using System;
using System.Drawing;
using Volpe.Cafe.Data;
using Volpe.Cafe.IO;
using Volpe.Cafe.Settings;
using Volpe.Utils;
namespace Volpe.Cafe.Reports.ReportTypes
{
    public class IndustryReport : ManufacturersReport
    {
        #region 
        public IndustryReport(string path, OutputSettings outputSettings)
            : base(path, TemplateType.IndustryReport, outputSettings)
        {
            base._autoBorder = true;
            base._autoChart = true;
            base._autoMerge = true;
            base._autoSummarize = true;
            base._fillColor = XlColor.Automatic;
            this._fitToPage = new Size(1, -1);
            base._font = new XlFont("Arial");
            base._headerAlignment = new XlTextAlignment(XlHAlign.Center, XlVAlign.Center);
            base._headerFillColor = XlColor.Automatic;
            base._headerFont = new XlFont("Arial", 10, false, true, false);
            base._landscape = true;
            base._margins = new XlMargins(0.5);
            base._numberFormat = "#,##0.00";
            base._repeatHeaders = true;
            base._textAlignment = new XlTextAlignment(XlHAlign.General, XlVAlign.Center);
            this._initialized = false;
            this._minYear = 0;
            this._maxYear = 0;
        }
        #endregion
        #region 
        public override void ParseData(ModelingSettings settings, int scenarioIndex, ModelYear year, Industry data,
            Industry baselineData)
        {
            if (!this._initialized)
            {
                this._minYear = data.MinYear.Year;
                this._maxYear = data.MaxYear.Year;
                this._mfrCount = data.ManufacturerCount;
                this._numSects = this._maxYear - this._minYear + 1;
                this._numTechs = settings.Technologies.Count;
                this.SetupBuffers(settings, scenarioIndex, year, data);
                this._initialized = true;
            }
            this.WriteData(year, data, baselineData);
        }
        private void SetupBuffers(ModelingSettings settings, int scenarioIndex, ModelYear year, Industry data)
        {
            this.SetupBuffer(settings, data, "Industry Overall", "Model Year Summary -- Industry Overall");
            this.SetupHeaders(settings);
            for (int i = 0; i < this._mfrCount; i++)
            {
                string mfrName = data.Manufacturers[i].Description.Name;
                mfrName = (mfrName.Length == 3) ? mfrName : Global.GetTitleCase(mfrName);
                this.SetupBuffer(settings, data, mfrName, "Model Year Summary -- " + mfrName);
                this.SetupHeaders(settings);
            }
        }
        private void WriteData(ModelYear year, Industry data, Industry baselineData)
        {
            int column = 3 + 3 * (year.Year - this._minYear);
            this.ActivateBuffer(0);
            this.Buffer[0, 0] = "Industry Overall";
            this.Buffer[0, column] = year.ToString() + " Total";
            this.Buffer[1, column] = "Current\nScenario";
            this.Buffer[1, column + 1] = "Delta\n(abs.)";
            this.Buffer[1, column + 2] = "Delta\n(%)";
            this.WriteIndustryData(year, column, data, baselineData);
            for (int i = 0; i < this._mfrCount; i++)
            {
                string mfrName = data.Manufacturers[i].Description.Name;
                mfrName = (mfrName.Length == 3) ? mfrName : Global.GetTitleCase(mfrName);
                this.ActivateBuffer(i + 1);
                this.Buffer[0, 0] = mfrName;
                this.Buffer[0, column] = year.ToString() + " Total";
                this.Buffer[1, column] = "Current\nScenario";
                this.Buffer[1, column + 1] = "Delta\n(abs.)";
                this.Buffer[1, column + 2] = "Delta\n(%)";
                this.WriteManufacturerData(year, column, data.Manufacturers[i]);
                this.WriteManufacturerDelta(year, column + 1, data.Manufacturers[i], baselineData.Manufacturers[i]);
            }
        }
        #endregion
        #region 
        private bool _initialized;        
        private int _minYear;
        private int _maxYear;
        private int _mfrCount;
        #endregion
    }
}

