using System;
using System.Collections.Specialized;
using Volpe.Cafe.Collections;
using Volpe.Cafe.IO;
namespace Volpe.Cafe.Data
{
    [Serializable]
    public class Industry : ICloneable
    {
        #region 
        private Industry()
        {
        }
        public Industry(string path)
            : this(path, "")
        {
        }
        public Industry(string path, string password)
            : this()
        {
            this._manufacturers = new ManufacturerCollection(32);
            this._modelingData = new IndustryModelingData();
            Input input = this.LoadFile(path, password);
            StringCollection errorList;
            bool hasErrors = false;
            try
            {   
                hasErrors = this.ParseFile(input, out errorList);
            }
            catch (Exception ex)
            {   
                throw new InputException("Errors were encountered while parsing file: " + ex.Message, path, ex);
            }
            finally
            {   
                input.Close();
            }
            if (hasErrors)
            {
                int maxErrCount = Math.Min(100, errorList.Count);
                string err = "\n\nThe input data file contains " + errorList.Count + " missing or invalid entries.";
                if (errorList.Count > maxErrCount) { err += "  (First " + maxErrCount + " errors are shown.)"; }
                err += "\n";
                for (int i = 0; i < maxErrCount; i++)
                {
                    err += ("\n" + errorList[i]);
                }
                throw new InputException(err, path);
            }
        }
        #endregion
        #region 
        #region 
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        public Industry Clone()
        {
            Industry ind = new Industry();
            ind._manufacturers = this._manufacturers.Clone();
            ind._modelingData  = this._modelingData.Clone();
            ind._minYear       = new ModelYear(this._minYear.Year);
            ind._maxYear       = new ModelYear(this._maxYear.Year);
            ind._minFp         = this._minFp;
            ind._maxFp         = this._maxFp;
            ind._minCw         = this._minCw;
            ind._maxCw         = this._maxCw;
            return ind;
        }
        #endregion
        public void WriteSummaryLog(LogWriter writer)
        {
            if (writer.Summary == null) { return; }
            writer.Summary.WriteLine("---------- Modeling Data ----------");
            writer.Summary.WriteLine("--- Model Years Analyzed ---\n  "
                + this._minYear.ToString() + " to " + this._maxYear.ToString() + "\n");
            writer.Summary.WriteLine("--- Valid Manufacturers ---");
            for (int i = 0, mfrCount = this._manufacturers.Count; i < mfrCount; i++)
            {
                Manufacturer mfr = this._manufacturers[i];
                writer.Summary.WriteLine(
                    "  " + mfr.Index + ". Code=" + mfr.Description.Code + ", Name=" + mfr.Description.Name +
                        "\n      Vehicles=" + mfr.VehicleCount +
                            " (Dom Autos=" + mfr.Vehicles.GetVehicleCountByType(VehicleType.Domestic | VehicleType.PassengerAuto) +
                            ", Imp Autos=" + mfr.Vehicles.GetVehicleCountByType(VehicleType.Imported | VehicleType.PassengerAuto) +
                            ", Dom LTs  =" + mfr.Vehicles.GetVehicleCountByType(VehicleType.Domestic | VehicleType.LightTruck   ) +
                            ", Imp LTs  =" + mfr.Vehicles.GetVehicleCountByType(VehicleType.Imported | VehicleType.LightTruck   ) +
                            ", Dom HLTs =" + mfr.Vehicles.GetVehicleCountByType(VehicleType.Domestic | VehicleType.HeavyLT      ) +
                            ", Imp HLTs =" + mfr.Vehicles.GetVehicleCountByType(VehicleType.Imported | VehicleType.HeavyLT      ) + ")" +
                        "\n      Engines=" + mfr.Engines.Count +
                        "\n      Transmissions=" + mfr.Transmissions.Count);
            }
            writer.Summary.WriteLine();
            writer.Summary.WriteLine("--- Industry Summary ---");
            int engs = 0, trns = 0, vehs = 0, da = 0, ia = 0, dlt = 0, ilt = 0, dhlt = 0, ihlt = 0;
            for (int i = 0, mfrCount = this._manufacturers.Count; i < mfrCount; i++)
            {
                Manufacturer mfr = this._manufacturers[i];
                engs  += mfr.Engines.Count;
                trns  += mfr.Transmissions.Count;
                vehs  += mfr.VehicleCount;
                da    += mfr.Vehicles.GetVehicleCountByType(VehicleType.Domestic | VehicleType.PassengerAuto);
                ia    += mfr.Vehicles.GetVehicleCountByType(VehicleType.Imported | VehicleType.PassengerAuto);
                dlt   += mfr.Vehicles.GetVehicleCountByType(VehicleType.Domestic | VehicleType.LightTruck   );
                ilt   += mfr.Vehicles.GetVehicleCountByType(VehicleType.Imported | VehicleType.LightTruck   );
                dhlt  += mfr.Vehicles.GetVehicleCountByType(VehicleType.Domestic | VehicleType.HeavyLT      );
                ihlt  += mfr.Vehicles.GetVehicleCountByType(VehicleType.Imported | VehicleType.HeavyLT      );
            }
            writer.Summary.WriteLine("  Total Manufacturers Analyzed:  " + this._manufacturers.Count);
            writer.Summary.WriteLine("  Total Vehicles Analyzed:  " + vehs);
            writer.Summary.WriteLine("    Domestic Autos: " + da);
            writer.Summary.WriteLine("    Imported Autos: " + ia);
            writer.Summary.WriteLine("    Domestic LTs:   " + dlt);
            writer.Summary.WriteLine("    Imported LTs:   " + ilt);
            writer.Summary.WriteLine("    Domestic HLTs:  " + dhlt);
            writer.Summary.WriteLine("    Imported HLTs:  " + ihlt);
            writer.Summary.WriteLine("  Total Engines Analyzed:   " + engs);
            writer.Summary.WriteLine("  Total Transmissions Analyzed:  " + trns);
            writer.Summary.WriteLine();
        }
        public VehicleCollection GetVehicles()
        {
            VehicleCollection vc = new VehicleCollection(this.VehicleCount);
            for (int i = 0; i < this._manufacturers.Count; i++)
            {
                vc.AddRange(this._manufacturers[i].Vehicles);
            }
            return vc;
        }
        public VehicleCollection GetModelYearVehicles()
        {
            VehicleCollection vc = new VehicleCollection(this.ModelYearVehicleCount);
            for (int i = 0; i < this._manufacturers.Count; i++)
            {
                vc.AddRange(this._manufacturers[i].ModelYearVehicles);
            }
            return vc;
        }
        private Input LoadFile(string path, string password)
        {
            return new Input(path, password);
        }
        private bool ParseFile(Input input, out StringCollection errorList)
        {
            string[] sheets = input.SheetNames;
            string[] names = {"Manufacturers", "Vehicles", "Engines", "Transmissions"};
            int   [] indexes = {-1, -1, -1, -1};
            for (int i = 0; i < sheets.Length; i++)
            {
                for (int j = 0; j < names.Length; j++)
                {
                    if (input.Compare(sheets[i], names[j], false)) { indexes[j] = i; break; }
                }
            }
            input.VerifyIndexes(indexes, names);
            errorList = new StringCollection();     
            int mfrWsMinYear, mfrWsMaxYear;         
            ManufacturerCollection   mc = this.ParseFile_LoadManufacturers(input, indexes[0], out mfrWsMinYear, out mfrWsMaxYear, errorList);
            VehicleCollection[]      vc = this.ParseFile_LoadVehicles     (input, indexes[1], mc, errorList);
            EngineCollection[]       ec = this.ParseFile_LoadEngines      (input, indexes[2], mc, errorList);
            TransmissionCollection[] tc = this.ParseFile_LoadTransmissions(input, indexes[3], mc, errorList);
            if (errorList.Count > 0) { return true; }
            int mfrIndex = 0;
            int indMinYear = int.MaxValue, indMaxYear = -1;
            for (int i = 0; i < mc.Count; i++)
            {
                int minYear, maxYear;
                mc[i].Initialize(vc[i], ec[i], tc[i], out minYear, out maxYear);
                if (mc[i].VehicleCount == 0)
                {
                    throw new Exception("\n\nManufacturer Code " + mc[i].Description.Code + " does not contain any valid vehicles.");
                }
                else
                {   
                    this._manufacturers.Add(mc[i]);
                    mc[i].SetIndex(mfrIndex++);
                    indMinYear = Math.Min(minYear, indMinYear);
                    indMaxYear = Math.Max(maxYear, indMaxYear);
                }
            }
            indMinYear = Math.Max(mfrWsMinYear, indMinYear);
            indMaxYear = Math.Min(mfrWsMaxYear, indMaxYear);
            if (indMinYear <  9999)              { this._minYear = new ModelYear(indMinYear); }
            if (indMaxYear >= ModelYear.MinYear) { this._maxYear = new ModelYear(indMaxYear); }
            this.UpdateMinMaxFpCw();
            return false;
        }
        private ManufacturerCollection ParseFile_LoadManufacturers(Input input, int index, out int mfrMinYear,
            out int mfrMaxYear, StringCollection errorList)
        {
            input.ActivateWorksheet(index);
            int rows = input.Rows;
            int cols = input.Columns;
            string[] names = {   
                                 "Manufacturer Code",                       
                                 "Manufacturer Name",                       
                                 "Cost Allocation Strategy",                
                                 "Avg. CW of MY02 Lt. Trucks",              
                                 "Discount Rate",                           
                                 "Optimize",                                
                                 "Willingness to Pay CAFE Fines",           
                                 "Available Domestic Auto Credits (mpg)",   
                                 "Available Imported Auto Credits (mpg)",   
                                 "Available Light Truck Credits (mpg)",     
                                 "Credits Apply to Baseline",               
                                 "AC Domestic Auto Credits (mpg)",          
                                 "AC Imported Auto Credits (mpg)",          
                                 "AC Light Truck Credits (mpg)",            
                                 "AC Credits Apply to Baseline"             
                             };
            int[] indexes = new int[names.Length];
            for (int i = 0; i < indexes.Length; i++) { indexes[i] = -1; }
            for (int i = 0; i < cols; i++)
            {
                string column0 = input.GetString(0, i);
                string column1 = input.GetString(1, i);
                int    column1_int = Global.GetInt32(column1);
                if      (input.Compare(column1, names[ 0], true)                                           ) { indexes[ 0] = i; }
                else if (input.Compare(column1, names[ 1], true)                                           ) { indexes[ 1] = i; }
                else if (input.Compare(column1, names[ 2], true)                                           ) { indexes[ 2] = i; }
                else if (input.Compare(column1, names[ 3], true)                                           ) { indexes[ 3] = i; }
                else if (input.Compare(column1, names[ 4], true)                                           ) { indexes[ 4] = i; }
                else if (input.Compare(column1, names[ 5], true)                                           ) { indexes[ 5] = i; }
                else if (input.Compare(column0, names[ 6], true) && ModelYear.IsValid(column1_int)         ) { indexes[ 6] = i; }
                else if (input.Compare(column0, names[ 7], true) && ModelYear.IsValid(column1_int)         ) { indexes[ 7] = i; }
                else if (input.Compare(column0, names[ 8], true) && ModelYear.IsValid(column1_int)         ) { indexes[ 8] = i; }
                else if (input.Compare(column0, names[ 9], true) && ModelYear.IsValid(column1_int)         ) { indexes[ 9] = i; }
                else if (input.Compare(column0, names[10], true) || input.Compare(column1, names[10], true)) { indexes[10] = i; }
                else if (input.Compare(column0, names[11], true) && ModelYear.IsValid(column1_int)         ) { indexes[11] = i; }
                else if (input.Compare(column0, names[12], true) && ModelYear.IsValid(column1_int)         ) { indexes[12] = i; }
                else if (input.Compare(column0, names[13], true) && ModelYear.IsValid(column1_int)         ) { indexes[13] = i; }
                else if (input.Compare(column0, names[14], true) || input.Compare(column1, names[10], true)) { indexes[14] = i; }
            }
            input.VerifyIndexes(indexes, names);
            input.GetMinMaxYears(1, indexes[6], out mfrMinYear, out mfrMaxYear);
            ManufacturerCollection mc = new ManufacturerCollection(32);
            for (int i = 2; i < rows; i++)
            {
                ManufacturerDescription md = this.ParseFile_LoadManufacturer(input, i, indexes, mfrMinYear, mfrMaxYear, errorList);
                if (md != null) { mc.Add(new Manufacturer(md)); }
            }
            return mc;
        }
        private ManufacturerDescription ParseFile_LoadManufacturer(Input input, int row, int[] indexes, int minYear,
            int maxYear, StringCollection errorList)
        {
            int    mfrCode = input.GetInt32 (row, indexes[0]);
            string mfrName = input.GetString(row, indexes[1]);
            if (mfrCode <= 0 || mfrName == string.Empty) { return null; }
            ManufacturerDescription md = new ManufacturerDescription();
            md.Code = mfrCode;
            md.Name = mfrName;
            md.CostAllocationStrategy   = (indexes[ 2] == -1) ? 0                :  input.GetInt32 (row, indexes[ 2]);
            md.AverageMY2002LtCw        = (indexes[ 3] == -1) ? 4329.48480892067 :  input.GetDouble(row, indexes[ 3]);
            md.DiscountRate             = (indexes[ 4] == -1) ? -1               :  input.GetDouble(row, indexes[ 4]);
            md.Optimize                 = (indexes[ 5] == -1) ? false            : (input.GetChar  (row, indexes[ 5]) == 'Y');
            md.CreditsApplyToBaseline   = (indexes[10] == -1) ? false            : (input.GetChar  (row, indexes[10]) == 'Y');
            md.ACCreditsApplyToBaseline = (indexes[14] == -1) ? false            : (input.GetChar  (row, indexes[14]) == 'Y');
            md.WillingToPayFines    = new bool  [Math.Max(16, maxYear - ModelYear.MinYear + 1)];
            md.AvailableCreditsDA   = new double[Math.Max(16, maxYear - ModelYear.MinYear + 1)];
            md.AvailableCreditsIA   = new double[Math.Max(16, maxYear - ModelYear.MinYear + 1)];
            md.AvailableCreditsLT   = new double[Math.Max(16, maxYear - ModelYear.MinYear + 1)];
            md.AvailableACCreditsDA = new double[Math.Max(16, maxYear - ModelYear.MinYear + 1)];
            md.AvailableACCreditsIA = new double[Math.Max(16, maxYear - ModelYear.MinYear + 1)];
            md.AvailableACCreditsLT = new double[Math.Max(16, maxYear - ModelYear.MinYear + 1)];
            int offset = minYear - ModelYear.MinYear;
            for (int i = 0; i < maxYear - minYear + 1; i++)
            {
                if (indexes[ 6] != -1) { md.WillingToPayFines   [offset + i] = input.GetString(row, indexes[ 6] + i).ToUpper().Equals("Y"); }
                if (indexes[ 7] != -1) { md.AvailableCreditsDA  [offset + i] = input.GetDouble(row, indexes[ 7] + i); }
                if (indexes[ 8] != -1) { md.AvailableCreditsIA  [offset + i] = input.GetDouble(row, indexes[ 8] + i); }
                if (indexes[ 9] != -1) { md.AvailableCreditsLT  [offset + i] = input.GetDouble(row, indexes[ 9] + i); }
                if (indexes[11] != -1) { md.AvailableACCreditsDA[offset + i] = input.GetDouble(row, indexes[11] + i); }
                if (indexes[12] != -1) { md.AvailableACCreditsIA[offset + i] = input.GetDouble(row, indexes[12] + i); }
                if (indexes[13] != -1) { md.AvailableACCreditsLT[offset + i] = input.GetDouble(row, indexes[13] + i); }
            }
            return md;
        }
        private VehicleCollection[] ParseFile_LoadVehicles(Input input, int index, ManufacturerCollection mfrs,
            StringCollection errorList)
        {
            input.ActivateWorksheet(index);
            int rows = input.Rows;
            int cols = input.Columns;
            string[] names = {   
                                 "Sales", "MSRP", "Price",          
                                 "Vehicle Code"                ,    
                                 "Manufacturer"                ,    
                                 "Model"                       ,    
                                 "Nameplate"                   ,    
                                 "Fuel Economy On Primary Fuel",    
                                 "Engine Code"                 ,    
                                 "Transmission Code"           ,    
                                 "Origin"                      ,    
                                 "Regulatory Class"            ,    
                                 "Technology Class"            ,    
                                 "Class"                       ,    
                                 "Style"                       ,    
                                 "Structure"                   ,    
                                 "Drive"                       ,    
                                 "Footprint"                   ,    
                                 "Curb Weight"                 ,    
                                 "GVWR"                        ,    
                                 "Seating (max)"               ,    
                                 "Fuel Capacity"               ,    
                                 "Type of Hybrid/Electric Vehicle", 
                                 "Predecessor"                 ,    
                                 "Refresh Years"               ,    
                                 "Redesign Years"              ,    
                                 "EPS", "IACC", "MHEV", "BISG", "CISG", "PSHEV", "2MHEV", "PHEV", "MS1", "MS2",
                                 "MS5", "ROLL", "LDB" , "SAXU", "SAXL", "AERO"
                             };
            int[] indexes = new int[names.Length];
            for (int i = 0; i < names.Length; i++) { indexes[i] = -1; }
            for (int i = 0; i < cols; i++)
            {
                string column = input.GetString(1, i);
                string smpCol = input.GetString(0, i);      
                for (int j = 0; j < names.Length; j++)
                {
                    if (indexes[j] == -1)
                    {
                        if      (input.Compare(smpCol, names[j], true)) { indexes[j] = i; break; }
                        else if (input.Compare(column, names[j], true)) { indexes[j] = i; break; }
                    }
                }
            }
            if (indexes[1] == -1 && indexes[2] != -1)
            {   
                indexes[1] = indexes[2];
            }
            else if (indexes[1] != -1 && indexes[2] == -1)
            {   
                indexes[2] = indexes[1];
            }
            input.VerifyIndexes(indexes, names);
            int salesMinYear, salesMaxYear;
            int msrpMinYear , msrpMaxYear ;
            int priceMinYear, priceMaxYear;
            input.GetMinMaxYears(1, indexes[0], out salesMinYear, out salesMaxYear);
            input.GetMinMaxYears(1, indexes[1], out msrpMinYear , out msrpMaxYear );
            input.GetMinMaxYears(1, indexes[2], out priceMinYear, out priceMaxYear);
            this._minYear = new ModelYear(Math.Max(Math.Max(salesMinYear, msrpMinYear), priceMinYear));
            this._maxYear = new ModelYear(Math.Min(Math.Min(salesMaxYear, msrpMaxYear), priceMaxYear));
            VehicleCollection[] vc = new VehicleCollection[mfrs.Count];
            for (int i = 0; i < mfrs.Count; i++)
            {
                vc[i] = new VehicleCollection(32);
            }
            for (int i = 2; i < rows; i++)
            {   
                int mfrIndex;
                VehicleDescription vd = this.ParseFile_LoadVehicle(input, i, indexes,
                    salesMinYear, salesMaxYear, msrpMinYear, msrpMaxYear, priceMinYear, priceMaxYear,
                    mfrs, out mfrIndex, errorList);
                if (vd != null && mfrIndex != -1) { vc[mfrIndex].Add(new Vehicle(vd)); }
            }
            return vc;
        }
        private VehicleDescription ParseFile_LoadVehicle(Input input, int row, int[] indexes,
            int salesMinYear, int salesMaxYear, int msrpMinYear, int msrpMaxYear, int priceMinYear, int priceMaxYear,
            ManufacturerCollection mfrs, out int mfrIndex, StringCollection errorList)
        {
            mfrIndex = -1;
            double curbWeigth = input.GetDouble(row, indexes[18]);  
            VehicleDescription vd =  new VehicleDescription(curbWeigth);
            string vehCode = input.GetString(row, indexes[3]);
            string mfrName = input.GetString(row, indexes[4]);
            string model   = input.GetString(row, indexes[5]);
            if (vehCode == string.Empty && mfrName == string.Empty && model == string.Empty) { return null; }
            if (vehCode == "X") { return null; }
			vd.Code                  = Global.GetInt32(vehCode); 
			vd.Manufacturer          = mfrName;                  
			vd.Model                 = model;                    
			vd.Nameplate             = input.GetString(row, indexes[ 6]);
			vd.FuelEconomy           = input.GetDouble(row, indexes[ 7]);
			vd.EngineCode            = input.GetInt32 (row, indexes[ 8]);
			vd.TransmissionCode      = input.GetInt32 (row, indexes[ 9]);
            vd.Origin                = input.GetChar  (row, indexes[10]);
            vd.RegulatoryIndicator   = input.GetString(row, indexes[11]);
            vd.TechnologyClass       = input.GetString(row, indexes[12]);
            vd.Class                 = input.GetString(row, indexes[13]);
            vd.Style                 = input.GetString(row, indexes[14]);
			vd.Structure             = input.GetString(row, indexes[15]);
			vd.Drive                 = input.GetChar  (row, indexes[16]);
			vd.Footprint             = input.GetDouble(row, indexes[17]);
			vd.CurbWeight            = curbWeigth;               
			vd.GVWR                  = input.GetDouble(row, indexes[19]);
            vd.Seating               = input.GetInt32 (row, indexes[20]);
            vd.FuelCapacity          = input.GetDouble(row, indexes[21]);
			vd.HybridType            = input.GetString(row, indexes[22]);
			vd.PredecessorCode       = input.GetInt32 (row, indexes[23]);
            vd.RefreshYears          = this.ParseFile_LoadVehicle_RefreshRedesign(input.GetString(row, indexes[24]).Split(','));
            vd.RedesignYears         = this.ParseFile_LoadVehicle_RefreshRedesign(input.GetString(row, indexes[25]).Split(','));
            vd.Sales = this.ParseFile_LoadVehicleSMP(input, row, indexes[0], salesMinYear, salesMaxYear);
            vd.Msrp  = this.ParseFile_LoadVehicleSMP(input, row, indexes[1], msrpMinYear , msrpMaxYear );
            vd.Price = this.ParseFile_LoadVehicleSMP(input, row, indexes[2], priceMinYear, priceMaxYear);
            int maxYr = (int)Math.Max(Math.Max(priceMaxYear, msrpMaxYear), salesMaxYear) - ModelYear.MinYear + 1;
            if (vd.Sales.Length < maxYr) { double[] sales = new double[maxYr]; vd.Sales.CopyTo(sales, 0); vd.Sales = sales; }
            if (vd.Msrp .Length < maxYr) { double[] msrp  = new double[maxYr]; vd.Msrp .CopyTo(msrp , 0); vd.Msrp  = msrp ; }
            if (vd.Price.Length < maxYr) { double[] price = new double[maxYr]; vd.Price.CopyTo(price, 0); vd.Price = price; }
            for (int i = 0; i < maxYr; i++)
            {
                if      (vd.Msrp[i] == 0 && vd.Price[i] != 0) { vd.Msrp [i] = vd.Price[i]; }
                else if (vd.Msrp[i] != 0 && vd.Price[i] == 0) { vd.Price[i] = vd.Msrp [i]; }
            }
            const int techCount = 16;   
            this.ParseFile_LoadTechApplicability(input, row, indexes, 26, techCount, out vd.UsedTechnologies,
                out vd.AvailableTechnologies, TechnologyIndexes.EPS);
            string hdr = "Vehicle (R:" + (row + 1).ToString("0000") + "):  ";
            mfrIndex = this.ParseFile_GetManufacturerIndex(input, mfrs, vd.Manufacturer);
            if (vd.Code < 1) { errorList.Add(hdr + "Vehicle Code is not valid."); }
            if (mfrIndex == -1) { errorList.Add(hdr + "Manufacturer is not specified or not valid."); }
            if (vd.Model == string.Empty) { errorList.Add(hdr + "Model is not specified."); }
            if (vd.Nameplate == string.Empty) { errorList.Add(hdr + "Nameplate is not specified."); }
            if (vd.FuelEconomy < 1) { errorList.Add(hdr + "Fuel Economy is not valid."); }
            if (vd.EngineCode < 1) { errorList.Add(hdr + "Engine Code is not valid."); }
            if (vd.TransmissionCode < 1) { errorList.Add(hdr + "Transmission Code is not valid."); }
            if (vd.Origin != 'I' && vd.Origin != 'D') { errorList.Add(hdr + "Origin is not valid."); }
            if (vd.RegulatoryIndicator != "PC" && vd.RegulatoryIndicator != "LT") { errorList.Add(hdr + "Regulatory Indicator is not valid."); }
            if (!Global.StringCompareAny(vd.TechnologyClass, new string[] {"Subcompact PC", "Subcompact Perf. PC", "Compact PC",
                "Compact Perf. PC", "Midsize PC", "Midsize Perf. PC", "Large PC", "Large Perf. PC", "Minivan LT", "Small LT",
                "Midsize LT", "Large LT"}, true)) { errorList.Add(hdr + "Technology Class is not valid."); }
            if (vd.Style == string.Empty) { errorList.Add(hdr + "Style is not specified."); }
            if (vd.Structure == string.Empty) { errorList.Add(hdr + "Structure is not specified."); }
            if (!Global.StringCompareAny(vd.Drive.ToString(), new string[] {"F", "R", "2", "4", "A"}, false)) { errorList.Add(hdr + "Class is not specified."); }
            if (vd.Footprint < 1) { errorList.Add(hdr + "Footprint is not valid."); }
            if (vd.CurbWeight < 1) { errorList.Add(hdr + "Curb Weight is not valid."); }
            if (vd.GVWR < 1) { errorList.Add(hdr + "GVWR is not valid."); }
            if (vd.Seating < 1) { errorList.Add(hdr + "Seating (Max) is not valid."); }
            if (vd.FuelCapacity < 1) { errorList.Add(hdr + "Fuel Capacity is not valid."); }
            if (vd.PredecessorCode < 0) { errorList.Add(hdr + "Predecessor Code is not valid."); }
            bool foundSlMsrp = false;
            for (int i = 0; i < maxYr; i++)
            {
                if (vd.Sales[i] > 0 && vd.Msrp[i] > 0) { foundSlMsrp = true; break; }
            }
            if (!foundSlMsrp) { errorList.Add(hdr + "No Sales and MSRP found for the same year."); }
            return vd;
        }
        private int[] ParseFile_LoadVehicle_RefreshRedesign(string[] years)
        {
            int count = years.Length;
            int[] iYears = new int[count];
            int iCount = 0;
            for (int i = 0; i < count; i++)
            {
                int year = Global.GetInt32(years[i]);
                if (year > 1900)
                {   
                    bool duplicate = false;
                    for (int j = 0; j < iCount; j++)
                    {
                        if (iYears[j] == year) { duplicate = true; break; }
                    }
                    if (!duplicate) { iYears[iCount++] = year; }
                }
            }
            int[] actualYears = new int[iCount];
            Array.Copy(iYears, actualYears, iCount);
            Array.Sort(actualYears);
            return actualYears;
        }
        private double[] ParseFile_LoadVehicleSMP(Input input, int row, int column, int minYear, int maxYear)
        {
            double[] values = new double[Math.Max(16, maxYear - ModelYear.MinYear + 1)];
            int years = maxYear - minYear + 1;
            int offset = minYear - ModelYear.MinYear;
            for (int i = 0; i < years; i++)
            {
                values[i + offset] = input.GetDouble(row, i + column);
            }
            return values;
        }
        private EngineCollection[] ParseFile_LoadEngines(Input input, int index, ManufacturerCollection mfrs,
            StringCollection errorList)
        {
            input.ActivateWorksheet(index);
            int rows = input.Rows;
            int cols = input.Columns;
            string[] names = {   
                                 "Engine Code"            ,     
                                 "Manufacturer"           ,     
                                 "Configuration"          ,     
                                 "Primary Fuel"           ,     
                                 "Engine Oil Viscosity"   ,     
                                 "Cycle"                  ,     
                                 "Fuel Delivery System"   ,     
                                 "Aspiration"             ,     
                                 "Valvetrain Design"      ,     
                                 "Valve Actuation/Timing" ,     
                                 "Valve Lift"             ,     
                                 "Cylinders"              ,     
                                 "Valves/Cylinder"        ,     
                                 "Deactivation"           ,     
                                 "Displacement"           ,     
                                 "Max. Horsepower"        ,     
                                 "Max. Torque"            ,     
                                 "LUB"  , "EFR" , "CCPS" , "DVVLS", "DEACS", "ICP"  , "DCP"  , "DVVLD", "CVVL", "DEACD",
                                 "DEACO", "CCPO", "DVVLO", "CDOHC", "SGDI" , "CBRST", "TRBDS", "EGRB" , "DSLT", "DSLC"
                             };
            int[] indexes = new int[names.Length];
            for (int i = 0; i < names.Length; i++) { indexes[i] = -1; }
            for (int i = 0; i < cols; i++)
            {
                string column = input.GetString(1, i);
                for (int j = 0; j < names.Length; j++)
                {
                    if (indexes[j] == -1 && input.Compare(column, names[j], true)) { indexes[j] = i; break; }
                }
            }
            input.VerifyIndexes(indexes, names);
            EngineCollection[] ec = new EngineCollection[mfrs.Count];
            for (int i = 0; i < mfrs.Count; i++)
            {
                ec[i] = new EngineCollection(32);
            }
            for (int i = 2; i < rows; i++)
            {   
                int mfrIndex;
                EngineDescription ed = this.ParseFile_LoadEngine(input, i, indexes, mfrs, out mfrIndex, errorList);
                if (ed != null && mfrIndex != -1) { ec[mfrIndex].Add(new Engine(ed)); }
            }
            return ec;
        }
        private EngineDescription ParseFile_LoadEngine(
            Input input, int row, int[] indexes, ManufacturerCollection mfrs, out int mfrIndex, StringCollection errorList)
        {
            mfrIndex = -1;
            EngineDescription ed = new EngineDescription();
            string engCode = input.GetString(row, indexes[0]);
            string mfrName = input.GetString(row, indexes[1]);
            if (engCode == string.Empty && mfrName == string.Empty) { return null; }
            if (engCode == "X") { return null; }
            ed.Code                 = Global.GetInt32(engCode); 
            ed.Manufacturer         = mfrName;                  
            ed.Configuration        = input.GetString(row, indexes[ 2]);
            ed.Fuel                 = input.GetString(row, indexes[ 3]); if (ed.Fuel != "D") { ed.Fuel = "G"; }
            ed.EngineOilViscosity   = input.GetString(row, indexes[ 4]);
            ed.Cycle                = input.GetChar  (row, indexes[ 5]);
            ed.FuelSystem           = input.GetString(row, indexes[ 6]);
            ed.Aspiration           = input.GetString(row, indexes[ 7]);
            ed.ValvetrainDesign     = input.GetString(row, indexes[ 8]);
            ed.ValveActuationTiming = input.GetString(row, indexes[ 9]);
            ed.ValveLift            = input.GetString(row, indexes[10]);
            ed.Cylinders            = input.GetInt32 (row, indexes[11]);
            ed.ValvesPerCylinder    = input.GetInt32 (row, indexes[12]);
            ed.Deactivation         = input.GetDouble(row, indexes[13]); if (ed.Deactivation == 0 && input.GetString(row, indexes[13]) == "Y") { ed.Deactivation = 0.3; }
            ed.Displacement         = input.GetDouble(row, indexes[14]);
            ed.Horsepower           = input.GetDouble(row, indexes[15]);
            ed.Torque               = input.GetDouble(row, indexes[16]);
            const int techCount = 20;   
            this.ParseFile_LoadTechApplicability(input, row, indexes, 17, techCount, out ed.UsedTechnologies,
                out ed.AvailableTechnologies, TechnologyIndexes.LUB);
            string hdr = "Engine (R:" + (row + 1).ToString("0000") + "):  ";
            mfrIndex = this.ParseFile_GetManufacturerIndex(input, mfrs, ed.Manufacturer);
            if (ed.Code < 1) { errorList.Add(hdr + "Engine Code is not valid."); }
            if (mfrIndex == -1) { errorList.Add(hdr + "Manufacturer is not specified or not valid."); }
            if (ed.Configuration == string.Empty) { errorList.Add(hdr + "Configuration is not specified."); }
            if (ed.Fuel == string.Empty) { errorList.Add(hdr + "Primary Fuel is not specified."); }
            if (ed.EngineOilViscosity == string.Empty) { errorList.Add(hdr + "Engine Oil Viscosity is not specified."); }
            if (ed.Cycle == '\0') { errorList.Add(hdr + "Cycle is not specified."); }
            if (ed.FuelSystem == string.Empty) { errorList.Add(hdr + "Fuel Delivery System is not specified."); }
            if (!Global.StringCompareAny(ed.Aspiration, new string[] {"NA", "S", "T", "T2"}, false)) { errorList.Add(hdr + "Aspiration is not valid."); }
            if (ed.Configuration != "R")
            {
                if (!Global.StringCompareAny(ed.ValvetrainDesign, new string[] {"OHV", "SOHC", "DOHC"}, false)) { errorList.Add(hdr + "Valvetrain Design is not valid."); }
                if (ed.ValveActuationTiming == string.Empty) { errorList.Add(hdr + "Valve Actuation/Timing is not specified."); }
                if (ed.ValveLift == string.Empty) { errorList.Add(hdr + "Valve Lift is not specified."); }
            }
            if (ed.Cylinders < 1) { errorList.Add(hdr + "Number of Cylinders is too small."); }
            if (ed.ValvesPerCylinder < 1) { errorList.Add(hdr + "Number of Valves/Cylinders is too small."); }
            if (ed.Deactivation < 0) { errorList.Add(hdr + "Deactivation is not valid."); }
            if (ed.Displacement < 0) { errorList.Add(hdr + "Displacement is not valid."); }
            if (ed.Horsepower < 0) { errorList.Add(hdr + "Horsepower is not valid."); }
            if (ed.Torque < 0) { errorList.Add(hdr + "Torque is not valid."); }
            return ed;
        }
        private double GetCompressionRatio(string value)
        {
            double ratio = 0;
            if (value != "")
            {
                int index = value.IndexOf(":");
                if (index != -1)
                {   
                    ratio = Global.GetDouble(value.Substring(0, index)) / Global.GetDouble(value.Substring(index + 1));
                }
                else
                {   
                    ratio = Global.GetDouble(value);
                }
            }
            return ratio;
        }
        private TransmissionCollection[] ParseFile_LoadTransmissions(Input input, int index, ManufacturerCollection mfrs,
            StringCollection errorList)
        {
            input.ActivateWorksheet(index);
            int rows = input.Rows;
            int cols = input.Columns;
            string[] names = {   
                                 "Transmission Code"      ,     
                                 "Manufacturer"           ,     
                                 "Type"                   ,     
                                 "Number of Forward Gears",     
                                 "6MAN", "IATC", "CVT", "NAUTO", "DCTAM"
                             };
            int[] indexes = new int[names.Length];
            for (int i = 0; i < names.Length; i++) { indexes[i] = -1; }
            for (int i = 0; i < cols; i++)
            {
                string column = input.GetString(1, i);
                for (int j = 0; j < names.Length; j++)
                {
                    if (indexes[j] == -1 && input.Compare(column, names[j], true)) { indexes[j] = i; break; }
                }
            }
            input.VerifyIndexes(indexes, names);
            TransmissionCollection[] tc = new TransmissionCollection[mfrs.Count];
            for (int i = 0; i < mfrs.Count; i++)
            {
                tc[i] = new TransmissionCollection(32);
            }
            for (int i = 2; i < rows; i++)
            {   
                int mfrIndex;
                TransmissionDescription td = this.ParseFile_LoadTransmission(input, i, indexes, mfrs, out mfrIndex, errorList);
                if (td != null && mfrIndex != -1) { tc[mfrIndex].Add(new Transmission(td)); }
            }
            return tc;
        }
        private TransmissionDescription ParseFile_LoadTransmission(
            Input input, int row, int[] indexes, ManufacturerCollection mfrs, out int mfrIndex, StringCollection errorList)
        {
            mfrIndex = -1;
            TransmissionDescription td = new TransmissionDescription();
            string trnCode = input.GetString(row, indexes[0]);
            string mfrName = input.GetString(row, indexes[1]);
            if (trnCode == string.Empty && mfrName == string.Empty) { return null; }
            if (trnCode == "X") { return null; }
            td.Code            = Global.GetInt32(trnCode); 
            td.Manufacturer    = mfrName;                  
            td.Type            = input.GetString(row, indexes[2]);
            td.NumForwardGears = input.GetInt32 (row, indexes[3]);
            const int techCount = 5;    
            this.ParseFile_LoadTechApplicability(input, row, indexes, 4, techCount, out td.UsedTechnologies,
                out td.AvailableTechnologies, TechnologyIndexes.MAN6);
            string hdr = "Transmission (R:" + (row + 1).ToString("0000") + "):  ";
            mfrIndex = this.ParseFile_GetManufacturerIndex(input, mfrs, td.Manufacturer);
            if (td.Code < 1) { errorList.Add(hdr + "Transmission Code is not valid."); }
            if (mfrIndex == -1) { errorList.Add(hdr + "Manufacturer is not specified or not valid."); }
            if (!Global.StringCompareAny(td.Type, new string[] {"A", "M", "CVT", "AMT", "DCT"}, false)) { errorList.Add(hdr + "Type is not valid."); }
            if (td.Type != "CVT" && td.NumForwardGears < 3) { errorList.Add(hdr + "Number of Forward Gears is too small."); }
            return td;
        }
        private void ParseFile_LoadTechApplicability(Input input, int row, int[] indexes, int colIndex,
            int techCount, out int[] usedTechnologies, out int[] availableTechnologies, int firstTechIndex)
        {
            int[] techAp = new int[techCount];
            int techUsedCount = 0, techAvailCount = 0;
            for (int i = 0; i < techCount; i++)
            {
                techAp[i] = this.ParseFile_ParseTechApplicability(input.GetString(row, indexes[colIndex + i]));
                if      (techAp[i] == 1) { techUsedCount ++; }
                else if (techAp[i] == 0) { techAvailCount++; }
            }
            usedTechnologies      = new int[techUsedCount ];
            availableTechnologies = new int[techAvailCount];
            for (int i = 0, ut = 0, at = 0; i < techCount; i++)
            {
                if      (techAp[i] == 1) { usedTechnologies     [ut++] = firstTechIndex + i; }
                else if (techAp[i] == 0) { availableTechnologies[at++] = firstTechIndex + i; }
            }
        }
        private int ParseFile_ParseTechApplicability(string value)
        {
            return (value == "SKIP") ? -1 : (value == "USED") ? 1 : 0;
        }
        private int ParseFile_GetManufacturerIndex(Input input, ManufacturerCollection mfrs, string mfrName)
        {
            for (int i = 0; i < mfrs.Count; i++)
            {
                if (input.Compare(mfrs[i].Description.Name, mfrName, false)) { return i; }
            }
            return -1;
        }
        internal void SetModelingData(IndustryModelingData value)
        {
            this._modelingData = value;
        }
        public void UpdateMinMaxFpCw()
        {
            this._minFp = this._minCw = 99999;
            this._maxFp = this._maxCw =    -1;
            for (int i = 0, mfrCount = this._manufacturers.Count; i < mfrCount; i++)
            {   
                Manufacturer mfr = this._manufacturers[i];
                for (int j = 0, vehCount = mfr.Vehicles.Count; j < vehCount; j++)
                {
                    VehicleDescription vd = mfr.Vehicles[j].Description;
                    double fp = vd.Footprint, cw = vd.CurbWeight;
                    if (fp < this._minFp) { this._minFp = fp; }
                    if (fp > this._maxFp) { this._maxFp = fp; }
                    if (cw < this._minCw) { this._minCw = cw; }
                    if (cw > this._maxCw) { this._maxCw = cw; }
                } 
            } 
        }
        public Industry GetMergedFleet()
        {
            Industry ind = this.Clone();
            ind.MergeManufacturers();
            return ind;
        }
        void MergeManufacturers()
        {
            VehicleCollection      vehs = new VehicleCollection();
            EngineCollection       engs = new EngineCollection();
            TransmissionCollection trns = new TransmissionCollection();
            for (int i = 0, mfrCount = this._manufacturers.Count; i < mfrCount; i++)
            {   
                vehs.AddRange(this._manufacturers[i].Vehicles     );
                engs.AddRange(this._manufacturers[i].Engines      );
                trns.AddRange(this._manufacturers[i].Transmissions);
            }
            int vehIndex = 1, engIndex = 1, trnIndex = 1;
            for (int i = 0, vehCount = vehs.Count; i < vehCount; i++)
            {   
                string mfrName = vehs[i].Description.Manufacturer;
                vehs[i].Description.Manufacturer = "MERGED FLEET";
                vehs[i].Description.Code         = vehIndex;
                vehs[i].Description.Model        = mfrName + "_" + vehs[i].Description.Model;
                vehs[i].Description.Nameplate    = mfrName + "_" + vehs[i].Description.Nameplate;
                vehs[i].Description.Origin       = 'D';
                for (int j = 0; j < vehs[i].Successors.Count; j++)
                {
                    vehs[i].Successors[j].Description.PredecessorCode = vehIndex;
                }
                if (vehs[i].Predecessor == null) { vehs[i].Description.PredecessorCode = 0; }
                vehs[i].Successors.Clear();
                vehIndex++;
            }
            for (int i = 0, engCount = engs.Count; i < engCount; i++)
            {   
                string mfrName = engs[i].Description.Manufacturer;
                engs[i].Description.Manufacturer = "MERGED FLEET";
                engs[i].Description.Code         = engIndex;
                for (int j = 0, vehCount = engs[i].Vehicles.Count; j < vehCount; j++)
                {   
                    engs[i].Vehicles[j].Description.EngineCode = engIndex;
                }
                engIndex++;
            }
            for (int i = 0, trnCount = trns.Count; i < trnCount; i++)
            {   
                string mfrName = trns[i].Description.Manufacturer;
                trns[i].Description.Manufacturer = "MERGED FLEET";
                trns[i].Description.Code         = trnIndex;
                for (int j = 0, vehCount = trns[i].Vehicles.Count; j < vehCount; j++)
                {   
                    trns[i].Vehicles[j].Description.TransmissionCode = trnIndex;
                }
                trnIndex++;
            }
            ManufacturerDescription md  = new ManufacturerDescription();
            md.Code                     = 1;
            md.Name                     = "MERGED FLEET";
            md.CostAllocationStrategy   = 0;
            md.AverageMY2002LtCw        = 4329.48480892067;
            md.DiscountRate             = 0.07;
            md.Optimize                 = true;
            md.WillingToPayFines        = new bool  [Math.Max(15, this._maxYear.Year - ModelYear.MinYear + 1)];
            md.AvailableCreditsDA       = new double[Math.Max(15, this._maxYear.Year - ModelYear.MinYear + 1)];
            md.AvailableCreditsIA       = new double[Math.Max(15, this._maxYear.Year - ModelYear.MinYear + 1)];
            md.AvailableCreditsLT       = new double[Math.Max(15, this._maxYear.Year - ModelYear.MinYear + 1)];
            md.CreditsApplyToBaseline   = false;
            md.AvailableACCreditsDA     = new double[Math.Max(15, this._maxYear.Year - ModelYear.MinYear + 1)];
            md.AvailableACCreditsIA     = new double[Math.Max(15, this._maxYear.Year - ModelYear.MinYear + 1)];
            md.AvailableACCreditsLT     = new double[Math.Max(15, this._maxYear.Year - ModelYear.MinYear + 1)];
            md.ACCreditsApplyToBaseline = false;
            int minYear, maxYear;
            Manufacturer mfr = new Manufacturer(md);
            mfr.Initialize(vehs, engs, trns, out minYear, out maxYear);
            mfr.SetIndex(0);
            this._manufacturers.Clear();
            this._manufacturers.Add(mfr);
        }
        #endregion
        #region 
        public ManufacturerCollection Manufacturers    { get { return this._manufacturers; } }
        public int ManufacturerCount { get { return (this._manufacturers == null) ? 0 : this._manufacturers.Count; } }
        public IndustryModelingData ModelingData { get { return this._modelingData; } }
        public int VehicleCount
        {
            get
            {
                int count = 0;
                for (int i = 0; i < this._manufacturers.Count; i++)
                {
                    count += this._manufacturers[i].VehicleCount;
                }
                return count;
            }
        }
        public int ModelYearVehicleCount
        {
            get
            {
                int count = 0;
                for (int i = 0; i < this._manufacturers.Count; i++)
                {
                    count += this._manufacturers[i].ModelYearVehicleCount;
                }
                return count;
            }
        }
        public ModelYear MinYear { get { return this._minYear; } }
        public ModelYear MaxYear { get { return this._maxYear; } }
        public double MinFp { get { return this._minFp; } }
        public double MaxFp { get { return this._maxFp; } }
        public double MinCw { get { return this._minCw; } }
        public double MaxCw { get { return this._maxCw; } }
        #endregion
        #region 
        private ManufacturerCollection _manufacturers;
        private IndustryModelingData _modelingData;
        private ModelYear _minYear;
        private ModelYear _maxYear;
        [NonSerialized]
        private double _minFp;
        [NonSerialized]
        private double _maxFp;
        [NonSerialized]
        private double _minCw;
        [NonSerialized]
        private double _maxCw;
        #endregion
    }
}

