using System;
using Volpe.Cafe;
using Volpe.Cafe.Collections;
namespace Volpe.Cafe.Data
{
    [Serializable]
    public class Manufacturer : ICloneable
    {
        #region 
        private Manufacturer()
        {
        }
        public Manufacturer(ManufacturerDescription description)
        {
            this._description            = description;
            this._modelingData           = new ManufacturerModelingData();
            this._vehicles               = new VehicleCollection(32);
            this._modelYearVehicles      = new VehicleCollection(32);
            this._engines                = new EngineCollection(32);
            this._transmissions          = new TransmissionCollection(32);
            this._namePlates             = new NamePlateCollection(32);
            this._index                  = -1;
            this._initialized            = false;
            this._lastEngCode            = -1;
            this._lastTrnCode            = -1;
        }
        #endregion
        #region 
        #region 
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        public Manufacturer Clone()
        {
            Manufacturer mfr = new Manufacturer();
            mfr._index                      = this._index;
            mfr._initialized                = this._initialized;
            mfr._lastEngCode                = this._lastEngCode;
            mfr._lastTrnCode                = this._lastTrnCode;
            mfr._description  = this._description.Clone();
            mfr._modelingData = this._modelingData.Clone();
            mfr._currentYear   = this._currentYear;
            mfr._vehicles      = this._vehicles.Clone();
            mfr._engines       = this._engines.Clone();
            mfr._transmissions = this._transmissions.Clone();
            mfr._namePlates    = this._namePlates.Clone();
            Engine      [] engs = mfr._engines.ToArray();
            Transmission[] trns = mfr._transmissions.ToArray();
            NamePlate   [] npts = mfr._namePlates.ToArray();
            Vehicle     [] vehs = mfr._vehicles.ToArray();
            for (int i = 0; i < engs.Length; i++)
            {
                int parentCode = ((engs[i].Parent == null) ? -1 : engs[i].Parent.Description.Code);
                engs[i].Initialize(mfr);
                engs[i].SetParent(parentCode);
            }
            for (int i = 0; i < trns.Length; i++)
            {
                int parentCode = ((trns[i].Parent == null) ? -1 : trns[i].Parent.Description.Code);
                trns[i].Initialize(mfr);
                trns[i].SetParent(parentCode);
            }
            for (int i = 0; i < npts.Length; i++)
            {
                npts[i].Initialize(mfr);
            }
            for (int i = 0; i < vehs.Length; i++)
            {
                vehs[i].Successors.Clear();
            }
            for (int i = 0; i < vehs.Length; i++)
            {   
                vehs[i].Initialize(mfr, vehs);
                NamePlate np = mfr.GetNamePlate(vehs[i].Description.Model);
                np.Vehicles.Add(vehs[i]);
                vehs[i].SetNamePlate(np);
            }
            mfr._modelYearVehicles = new VehicleCollection(32);
            if (mfr._currentYear != null)
            {
                mfr.InitializeModelYearVehicles(mfr._currentYear);
            }
            return mfr;
        }
        #endregion
        public override string ToString()
        {
            return this._description.Name;
        }
        internal void Initialize(VehicleCollection vehicles, EngineCollection engines, TransmissionCollection transmissions,
            out int minYear, out int maxYear)
        {
            Engine[] engs = engines.ToArray();
            Transmission[] trns = transmissions.ToArray();
            Vehicle[] vehs = vehicles.ToArray();
            for (int i = 0; i < engs.Length; i++)
            {
                engs[i].Initialize(this);
                this._engines.Add(engs[i]);
                if (this._lastEngCode < engs[i].Description.Code)
                {
                    this._lastEngCode = engs[i].Description.Code;
                }
            }
            for (int i = 0; i < trns.Length; i++)
            {
                trns[i].Initialize(this);
                this._transmissions.Add(trns[i]);
                if (this._lastTrnCode < trns[i].Description.Code)
                {
                    this._lastTrnCode = trns[i].Description.Code;
                }
            }
            for (int i = 0; i < vehs.Length; i++)
            {
                vehs[i].Initialize(this, vehs);
                NamePlate np = this.GetNamePlate(vehs[i].Description.Model);
                np.Vehicles.Add(vehs[i]);
                vehs[i].SetNamePlate(np);
                this._vehicles.Add(vehs[i]);
            }
            for (int i = 0; i < vehs.Length; i++)
            {
                vehs[i].ValidatePredecessor();
            }
            VehicleCollection vc = this.Vehicles;
            minYear = int.MaxValue;
            maxYear = -1;
            for (int i = 0, vehCount = this.VehicleCount; i < vehCount; i++)
            {
                Vehicle v = vc[i];
                if (v.ValidYears != null)
                {
                    minYear = Math.Min(v.MinYear.Year, minYear);
                    maxYear = Math.Max(v.MaxYear.Year, maxYear);
                }
            }
            this._initialized = true;
        }
        private NamePlate GetNamePlate(string name)
        {
            for (int i = 0; i < this._namePlates.Count; i++)
            {
                if (this._namePlates[i].Name.Equals(name.Trim().ToUpper()))
                {    
                    return this._namePlates[i];
                }
            }
            NamePlate np = new NamePlate(name);
            np.Initialize(this);
            this._namePlates.Add(np);
            return np;
        }
        internal void SetIndex(int index)
        {
            this._index = index;
        }
        public VehicleCollection InitializeModelYearVehicles(ModelYear year)
        {
            this._currentYear = year;
            int vehCount = this._vehicles.Count;
            this._modelYearVehicles = new VehicleCollection(vehCount);
            for (int i = 0; i < vehCount; i++)
            {
                if (this._vehicles[i].IsValid(year)) { this._modelYearVehicles.Add(this._vehicles[i]); }
            }
            EngineCollection       ec  = this.Engines;
            TransmissionCollection tc  = this.Transmissions;
            NamePlateCollection    npc = this.NamePlates;
            for (int i = 0, count = ec .Count; i < count; i++) { Engine       eng = ec [i]; eng.InitializeModelYearVehicles(year); }
            for (int i = 0, count = tc .Count; i < count; i++) { Transmission trn = tc [i]; trn.InitializeModelYearVehicles(year); }
            for (int i = 0, count = npc.Count; i < count; i++) { NamePlate    np  = npc[i]; np .InitializeModelYearVehicles(year); }
            return this._modelYearVehicles;
        }
        internal int GetNextEngineCode()
        {
            return ++this._lastEngCode;
        }
        internal int GetNextTransmissionCode()
        {
            return ++this._lastTrnCode;
        }
        internal void SetModelingData(ManufacturerModelingData value)
        {
            this._modelingData = value;
        }
        #endregion
        #region 
        public int Index { get { return this._index; } }
        public bool Initialized { get { return this._initialized; } }
        public ManufacturerDescription Description { get { return this._description; } }
        public ManufacturerModelingData ModelingData { get { return this._modelingData; } }
        public VehicleCollection Vehicles { get { return this._vehicles; } }
        public VehicleCollection ModelYearVehicles { get { return this._modelYearVehicles; } }
        public int VehicleCount { get { return this._vehicles.Count; } }
        public int ModelYearVehicleCount { get { return this._modelYearVehicles.Count; } }
        public EngineCollection Engines { get { return this._engines; } }
        public TransmissionCollection Transmissions { get { return this._transmissions; } }
        public NamePlateCollection NamePlates { get { return this._namePlates; } }
        internal int CurrentYear { get { return this._currentYear.Year; } set { this._currentYear = new ModelYear(value); } }
        internal int LastEngCode { get { return this._lastEngCode; } set { this._lastEngCode = value; } }
        internal int LastTrnCode { get { return this._lastTrnCode; } set { this._lastTrnCode = value; } }
        #endregion
        #region 
        private int _index;
        private bool _initialized;
        private ManufacturerDescription _description;
        private ManufacturerModelingData _modelingData;
        private VehicleCollection _vehicles;
        private VehicleCollection _modelYearVehicles;
        private ModelYear _currentYear;
        private EngineCollection _engines;
        private TransmissionCollection _transmissions;
        private NamePlateCollection _namePlates;
        private int _lastEngCode;
        private int _lastTrnCode;
        #endregion
    }
}

