using System;
using Volpe.Cafe.Collections;
using Volpe.Cafe.IO;
namespace Volpe.Cafe.Data.MonteCarlo
{
    [Serializable]
    public struct TrialData : ICloneable
    {
        #region 
        public TrialData(TrialInfo info)
        {
            this.TrialInfo           = info;
            this.HasData             = false;
            this.MfrAvePriceIncrease = null;
            this.MfrTechCost         = null;
            this.TechCost            = null;
            this.SocialBenefits      = null;
            this.FuelSavings         = null;
            this.BCRatio             = null;
        }
        #endregion
        #region 
        #region 
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        public TrialData Clone()
        {
            TrialData td = new TrialData();
            td.TrialInfo = this.TrialInfo.Clone();
            if (this.MfrAvePriceIncrease != null)
            {
                int count = this.MfrAvePriceIncrease.Length;
                td.MfrAvePriceIncrease = new double[count][];
                for (int i = 0; i < count; i++)
                {
                    td.MfrAvePriceIncrease[i] = (double[])Global.CloneArray(this.MfrAvePriceIncrease[i], typeof(double));
                }
            }
            if (this.MfrTechCost != null)
            {
                int count = this.MfrTechCost.Length;
                td.MfrTechCost = new double[count][];
                for (int i = 0; i < count; i++)
                {
                    td.MfrTechCost[i] = (double[])Global.CloneArray(this.MfrTechCost[i], typeof(double));
                }
            }
            td.TechCost       = (double[])Global.CloneArray(this.TechCost      , typeof(double));
            td.SocialBenefits = (double[])Global.CloneArray(this.SocialBenefits, typeof(double));
            td.FuelSavings    = (double[])Global.CloneArray(this.FuelSavings   , typeof(double));
            td.BCRatio        = (double[])Global.CloneArray(this.BCRatio       , typeof(double));
            return td;
        }
        #endregion
        public void SaveTrialData(Scenario scenario, int minYearIndex, int yearCount, TrialData baseTd, LogWriter writer,
            Industry[] modelData)
        {
            bool isBaseline = scenario.IsBaseline;
            this.MfrAvePriceIncrease = new double[yearCount][];
            this.MfrTechCost         = new double[yearCount][];
            this.TechCost            = new double[yearCount];
            this.SocialBenefits      = new double[yearCount];
            this.FuelSavings         = new double[yearCount];
            this.BCRatio             = new double[yearCount];
            for (int j = 0; j < yearCount; j++)
            {
                Industry ind = modelData[minYearIndex + j];
                int mfrCount = ind.ManufacturerCount;
                this.MfrAvePriceIncrease[j] = new double[mfrCount];
                this.MfrTechCost[j] = new double[mfrCount];
                for (int k = 0; k < mfrCount; k++)
                {   
                    ManufacturerModelingData mmd = ind.Manufacturers[k].ModelingData;
                    this.MfrAvePriceIncrease[j][k] = mmd.RegCost.TotalAll / mmd.Sales.TotalAll;
                    this.MfrTechCost        [j][k] = mmd.TechCost.TotalAll * 0.000001;  
                    if (!isBaseline)
                    {   
                        this.MfrAvePriceIncrease[j][k] -= baseTd.MfrAvePriceIncrease[j][k];
                        this.MfrTechCost        [j][k] -= baseTd.MfrTechCost[j][k];
                    }
                    this.TechCost[j] += this.MfrTechCost[j][k];
                }
                AggregateEffectsData ied = ind.ModelingData.EffectsData;
                this.SocialBenefits[j]   = ied.RcDiscSocialBenefits.TotalAll * 0.001;   
                this.FuelSavings[j]      = ied.RcGallons.TotalAll;
                if (!isBaseline)
                {   
                    this.SocialBenefits[j] = baseTd.SocialBenefits[j] - this.SocialBenefits[j];
                    this.FuelSavings   [j] = baseTd.FuelSavings   [j] - this.FuelSavings[j];
                    this.BCRatio       [j] = this.SocialBenefits  [j] / this.TechCost[j];
                }
            }
            this.HasData = true;
        }
        public static string CreateCsvHeader(Industry data, bool includeTrialInfo)
        {
            ManufacturerCollection mc = data.Manufacturers;
            int mfrCount = mc.Count;
            int minYear = data.MinYear.Year;
            int maxYear = data.MaxYear.Year;
            string mfrAp = "", mfrTc = "", tc = "", sb = "", fs = "", bcr = "";
            for (int i = minYear; i <= maxYear; i++)
            {
                for (int j = 0; j < mfrCount; j++)
                {
                    mfrAp += "AvePrice_" + mc[j].Description.Name + "(" + i + "),";
                    mfrTc += "TechCost_" + mc[j].Description.Name + "(" + i + "),";
                }
                tc  += "TechCost(" + i + "),";
                sb  += "SocialBenefits(" + i + "),";
                fs  += "FuelSavings(" + i + "),";
                bcr += "BCRatio(" + i + "),";
            }
            return ((includeTrialInfo) ? TrialInfo.CreateCsvHeader() + "," : "Index,DiscountRate,") +
                mfrAp + mfrTc + tc + sb + fs + bcr.Remove(bcr.Length - 1, 1);
        }
        public string ToCsvString(bool includeTrialInfo)
        {
            return ((includeTrialInfo) ? this.TrialInfo.ToCsvString() :
                                         (this.TrialInfo.Index + "," + this.TrialInfo.DiscountRate)) + "," +
                this.ArrToCsv(this.MfrAvePriceIncrease) + "," +
                this.ArrToCsv(this.MfrTechCost) + "," +
                this.ArrToCsv(this.TechCost) + "," +
                this.ArrToCsv(this.SocialBenefits) + "," +
                this.ArrToCsv(this.FuelSavings) + "," +
                this.ArrToCsv(this.BCRatio);
        }
        private string ArrToCsv(double[] arr)
        {
            string result = "";
            for (int i = 0, count = arr.Length; i < count; i++)
            {
                result += arr[i] + ",";
            }
            return result.Remove(result.Length - 1, 1);
        }
        private string ArrToCsv(double[][] arr)
        {
            string result = "";
            for (int i = 0, count = arr.Length; i < count; i++)
            {
                result += this.ArrToCsv(arr[i]) + ",";
            }
            return result.Remove(result.Length - 1, 1);
        }
        #endregion
        #region 
        public bool HasData;
        public TrialInfo TrialInfo;
        public double[][] MfrAvePriceIncrease;
        public double[][] MfrTechCost;
        public double[] TechCost;
        public double[] SocialBenefits;
        public double[] FuelSavings;
        public double[] BCRatio;
        #endregion
    }
}

