using System;
using System.Runtime.Serialization;
using Volpe.Cafe.Collections;
using Volpe.Cafe.Settings;
namespace Volpe.Cafe.Data.MonteCarlo
{
    [Serializable]
    public struct TrialInfo : ICloneable, ISerializable
    {
        #region 
        TrialInfo(SerializationInfo info, StreamingContext context)
        {
            this.Index                = info.GetInt32 ("Index");
            this.FuelPriceEstimates   = (Estimates)info.GetValue("FuelPriceEstimates", typeof(Estimates));
            this.DiscountRate         = info.GetDouble("DiscountRate");
            this.ReboundEffect        = info.GetDouble("ReboundEffect");
            this.MonopsonyCost        = info.GetDouble("MonopsonyCost");
            this.PriceShockCost       = info.GetDouble("PriceShockCost");
            this.MilitarySecurityCost = info.GetDouble("MilitarySecurityCost");
            this.TechCostScaleFactor  = (double[])info.GetValue("TechCostScaleFactor", typeof(double[]));
            this.TechFcScaleFactor    = (double[])info.GetValue("TechFcScaleFactor", typeof(double[]));
            try
            {
                this.CO2Estimates = (CO2Estimates)info.GetValue("CO2Estimates", typeof(CO2Estimates));
            }
            catch
            {
                this.CO2Estimates = CO2Estimates.Average;
            }
        }
        #endregion
        #region 
        #region 
        public void GetObjectData(SerializationInfo info, StreamingContext context)
        {
            info.AddValue("Index"               , this.Index);
            info.AddValue("FuelPriceEstimates"  , this.FuelPriceEstimates);
            info.AddValue("DiscountRate"        , this.DiscountRate);
            info.AddValue("ReboundEffect"       , this.ReboundEffect);
            info.AddValue("CO2Estimates"        , this.CO2Estimates);
            info.AddValue("MonopsonyCost"       , this.MonopsonyCost);
            info.AddValue("PriceShockCost"      , this.PriceShockCost);
            info.AddValue("MilitarySecurityCost", this.MilitarySecurityCost);
            info.AddValue("TechCostScaleFactor" , this.TechCostScaleFactor);
            info.AddValue("TechFcScaleFactor"   , this.TechFcScaleFactor);
        }
        #endregion
        #region 
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        public TrialInfo Clone()
        {
            TrialInfo ti = new TrialInfo();
            ti.Index                = this.Index;
            ti.FuelPriceEstimates   = this.FuelPriceEstimates;
            ti.DiscountRate         = this.DiscountRate;
            ti.ReboundEffect        = this.ReboundEffect;
            ti.CO2Estimates         = this.CO2Estimates;
            ti.MonopsonyCost        = this.MonopsonyCost;
            ti.PriceShockCost       = this.PriceShockCost;
            ti.MilitarySecurityCost = this.MilitarySecurityCost;
            ti.TechCostScaleFactor  = (double[])Global.CloneArray(this.TechCostScaleFactor, typeof(double));
            ti.TechFcScaleFactor    = (double[])Global.CloneArray(this.TechFcScaleFactor  , typeof(double));
            return ti;
        }
        #endregion
        public void InitializeTrial(Industry data, ModelingSettings settings)
        {
            settings.OperatingModes.FuelPriceEstimates = this.FuelPriceEstimates;
            settings.OperatingModes.CO2Estimates       = this.CO2Estimates;
            ManufacturerCollection mfrs = data.Manufacturers;
            for (int j = 0, mfrCount = data.ManufacturerCount; j < mfrCount; j++)
            {
                mfrs[j].Description.DiscountRate = this.DiscountRate;
            }
            settings.Parameters.UpdateForMonteCarlo(this.DiscountRate, this.ReboundEffect, this.MonopsonyCost,
                this.PriceShockCost, this.MilitarySecurityCost);
            TechnologyCollection techs = settings.Technologies;
            double[] costScale = this.TechCostScaleFactor;
            double[] fcScale   = this.TechFcScaleFactor;
            for (int i = 0, techCount = techs.Count; i < techCount; i++)
            {
                Technology tech = techs[i];
                TechnologyAttributes[] techAttr = tech.Attributes.ToArray();
                int techAttrCount = techAttr.Length;
                for (int j = 0; j < techAttrCount; j++)
                {
                    techAttr[j].CostHigh *= costScale[i];
                    techAttr[j].CostLow  *= costScale[i];
                    techAttr[j].FcHigh   *= fcScale[i];
                    techAttr[j].FcLow    *= fcScale[i];
                } 
            } 
        }
        public static string CreateCsvHeader()
        {
            string cost = "", fc = "";
            for (int i = 0; i < TechnologyIndexes.TechnologyCount; i++)
            {
                cost += "Cost(" + TechnologyIndexes.TechAbbr[i] + "),";
                fc   += "FC("   + TechnologyIndexes.TechAbbr[i] + "),";
            }
            return "Index,FuelPriceEstimates,DiscountRate,ReboundEffect,CO2Estimates,MonopsonyCost,PriceShockCost," +
                "MilitarySecurityCost," + cost + fc.Remove(fc.Length - 1, 1);
        }
        public string ToCsvString()
        {
            return
                this.Index                + "," +
                this.FuelPriceEstimates   + "," +
                this.DiscountRate         + "," +
                this.ReboundEffect        + "," +
                this.CO2Estimates         + "," +
                this.MonopsonyCost        + "," +
                this.PriceShockCost       + "," +
                this.MilitarySecurityCost + "," +
                this.ArrToCsv(this.TechCostScaleFactor) + "," +
                this.ArrToCsv(this.TechFcScaleFactor);
        }
        public void FromCsvString(string value)
        {
            string[] parts = value.Split(',');
            this.Index                = int.Parse(parts[0]);
            this.FuelPriceEstimates   =
                (Global.StringCompare(parts[1], "Low" , false)) ? Estimates.Low  :
                (Global.StringCompare(parts[1], "High", false)) ? Estimates.High : Estimates.Average;
            this.DiscountRate         = double.Parse(parts[2]);
            this.ReboundEffect        = double.Parse(parts[3]);
            this.CO2Estimates         =
                (Global.StringCompare(parts[4], "Low"     , false)) ? CO2Estimates.Low      :
                (Global.StringCompare(parts[4], "High"    , false)) ? CO2Estimates.High     :
                (Global.StringCompare(parts[4], "VeryHigh", false)) ? CO2Estimates.VeryHigh : CO2Estimates.Average;
            this.MonopsonyCost        = double.Parse(parts[5]);
            this.PriceShockCost       = double.Parse(parts[6]);
            this.MilitarySecurityCost = double.Parse(parts[7]);
            this.TechCostScaleFactor  = new double[TechnologyIndexes.TechnologyCount];
            this.TechFcScaleFactor    = new double[TechnologyIndexes.TechnologyCount];
            for (int i = 0; i < TechnologyIndexes.TechnologyCount; i++)
            {   
                this.TechCostScaleFactor[i] = double.Parse(parts[8 + i]);
                this.TechFcScaleFactor  [i] = double.Parse(parts[8 + TechnologyIndexes.TechnologyCount + i]);
            }
        }
        string ArrToCsv(double[] arr)
        {
            string result = "";
            for (int i = 0, count = arr.Length; i < count; i++)
            {
                result += arr[i] + ",";
            }
            return result.Remove(result.Length - 1, 1);
        }
        #endregion
        #region 
        public int Index;
        public Estimates FuelPriceEstimates;
        public double DiscountRate;
        public double ReboundEffect;
        public CO2Estimates CO2Estimates;
        public double MonopsonyCost;
        public double PriceShockCost;
        public double MilitarySecurityCost;
        public double[] TechCostScaleFactor;
        public double[] TechFcScaleFactor;
        #endregion
    }
}

