using java.io;
using java.util.zip;
using System;
using System.IO;
using System.Collections;
namespace Volpe.Utils
{
    [Serializable]
    public class ZipUtilities
    {
        #region 
        public ZipUtilities(ZipFormat format)
            : this(format, ZipLevel.Default)
        {
        }
        public ZipUtilities(ZipFormat format, ZipLevel level)
            : this(format, ZipMethod.Deflated, ZipStrategy.Default, level)
        {
        }
        public ZipUtilities(ZipFormat format, ZipMethod method, ZipStrategy strategy)
            : this(format, method, strategy, ZipLevel.Default)
        {
        }
        public ZipUtilities(ZipFormat format, ZipMethod method, ZipStrategy strategy, ZipLevel level)
        {
            this.Format   = format;
            this.Method   = method;
            this.Strategy = strategy;
            this.Level    = level;
            this._bufferSize = 4096;
        }
        #endregion
        #region 
        #region 
        public void Compress(string file)
        {
            this.Compress(new string[] {file});
        }
        public void Compress(string file, string outPath)
        {
            this.Compress(new string[] {file}, outPath);
        }
        public void Compress(string[] files)
        {
            string dirName = Path.GetDirectoryName(files[0]);
            string outPath = dirName + "\\" + ((files.Length == 1) ?
                Path.GetFileNameWithoutExtension(files[0]) :
                Path.GetFileName(dirName)) + this._outExtension;
            this.Compress(files, outPath);
        }
        public void Compress(string[] files, string outPath)
        {
            int fileCount = files.Length;
            Stream[] streams = new Stream[fileCount];
            string[] streamNames = new string[fileCount];
            for (int i = 0; i < fileCount; i++)
            {
                streams[i] = new FileStream(files[i], FileMode.Open, FileAccess.Read, FileShare.Read);
                streamNames[i] = Path.GetFileName(files[i]);
            }
            this.Compress(streams, streamNames, outPath);
            for (int i = 0; i < fileCount; i++)
            {
                streams[i].Close();
            }
        }
        public void Compress(Stream stream, string streamName, string outPath)
        {
            this.Compress(new Stream[] {stream}, new string[] {streamName}, outPath);
        }
        public void Compress(Stream[] streams, string[] streamNames, string outPath)
        {
            if (streams == null || streamNames == null)
            {
                throw new ArgumentNullException("streams or streamNames", "The streams or streamNames specified is null");
            }
            if (streams.Length != streamNames.Length)
            {
                throw new ArgumentException("The number of stream names specified does not match the number of streams provided.");
            }
            if      (this._format == ZipFormat.Zip ) { this.CompressZip (streams, streamNames, outPath); }
            else if (this._format == ZipFormat.GZip) { this.CompressGZip(streams, streamNames, outPath); }
            else                  { throw new DataFormatException("The ZipFormat provided is unknown."); }
        }
        private void CompressZip(Stream[] streams, string[] streamNames, string outPath)
        {
            ZipOutputStream zip = new ZipOutputStream(new FileOutputStream(outPath));
            zip.setLevel((int)this._level);
            zip.setMethod((int)this._method);
            zip.def.setStrategy((int)this._strategy);
            byte[] buffer = new byte[this._bufferSize];
            int streamCount = streams.Length;
            for (int i = 0; i < streamCount; i++)
            {
                zip.putNextEntry(new ZipEntry(streamNames[i]));
                int bytesRead = 0;
                do  
                {
                    bytesRead = streams[i].Read(buffer, 0, this._bufferSize);
                    zip.write(this.ToSByte(buffer, bytesRead), 0, bytesRead);
                } while (bytesRead > 0);
                zip.closeEntry();
            }
            zip.close();
        }
        private void CompressGZip(Stream[] streams, string[] streamNames, string outPath)
        {
            string   path   = null;
            Deflater def    = this.Deflater;
            byte[]   buffer = new byte[this._bufferSize];
            int streamCount = streams.Length;
            for (int i = 0; i < streamCount; i++)
            {   
                path = (streamCount == 1) ? outPath : Path.GetDirectoryName(outPath) + "\\" + streamNames[i] + this._outExtension;
                GZIPOutputStream gzip = new GZIPOutputStream(new FileOutputStream(path));
                gzip.def = this.Deflater;
                int bytesRead = 0;
                do  
                {
                    gzip.write(this.ToSByte(buffer, bytesRead), 0, bytesRead);
                    bytesRead = streams[i].Read(buffer, 0, this._bufferSize);
                } while (bytesRead > 0);
                gzip.finish();
                gzip.close();
            }
        }
        #endregion
        #region 
        public void Decompress(string file)
        {
            string outPath = Path.GetDirectoryName(file) + "\\" + Path.GetFileNameWithoutExtension(file);
            this.Decompress(file, outPath);
        }
        public void Decompress(Stream stream, string outPath)
        {
            Stream[] outStreams = null;
            try
            {
                this.Decompress(stream, out outStreams, false, outPath);
            }
            catch (Exception ex)
            {
                throw ex;   
            }
            finally
            {
                if (outStreams != null)
                {
                    int streamCount = outStreams.Length;
                    for (int i = 0; i < streamCount; i++)
                    {
                        if (outStreams[i] != null) { outStreams[i].Close(); }
                    }
                }
            } 
        }
        public void Decompress(string file, string outPath)
        {
            Stream[] outStreams = null;
            try
            {
                this.Decompress(file, out outStreams, false, outPath);
            }
            catch (Exception ex)
            {
                throw ex;   
            }
            finally
            {
                if (outStreams != null)
                {
                    int streamCount = outStreams.Length;
                    for (int i = 0; i < streamCount; i++)
                    {
                        if (outStreams[i] != null)
                        {
                            outStreams[i].Close();
                        }
                    }
                }
            } 
        }
        public void Decompress(Stream stream, out Stream[] outStreams, bool decompressToMemory, string outPath)
        {
            stream.Seek(0, SeekOrigin.Begin);
            byte[] buffer = new byte[32768];
            string tmpFile = Path.GetTempFileName();
            FileStream tmpStream = null;
            try
            {
                tmpStream = new FileStream(tmpFile, FileMode.Open, FileAccess.Write, FileShare.None);
                int bytesRead = 0;
                do
                {
                    tmpStream.Write(buffer, 0, bytesRead);
                    bytesRead = stream.Read(buffer, 0, 32768);
                } while (bytesRead > 0);
                tmpStream.Close();
                tmpStream = null;
                this.Decompress(tmpFile, out outStreams, decompressToMemory, outPath);
            }
            catch (Exception ex)
            {
                throw ex;    
            }
            finally
            {   
                if (tmpStream != null) { tmpStream.Close(); }
                System.IO.File.Delete(tmpFile);         
            }
        }
        public void Decompress(string file, out Stream[] outStreams, bool decompressToMemory, string outPath)
        {
            if (this._format == ZipFormat.Zip)
            {
                this.DecompressZip(file, out outStreams, decompressToMemory, outPath);
            }
            else if (this._format == ZipFormat.GZip)
            {
                outStreams = new Stream[1];
                outStreams[0] = (decompressToMemory) ? (Stream)new MemoryStream() :
                    (Stream)new FileStream(outPath, FileMode.Create, FileAccess.Write, FileShare.None);
                this.DecompressGZip(file, ref outStreams[0]);
            }
            else
            {
                throw new DataFormatException("The ZipFormat provided is unknown.");
            }
        }
        private void DecompressZip(string file, out Stream[] outStreams, bool decompressToMemory, string outPath)
        {
            ZipInputStream zip = new ZipInputStream(new FileInputStream(file));
            ArrayList streams = new ArrayList();
            Stream stream;
            ZipEntry entry = null;
            do
            {
                entry = zip.getNextEntry();
                if (entry == null) { break; }
                stream = (decompressToMemory) ? (Stream)new MemoryStream() :
                    (Stream)new FileStream(outPath + "\\" + entry.getName(), FileMode.Create, FileAccess.Write, FileShare.None);
                sbyte[] buffer = new sbyte[this._bufferSize];
                int bytesRead = 0;
                do  
                {
                    stream.Write(this.FromSByte(buffer, bytesRead), 0, bytesRead);
                    bytesRead = zip.read(buffer, 0, this._bufferSize);
                } while (bytesRead > 0);
                zip.closeEntry();
                streams.Add(stream);
            } while (true);
            int streamCount = streams.Count;
            outStreams = new Stream[streamCount];
            for (int i = 0; i < streamCount; i++)
            {
                outStreams[i] = (Stream)streams[i];
            }
            zip.close();
        }
        private void DecompressGZip(string file, ref Stream outStream)
        {
            GZIPInputStream gzip = new GZIPInputStream(new FileInputStream(file));
            sbyte[] buffer = new sbyte[this._bufferSize];
            int bytesRead = 0;
            do  
            {
                outStream.Write(this.FromSByte(buffer, bytesRead), 0, bytesRead);
                bytesRead = gzip.read(buffer, 0, this._bufferSize);
            } while (bytesRead > 0);
            gzip.close();
        }
        #endregion
        private sbyte[] ToSByte(byte[] buffer, int length)
        {
            sbyte[] buf = new sbyte[length];
            for (int i = 0; i < length; i++)
            {
                buf[i] = (sbyte)buffer[i];
            }
            return buf;
        }
        private byte[] FromSByte(sbyte[] buffer, int length)
        {
            byte[] buf = new byte[length];
            for (int i = 0; i < length; i++)
            {
                buf[i] = (byte)buffer[i];
            }
            return buf;
        }
        #endregion
        #region 
        public Deflater Deflater
        {
            get
            {
                Deflater def = new Deflater((int)this._level, (this._format == ZipFormat.GZip));
                def.setStrategy((int)this._strategy);
                return def;
            }
        }
        public string OutputExtension
        {
            get { return this._outExtension; }
        }
        public ZipFormat Format
        {
            get { return this._format; }
            set
            {
                this._format = value;
                this._outExtension = (value == ZipFormat.GZip) ? ".gz" : (value == ZipFormat.Zip) ? ".zip" : "";
            }
        }
        public ZipMethod Method
        {
            get { return this._method; }
            set { this._method = value; }
        }
        public ZipStrategy Strategy
        {
            get { return this._strategy; }
            set { this._strategy = value; }
        }
        public ZipLevel Level
        {
            get { return this._level; }
            set { this._level = ((value < ZipLevel.Level0 || value > ZipLevel.Level9) ? ZipLevel.Default : value); }
        }
        #endregion
        #region 
        private ZipFormat   _format;
        private ZipMethod   _method;
        private ZipStrategy _strategy;
        private ZipLevel    _level;
        private string _outExtension;
        private int    _bufferSize;
        #endregion
    }
}

