#region << Using Directives >>
using System;
using System.Collections;
using Volpe.Cafe;
using Volpe.Cafe.Collections;
using Volpe.Cafe.Data;
using RC = Volpe.Cafe.RegulatoryClass;
using VT = Volpe.Cafe.VehicleType;
#endregion
namespace Volpe.Cafe.Model
{
    public sealed class Standards
    {
        #region 
        #region 
        public static RCDouble GetIndustryStandard(Industry data, Scenario scen, ModelYear year, bool calcUnregulated)
        {
            RCDouble standard = new RCDouble();
            standard.DomesticAuto = Standards.GetIndustryStandard(data, scen, year, RC.DomesticAuto);
            standard.ImportedAuto = Standards.GetIndustryStandard(data, scen, year, RC.ImportedAuto);
            standard.LightTruck   = Standards.GetIndustryStandard(data, scen, year, RC.LightTruck);
            if (calcUnregulated) { standard.Unregulated = Standards.GetIndustryStandard(data, scen, year, RC.Unregulated); }
            return standard;
        }
        public static double GetIndustryStandard(Industry data, Scenario scen, ModelYear year, RC regClass)
        {
            ManufacturerCollection mfrs = data.Manufacturers;
            double standardSum = 0, salesSum    = 0;
            for (int i = 0, mfrCount = mfrs.Count; i < mfrCount; i++)
            {   
                Manufacturer manufacturer = mfrs[i];
                double mfrSales = Standards.GetSales   (manufacturer,       year, regClass);
                double mfrStnd  = Standards.GetStandard(manufacturer, scen, year, regClass, null, 0);
                standardSum += (mfrSales == 0 || mfrStnd == 0) ? 0 : mfrSales / mfrStnd;
                salesSum    += mfrSales;
            }
            return salesSum / standardSum;
        }
        public static RCDouble GetStandard(Manufacturer mfr, Scenario scen, ModelYear year)
        {
            RCDouble standard = new RCDouble();
            standard.DomesticAuto = Standards.GetStandard(mfr, scen, year, RC.DomesticAuto);
            standard.ImportedAuto = Standards.GetStandard(mfr, scen, year, RC.ImportedAuto);
            standard.LightTruck   = Standards.GetStandard(mfr, scen, year, RC.LightTruck);
            standard.Unregulated  = Standards.GetStandard(mfr, scen, year, RC.Unregulated);
            return standard;
        }
        public static double GetStandard(Manufacturer mfr, Scenario scen, ModelYear year, RC regClass)
        {
            return Standards.GetStandard(mfr, scen, year, regClass, null, 0);
        }
        public static double GetStandard(Manufacturer mfr, Scenario scen, ModelYear year, RC regClass, TechImpact[] techImpacts,
            int techImpactCount)
        {
            if (techImpactCount > 0 && (techImpacts == null || techImpacts.Length < techImpactCount))
            {   
                throw new ArgumentOutOfRangeException("techImpacts", "techImpacts is null or less than techImpactCount.");
            }
            ManufacturerModelingData mmd = mfr.ModelingData;
            double standard = 0;
            int yrIndex = year.Index;
            int fncType;
            double[,] fncCoeff;
            int numCoeff = Standards.GetCoefficients(regClass, scen, year, out fncType, out fncCoeff);
            if (fncType == 0 || fncCoeff == null || regClass == RC.Unregulated)
            {   
                standard = 0;
            }
            else if (fncType == 1)
            {   
                standard = fncCoeff[0, yrIndex];
            }
            else if (fncType == 99)
            {   
                standard = mmd.DefaultStandard[yrIndex][regClass];
            }
            else
            {   
                double prevStandard = mmd.PreliminaryStandard[regClass];
                bool noPrevStandard = (prevStandard == 0 || double.IsNaN(prevStandard));
                bool areaSystem     = FunctionInformation.IsAreaBasedFunction(fncType);
                bool areaChanged    = noPrevStandard, weightChanged = noPrevStandard;
                if (!noPrevStandard)
                {
                    if (areaSystem)
                    {   
                    }
                    else
                    {   
                        for (int i = 0; i < techImpactCount; i++)
                        {
                            if (!techImpacts[i].Ignore && techImpacts[i].DeltaCurbWeight[yrIndex] != 0) { weightChanged = true; break; }
                        }
                    }
                }
                if (areaChanged || weightChanged)
                {   
                    double sales  = 0;      
                    double stdSum = 0;      
                    VehicleCollection vehs = mfr.ModelYearVehicles;
                    for (int i = 0, vehCount = vehs.Count; i < vehCount; i++)
                    {
                        Vehicle veh = vehs[i];
                        if (veh.RegClass == regClass)
                        {
                            VehicleDescription vd = veh.Description;
                            double target = 0;
                            int idx = -1;
                            for (int j = 0; j < techImpactCount; j++)
                            {
                                if (!techImpacts[j].Ignore && veh == techImpacts[j].Vehicle[yrIndex]) { idx = j; break; }
                            }
                            target = Standards.GetTarget(veh, scen, year, (idx == -1) ? double.NaN : vd.Footprint,
                                (idx == -1) ? double.NaN : vd.CurbWeight + techImpacts[idx].DeltaCurbWeight[yrIndex], fncType, fncCoeff);
                            sales  += vd.Sales[yrIndex];
                            stdSum += vd.Sales[yrIndex] * target;
                        } 
                    } 
                    standard = sales / stdSum;
                }
                else
                {   
                    standard = prevStandard;
                }
            }
            mmd.PreliminaryStandard[regClass] = Math.Round(standard, 1);
            if (regClass == RC.LightTruck)
            {
                if (standard < scen.LtMinStndMpg[yrIndex]) { standard = scen.LtMinStndMpg[yrIndex]; }
                double pctIndStnd = scen.LtMinStndPct[yrIndex] * mmd.AverageStandard.LightTruck;
                if (standard < pctIndStnd) { standard = pctIndStnd; }
            }
            else if (regClass == RC.DomesticAuto || regClass == RC.ImportedAuto)
            {
                if (standard < scen.AutoMinStndMpg[yrIndex]) { standard = scen.AutoMinStndMpg[yrIndex]; }
                double pctIndStnd = scen.AutoMinStndPct[yrIndex] * mmd.AverageStandard[regClass];
                if (standard < pctIndStnd) { standard = pctIndStnd; }
            }
            return Math.Round(standard, 1);
        }
        public static double GetTarget(Vehicle vehicle, Scenario scen, ModelYear year)
        {
            return Standards.GetTarget(vehicle, scen, year, double.NaN, double.NaN, 0, null);
        }
        public static double GetTarget(Vehicle vehicle, Scenario scen, ModelYear year, double area, double weight, int fncType,
            double[,] fncCoeff)
        {
            if (double.IsNaN(area  ) || area   <= 0) { area   = vehicle.Description.Footprint;  }
            if (double.IsNaN(weight) || weight <= 0) { weight = vehicle.Description.CurbWeight; }
            if (fncType == 0 || fncCoeff == null)
            {   
                Standards.GetCoefficients(vehicle.RegClass, scen, year, out fncType, out fncCoeff);
            }
            return Standards.GetTarget(area, weight, fncType, fncCoeff, year.Index);
        }
        public static double GetTarget(double area, double weight, int fncType, double[] fncCoeff)
        {
            return Standards.GetTarget(area, weight, fncType, fncCoeff[0], fncCoeff[1], fncCoeff[2], fncCoeff[3]);
        }
        public static double GetTarget(double area, double weight, int fncType, double[,] fncCoeff, int yrIndex)
        {
            return Standards.GetTarget(area, weight, fncType, fncCoeff[0, yrIndex], fncCoeff[1, yrIndex], fncCoeff[2, yrIndex],
                fncCoeff[3, yrIndex]);
        }
        public static double GetTarget(double area, double weight, int fncType, double A, double B, double C, double D)
        {
            double target = 0.0, invA = 1.0 / A, invB = 1.0 / B;
            double exp;
            switch (fncType)
            {
                case 1:
                    target = invA;
                    break;
                case 2:     
                case 4:     
                    exp    = Math.Exp((((fncType == 2) ? area : weight) - C) / D);
                    target = invA + (invB - invA) * (exp / (1.0 + exp));
                    break;
                case 3:     
                case 5:     
                    exp    = Math.Exp(1.0 - ((fncType == 3) ? area : weight) / C);
                    target = invA - exp * invB;
                    break;
                case 6:     
                case 7:     
                    target = C * ((fncType == 6) ? area : weight) + D;                  
                    target = (target < invA) ? invA : (target > invB) ? invB : target;  
                    break;
                case 8:     
                case 9:     
                    target = C * ((fncType == 8) ? area : weight) + D;                  
                    target = (target > invB) ? invB : target;                           
                    break;
            }
            return target;
        }
        public static RC GetRegClass(Vehicle vehicle, Scenario scen, ModelYear year)
        {
            VT vehType = vehicle.VehicleType;
            if (vehType == VT.None) { return RC.Unregulated; }
            else
            {   
                string merging = scen.RegulatoryMerging[year.Index].ToUpper();
                bool   isRegLt = (vehicle.Description.RegulatoryIndicator == "LT"); 
                if      (isRegLt                                || merging == "RC3") { return RC.LightTruck;   }
                else if ((vehType & VT.Domestic) == VT.Domestic || merging == "RC1") { return RC.DomesticAuto; }
                else if ((vehType & VT.Imported) == VT.Imported)                     { return RC.ImportedAuto; }
                else                                                                 { return RC.Unregulated;  }
            }
        }
        public static bool IsSupportedFunction(int fncType)
        {
            return FunctionInformation.IsSupportedFunction(fncType);
        }
        public static bool IsFlatStandard(int fncType)
        {
            return FunctionInformation.IsFlatStandard(fncType);
        }
        public static bool IsAreaBasedFunction(int fncType)
        {
            return FunctionInformation.IsAreaBasedFunction(fncType);
        }
        public static bool IsWeightBasedFunction(int fncType)
        {
            return FunctionInformation.IsWeightBasedFunction(fncType);
        }
        #endregion
        #region 
        public static RCDouble GetSales(Manufacturer mfr, ModelYear year)
        {
            RCDouble sales = new RCDouble();
            sales.DomesticAuto = Standards.GetSales(mfr, year, RC.DomesticAuto);
            sales.ImportedAuto = Standards.GetSales(mfr, year, RC.ImportedAuto);
            sales.LightTruck   = Standards.GetSales(mfr, year, RC.LightTruck);
            sales.Unregulated  = Standards.GetSales(mfr, year, RC.Unregulated);
            return sales;
        }
        public static double GetSales(Manufacturer mfr, ModelYear year, RC regClass)
        {
            double sales = 0;
            int yrIndex = year.Index;
            VehicleCollection vehs = mfr.ModelYearVehicles;
            for (int i = 0, vehCount = vehs.Count; i < vehCount; i++)
            {   
                if (vehs[i].RegClass == regClass)
                {
                    sales += vehs[i].Description.Sales[yrIndex];
                }
            }
            return sales;
        }
        #endregion
        #region 
        public static RCDouble GetFines(Manufacturer mfr, Parameters parameters)
        {
            RCDouble fines = new RCDouble();
            fines.DomesticAuto = Standards.GetFines(mfr, parameters, RC.DomesticAuto);
            fines.ImportedAuto = Standards.GetFines(mfr, parameters, RC.ImportedAuto);
            fines.LightTruck   = Standards.GetFines(mfr, parameters, RC.LightTruck  );
            fines.Unregulated  = Standards.GetFines(mfr, parameters, RC.Unregulated );
            return fines;
        }
        public static double GetFines(Manufacturer mfr, Parameters parameters, RC regClass)
        {
            ManufacturerModelingData mmd = mfr.ModelingData;
            double credits  = mmd.Credits[regClass];            
            double fineRate = parameters.EconomicValues.Kf;     
            return (double.IsNaN(credits)) ? 0 : -fineRate * Math.Min(Math.Round(credits, 1), 0);
        }
        public static RCDouble TestFines(Manufacturer mfr, Scenario scen, ModelYear year, Parameters parameters,
            RCDouble newCafe, TechImpact[] techImpacts, int techImpactCount)
        {
            RCDouble fines = new RCDouble();
            fines.DomesticAuto = Standards.TestFines(mfr, scen, year, parameters, RC.DomesticAuto, newCafe.DomesticAuto,
                techImpacts, techImpactCount);
            fines.ImportedAuto = Standards.TestFines(mfr, scen, year, parameters, RC.ImportedAuto, newCafe.ImportedAuto,
                techImpacts, techImpactCount);
            fines.LightTruck   = Standards.TestFines(mfr, scen, year, parameters, RC.LightTruck  , newCafe.LightTruck  ,
                techImpacts, techImpactCount);
            fines.Unregulated  = Standards.TestFines(mfr, scen, year, parameters, RC.Unregulated , newCafe.Unregulated ,
                techImpacts, techImpactCount);
            return fines;
        }
        public static double TestFines(Manufacturer mfr, Scenario scen, ModelYear year, Parameters parameters, RC regClass,
            double newCafe, TechImpact[] techImpacts, int techImpactCount)
        {
            double credits = Standards.TestCredits(mfr, scen, year, parameters, regClass, newCafe, techImpacts, techImpactCount);
            double fineRate = parameters.EconomicValues.Kf;
            return -fineRate * Math.Min(credits, 0);
        }
        #endregion
        #region 
        public static RCDouble GetCredits(Manufacturer mfr, Scenario scen, ModelYear year, Parameters parameters)
        {
            RCDouble credits = new RCDouble();
            credits.DomesticAuto = Standards.GetCredits(mfr, scen, year, parameters, RC.DomesticAuto);
            credits.ImportedAuto = Standards.GetCredits(mfr, scen, year, parameters, RC.ImportedAuto);
            credits.LightTruck   = Standards.GetCredits(mfr, scen, year, parameters, RC.LightTruck  );
            credits.Unregulated  = Standards.GetCredits(mfr, scen, year, parameters, RC.Unregulated );
            return credits;
        }
        public static double GetCredits(Manufacturer mfr, Scenario scen, ModelYear year, Parameters parameters, RC regClass)
        {
            ManufacturerModelingData mmd = mfr.ModelingData;
            double cafe = mmd.CAFE[regClass];
            if (double.IsNaN(cafe)) { return 0; }   
            double sales    = mmd.Sales   [regClass];
            double standard = mmd.Standard[regClass];
            cafe =
                Standards.AdjustForACCredits(mfr, scen, year, parameters, regClass, cafe) +
                Standards.GetFFVCredits(mfr, scen, year, regClass);
            return (double.IsNaN(standard) || double.IsNaN(sales)) ? 0 : sales * (Math.Round(cafe, 1) - standard);
        }
        public static RCDouble TestCredits(Manufacturer mfr, Scenario scen, ModelYear year, Parameters parameters,
            RCDouble newCafe, TechImpact[] techImpacts, int techImpactCount)
        {
            RCDouble credits = new RCDouble();
            credits.DomesticAuto = Standards.TestCredits(mfr, scen, year, parameters, RC.DomesticAuto, newCafe.DomesticAuto,
                techImpacts, techImpactCount);
            credits.ImportedAuto = Standards.TestCredits(mfr, scen, year, parameters, RC.ImportedAuto, newCafe.ImportedAuto,
                techImpacts, techImpactCount);
            credits.LightTruck   = Standards.TestCredits(mfr, scen, year, parameters, RC.LightTruck  , newCafe.LightTruck  ,
                techImpacts, techImpactCount);
            credits.Unregulated  = Standards.TestCredits(mfr, scen, year, parameters, RC.Unregulated , newCafe.Unregulated ,
                techImpacts, techImpactCount);
            return credits;
        }
        public static double TestCredits(Manufacturer mfr, Scenario scen, ModelYear year, Parameters parameters, RC regClass,
            double newCafe, TechImpact[] techImpacts, int techImpactCount)
        {
            ManufacturerModelingData mmd = mfr.ModelingData;
            double sales    = mmd.Sales[regClass];
            double standard = Standards.GetStandard(mfr, scen, year, regClass, techImpacts, techImpactCount);
            newCafe =
                Standards.AdjustForACCredits(mfr, scen, year, parameters, regClass, newCafe) +
                Standards.GetFFVCredits(mfr, scen, year, regClass);
            return sales * (newCafe - standard);
        }
        static double GetFFVCredits(Manufacturer mfr, Scenario scen, ModelYear year, RC regClass)
        {
            if (mfr.Description.CreditsApplyToBaseline || !scen.IsBaseline)
            {
                return
                    (regClass == RC.DomesticAuto) ? mfr.Description.AvailableCreditsDA[year.Index] :
                    (regClass == RC.ImportedAuto) ? mfr.Description.AvailableCreditsIA[year.Index] :
                    (regClass == RC.LightTruck  ) ? mfr.Description.AvailableCreditsLT[year.Index] : 0;
            }
            return 0;
        }
        static double AdjustForACCredits(Manufacturer mfr, Scenario scen, ModelYear year, Parameters parameters, RC regClass,
            double cafe)
        {
            if (mfr.Description.ACCreditsApplyToBaseline || !scen.IsBaseline)
            {   
                double acCredit =
                    (regClass == RC.DomesticAuto) ? mfr.Description.AvailableACCreditsDA[year.Index] :
                    (regClass == RC.ImportedAuto) ? mfr.Description.AvailableACCreditsIA[year.Index] :
                    (regClass == RC.LightTruck  ) ? mfr.Description.AvailableACCreditsLT[year.Index] : 0;
                double co2 = parameters.GasolineCarbonContent * parameters.GasolineMassDensity * 44 / 12;
                return 1 / (1 / cafe - acCredit / co2);
            }
            else
            {   
                return cafe;
            }
        }
        #endregion
        #region 
        public static RCDouble GetCAFE(Manufacturer mfr, ModelYear year, out RCDouble sales, out RCDouble salesOverFe)
        {
            RCDouble cafe = new RCDouble();
            sales         = new RCDouble();
            salesOverFe   = new RCDouble();
            cafe.DomesticAuto = Standards.GetCAFE(mfr, year, RC.DomesticAuto, out sales.DomesticAuto, out salesOverFe.DomesticAuto);
            cafe.ImportedAuto = Standards.GetCAFE(mfr, year, RC.ImportedAuto, out sales.ImportedAuto, out salesOverFe.ImportedAuto);
            cafe.LightTruck   = Standards.GetCAFE(mfr, year, RC.LightTruck  , out sales.LightTruck  , out salesOverFe.LightTruck  );
            cafe.Unregulated  = Standards.GetCAFE(mfr, year, RC.Unregulated , out sales.Unregulated , out salesOverFe.Unregulated );
            return cafe;
        }
        public static double GetCAFE(Manufacturer mfr, ModelYear year, RC regClass, out double sales, out double salesOverFe)
        {
            int yrIndex = year.Index;
            sales       = 0;
            salesOverFe = 0;
            VehicleCollection vehs = mfr.ModelYearVehicles;
            for (int i = 0, vehCount = vehs.Count; i < vehCount; i++)
            {
                Vehicle veh = vehs[i];
                if (veh.RegClass == regClass)
                {
                    double vehSales = veh.Description.Sales[yrIndex];
                    double vehFe    = Math.Round(veh.Description.FuelEconomy, 1);
                    double vehSalesOverFe = vehSales / vehFe;
                    sales       += vehSales;
                    salesOverFe += vehSalesOverFe;
                    veh.ModelingData.SalesOverFe = vehSalesOverFe;
                } 
            } 
            return sales / salesOverFe;
        }
        public static RCDouble TestCAFE(Manufacturer mfr, ModelYear year, TechImpact[] techImpacts, int techImpactCount,
            bool useRounding)
        {
            RCDouble cafe = new RCDouble();
            cafe.DomesticAuto = Standards.TestCAFE(mfr, year, RC.DomesticAuto, techImpacts, techImpactCount, useRounding);
            cafe.ImportedAuto = Standards.TestCAFE(mfr, year, RC.ImportedAuto, techImpacts, techImpactCount, useRounding);
            cafe.LightTruck   = Standards.TestCAFE(mfr, year, RC.LightTruck  , techImpacts, techImpactCount, useRounding);
            cafe.Unregulated  = Standards.TestCAFE(mfr, year, RC.Unregulated , techImpacts, techImpactCount, useRounding);
            return cafe;
        }
        public static double TestCAFE(Manufacturer mfr, ModelYear year, RC regClass, TechImpact[] techImpacts,
            int techImpactCount, bool useRounding)
        {
            if (techImpactCount > 0 && (techImpacts == null || techImpacts.Length < techImpactCount))
            {   
                throw new ArgumentOutOfRangeException("techImpacts", "techImpacts is null or less than techImpactCount.");
            }
            int yrIndex = year.Index;
            ManufacturerModelingData mmd = mfr.ModelingData;
            double sales       = mmd.Sales[regClass];
            double salesOverFe = mmd.SalesOverFE[regClass];
            for (int i = 0; i < techImpactCount; i++)
            {
                if (!techImpacts[i].Ignore && techImpacts[i].Vehicle[yrIndex].RegClass == regClass)
                {
                    VehicleDescription vd = techImpacts[i].Vehicle[yrIndex].Description;
                    double vehSales = vd.Sales[yrIndex];
                    double vehFe    = (useRounding) ? Math.Round(vd.FuelEconomy + techImpacts[i].DeltaFuelEconomy[yrIndex], 1)
                                                    : vd.FuelEconomy + techImpacts[i].DeltaFuelEconomy[yrIndex];
                    double vehSalesOverFe  = techImpacts[i].Vehicle[yrIndex].ModelingData.SalesOverFe;
                    double salesOverFeDiff = vehSales / vehFe - vehSalesOverFe;
                    salesOverFe += salesOverFeDiff;
                }
            }
            return sales / salesOverFe;
        }
        #endregion
        #region 
        public static int GetCoefficients(RC regClass, Scenario scen, ModelYear year, out int fncType, out double[,] fncCoeff)
        {
            int yrIndex = year.Index;
            if (regClass == RC.DomesticAuto || regClass == RC.ImportedAuto)
            {
                fncType  = scen.AutoFunctionType[yrIndex];
                fncCoeff = scen.AutoCoefficients;
            }
            else if (regClass == RC.LightTruck)
            {
                fncType  = scen.LtFunctionType[yrIndex];
                fncCoeff = scen.LtCoefficients;
            }
            else
            {   
                fncType = 0;
                fncCoeff = null;
                return 0;
            }
            return fncCoeff.GetUpperBound(0) + 1;
        }
        [Obsolete("For improved performance, use: GetCoefficients(RegulatoryClass, Scenario, ModelYear, out int, out double[,]).")]
        public static int GetCoefficients(RC regClass, Scenario scen, ModelYear year, out int fncType, out double[] fncCoeff)
        {
            int yrIndex = year.Index;
            double[,] coeff;
            int numCoeff = Standards.GetCoefficients(regClass, scen, year, out fncType, out coeff);
            if (fncType == 0 || coeff == null) { fncCoeff = null; return 0; }
            fncCoeff = new double[numCoeff];
            for (int i = 0; i < numCoeff; i++)
            {
                fncCoeff[i] = coeff[i, yrIndex];
            }
            return numCoeff;
        }
        #endregion
        #endregion
    }
}

