using System;
using System.Drawing;
using System.Globalization;
using System.IO;
using Volpe.Utils;
namespace Volpe.Cafe.IO
{
    [Serializable]
    public class Input
    {
        #region 
        public Input(string path)
            : this(path, null)
        {
        }
        public Input(string path, string password)
        {
            if (!File.Exists(path))
            {
                throw new FileNotFoundException( "Error opening Excel workbook.  The specified file was not found.", path);
            }
            this._xlu = new XlUtilities(path, true, password);
            this._sheets = this._xlu.GetWorksheetNames();
            this.ParseTitle();
        }
        #endregion
        #region 
        private void ParseTitle()
        {
            string[] title = this._xlu.WorkbookTitle.Split(',');
            if (title != null)
            {
                if (title.Length >= 1) { this._title = title[0]; }
                if (title.Length >= 2) { this._version = (float)Global.GetDouble(title[1]); }
            }
        }
        public void Close()
        {
            this._xlu.Close(false);
        }
        public void ActivateWorksheet(string name)
        {
            this._buffer = this._xlu.GetData(name);
        }
        public void ActivateWorksheet(int index)
        {
            this._buffer = this._xlu.GetData(index + 1);
        }
        public void ActivateWorksheet(string name, XlCell firstCell, XlCell lastCell)
        {
            this._buffer = this._xlu.GetData(name, firstCell, lastCell);
        }
        public void ActivateWorksheet(int index, XlCell firstCell, XlCell lastCell)
        {
            this._buffer = this._xlu.GetData(index + 1, firstCell, lastCell);
        }
        public bool GetBool(int row, int column)
        {
            return Global.GetBool(this._buffer[row + 1, column + 1]);
        }
        public int GetInt32(int row, int column)
        {
            return Global.GetInt32(this._buffer[row + 1, column + 1]);
        }
        public double GetDouble(int row, int column)
        {
            return Global.GetDouble(this._buffer[row + 1, column + 1]);
        }
        public char GetChar(int row, int column)
        {
            return Global.GetChar(this._buffer[row + 1, column + 1]);
        }
        public string GetString(int row, int column)
        {
            return Global.GetString(this._buffer[row + 1, column + 1]);
        }
        public void GetMinMaxYears(int row, int startColumn, out int minYear, out int maxYear)
        {
            minYear = 0;
            maxYear = 0;
            if (row == -1)
            {
                throw new ArgumentException("The value of row must be greater than or equal to 0.", "row");
            }
            else if (startColumn == -1)
            {   
                return;
            }
            int cols = this.Columns;
            int numYears = 0;
            minYear = this.GetInt32(row, startColumn);
            if (minYear == 0)
            {   
                minYear = Global.GetInt32(this.GetString(row, startColumn).Replace("MY", ""));
            }
            if (minYear < 2000)
            {
                this.Close();
                throw new InputException("Only model years 2000 and above are supported.", this._xlu.Path);
            }
            int prev = minYear - 1, current = minYear;
            while (prev + 1 == current)
            {
                numYears++;
                prev = current;
                if (startColumn + numYears >= cols)
                {   
                    break;
                }
                current = this.GetInt32(row, startColumn + numYears);
                if (current == 0)
                {
                    current = Global.GetInt32(this.GetString(row, startColumn + numYears).Replace("MY", ""));
                }
            }
            maxYear = prev;
        }
        public bool Compare(string value1, string value2, bool fuzzy)
        {
            return Global.StringCompare(value1, value2, fuzzy);
        }
        public void VerifyIndexes(int[] indexes, string[] names)
        {
            this.VerifyIndexes(indexes, names, 0, indexes.Length);
        }
        public void VerifyIndexes(int[] indexes, string[] names, int startIndex, int length)
        {
            int endIndex = startIndex + length;
            if (indexes.Length != names.Length)
            {
                throw new ArgumentException(
                    "The length of the indexes array does not match the length of the names array.", "names");
            }
            if (startIndex < 0 || indexes.Length < endIndex || names.Length < endIndex)
            {
                throw new ArgumentOutOfRangeException("startIndex or length",
                    "The startIndex and/or length is out side the bounds of the indexes and names arrays.");
            }
            string missing = "";
            for (int i = startIndex; i < endIndex; i++)
            {
                if (indexes[i] == -1)
                {
                    missing += "\n\t" + names[i] + " section or entry not found.";
                }
            }
            if (missing != "")
            {   
                string path = this._xlu.Path;
                this.Close();
                throw new InputException("\n" + missing, path);
            }
        }
        #endregion
        #region 
        public XlUtilities XlUtils { get { return this._xlu; } }
        public string[] SheetNames { get { return this._sheets; } }
        public object[,] Buffer
        {
            get
            {
                int rows = this.Rows;
                int columns = this.Columns;
                object[,] buffer = new object[rows, columns];
                for (int i = 0; i < rows; i++)
                {
                    for (int j = 0; j < columns; j++)
                    {
                        buffer[i, j] = this._buffer[i + 1, j + 1];
                    }
                }
                return buffer;
            }
        }
        public int Rows { get { return this._buffer.GetUpperBound(0); } }
        public int Columns { get { return this._buffer.GetUpperBound(1); } }
        public string Title { get { return this._title; } }
        public float Version { get { return this._version; } }
        #endregion
        #region 
        protected XlUtilities _xlu;
        protected string[] _sheets;
        protected object[,] _buffer;
        protected string _title;
        protected float _version;
        #endregion
    }
}

