using System;
using System.IO;
using Volpe.Cafe;
using Volpe.Cafe.Collections;
using Volpe.Cafe.Data;
using Volpe.Cafe.Model;
namespace Volpe.Cafe.IO
{
    [Serializable]
    public sealed class LogWriter
    {
        #region 
        LogWriter()
        {
            this._isEmpty = true;
        }
        public LogWriter(string basePath)
            : this(basePath, false)
        {
        }
        public LogWriter(string basePath, bool writeExtLogs)
        {
            if (!Directory.Exists(basePath))
            {
                throw new IOException("The specified basePath could not be found.");
            }
            this._basePath      = basePath;
            this._writesExtLogs = writeExtLogs;
            this._isEmpty       = false;
        }
        ~LogWriter()
        {
            this.Close();
        }
        #endregion
        #region 
        string GenerateComplianceLogHeader(int scenIndex, int year, Manufacturer mfr)
        {
            string header = "Scen:\t" + scenIndex + "\nYear:\t" + year.ToString() +
                "\nMfr:\t" + Global.GetTitleCase(mfr.Description.Name, 4) + "\n\n" +
                "LN\tCF\tApplied\tEff\tTotal Delta RPE\tTotal Delta Fines\tTotal FC Savings\tTotal Affected Sales\t" +
                "Reg-Class\tStandard\tCAFE\tNew CAFE\tFines\tNew Fines\tVeh\tTechs\tYear\tFE\tNew FE\tSales\tCost\tImprv\tDelta CW\tDelta FE\t";
            return header;
        }
        void WriteLog(TextWriter tw, ref string value)
        {
            tw.Write(value);
        }
        public void WriteError(string value)
        {
            if (this._errors != null) { this.WriteLog(this._errors, ref value); }
        }
        public void WriteWarning(string value)
        {
            if (this._warnings != null) { this.WriteLog(this._warnings, ref value); }
        }
        public void WriteSummary(string value)
        {
            if (this._summary != null) { this.WriteLog(this._summary, ref value); }
        }
        public void WriteCompliance(Scenario scenario, ModelYear year, Manufacturer mfr, string value)
        {
            ComplianceWriter cw = this.GetCompliance(scenario, year, mfr);
            if (cw != null) { cw.Write(value); }
        }
        public void WriteExtCompliance(Scenario scenario, ModelYear year, Manufacturer mfr, string value)
        {
            ComplianceWriter cw = this.GetExtCompliance(scenario, year, mfr);
            if (cw != null) { cw.Write(value); }
        }
        public void WriteErrorLine(string value)
        {
            this.WriteError(value); this.WriteError(NewLine);
        }
        public void WriteWarningLine(string value)
        {
            this.WriteWarning(value); this.WriteWarning(NewLine);
        }
        public void WriteSummaryLine(string value)
        {
            this.WriteSummary(value); this.WriteSummary(NewLine);
        }
        public void WriteComplianceLine(Scenario scenario, ModelYear year, Manufacturer mfr, string value)
        {
            this.WriteCompliance(scenario, year, mfr, value);
            this.WriteCompliance(scenario, year, mfr, NewLine);
        }
        public void WriteExtComplianceLine(Scenario scenario, ModelYear year, Manufacturer mfr, string value)
        {
            this.WriteExtCompliance(scenario, year, mfr, value);
            this.WriteExtCompliance(scenario, year, mfr, NewLine);
        }
        public void Open(bool append,
            int scenCount, int mfrCount, int minYearIndex, int maxYearIndex, ManufacturerCollection mfrs)
        {
            if (this._basePath == null) { return; }
            this._errors   = new StreamWriter(this._basePath + "\\Errors.txt"  , append);
            this._warnings = new StreamWriter(this._basePath + "\\Warnings.txt", append);
            this._summary  = new StreamWriter(this._basePath + "\\Summary.txt" , append);
            this.OpenCompliance(append, scenCount, mfrCount, minYearIndex, maxYearIndex, mfrs);
            this._isOpen = true;
        }
        void OpenCompliance(bool append,
            int scenCount, int mfrCount, int minYearIndex, int maxYearIndex, ManufacturerCollection mfrs)
        {
            if (mfrs == null) { return; }
            this._scenCount    = scenCount;
            this._yearCount    = maxYearIndex - minYearIndex + 1;
            this._mfrCount     = mfrCount;
            this._minYearIndex = minYearIndex;
            this._maxYearIndex = maxYearIndex;
            int compLogCount = this._scenCount * this._yearCount * this._mfrCount;
            this._compliance = new ComplianceWriter[compLogCount];
            if (this._writesExtLogs) { this._extCompliance = new ComplianceWriter[compLogCount]; }
            for (int i = 0; i < this._scenCount; i++)
            {
                for (int j = 0; j < this._yearCount; j++)
                {
                    int modelYear = j + this._minYearIndex + ModelYear.MinYear;
                    for (int k = 0; k < this._mfrCount; k++)
                    {   
                        int compIdx = (i * this._yearCount + j) * this._mfrCount + k;
                        string header = this.GenerateComplianceLogHeader(i, modelYear, mfrs[k]);
                        this._compliance[compIdx] = new ComplianceWriter(this._basePath +
                            "\\compliance_" + i + "_" + modelYear + "_" + mfrs[k] + ".txt", append);
                        this._compliance[compIdx].WriteLine(header);
                        if (this._writesExtLogs)
                        {   
                            this._extCompliance[compIdx] = new ComplianceWriter(this._basePath +
                                "\\ext-compliance_" + i + "_" + modelYear + "_" + mfrs[k] + ".txt", append);
                            this._extCompliance[compIdx].WriteLine(header);
                        }
                    } 
                } 
            } 
        }
        public void Close()
        {
            this.CloseErrors();
            this.CloseWarnings();
            this.CloseSummary();
            this.CloseCompliance();
            this.CloseExtCompliance();
            this._isOpen = false;
        }
        void CloseErrors()
        {
            if (this._errors != null ) { this.Errors.Close(); this._errors = null; }
        }
        void CloseWarnings()
        {
            if (this._warnings != null) { this.Warnings.Close(); this._warnings = null; }
        }
        void CloseSummary()
        {
            if (this._summary != null) { this.Summary.Close(); this._summary = null; }
        }
        void CloseCompliance(int index)
        {
            if (this._compliance != null && 0 <= index && index < this._compliance.Length &&
                this._compliance[index] != null)
            {
                this._compliance[index].Close();
                this._compliance[index] = null;
            }
        }
        void CloseCompliance()
        {
            if (this._compliance != null)
            {
                for (int i = 0, count = this._compliance.Length; i < count; i++)
                {
                    this.CloseCompliance(i);
                }
                this._compliance = null;
            }
        }
        void CloseExtCompliance(int index)
        {
            if (this._extCompliance != null && 0 <= index && index < this._extCompliance.Length &&
                this._extCompliance[index] != null)
            {
                this._extCompliance[index].Close();
                this._extCompliance[index] = null;
            }
        }
        void CloseExtCompliance()
        {
            if (this._extCompliance != null)
            {
                for (int i = 0, count = this._extCompliance.Length; i < count; i++)
                {
                    this.CloseExtCompliance(i);
                }
                this._extCompliance = null;
            }
        }
        public ComplianceWriter GetCompliance(Scenario scenario, ModelYear year, Manufacturer mfr)
        {
            int snIndex  = (scenario == null) ? 0 : scenario.Index;
            int myIndex  = (year     == null) ? 0 : year.Index - this._minYearIndex;
            int mfrIndex = (mfr      == null) ? 0 : mfr.Index;
            int logIndex = (snIndex * this._yearCount + myIndex) * this._mfrCount + mfrIndex;
            if (this._compliance == null || logIndex < 0 || this._compliance.Length <= logIndex)
            {
                return null;
            }
            else { return this._compliance[logIndex]; }
        }
        public ComplianceWriter GetExtCompliance(Scenario scenario, ModelYear year, Manufacturer mfr)
        {
            int snIndex  = (scenario == null) ? 0 : scenario.Index;
            int myIndex  = (year     == null) ? 0 : year.Index - this._minYearIndex;
            int mfrIndex = (mfr      == null) ? 0 : mfr.Index;
            int logIndex = (snIndex * this._yearCount + myIndex) * this._mfrCount + mfrIndex;
            if (this._extCompliance == null || logIndex < 0 || this._extCompliance.Length <= logIndex)
            {
                return null;
            }
            else { return this._extCompliance[logIndex]; }
        }
        #endregion
        #region 
        public string LogPath
        {
            get { return this._basePath; }
            set
            {
                if (this._isEmpty) { throw new InvalidOperationException("Cannot modify an empty LogWriter--it is read-only."); }
                if (this._isOpen)  { throw new InvalidOperationException("The LogWriter is opened; path cannot be modified."); }
                this._basePath = value;
            }
        }
        public bool WritesExtLogs { get { return this._writesExtLogs; } }
        public bool IsEmpty { get { return this._isEmpty; } }
        public bool IsOpen { get { return this._isOpen; } }
        public TextWriter Errors   { get { return this._errors;   } }
        public TextWriter Warnings { get { return this._warnings; } }
        public TextWriter Summary  { get { return this._summary;  } }
        public ComplianceWriter[] Compliance    { get { return this._compliance; } }
        public ComplianceWriter[] ExtCompliance { get { return this._extCompliance; } }
        #endregion
        #region 
        public static readonly LogWriter Empty = new LogWriter();
        const string NewLine = "\n";
        string _basePath;
        bool   _writesExtLogs;
        bool   _isEmpty;
        bool   _isOpen;
        TextWriter _errors;
        TextWriter _warnings;
        TextWriter _summary;
        ComplianceWriter[] _compliance;
        ComplianceWriter[] _extCompliance;
        int _scenCount, _yearCount, _mfrCount, _minYearIndex, _maxYearIndex;
        #endregion
    }
}

