#region << Using Directives >>
using System;
using Volpe.Cafe;
using Volpe.Cafe.Collections;
using Volpe.Cafe.Data;
using TI = Volpe.Cafe.TechnologyIndexes;
#endregion
namespace Volpe.Cafe.Model
{
    public sealed class TechnologyApplicability
    {
        #region 
        static TechnologyApplicability()
        {
            TechSupersedes       = new int[TI.TechnologyCount][];
            TechDisallowBackfill = new int[TI.TechnologyCount][];
            TechKeys = new ulong[63];
            for (int i = 0; i < 63; i++) { TechKeys[i] = (ulong)Math.Pow(2, i); }
            TechTypes = new TechnologyType[] {TechnologyType.Engine, TechnologyType.Transmission, TechnologyType.Elec,
                TechnologyType.Hybrid, TechnologyType.MSR, TechnologyType.DLR, TechnologyType.Aerodynamics, TechnologyType.Other};
            TechTypeCount = TechTypes.Length;
            for (int i = 0; i < TI.TechnologyCount; i++)
            {
                TechSupersedes      [i] = new int[0];
                TechDisallowBackfill[i] = new int[0];
                switch (i)
                {
                    case TI.DCP:
                        TechSupersedes[i] = new int[] {TI.ICP};
                        break;
                    case TI.DVVLD:
                        TechDisallowBackfill[i] = new int[] {TI.CVVL};
                        break;
                    case TI.CVVL:
                        TechDisallowBackfill[i] = new int[] {TI.DVVLD};
                        break;
                    case TI.DVVLO:
                        TechDisallowBackfill[i] = new int[] {TI.CDOHC};
                        break;
                    case TI.CDOHC:
                        TechSupersedes[i] = new int[] {TI.CCPO};
                        TechDisallowBackfill[i] = new int[] {TI.DVVLO};
                        break;
                    case TI.TRBDS:
                        TechSupersedes[i] = new int[] {TI.DEACS, TI.DEACD, TI.DEACO};
                        TechDisallowBackfill[i] = new int[] {TI.DSLC};
                        break;
                    case TI.EGRB:
                        TechSupersedes[i] = new int[] {TI.DEACS, TI.DEACD, TI.DEACO};
                        TechDisallowBackfill[i] = new int[] {TI.DSLC};
                        break;
                    case TI.DSLT:
                        TechSupersedes[i] = new int[] {TI.LUB, TI.EFR, TI.CCPS, TI.DVVLS, TI.DEACS, TI.ICP, TI.DCP, TI.DVVLD, 
                            TI.CVVL, TI.DEACD, TI.DEACO, TI.CCPO, TI.DVVLO, TI.CDOHC, TI.SGDI, TI.CBRST, TI.TRBDS, TI.EGRB};
                        TechDisallowBackfill[i] = new int[] {TI.DSLC};
                        break;
                    case TI.DSLC:
                        TechSupersedes[i] = new int[] {TI.LUB, TI.EFR, TI.CCPS, TI.DVVLS, TI.DEACS, TI.ICP, TI.DCP, TI.DVVLD, 
                            TI.CVVL, TI.DEACD, TI.DEACO, TI.CCPO, TI.DVVLO, TI.CDOHC, TI.SGDI, TI.CBRST};
                        TechDisallowBackfill[i] = new int[] {TI.TRBDS, TI.EGRB, TI.DSLT};
                        break;
                    case TI.CVT:
                        TechSupersedes[i] = new int[] {TI.IATC};
                        TechDisallowBackfill[i] = new int[] {TI.NAUTO, TI.DCTAM};
                        break;
                    case TI.NAUTO:
                        TechSupersedes[i] = new int[] {TI.IATC};
                        TechDisallowBackfill[i] = new int[] {TI.CVT};
                        break;
                    case TI.DCTAM:
                        TechSupersedes[i] = new int[] {TI.IATC, TI.NAUTO};
                        TechDisallowBackfill[i] = new int[] {TI.CVT};
                        break;
                    case TI.MHEV:
                        TechSupersedes[i] = new int[] {TI.CBRST};
                        break;
                    case TI.BISG:
                        TechSupersedes[i] = new int[] {TI.CBRST, TI.MHEV};
                        break;
                    case TI.CISG:
                        TechSupersedes[i] = new int[] {TI.CBRST, TI.MHEV, TI.BISG};
                        break;
                    case TI.PSHEV:
                        TechSupersedes[i] = new int[] {TI.CBRST, TI.IATC, TI.CVT, TI.NAUTO, TI.DCTAM, TI.MHEV, TI.BISG, TI.CISG};
                        TechDisallowBackfill[i] = new int[] {TI.MHEV2, TI.PHEV};
                        break;
                    case TI.MHEV2:
                        TechSupersedes[i] = new int[] {TI.CBRST, TI.IATC, TI.CVT, TI.NAUTO, TI.DCTAM, TI.MHEV, TI.BISG, TI.CISG};
                        TechDisallowBackfill[i] = new int[] {TI.PSHEV, TI.PHEV};
                        break;
                    case TI.PHEV:
                        TechSupersedes[i] = new int[] {TI.CBRST, TI.IATC, TI.CVT, TI.NAUTO, TI.DCTAM, TI.MHEV, TI.BISG, TI.CISG};
                        TechDisallowBackfill[i] = new int[] {TI.PSHEV, TI.MHEV2};
                        break;
                } 
            } 
        }
        #endregion
        #region 
        #region 
        public static void Initialize(Vehicle vehicle, ModelingSettings settings)
        {
            VehicleDescription      vehDescr = vehicle.Description;
            EngineDescription       engDescr = vehicle.Engine.Description;
            TransmissionDescription trnDescr = vehicle.Transmission.Description;
            VehicleModelingData     vmd      = vehicle.ModelingData;
            TechnologyClass      techClass    = vehicle.TechnologyClass;
            TechnologyCollection technologies = settings.Technologies;
            vmd.TechAvailable         = new bool[TI.TechnologyCount];
            vmd.TechUsed              = new bool[TI.TechnologyCount];
            vmd.TechApplied           = new bool[TI.TechnologyCount];
            vmd.TechSuperseded        = new bool[TI.TechnologyCount];
            vmd.TechBackfilled        = new bool[TI.TechnologyCount];
            vmd.TechPhaseInBackfilled = new bool[TI.TechnologyCount];
            vmd.TechEnabled           = new bool[TI.TechnologyCount];
            vmd.TechIgnored           = new bool[TI.TechnologyCount];
            vmd.TechReserved          = new bool[TI.TechnologyCount];
            vmd.TechLingering         = new bool[TI.TechnologyCount];
            vmd.TechLingerYear        = null;
            bool[] techUnused         = new bool[TI.TechnologyCount];
            for (int i = 0; i < TI.TechnologyCount; i++)
            {
                string               abbr     = technologies[i].Abbr;
                TechnologyType       type     = technologies[i].Type;
                TechnologyAttributes techAttr = technologies[i].Attributes[techClass];
                vmd.TechAvailable[i] = techAttr.Applicable;
                TechnologyOverride techOverride =
                    (type == TechnologyType.Engine      ) ? engDescr.Overrides[abbr] :
                    (type == TechnologyType.Transmission) ? trnDescr.Overrides[abbr] : vehDescr.Overrides[abbr];
                if      (techOverride == TechnologyOverride.Used  ) { vmd.TechUsed     [i] = true;  }
                else if (techOverride == TechnologyOverride.Skip  ) { vmd.TechAvailable[i] = false; }
                else if (techOverride == TechnologyOverride.Force ) { vmd.TechAvailable[i] = true;  }
                else if (techOverride == TechnologyOverride.Unused) { techUnused       [i] = true;  }
            }
            if (!(vmd.TechUsed[TI.DSLT] || vmd.TechUsed[TI.DSLC]) && engDescr.Fuel == "D")
            {   
                vmd.TechUsed[TI.DSLC] = !techUnused[TI.DSLC];
            }
            if (vmd.TechUsed[TI.DSLT] || vmd.TechUsed[TI.DSLC])
            {   
                bool dsltUsed = vmd.TechUsed[TI.DSLT];
                bool dslcUsed = vmd.TechUsed[TI.DSLC];
                RectifyDependencies_DisableAllEng(vmd.TechUsed);
                vmd.TechUsed[TI.DSLT] = dsltUsed;
                vmd.TechUsed[TI.DSLC] = dslcUsed;
            }
            else
            {   
                if (!vmd.TechUsed[TI.LUB] && !Global.StringCompareAny(engDescr.EngineOilViscosity,
                    new string[] {"", "5w30", "10w30", "10w40", "15w40", "Lub1", "Lub2"}, false))
                {    
                    vmd.TechUsed[TI.LUB] = !techUnused[TI.LUB];
                }
                if (Global.StringCompare(engDescr.Configuration, "R", false))
                {   
                    TechnologyApplicability.RectifyDependencies_DisableDohcEng(vmd.TechAvailable);
                    TechnologyApplicability.RectifyDependencies_DisableSohcEng(vmd.TechAvailable);
                    TechnologyApplicability.RectifyDependencies_DisableOhvEng (vmd.TechAvailable);
                    vmd.TechAvailable[TI.TRBDS] = false;
                    vmd.TechAvailable[TI.EGRB ] = false;
                }
                else
                {   
                    bool sohcPath = Global.StringCompare   (engDescr.ValvetrainDesign, "SOHC", false);
                    bool dohcPath = Global.StringCompare   (engDescr.ValvetrainDesign, "DOHC", false);
                    bool ohvPath  = Global.StringCompareAny(engDescr.ValvetrainDesign, new string[] {"", "OHV"}, false);
                    if (sohcPath)
                    {
                        TechnologyApplicability.RectifyDependencies_DisableDohcEng(vmd.TechAvailable);
                        TechnologyApplicability.RectifyDependencies_DisableOhvEng (vmd.TechAvailable);
                    }
                    else if (dohcPath)
                    {
                        TechnologyApplicability.RectifyDependencies_DisableSohcEng(vmd.TechAvailable);
                        TechnologyApplicability.RectifyDependencies_DisableOhvEng (vmd.TechAvailable);
                    }
                    else if (ohvPath)
                    {
                        TechnologyApplicability.RectifyDependencies_DisableSohcEng(vmd.TechAvailable);
                        TechnologyApplicability.RectifyDependencies_DisableDohcEng(vmd.TechAvailable);
                    }
                    if (!Global.StringCompareAny(engDescr.ValveActuationTiming, new string[] {"", "F", "E", "N/A"}, false))
                    {   
                        if (sohcPath && Global.StringCompareAny(engDescr.ValveActuationTiming,
                            new string[] {"CC", "DCL", "EIE", "VCT", "VCT+VVTE+ICP"}, false))
                        {
                            vmd.TechUsed[TI.CCPS] = !techUnused[TI.CCPS];
                        }
                        else if (dohcPath)
                        {
                            if (Global.StringCompareAny(engDescr.ValveActuationTiming,
                                new string[] {"CC", "EIE", "IIE", "TIVCT", "VCT+VVTE", "VCT+VVTE+ICP"}, false))
                            {
                                vmd.TechUsed[TI.DCP] = !techUnused[TI.DCP];
                            }
                            else if (Global.StringCompareAny(engDescr.ValveActuationTiming,
                                new string[] {"ICP", "VCT", "VCT+ICP", "VVTE"}, false))
                            {
                                vmd.TechUsed[TI.ICP] = !techUnused[TI.ICP];
                            }
                        }
                        else if (ohvPath && Global.StringCompareAny(engDescr.ValveActuationTiming,
                            new string[] {"EIE", "ICP"}, false))
                        {
                            vmd.TechUsed[TI.CCPO] = !techUnused[TI.CCPO];
                        }
                    }
                    if (!Global.StringCompareAny(engDescr.ValveLift, new string[] {"", "F", "N/A"}, false))
                    {   
                        if (sohcPath && Global.StringCompareAny(engDescr.ValveActuationTiming,
                            new string[] {"CV", "SVI", "SVIE"}, false))
                        {
                            vmd.TechUsed[TI.DVVLS] = !techUnused[TI.DVVLS];
                        }
                        else if (dohcPath)
                        {
                            if (Global.StringCompareAny(engDescr.ValveLift, new string[] {"CV"}, false))
                            {
                                vmd.TechUsed[TI.CVVL] = !techUnused[TI.CVVL];
                            }
                            else if (Global.StringCompareAny(engDescr.ValveLift, new string[] {"SVI", "SVIE"}, false))
                            {
                                vmd.TechUsed[TI.DVVLD] = !techUnused[TI.DVVLD];
                            }
                        }
                        else if (ohvPath && Global.StringCompareAny(engDescr.ValveLift, new string[] {"SVI", "SVIE"}, false))
                        {
                            vmd.TechUsed[TI.DVVLO] = !techUnused[TI.DVVLO];
                        }
                    }
                    if (engDescr.Deactivation == 0.3 || engDescr.Deactivation == 0.5)
                    {    
                        if      (sohcPath) { vmd.TechUsed[TI.DEACS] = !techUnused[TI.DEACS]; }
                        else if (dohcPath) { vmd.TechUsed[TI.DEACD] = !techUnused[TI.DEACD]; }
                        else if (ohvPath ) { vmd.TechUsed[TI.DEACO] = !techUnused[TI.DEACO]; }
                    }
                    else if (engDescr.Cylinders < 6 || engDescr.Configuration.ToUpper() == "I")
                    {   
                        TechnologyApplicability.RectifyDependencies_DisableDeacEng(vmd.TechAvailable);
                    }
                    if (!vmd.TechUsed[TI.TRBDS] && Global.StringCompareAny(engDescr.Aspiration.ToString(),
                        new string[] {"T", "S"}, false))
                    {   
                        vmd.TechUsed[TI.TRBDS] = !techUnused[TI.TRBDS];
                    }
                }
                if (!vmd.TechUsed[TI.SGDI])
                {
                    if (Global.StringCompare(engDescr.FuelSystem, "SIDI", false) ||
                        (char.ToUpper(engDescr.Cycle) == 'O' && Global.StringCompare(engDescr.FuelSystem, "DI", false)))
                    {   
                        vmd.TechUsed[TI.SGDI] = !techUnused[TI.SGDI];
                    }
                }
            } 
            string trnType    = trnDescr.Type   .ToUpper();
            string trnControl = trnDescr.Control.ToUpper();
            if (trnType == "M" || (trnType == "C" && trnControl == "M"))
            {   
                if (techUnused[TI.MAN6] || trnDescr.NumForwardGears < 6)
                {   
                    TechnologyApplicability.RectifyDependencies_DisableAutoTrn(vmd.TechAvailable);
                    TechnologyApplicability.RectifyDependencies_DisableAllHev (vmd.TechAvailable);
                    TechnologyApplicability.RectifyDependencies_DisableDeacEng(vmd.TechAvailable);
                }
                else
                {   
                    vmd.TechUsed[TI.MAN6] = !techUnused[TI.MAN6];
                }
            } 
            else
            {   
                TechnologyApplicability.RectifyDependencies_DisableManualTrn(vmd.TechAvailable);
                if (trnType.StartsWith("CVT"))
                {   
                    vmd.TechUsed[TI.CVT] = !techUnused[TI.CVT];
                }
                else if (trnType == "AMT" || trnType == "DCT" || (trnType == "C" && trnControl == "A"))
                {   
                    vmd.TechUsed[TI.DCTAM] = !techUnused[TI.DCTAM];
                }
                else if (trnType == "A" || trnType == "T")
                {   
                    if (trnDescr.Logic.ToUpper() == "A")
                    {   
                        vmd.TechUsed[TI.IATC] = !techUnused[TI.IATC];
                    }
                    if (trnDescr.NumForwardGears >= 6)
                    {   
                        vmd.TechUsed[TI.NAUTO] = !techUnused[TI.NAUTO];
                    }
                } 
            } 
            if (!vmd.TechUsed[TI.CVT] &&
                Global.StringCompareAny(vehDescr.Structure, new string[] {"frame", "ladder"}, false) &&
                Global.StringCompareAny(vehDescr.Drive.ToString(), new string[] {"4", "R"}, false))
            {   
                vmd.TechAvailable[TI.CVT] = false;
            }
            if (!settings.OperatingModes.IgnoreWeightInMsApplication)
            {
                if (vehDescr.CurbWeight < 5000)
                {   
                    vmd.TechAvailable[TI.MS1] = false;
                    vmd.TechAvailable[TI.MS2] = false;
                    vmd.TechAvailable[TI.MS5] = false;
                }
            }
            if (vehDescr.Drive != '4')
            {
                vmd.TechAvailable[TI.SAXL] = false;
                vmd.TechAvailable[TI.SAXU] = false;
            }
            TechnologyApplicability.RectifyDependencies(vehicle, null, -1, settings);
            for (int i = 0; i < TI.TechnologyCount; i++)
            {
                string         abbr = technologies[i].Abbr;
                TechnologyType type = technologies[i].Type;
                TechnologyOverride techOverride =
                    (type == TechnologyType.Engine      ) ? engDescr.Overrides[abbr] :
                    (type == TechnologyType.Transmission) ? trnDescr.Overrides[abbr] :
                    (type == TechnologyType.Hybrid      ) ? trnDescr.Overrides[abbr] : vehDescr.Overrides[abbr];
                if      (techOverride == TechnologyOverride.Skip ) { vmd.TechAvailable[i] = false; }
                else if (techOverride == TechnologyOverride.Force) { vmd.TechAvailable[i] = true;  }
            }
        }
        public static void Enable(Vehicle vehicle, ModelYear modelYear, int techIndex, ModelingSettings settings)
        {
            VehicleDescription   vehDescr     = vehicle.Description;
            VehicleModelingData  vmd          = vehicle.ModelingData;
            TechnologyClass      techClass    = vehicle.TechnologyClass;
            TechnologyCollection technologies = settings.Technologies;
            int                  year         = modelYear.Year;
            if (techIndex == -1)
            {   
                for (int i = 0; i < TI.TechnologyCount; i++)
                {
                    vmd.TechEnabled [i] = vmd.TechAvailable[i] &&
                        (settings.OperatingModes.IgnoreYearAvailable || technologies[i].Attributes[techClass].Year <= year);
                    vmd.TechIgnored [i] = false;
                    vmd.TechReserved[i] = false;
                }
            }
            else
            {   
                vmd.TechEnabled[techIndex] = vmd.TechAvailable[techIndex] &&
                    (technologies[techIndex].Attributes[techClass].Year <= year);
            }
            if (!settings.OperatingModes.IgnoreRefreshRedesign)
            {   
                bool atRedesign, inShadowOfRedesign;
                bool atRefresh , inShadowOfRefresh ;
                vehicle.GetRedesignState(modelYear, out atRedesign, out inShadowOfRedesign);
                vehicle.GetRefreshState (modelYear, out atRefresh , out inShadowOfRefresh );
                int sIdx = (techIndex == -1) ? 0                  : techIndex;
                int eIdx = (techIndex == -1) ? TI.TechnologyCount : techIndex + 1;
                for (int i = sIdx; i < eIdx; i++)
                {   
                    if (vmd.TechEnabled[i])
                    {
                        if (inShadowOfRedesign)
                        {   
                            if (TI.IsTiedToRedesign(i))
                            {
                                vmd.TechEnabled[i] = false;
                            }
                            else if (!atRefresh)
                            {   
                                if (TI.IsTiedToRefresh(i)) { vmd.TechEnabled[i] = false; }
                            }
                        }
                        else if (!atRedesign && inShadowOfRefresh)
                        {   
                            if (TI.IsTiedToRefresh(i)) { vmd.TechEnabled[i] = false; }
                        }
                    } 
                } 
            }
            TechnologyApplicability.RectifyDependencies(vehicle, modelYear, techIndex, settings);
        }
        static void RectifyDependencies(Vehicle vehicle, ModelYear year, int techIdx, ModelingSettings settings)
        {
            VehicleModelingData vmd = vehicle.ModelingData;     
            bool updateTechEnabled = (year != null);
            bool updateSuperseded  = (techIdx != -1);
            bool[] techAE   = (updateTechEnabled) ? vmd.TechEnabled : vmd.TechAvailable;
            bool[] techLing = vmd.TechLingering;
            bool[] techUsed = vmd.TechUsed;
            int sIdx = (techIdx == -1) ? 0                  : techIdx;
            int eIdx = (techIdx == -1) ? TI.TechnologyCount : techIdx + 1;
            bool hevRectified = false;  
            for (int i = sIdx; i < eIdx; i++)
            {   
                if (techUsed[i]) { techAE[i] = false; }
                RectifyDependenciesHelper(i, techUsed, techAE, updateTechEnabled, ref hevRectified, vehicle, year, techIdx,
                    settings);
                if (updateTechEnabled)
                {   
                    RectifyDependenciesHelper(i, techUsed, techLing, updateTechEnabled, ref hevRectified, vehicle, year, techIdx,
                        settings);
                }
                if (techUsed[i] && vmd.TechApplied[i] && updateSuperseded)
                {   
                    int[] techSupersedes = TechSupersedes[i];
                    for (int j = 0, techSupCount = techSupersedes.Length; j < techSupCount; j++)
                    {   
                        vmd.TechSuperseded[techSupersedes[j]] = vmd.TechUsed[techSupersedes[j]];
                    }
                }
            } 
        }
        static void RectifyDependenciesHelper(int i, bool[] techUsed, bool[] techArr, bool updateTechEnabled,
            ref bool hevRectified, Vehicle vehicle, ModelYear year, int techIdx, ModelingSettings settings)
        {
            switch (i)
            {
                case TI.LUB:
                case TI.EFR:
                    break;
                case TI.CCPS:
                case TI.DVVLS:
                case TI.DEACS:
                    break;
                case TI.ICP:
                case TI.DEACD:
                    break;
                case TI.DCP:
                    if (techUsed[i]) { techArr[TI.ICP] = false; }
                    break;
                case TI.DVVLD:
                    if (techUsed[i]) { techArr[TI.CVVL] = false; }
                    break;
                case TI.CVVL:
                    if (techUsed[i]) { techArr[TI.DVVLD] = false; }
                    break;
                case TI.DEACO:
                case TI.CCPO:
                    break;
                case TI.DVVLO:
                    if (techUsed[i]) { techArr[TI.CDOHC] = false; }
                    break;
                case TI.CDOHC:
                    if (techUsed[i]) { techArr[TI.DVVLO] = false; }
                    break;
                case TI.SGDI:
                case TI.CBRST:
                    break;
                case TI.TRBDS:
                case TI.EGRB:
                    if (techUsed[i])
                    {   
                        TechnologyApplicability.RectifyDependencies_DisableDeacEng(techArr);
                        techArr[TI.DSLC] = false;
                    }
                    break;
                case TI.DSLT:
                case TI.DSLC:
                    if (techUsed[i]) { TechnologyApplicability.RectifyDependencies_DslUsed(techArr); }
                    break;
                case TI.MAN6:
                    if (techUsed[i])
                    {   
                        TechnologyApplicability.RectifyDependencies_DisableAutoTrn(techArr);
                        TechnologyApplicability.RectifyDependencies_DisableAllHev (techArr);
                        TechnologyApplicability.RectifyDependencies_DisableDeacEng(techArr);
                    }
                    break;
                case TI.IATC:
                case TI.CVT:
                case TI.NAUTO:
                case TI.DCTAM:
                    if (techUsed[i])
                    {   
                        TechnologyApplicability.RectifyDependencies_DisableManualTrn(techArr);
                        techArr[TI.IATC] = false;
                        if (i == TI.CVT)
                        {   
                            techArr[TI.NAUTO] = false;
                            techArr[TI.DCTAM] = false;
                            techArr[TI.MHEV2] = false;
                        }
                        else if (i == TI.NAUTO || i == TI.DCTAM)
                        {   
                            techArr[TI.CVT  ] = false;
                            techArr[TI.NAUTO] = false;
                        }
                    }
                    if (updateTechEnabled && !hevRectified)
                    {
                        RectifyDependencies_EnableHev(techArr, techUsed, vehicle, year, techIdx, settings);
                        hevRectified = true;    
                    }
                    break;
                case TI.EPS:
                case TI.IACC:
                    break;
                case TI.MHEV:
                case TI.BISG:
                case TI.CISG:
                    if (techUsed[i])
                    {   
                        techArr[TI.CBRST] = false;
                        if (i == TI.BISG) { RectifyDependencies_DisableAllElec(techArr, false); }
                        if (i == TI.CISG) { RectifyDependencies_DisableAllElec(techArr, true); }
                    }
                    break;
                case TI.PSHEV:
                case TI.MHEV2:
                case TI.PHEV:
                    if (techUsed[i]) { TechnologyApplicability.RectifyDependencies_HevUsed(techArr, settings.OperatingModes); }
                    break;
                case TI.MS1:
                case TI.MS2:
                case TI.MS5:
                    break;
                case TI.ROLL:
                case TI.LDB:
                case TI.SAXU:
                case TI.SAXL:
                    break;
                case TI.AERO:
                    break;
            } 
        }
        static void RectifyDependencies_DslUsed(bool[] techAE)
        {
            TechnologyApplicability.RectifyDependencies_DisableAllEng(techAE);
        }
        static void RectifyDependencies_HevUsed(bool[] techAE, OperatingModes opModes)
        {
            if (opModes.ShutoffEngineTechsAfterHEV)       { TechnologyApplicability.RectifyDependencies_DisableAllEng (techAE); }
            else                                          { techAE[TI.CBRST] = false; }     
            if (opModes.ShutoffTransmissionTechsAfterHEV) { TechnologyApplicability.RectifyDependencies_DisableAllTrn (techAE); }
            if (opModes.ShutoffElecTechsAfterHEV)         { TechnologyApplicability.RectifyDependencies_DisableAllElec(techAE, true); }
            TechnologyApplicability.RectifyDependencies_DisableAllHev(techAE);
        }
        static void RectifyDependencies_DisableAllEng(bool[] techAE)
        {
            TechnologyApplicability.RectifyDependencies_DisableSohcEng(techAE);
            TechnologyApplicability.RectifyDependencies_DisableDohcEng(techAE);
            TechnologyApplicability.RectifyDependencies_DisableOhvEng (techAE);
            techAE[TI.LUB  ] = false;
            techAE[TI.EFR  ] = false;
            techAE[TI.SGDI ] = false;
            techAE[TI.CBRST] = false;
            techAE[TI.TRBDS] = false;
            techAE[TI.EGRB ] = false;
            techAE[TI.DSLC ] = false;
            techAE[TI.DSLT ] = false;
        }
        static void RectifyDependencies_DisableSohcEng(bool[] techAE)
        {
            techAE[TI.CCPS ] = false;
            techAE[TI.DVVLS] = false;
            techAE[TI.DEACS] = false;
        }
        static void RectifyDependencies_DisableDohcEng(bool[] techAE)
        {
            techAE[TI.ICP  ] = false;
            techAE[TI.DCP  ] = false;
            techAE[TI.DVVLD] = false;
            techAE[TI.CVVL ] = false;
            techAE[TI.DEACD] = false;
        }
        static void RectifyDependencies_DisableOhvEng(bool[] techAE)
        {
            techAE[TI.DEACO] = false;
            techAE[TI.CCPO ] = false;
            techAE[TI.DVVLO] = false;
            techAE[TI.CDOHC] = false;
        }
        static void RectifyDependencies_DisableDeacEng(bool[] techAE)
        {
            techAE[TI.DEACS] = false;
            techAE[TI.DEACD] = false;
            techAE[TI.DEACO] = false;
        }
        static void RectifyDependencies_DisableAllTrn(bool[] techAE)
        {
            TechnologyApplicability.RectifyDependencies_DisableManualTrn(techAE);
            TechnologyApplicability.RectifyDependencies_DisableAutoTrn  (techAE);
        }
        static void RectifyDependencies_DisableManualTrn(bool[] techAE)
        {
            techAE[TI.MAN6] = false;
        }
        static void RectifyDependencies_DisableAutoTrn(bool[] techAE)
        {
            techAE[TI.IATC ] = false;
            techAE[TI.CVT  ] = false;
            techAE[TI.NAUTO] = false;
            techAE[TI.DCTAM] = false;
        }
        static void RectifyDependencies_DisableAllElec(bool[] techAE, bool disableCISG)
        {
            techAE[TI.EPS ] = false;
            techAE[TI.IACC] = false;
            techAE[TI.MHEV] = false;
            techAE[TI.BISG] = false;
            if (disableCISG) { techAE[TI.CISG] = false; }
        }
        static void RectifyDependencies_DisableAllHev(bool[] techAE)
        {
            techAE[TI.PSHEV] = false;
            techAE[TI.MHEV2] = false;
            techAE[TI.PHEV ] = false;
        }
        static void RectifyDependencies_EnableHev(bool[] techEnabled, bool[] techUsed, Vehicle vehicle, ModelYear year,
            int techIdx, ModelingSettings settings)
        {
            bool[] techAvailable = vehicle.ModelingData.TechAvailable;
            bool hevSatisfied =
                (!techAvailable[TI.IATC ] || techUsed[TI.IATC ]) &&
                (!techAvailable[TI.CVT  ] || techUsed[TI.CVT  ]) &&
                (!techAvailable[TI.NAUTO] || techUsed[TI.NAUTO]) &&
                (!techAvailable[TI.DCTAM] || techUsed[TI.DCTAM]);
            if (!hevSatisfied)
            {   
                TechnologyApplicability.RectifyDependencies_DisableAllHev(techEnabled);
            }
            else if (techIdx != -1)
            {   
                TechnologyApplicability.Enable(vehicle, year, TI.PSHEV, settings);
                if (!techUsed[TI.CVT])
                {   
                    TechnologyApplicability.Enable(vehicle, year, TI.MHEV2, settings);
                }
                TechnologyApplicability.Enable(vehicle, year, TI.PHEV, settings);
            }
        }
        #endregion
        #region 
        #region 
        static double GetSynergy(int techIndex, bool[] techUsedOrApplied, bool[] techSuperseded, double[] techSynergies,
            SynergyType[] synergyType, TechnologyCollection techGroup, ulong techKey)
        {
            double synergy = 0.0;
            for (int i = 0; i < TI.TechnologyCount; i++)
            {
                if (techUsedOrApplied[i] &&
                    (synergyType[i] != SynergyType.Physical || !techSuperseded[i])) { synergy += techSynergies[i]; }
            }
            if (techGroup != null && techKey > 0)
            {
                for (int i = 0, techCount = techGroup.Count; i < techCount; i++)
                {   
                    if ((techKey & TechKeys[i]) != 0)
                    {   
                        if (techGroup[i].Index >= techIndex) { break; }
                        synergy += techSynergies[techGroup[i].Index];
                    }
                } 
            }
            return synergy;
        }
        static double GetSynergy(int techIndex, bool[] techApplied, bool[] techSuperseded, double[] techSynergies,
            SynergyType[] synergyType)
        {
            double synergy = 0.0;
            for (int i = 0; i < techIndex; i++)
            {
                if (techApplied[i] &&
                    (synergyType[i] != SynergyType.Physical || !techSuperseded[i])) { synergy += techSynergies[i]; }
            }
            return synergy;
        }
        #endregion
        #region 
        public static double GetImprovement(Technology tech, TechnologyClass techClass, Estimates estimate, bool[] techUsed,
            bool[] techSuperseded, bool clipAtZero)
        {
            return GetImprovement(tech, techClass, estimate, techUsed, techSuperseded, null, 0, clipAtZero);
        }
        public static double GetImprovement(Technology tech, TechnologyClass techClass, Estimates estimate, bool[] techUsed,
            bool[] techSuperseded, TechnologyCollection techGroup, ulong techKey, bool clipAtZero)
        {
            TechnologyAttributes ta = tech.Attributes[techClass];
            double imprv = (estimate == Estimates.Low) ? ta.FcLow :
                (estimate == Estimates.High) ? ta.FcHigh : (ta.FcLow + ta.FcHigh) / 2;
            double synergy = GetSynergy(tech.Index, techUsed, techSuperseded, tech.ImprvSynergies[techClass].Synergies,
                tech.ImprvSynergies[techClass].Type, techGroup, techKey);
            imprv += synergy;
            return (clipAtZero && imprv < 0) ? 0 : imprv;
        }
        #endregion
        #region 
        public static double GetCost(Technology tech, Vehicle veh, Industry data, Estimates estimate, ModelYear year,
            int startYear, bool adjustForTechApplied, bool clipAtZero)
        {
            return GetCost(tech, veh, data, estimate, year, startYear, null, 0, adjustForTechApplied, clipAtZero);
        }
        public static double GetCost(Technology tech, Vehicle veh, Industry data, Estimates estimate, ModelYear year,
            int startYear, TechnologyCollection techGroup, ulong techKey, bool clipAtZero)
        {
            return GetCost(tech, veh, data, estimate, year, startYear, techGroup, techKey, false, clipAtZero);
        }
        static double GetCost(Technology tech, Vehicle veh, Industry data, Estimates estimate, ModelYear year,
            int startYear, TechnologyCollection techGroup, ulong techKey, bool adjustForTechApplied, bool clipAtZero)
        {
            VehicleModelingData vmd = veh.ModelingData;
            double techCost     = GetEstimatedCost(tech, veh, estimate);
            double learningRate = GetLearningRate(tech, veh, data, year, startYear);
            double synergy      = (adjustForTechApplied) ?
                GetSynergy(tech.Index, vmd.TechApplied, vmd.TechSuperseded, tech.CostSynergies[veh.TechnologyClass].Synergies,
                           tech.CostSynergies[veh.TechnologyClass].Type) :
                GetSynergy(tech.Index, vmd.TechApplied, vmd.TechSuperseded, tech.CostSynergies[veh.TechnologyClass].Synergies,
                           tech.CostSynergies[veh.TechnologyClass].Type, techGroup, techKey);
            techCost = techCost * learningRate + synergy;
            return (clipAtZero && techCost < 0) ? 0 : techCost;
        }
        static double GetEstimatedCost(Technology tech, Vehicle veh, Estimates estimate)
        {
            TechnologyClass      techClass = veh.TechnologyClass;
            TechnologyAttributes ta        = tech.Attributes[techClass];
            double               techCost  = (estimate == Estimates.Low ) ? ta.CostLow  :
                (estimate == Estimates.High) ? ta.CostHigh : (ta.CostHigh + ta.CostLow) / 2;
            if (TI.IsMaterialSubstitution(tech.Index))
            {   
                techCost *= (veh.Description.BaseWeight * ta.Aux);
            }
            else if (TI.IsCylinderBased(tech.Index))
            {   
                techCost *= veh.Engine.Description.Cylinders;
            }
            else if (TI.IsConfigurationBased(tech.Index))
            {   
                string engConfig = veh.Engine.Description.Configuration.ToUpper();
                int costMult = (engConfig == "R" || engConfig == "I") ? 1 :
                    (engConfig == "F" || engConfig == "H" || engConfig == "V") ? 2 : (engConfig == "W") ? 4 : 0;
                techCost *= costMult;
            }
            return techCost;
        }
        static double GetLearningRate(Technology tech, Vehicle veh, Industry data, ModelYear year,
            int startYear)
        {
            TechnologyClass techClass = veh.TechnologyClass;
            TechnologyAttributes   ta = tech.Attributes[techClass];
            if (ta.LearningRate != 0 && ta.LearningStartYear < year.Year)
            {
                if (ta.LearningType == LearningType.Volume && ta.LearningThreshold >= 1)
                {   
                    double          volume      = 0;
                    RCDouble[]      techVolumes = data.ModelingData.TechUsedSales;
                    RegulatoryClass regClass    = veh.RegClass;
                    if (regClass == RegulatoryClass.LightTruck)
                    {   
                        volume = (TI.IsDieselization(tech.Index)) ?
                            techVolumes[TI  .DSLT ][RegulatoryClass.LightTruck] +
                            techVolumes[TI  .DSLC ][RegulatoryClass.LightTruck] :
                            techVolumes[tech.Index][RegulatoryClass.LightTruck];
                    }
                    else if (regClass != RegulatoryClass.Unregulated)
                    {   
                        volume = (TI.IsDieselization(tech.Index)) ?
                            techVolumes[TI  .DSLT ][RegulatoryClass.DomesticAuto] +
                            techVolumes[TI  .DSLC ][RegulatoryClass.DomesticAuto] +
                            techVolumes[TI  .DSLT ][RegulatoryClass.ImportedAuto] +
                            techVolumes[TI  .DSLC ][RegulatoryClass.ImportedAuto] :
                            techVolumes[tech.Index][RegulatoryClass.DomesticAuto] +
                            techVolumes[tech.Index][RegulatoryClass.ImportedAuto];
                    }
                    if (volume >= ta.LearningThreshold)
                    {
                        return GetLearningRate_Volume(volume, ta.LearningThreshold, ta.LearningRate);
                    }
                }
                else if (ta.LearningType == LearningType.Time)
                {   
                    return GetLearningRate_Time(year.Year, startYear, ta.LearningStartYear, ta.LearningRate);
                }
            }
            return 1;
        }
        static double GetLearningRate_Volume(double volume, double threshold, double rate)
        {
            return Math.Pow(1.0 - rate, Math.Min(Math.Floor(volume / threshold), 2));
        }
        static double GetLearningRate_Time(int year, int startYear, int learningStartYear, double rate)
        {
            if (startYear < learningStartYear) { startYear = learningStartYear; }
            return Math.Pow(1.0 - rate, year - startYear);
        }
        #endregion
        #endregion
        public static bool CanBackfill(int techIndex, TechnologyCollection techs, bool[] techUsed)
        {
            for (int i = 0, count = techs.Count; i < count; i++)
            {   
                if (!CanBackfillHelper(techIndex, techs[i].Index)) { return false; }
            }
            for (int i = 0; i < TI.TechnologyCount; i++)
            {   
                if (techUsed[i] && !CanBackfillHelper(techIndex, i)) { return false; }
            }
            return true;
        }
        public static bool CanBackfill(int techIndex, TechnologyCollection techGroup, ulong techKey, bool[] techUsed)
        {
            int trueTechIndex = techGroup[techIndex].Index;
            for (int i = 0, techCount = techGroup.Count; i < techCount; i++)
            {   
                if ((techKey & TechKeys[i]) != 0)
                {   
                    if (!CanBackfillHelper(trueTechIndex, techGroup[i].Index)) { return false; }
                }
            }
            for (int i = 0; i < TI.TechnologyCount; i++)
            {   
                if (techUsed[i] && !CanBackfillHelper(trueTechIndex, i)) { return false; }
            }
            return true;
        }
        static bool CanBackfillHelper(int techIndex, int selTechIndex)
        {
            if (techIndex == selTechIndex) { return false; }
            int[] techDisallowBackfill = TechDisallowBackfill[selTechIndex];
            for (int j = 0, disallowCount = techDisallowBackfill.Length; j < disallowCount; j++)
            {
                if (techIndex == techDisallowBackfill[j])
                {   
                    return false;
                }
            }
            return true;
        }
        public static bool IsTechValid(ManufacturerModelingData mmd, VehicleModelingData vmd, RegulatoryClass regClass,
            int techIdx, bool ignorePhaseIn, bool backfilling, ManufacturerModelingData lMmd, out bool lingering)
        {
            lingering = false;
            bool valid = (vmd.TechEnabled[techIdx] && !vmd.TechUsed[techIdx] &&
                (backfilling || (!vmd.TechReserved[techIdx] && !vmd.TechIgnored[techIdx] &&
                (ignorePhaseIn || !mmd.TechExhausted[techIdx][regClass]))));
            if (valid)
            {
                return true;
            }
            else if (vmd.TechLingering[techIdx]) 
            {   
                lingering = (!vmd.TechUsed[techIdx]) && 
                    (ignorePhaseIn || !lMmd.TechExhausted[techIdx][regClass]);
                return lingering;
            }
            return false;
        }
        public static void InstallTechnologyOnVehicle(Vehicle veh, Technology tech, ModelYear year, ModelingSettings settings,
            bool backfilling, bool isExhausted)
        {
            VehicleModelingData     vmd = veh.ModelingData;
            VehicleDescription      vd  = veh.Description;
            EngineDescription       ed  = veh.Engine.Description;
            TransmissionDescription td  = veh.Transmission.Description;
            vmd.TechApplied[tech.Index] = true;
            vmd.TechUsed   [tech.Index] = true;
            if (backfilling)
            {
                vmd.TechBackfilled[tech.Index] = true;
                if (isExhausted) { vmd.TechPhaseInBackfilled[tech.Index] = true; }
            }
            switch (tech.Index)
            {
                case TI.LUB:
                    ed.EngineOilViscosity = "LUB1";
                    break;
                case TI.CDOHC:
                    ed.ValvetrainDesign = "DOHC";
                    ed.ValvesPerCylinder = 4;
                    break;
                case TI.DVVLS:
                case TI.CVVL:
                    ed.ValveLift = "CV";
                    break;
                case TI.DVVLD:
                case TI.DVVLO:
                    ed.ValveLift = "SVI";
                    break;
                case TI.TRBDS:
                    ed.Aspiration = 'S';
                    break;
                case TI.DSLC:
                case TI.DSLT:
                    ed.ValveActuationTiming = "F";
                    ed.ValveLift = "F";
                    ed.Fuel = "D";
                    ed.FuelSystem = "DI";
                    ed.Aspiration = 'T';
                    ed.Cycle = 'D';
                    ed.AirFuelRatio = 0;
                    ed.MinCompressionRatio = 0;
                    ed.MaxCompressionRatio = 0;
                    break;
                case TI.MAN6:
                    td.NumForwardGears = 6;
                    break;
                case TI.IATC:
                    td.Type    = "T";
                    td.Control = "A";
                    td.Logic   = "A";
                    break;
                case TI.CVT:
                    td.Type = "CVT";
                    break;
                case TI.NAUTO:
                    td.NumForwardGears = 6;
                    td.Control = "A";
                    td.Logic = "";
                    break;
                case TI.DCTAM:
                    td.Type = "C";
                    td.Control = "A";
                    td.Logic = "";
                    break;
                case TI.PSHEV:
                case TI.MHEV2:
                case TI.PHEV:
                    vd.HybridType = "E";
                    break;
            }
            TechnologyApplicability.Enable(veh, year, tech.Index, settings);
        }
        public static void CheckPhaseIn(ManufacturerModelingData mmd, RegulatoryClass regClass, Technology tech, ModelYear year)
        {
            if (mmd.TechExhausted[tech.Index][regClass]) { return; }
            int[]  phaseInPairs  = new int[] {-1, -1, -1, -1};
            switch (tech.Index)
            {
                case TI.CCPS: case TI.ICP: case TI.DCP: case TI.CCPO:       
                    phaseInPairs[0] = TI.CCPS; phaseInPairs[1] = TI.ICP; phaseInPairs[2] = TI.DCP; phaseInPairs[3] = TI.CCPO;
                    break;
                case TI.DVVLS: case TI.DVVLD: case TI.CVVL: case TI.DVVLO:  
                    phaseInPairs[0] = TI.DVVLS; phaseInPairs[1] = TI.DVVLD; phaseInPairs[2] = TI.CVVL; phaseInPairs[3] = TI.DVVLO;
                    break;
                case TI.DEACS: case TI.DEACD: case TI.DEACO:                
                    phaseInPairs[0] = TI.DEACS; phaseInPairs[1] = TI.DEACD; phaseInPairs[2] = TI.DEACO;
                    break;
                case TI.DSLC: case TI.DSLT:                                 
                    phaseInPairs[0] = TI.DSLC; phaseInPairs[1] = TI.DSLT;
                    break;
                case TI.PSHEV: case TI.MHEV2: case TI.PHEV:                 
                    phaseInPairs[1] = TI.PSHEV; phaseInPairs[2] = TI.MHEV2; phaseInPairs[3] = TI.PHEV;
                    break;
                case TI.SAXL: case TI.SAXU:                                 
                    phaseInPairs[0] = TI.SAXL; phaseInPairs[1] = TI.SAXU;
                    break;
                default:                                                    
                    phaseInPairs[0] = tech.Index;
                    break;
            }
            double appliedSales = 0D;
            for (int i = 0; i < phaseInPairs.Length; i++)
            {
                if (phaseInPairs[i] != -1) { appliedSales += mmd.TechAppliedSales[phaseInPairs[i]][regClass]; }
            }
            double sales = mmd.Sales[regClass];
            if (sales > 0 && appliedSales >= sales * tech.GetPhaseIn(year.Index))
            {   
                for (int i = 0; i < phaseInPairs.Length; i++)
                {
                    if (phaseInPairs[i] != -1) { mmd.TechExhausted[phaseInPairs[i]][regClass] = true; }
                }
            }
        }
        #endregion
        #region 
        static readonly int[][] TechSupersedes;
        static readonly int[][] TechDisallowBackfill;
        public static readonly ulong[] TechKeys;
        public static TechnologyType[] TechTypes;
        public static int TechTypeCount;
        #endregion
    }
}

