using System;
using System.Drawing;
using Volpe.Cafe.Data;
using Volpe.Cafe.IO;
using Volpe.Utils;
using CGlobal = Volpe.Cafe.Global;
namespace Volpe.Cafe.Reporter.ReportTypes
{
    public class EAReport : Output
    {
        #region 
        public EAReport(string path, EncryptionSettings cryptoSettings)
            : base(path, TemplateType.EAReport, cryptoSettings, true)
        {
            base._autoBorder      = true;
            base._autoChart       = true;
            base._autoMerge       = true;
            base._autoSummarize   = true;
            base._fillColor       = XlColor.Automatic;
            this._fitToPage       = new Size(1, -1);
            base._font            = new XlFont("Arial");
            base._headerAlignment = new XlTextAlignment(XlHAlign.Center, XlVAlign.Center);
            base._headerFillColor = XlColor.Automatic;
            base._headerFont      = new XlFont("Arial", 10, false, true, false);
            base._landscape       = true;
            base._margins         = new XlMargins(0.5);
            base._numberFormat    = "#,##0.00";
            base._repeatHeaders   = false;
            base._textAlignment   = new XlTextAlignment(XlHAlign.General, XlVAlign.Center);
            this._initialized = false;
            this._numScen     = 0;
            this._minYear     = 0;
            this._maxYear     = 0;
            this._numSects    = 0;
            this._tableRows   = 0;
        }
        #endregion
        #region 
        public void ParseData(int numScen, int scenarioIndex, ModelYear year, Industry data)
        {
            if (!this._initialized)
            {
                this._numScen   = numScen;
                this._minYear   = data.MinYear.Year;
                this._maxYear   = data.MaxYear.Year;
                this._tableRows = this._maxYear - this._minYear + 5;
                this._numSects  = this._maxYear - this._minYear + 38;
                this._scenarioAdded = new bool[numScen];
                this.SetupBuffer("Fuel Consumption (k. gal)");
                this.SetupBuffer("CO2 (mmT)" );
                this.SetupBuffer("CO (tons)" );
                this.SetupBuffer("VOC (tons)");
                this.SetupBuffer("NOX (tons)");
                this.SetupBuffer("PM (tons)" );
                this.SetupBuffer("SOX (tons)");
                this._initialized = true;
            }
            this.WriteData(scenarioIndex, year, data);
            this._scenarioAdded[scenarioIndex] = true;
        }
        private void SetupBuffer(string name)
        {
            int rows = this._numScen * (this._tableRows + 1) - 1;
            int cols = this._numSects;
            this.AddBuffer(rows, cols);
            this.Buffer.Header = new XlCell(0, 1);
            this.Buffer.Name = name;
            this.Buffer.Title = "EA Summary -- &[Tab]";
            int numYears = this._maxYear - this._minYear + 1;
            for (int i = 0; i < this._numScen; i++)
            {
                int rowOffset = i * (this._tableRows + 1);
                this.Buffer[0 + rowOffset, 0] = name + " -- Sn " + i + ", Years " +
                    this._minYear + " - " + this._maxYear;
                this.Buffer[1 + rowOffset, 0] = "Model\nYears";
                this.Buffer[1 + rowOffset, 1] = "Calendar Years";
                for (int j = 0; j < numYears; j++)
                {
                    this.Buffer[3 + j + rowOffset, 0] = this._minYear + j;
                }
                this.Buffer[3 + numYears + rowOffset, 0] = "Total";
                for (int j = 0; j < numYears + 35; j++)
                {
                    this.Buffer[2 + rowOffset, 1 + j] = this._minYear + j;
                }
                this.Buffer[1 + rowOffset, CGlobal.CY + numYears] = "Total";
            }
        }
        private void WriteData(int scenarioIndex, ModelYear year, Industry data)
        {
            AggregateEffectsData ied = data.ModelingData.EffectsData;
            int zeroBasedMyIdx = year.Year - this._minYear;
            int numMy = this._maxYear - this._minYear + 1;
            int row = 3 + zeroBasedMyIdx + scenarioIndex * (this._tableRows + 1);
            int endRow = 3 + numMy + scenarioIndex * (this._tableRows + 1);
            for (int i = 0; i < CGlobal.CY; i++)
            {
                int col = 1 + zeroBasedMyIdx + i;
                this.WriteValue(0, ied.Gallons     [i].Total, row, col, endRow);    
                this.WriteValue(1, ied.CO2Emissions[i].Total, row, col, endRow);    
                this.WriteValue(2, ied.COEmissions [i].Total, row, col, endRow);    
                this.WriteValue(3, ied.VOCEmissions[i].Total, row, col, endRow);    
                this.WriteValue(4, ied.NOXEmissions[i].Total, row, col, endRow);    
                this.WriteValue(5, ied.PMEmissions [i].Total, row, col, endRow);    
                this.WriteValue(6, ied.SOXEmissions[i].Total, row, col, endRow);    
            } 
        }
        private void WriteValue(int bufIndex, double value, int row, int col, int endRow)
        {
            object cell;
            this.Buffers[bufIndex][row, col]    = value;
            cell = this.Buffers[bufIndex][row, this._numSects - 1];
            this.Buffers[bufIndex][row, this._numSects - 1] = (cell == null) ? value : (double)cell + value;
            cell = this.Buffers[bufIndex][endRow, col];
            this.Buffers[bufIndex][endRow, col] = (cell == null) ? value : (double)cell + value;
        }
        #region 
        public override void WriteBuffers(Input template, bool templateFirstBuffer)
        {
            int bufCount = this.BufferCount;
            int remove = 0;
            for (int i = 0; i < this._numScen; i++)
            {
                if (!this._scenarioAdded[i])
                {    
                    int sIdx = (i + 1) * (this._tableRows + 1);
                    int eIdx = this._numScen * (this._tableRows + 1) - 1;
                    for (int bufIdx = 0; bufIdx < bufCount; bufIdx++)
                    {
                        OutputBuffer buf = this.Buffers[bufIdx];
                        for (int j = sIdx; j < eIdx; j++)
                        {
                            int j2 = j - this._tableRows - 1;
                            for (int k = 0; k < this._numSects; k++)
                            {
                                buf[j2, k] = buf[j, k];
                            }
                        }
                    }
                    remove++;
                }
            }
            if (remove > 0)
            {
                this._numScen -= remove;
                for (int bufIdx = 0; bufIdx < bufCount; bufIdx++)
                {
                    this.Buffers[bufIdx].ResizeBuffer(
                        new XlSize(this._numScen * (this._tableRows + 1) - 1, this._numSects), true);
                }
            }
            base.WriteBuffers(template, templateFirstBuffer);
        }
        protected override void MergeHeaders()
        {
            if (this._numSects < 1) { return; }
            for (int i = 0; i < this._numScen; i++)
            {
                int rowOffset = i * (this._tableRows + 1);
                int snDivRow = rowOffset + this._tableRows + 1;        
                this._xlu.MergeCells(new XlCell(1 + rowOffset, 1), new XlCell(1 + rowOffset, this._numSects));
                this._xlu.MergeCells(new XlCell(2 + rowOffset, 2), new XlCell(2 + rowOffset, this._numSects - 1));
                this._xlu.MergeCells(new XlCell(2 + rowOffset, 1), new XlCell(3 + rowOffset, 1));
                this._xlu.MergeCells(new XlCell(2 + rowOffset, this._numSects), new XlCell(3 + rowOffset, this._numSects));
                if (i < this._numScen - 1)
                {
                    this._xlu.MergeCells(new XlCell(snDivRow, 1), new XlCell(snDivRow, this._numSects));
                }
            } 
        }
        protected override void SetBorders(XlCell rHeader1, XlCell rHeader2, XlCell cHeader1, XlCell cHeader2,
            XlCell dataCell1, XlCell dataCell2)
        {
            if (this._numSects < 1)
            {
                return;
            }
            for (int i = 0; i < this._numScen; i++)
            {
                int rowOffset = i * (this._tableRows + 1);
                int totalsRow = rowOffset + this._tableRows;
                this._xlu.SetBorders(new XlCell(1 + rowOffset, 1), new XlCell(totalsRow, this._numSects),
                    false, true, true, false, XlColor.DarkBlue);
                this._xlu.SetBorders(new XlCell(1 + rowOffset, 1), new XlCell(totalsRow, this._numSects),
                    true, false, true, true, XlColor.DarkBlue);
                this._xlu.SetBorders(new XlCell(4 + rowOffset, 2),
                    new XlCell(totalsRow - 1, this._numSects - 1), false, true, false, false, XlColor.DarkBlue);
                this._xlu.SetBorders(new XlCell(4 + rowOffset, 1),
                    new XlCell(totalsRow - 1, this._numSects), true, true, false, false, true, true,
                    XlColor.DarkBlue);
                this._xlu.SetBorders(new XlCell(2 + rowOffset, 2),
                    new XlCell(totalsRow, this._numSects - 1), false, false, true, true, true, true,
                    XlColor.DarkBlue);
            } 
        }
        protected override void SetNumberFormat(XlCell dataCell1, XlCell dataCell2)
        {
            base.SetNumberFormat(dataCell1, dataCell2);
            if (this._numSects < 1)
            {
                return;
            }
            for (int i = 0; i < this._numScen; i++)
            {
                int rowOffset = i * (this._tableRows + 1);
                int snDivRow = rowOffset + this._tableRows + 1;        
                this._xlu.SetNumberFormat(new XlCell(1 + rowOffset, 1), new XlCell(snDivRow, 1),
                    XlNumberFormats.General);
                this._xlu.SetNumberFormat(new XlCell(3 + rowOffset, 1), new XlCell(3 + rowOffset, this._numSects),
                    XlNumberFormats.General);
            } 
        }
        protected override void SetColor(XlCell rHeader1, XlCell rHeader2, XlCell cHeader1, XlCell cHeader2,
            XlCell dataCell1, XlCell dataCell2)
        {
            base.SetColor(rHeader1, rHeader2, cHeader1, cHeader2, dataCell1, dataCell2);
            if (this._numSects < 1)
            {
                return;
            }
            for (int i = 0; i < this._numScen; i++)
            {
                int rowOffset = i * (this._tableRows + 1);
                int totalsRow = rowOffset + this._tableRows;
                this._xlu.SetFillColor(new XlCell(1 + rowOffset, 1), new XlCell(1 + rowOffset, this._numSects),
                    XlColor.LightYellow);
                this._xlu.SetFillColor(new XlCell(totalsRow, 1), new XlCell(totalsRow, this._numSects - 1),
                    XlColor.LightGreen);
            } 
        }
        protected override void SetFont(XlCell rHeader1, XlCell rHeader2, XlCell cHeader1, XlCell cHeader2,
            XlCell dataCell1, XlCell dataCell2)
        {
            base.SetFont(rHeader1, rHeader2, cHeader1, cHeader2, dataCell1, dataCell2);
            if (this._numSects < 1)
            {
                return;
            }
            for (int i = 0; i < this._numScen; i++)
            {
                int rowOffset = i * (this._tableRows + 1);
                this._xlu.SetFont(new XlCell(1 + rowOffset, 1), new XlCell(3 + rowOffset, this._numSects),
                    this._headerFont);
            } 
        }
        protected override void AlignText(XlCell rHeader1, XlCell rHeader2, XlCell cHeader1, XlCell cHeader2,
            XlCell dataCell1, XlCell dataCell2)
        {
            base.AlignText(rHeader1, rHeader2, cHeader1, cHeader2, dataCell1, dataCell2);
            if (this._numSects < 1)
            {
                return;
            }
            for (int i = 0; i < this._numScen; i++)
            {
                int rowOffset = i * (this._tableRows + 1);
                this._xlu.AlignText(new XlCell(1 + rowOffset, 1), new XlCell(3 + rowOffset, this._numSects), 
                    this._headerAlignment);
                this._xlu.AlignText(new XlCell(2 + rowOffset, this._numSects),
                    new XlCell(3 + rowOffset, this._numSects), new XlTextAlignment(XlHAlign.Center));
            } 
        }
        #endregion
        #endregion
        #region 
        private bool _initialized;        
        private int _numScen;
        private int _minYear;
        private int _maxYear;
        private int _numSects;
        private int _tableRows;            
        private bool[] _scenarioAdded;    
        #endregion
    }
}

