using System;
using System.Drawing;
using Volpe.Cafe.Collections;
using Volpe.Cafe.Data;
using Volpe.Cafe.IO;
using Volpe.Utils;
using CGlobal = Volpe.Cafe.Global;
namespace Volpe.Cafe.Reporter.ReportTypes
{
    public class ScenSummaryReport : Output
    {
        #region 
        public ScenSummaryReport(string path, EncryptionSettings cryptoSettings)
            : base(path, TemplateType.ScenarioSummaryReport, cryptoSettings, true)
        {
            base._autoBorder = true;
            base._autoChart = true;
            base._autoMerge = true;
            base._autoSummarize = true;
            base._fillColor = XlColor.Automatic;
            this._fitToPage = new Size(1, -1);
            base._font = new XlFont("Arial");
            base._headerAlignment = new XlTextAlignment(XlHAlign.Center, XlVAlign.Center);
            base._headerFillColor = XlColor.Automatic;
            base._headerFont = new XlFont("Arial", 10, false, true, false);
            base._landscape = true;
            base._margins = new XlMargins(0.5);
            base._numberFormat = "#,##0";
            base._repeatHeaders = false;
            base._textAlignment = new XlTextAlignment(XlHAlign.General, XlVAlign.Center);
            this._initialized = false;
        }
        #endregion
        #region 
        public void ParseData(int scenCount, int scenIndex, string scenName, ModelYear year, Industry data,
            Industry baseData)
        {
            if (!this._initialized)
            {
                this._scenCount = scenCount;
                this._minYear = data.MinYear.Year;
                this._maxYear = data.MaxYear.Year;
                this._yearCount = this._maxYear - this._minYear + 1;
                this._mfrCount = data.ManufacturerCount;
                this._scenarioAdded = new bool[scenCount];
                this._yearCostsByMfr = new double[scenCount][];
                for (int i = 0; i < scenCount; i++) { this._yearCostsByMfr[i] = new double[this._mfrCount]; }
                this._yearCostsBigN         = new double[scenCount];
                this._yearFuelSavingsBigN   = new double[scenCount];
                this._yearTotalBenefitsBigN = new double[scenCount];
                this._yearNetBenefitsBigN   = new double[scenCount];
                this._yearCostsAll          = new double[scenCount];
                this._yearFuelSavings       = new double[scenCount];
                this._yearTotalBenefits     = new double[scenCount];
                this._yearNetBenefits       = new double[scenCount];
                this.SetupBuffer(data.Manufacturers);
                this._initialized = true;
            }
            this.WriteData(scenIndex, Global.GetTitleCase(scenName), year, data, baseData);
        }
        private void SetupBuffer(ManufacturerCollection mfrs)
        {
            int rows = 2 * this._mfrCount + 24;
            int cols = (this._scenCount - 1) * (this._yearCount + 1) + 1;
            this.AddBuffer(rows, cols);
            this.Buffer.Header = new XlCell(3, 1);
            this.Buffer.Name = "Scenario Summary";
            this.Buffer.Title = "&[Tab]";
            this._bigN = 0;
            for (int i = 0; i < this._mfrCount; i++)
            {
                if (mfrs[i].Description.Optimize) { this._bigN++; }
            }
            int r2 = this._mfrCount + 20;   
            this.Buffer[0, 0] = "Total Costs ($M) and Benefits";
            this.Buffer[1, 0] = "Manufacturer";
            this.Buffer[r2, 0] = "Average Vehicle Price Increases";
            this.Buffer[r2 + 1, 0] = "Manufacturer";
            for (int i = 0; i < this._scenCount - 1; i++)
            {
                int colOffset = i * (this._yearCount + 1);
                for (int j = 0; j < this._yearCount; j++)
                {
                    this.Buffer[2,      colOffset + j + 1] = this._minYear + j;
                    this.Buffer[r2 + 2, colOffset + j + 1] = this._minYear + j;
                }
                this.Buffer[2, colOffset + this._yearCount + 1] = "Total";
            }
            for (int i = 0; i < this._mfrCount; i++)
            {
                string mfrName = mfrs[i].Description.Name;
                mfrName = (mfrName.Length == 3) ? mfrName : Global.GetTitleCase(mfrName);
                this.Buffer[3 + i, 0] = mfrName;
                this.Buffer[r2 + 3 + i, 0] = mfrName;
            }
            int row = 3 + this._mfrCount;
            this.Buffer[r2 + row, 0] = "Average";
            this.Buffer[row++, 0] = "Big-" + this._bigN + " Summary";
            this.Buffer[row++, 0] = "Total";
            this.Buffer[row++, 0] = "Fuel Savings (BG)";
            this.Buffer[row++, 0] = "Preliminary Standard (MPG)";
            this.Buffer[row++, 0] = "Final Standard (MPG)";
            this.Buffer[row++, 0] = "Total Benefits";
            this.Buffer[row++, 0] = "Net Benefits";
            this.Buffer[row++, 0] = "Benefit:Cost Ratio";
            this.Buffer[row++, 0] = "Entire Industry Summary";
            this.Buffer[row++, 0] = "Total";
            this.Buffer[row++, 0] = "Fuel Savings (BG)";
            this.Buffer[row++, 0] = "Preliminary Standard (MPG)";
            this.Buffer[row++, 0] = "Final Standard (MPG)";
            this.Buffer[row++, 0] = "Total Benefits";
            this.Buffer[row++, 0] = "Net Benefits";
            this.Buffer[row++, 0] = "Benefit:Cost Ratio";
        }
        private void WriteData(int scenIndex, string scenName, ModelYear year, Industry data, Industry baseData)
        {
            int row1 = 1;
            int row2 = this._mfrCount + 21;
            int col = 1 + (this._yearCount + 1) * (scenIndex - 1) + (year.Year - this._minYear);
            if (year.Year == this._minYear)
            {
                this.Buffer[row1, col] = scenName;
                this.Buffer[row2, col] = scenName;
            }
            row1+=2; row2+=2;
            ManufacturerCollection mfrs     = data.Manufacturers;
            ManufacturerCollection baseMfrs = baseData.Manufacturers;
            double indTc    = 0;
            double indBigN  = 0;
            double indSales = 0, indBaseSales = 0;
            double indRc    = 0, indBaseRc    = 0;      
            for (int i = 0; i < this._mfrCount; i++)
            {
                Manufacturer mfr = mfrs[i], baseMfr = baseMfrs[i];
                ManufacturerModelingData mmd = mfr.ModelingData, baseMmd = baseMfr.ModelingData;
                double deltaTc   = (mmd.TechCost.Total - baseMmd.TechCost.Total) / 1000000;     
                double mfrRc     = mmd.RegCost.Total;                                           
                double mfrBaseRc = baseMmd.RegCost.Total;
                double deltaAveRc  = (mfrRc / mmd.Sales.Total - mfrBaseRc / baseMmd.Sales.Total);
                indSales     += mmd.Sales.Total;
                indBaseSales += baseMmd.Sales.Total;
                indTc += deltaTc;
                if (mfr.Description.Optimize) { indBigN += deltaTc; }
                indRc     += mfrRc;
                indBaseRc += mfrBaseRc;
                this.Buffer[row1++, col] = deltaTc;
                this.Buffer[row2++, col] = deltaAveRc;
                this._yearCostsByMfr[scenIndex][i] += deltaTc;
            }
            double fuelSavingsBigN = 0;
            for (int i = 0; i < this._mfrCount; i++)
            {
                Manufacturer mfr = data.Manufacturers[i];
                if (mfr.Description.Optimize)
                {
                    fuelSavingsBigN += (baseData.Manufacturers[i].ModelingData.FuelConsumption.Total -
                        mfr.ModelingData.FuelConsumption.Total) / 1000000;
                }
            }
            double prelimStandardBigN = Calculate.IndStandard(data, year, true, false, true);
            double standardBigN       = Calculate.IndStandard(data, year, true, false, false);
            double totalBenefitsBigN  = (Calculate.IndBenefits(baseData, true).Total - Calculate.IndBenefits(data, true).Total) / 1000;
            double netBenefitsBigN    = totalBenefitsBigN - indBigN;
            double bcrBigN            = totalBenefitsBigN / indBigN;
            AggregateEffectsData ied = data.ModelingData.EffectsData, baseIed = baseData.ModelingData.EffectsData;
            double fuelSavings       = (baseIed.RcGallons.Total - ied.RcGallons.Total) / 1000000;
            double prelimStandard    = Calculate.IndStandard(data, year, false, false, true);
            double standard          = Calculate.IndStandard(data, year, false, false, false);
            double totalBenefits     = (baseIed.RcDiscSocialBenefits.Total - ied.RcDiscSocialBenefits.Total) / 1000;
            double netBenefits       = totalBenefits - indTc;
            double bcr               = totalBenefits / indTc;
            row1++;     
            this.Buffer[row1++, col] = indBigN;
            this.Buffer[row1++, col] = fuelSavingsBigN;
            this.Buffer[row1++, col] = prelimStandardBigN;
            this.Buffer[row1++, col] = standardBigN;
            this.Buffer[row1++, col] = totalBenefitsBigN;
            this.Buffer[row1++, col] = netBenefitsBigN;
            this.Buffer[row1++, col] = (double.IsNaN(bcrBigN) || double.IsInfinity(bcrBigN)) ? "N/A" : bcrBigN.ToString();
            row1++;     
            this.Buffer[row1++, col] = indTc;
            this.Buffer[row1++, col] = fuelSavings;
            this.Buffer[row1++, col] = prelimStandard;
            this.Buffer[row1++, col] = standard;
            this.Buffer[row1++, col] = totalBenefits;
            this.Buffer[row1++, col] = netBenefits;
            this.Buffer[row1  , col] = (double.IsNaN(bcr) || double.IsInfinity(bcr)) ? "N/A" : bcr.ToString();
            double indDeltaAveRc   = (indRc / indSales - indBaseRc / indBaseSales);
            this.Buffer[row2, col] = indDeltaAveRc;
            this._yearCostsBigN        [scenIndex] += indBigN;          
            this._yearFuelSavingsBigN  [scenIndex] += fuelSavingsBigN;
            this._yearTotalBenefitsBigN[scenIndex] += totalBenefitsBigN;
            this._yearNetBenefitsBigN  [scenIndex] += netBenefitsBigN;
            this._yearCostsAll         [scenIndex] += indTc;            
            this._yearFuelSavings      [scenIndex] += fuelSavings;
            this._yearTotalBenefits    [scenIndex] += totalBenefits;
            this._yearNetBenefits      [scenIndex] += netBenefits;
        }
        private void WriteYearSumData(int scenIndex)
        {
            int row = 3;
            int col = 1 + (this._yearCount + 1) * (scenIndex - 1) + this._yearCount;
            for (int i = 0; i < this._mfrCount; i++)
            {
                this.Buffer[row++, col] = this._yearCostsByMfr[scenIndex][i];
            }
            row++;          
            this.Buffer[row++, col] = this._yearCostsBigN[scenIndex];
            this.Buffer[row++, col] = this._yearFuelSavingsBigN[scenIndex];
            row+=2;         
            this.Buffer[row++, col] = this._yearTotalBenefitsBigN[scenIndex];
            this.Buffer[row++, col] = this._yearNetBenefitsBigN[scenIndex];
            double bcrBigN = this._yearTotalBenefitsBigN[scenIndex] / this._yearCostsBigN[scenIndex];
            this.Buffer[row++, col] = double.IsNaN(bcrBigN) ? "N/A" : bcrBigN.ToString();
            row++;          
            this.Buffer[row++, col] = this._yearCostsAll[scenIndex];
            this.Buffer[row++, col] = this._yearFuelSavings[scenIndex];
            row+=2;         
            this.Buffer[row++, col] = this._yearTotalBenefits[scenIndex];
            this.Buffer[row++, col] = this._yearNetBenefits[scenIndex];
            double bcr = this._yearTotalBenefits[scenIndex] / this._yearCostsAll[scenIndex];
            this.Buffer[row, col] = double.IsNaN(bcr) ? "N/A" : bcr.ToString();
        }
        #region 
        public override void WriteBuffers(Input template, bool templateFirstBuffer)
        {
            for (int i = 1; i < this._scenCount; i++)
            {    
                this.WriteYearSumData(i);
            }
            base.WriteBuffers(template, templateFirstBuffer);
        }
        protected override void MergeHeaders()
        {
            int cols = (this._scenCount - 1) * (this._yearCount + 1) + 1;
            int row1 = 1;
            int row2 = this._mfrCount + 21;
            this._xlu.MergeCells(new XlCell(row1, 1), new XlCell(row1, cols));
            this._xlu.MergeCells(new XlCell(row1 + 1, 1), new XlCell(row1 + 2, 1));
            this._xlu.MergeCells(new XlCell(row2, 1), new XlCell(row2, cols));
            this._xlu.MergeCells(new XlCell(row2 + 1, 1), new XlCell(row2 + 2, 1));
            int wdth = this._yearCount + 1;
            for (int i = 0; i < this._scenCount - 1; i++)
            {
                this._xlu.MergeCells(new XlCell(row1 + 1, 2 + i * wdth), new XlCell(row1 + 1, 1 + (i + 1) * wdth));
                this._xlu.MergeCells(new XlCell(row2 + 1, 2 + i * wdth), new XlCell(row2 + 1, (i + 1) * wdth));
                this._xlu.MergeCells(new XlCell(row2 + 1, 1 + (i + 1) * wdth), new XlCell(row2 + 2, 1 + (i + 1) * wdth));
                int bigNSum = row1 + this._mfrCount + 3;
                int indSum  = bigNSum + 8;
                this._xlu.MergeCells(new XlCell(bigNSum, 2 + i * wdth), new XlCell(bigNSum, 1 + (i + 1) * wdth));
                this._xlu.MergeCells(new XlCell(indSum , 2 + i * wdth), new XlCell(indSum , 1 + (i + 1) * wdth));
            }
            this._xlu.MergeCells(new XlCell(row2 - 1, 1), new XlCell(row2 - 1, cols));
        }
        protected override void SetBorders(XlCell rHeader1, XlCell rHeader2, XlCell cHeader1, XlCell cHeader2,
            XlCell dataCell1, XlCell dataCell2)
        {
            base.SetBorders(rHeader1, rHeader2, cHeader1, cHeader2, dataCell1, dataCell2);
            int cols = (this._scenCount - 1) * (this._yearCount + 1) + 1;
            int row2 = this._mfrCount + 21;
            this._xlu.SetBorders(new XlCell(row2, 2), new XlCell(row2 + 2, cols), false, true, true, false, XlColor.DarkBlue);
            this._xlu.SetBorders(new XlCell(row2, 1), new XlCell(row2 + 2, cols), true, false, true, true, XlColor.DarkBlue);
            this._xlu.SetBorders(new XlCell(row2 - 1, 1), new XlCell(row2 - 1, cols), true, false, true, true, XlColor.DarkBlue);
        }
        protected override void SetNumberFormat(XlCell dataCell1, XlCell dataCell2)
        {
            base.SetNumberFormat(dataCell1, dataCell2);
            int cols = (this._scenCount - 1) * (this._yearCount + 1) + 1;
            int row2 = this._mfrCount + 21;
            this._xlu.SetNumberFormat(new XlCell(row2 - 15, 1), new XlCell(row2 - 15, cols), "#,##0.000");
            this._xlu.SetNumberFormat(new XlCell(row2 - 14, 1), new XlCell(row2 - 13, cols), "#,##0.0"  );
            this._xlu.SetNumberFormat(new XlCell(row2 - 10, 1), new XlCell(row2 - 10, cols), "#,##0.00" );
            this._xlu.SetNumberFormat(new XlCell(row2 - 7, 1), new XlCell(row2 - 7, cols), "#,##0.000");
            this._xlu.SetNumberFormat(new XlCell(row2 - 6, 1), new XlCell(row2 - 5, cols), "#,##0.0"  );
            this._xlu.SetNumberFormat(new XlCell(row2 - 2, 1), new XlCell(row2 - 2, cols), "#,##0.00" );
            this._xlu.SetNumberFormat(new XlCell(row2 + 2, 1), new XlCell(row2 + 2, cols - 1), XlNumberFormats.General);
        }
        protected override void SetColor(XlCell rHeader1, XlCell rHeader2, XlCell cHeader1, XlCell cHeader2,
            XlCell dataCell1, XlCell dataCell2)
        {
            base.SetColor(rHeader1, rHeader2, cHeader1, cHeader2, dataCell1, dataCell2);
            int cols = (this._scenCount - 1) * (this._yearCount + 1) + 1;
            int row1 = 1;
            int row2 = this._mfrCount + 21;
            this._xlu.SetFillColor(new XlCell(row1, 1), new XlCell(row1, cols), XlColor.LightYellow);
            this._xlu.SetFillColor(new XlCell(row2, 1), new XlCell(row2, cols - 1), XlColor.LightYellow);
            int bigNSum = row1 + this._mfrCount + 3;
            int indSum  = bigNSum + 8;
            this._xlu.SetFillColor(new XlCell(bigNSum, 1), new XlCell(bigNSum, 1), XlColor.PaleBlue);
            this._xlu.SetFillColor(new XlCell(indSum , 1), new XlCell(indSum , 1), XlColor.PaleBlue);
        }
        protected override void SetFont(XlCell rHeader1, XlCell rHeader2, XlCell cHeader1, XlCell cHeader2,
            XlCell dataCell1, XlCell dataCell2)
        {
            base.SetFont(rHeader1, rHeader2, cHeader1, cHeader2, dataCell1, dataCell2);
            int cols = (this._scenCount - 1) * (this._yearCount + 1) + 1;
            int row2 = this._mfrCount + 21;
            this._xlu.SetFont(new XlCell(row2, 1), new XlCell(row2 + 2, cols - 1), this._headerFont);
        }
        protected override void AlignText(XlCell rHeader1, XlCell rHeader2, XlCell cHeader1, XlCell cHeader2,
            XlCell dataCell1, XlCell dataCell2)
        {
            base.AlignText(rHeader1, rHeader2, cHeader1, cHeader2, dataCell1, dataCell2);
            int cols = (this._scenCount - 1) * (this._yearCount + 1) + 1;
            int row2 = this._mfrCount + 21;
            this._xlu.AlignText(new XlCell(row2, 1), new XlCell(row2 + 2, cols - 1), this._headerAlignment);
        }
        #endregion
        #endregion
        #region 
        private bool _initialized;      
        private int  _scenCount;
        private int  _yearCount;
        private int  _mfrCount;
        private int  _minYear;
        private int  _maxYear;
        private bool[] _scenarioAdded;  
        private int    _bigN;           
        private double[][] _yearCostsByMfr;
        private double[]   _yearCostsBigN        , _yearCostsAll,
                           _yearFuelSavingsBigN  , _yearFuelSavings,
                           _yearTotalBenefitsBigN, _yearTotalBenefits,
                           _yearNetBenefitsBigN  , _yearNetBenefits;
        #endregion
    }
}

