#region << Using Directives >>
using System;
using System.Collections;
using System.IO;
using System.Reflection;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters.Binary;
using System.Text;
using System.Threading;
using Volpe.Cafe;
using Volpe.Cafe.Collections;
using Volpe.Cafe.Data;
using Volpe.Cafe.Data.MonteCarlo;
using Volpe.Cafe.Data.Optimization;
using Volpe.Cafe.IO;
using Volpe.Cafe.Model;
using Volpe.Cafe.Model.Optimization;
using Volpe.Ui;
using Volpe.Utils;
using IOPath = System.IO.Path;
using CafeReporter = Volpe.Cafe.Reporter.Reporter;
#endregion
namespace Volpe.Cafe.Ui
{
    [Serializable]
    public class Session : ISerializable
    {
        #region 
        [Serializable]
        public struct ProgressInfo
        {
            internal ProgressInfo(int scenCount, int yearCount, int mfrCount, int minYear)
            {
                this._scenCount = scenCount;
                this._yearCount = yearCount;
                this._mfrCount  = mfrCount;
                this._minYear   = minYear;
                this._rawCurrent = "";
                this._current    = new string[(int)Math.Ceiling((double)scenCount / 4D) * 4];
                this._additionalInfo = null;
            }
            internal void AddProgressInfo(IModelingProgress progress)
            {
                if (progress == null) { return; }
                this._rawCurrent = progress.ToString();
                Scenario[] scens = progress.Scenarios;
                if (scens != null && this._scenCount > 0)
                {
                    int scenCount = scens.Length;
                    ModelYear[] years = progress.ModelYears;
                    Manufacturer[][] mfrs = progress.Manufacturers;
                    for (int i = 0; i < scenCount; i++)
                    {
                        if (scens[i] != null)
                        {   
                            string progressStr = progress.ToString(i);
                            int scenIndex = scens[i].Index;
                            int curLen = this._current.Length;
                            if (curLen < scenIndex)
                            {   
                                string[] newCurrent = new string[curLen + 4];
                                Array.Copy(this._current, newCurrent, curLen);
                                this._current = newCurrent;
                            }
                            this._current[scenIndex] = progressStr;
                        }
                    } 
                }
                this._additionalInfo = progress.AdditionalInfo;
            }
            internal void Clear()
            {
                this._scenCount = 0;
                this._yearCount = 0;
                this._mfrCount  = 0;
                this._minYear   = 0;
                this._current   = null;
                this._additionalInfo = null;
            }
            internal void SetAdditionalInfo(string message, bool append)
            {
                if (this._additionalInfo == null || !append) { this._additionalInfo = message; }
                else                                         { this._additionalInfo += "\n" + message; }
            }
            public string GetCurrent()
            {
                string progress = "";
                if (this._current != null)
                {
                    for (int i = 0; i < this._current.Length; i++)
                    {
                        if (this._current[i] != null && this._current[i] != "")
                        {
                            progress += this._current[i] + "\r\n" + new string('-', 25) + "\r\n";
                        }
                    }
                }
                if (progress == "") { progress = this._rawCurrent; }
                return progress;
            }
            public object GetAdditionalInfo()
            {
                return this._additionalInfo;
            }
            private int _scenCount;
            private int _yearCount;
            private int _mfrCount;
            private int _minYear;
            [NonSerialized]
            private string _rawCurrent;
            private string[] _current;
            private object _additionalInfo;
        }
        private sealed class ActiveSessionInfo
        {
            public ActiveSessionInfo(string sessionText)
            {
                this.SessionText = sessionText;
            }
            public string SessionText;
        }
        #endregion
        #region 
        public event PromptEventHandler Prompt;
        public event EventHandler Updating;
        #endregion
        #region 
        public Session()
            : this(null, null)
        {
        }
        public Session(string name, string path)
        {
            this._loaded = false;
            this._index = Session.SessionCount++;
            this._name = (name == null || name.Trim() == "") ? "Session " + this._index : name.Trim();
            this._path = path;
            this._envPath = Cafe2d.StartupPath.ToUpper();
            this._settings = new ModelingSettings(Cafe2d.DefaultDataPath, Cafe2d.DefaultOutputPath);
            this._refreshInterval = 1000;
            this._stopRefreshing = false;
            this._sesInfo = new ActiveSessionInfo(this.Text);
            this._status = "Ready";
            Session.ActiveSessions.Add(this._sesInfo);
        }
        protected Session(SerializationInfo info, StreamingContext context)
        {
            this._index = Session.SessionCount++;    
            this._name = info.GetString("_name");
            this._path = info.GetString("_path");
            this._envPath = info.GetString("_envPath");
            this._data = (Industry)info.GetValue("_data", typeof(Industry));
            this._settings = (ModelingSettings)info.GetValue("_settings", typeof(ModelingSettings));
            this._compliance = (ICompliance)info.GetValue("_compliance", typeof(ICompliance));
            this._progressInfo = (Session.ProgressInfo)info.GetValue("_progressInfo", typeof(Session.ProgressInfo));
            this._refreshInterval = info.GetInt32("_refreshInterval");
            this._status = info.GetString("_status");
            this._envChanged = (this._envPath != Cafe2d.StartupPath.ToUpper());
        }
        ~Session()
        {
            this.Stop(false);
            this.Prompt   = null;
            this.Updating = null;
            this._name = null;
            this._path = null;
            this._data = null;
            this._settings   = null;
            this._compliance = null;
            this._progressInfo.Clear();
            this.Dispose(false);
        }
        internal void Dispose()
        {
            this.Dispose(true);
        }
        private void Dispose(bool disposing)
        {
            if (!this._disposed)
            {
                Session.ActiveSessions.Remove(this._sesInfo);
                this._sesInfo = null;
                this._disposed = true;
            }
        }
        #endregion
        #region 
        #region 
        public void GetObjectData(SerializationInfo info, StreamingContext context)
        {
            info.AddValue("_name", this._name);
            info.AddValue("_path", this._path);
            info.AddValue("_envPath", this._envPath);
            info.AddValue("_data", this._data);
            info.AddValue("_settings", this._settings);
            info.AddValue("_compliance", this._compliance);
            info.AddValue("_progressInfo", this._progressInfo);
            info.AddValue("_refreshInterval", this._refreshInterval);
            info.AddValue("_status", this._status);
        }
        #endregion
        #region 
        protected void OnPrompt(PromptEventArgs e)
        {
            if (this.Prompt != null) { this.Prompt(this, e); }
        }
        protected void OnUpdating()
        {
            if (this.Updating != null) { this.Updating(this, EventArgs.Empty); }
        }
        #endregion
        public void Refresh()
        {
            if (this._compliance != null) { this._progressInfo.AddProgressInfo  (this._compliance.Progress); }
            if (this._reporter   != null) { this._progressInfo.SetAdditionalInfo(this._reporter.Progress, this._running); }
            this.OnUpdating();
        }
        public void Load(string path, bool silent)
        {
            this._path = path;
            this.Load(silent);
        }
        public void Load(bool silent)
        {
            if (this._running || this._path == null || !File.Exists(this._path))
            {
                if (!silent)
                {
                    string message;
                    if (this._running)
                    {
                        message = "The session is currently running.  Stop this session and try again.";
                    }
                    else
                    {
                        message = "The load path was not specified or the specified path does not exist.";
                    }
                    this.OnPrompt(new PromptEventArgs("Error Loading Session", message, PromptOption.Ok));
                }
                return;
            }
            ZipUtilities compressor = Cafe2d.Settings.Compressor;
            BinaryFormatter bf = new BinaryFormatter();
            Stream[] streams = null;
            try
            {
                compressor.Decompress(this._path, out streams, true, null);
                streams[0].Seek(0, SeekOrigin.Begin);
                Session session = (Session)bf.Deserialize(streams[0]);
                if (session._envChanged)
                {
                    PromptEventArgs e = new PromptEventArgs("Environment Changed", "It appears that the environment of this " +
                        "session has changed.  The loaded session may have been saved by a different user or system.\n\n" +
                        "Would you like to attempt to update the session paths?", PromptOption.YesNo);
                    if (silent) { e.PromptResult = PromptResult.Yes; } else { this.OnPrompt(e); }
                    if (e.PromptResult == PromptResult.Yes)
                    {    
                        session._envPath = Cafe2d.StartupPath.ToUpper();
                        session._settings.OutputSettings.OutputPath = Cafe2d.DefaultOutputPath;
                        InputSettings curIS = session._settings.InputSettings;
                        InputSettings newIS = new InputSettings(Cafe2d.DefaultDataPath);
                        if (curIS.EmissionsRatesFileLoaded)
                        {
                            newIS.EmissionsRatesFileLoaded = true;
                            newIS.EmissionsRatesFile = Cafe2d.DefaultDataPath +
                                curIS.EmissionsRatesFile.Substring(curIS.EmissionsRatesFile.LastIndexOf('\\'));
                        }
                        if (curIS.MarketDataFileLoaded)
                        {
                            newIS.MarketDataFileLoaded = true;
                            newIS.MarketDataFile = Cafe2d.DefaultDataPath +
                                curIS.MarketDataFile.Substring(curIS.MarketDataFile.LastIndexOf('\\'));
                        }
                        if (curIS.ParametersFileLoaded)
                        {
                            newIS.ParametersFileLoaded = true;
                            newIS.ParametersFile = Cafe2d.DefaultDataPath +
                                curIS.ParametersFile.Substring(curIS.ParametersFile.LastIndexOf('\\'));
                        }
                        if (curIS.ScenariosFileLoaded)
                        {
                            newIS.ScenariosFileLoaded = true;
                            newIS.ScenariosFile = Cafe2d.DefaultDataPath +
                                curIS.ScenariosFile.Substring(curIS.ScenariosFile.LastIndexOf('\\'));
                        }
                        if (curIS.TechnologiesFileLoaded)
                        {
                            newIS.TechnologiesFileLoaded = true;
                            newIS.TechnologiesFile = Cafe2d.DefaultDataPath +
                                curIS.TechnologiesFile.Substring(curIS.TechnologiesFile.LastIndexOf('\\'));
                        }
                        session._settings.InputSettings = newIS;
                    }
                } 
                this._name = session._name;
                this._path = (session._envChanged) ? null : session._path;
                this._data = session._data;
                this._settings = session._settings;
                this._compliance = session._compliance;
                this._progressInfo = session._progressInfo;
                this._refreshInterval = session._refreshInterval;
                this._sesInfo.SessionText = this.Text;
                session._compliance = null;
            }
            catch (Exception ex)
            {
                if (!silent)
                {
                    this.OnPrompt(new PromptEventArgs("Error Loading Session", "Errors were encountered while loading the " +
                        "session data.\n\nThe session format may be outdated or no longer supported.", PromptOption.Ok));
                    Console.WriteLine(ex.ToString());
                }
                return;
            }
            finally
            {
                if (streams != null)
                {
                    for (int i = 0; i < streams.Length; i++)
                    {
                        if (streams[i] != null)
                        {
                            streams[i].Close();
                        }
                    }
                }
            }
            this._loaded = true;
        }
        public void Save(string path)
        {
            this._path = path;
            this.Save();
        }
        public void Save()
        {
            if (this._running || this._path == null || this._path.Trim() == "")
            {
                this.OnPrompt(new PromptEventArgs("Error Saving Session", "The session is running, or the save path was not " +
                    "specified.", PromptOption.Ok));
                return;
            }
            ZipUtilities compressor = Cafe2d.Settings.Compressor;
            MemoryStream stream = new MemoryStream();
            BinaryFormatter bf = new BinaryFormatter();
            bf.Serialize(stream, this);
            stream.Seek(0, SeekOrigin.Begin);
            compressor.Compress(new Stream[] {stream}, new string[] {"cmsd"}, this._path);
            stream.Close();
        }
        public void Start()
        {
            if (this._compliance == null || this._compliance.Running) { return; }
            LogWriter logWriter = null;
            string logPath = this._settings.OutputSettings.OutputPath + "\\" + this._name + "\\logs";
            Directory.CreateDirectory(logPath);
            logWriter = new LogWriter(logPath);
            bool noSets = (this._settings == null || this._settings.Scenarios == null);
            bool noData = (this._data == null);
            int scenCount = (noSets) ? 0 : this._settings.Scenarios.Count;
            int minYear   = (noData) ? 0 : this._data.MinYear.Year;
            int yearCount = (noData) ? 0 : this._data.MaxYear.Year - minYear + 1;
            int mfrCount  = (noData) ? 0 : this._data.ManufacturerCount;
            int minYearIndex = (noData) ? 0 : this._data.MinYear.Index;
            int maxYearIndex = (noData) ? 0 : this._data.MaxYear.Index;
            ManufacturerCollection mfrs = (noData) ? null : this._data.Manufacturers;
            this._progressInfo = new ProgressInfo(scenCount, yearCount, mfrCount, minYear);
            this.RegisterCompliance();
            this._running = true;   
            if (!this._stopRefreshing) { this.InitRefreshThread(); }
            this._compliance.Start((this._data == null) ? null : this._data.Clone(),
                (this._settings == null) ? null : this._settings.Clone(), logWriter);
            this._settings.Changes.Reset();
        }
        public void Stop()
        {
            this.Stop(true);
        }
        public void Stop(bool stopWhenReady)
        {
            if (this._compliance != null)
            {    
                this._compliance.Abort(stopWhenReady);
            }
        }
        public void GenerateReports()
        {
            try
            {
                if (this._compliance == null || !this._compliance.Completed)
                {
                    this.OnPrompt(new PromptEventArgs("Model Not Completed",
                        "The reports cannot be generated because the model has not yet completed successfully.", PromptOption.Ok));
                    return;
                }
                this.GenerateReportsInternal();
            }
            catch (Exception ex)
            {
                if (this._reporter != null)
                {
                    this._reporter.ReportingCompleted -= new EventHandler(this.Reporter_ReportingCompleted);
                    this._reporter.ReportingStopped   -= new EventHandler(this.Reporter_ReportingStopped);
                    this._reporter.ReportingProgress  -= new EventHandler(this.Reporter_ReportingProgress);
                }
                throw ex;
            }
        }
        private void GenerateReportsInternal()
        {
            if (!this._running || this._reporting)
            {
                this._reporting = true;
                this._status = "Preparing to generate reports ...";
                ModelingSettings settings   = this._compliance.Settings.Clone();
                settings.OutputSettings     = this._settings.OutputSettings;
                settings.EncryptionSettings = this._settings.EncryptionSettings;
                string path            = settings.OutputSettings.OutputPath + "\\" + this._name + "\\reports";
                string templatePath = Cafe2d.DefaultDataPath + "\\template.xls";
                bool             isOpt      = (this._compliance is Optimization);
                OptimizationData optData    =  this._compliance.OptimizationData;
                IterationsData[] optIndData = (optData != null) ? optData.IndustryData : null;
                this._reporter = new CafeReporter(this._compliance, path, templatePath, this._compliance.Data, settings,
                    isOpt, 1, optIndData);
                this._reporter.ReportingCompleted += new EventHandler(this.Reporter_ReportingCompleted);
                this._reporter.ReportingStopped   += new EventHandler(this.Reporter_ReportingStopped);
                this._reporter.ReportingProgress  += new EventHandler(this.Reporter_ReportingProgress);
                this._status = "Generating Reports ...";
                this._reporter.GenerateReports(true);
                this.Refresh();
            }
        }
        public void CancelReports()
        {
            if (this._reporting && this._reporter != null)
            {
                this._reporter.CancelReports();
                this._status = "Canceling Reports (this may take some time) ...";
                this.Refresh();
            }
        }
        private void Reporter_ReportingCompletedOrStopped(string status)
        {
            this._reporter.ReportingCompleted -= new EventHandler(this.Reporter_ReportingCompleted);
            this._reporter.ReportingStopped   -= new EventHandler(this.Reporter_ReportingStopped  );
            this._reporter.ReportingProgress  -= new EventHandler(this.Reporter_ReportingProgress );
            this._reporter  = null;
            this._reporting = false;
            this._status = status;
            this._progressInfo.SetAdditionalInfo(null, false);
            this.Refresh();
        }
        private void Reporter_ReportingCompleted(object sender, EventArgs e)
        {
            this.Reporter_ReportingCompletedOrStopped("Reporting Completed!");
        }
        private void Reporter_ReportingStopped(object sender, EventArgs e)
        {
            this.Reporter_ReportingCompletedOrStopped("Reporting Stopped.");
        }
        private void Reporter_ReportingProgress(object sender, EventArgs e)
        {
            this.Refresh();
        }
        private void RegisterCompliance()
        {
            this._compliance.ModelingChanged += new ModelingEventHandler(this.Compliance_ModelingChanged);
            this._compliance.Prompt          += new PromptEventHandler  (this.Compliance_Prompt);
        }
        private void DeRegisterCompliance()
        {
            this._compliance.ModelingChanged -= new ModelingEventHandler(this.Compliance_ModelingChanged);
            this._compliance.Prompt          -= new PromptEventHandler  (this.Compliance_Prompt);
        }
        private void Compliance_ModelingChanged(object sender, ModelingEventArgs e)
        {
            ModelingState state = e.State;
            this._running = (state == ModelingState.Running || state == ModelingState.StopRequested ||
                state == ModelingState.StartRequested);
            if      (state == ModelingState.StartRequested) { this._status = "Starting the Modeling Process ...";     }
            else if (state == ModelingState.Running)        { this._status = "Model Running ...";                     }
            else if (state == ModelingState.StopRequested)  { this._status = "Stopping the Modeling Process ...";     }
            else if (state == ModelingState.Stopped)        { this._status = "Modeling Stopped.";                     }
            else if (state == ModelingState.Completed)      { this._status = "Modeling Completed!";                   }
            else if (state == ModelingState.Unstarted)      { this._status = "The Modeling Process has not Started."; }
            if (!this._running)
            {   
                this.DeRegisterCompliance();
                if (this._compliance != null && this._compliance.Progress != null)
                {
                    this._compliance.Progress.AdditionalInfo = null;
                    this._progressInfo.SetAdditionalInfo(null, false);
                }
            }
            this.OnUpdating();
        }
        private void Compliance_Prompt(object sender, PromptEventArgs e)
        {
            this.OnPrompt(e);
        }
        void InitRefreshThread()
        {
            if (this._running)
            {
                this._stopRefreshing = false;
                this._refreshThread = new Thread(new ThreadStart(this.Session_Refresh));
                this._refreshThread.Name = "SessionRefreshThread";
                this._refreshThread.Start();
            }
        }
        void StopRefreshThread()
        {
            this._stopRefreshing = true;
        }
        void Session_Refresh()
        {
            while (this._running && !this._stopRefreshing)
            {
                for (int i = 0; i < this._refreshInterval / Session.MinRefreshInterval; i++)
                {
                    Thread.Sleep(Session.MinRefreshInterval);
                    if (!this._running || this._stopRefreshing) { break; }
                }
                this.Refresh();
            }
        }
        #endregion
        #region 
        public bool Loaded
        {
            get { return this._loaded; }
        }
        public int Index
        {
            get { return this._index; }
        }
        public string Text
        {
            get { return this._index + ". " + this._name; }
        }
        public string Name
        {
            get { return this._name; }
            set
            {
                this._name = value;
                this._sesInfo.SessionText = this.Text;    
            }
        }
        public string Path
        {
            get { return this._path; }
            set { this._path = value; }
        }
        public Industry Data
        {
            get { return this._data; }
            set { this._data = value; }
        }
        public ModelingSettings Settings
        {
            get { return this._settings; }
            set { this._settings = value; }
        }
        public ICompliance Compliance
        {
            get { return this._compliance; }
            set { this._compliance = value; }
        }
        public Session.ProgressInfo ProgressInformation
        {
            get { return this._progressInfo; }
        }
        public bool Running
        {
            get { return this._running; }
        }
        public string Status
        {
            get { return this._status; }
        }
        public bool Reporting
        {
            get
            {
                return this._reporting;
            }
        }
        public long Runtime
        {
            get
            {
                return (this._compliance.State == ModelingState.Unstarted || this._compliance.Runtime < 0) ? 0 :
                    this._compliance.Runtime;
            }
        }
        public int RefreshInterval
        {
            get { return this._refreshInterval; }
            set
            {
                if (value == -1) { this.StopRefreshThread(); }
                int lastRefInterval = this._refreshInterval;
                this._refreshInterval = (value < MinRefreshInterval) ? MinRefreshInterval :
                    value - (value % MinRefreshInterval);   
                if (this._stopRefreshing && value != -1) { this.InitRefreshThread(); }
            }
        }
        #endregion
        #region 
        public const int MinRefreshInterval = 125;
        private static int SessionCount = 1;
        private static ArrayList ActiveSessions = new ArrayList();
        private Session.ActiveSessionInfo _sesInfo;
        private bool _disposed;
        private bool _loaded;
        private int _index;
        private string _name;
        private string _path;
        private string _envPath;
        private bool _envChanged;
        private Industry _data;
        private ModelingSettings _settings;
        private ICompliance _compliance;
        private ProgressInfo _progressInfo;
        private bool _running;
        private string _status;
        private CafeReporter _reporter;
        private bool _reporting;
        private int _refreshInterval;
        bool _stopRefreshing;
        private Thread _refreshThread;
        #endregion
    }
}

