using System;
using Volpe.Cafe.Collections;
using Volpe.Cafe.IO;
namespace Volpe.Cafe.Data
{
    [Serializable]
    public class Industry : ICloneable
    {
        #region 
        private Industry()
        {
        }
        public Industry(string path)
            : this(path, "")
        {
        }
        public Industry(string path, string password)
            : this()
        {
            this._allManufacturers = new ManufacturerCollection(32);
            this._manufacturers = new ManufacturerCollection(32);
            this._modelingData = new IndustryModelingData();
            Input input = this.LoadFile(path, password);
            try
            {    
                this.ParseFile(input);
            }
            catch (Exception ex)
            {    
				Console.WriteLine(ex.Message);		
                throw new InputException("Errors were encountered while parsing file: " + ex.Message, path, ex);
            }
            finally
            {    
                input.Close();
            }
        }
        #endregion
        #region 
        #region 
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        public Industry Clone()
        {
            Industry ind = new Industry();
            ind._modelingData = this._modelingData.Clone();
            ind._minYear = new ModelYear(this._minYear.Year);
            ind._maxYear = new ModelYear(this._maxYear.Year);
            ind._allManufacturers = (ManufacturerCollection)this._allManufacturers.Clone();
            ind._manufacturers = new ManufacturerCollection(this._manufacturers.Count);
            for (int i = 0; i < ind._allManufacturers.Count; i++)
            {
                if (ind._allManufacturers[i].Valid)
                {   
                    ind._manufacturers.Add(ind._allManufacturers[i]);
                }
            }
            return ind;
        }
        #endregion
        public void WriteWarningsLog(LogWriter writer)
        {
            if (writer.Warnings == null) { return; }
            for (int i = 0; i < this._allManufacturers.Count; i++)
            {
                Manufacturer mfr = this._allManufacturers[i];
                EngineCollection engs = mfr.Engines;
                TransmissionCollection trns = mfr.Transmissions;
                VehicleCollection discVehs = mfr.DiscardedVehicles;
                VehicleCollection allVehs = mfr.GetVehicles();
                writer.Warnings.WriteLine("Analyzing Manufacturer: Code=" + mfr.Description.Code +
                    ", Name=" + mfr.Description.Name);
                writer.Warnings.WriteLine("  Analyzing Engines (total " + engs.Count + "): ");
                if (mfr.DiscardedEngineCount > 0)
                {
                    for (int j = 0, count = engs.Count; j < count; j++)
                    {
                        Engine eng = engs[j];
                        if (!eng.Valid)
                        {
                            writer.Warnings.WriteLine("    Engine Discarded: Code=" + eng.Description.Code +
                                ", Name=" + eng.Description.Name +
                                this.WriteWarningsLog_GetInvalidReasonsString("      ", eng.InvalidReasons));
                        }
                    }
                    writer.Warnings.WriteLine("    -----\r\n    Total Discarded: " + mfr.DiscardedEngineCount);
                }
                writer.Warnings.WriteLine("  Analyzing Transmissions (total " + trns.Count + "): ");
                if (mfr.DiscardedTransmissionCount > 0)
                {
                    for (int j = 0, count = trns.Count; j < count; j++)
                    {
                        Transmission trn = trns[j];
                        if (!trn.Valid)
                        {
                            writer.Warnings.WriteLine("    Transmission Discarded: Code=" + trn.Description.Code +
                                ", Name=" + trn.Description.Name +
                                this.WriteWarningsLog_GetInvalidReasonsString("      ", trn.InvalidReasons));
                        }
                    }
                    writer.Warnings.WriteLine("    -----\r\n    Total Discarded: " + mfr.DiscardedTransmissionCount);
                }
                writer.Warnings.WriteLine("  Analyzing Vehicles (total " + mfr.AllVehicleCount + "): ");
                if (mfr.VehicleCount > 0)
                {
                    for (int j = 0, count = allVehs.Count; j < count; j++)
                    {
                        Vehicle veh = allVehs[j];
                        string[] loadWarnings = veh.LoadWarnings;
                        if (loadWarnings != null && loadWarnings.Length > 0)
                        {
                            writer.Warnings.WriteLine(
                                "    Vehicle Accepted with Warnings: Code=" + veh.Description.Code +
                                ", Model=" + veh.Description.Model + ", Nameplate=" + veh.Description.Nameplate +
                                ", Eng-Code=" + veh.Description.EngineCode +
                                ", Trn-Code=" + veh.Description.TransmissionCode +
                                this.WriteWarningsLog_GetInvalidReasonsString("      ", loadWarnings));
                        }
                    }
                }
                if (mfr.DiscardedVehicleCount > 0)
                {
                    for (int j = 0, count = discVehs.Count; j < count; j++)
                    {
                        Vehicle veh = discVehs[j];
                        writer.Warnings.WriteLine(
                            "    Vehicle Discarded: Code=" + veh.Description.Code +
                            ", Model=" + veh.Description.Model + ", Nameplate=" + veh.Description.Nameplate +
                            ", Eng-Code=" + veh.Description.EngineCode +
                            ", Trn-Code=" + veh.Description.TransmissionCode +
                            this.WriteWarningsLog_GetInvalidReasonsString("      ", veh.InvalidReasons));
                    }
                    writer.Warnings.WriteLine("    -----\r\n    Total Discarded: " + mfr.DiscardedVehicleCount);
                }
                writer.Warnings.WriteLine("Manufacturer " + ((mfr.Valid) ? "Accepted w/Index=" + mfr.Index + "." :
                    "Discarded!" + this.WriteWarningsLog_GetInvalidReasonsString("  ", mfr.InvalidReasons)));
                writer.Warnings.WriteLine(new string('-', 25));
            }
            if (this._manufacturers.Count == 0) { writer.Warnings.WriteLine("Warning!  No valid Manufacturers found."); }
        }
        private string WriteWarningsLog_GetInvalidReasonsString(string tab, string[] reasons)
        {
            string s = "";
            if (reasons != null)
            {
                for (int i = 0, count = reasons.Length; i < count; i++)
                {
                    s += ("\r\n" + tab + "* " + reasons[i]);
                }
            }
            return s;
        }
        public void WriteSummaryLog(LogWriter writer)
        {
            if (writer.Summary == null) { return; }
            writer.Summary.WriteLine("---------- Modeling Data ----------");
            writer.Summary.WriteLine("--- Model Years Analyzed ---\r\n  "
                + this._minYear.ToString() + " to " + this._maxYear.ToString() + "\r\n");
            writer.Summary.WriteLine("--- Valid Manufacturers ---");
            for (int i = 0, mfrCount = this._manufacturers.Count; i < mfrCount; i++)
            {
                Manufacturer mfr = this._manufacturers[i];
                writer.Summary.WriteLine(
                    "  " + mfr.Index + ". Code=" + mfr.Description.Code + ", Name=" + mfr.Description.Name +
                        "\r\n      Vehicles=" + mfr.VehicleCount +
                            " (Dom Autos=" + mfr.DomesticPassengerAutos.Count +
                            ", Imp Autos=" + mfr.ImportedPassengerAutos.Count +
                            ", Lt Trucks=" + mfr.LightTrucks.Count +
                            ", Heavy LTs=" + mfr.HeavyLightTrucks.Count + ")" +
                        "\r\n      Discarded Vehs=" + mfr.DiscardedVehicleCount +
                        "\r\n      Engines=" + mfr.NonDiscardedEngineCount +
                        "\r\n      Discarded Engs=" + mfr.DiscardedEngineCount +
                        "\r\n      Transmissions=" + mfr.NonDiscardedTransmissionCount +
                        "\r\n      Discarded Trns=" + mfr.DiscardedTransmissionCount);
            }
            writer.Summary.WriteLine();
            writer.Summary.WriteLine("--- Discarded Manufacturers ---");
            int engs = 0, dEngs = 0, trns = 0, dTrns = 0, vehs = 0, dVehs = 0, doms = 0, imps = 0, lts = 0, hlts = 0;
            for (int i = 0, mfrCount = this._allManufacturers.Count; i < mfrCount; i++)
            {
                Manufacturer mfr = this._allManufacturers[i];
                engs  += mfr.NonDiscardedEngineCount;
                dEngs += mfr.DiscardedEngineCount;
                trns  += mfr.NonDiscardedTransmissionCount;
                dTrns += mfr.DiscardedTransmissionCount;
                vehs  += mfr.VehicleCount;
                dVehs += mfr.DiscardedVehicleCount;
                doms  += mfr.DomesticPassengerAutos.Count;
                imps  += mfr.ImportedPassengerAutos.Count;
                lts   += mfr.LightTrucks.Count;
                hlts  += mfr.HeavyLightTrucks.Count;
                if (!mfr.Valid)
                {
                    writer.Summary.WriteLine("  Code=" + mfr.Description.Code + ", Name=" + mfr.Description.Name +
                        " (Vehicles=" + mfr.VehicleCount + ", Engines=" + mfr.Engines.Count + ", Transmissions=" +
                        mfr.Transmissions.Count + ")");
                }
            }
            writer.Summary.WriteLine();
            writer.Summary.WriteLine("--- Industry Summary ---");
            writer.Summary.WriteLine("  Total Manufacturers Analyzed:  " + this._manufacturers.Count);
            writer.Summary.WriteLine("  Total Manufacturers Discarded: " + (this._allManufacturers.Count -
                                                                            this._manufacturers.Count));
            writer.Summary.WriteLine("  Total Vehicles Analyzed:  " + vehs);
            writer.Summary.WriteLine("    Domestic Autos:   " + doms);
            writer.Summary.WriteLine("    Imported Autos:   " + imps);
            writer.Summary.WriteLine("    Light Trucks:     " + lts);
            writer.Summary.WriteLine("    Heavy Lt. Trucks: " + hlts);
            writer.Summary.WriteLine("  Total Vehicles Discarded: " + dVehs);
            writer.Summary.WriteLine("  Total Engines Analyzed:   " + engs);
            writer.Summary.WriteLine("  Total Engines Discarded:  " + dEngs);
            writer.Summary.WriteLine("  Total Transmissions Analyzed:  " + trns);
            writer.Summary.WriteLine("  Total Transmissions Discarded: " + dTrns);
            writer.Summary.WriteLine();
        }
        public VehicleCollection GetVehicles()
        {
            VehicleCollection vc = new VehicleCollection(this.VehicleCount);
            for (int i = 0; i < this._manufacturers.Count; i++)
            {
                vc.AddRange(this._manufacturers[i].GetVehicles());
            }
            return vc;
        }
        public VehicleCollection GetModelYearVehicles()
        {
            VehicleCollection vc = new VehicleCollection(this.ModelYearVehicleCount);
            for (int i = 0; i < this._manufacturers.Count; i++)
            {
                vc.AddRange(this._manufacturers[i].ModelYearVehicles);
            }
            return vc;
        }
        private Input LoadFile(string path, string password)
        {
            return new Input(path, password);
        }
        private void ParseFile(Input input)
        {
            string[] sheets = input.SheetNames;
            string[] names = {"Manufacturers", "Vehicles", "Engines", "Transmissions"};
            int[] indexes = {-1, -1, -1, -1};
            for (int i = 0; i < sheets.Length; i++)
            {
                for (int j = 0; j < names.Length; j++)
                {
                    if (input.Compare(sheets[i], names[j], false))
                    {
                        indexes[j] = i;
                        break;
                    }
                }
            }
            input.VerifyIndexes(indexes, names);
            ManufacturerCollection mc;
            VehicleCollection[] vc;
            EngineCollection[] ec;
            TransmissionCollection[] tc;
            int mfrWsMinYear, mfrWsMaxYear;        
            mc = this.ParseFile_LoadManufacturers(input, indexes[0], out mfrWsMinYear, out mfrWsMaxYear);
            vc = this.ParseFile_LoadVehicles(input, indexes[1], mc);
            ec = this.ParseFile_LoadEngines(input, indexes[2], mc);
            tc = this.ParseFile_LoadTransmissions(input, indexes[3], mc);
            this._allManufacturers = mc;
            int mfrIndex = 0;
            int indMinYear = int.MaxValue, indMaxYear = -1;
            for (int i = 0; i < this._allManufacturers.Count; i++)
            {
                int minYear, maxYear;
                this._allManufacturers[i].Initialize(vc[i], ec[i], tc[i]);
                this._allManufacturers[i].Validate(out minYear, out maxYear);
                if (this._allManufacturers[i].Valid)
                {   
                    this._manufacturers.Add(this._allManufacturers[i]);
                    this._allManufacturers[i].SetIndex(mfrIndex++);
                    indMinYear = Math.Min(minYear, indMinYear);
                    indMaxYear = Math.Max(maxYear, indMaxYear);
                }
            }
            indMinYear = Math.Max(mfrWsMinYear, indMinYear);
            indMaxYear = Math.Min(mfrWsMaxYear, indMaxYear);
            if (indMinYear <  9999)              { this._minYear = new ModelYear(indMinYear); }
            if (indMaxYear >= ModelYear.MinYear) { this._maxYear = new ModelYear(indMaxYear); }
            double fuelCapcitySum    = 0d;
            int    fuelCapacityCount = 0;
            for (int i = 0, mfrCount = this._manufacturers.Count; i < mfrCount; i++)
            {
                VehicleCollection vehs = this._manufacturers[i].GetVehicles();
                for (int j = 0, vehCount = vehs.Count; j < vehCount; j++)
                {
                    if (vehs[j].Description.FuelCapacity > 0)
                    {
                        fuelCapcitySum += vehs[j].Description.FuelCapacity;
                        fuelCapacityCount++;
                    }
                }
            }
            fuelCapcitySum /= fuelCapacityCount;
            for (int i = 0, mfrCount = this._manufacturers.Count; i < mfrCount; i++)
            {
                VehicleCollection vehs = this._manufacturers[i].GetVehicles();
                for (int j = 0, vehCount = vehs.Count; j < vehCount; j++)
                {
                    if (vehs[j].Description.FuelCapacity <= 0) { vehs[j].Description.FuelCapacity = fuelCapcitySum; }
                }
            }
            this.UpdateMinMaxFpCw();
        }
        private ManufacturerCollection ParseFile_LoadManufacturers(Input input, int index, out int mfrMinYear,
            out int mfrMaxYear)
        {
            ManufacturerCollection mc;
            input.ActivateWorksheet(index);
            int rows = input.Rows;
            int cols = input.Columns;
            string[] names = {
                                 "Manufacturer Code", "Manufacturer Name", "Cost Allocation Strategy",
                                 "Avg. CW of MY02 Lt. Trucks", "Discount Rate", "Optimize",
                                 "Willingness to Pay CAFE Fines", "Available Credits (vehicle-mpg)",
                                 "Applies to Baseline"
                             };
            int[] indexes = new int[names.Length];
            for (int i = 0; i < indexes.Length; i++) { indexes[i] = -1; }
            for (int i = 0; i < cols; i++)
            {
                string column0 = input.GetString(0, i);
                string column1 = input.GetString(1, i);
                int    column1_int = Global.GetInt32(column1);
                if      (input.Compare(column1, names[1], true)                                         ) { indexes[1] = i; }
                else if (input.Compare(column1, names[0], true) || input.Compare(column1, "Code", false)) { indexes[0] = i; }
                else if (input.Compare(column1, names[2], true)                                         ) { indexes[2] = i; }
                else if (input.Compare(column1, names[3], true)                                         ) { indexes[3] = i; }
                else if (input.Compare(column1, names[4], true)                                         ) { indexes[4] = i; }
                else if (input.Compare(column1, names[5], true)                                         ) { indexes[5] = i; }
                else if (input.Compare(column0, names[6], true) && ModelYear.IsValid(column1_int)       ) { indexes[6] = i; }
                else if (input.Compare(column0, names[7], true) && ModelYear.IsValid(column1_int)       ) { indexes[7] = i; }
                else if (input.Compare(column1, names[8], true)                                         ) { indexes[8] = i; }
            }
            input.VerifyIndexes(new int[] {indexes[0], indexes[1]}, new string[] {names[0], names[1]});
            input.GetMinMaxYears(1, indexes[6], out mfrMinYear, out mfrMaxYear);
            mc = new ManufacturerCollection(32);
            for (int i = 2; i < rows; i++)
            {
                ManufacturerDescription md = this.ParseFile_LoadManufacturer(input, i, indexes, mfrMinYear, mfrMaxYear);
                if (md != null) { mc.Add(new Manufacturer(md)); }
            }
            return mc;
        }
        private ManufacturerDescription ParseFile_LoadManufacturer(Input input, int row, int[] indexes, int minYear,
            int maxYear)
        {
            int    mfrCode = input.GetInt32 (row, indexes[0]);
            string mfrName = input.GetString(row, indexes[1]);
            if (mfrCode <= 0 || mfrName == string.Empty) { return null; }
            ManufacturerDescription md = new ManufacturerDescription();
            md.Code = mfrCode;
            md.Name = mfrName;
            md.CostAllocationStrategy = (indexes[2] == -1) ? 0                :  input.GetInt32 (row, indexes[2]);
            md.AverageMY2002LtCw      = (indexes[3] == -1) ? 4329.48480892067 :  input.GetDouble(row, indexes[3]);
            md.DiscountRate           = (indexes[4] == -1) ? -1               :  input.GetDouble(row, indexes[4]);
            md.Optimize               = (indexes[5] == -1) ? false            : (input.GetChar  (row, indexes[5]) == 'Y');
            md.CreditsApplyToBaseline = (indexes[8] == -1) ? false            : (input.GetChar  (row, indexes[8]) == 'Y');
            md.WillingToPayFines = new bool  [Math.Max(15, maxYear - ModelYear.MinYear + 1)];
            md.AvailableCredits  = new double[Math.Max(15, maxYear - ModelYear.MinYear + 1)];
            int offset = minYear - ModelYear.MinYear;
            for (int i = 0; i < maxYear - minYear + 1; i++)
            {   
                if (indexes[6] != -1) { md.WillingToPayFines[offset + i] = input.GetString(row, indexes[6] + i).ToUpper().Equals("Y"); }
                if (indexes[7] != -1) { md.AvailableCredits [offset + i] = input.GetDouble(row, indexes[7] + i); }
            }
            return md;
        }
        private VehicleCollection[] ParseFile_LoadVehicles(Input input, int index, ManufacturerCollection mfrs)
        {
            VehicleCollection[] vc = null;
            input.ActivateWorksheet(index);
            int rows = input.Rows;
            int cols = input.Columns;
            string[] names = {   
                                 "Sales", "MSRP", "Price",          
                                 "Number"                      ,    
                                 "Manufacturer"                ,    
                                 "Model"                       ,    
                                 "Nameplate"                   ,    
                                 "Fuel Economy"                ,    
                                 "Actual FE (FFVs)"            ,    
                                 "Engine Code"                 ,    
                                 "Transmission Code"           ,    
                                 "Style"                       ,    
                                 "Class"                       ,    
                                 "Structure"                   ,    
                                 "Drive"                       ,    
                                 "Wheelbase"                   ,    
                                 "Track Width (front)"         ,    
                                 "Track Width (rear)"          ,    
                                 "Footprint"                   ,    
                                 "Curb Weight"                 ,    
                                 "Test Weight"                 ,    
                                 "GVWR"                        ,    
                                 "Fuel Capacity"               ,    
                                 "Seating (max)"               ,    
                                 "Type (Hybridization)"        ,    
                                 "Origin"                      ,    
                                 "US Content"                  ,    
                                 "Canadian Content"            ,    
                                 "Mexican Content"             ,    
                                 "Predecessor"                 ,    
                                 "Redesign Years"              ,    
                                 "Refresh Years"               ,    
                                 "Technology Class"            ,    
								 "Vehicle Number"              ,	
								 "Fuel Economy on Primary Fuel",	
								 "Fuel Economy on Secondary Fuel",	
								 "Subclass"                    ,    
								 "Type of Hybrid/Electric Vehicle",	
                                 "Employment Hours Per Vehicle",    
                                 "% 2 DR Cars"                 ,    
                                 "Market Segment - Auto News"  ,    
                                 "Cohort"                      ,    
                                 "Outlier"                     ,    
                                 "EPS", "IACC", "MHEV", "HVIA", "BISG", "CISG", "PSHEV", "2MHEV", "PHEV", "MS1",
                                 "MS2", "MS5" , "ROLL", "LDB" , "SAXU", "SAXL", "AERO"
                             };
            int[] indexes = new int[names.Length];
            for (int i = 0; i < names.Length; i++) { indexes[i] = -1; }
            for (int i = 0; i < cols; i++)
            {
                string column = input.GetString(1, i);
                string smpCol = input.GetString(0, i);        
                for (int j = 0; j < names.Length; j++)
                {
                    if      (input.Compare(smpCol, names[j], true)) { indexes[j] = i; break; }
                    else if (input.Compare(column, names[j], true)) { indexes[j] = i; break; }
                }
            }
            if (indexes[1] == -1 && indexes[2] != -1)
            {   
                indexes[1] = indexes[2];
            }
            else if (indexes[1] != -1 && indexes[2] == -1)
            {   
                indexes[2] = indexes[1];
            }
			if (indexes[ 3] == -1 && indexes[33] != -1)
				indexes[ 3] = indexes[33];		
			else if (indexes[ 3] != -1 && indexes[33] == -1)
				indexes[33] = indexes[ 3];		
			if (indexes[ 7] == -1 && indexes[34] != -1)
				indexes[ 7] = indexes[34];		
			else if (indexes[ 7] != -1 && indexes[34] == -1)
				indexes[34] = indexes[ 7];		
			if (indexes[ 8] == -1 && indexes[34] != -1)
				indexes[ 8] = indexes[34];		
            if (indexes[35] == -1) { indexes[35] = indexes[7]; }
			if (indexes[32] == -1 && indexes[36] != -1)
				indexes[32] = indexes[36];		
			else if (indexes[32] != -1 && indexes[36] == -1)
				indexes[36] = indexes[32];		
			if (indexes[12] == -1 && indexes[36] != -1)
				indexes[12] = indexes[36];		
			else if (indexes[12] != -1 && indexes[36] == -1)
				indexes[36] = indexes[12];		
			if (indexes[24] == -1 && indexes[37] != -1)
				indexes[24] = indexes[37];		
			else if (indexes[24] != -1 && indexes[37] == -1)
				indexes[37] = indexes[24];		
            input.VerifyIndexes(indexes, names, 0, 38);
            int salesMinYear, salesMaxYear;
            int msrpMinYear, msrpMaxYear;
            int priceMinYear, priceMaxYear;
            input.GetMinMaxYears(1, indexes[0], out salesMinYear, out salesMaxYear);
            input.GetMinMaxYears(1, indexes[1], out msrpMinYear, out msrpMaxYear);
            input.GetMinMaxYears(1, indexes[2], out priceMinYear, out priceMaxYear);
            this._minYear = new ModelYear(Math.Max(Math.Max(salesMinYear, msrpMinYear), priceMinYear));
            this._maxYear = new ModelYear(Math.Min(Math.Min(salesMaxYear, msrpMaxYear), priceMaxYear));
            vc = new VehicleCollection[mfrs.Count];
            for (int i = 0; i < mfrs.Count; i++)
            {
                vc[i] = new VehicleCollection(32);
            }
            for (int i = 2; i < rows; i++)
            {   
                VehicleDescription vd = this.ParseFile_LoadVehicle(input, i, indexes,
                    salesMinYear, salesMaxYear, msrpMinYear, msrpMaxYear, priceMinYear, priceMaxYear);
                int mfrIdx = this.ParseFile_GetManufacturerIndex(input, mfrs, vd.Manufacturer);
                if (mfrIdx != -1)
                {    
                    vc[mfrIdx].Add(new Vehicle(vd));
                }
            }
            return vc;
        }
        private VehicleDescription ParseFile_LoadVehicle(Input input, int row, int[] indexes,
            int salesMinYear, int salesMaxYear, int msrpMinYear, int msrpMaxYear, int priceMinYear, int priceMaxYear)
        {
            double curbWeigth = input.GetDouble(row, indexes[19]);
            double testWeight = input.GetDouble(row, indexes[20]);
            if (curbWeigth == 0) { curbWeigth = testWeight - 300; }
            VehicleDescription vd =  new VehicleDescription(curbWeigth);
			vd.Code                  = input.GetInt32 (row, indexes[ 3]);
			vd.Manufacturer          = input.GetString(row, indexes[ 4]);
			vd.Model                 = input.GetString(row, indexes[ 5]);
			vd.Nameplate             = input.GetString(row, indexes[ 6]);
			vd.RatedFuelEconomy      = input.GetDouble(row, indexes[ 7]);
			vd.ActualFuelEconomy     = input.GetDouble(row, indexes[ 8]);
			vd.EngineCode            = input.GetInt32 (row, indexes[ 9]);
			vd.TransmissionCode      = input.GetInt32 (row, indexes[10]);
			vd.Style                 = input.GetString(row, indexes[11]);
			vd.Class                 = input.GetString(row, indexes[12]);
			vd.Structure             = input.GetString(row, indexes[13]);
			vd.Drive                 = input.GetChar  (row, indexes[14]);
			vd.Wheelbase             = input.GetDouble(row, indexes[15]);
			vd.FrontTrackWidth       = input.GetDouble(row, indexes[16]);
			vd.RearTrackWidth        = input.GetDouble(row, indexes[17]);
			vd.Footprint             = input.GetDouble(row, indexes[18]);
			vd.CurbWeight            = curbWeigth;
			vd.TestWeight            = testWeight;
			vd.GVWR                  = input.GetDouble(row, indexes[21]);
			vd.FuelCapacity          = input.GetDouble(row, indexes[22]);
			vd.Seating               = input.GetInt32 (row, indexes[23]);
			vd.HybridType            = input.GetString(row, indexes[24]);
			vd.Origin                = input.GetChar  (row, indexes[25]);
			vd.USContent             = input.GetDouble(row, indexes[26]);
			vd.CanadianContent       = input.GetDouble(row, indexes[27]);
			vd.MexicanContent        = input.GetDouble(row, indexes[28]);
			vd.PredecessorCode       = input.GetInt32 (row, indexes[29]);
			string[] redesignYears   = input.GetString(row, indexes[30]).Split(',');
			vd.RedesignYears         = this.ParseFile_LoadVehicle_RefreshRedesign(redesignYears);
			string[] refreshYears    = input.GetString(row, indexes[31]).Split(',');
			vd.RefreshYears          = this.ParseFile_LoadVehicle_RefreshRedesign(refreshYears);
			vd.EmploymentHours       = (indexes[38] != -1) ? input.GetDouble(row, indexes[38]) : 0;
			vd.Percent2Dr            = (indexes[39] != -1) ? input.GetDouble(row, indexes[39]) : 0;
			vd.AutoNewsMarketSegment = (indexes[40] != -1) ? input.GetInt32 (row, indexes[40]) : 0;
			vd.Cohort                = (indexes[41] != -1) ? input.GetInt32 (row, indexes[41]) : 0;
			vd.Outlier               = (indexes[42] != -1) ? input.GetBool  (row, indexes[42]) : false;
            string   techClass      = input.GetString(row, indexes[32]);
            string[] techClassNames =
                new string[] {
                                 "Subcompact PC", "Subcompact Perf. PC", "Compact PC", "Compact Perf. PC",
                                 "Midsize PC"   , "Midsize Perf. PC"   , "Large PC"  , "Large Perf. PC"  ,
                                 "Minivan LT"   , "Small LT"           , "Midsize LT", "Large LT"
                             };
            TechnologyClass[] techClasses = 
                new TechnologyClass[] {
                                          TechnologyClass.SubcompactPC, TechnologyClass.SubcompactPerfPC,
                                          TechnologyClass.CompactPC   , TechnologyClass.CompactPerfPC   ,
                                          TechnologyClass.MidsizePC   , TechnologyClass.MidsizePerfPC   ,
                                          TechnologyClass.LargePC     , TechnologyClass.LargePerfPC     ,
                                          TechnologyClass.MinivanLT   , TechnologyClass.SmallLT         ,
                                          TechnologyClass.MidsizeLT   , TechnologyClass.LargeLT
                                      };
            vd.TechnologyClass = TechnologyClass.None;
            for (int i = 0; i < techClassNames.Length; i++)
            {
                if (Global.StringCompare(techClass, techClassNames[i], true)) { vd.TechnologyClass = techClasses[i]; break; }
            }
            vd.Overrides.EPS   = this.ParseFile_ParseTechOverride((indexes[43] == -1) ? "" : input.GetString(row, indexes[43]));
            vd.Overrides.IACC  = this.ParseFile_ParseTechOverride((indexes[44] == -1) ? "" : input.GetString(row, indexes[44]));
            vd.Overrides.MHEV  = this.ParseFile_ParseTechOverride((indexes[45] == -1) ? "" : input.GetString(row, indexes[45]));
            vd.Overrides.HVIA  = this.ParseFile_ParseTechOverride((indexes[46] == -1) ? "" : input.GetString(row, indexes[46]));
            vd.Overrides.BISG  = this.ParseFile_ParseTechOverride((indexes[47] == -1) ? "" : input.GetString(row, indexes[47]));
            vd.Overrides.CISG  = this.ParseFile_ParseTechOverride((indexes[48] == -1) ? "" : input.GetString(row, indexes[48]));
            vd.Overrides.PSHEV = this.ParseFile_ParseTechOverride((indexes[49] == -1) ? "" : input.GetString(row, indexes[49]));
            vd.Overrides.MHEV2 = this.ParseFile_ParseTechOverride((indexes[50] == -1) ? "" : input.GetString(row, indexes[50]));
            vd.Overrides.PHEV  = this.ParseFile_ParseTechOverride((indexes[51] == -1) ? "" : input.GetString(row, indexes[51]));
            vd.Overrides.MS1   = this.ParseFile_ParseTechOverride((indexes[52] == -1) ? "" : input.GetString(row, indexes[52]));
            vd.Overrides.MS2   = this.ParseFile_ParseTechOverride((indexes[53] == -1) ? "" : input.GetString(row, indexes[53]));
            vd.Overrides.MS5   = this.ParseFile_ParseTechOverride((indexes[54] == -1) ? "" : input.GetString(row, indexes[54]));
            vd.Overrides.ROLL  = this.ParseFile_ParseTechOverride((indexes[55] == -1) ? "" : input.GetString(row, indexes[55]));
            vd.Overrides.LDB   = this.ParseFile_ParseTechOverride((indexes[56] == -1) ? "" : input.GetString(row, indexes[56]));
            vd.Overrides.SAXU  = this.ParseFile_ParseTechOverride((indexes[57] == -1) ? "" : input.GetString(row, indexes[57]));
            vd.Overrides.SAXL  = this.ParseFile_ParseTechOverride((indexes[58] == -1) ? "" : input.GetString(row, indexes[58]));
            vd.Overrides.AERO  = this.ParseFile_ParseTechOverride((indexes[59] == -1) ? "" : input.GetString(row, indexes[59]));
            vd.Sales = this.ParseFile_LoadVehicleSMP(input, row, indexes[0], salesMinYear, salesMaxYear);
            vd.Msrp  = this.ParseFile_LoadVehicleSMP(input, row, indexes[1], msrpMinYear , msrpMaxYear );
            vd.Price = this.ParseFile_LoadVehicleSMP(input, row, indexes[2], priceMinYear, priceMaxYear);
            int maxYears = (int)Math.Max(Math.Max(priceMaxYear, msrpMaxYear), salesMaxYear) - ModelYear.MinYear + 1;
            if (vd.Sales.Length < maxYears)
            {
                double[] sales = new double[maxYears];
                vd.Sales.CopyTo(sales, 0);
                vd.Sales = sales;
            }
            if (vd.Msrp.Length < maxYears)
            {
                double[] msrp = new double[maxYears];
                vd.Msrp.CopyTo(msrp, 0);
                vd.Msrp = msrp;
            }
            if (vd.Price.Length < maxYears)
            {
                double[] price = new double[maxYears];
                vd.Price.CopyTo(price, 0);
                vd.Price = price;
            }
            for (int i = 0; i < maxYears; i++)
            {
                if      (vd.Msrp[i] == 0 && vd.Price[i] != 0) { vd.Msrp [i] = vd.Price[i]; }
                else if (vd.Msrp[i] != 0 && vd.Price[i] == 0) { vd.Price[i] = vd.Msrp [i]; }
            }
            return vd;
        }
        private int[] ParseFile_LoadVehicle_RefreshRedesign(string[] years)
        {
            int count = years.Length;
            int[] iYears = new int[count];
            int iCount = 0;
            for (int i = 0; i < count; i++)
            {
                int year = Global.GetInt32(years[i]);
                if (year > 1900)
                {    
                    bool duplicate = false;
                    for (int j = 0; j < iCount; j++)
                    {
                        if (iYears[j] == year) { duplicate = true; break; }
                    }
                    if (!duplicate) { iYears[iCount++] = year; }
                }
            }
            int[] actualYears = new int[iCount];
            Array.Copy(iYears, actualYears, iCount);
            Array.Sort(actualYears);
            return actualYears;
        }
        private double[] ParseFile_LoadVehicleSMP(Input input, int row, int column, int minYear, int maxYear)
        {
            double[] values = new double[Math.Max(15, maxYear - 1999)];
            int years = maxYear - minYear + 1;
            int offset = minYear - ModelYear.MinYear;
            for (int i = 0; i < years; i++)
            {
                values[i + offset] = input.GetDouble(row, i + column);
            }
            return values;
        }
        private EngineCollection[] ParseFile_LoadEngines(Input input, int index, ManufacturerCollection mfrs)
        {
            EngineCollection[] ec;
            input.ActivateWorksheet(index);
            int rows = input.Rows;
            int cols = input.Columns;
            string[] names = {   
                                 "Engine Code"            ,     
                                 "Manufacturer"           ,     
                                 "Name"                   ,     
                                 "Configuration"          ,     
                                 "Fuel"                   ,     
                                 "Engine Oil Viscosity"   ,     
                                 "Cycle"                  ,     
                                 "Air/Fuel Ratio"         ,     
                                 "Fuel System"            ,     
                                 "Aspiration"             ,     
                                 "Valvetrain Design"      ,     
                                 "Valve Actuation/Timing" ,     
                                 "Valve Lift"             ,     
                                 "Cylinders"              ,     
                                 "Valves/Cylinder"        ,     
                                 "Deactivation"           ,     
                                 "Displacement"           ,     
                                 "Compression Ratio (Min)",     
                                 "Compression Ratio (Max)",     
                                 "Horsepower"             ,     
                                 "Torque"                 ,     
								 "Fuel Delivery System"   ,     
								 "Max. Horsepower"        ,     
								 "Max. Torque"            ,     
								 "Primary Fuel"           ,     
                                 "LUB"  , "EFR" , "CCPS" , "DVVLS", "DEACS", "ICP"  , "DCP"  , "DVVLD", "CVVL", "DEACD",
                                 "DEACO", "CCPO", "DVVLO", "CDOHC", "SGDI" , "CBRST", "TRBDS", "EGRB" , "DSLT", "DSLC"
                             };
            int[] indexes = new int[names.Length];
            for (int i = 0; i < names.Length; i++) { indexes[i] = -1; }
            for (int i = 0; i < cols; i++)
            {
                string column = input.GetString(0, i);
                if (input.Compare(column, "Code", false)) { indexes[0] = i; }
                else
                {
                    for (int j = 0; j < names.Length; j++)
                    {
                        if (input.Compare(column, names[j], true)) { indexes[j] = i; break; }
                    }
                }
            }
			if (indexes[4] == -1 && indexes[24] != -1)
				indexes[4] = indexes[24];				
			else if (indexes[4] != -1 && indexes[24] == -1)
				indexes[24] = indexes[4];				
			if (indexes[8] == -1 && indexes[21] != -1)
				indexes[8] = indexes[21];				
			else if (indexes[8] != -1 && indexes[21] == -1)
				indexes[21] = indexes[8];				
			if (indexes[19] == -1 && indexes[22] != -1)
				indexes[19] = indexes[22];				
			else if (indexes[19] != -1 && indexes[22] == -1)
				indexes[22] = indexes[19];				
			if (indexes[20] == -1 && indexes[23] != -1)
				indexes[20] = indexes[23];				
			else if (indexes[20] != -1 && indexes[23] == -1)
				indexes[23] = indexes[20];				
            input.VerifyIndexes(indexes, names, 0, 21);
            ec = new EngineCollection[mfrs.Count];
            for (int i = 0; i < mfrs.Count; i++)
            {
                ec[i] = new EngineCollection(32);
            }
            for (int i = 1; i < rows; i++)
            {   
                EngineDescription ed = this.ParseFile_LoadEngine(input, i, indexes);
                int mfrIdx = this.ParseFile_GetManufacturerIndex(input, mfrs, ed.Manufacturer);
                if (mfrIdx != -1)
                {   
                    ec[mfrIdx].Add(new Engine(ed));
                }
            }
            return ec;
        }
        private EngineDescription ParseFile_LoadEngine(Input input, int row, int[] indexes)
        {
            EngineDescription ed = new EngineDescription();
            ed.Code                 = input.GetInt32 (row, indexes[ 0]);
            ed.Manufacturer         = input.GetString(row, indexes[ 1]);
            ed.Name                 = input.GetString(row, indexes[ 2]);
            ed.Configuration        = input.GetString(row, indexes[ 3]);
            ed.Fuel                 =(input.GetString(row, indexes[ 4]) == "D") ? "D" : "G";
            ed.EngineOilViscosity   = input.GetString(row, indexes[ 5]);
            ed.Cycle                = input.GetChar  (row, indexes[ 6]);
            ed.AirFuelRatio         = input.GetDouble(row, indexes[ 7]);
            ed.FuelSystem           = input.GetString(row, indexes[ 8]);
            ed.Aspiration           = input.GetChar  (row, indexes[ 9]);
            ed.ValvetrainDesign     = input.GetString(row, indexes[10]);
            ed.ValveActuationTiming = input.GetString(row, indexes[11]);
            ed.ValveLift            = input.GetString(row, indexes[12]);
            ed.Cylinders            = input.GetInt32 (row, indexes[13]);
            ed.ValvesPerCylinder    = input.GetInt32 (row, indexes[14]);
            ed.Deactivation         = input.GetDouble(row, indexes[15]);
            ed.Displacement         = input.GetDouble(row, indexes[16]);
            ed.MinCompressionRatio  = this.GetCompressionRatio(input.GetString(row, indexes[17]));
            ed.MaxCompressionRatio  = this.GetCompressionRatio(input.GetString(row, indexes[18]));
            ed.Horsepower           = input.GetDouble(row, indexes[19]);
            ed.Torque               = input.GetDouble(row, indexes[20]);
            ed.Overrides.LUB   = this.ParseFile_ParseTechOverride((indexes[25] == -1) ? "" : input.GetString(row, indexes[25]));
            ed.Overrides.EFR   = this.ParseFile_ParseTechOverride((indexes[26] == -1) ? "" : input.GetString(row, indexes[26]));
            ed.Overrides.CCPS  = this.ParseFile_ParseTechOverride((indexes[27] == -1) ? "" : input.GetString(row, indexes[27]));
            ed.Overrides.DVVLS = this.ParseFile_ParseTechOverride((indexes[28] == -1) ? "" : input.GetString(row, indexes[28]));
            ed.Overrides.DEACS = this.ParseFile_ParseTechOverride((indexes[29] == -1) ? "" : input.GetString(row, indexes[29]));
            ed.Overrides.ICP   = this.ParseFile_ParseTechOverride((indexes[30] == -1) ? "" : input.GetString(row, indexes[30]));
            ed.Overrides.DCP   = this.ParseFile_ParseTechOverride((indexes[31] == -1) ? "" : input.GetString(row, indexes[31]));
            ed.Overrides.DVVLD = this.ParseFile_ParseTechOverride((indexes[32] == -1) ? "" : input.GetString(row, indexes[32]));
            ed.Overrides.CVVL  = this.ParseFile_ParseTechOverride((indexes[33] == -1) ? "" : input.GetString(row, indexes[33]));
            ed.Overrides.DEACD = this.ParseFile_ParseTechOverride((indexes[34] == -1) ? "" : input.GetString(row, indexes[34]));
            ed.Overrides.DEACO = this.ParseFile_ParseTechOverride((indexes[35] == -1) ? "" : input.GetString(row, indexes[35]));
            ed.Overrides.CCPO  = this.ParseFile_ParseTechOverride((indexes[36] == -1) ? "" : input.GetString(row, indexes[36]));
            ed.Overrides.DVVLO = this.ParseFile_ParseTechOverride((indexes[37] == -1) ? "" : input.GetString(row, indexes[37]));
            ed.Overrides.CDOHC = this.ParseFile_ParseTechOverride((indexes[38] == -1) ? "" : input.GetString(row, indexes[38]));
            ed.Overrides.SGDI  = this.ParseFile_ParseTechOverride((indexes[39] == -1) ? "" : input.GetString(row, indexes[39]));
            ed.Overrides.CBRST = this.ParseFile_ParseTechOverride((indexes[40] == -1) ? "" : input.GetString(row, indexes[40]));
            ed.Overrides.TRBDS = this.ParseFile_ParseTechOverride((indexes[41] == -1) ? "" : input.GetString(row, indexes[41]));
            ed.Overrides.EGRB  = this.ParseFile_ParseTechOverride((indexes[42] == -1) ? "" : input.GetString(row, indexes[42]));
            ed.Overrides.DSLT  = this.ParseFile_ParseTechOverride((indexes[43] == -1) ? "" : input.GetString(row, indexes[43]));
            ed.Overrides.DSLC  = this.ParseFile_ParseTechOverride((indexes[44] == -1) ? "" : input.GetString(row, indexes[44]));
            return ed;
        }
        private double GetCompressionRatio(string value)
        {
            double ratio = 0;
            if (value != "")
            {
                int index = value.IndexOf(":");
                if (index != -1)
                {   
                    ratio = Global.GetDouble(value.Substring(0, index)) / Global.GetDouble(value.Substring(index + 1));
                }
                else
                {   
                    ratio = Global.GetDouble(value);
                }
            }
            return ratio;
        }
        private TransmissionCollection[] ParseFile_LoadTransmissions(Input input, int index, ManufacturerCollection mfrs)
        {
            TransmissionCollection[] tc;
            input.ActivateWorksheet(index);
            int rows = input.Rows;
            int cols = input.Columns;
            string[] names = {   
                                 "Transmission Code"      ,     
                                 "Manufacturer"           ,     
                                 "Name"                   ,     
                                 "Type"                   ,     
                                 "Number of Forward Gears",     
                                 "Control"                ,     
                                 "Logic"                  ,     
								 "Clutch Type"            ,		
                                 "6MAN", "IATC", "CVT", "NAUTO", "DCTAM"
                             };
            int[] indexes = new int[names.Length];
            for (int i = 0; i < names.Length; i++) { indexes[i] = -1; }
            for (int i = 0; i < cols; i++)
            {
                string column = input.GetString(0, i);
                if (input.Compare(column, "Code", false)) { indexes[0] = i; }
                else
                {
                    for (int j = 0; j < names.Length; j++)
                    {
                        if (input.Compare(column, names[j], true)) { indexes[j] = i; break; }
                    }
                }
            }
			bool hasClutchType = false;
			if (indexes[5] == -1 && indexes[7] != -1)
			{
				indexes[5] = indexes[7];				
				hasClutchType = true;
			}
            input.VerifyIndexes(indexes, names, 0, 7);
            tc = new TransmissionCollection[mfrs.Count];
            for (int i = 0; i < mfrs.Count; i++)
            {
                tc[i] = new TransmissionCollection(32);
            }
            for (int i = 1; i < rows; i++)
            {   
                TransmissionDescription td = this.ParseFile_LoadTransmission(input, i, indexes, hasClutchType);
                int mfrIdx = this.ParseFile_GetManufacturerIndex(input, mfrs, td.Manufacturer);
                if (mfrIdx != -1)
                {   
                    tc[mfrIdx].Add(new Transmission(td));
                }
            }
            return tc;
        }
        private TransmissionDescription ParseFile_LoadTransmission(Input input, int row, int[] indexes, bool hasClutchType)
        {
            TransmissionDescription td =  new TransmissionDescription();
            td.Code            = input.GetInt32 (row, indexes[0]);
            td.Manufacturer    = input.GetString(row, indexes[1]);
            td.Name            = input.GetString(row, indexes[2]);
            td.NumForwardGears = input.GetInt32 (row, indexes[4]);
			if (hasClutchType)
			{	
				string ct			= input.GetString(row, indexes[7]);	
				string tt			= input.GetString(row, indexes[3]);	
				if (tt.ToUpper() == "M")
				{	
					td.Control = "M";	
					td.Type    = "C";	
				}
				else if (tt.ToUpper() == "A")
				{	
					td.Control = "A";	
					td.Type	   = "T";	
				}
				else
				{	
					td.Control = "A";	
					td.Type    = tt;
				}
			}
			else
			{	
				td.Control         = input.GetString(row, indexes[5]);
				td.Type            = input.GetString(row, indexes[3]);
			}
            td.Logic           = input.GetString(row, indexes[6]);
            td.Overrides.MAN6  = this.ParseFile_ParseTechOverride((indexes[ 8] == -1) ? "" : input.GetString(row, indexes[ 8]));
            td.Overrides.IATC  = this.ParseFile_ParseTechOverride((indexes[ 9] == -1) ? "" : input.GetString(row, indexes[ 9]));
            td.Overrides.CVT   = this.ParseFile_ParseTechOverride((indexes[10] == -1) ? "" : input.GetString(row, indexes[10]));
            td.Overrides.NAUTO = this.ParseFile_ParseTechOverride((indexes[11] == -1) ? "" : input.GetString(row, indexes[11]));
            td.Overrides.DCTAM = this.ParseFile_ParseTechOverride((indexes[12] == -1) ? "" : input.GetString(row, indexes[12]));
            return td;
        }
        private TechnologyOverride ParseFile_ParseTechOverride(string str)
        {
            switch (str.ToUpper())
            {
                case "USED"  : return TechnologyOverride.Used;
                case "SKIP"  : return TechnologyOverride.Skip;
                case "FORCE" : return TechnologyOverride.Force;
                case "UNUSED": return TechnologyOverride.Unused;
                default:       return TechnologyOverride.None;
            }
        }
        private int ParseFile_GetManufacturerIndex(Input input, ManufacturerCollection mfrs, string mfrName)
        {
            for (int i = 0; i < mfrs.Count; i++)
            {
                if (input.Compare(mfrs[i].Description.Name, mfrName, false)) { return i; }
            }
            return -1;
        }
        internal void SetModelingData(IndustryModelingData value)
        {
            this._modelingData = value;
        }
        public void UpdateMinMaxFpCw()
        {
            this._minFp = this._minCw = 99999;
            this._maxFp = this._maxCw =    -1;
            for (int i = 0, mfrCount = this._manufacturers.Count; i < mfrCount; i++)
            {   
                Manufacturer mfr = this._manufacturers[i];
                this.UpdateMinMaxFpCw_Helper(mfr.DomesticPassengerAutos);
                this.UpdateMinMaxFpCw_Helper(mfr.ImportedPassengerAutos);
                this.UpdateMinMaxFpCw_Helper(mfr.LightTrucks);
                this.UpdateMinMaxFpCw_Helper(mfr.HeavyLightTrucks);
            } 
        }
        void UpdateMinMaxFpCw_Helper(VehicleCollection vehs)
        {
            for (int i = 0, vehCount = vehs.Count; i < vehCount; i++)
            {
                VehicleDescription vd = vehs[i].Description;
                double fp = vd.Footprint, cw = vd.CurbWeight;
                if (fp < this._minFp) { this._minFp = fp; }
                if (fp > this._maxFp) { this._maxFp = fp; }
                if (cw < this._minCw) { this._minCw = cw; }
                if (cw > this._maxCw) { this._maxCw = cw; }
            } 
        }
        public Industry GetMergedFleet()
        {
            Industry ind = this.Clone();
            ind.MergeManufacturers();
            return ind;
        }
        void MergeManufacturers()
        {
            VehicleCollection      vehs = new VehicleCollection();
            EngineCollection       engs = new EngineCollection();
            TransmissionCollection trns = new TransmissionCollection();
            for (int i = 0, mfrCount = this._manufacturers.Count; i < mfrCount; i++)
            {   
                vehs.AddRange(this._manufacturers[i].DomesticPassengerAutos);
                vehs.AddRange(this._manufacturers[i].ImportedPassengerAutos);
                vehs.AddRange(this._manufacturers[i].LightTrucks           );
                vehs.AddRange(this._manufacturers[i].DiscardedVehicles     );
                engs.AddRange(this._manufacturers[i].Engines               );
                trns.AddRange(this._manufacturers[i].Transmissions         );
            }
            for (int i = 0, mfrCount = this._manufacturers.Count; i < mfrCount; i++)
            {
                vehs.AddRange(this._manufacturers[i].HeavyLightTrucks      );
            }
            int vehIndex = 1, engIndex = 1, trnIndex = 1;
            for (int i = 0, vehCount = vehs.Count; i < vehCount; i++)
            {   
                string mfrName = vehs[i].Description.Manufacturer;
                vehs[i].Description.Manufacturer = "MERGED FLEET";
                vehs[i].Description.Code         = vehIndex;
                vehs[i].Description.Model        = mfrName + "_" + vehs[i].Description.Model;
                vehs[i].Description.Nameplate    = mfrName + "_" + vehs[i].Description.Nameplate;
                for (int j = 0; j < vehs[i].Successors.Count; j++)
                {
                    vehs[i].Successors[j].Description.PredecessorCode = vehIndex;
                }
                if (vehs[i].Predecessor == null) { vehs[i].Description.PredecessorCode = 0; }
                vehs[i].Successors.Clear();
                vehIndex++;
            }
            for (int i = 0, engCount = engs.Count; i < engCount; i++)
            {   
                string mfrName = engs[i].Description.Manufacturer;
                engs[i].Description.Manufacturer = "MERGED FLEET";
                engs[i].Description.Code         = engIndex;
                engs[i].Description.Name         = mfrName + "_" + engs[i].Description.Name;
                for (int j = 0, vehCount = engs[i].Vehicles.Count; j < vehCount; j++)
                {   
                    engs[i].Vehicles[j].Description.EngineCode = engIndex;
                }
                engIndex++;
            }
            for (int i = 0, trnCount = trns.Count; i < trnCount; i++)
            {   
                string mfrName = trns[i].Description.Manufacturer;
                trns[i].Description.Manufacturer = "MERGED FLEET";
                trns[i].Description.Code         = trnIndex;
                trns[i].Description.Name         = mfrName + "_" + trns[i].Description.Name;
                for (int j = 0, vehCount = trns[i].Vehicles.Count; j < vehCount; j++)
                {   
                    trns[i].Vehicles[j].Description.TransmissionCode = trnIndex;
                }
                trnIndex++;
            }
            ManufacturerDescription md = new ManufacturerDescription();
            md.Code                   = 1;
            md.Name                   = "MERGED FLEET";
            md.CostAllocationStrategy = 0;
            md.AverageMY2002LtCw      = 4329.48480892067;
            md.DiscountRate           = 0.07;
            md.Optimize               = true;
            md.WillingToPayFines      = new bool  [Math.Max(15, this._maxYear.Year - ModelYear.MinYear + 1)];
            md.AvailableCredits       = new double[Math.Max(15, this._maxYear.Year - ModelYear.MinYear + 1)];
            Array.Clear(md.WillingToPayFines, 0, md.WillingToPayFines.Length);
            Array.Clear(md.AvailableCredits , 0, md.AvailableCredits.Length);
            md.CreditsApplyToBaseline = false;
            int minYear, maxYear;
            Manufacturer mfr = new Manufacturer(md);
            mfr.Initialize(vehs, engs, trns);
            mfr.Validate(out minYear, out maxYear);
            mfr.SetIndex(0);
            this._allManufacturers.Clear();
            this._manufacturers   .Clear();
            this._allManufacturers.Add(mfr);
            this._manufacturers   .Add(mfr);
        }
        #endregion
        #region 
        public int AllVehicleCount
        {
            get
            {
                int count = 0;
                for (int i = 0; i < this._allManufacturers.Count; i++)
                {
                    count += this._allManufacturers[i].VehicleCount;
                }
                return count;
            }
        }
        public int ModelYearVehicleCount
        {
            get
            {
                int count = 0;
                for (int i = 0; i < this._manufacturers.Count; i++)
                {
                    count += this._manufacturers[i].ModelYearVehicleCount;
                }
                return count;
            }
        }
        public int VehicleCount
        {
            get
            {
                int count = 0;
                for (int i = 0; i < this._manufacturers.Count; i++)
                {
                    count += this._manufacturers[i].VehicleCount;
                }
                return count;
            }
        }
        public int AllManufacturerCount { get { return (this._allManufacturers == null) ? 0 : this._allManufacturers.Count; } }
        public int ManufacturerCount    { get { return (this._manufacturers == null) ? 0 : this._manufacturers.Count; } }
        public ManufacturerCollection AllManufacturers { get { return this._allManufacturers; } }
        public ManufacturerCollection Manufacturers    { get { return this._manufacturers; } }
        public IndustryModelingData ModelingData { get { return this._modelingData; } }
        public ModelYear MinYear { get { return this._minYear; } }
        public ModelYear MaxYear { get { return this._maxYear; } }
        public double MinFp { get { return this._minFp; } }
        public double MaxFp { get { return this._maxFp; } }
        public double MinCw { get { return this._minCw; } }
        public double MaxCw { get { return this._maxCw; } }
        #endregion
        #region 
        private ManufacturerCollection _allManufacturers;
        private ManufacturerCollection _manufacturers;
        private IndustryModelingData _modelingData;
        private ModelYear _minYear;
        private ModelYear _maxYear;
        [NonSerialized]
        private double _minFp;
        [NonSerialized]
        private double _maxFp;
        [NonSerialized]
        private double _minCw;
        [NonSerialized]
        private double _maxCw;
        #endregion
    }
}

