#region << Using Directives >>
using System;
using System.Drawing;
using System.IO;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using Excel   = Microsoft.Office.Interop.Excel;
using IOPath  = System.IO.Path;
using Missing = System.Reflection.Missing;
#endregion
namespace Volpe.Utils
{
    [Serializable]
    public class XlUtilities
    {
        #region 
        public XlUtilities(string path)
            : this(path, false)
        {
        }
        public XlUtilities(string path, bool readOnly)
            : this(path, readOnly, null)
        {
        }
        public XlUtilities(string path, bool readOnly, string password)
            : this(path, readOnly, password, null, XlCryptoSettings.OfficeStandard)
        {
        }
        public XlUtilities(string path, bool readOnly, string passwordToOpen, string passwordToModify,
            XlCryptoSettings encryptionOptions)
        {
            this.Open(path, readOnly, passwordToOpen, passwordToModify, encryptionOptions);
        }
        ~XlUtilities()
        {
            if (!this._disposed) { this.Close(false); }
        }
        #endregion
        #region 
        #region 
        public static bool ShowExcelApplication()
        {
            if (XlUtilities.ExcelApp != null && XlUtilities.ExcelApp.Workbooks.Count > 0)
            {
                try
                {
                    XlUtilities.ExcelApp.WindowState = Excel.XlWindowState.xlMaximized;
                    XlUtilities.ExcelApp.Visible = true;
                    return true;
                } catch {}
            }
            return false;
        }
        public static void HideExcelApplication()
        {
            if (XlUtilities.ExcelApp != null && XlUtilities.ExcelApp.Workbooks.Count > 0)
            {
                XlUtilities.ExcelApp.Visible = false;
            }
        }
        #endregion
        #region 
        protected void LoadExcel()
        {
            if (XlUtilities.ExcelApp == null)
            {
                XlUtilities.ExcelApp = new Excel.Application();
                XlUtilities.ExcelApp.DisplayAlerts = false;
                XlUtilities.ExcelApp.UserControl   = false;
                XlUtilities.ExcelApp.Visible       = false;
            }
        }
        protected void UnLoadExcel()
        {
            if (XlUtilities.ExcelApp != null && XlUtilities.ExcelApp.Workbooks.Count == 0)
            {
                XlUtilities.ExcelApp.Quit();
                Marshal.ReleaseComObject(XlUtilities.ExcelApp);
                XlUtilities.ExcelApp = null;
            }
        }
        #endregion
        #region 
        protected void Open(string path, bool readOnly, string passwordToOpen, string passwordToModify,
            XlCryptoSettings encryptionOptions)
        {
            if (readOnly && !File.Exists(path)) { throw new FileNotFoundException("The specified input file was not found.", path); }
            FileInfo fi = new FileInfo(path);
            if (!fi.Extension.StartsWith(".xl")) { throw new IOException("The specified input file is not an Excel workbook file."); }
            fi = null;
            path = IOPath.GetFullPath(path);
            this.LoadExcel();
            if (File.Exists(path))
            {   
                this._workbook = XlUtilities.ExcelApp.Workbooks.Open(path, NIL, readOnly, NIL,
                    (passwordToOpen == null) ? NIL : passwordToOpen, (passwordToModify == null) ? NIL : passwordToModify,
                    NIL, NIL, NIL, NIL, NIL, NIL, NIL, NIL, NIL);
                this._readOnly = readOnly;
            }
            else
            {   
                this._workbook = XlUtilities.ExcelApp.Workbooks.Add(NIL);
                this.SaveAs(path, passwordToOpen, passwordToModify, encryptionOptions);
                this._readOnly = false;
            }
            this.Activate();
            this._path = path;
            this._passwordProtected = (passwordToOpen != null);
            this._opened   = true;
            this._disposed = false;
        }
        public void Save()
        {
            if      (this._disposed) { throw new ObjectDisposedException(base.ToString()); }
            else if (this._readOnly)
            {
                throw new InvalidOperationException("Cannot save a workbook that is marked as read-only.  " +
                    "Try using Save As instead.");
            }
            else { this._workbook.Save(); }
        }
        public void SaveAs(string path, string passwordToOpen, string passwordToModify, XlCryptoSettings encryptionOptions)
        {
            path = IOPath.GetFullPath(path);
            if (this._disposed) { throw new ObjectDisposedException(base.ToString()); }
            else
            {   
                if (!encryptionOptions.IsEmpty)
                {
                    this._workbook.SetPasswordEncryptionOptions(encryptionOptions.EncryptionProvider,
                        encryptionOptions.EncryptionAlgorithm, encryptionOptions.EncryptionKeyLength,
                        encryptionOptions.EncryptDocumentProperties);
                }
                this._workbook.SaveAs(path, NIL, (passwordToOpen == null) ? NIL : passwordToOpen,
                    (passwordToModify == null) ? NIL : passwordToModify, (passwordToModify != null), false,
                    Excel.XlSaveAsAccessMode.xlNoChange, NIL, NIL, NIL, NIL, NIL);
                this._path = path;
                this._readOnly = false;
                this._passwordProtected = (passwordToOpen != null);
            }
        }
        public void Close()
        {
            this.Close(false);
        }
        public void Close(bool saveChanges)
        {
            if (this._opened && this._disposed) { throw new ObjectDisposedException(base.ToString()); }
            else
            {
                if (this._opened)
                {
                    this._workbook.Close(saveChanges && !this._readOnly, Type.Missing, Type.Missing);
                    Marshal.ReleaseComObject(this._workbook);
                    this._disposed = true;
                    this._opened = false;
                }
                this.UnLoadExcel();
                GC.Collect();
                GC.WaitForPendingFinalizers();
                GC.Collect();
            }
        }
        public void Activate()
        {
            this.Workbook.Activate();
        }
        #endregion
        #region 
        public object[,] GetData()
        {
            return this.GetData(false);
        }
        public object[,] GetData(XlCell cell1, XlCell cell2)
        {
            return GetData(cell1, cell2, false);
        }
        public object[,] GetData(bool getFormula)
        {
            return this.GetData(new XlCell(1, 1), this.GetLastCell(), getFormula);
        }
        public object[,] GetData(XlCell cell1, XlCell cell2, bool getFormula)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.Workbook.ActiveSheet;
            return GetData(ref sheet, cell1, cell2, getFormula, true);
        }
        public object[,] GetData(object index)
        {
            return this.GetData(index, false);
        }
        public object[,] GetData(object index, XlCell cell1, XlCell cell2)
        {
            return GetData(index, cell1, cell2, false);
        }
        public object[,] GetData(object index, bool getFormula)
        {
            return this.GetData(index, new XlCell(1, 1), this.GetLastCell(index), getFormula);
        }
        public object[,] GetData(object index, XlCell cell1, XlCell cell2, bool getFormula)
        {
            Excel.Worksheet sheet = this.GetWorksheet(index);
            return GetData(ref sheet, cell1, cell2, getFormula, true);
        }
        protected object[,] GetData(ref Excel.Worksheet sheet, XlCell cell1, XlCell cell2, bool getFormula,
            bool releaseComObject)
        {
            Excel.Range range = this.GetRange(ref sheet, cell1, cell2, false);
            object[,] data = (getFormula) ? (object[,])range.Formula :
                (object[,])range.get_Value(Excel.XlRangeValueDataType.xlRangeValueDefault);
            Marshal.ReleaseComObject(range);
            range = null;
            if (releaseComObject)
            {
                Marshal.ReleaseComObject(sheet);
                sheet = null;
            }
            return data;
        }
        public void SetData(object[,] buffer)
        {
            this.SetData(buffer, new XlCell(1, 1));
        }
        public void SetData(object[,] buffer, XlCell cell)
        {
            int rows = buffer.GetUpperBound(0) + 1 - buffer.GetLowerBound(0);
            int cols = buffer.GetUpperBound(1) + 1 - buffer.GetLowerBound(1);
            this.SetData(buffer, cell, new XlCell(rows, cols));
        }
        public void SetData(object[,] buffer, XlCell cell1, XlCell cell2)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.Workbook.ActiveSheet;
            this.SetData(ref sheet, buffer, cell1, cell2, true);
        }
        public void SetData(object index, object[,] buffer)
        {
            this.SetData(index, buffer, new XlCell(1, 1));
        }
        public void SetData(object index, object[,] buffer, XlCell cell)
        {
            int rows = buffer.GetUpperBound(0) + 1 - buffer.GetLowerBound(0);
            int cols = buffer.GetUpperBound(1) + 1 - buffer.GetLowerBound(1);
            this.SetData(index, buffer, cell, new XlCell(rows, cols));
        }
        public void SetData(object index, object[,] buffer, XlCell cell1, XlCell cell2)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.GetWorksheet(index);
            this.SetData(ref sheet, buffer, cell1, cell2, true);
        }
        protected void SetData(ref Excel.Worksheet sheet, object[,] buffer, XlCell cell1, XlCell cell2,
            bool releaseComObject)
        {
            Excel.Range range = this.GetRange(ref sheet, cell1, cell2, false);
            range.set_Value(NIL, buffer);
            Marshal.ReleaseComObject(range);
            range = null;
            if (releaseComObject)
            {
                Marshal.ReleaseComObject(sheet);
                sheet = null;
            }
        }
        #endregion
        #region 
        public void AutoFilter(XlCell cell1, XlCell cell2)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.Workbook.ActiveSheet;
            this.AutoFilter(ref sheet, cell1, cell2, true);
        }
        public void AutoFilter(object index, XlCell cell1, XlCell cell2)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.GetWorksheet(index);
            this.AutoFilter(ref sheet, cell1, cell2, true);
        }
        protected void AutoFilter(ref Excel.Worksheet sheet, XlCell cell1, XlCell cell2, bool releaseComObject)
        {
            Excel.Range range = this.GetRange(ref sheet, cell1, cell2, false);
            Marshal.ReleaseComObject(range);
            range = null;
            if (releaseComObject)
            {
                Marshal.ReleaseComObject(sheet);
                sheet = null;
            }
        }
        public void SortData(XlCell cell1, XlCell cell2, int key, bool descending)
        {
            this.SortData(cell1, cell2, key, descending, -1, false);
        }
        public void SortData(XlCell cell1, XlCell cell2, int key1, bool descending1, int key2, bool descending2)
        {
            this.SortData(cell1, cell2, key1, descending1, key2, descending2, -1, false);
        }
        public void SortData(XlCell cell1, XlCell cell2, int key1, bool descending1, int key2, bool descending2,
            int key3, bool descending3)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.Workbook.ActiveSheet;
            this.SortData(ref sheet, cell1, cell2, key1, descending1, key2, descending2, key3, descending3, true);
        }
        public void SortData(object index, XlCell cell1, XlCell cell2, int key, bool descending)
        {
            this.SortData(index, cell1, cell2, key, descending, -1, false);
        }
        public void SortData(object index, XlCell cell1, XlCell cell2, int key1, bool descending1, int key2,
            bool descending2)
        {
            this.SortData(index, cell1, cell2, key1, descending1, key2, descending2, -1, false);
        }
        public void SortData(object index, XlCell cell1, XlCell cell2, int key1, bool descending1, int key2, bool descending2,
            int key3, bool descending3)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.GetWorksheet(index);
            this.SortData(ref sheet, cell1, cell2, key1, descending1, key2, descending2, key3, descending3, true);
        }
        protected void SortData(ref Excel.Worksheet sheet, XlCell cell1, XlCell cell2, int key1, bool descending1,
            int key2, bool descending2, int key3, bool descending3, bool releaseComObject)
        {
            Excel.Range range = this.GetRange(ref sheet, cell1, cell2, false);
            object sk1 = (key1 <= 0) ? null : range.Cells[key1, NIL];
            object sk2 = (key2 <= 0) ? null : range.Cells[key2, NIL];
            object sk3 = (key3 <= 0) ? null : range.Cells[key3, NIL];
            Excel.XlSortOrder so1 = (descending1) ? Excel.XlSortOrder.xlDescending : Excel.XlSortOrder.xlAscending;
            Excel.XlSortOrder so2 = (descending2) ? Excel.XlSortOrder.xlDescending : Excel.XlSortOrder.xlAscending;
            Excel.XlSortOrder so3 = (descending3) ? Excel.XlSortOrder.xlDescending : Excel.XlSortOrder.xlAscending;
            range.Sort(sk1, so1, sk2, NIL, so2, sk3, so3, Excel.XlYesNoGuess.xlNo, 1, false,
                Excel.XlSortOrientation.xlSortColumns, Excel.XlSortMethod.xlPinYin, Excel.XlSortDataOption.xlSortNormal,
                Excel.XlSortDataOption.xlSortNormal, Excel.XlSortDataOption.xlSortNormal);
            if (sk3 != null) { Marshal.ReleaseComObject(sk3); }
            if (sk2 != null) { Marshal.ReleaseComObject(sk2); }
            if (sk1 != null) { Marshal.ReleaseComObject(sk1); }
            Marshal.ReleaseComObject(range);
            sk1 = sk2 = sk3 = null;
            range = null;
            if (releaseComObject)
            {
                Marshal.ReleaseComObject(sheet);
                sheet = null;
            }
        }
        #endregion
        #region 
        public void SetFont(XlCell cell1, XlCell cell2, XlFont font)
        {
            this.SetFont(cell1, cell2, font, XlColor.Automatic);
        }
        public void SetFont(XlCell cell1, XlCell cell2, XlFont font, XlColor fillColor)
        {
            this.SetFont(cell1, cell2, font, fillColor, null, 0);
        }
        public void SetFont(XlCell cell1, XlCell cell2, XlFont font, string numberFormat)
        {
            this.SetFont(cell1, cell2, font, XlColor.Automatic, numberFormat, 0);
        }
        public void SetFont(XlCell cell1, XlCell cell2, XlFont font, int indent)
        {
            this.SetFont(cell1, cell2, font, XlColor.Automatic, null, indent);
        }
        public void SetFont(XlCell cell1, XlCell cell2, XlFont font, XlColor fillColor, string numberFormat, int indent)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.Workbook.ActiveSheet;
            this.SetFont(ref sheet, cell1, cell2, font, fillColor, numberFormat, indent, true);
        }
        public void SetFont(object index, XlCell cell1, XlCell cell2, XlFont font)
        {
            this.SetFont(index, cell1, cell2, font, XlColor.Automatic);
        }
        public void SetFont(object index, XlCell cell1, XlCell cell2, XlFont font, XlColor fillColor)
        {
            this.SetFont(index, cell1, cell2, font, fillColor, null, 0);
        }
        public void SetFont(object index, XlCell cell1, XlCell cell2, XlFont font, string numberFormat)
        {
            this.SetFont(index, cell1, cell2, font, XlColor.Automatic, numberFormat, 0);
        }
        public void SetFont(object index, XlCell cell1, XlCell cell2, XlFont font, int indent)
        {
            this.SetFont(index, cell1, cell2, font, XlColor.Automatic, null, indent);
        }
        public void SetFont(object index, XlCell cell1, XlCell cell2, XlFont font, XlColor fillColor, string numberFormat,
            int indent)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.GetWorksheet(index);
            this.SetFont(ref sheet, cell1, cell2, font, fillColor, numberFormat, indent, true);
        }
        protected void SetFont(ref Excel.Worksheet sheet, XlCell cell1, XlCell cell2, XlFont font, XlColor fillColor,
            string numberFormat, int indent, bool releaseComObject)
        {
            if (indent < 0)
            {
                throw new ArgumentOutOfRangeException("indent", indent,
                    "The value of indentation must be greater than or equal to zero.");
            }
            Excel.Range range = this.GetRange(ref sheet, cell1, cell2, false);
            range.Font.Name          = font.Name;
            range.Font.Size          = font.Size;
            range.Font.Italic        = font.Italic;
            range.Font.Bold          = font.Bold;
            range.Font.Underline     = font.Underline;
            range.Font.Strikethrough = font.Strikethrough;
            range.Font.Subscript     = font.Subscript;
            range.Font.Superscript   = font.Superscript;
            range.Font.ColorIndex    = (int)font.Color;
            if (fillColor != XlColor.Automatic)
            {
                range.Interior.ColorIndex = (int)fillColor;
                range.Interior.Pattern = Excel.Constants.xlSolid;
            }
            range.NumberFormat = numberFormat;
            range.IndentLevel = indent;
            Marshal.ReleaseComObject(range);
            range = null;
            if (releaseComObject)
            {
                Marshal.ReleaseComObject(sheet);
                sheet = null;
            }
        }
        public void SetFontColor(XlCell cell1, XlCell cell2, XlColor color)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.Workbook.ActiveSheet;
            this.SetFontColor(ref sheet, cell1, cell2, color, true);
        }
        public void SetFontColor(object index, XlCell cell1, XlCell cell2, XlColor color)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.GetWorksheet(index);
            this.SetFontColor(ref sheet, cell1, cell2, color, true);
        }
        protected void SetFontColor(ref Excel.Worksheet sheet, XlCell cell1, XlCell cell2, XlColor color,
            bool releaseComObject)
        {
            Excel.Range range = this.GetRange(ref sheet, cell1, cell2, false);
            range.Font.ColorIndex = (int)color;
            Marshal.ReleaseComObject(range);
            range = null;
            if (releaseComObject)
            {
                Marshal.ReleaseComObject(sheet);
                sheet = null;
            }
        }
        public void SetFillColor(XlCell cell1, XlCell cell2, XlColor color)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.Workbook.ActiveSheet;
            this.SetFillColor(ref sheet, cell1, cell2, color, true);
        }
        public void SetFillColor(object index, XlCell cell1, XlCell cell2, XlColor color)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.GetWorksheet(index);
            this.SetFillColor(ref sheet, cell1, cell2, color, true);
        }
        protected void SetFillColor(ref Excel.Worksheet sheet, XlCell cell1, XlCell cell2, XlColor color, bool releaseComObject)
        {
            Excel.Range range = this.GetRange(ref sheet, cell1, cell2, false);
            if (color == XlColor.Automatic)
            {
                range.Interior.ColorIndex = XlUtilities.xlNone;
            }
            else
            {
                range.Interior.ColorIndex = (int)color;
                range.Interior.Pattern = Excel.Constants.xlSolid;
            }
            Marshal.ReleaseComObject(range);
            range = null;
            if (releaseComObject)
            {
                Marshal.ReleaseComObject(sheet);
                sheet = null;
            }
        }
        public void SetNumberFormat(XlCell cell1, XlCell cell2, string format)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.Workbook.ActiveSheet;
            this.SetNumberFormat(ref sheet, cell1, cell2, format, true);
        }
        public void SetNumberFormat(object index, XlCell cell1, XlCell cell2, string format)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.GetWorksheet(index);
            this.SetNumberFormat(ref sheet, cell1, cell2, format, true);
        }
        protected void SetNumberFormat(ref Excel.Worksheet sheet, XlCell cell1, XlCell cell2, string format, bool releaseComObject)
        {
            Excel.Range range = this.GetRange(ref sheet, cell1, cell2, false);
            range.NumberFormat = format;
            Marshal.ReleaseComObject(range);
            range = null;
            if (releaseComObject)
            {
                Marshal.ReleaseComObject(sheet);
                sheet = null;
            }
        }
        public void SetIndent(XlCell cell1, XlCell cell2, int indent)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.Workbook.ActiveSheet;
            this.SetIndent(ref sheet, cell1, cell2, indent, true);
        }
        public void SetIndent(object index, XlCell cell1, XlCell cell2, int indent)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.GetWorksheet(index);
            this.SetIndent(ref sheet, cell1, cell2, indent, true);
        }
        protected void SetIndent(ref Excel.Worksheet sheet, XlCell cell1, XlCell cell2, int indent,
            bool releaseComObject)
        {
            if (indent < 0)
            {
                throw new ArgumentOutOfRangeException("indent", indent,
                    "The value of indentation must be greater than or equal to zero.");
            }
            Excel.Range range = this.GetRange(ref sheet, cell1, cell2, false);
            range.IndentLevel = indent;
            Marshal.ReleaseComObject(range);
            range = null;
            if (releaseComObject)
            {
                Marshal.ReleaseComObject(sheet);
                sheet = null;
            }
        }
        #endregion
        #region 
        public void MergeCells(XlCell cell1, XlCell cell2)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.Workbook.ActiveSheet;
            this.MergeCells(ref sheet, cell1, cell2, true);
        }
        public void MergeCells(object index, XlCell cell1, XlCell cell2)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.GetWorksheet(index);
            this.MergeCells(ref sheet, cell1, cell2, true);
        }
        protected void MergeCells(ref Excel.Worksheet sheet, XlCell cell1, XlCell cell2, bool releaseComObject)
        {
            Excel.Range range = this.GetRange(ref sheet, cell1, cell2, false);
            range.Merge(NIL);
            Marshal.ReleaseComObject(range);
            range = null;
            if (releaseComObject)
            {
                Marshal.ReleaseComObject(sheet);
                sheet = null;
            }
        }
        public void AlignText(XlCell cell1, XlCell cell2, XlTextAlignment textAlign)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.Workbook.ActiveSheet;
            this.AlignText(ref sheet, cell1, cell2, textAlign, true);
        }
        public void AlignText(object index, XlCell cell1, XlCell cell2, XlTextAlignment textAlign)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.GetWorksheet(index);
            this.AlignText(ref sheet, cell1, cell2, textAlign, true);
        }
        protected void AlignText(ref Excel.Worksheet sheet, XlCell cell1, XlCell cell2, XlTextAlignment textAlign,
            bool releaseComObject)
        {
            Excel.Range range = this.GetRange(ref sheet, cell1, cell2, false);
            Excel.XlHAlign hAlign =
                (textAlign.Horizontal == XlHAlign.Center) ? Excel.XlHAlign.xlHAlignCenter :
                (textAlign.Horizontal == XlHAlign.Left  ) ? Excel.XlHAlign.xlHAlignLeft :
                (textAlign.Horizontal == XlHAlign.Right ) ? Excel.XlHAlign.xlHAlignRight :
                Excel.XlHAlign.xlHAlignGeneral;
            Excel.XlVAlign vAlign =
                (textAlign.Vertical == XlVAlign.Center) ? Excel.XlVAlign.xlVAlignCenter :
                (textAlign.Vertical == XlVAlign.Top   ) ? Excel.XlVAlign.xlVAlignTop :
                Excel.XlVAlign.xlVAlignBottom;
            range.HorizontalAlignment = hAlign;
            range.VerticalAlignment   = vAlign;
            Marshal.ReleaseComObject(range);
            range = null;
            if (releaseComObject)
            {
                Marshal.ReleaseComObject(sheet);
                sheet = null;
            }
        }
        public void AutoFitText(XlCell cell1, XlCell cell2)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.Workbook.ActiveSheet;
            this.AutoFitText(ref sheet, cell1, cell2, true);
        }
        public void AutoFitText(object index, XlCell cell1, XlCell cell2)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.GetWorksheet(index);
            this.AutoFitText(ref sheet, cell1, cell2, true);
        }
        protected void AutoFitText(ref Excel.Worksheet sheet, XlCell cell1, XlCell cell2, bool releaseComObject)
        {
            Excel.Range range = this.GetRange(ref sheet, cell1, cell2, false);
            range.RowHeight   = 128;
            range.ColumnWidth = 128;
            range.Rows.AutoFit();
            range.Columns.AutoFit();
            Marshal.ReleaseComObject(range);
            range = null;
            if (releaseComObject)
            {
                Marshal.ReleaseComObject(sheet);
                sheet = null;
            }
        }
        public void SetRowHeight(XlCell cell1, XlCell cell2, double rowHeight)
        {
            this.SetHeightAndWidth(cell1, cell2, rowHeight, -1);
        }
        public void SetRowHeight(object index, XlCell cell1, XlCell cell2, double rowHeight)
        {
            this.SetHeightAndWidth(index, cell1, cell2, rowHeight, -1);
        }
        public void SetColumnWidth(XlCell cell1, XlCell cell2, double columnWidth)
        {
            this.SetHeightAndWidth(cell1, cell2, -1, columnWidth);
        }
        public void SetColumnWidth(object index, XlCell cell1, XlCell cell2, double columnWidth)
        {
            this.SetHeightAndWidth(index, cell1, cell2, -1, columnWidth);
        }
        public void SetHeightAndWidth(XlCell cell1, XlCell cell2, double rowHeight, double columnWidth)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.Workbook.ActiveSheet;
            this.SetHeightAndWidth(ref sheet, cell1, cell2, rowHeight, columnWidth, true);
        }
        public void SetHeightAndWidth(object index, XlCell cell1, XlCell cell2, double rowHeight, double columnWidth)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.GetWorksheet(index);
            this.SetHeightAndWidth(ref sheet, cell1, cell2, rowHeight, columnWidth, true);
        }
        protected void SetHeightAndWidth(ref Excel.Worksheet sheet, XlCell cell1, XlCell cell2, double rowHeight,
            double columnWidth, bool releaseComObject)
        {
            Excel.Range range = this.GetRange(ref sheet, cell1, cell2, false);
            if (rowHeight   >= 0) { range.RowHeight   = rowHeight; }
            if (columnWidth >= 0) { range.ColumnWidth = columnWidth; }
            Marshal.ReleaseComObject(range);
            range = null;
            if (releaseComObject)
            {
                Marshal.ReleaseComObject(sheet);
                sheet = null;
            }
        }
        #endregion
        #region 
        public void SetBorders(XlCell cell1, XlCell cell2, bool exterior, bool interior, bool solid, bool thick,
            XlColor borderColor)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.Workbook.ActiveSheet;
            this.SetBorders(ref sheet, cell1, cell2, exterior, exterior, exterior, exterior, interior, solid, thick, borderColor,
                true);
        }
        public void SetBorders(XlCell cell1, XlCell cell2, bool top, bool bottom, bool left, bool right, bool solid,
            bool thick, XlColor borderColor)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.Workbook.ActiveSheet;
            this.SetBorders(ref sheet, cell1, cell2, top, bottom, left, right, false, solid, thick, borderColor, true);
        }
        public void SetBorders(object index, XlCell cell1, XlCell cell2, bool exterior, bool interior, bool solid,
            bool thick, XlColor borderColor)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.GetWorksheet(index);
            this.SetBorders(ref sheet, cell1, cell2, exterior, exterior, exterior, exterior, interior, solid, thick, borderColor,
                true);
        }
        public void SetBorders(object index, XlCell cell1, XlCell cell2, bool top, bool bottom, bool left, bool right,
            bool solid, bool thick, XlColor borderColor)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.GetWorksheet(index);
            this.SetBorders(ref sheet, cell1, cell2, top, bottom, left, right, false, solid, thick, borderColor, true);
        }
        protected void SetBorders(ref Excel.Worksheet sheet, XlCell cell1, XlCell cell2, bool top, bool bottom,
            bool left, bool right, bool interior, bool solid, bool thick, XlColor borderColor, bool releaseComObject)
        {
            Excel.XlLineStyle borderStyle = (solid) ? Excel.XlLineStyle.xlContinuous : Excel.XlLineStyle.xlDot;
            Excel.XlBorderWeight borderWeight = (solid && thick) ? Excel.XlBorderWeight.xlMedium :
                (solid || thick) ? Excel.XlBorderWeight.xlThin : Excel.XlBorderWeight.xlHairline;
            this.SetBorders(ref sheet, cell1, cell2, top, bottom, left, right, interior, borderStyle, borderWeight, borderColor,
                releaseComObject);
        }
        protected void SetBorders(ref Excel.Worksheet sheet, XlCell cell1, XlCell cell2, bool top, bool bottom, bool left,
            bool right, bool interior, Excel.XlLineStyle borderStyle, Excel.XlBorderWeight borderWeight, XlColor borderColor,
            bool releaseComObject)
        {
            Excel.Range range = this.GetRange(ref sheet, cell1, cell2, false);
            if (top)   this.SetBorder(range, Excel.XlBordersIndex.xlEdgeTop   , borderStyle, borderWeight, borderColor);
            if (bottom)this.SetBorder(range, Excel.XlBordersIndex.xlEdgeBottom, borderStyle, borderWeight, borderColor);
            if (left)  this.SetBorder(range, Excel.XlBordersIndex.xlEdgeLeft  , borderStyle, borderWeight, borderColor);
            if (right) this.SetBorder(range, Excel.XlBordersIndex.xlEdgeRight , borderStyle, borderWeight, borderColor);
            if (interior)
            {   
                if (cell1.Row != cell2.Row)
                {
                    this.SetBorder(range, Excel.XlBordersIndex.xlInsideHorizontal, borderStyle, borderWeight, borderColor);
                }
                if (cell1.Column != cell2.Column)
                {
                    this.SetBorder(range, Excel.XlBordersIndex.xlInsideVertical, borderStyle, borderWeight, borderColor);
                }
            }
            Marshal.ReleaseComObject(range);
            range = null;
            if (releaseComObject)
            {
                Marshal.ReleaseComObject(sheet);
                sheet = null;
            }
        }
        private void SetBorder(Excel.Range range, Excel.XlBordersIndex index, Excel.XlLineStyle style,
            Excel.XlBorderWeight weight, XlColor color)
        {
            Excel.Borders borders = range.Borders;
            Excel.Border  border  = borders[index];
            border.LineStyle  = style;
            border.Weight     = weight;
            border.ColorIndex = (int)color;
            Marshal.ReleaseComObject(border);
            Marshal.ReleaseComObject(borders);
            border  = null;
            borders = null;
        }
        #endregion
        #region 
        public void SetHeader(XlHeaderFooterText header)
        {
            this.SetHeaderAndFooter(header, null);
        }
        public void SetHeader(object index, XlHeaderFooterText header)
        {
            this.SetHeaderAndFooter(index, header, null);
        }
        public void SetFooter(XlHeaderFooterText footer)
        {
            this.SetHeaderAndFooter(null, footer);
        }
        public void SetFooter(object index, XlHeaderFooterText footer)
        {
            this.SetHeaderAndFooter(index, null, footer);
        }
        public void SetHeaderAndFooter(XlHeaderFooterText header, XlHeaderFooterText footer)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.Workbook.ActiveSheet;
            this.SetHeaderAndFooter(ref sheet, header, footer, true);
        }
        public void SetHeaderAndFooter(object index, XlHeaderFooterText header, XlHeaderFooterText footer)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.GetWorksheet(index);
            this.SetHeaderAndFooter(ref sheet, header, footer, true);
        }
        protected void SetHeaderAndFooter(ref Excel.Worksheet sheet, XlHeaderFooterText header,
            XlHeaderFooterText footer, bool releaseComObject)
        {
            if (this._disposed) { throw new ObjectDisposedException(base.ToString()); }
            if (header != null)
            {
                sheet.PageSetup.LeftHeader = header.Left;
                sheet.PageSetup.CenterHeader = header.Center;
                sheet.PageSetup.RightHeader = header.Right;
            }
            if (footer != null)
            {
                sheet.PageSetup.LeftFooter = footer.Left;
                sheet.PageSetup.CenterFooter = footer.Center;
                sheet.PageSetup.RightFooter = footer.Right;
            }
            if (releaseComObject)
            {
                Marshal.ReleaseComObject(sheet);
                sheet = null;
            }
        }
        public void PageSetup(bool landscape, Size fitToPage, XlMargins margins)
        {
            this.PageSetup(landscape, fitToPage, margins, -1, -1);
        }
        public void PageSetup(bool landscape, Size fitToPage, XlMargins margins, int titleRows, int titleCols)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.Workbook.ActiveSheet;
            this.PageSetup(ref sheet, landscape, fitToPage, margins, titleRows, titleCols, true);
        }
        public void PageSetup(object index, bool landscape, Size fitToPage, XlMargins margins)
        {
            this.PageSetup(index, landscape, fitToPage, margins, -1, -1);
        }
        public void PageSetup(object index, bool landscape, Size fitToPage, XlMargins margins, int titleRows,
            int titleCols)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.GetWorksheet(index);
            this.PageSetup(ref sheet, landscape, fitToPage, margins, titleRows, titleCols, true);
        }
        protected void PageSetup(ref Excel.Worksheet sheet, bool landscape, Size fitToPage, XlMargins margins, int titleRows,
            int titleCols, bool releaseComObject)
        {
            if (this._disposed) { throw new ObjectDisposedException(base.ToString()); }
            sheet.PageSetup.Orientation = (landscape) ? Excel.XlPageOrientation.xlLandscape : Excel.XlPageOrientation.xlPortrait;
            sheet.PageSetup.FitToPagesWide = (fitToPage.Width  > 0) ? (object)fitToPage.Width  : (object)false;
            sheet.PageSetup.FitToPagesTall = (fitToPage.Height > 0) ? (object)fitToPage.Height : (object)false;
            sheet.PageSetup.Zoom = (fitToPage.Width > 0 || fitToPage.Height > 0) ? (object)false : (object)100;
            if (margins != null)
            {
                sheet.PageSetup.TopMargin          = margins.PtTop;
                sheet.PageSetup.BottomMargin       = margins.PtBottom;
                sheet.PageSetup.LeftMargin         = margins.PtLeft;
                sheet.PageSetup.RightMargin        = margins.PtRight;
                sheet.PageSetup.HeaderMargin       = margins.PtHeader;
                sheet.PageSetup.FooterMargin       = margins.PtFooter;
                sheet.PageSetup.CenterHorizontally = margins.CenterHorizontally;
                sheet.PageSetup.CenterVertically   = margins.CenterVertically;
            }
            this.PrintTitleRowsCols(ref sheet, titleRows, titleCols, releaseComObject);
        }
        public void PrintTitleRowsCols(int titleRows, int titleCols)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.Workbook.ActiveSheet;
            this.PrintTitleRowsCols(ref sheet, titleRows, titleCols, true);
        }
        public void PrintTitleRowsCols(object index, int titleRows, int titleCols)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.GetWorksheet(index);
            this.PrintTitleRowsCols(ref sheet, titleRows, titleCols, true);
        }
        protected void PrintTitleRowsCols(ref Excel.Worksheet sheet, int titleRows, int titleCols,
            bool releaseComObject)
        {
            if (this._disposed) { throw new ObjectDisposedException(base.ToString()); }
            if (titleRows > 0) sheet.PageSetup.PrintTitleRows    = "$1:$" + titleRows.ToString();
            if (titleCols > 0) sheet.PageSetup.PrintTitleColumns = "$A:$" + this.ConvertExcelColumnIndex(titleCols).ToString();
            if (releaseComObject)
            {
                Marshal.ReleaseComObject(sheet);
                sheet = null;
            }
        }
        #endregion
        #region 
        public void InsertPageBreak(int row)
        {
            this.InsertPageBreaks(new int[] {row});
        }
        public void InsertPageBreak(object index, int row)
        {
            this.InsertPageBreaks(index, new int[] {row});
        }
        public void InsertPageBreaks(int[] rows)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.Workbook.ActiveSheet;
            this.InsertPageBreaks(ref sheet, rows, true);
        }
        public void InsertPageBreaks(object index, int[] rows)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.GetWorksheet(index);
            this.InsertPageBreaks(ref sheet, rows, true);
        }
        protected void InsertPageBreaks(ref Excel.Worksheet sheet, int[] rows, bool releaseComObject)
        {
            if (rows == null || rows.Length == 0)
            {
                throw new ArgumentNullException("rows", "The page break rows cannot be null or have zero (0) length.");
            }
            for (int i = 0; i < rows.Length; i++)
            {
                object cell = sheet.Cells[rows[i], 1];
                sheet.HPageBreaks.Add(cell);
                Marshal.ReleaseComObject(cell);
                cell = null;
            }
            if (releaseComObject)
            {
                Marshal.ReleaseComObject(sheet);
                sheet = null;
            }
        }
        public void FreezePanes(XlCell cell)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.Workbook.ActiveSheet;
            this.FreezePanes(ref sheet, cell, true);
        }
        public void FreezePanes(object index, XlCell cell)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.GetWorksheet(index);
            this.FreezePanes(ref sheet, cell, true);
        }
        protected void FreezePanes(ref Excel.Worksheet sheet, XlCell cell, bool releaseComObject)
        {
            if (this._disposed) { throw new ObjectDisposedException(base.ToString()); }
            if (cell.IsValid)
            {
                Excel.Range range = (Excel.Range)sheet.Cells[cell.Row, cell.Column];
                range.Select();
                XlUtilities.ExcelApp.ActiveWindow.FreezePanes = true;
                Marshal.ReleaseComObject(range);
                range = null;
            }
            if (releaseComObject)
            {
                Marshal.ReleaseComObject(sheet);
                sheet = null;
            }
        }
        public void UnFreezePanes()
        {
            if (this.Workbook != null)
            {
                XlUtilities.ExcelApp.ActiveWindow.FreezePanes = false;
            }
        }
        #endregion
        #region 
        public void AddExcelName(string name, string expression)
        {
            Excel.Names xlNames = this.Workbook.Names;
            Excel.Name  xlName  = xlNames.Add(name, expression, true, NIL, NIL, NIL, NIL, NIL, NIL, NIL, NIL);
            Marshal.ReleaseComObject(xlName);
            Marshal.ReleaseComObject(xlNames);
            xlName  = null;
            xlNames = null;
        }
        public void RemoveExcelName(string name)
        {
            Excel.Names xlNames = this.Workbook.Names;
            Excel.Name  xlName  = xlNames.Item(name, NIL, NIL);
            xlName.Delete();
            Marshal.ReleaseComObject(xlName);
            Marshal.ReleaseComObject(xlNames);
            xlName  = null;
            xlNames = null;
        }
        #endregion
        #region 
        public void SetWorksheetName(string name)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.Workbook.ActiveSheet;
            this.SetWorksheetName(ref sheet, name, true);
        }
        public void SetWorksheetName(object index, string name)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.GetWorksheet(index);
            this.SetWorksheetName(ref sheet, name, true);
        }
        protected void SetWorksheetName(ref Excel.Worksheet sheet, string name, bool releaseComObject)
        {
            name = name.Trim();
            if (name == null || name == "")
            {
                throw new ArgumentNullException("name", "The worksheet name cannot be null or empty.");
            }
            this.SetWorksheetName_Helper(sheet, name, 1);
            if (releaseComObject)
            {   
                Marshal.ReleaseComObject(sheet);
                sheet = null;
            }
        }
        private void SetWorksheetName_Helper(Excel.Worksheet sheet, string name, int retries)
        {
            if (this._disposed) { throw new ObjectDisposedException(base.ToString()); }
            try
            {
                sheet.Name = name;
            }
            catch (Exception ex)
            {   
                if (retries <= 0)
                {   
                    throw new ArgumentException("The worksheet name cannot be set: " + ex.Message, "name", ex);
                }
                name += ("_" + DateTime.Now.Hour + "." + DateTime.Now.Minute + "." + DateTime.Now.Second);
                this.SetWorksheetName_Helper(sheet, name, retries - 1);
            }
        }
        public void AddWorksheet(string name)
        {
            Excel.Sheets sheets = this.Workbook.Worksheets;
            this.AddWorksheet(sheets.Count + 1, name);
            Marshal.ReleaseComObject(sheets);
            sheets = null;
        }
        public void AddWorksheet(int index, string name)
        {
            Excel.Sheets sheets = this.Workbook.Worksheets;
            object before = NIL;
            object after  = NIL;
            if (index <= 1)
            {
                before = sheets[1];
                index = 1;
            }
            else if (index > sheets.Count)
            {
                after = sheets[sheets.Count];
                index = sheets.Count + 1;
            }
            else
            {
                before = sheets[index];
            }
            sheets.Add(before, after, NIL, NIL);
            this.SetWorksheetName(index, name);
            if      (after  != NIL) { Marshal.ReleaseComObject(after); }
            else if (before != NIL) { Marshal.ReleaseComObject(before); }
            Marshal.ReleaseComObject(sheets);
            after = null;
            before = null;
            sheets = null;
        }
        public void DeleteWorksheet(object index)
        {
            Excel.Worksheet sheet = this.GetWorksheet(index);
            sheet.Delete();
            Marshal.ReleaseComObject(sheet);
            sheet = null;
        }
        public void DeleteAllWorksheets()
        {
            Excel.Sheets sheets = this.Workbook.Sheets;
            this.AddWorksheet(1, "__XLU_TEMP_SHEET__");
            for (int i = sheets.Count; i > 1; i--)
            {
                this.DeleteWorksheet(i);
            }
            Marshal.ReleaseComObject(sheets);
            sheets = null;
        }
        public void ActivateWorksheet(object index)
        {
            Excel.Worksheet sheet = this.GetWorksheet(index);
            sheet.Activate();
            Marshal.ReleaseComObject(sheet);
            sheet = null;
        }
        public void CopyWorksheet(XlUtilities source, object index)
        {
            this.CopyWorksheet(source, index, 1);
        }
        public void CopyWorksheet(XlUtilities source, object sourceIndex, int destinationIndex)
        {
            Excel.Sheets sheets = this.Workbook.Worksheets;
            object before = NIL;
            object after  = NIL;
            if      (destinationIndex <= 1)           { before = sheets[1]; }
            else if (destinationIndex > sheets.Count) { after = sheets[sheets.Count]; }
            else                                      { before = sheets[destinationIndex]; }
            Excel.Worksheet sourceSheet = source.GetWorksheet(sourceIndex);
            sourceSheet.Copy(before, after);
            Marshal.ReleaseComObject(sourceSheet);
            if      (before != NIL) { Marshal.ReleaseComObject(before); }
            else if (after  != NIL) { Marshal.ReleaseComObject(after); }
            Marshal.ReleaseComObject(sheets);
            sourceSheet = null;
            after  = null;
            before = null;
            sheets = null;
        }
        #endregion
        #region 
        public void DrawChart(XlChartFormat format)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.Workbook.ActiveSheet;
            this.DrawChart(ref sheet, format, true);
        }
        public void DrawChart(object index, XlChartFormat format)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.GetWorksheet(index);
            this.DrawChart(ref sheet, format, true);
        }
        protected void DrawChart(ref Excel.Worksheet sheet, XlChartFormat format, bool releaseComObject)
        {
            if (!format.IsValid)
            {
                throw new ArgumentException("The chart format provided does not contain valid formatting information.", "format");
            }
            this.DrawChart(ref sheet, format.Type, format.Name, format.Title, format.PlotColor, format.SeriesFormat,
                format.CategoryAxisFormat, format.ValuesAxisFormat, releaseComObject);
        }
        public void DrawChart(XlChartType type, string name, string title, XlColor plotColor, XlSeriesFormat[] series,
            XlAxisFormat categoryAxisFormat, XlAxisFormat valuesAxisFormat)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.Workbook.ActiveSheet;
            this.DrawChart(ref sheet, type, name, title, plotColor, series, categoryAxisFormat, valuesAxisFormat,true);
        }
        public void DrawChart(object index, XlChartType type, string name, string title, XlColor plotColor,
            XlSeriesFormat[] series, XlAxisFormat categoryAxisFormat, XlAxisFormat valuesAxisFormat)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.GetWorksheet(index);
            this.DrawChart(ref sheet, type, name, title, plotColor, series, categoryAxisFormat, valuesAxisFormat, true);
        }
        protected void DrawChart(ref Excel.Worksheet sheet, XlChartType type, string name, string title,
            XlColor plotColor, XlSeriesFormat[] series, XlAxisFormat categoryAxisFormat, XlAxisFormat valuesAxisFormat,
            bool releaseComObject)
        {
            Excel.Chart chart = this.AddChart(ref sheet);
            chart.ChartType = (Excel.XlChartType)type;
            Excel.SeriesCollection chartSeries = (Excel.SeriesCollection)chart.SeriesCollection(NIL);
            for (int i = 0, count = chartSeries.Count; i < count; i++)
            {
                Excel.Series s = (Excel.Series)chartSeries.Item(i + 1);
                s.Delete();
                Marshal.ReleaseComObject(s);
                s = null;
            }
            string sheetName = sheet.Name;
            for (int i = 0, count = series.Length; i < count; i++)
            {
                this.DrawSeriesOnChart(ref chart, series[i], ref sheet);
            }
            chart.Location(Excel.XlChartLocation.xlLocationAsNewSheet, name);
            chart.PlotArea.Interior.ColorIndex = (plotColor == XlColor.Automatic) ? 2 : (int)plotColor;
            chart.HasTitle = (title != null && title.Trim() != "");
            if (chart.HasTitle) { chart.ChartTitle.Text = title.Trim(); }
            if (XlChartTypeInfo.Is3D(type))
            {   
                chart.Elevation = 40;
            }
            Excel.Axis xAxis = (Excel.Axis)chart.Axes(Excel.XlAxisType.xlCategory, Excel.XlAxisGroup.xlPrimary);
            Excel.Axis yAxis = (Excel.Axis)chart.Axes(Excel.XlAxisType.xlValue   , Excel.XlAxisGroup.xlPrimary);
            this.DrawChart_Helper_FormatAxis(xAxis, categoryAxisFormat, type);
            this.DrawChart_Helper_FormatAxis(yAxis, valuesAxisFormat  , type);
            Marshal.ReleaseComObject(xAxis);
            Marshal.ReleaseComObject(yAxis);
            Marshal.ReleaseComObject(chartSeries);
            Marshal.ReleaseComObject(chart);
            xAxis = null;
            yAxis = null;
            chartSeries = null;
            chart = null;
            if (releaseComObject)
            {
                Marshal.ReleaseComObject(sheet);
                sheet = null;
            }
        }
        private void DrawChart_Helper_FormatAxis(Excel.Axis axis, XlAxisFormat format, XlChartType type)
        {
            if (format == null) { return; }
            axis.Border.LineStyle = (format.SolidBorder) ? Excel.XlLineStyle.xlContinuous : Excel.XlLineStyle.xlDot;
            axis.Border.Weight      = (format.ThickBorder) ? Excel.XlBorderWeight.xlThick    : Excel.XlBorderWeight.xlThin;
            if (double.IsNaN(format.CrossesAt))
            {
                axis.Crosses = Excel.XlAxisCrosses.xlAxisCrossesAutomatic;
            }
            else
            {
                axis.Crosses = Excel.XlAxisCrosses.xlAxisCrossesCustom;
                axis.CrossesAt = format.CrossesAt;
            }
            XlGridlinesFormat gridlines = format.GridlinesFormat;
            if (gridlines != null)
            {   
                axis.HasMajorGridlines = gridlines.HasMajorGridlines;
                axis.HasMinorGridlines = gridlines.HasMinorGridlines;
                if (gridlines.HasMajorGridlines)
                {
                    axis.MajorGridlines.Border.ColorIndex = (int)gridlines.MajorGridlinesColor;
                    axis.MajorGridlines.Border.LineStyle  = (gridlines.MajorGridlinesSolidBorder) ?
                        Excel.XlLineStyle.xlContinuous : Excel.XlLineStyle.xlDot;
                    axis.MajorGridlines.Border.Weight     = (gridlines.MajorGridlinesThickBorder) ?
                        Excel.XlBorderWeight.xlThick : Excel.XlBorderWeight.xlThin;
                }
                if (gridlines.HasMinorGridlines)
                {
                    axis.MinorGridlines.Border.ColorIndex = (int)gridlines.MinorGridlinesColor;
                    axis.MajorGridlines.Border.LineStyle  = (gridlines.MinorGridlinesSolidBorder) ?
                        Excel.XlLineStyle.xlContinuous : Excel.XlLineStyle.xlDot;
                    axis.MajorGridlines.Border.Weight     = (gridlines.MinorGridlinesThickBorder) ?
                        Excel.XlBorderWeight.xlThick : Excel.XlBorderWeight.xlThin;
                }
            }
            if (axis.Type == Excel.XlAxisType.xlValue || XlChartTypeInfo.IsXYScatter(type))
            {   
                axis.MajorUnitIsAuto = double.IsNaN(format.MajorUnit);
                if (!axis.MajorUnitIsAuto) { axis.MajorUnit = format.MajorUnit; }
                axis.MinorUnitIsAuto = double.IsNaN(format.MinorUnit);
                if (!axis.MinorUnitIsAuto) { axis.MinorUnit = format.MinorUnit; }
                axis.MaximumScaleIsAuto = double.IsNaN(format.MaximumScale);
                if (!axis.MaximumScaleIsAuto) { axis.MaximumScale = format.MaximumScale; }
                axis.MinimumScaleIsAuto = double.IsNaN(format.MinimumScale);
                if (!axis.MinimumScaleIsAuto) { axis.MinimumScale = format.MinimumScale; }
                axis.ScaleType = (format.LogorithmicScale) ? Excel.XlScaleType.xlScaleLogarithmic :
                    Excel.XlScaleType.xlScaleLinear;
            }
            axis.ReversePlotOrder = format.ReversePlotOrder;
            if (format.HasTitle)
            {
                XlAxisTitleFormat titleFormat = format.AxisTitle;
                axis.HasTitle = true;
                axis.AxisTitle.Caption = titleFormat.Caption;
                axis.AxisTitle.Shadow  = titleFormat.Shadow;
                if (!double.IsNaN(titleFormat.Left)) { axis.AxisTitle.Left = titleFormat.Left; }
                if (!double.IsNaN(titleFormat.Top))  { axis.AxisTitle.Top  = titleFormat.Top; }
                axis.AxisTitle.Font.Name          = titleFormat.Font.Name;
                axis.AxisTitle.Font.Size          = titleFormat.Font.Size;
                axis.AxisTitle.Font.Italic        = titleFormat.Font.Italic;
                axis.AxisTitle.Font.Bold          = titleFormat.Font.Bold;
                axis.AxisTitle.Font.Underline     = titleFormat.Font.Underline;
                axis.AxisTitle.Font.Strikethrough = titleFormat.Font.Strikethrough;
                axis.AxisTitle.Font.Subscript     = titleFormat.Font.Subscript;
                axis.AxisTitle.Font.Superscript   = titleFormat.Font.Superscript;
                axis.AxisTitle.Font.ColorIndex    = (int)titleFormat.Font.Color;
                XlTextAlignment textAlign = titleFormat.TextAlignment;
                if (textAlign != null)
                {
                    Excel.XlHAlign hAlign =
                        (textAlign.Horizontal == XlHAlign.Left)  ? Excel.XlHAlign.xlHAlignLeft  :
                        (textAlign.Horizontal == XlHAlign.Right) ? Excel.XlHAlign.xlHAlignRight :
                        Excel.XlHAlign.xlHAlignCenter;
                    Excel.XlVAlign vAlign =
                        (textAlign.Vertical == XlVAlign.Top)    ? Excel.XlVAlign.xlVAlignTop    :
                        (textAlign.Vertical == XlVAlign.Bottom) ? Excel.XlVAlign.xlVAlignBottom :
                        Excel.XlVAlign.xlVAlignCenter;
                    axis.AxisTitle.HorizontalAlignment = hAlign;
                    axis.AxisTitle.VerticalAlignment   = vAlign;
                }
                axis.AxisTitle.Orientation = titleFormat.TextOrientation;
            }
            XlTickLabelsFormat tickFormat = format.TickLabels;
            if (tickFormat != null)
            {   
                axis.TickLabels.Font.Name          = tickFormat.Font.Name;
                axis.TickLabels.Font.Size          = tickFormat.Font.Size;
                axis.TickLabels.Font.Italic        = tickFormat.Font.Italic;
                axis.TickLabels.Font.Bold          = tickFormat.Font.Bold;
                axis.TickLabels.Font.Underline     = tickFormat.Font.Underline;
                axis.TickLabels.Font.Strikethrough = tickFormat.Font.Strikethrough;
                axis.TickLabels.Font.Subscript     = tickFormat.Font.Subscript;
                axis.TickLabels.Font.Superscript   = tickFormat.Font.Superscript;
                axis.TickLabels.Font.ColorIndex    = (int)tickFormat.Font.Color;
                axis.TickLabels.NumberFormat       = tickFormat.NumberFormat;
                axis.TickLabels.NumberFormatLinked = tickFormat.NumberFormatLinked;
                axis.TickLabels.Offset             = tickFormat.Offset;
                axis.TickLabels.Orientation        = (Excel.XlTickLabelOrientation)tickFormat.TextOrientation;
            }
        }
        protected void DrawSeriesOnChart(ref Excel.Chart chart, XlSeriesFormat series, ref Excel.Worksheet sheet)
        {
            if (!series.IsValid) { return; }
            Excel.SeriesCollection chartSeries = (Excel.SeriesCollection)chart.SeriesCollection(NIL);
            Excel.Series s = chartSeries.NewSeries();
            s.Name = (series.Name == null) ? null : "=\"" + series.Name +"\"";
            bool hasVals = false;
            bool hasXVals = false;
            if (series.UseCustomValues)
            {   
                s.XValues = series.CustomXValues;
                s.Values  = series.CustomValues;
                hasVals = hasXVals = true;
            }
            else
            {   
                object[,] vals = this.GetData(ref sheet, series.Values.Cell1, series.Values.Cell2, true, false);
                object[,] xVals = this.GetData(ref sheet, series.XValues.Cell1, series.XValues.Cell2, true, false);
                for (int i = 1, count = vals.GetUpperBound(1) + 1; i < count; i++)
                {
                    if (Information.IsNumeric(vals[1, i]))
                    {
                        hasVals = true;
                        break;
                    }
                }
                if (hasVals)
                {   
                    for (int i = 1, count = xVals.GetUpperBound(1) + 1; i < count; i++)
                    {
                        if (Information.IsNumeric(xVals[1, i]))
                        {
                            hasXVals = true;
                            break;
                        }
                    }
                }
                if (hasVals && hasXVals)
                {
                    string sheetName = sheet.Name;
                    int xR1 = series.XValues.Cell1.Row, xC1 = series.XValues.Cell1.Column;
                    int xR2 = series.XValues.Cell2.Row, xC2 = series.XValues.Cell2.Column;
                    s.XValues = "='" + sheetName + "'!R" + xR1 + "C" + xC1 + ":R" + xR2 + "C" + xC2;
                    int vR1 = series.Values.Cell1.Row, vC1 = series.Values.Cell1.Column;
                    int vR2 = series.Values.Cell2.Row, vC2 = series.Values.Cell2.Column;
                    s.Values  = "='" + sheetName + "'!R" + vR1 + "C" + vC1 + ":R" + vR2 + "C" + vC2;
                }
            }
            if (hasVals && hasXVals)
            {   
                s.Border.ColorIndex = (int)series.Color;
                s.Border.LineStyle  = (series.SolidLine) ? Excel.XlLineStyle.xlContinuous : Excel.XlLineStyle.xlDot;
                s.Border.Weight     = (series.ThickLine) ? Excel.XlBorderWeight.xlThick   : Excel.XlBorderWeight.xlThin;
                s.Interior.PatternColorIndex = (series.PatternColor == XlColor.Automatic) ?
                    (int)Excel.XlColorIndex.xlColorIndexAutomatic : (int)series.PatternColor;
                s.Interior.InvertIfNegative = series.InvertIfNegative;
                s.Shadow = series.Shadow;
                s.HasDataLabels = series.HasDataLabels;
                XlDataLabelsFormat labelsFormat = series.DataLabels;
                if (series.HasDataLabels && series.DataLabels != null)
                {
                    Excel.DataLabels dl = (Excel.DataLabels)s.DataLabels(NIL);
                    dl.NumberFormat       = labelsFormat.NumberFormat;
                    dl.NumberFormatLinked = labelsFormat.NumberFormatLinked;
                    XlTextAlignment textAlign = labelsFormat.TextAlignment;
                    if (textAlign != null)
                    {
                        Excel.XlHAlign hAlign =
                            (textAlign.Horizontal == XlHAlign.Left)  ? Excel.XlHAlign.xlHAlignLeft  :
                            (textAlign.Horizontal == XlHAlign.Right) ? Excel.XlHAlign.xlHAlignRight :
                            Excel.XlHAlign.xlHAlignCenter;
                        Excel.XlVAlign vAlign =
                            (textAlign.Vertical == XlVAlign.Top)    ? Excel.XlVAlign.xlVAlignTop    :
                            (textAlign.Vertical == XlVAlign.Bottom) ? Excel.XlVAlign.xlVAlignBottom :
                            Excel.XlVAlign.xlVAlignCenter;
                        dl.HorizontalAlignment = hAlign;
                        dl.VerticalAlignment   = vAlign;
                    }
                    dl.Orientation      = labelsFormat.TextOrientation;
                    dl.Shadow           = labelsFormat.Shadow;
                    dl.ShowCategoryName = labelsFormat.ShowCategoryName;
                    dl.ShowLegendKey    = labelsFormat.ShowLegendKey;
                    dl.ShowPercentage   = labelsFormat.ShowPercentage;
                    dl.ShowSeriesName   = labelsFormat.ShowSeriesName;
                    dl.ShowValue        = labelsFormat.ShowValue;
                }
                XlChartType type = (XlChartType)chart.Type;
                if (XlChartTypeInfo.IsLine(type) || XlChartTypeInfo.IsXYScatter(type) || XlChartTypeInfo.IsRadar(type))
                {
                    if (series.MarkerSize == -1)
                    {
                        s.MarkerStyle = Excel.XlMarkerStyle.xlMarkerStyleNone;
                    }
                    else
                    {
                        s.MarkerStyle = Excel.XlMarkerStyle.xlMarkerStyleAutomatic;
                        s.MarkerForegroundColorIndex = (series.MarkerForeColor == XlColor.Automatic) ?
                            Excel.XlColorIndex.xlColorIndexAutomatic : (Excel.XlColorIndex)series.MarkerForeColor;
                        s.MarkerBackgroundColorIndex = (series.MarkerBackColor == XlColor.Automatic) ?
                            Excel.XlColorIndex.xlColorIndexAutomatic : (Excel.XlColorIndex)series.MarkerBackColor;
                        s.MarkerSize = series.MarkerSize;
                    }
                }
            } 
            Marshal.ReleaseComObject(s);
            Marshal.ReleaseComObject(chartSeries);
            s = null;
            chartSeries = null;
        }
        protected Excel.Chart AddChart(ref Excel.Worksheet sheet)
        {
            Excel.Sheets charts = this.Workbook.Charts;
            Excel.Chart  chart  = (Excel.Chart)charts.Add(NIL, sheet, NIL, NIL);
            Marshal.ReleaseComObject(charts);
            charts = null;
            return chart;
        }
        #endregion
        public string[] GetWorksheetNames()
        {
            Excel.Sheets sheets = this._workbook.Worksheets;
            int numSheets = sheets.Count;
            string[] names = new string[numSheets];
            for (int i = 0; i < numSheets; i++)
            {
                names[i] = ((Excel.Worksheet)sheets[i + 1]).Name;
            }
            Marshal.ReleaseComObject(sheets);
            sheets = null;
            return names;
        }
        public XlCell GetLastCell()
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.Workbook.ActiveSheet;
            return this.GetLastCell(ref sheet, true);
        }
        public XlCell GetLastCell(object index)
        {
            Excel.Worksheet sheet = this.GetWorksheet(index);
            return this.GetLastCell(ref sheet, true);
        }
        protected XlCell GetLastCell(ref Excel.Worksheet sheet, bool releaseComObject)
        {
            Excel.Range range = sheet.Cells.SpecialCells(Excel.XlCellType.xlCellTypeLastCell, NIL);
            XlCell cell = new XlCell(range.Row, range.Column);
            Marshal.ReleaseComObject(range);
            range = null;
            if (releaseComObject)
            {
                Marshal.ReleaseComObject(sheet);
                sheet = null;
            }
            return cell;
        }
        protected Excel.Sheets GetWorksheets()
        {
            return this.Workbook.Worksheets;
        }
        protected Excel.Worksheet GetWorksheet(object index)
        {
            Excel.Sheets sheets = this.Workbook.Worksheets;
            Excel.Worksheet sheet = (Excel.Worksheet)sheets[index];
            Marshal.ReleaseComObject(sheets);
            sheets = null;
            return sheet;
        }
        protected Excel.Range GetRange(XlCell cell1, XlCell cell2)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.Workbook.ActiveSheet;
            return this.GetRange(ref sheet, cell1, cell2, true);
        }
        protected Excel.Range GetRange(object index, XlCell cell1, XlCell cell2)
        {
            Excel.Worksheet sheet = (Excel.Worksheet)this.GetWorksheet(index);
            return this.GetRange(ref sheet, cell1, cell2, true);
        }
        protected Excel.Range GetRange(ref Excel.Worksheet sheet, XlCell cell1, XlCell cell2, bool releaseComObject)
        {
            if (this._disposed) { throw new ObjectDisposedException(base.ToString()); }
            bool isEmpty = (cell1.IsEmpty && cell2.IsEmpty);
            if (!isEmpty)
            {   
                if      (!cell1.IsValid) { throw new ArgumentOutOfRangeException("cell1",
                                               "The cells specified must represent a valid (Row, Column) location."); }
                else if (!cell2.IsValid) { throw new ArgumentOutOfRangeException("cell2",
                                               "The cells specified must represent a valid (Row, Column) location."); }
            }
            Excel.Range range;
            if (isEmpty)
            {   
                range = sheet.Cells;
            }
            else
            {   
                object c1 = sheet.Cells[cell1.Row, cell1.Column];
                object c2 = sheet.Cells[cell2.Row, cell2.Column];
                range = sheet.get_Range(c1, c2);
                Marshal.ReleaseComObject(c1);
                Marshal.ReleaseComObject(c2);
                c1 = null;
                c2 = null;
            }
            if (releaseComObject)
            {
                Marshal.ReleaseComObject(sheet);
                sheet = null;
            }
            return range;
        }
        public object ConvertExcelColumnIndex(object index)
        {
            double d;
            if (double.TryParse(index.ToString(), System.Globalization.NumberStyles.Any, null, out d))
            {
                int i = (int)d;
                index = (i < 27) ? ((char)(64 + i)).ToString() :
                    (i % 26 == 0) ? ((char)(63 + i / 26)).ToString() + "Z" :
                    ((char)(64 + i / 26)).ToString() + ((char)(64 + i % 26)).ToString();
            }
            return index;
        }
        #endregion
        #region 
        public string Path
        {
            get
            {
                if (this._disposed) { throw new ObjectDisposedException(base.ToString()); }
                return this._path;
            }
        }
        public Excel.Workbook Workbook
        {
            get
            {
                if (this._disposed) { throw new ObjectDisposedException(base.ToString()); }
                this._workbook.Activate();  
                return this._workbook;
            }
        }
        public Excel.Worksheet ActiveSheet { get { return (Excel.Worksheet)this.Workbook.ActiveSheet; } }
        public string WorkbookTitle    { get { return this.Workbook.Title;    } }
        public string WorkbookSubject  { get { return this.Workbook.Subject;  } }
        public string WorkbookAuthor   { get { return this.Workbook.Author;   } }
        public string WorkbookComments { get { return this.Workbook.Comments; } }
        public int WorksheetCount
        {
            get
            {
                Excel.Sheets sheets = this.Workbook.Worksheets;
                int numSheets = sheets.Count;
                Marshal.ReleaseComObject(sheets);
                sheets = null;
                return numSheets;
            }
        }
        public bool ReadOnly
        {
            get
            {
                if (this._disposed) { throw new ObjectDisposedException(base.ToString()); }
                return this._readOnly;
            }
        }
        public bool PasswordProtected
        {
            get
            {
                if (this._disposed) { throw new ObjectDisposedException(base.ToString()); }
                return this._passwordProtected;
            }
        }
        public bool Opened { get { return this._opened; } }
        public bool Disposed { get { return this._disposed; } }
        #endregion
        #region 
        private const int xlNone = (int)Excel.Constants.xlNone;
        private static Excel.Application ExcelApp;
        private static readonly object NIL = Missing.Value;
        private string _path;
        private Excel.Workbook _workbook;
        private bool _readOnly;
        private bool _passwordProtected;
        private bool _opened;
        private bool _disposed;
        #endregion
    }
}

