using System;
using System.Drawing;
using Volpe.Cafe.Data;
using Volpe.Cafe.IO;
using Volpe.Utils;
namespace Volpe.Cafe.Reporter.ReportTypes
{
    public class EffectsReport : Output
    {
        #region 
        public EffectsReport(string path, EncryptionSettings cryptoSettings)
            : base(path, TemplateType.EffectsReport, cryptoSettings, true)
        {
            base._autoBorder = true;
            base._autoChart = true;
            base._autoMerge = true;
            base._autoSummarize = true;
            base._fillColor = XlColor.Automatic;
            this._fitToPage = new Size(1, -1);
            base._font = new XlFont("Arial");
            base._headerAlignment = new XlTextAlignment(XlHAlign.Center, XlVAlign.Center);
            base._headerFillColor = XlColor.Automatic;
            base._headerFont = new XlFont("Arial", 10, false, true, false);
            base._landscape = true;
            base._margins = new XlMargins(0.5);
            base._numberFormat = "#,##0";
            base._repeatHeaders = true;
            base._textAlignment = new XlTextAlignment(XlHAlign.General, XlVAlign.Center);
            this._initialized = false;
            this._minYear = 0;
            this._maxYear = 0;
            this._numSects = 0;
        }
        #endregion
        #region 
        public void ParseData(int scenarioIndex, ModelYear year, Industry data, Industry baselineData)
        {
            if (!this._initialized)
            {
                this._minYear = data.MinYear.Year;
                this._maxYear = data.MaxYear.Year;
                this._numSects = this._maxYear - this._minYear + 1;
                this.SetupBuffer("Sn " + scenarioIndex + ", MY " + this._minYear.ToString() + " - " +
                    this._maxYear.ToString(), "Effects Summary");
                this.SetupHeaders(data);
                this._initialized = true;
            }
            this.WriteData(year, data, baselineData);
        }
        private void SetupBuffer(string name, string title)
        {
            int rows = 130;
            int cols = 2 + 3 * this._numSects;
            this.AddBuffer(rows, cols);
            this.Buffer.FreezePanes = new XlCell(2, 2);
            this.Buffer.Header = new XlCell(2, 2);
            this.Buffer.Name = name;
            this.Buffer.Title = title + " -- &[Tab]";
        }
        private void SetupHeaders(Industry data)
        {
            string[] headers;
            string[] subHeaders;
            headers = new string[] {"Lifetime Fuel Consumption (k gal.)", "Sales", "Lifetime VMT (k mi.)",
                "Average Fuel Economy (mpg)"};
            subHeaders = new string[] {"Gas", "Diesel", "Total"};
            this.SetupSubHeaders(2, "Energy Consumption", headers, subHeaders);
            headers = new string[] {"CO2 (mmT)", "CO (tons)", "VOC (tons)", "NOX (tons)", "PM (tons)", "SOX (tons)",
                "Sales", "Lifetime VMT (k mi.)", "Average Fuel Economy (mpg)"};
            subHeaders = new string[] {"LDGV", "LDDV", "LDGT1", "LDDT1", "LDGT2", "LDDT2", "HDGV2b", "HDDV2b", "Total"};
            this.SetupSubHeaders(16, "Emissions", headers, subHeaders);
            headers = new string[] {"Total Lifetime Pretax Fuel Expenditures", "Fuel Tax Revenues", "Travel Value",
                "Refueling Time Value", "Petroleum Market Externalities", "Congestion Costs", "Noise Costs",
                "Accident Costs", "CO2", "CO", "VOC", "NOX", "PM", "SOX"};
            int hCount = headers.Length;
            this.Buffer[99, 0] = "Undiscounted Owner and Societal Costs (k $)";
            this.Buffer[115, 0] = "Discounted Owner and Societal Costs (k $)";
            for (int i = 0; i < hCount; i++)
            {
                this.Buffer[100 + i, 0] = headers[i];
                this.Buffer[116 + i, 0] = headers[i];
            }
        }
        private void SetupSubHeaders(int row, string title, string[] headers, string[] subHeaders)
        {
            int hCount = headers.Length;
            int shCount = subHeaders.Length;
            this.Buffer[row, 0] = title;
            for (int i = 0; i < hCount; i++)
            {
                this.Buffer[row + 1 + shCount * i, 0] = headers[i];
                for (int j = 0; j < shCount; j++)
                {
                    this.Buffer[row + 1 + shCount * i + j, 1] = subHeaders[j];
                }
            }
        }
        private void WriteData(ModelYear year, Industry data, Industry baselineData)
        {
            this.Buffer[0, 0] = "Model Year";
            int rowCount = this.Buffer.Size.Rows;
            int column = 2 + 3 * (year.Year - this._minYear);
            int row = 3;
            this.Buffer[0, column] = year.ToString() + " Total";
            this.Buffer[1, column] = "Current\nScenario";
            this.Buffer[1, column + 1] = "Delta\n(abs.)";
            this.Buffer[1, column + 2] = "Delta\n(%)";
            AggregateEffectsData ied = data.ModelingData.EffectsData;
            this.WriteValueAggr(ref row, column, this.SumArray(ied.Gallons));
            this.WriteValueAggr(ref row, column, ied.Sales);
            this.WriteValueAggr(ref row, column, this.SumArray(ied.Vmt));
            this.WriteValue(ref row, column, ied.FuelEconomy, true, true, ied.Sales);
            row += 2;
            this.WriteValue(ref row, column, this.SumArray(ied.CO2Emissions));
            this.WriteValue(ref row, column, this.SumArray(ied.COEmissions));
            this.WriteValue(ref row, column, this.SumArray(ied.VOCEmissions));
            this.WriteValue(ref row, column, this.SumArray(ied.NOXEmissions));
            this.WriteValue(ref row, column, this.SumArray(ied.PMEmissions));
            this.WriteValue(ref row, column, this.SumArray(ied.SOXEmissions));
            this.WriteValue(ref row, column, ied.Sales);
            this.WriteValue(ref row, column, this.SumArray(ied.Vmt));
            this.WriteValue(ref row, column, ied.FuelEconomy, false, true, ied.Sales);
            row += 2;
            this.Buffer[row++, column] = this.SumArray(ied.PreTaxFuel).Total;
            this.Buffer[row++, column] = this.SumArray(ied.FuelTax).Total;
            this.Buffer[row++, column] = this.SumArray(ied.DriveSurplus).Total;
            this.Buffer[row++, column] = this.SumArray(ied.RefuelSurplus).Total;
            this.Buffer[row++, column] = this.SumArray(ied.EconomicCosts).Total;
            this.Buffer[row++, column] = this.SumArray(ied.CongestionCosts).Total;
            this.Buffer[row++, column] = this.SumArray(ied.NoiseCosts).Total;
            this.Buffer[row++, column] = this.SumArray(ied.AccidentCosts).Total;
            this.Buffer[row++, column] = this.SumArray(ied.CO2Damage).Total;
            this.Buffer[row++, column] = this.SumArray(ied.CODamage).Total;
            this.Buffer[row++, column] = this.SumArray(ied.VOCDamage).Total;
            this.Buffer[row++, column] = this.SumArray(ied.NOXDamage).Total;
            this.Buffer[row++, column] = this.SumArray(ied.PMDamage).Total;
            this.Buffer[row++, column] = this.SumArray(ied.SOXDamage).Total;
            row += 2;
            this.Buffer[row++, column] = this.SumArray(ied.DiscPreTaxFuel).Total;
            this.Buffer[row++, column] = this.SumArray(ied.DiscFuelTax).Total;
            this.Buffer[row++, column] = this.SumArray(ied.DiscDriveSurplus).Total;
            this.Buffer[row++, column] = this.SumArray(ied.DiscRefuelSurplus).Total;
            this.Buffer[row++, column] = this.SumArray(ied.DiscEconomicCosts).Total;
            this.Buffer[row++, column] = this.SumArray(ied.DiscCongestionCosts).Total;
            this.Buffer[row++, column] = this.SumArray(ied.DiscNoiseCosts).Total;
            this.Buffer[row++, column] = this.SumArray(ied.DiscAccidentCosts).Total;
            this.Buffer[row++, column] = this.SumArray(ied.CO2DiscDamage).Total;
            this.Buffer[row++, column] = this.SumArray(ied.CODiscDamage).Total;
            this.Buffer[row++, column] = this.SumArray(ied.VOCDiscDamage).Total;
            this.Buffer[row++, column] = this.SumArray(ied.NOXDiscDamage).Total;
            this.Buffer[row++, column] = this.SumArray(ied.PMDiscDamage).Total;
            this.Buffer[row++, column] = this.SumArray(ied.SOXDiscDamage).Total;
            if (baselineData != null)
            {
                AggregateEffectsData baseIed = baselineData.ModelingData.EffectsData;
                row = 3;    
                column++;   
                this.WriteValueAggr(ref row, column, this.SumArray(ied.Gallons), this.SumArray(baseIed.Gallons));
                this.WriteValueAggr(ref row, column, ied.Sales, baseIed.Sales);
                this.WriteValueAggr(ref row, column, this.SumArray(ied.Vmt), this.SumArray(baseIed.Vmt));
                this.WriteValue(ref row, column, ied.FuelEconomy, baseIed.FuelEconomy, true, true, ied.Sales, baseIed.Sales);
                row += 2;
                this.WriteValue(ref row, column, this.SumArray(ied.CO2Emissions), this.SumArray(baseIed.CO2Emissions));
                this.WriteValue(ref row, column, this.SumArray(ied.COEmissions),  this.SumArray(baseIed.COEmissions));
                this.WriteValue(ref row, column, this.SumArray(ied.VOCEmissions), this.SumArray(baseIed.VOCEmissions));
                this.WriteValue(ref row, column, this.SumArray(ied.NOXEmissions), this.SumArray(baseIed.NOXEmissions));
                this.WriteValue(ref row, column, this.SumArray(ied.PMEmissions),  this.SumArray(baseIed.PMEmissions));
                this.WriteValue(ref row, column, this.SumArray(ied.SOXEmissions), this.SumArray(baseIed.SOXEmissions));
                this.WriteValue(ref row, column, ied.Sales, baseIed.Sales);
                this.WriteValue(ref row, column, this.SumArray(ied.Vmt), this.SumArray(baseIed.Vmt));
                this.WriteValue(ref row, column, ied.FuelEconomy, baseIed.FuelEconomy, false, true, ied.Sales, baseIed.Sales);
                row += 2;
                this.WriteCostsValue(ref row, column, this.SumArray(ied.PreTaxFuel), this.SumArray(baseIed.PreTaxFuel));
                this.WriteCostsValue(ref row, column, this.SumArray(ied.FuelTax), this.SumArray(baseIed.FuelTax));
                this.WriteCostsValue(ref row, column, this.SumArray(ied.DriveSurplus), this.SumArray(baseIed.DriveSurplus));
                this.WriteCostsValue(ref row, column, this.SumArray(ied.RefuelSurplus), this.SumArray(baseIed.RefuelSurplus));
                this.WriteCostsValue(ref row, column, this.SumArray(ied.EconomicCosts), this.SumArray(baseIed.EconomicCosts));
                this.WriteCostsValue(ref row, column, this.SumArray(ied.CongestionCosts), this.SumArray(baseIed.CongestionCosts));
                this.WriteCostsValue(ref row, column, this.SumArray(ied.NoiseCosts), this.SumArray(baseIed.NoiseCosts));
                this.WriteCostsValue(ref row, column, this.SumArray(ied.AccidentCosts), this.SumArray(baseIed.AccidentCosts));
                this.WriteCostsValue(ref row, column, this.SumArray(ied.CO2Damage), this.SumArray(baseIed.CO2Damage));
                this.WriteCostsValue(ref row, column, this.SumArray(ied.CODamage), this.SumArray(baseIed.CODamage));
                this.WriteCostsValue(ref row, column, this.SumArray(ied.VOCDamage), this.SumArray(baseIed.VOCDamage));
                this.WriteCostsValue(ref row, column, this.SumArray(ied.NOXDamage), this.SumArray(baseIed.NOXDamage));
                this.WriteCostsValue(ref row, column, this.SumArray(ied.PMDamage), this.SumArray(baseIed.PMDamage));
                this.WriteCostsValue(ref row, column, this.SumArray(ied.SOXDamage), this.SumArray(baseIed.SOXDamage));
                row += 2;
                this.WriteCostsValue(ref row, column, this.SumArray(ied.DiscPreTaxFuel), this.SumArray(baseIed.DiscPreTaxFuel));
                this.WriteCostsValue(ref row, column, this.SumArray(ied.DiscFuelTax), this.SumArray(baseIed.DiscFuelTax));
                this.WriteCostsValue(ref row, column, this.SumArray(ied.DiscDriveSurplus), this.SumArray(baseIed.DiscDriveSurplus));
                this.WriteCostsValue(ref row, column, this.SumArray(ied.DiscRefuelSurplus), this.SumArray(baseIed.DiscRefuelSurplus));
                this.WriteCostsValue(ref row, column, this.SumArray(ied.DiscEconomicCosts), this.SumArray(baseIed.DiscEconomicCosts));
                this.WriteCostsValue(ref row, column, this.SumArray(ied.DiscCongestionCosts), this.SumArray(baseIed.DiscCongestionCosts));
                this.WriteCostsValue(ref row, column, this.SumArray(ied.DiscNoiseCosts), this.SumArray(baseIed.DiscNoiseCosts));
                this.WriteCostsValue(ref row, column, this.SumArray(ied.DiscAccidentCosts), this.SumArray(baseIed.DiscAccidentCosts));
                this.WriteCostsValue(ref row, column, this.SumArray(ied.CO2DiscDamage), this.SumArray(baseIed.CO2DiscDamage));
                this.WriteCostsValue(ref row, column, this.SumArray(ied.CODiscDamage), this.SumArray(baseIed.CODiscDamage));
                this.WriteCostsValue(ref row, column, this.SumArray(ied.VOCDiscDamage), this.SumArray(baseIed.VOCDiscDamage));
                this.WriteCostsValue(ref row, column, this.SumArray(ied.NOXDiscDamage), this.SumArray(baseIed.NOXDiscDamage));
                this.WriteCostsValue(ref row, column, this.SumArray(ied.PMDiscDamage), this.SumArray(baseIed.PMDiscDamage));
                this.WriteCostsValue(ref row, column, this.SumArray(ied.SOXDiscDamage), this.SumArray(baseIed.SOXDiscDamage));
            }
        }
        private void WriteCostsValue(ref int row, int col, M6Double value, M6Double baseline)
        {
            this.Buffer[row, col] = value.Total - baseline.Total;
            this.Buffer[row++, col + 1] = (baseline.Total == 0) ? 0 :
                (value.Total - baseline.Total) / baseline.Total;
        }
        private void WriteValue(ref int row, int col, M6Double value)
        {
            this.WriteValue(ref row, col, value, false, false, null);
        }
        private void WriteValueAggr(ref int row, int col, M6Double value)
        {
            this.WriteValue(ref row, col, value, true, false, null);
        }
        private void WriteValue(ref int row, int col, M6Double value, bool isAggr, bool isFe, object sales)
        {
            value.ZeroNaNs();
            if (isAggr)
            {
                if (!isFe)
                {
                    this.Buffer[row++, col] = value.Gasoline;
                    this.Buffer[row++, col] = value.Diesel;
                }
            }
            else
            {
                this.Buffer[row++, col] = value.Ldgv;
                this.Buffer[row++, col] = value.Lddv;
                this.Buffer[row++, col] = value.Ldgt1;
                this.Buffer[row++, col] = value.Lddt1;
                this.Buffer[row++, col] = value.Ldgt2;
                this.Buffer[row++, col] = value.Lddt2;
                this.Buffer[row++, col] = value.Hdgv2b;
                this.Buffer[row++, col] = value.Hddv2b;
            }
            if (isFe)
            {
                M6Double m6Sales = (M6Double)sales;
                M6Double m6SalesOverFe = m6Sales / value;
                if (isAggr)
                {
                    this.Buffer[row++, col] = m6Sales.Gasoline / m6SalesOverFe.Gasoline;
                    this.Buffer[row++, col] = m6Sales.Diesel   / m6SalesOverFe.Diesel;
                }
                this.Buffer[row++, col] = m6Sales.Total / m6SalesOverFe.Total;
            }
            else
            {
                this.Buffer[row++, col] = value.Total;
            }
        }
        private void WriteValue(ref int row, int col, M6Double value, M6Double baseline)
        {
            this.WriteValue(ref row, col, value, baseline, false, false, null, null);
        }
        private void WriteValueAggr(ref int row, int col, M6Double value, M6Double baseline)
        {
            this.WriteValue(ref row, col, value, baseline, true, false, null, null);
        }
        private void WriteValue(ref int row, int col, M6Double value, M6Double baseline, bool isAggr, bool isFe,
            object sales, object baseSales)
        {
            value.ZeroNaNs();
            baseline.ZeroNaNs();
            if (isAggr)
            {
                if (!isFe)
                {
                    this.Buffer[row, col] = value.Gasoline - baseline.Gasoline;
                    this.Buffer[row++, col + 1] = (baseline.Gasoline == 0) ? 0 :
                        (value.Gasoline - baseline.Gasoline) / baseline.Gasoline;
                    this.Buffer[row, col] = value.Diesel - baseline.Diesel;
                    this.Buffer[row++, col + 1] = (baseline.Diesel == 0) ? 0 :
                        (value.Diesel - baseline.Diesel) / baseline.Diesel;
                }
            }
            else
            {
                this.Buffer[row, col] = value.Ldgv - baseline.Ldgv;
                this.Buffer[row++, col + 1] = (baseline.Ldgv == 0) ? 0 :
                    (value.Ldgv - baseline.Ldgv) / baseline.Ldgv;
                this.Buffer[row, col] = value.Lddv - baseline.Lddv;
                this.Buffer[row++, col + 1] = (baseline.Lddv == 0) ? 0 :
                    (value.Lddv - baseline.Lddv) / baseline.Lddv;
                this.Buffer[row, col] = value.Ldgt1 - baseline.Ldgt1;
                this.Buffer[row++, col + 1] = (baseline.Ldgt1 == 0) ? 0 :
                    (value.Ldgt1 - baseline.Ldgt1) / baseline.Ldgt1;
                this.Buffer[row, col] = value.Lddt1 - baseline.Lddt1;
                this.Buffer[row++, col + 1] = (baseline.Lddt1 == 0) ? 0 :
                    (value.Lddt1 - baseline.Lddt1) / baseline.Lddt1;
                this.Buffer[row, col] = value.Ldgt2 - baseline.Ldgt2;
                this.Buffer[row++, col + 1] = (baseline.Ldgt2 == 0) ? 0 :
                    (value.Ldgt2 - baseline.Ldgt2) / baseline.Ldgt2;
                this.Buffer[row, col] = value.Lddt2 - baseline.Lddt2;
                this.Buffer[row++, col + 1] = (baseline.Lddt2 == 0) ? 0 :
                    (value.Lddt2 - baseline.Lddt2) / baseline.Lddt2;
                this.Buffer[row, col] = value.Hdgv2b - baseline.Hdgv2b;
                this.Buffer[row++, col + 1] = (baseline.Hdgv2b == 0) ? 0 :
                    (value.Hdgv2b - baseline.Hdgv2b) / baseline.Hdgv2b;
                this.Buffer[row, col] = value.Hddv2b - baseline.Hddv2b;
                this.Buffer[row++, col + 1] = (baseline.Hddv2b == 0) ? 0 :
                    (value.Hddv2b - baseline.Hddv2b) / baseline.Hddv2b;
            }
            if (isFe)
            {
                M6Double baseM6Sales = (M6Double)baseSales;
                M6Double baseM6SalesOverFe = baseM6Sales / baseline;
                M6Double altM6Sales = (M6Double)sales;
                M6Double altM6SalesOverFe = altM6Sales / value;
                double baseVal = 0;
                double altVal = 0;
                if (isAggr)
                {
                    baseVal = baseM6Sales.Gasoline / baseM6SalesOverFe.Gasoline;
                    altVal    = altM6Sales .Gasoline / altM6SalesOverFe .Gasoline;
                    this.Buffer[row, col] = altVal - baseVal;
                    this.Buffer[row++, col + 1] = (baseVal == 0) ? 0 : (altVal - baseVal) / baseVal;
                    baseVal = baseM6Sales.Diesel / baseM6SalesOverFe.Diesel;
                    altVal    = altM6Sales .Diesel / altM6SalesOverFe .Diesel;
                    this.Buffer[row, col] = altVal - baseVal;
                    this.Buffer[row++, col + 1] = (baseVal == 0) ? 0 : (altVal - baseVal) / baseVal;
                }
                baseVal = baseM6Sales.Total / baseM6SalesOverFe.Total;
                altVal  = altM6Sales .Total / altM6SalesOverFe .Total;
                this.Buffer[row, col] = altVal - baseVal;
                this.Buffer[row++, col + 1] = (baseVal == 0) ? 0 : (altVal - baseVal) / baseVal;
            }
            else
            {
                this.Buffer[row, col] = value.Total - baseline.Total;
                this.Buffer[row++, col + 1] = (baseline.Total == 0) ? 0 :
                    (value.Total - baseline.Total) / baseline.Total;
            }
        }
        private M6Double SumArray(M6Double[] values)
        {
            M6Double value = new M6Double();
            for (int i = 0; i < values.Length; i++)
            {
                value.Ldgv += values[i].Ldgv;
                value.Lddv += values[i].Lddv;
                value.Ldgt1 += values[i].Ldgt1;
                value.Lddt1 += values[i].Lddt1;
                value.Ldgt2 += values[i].Ldgt2;
                value.Lddt2 += values[i].Lddt2;
                value.Hdgv2b += values[i].Hdgv2b;
                value.Hddv2b += values[i].Hddv2b;
            }
            return value;
        }
        #region 
        protected override void SetNumberFormat(XlCell dataCell1, XlCell dataCell2)
        {
            base.SetNumberFormat(dataCell1, dataCell2);
            if (this._numSects < 1)
            {
                return;
            }
            int rows = this.Buffer.Size.Rows;
            int cols = this.Buffer.Size.Columns;
            this._xlu.SetNumberFormat(new XlCell(13, 3), new XlCell(15, cols),
                XlNumberFormats.NumericDecimal2);
            this._xlu.SetNumberFormat(new XlCell(90, 3), new XlCell(98, cols),
                XlNumberFormats.NumericDecimal2);
            for (int i = 0; i < this._numSects; i++)
            {
                int col = 5 + 3 * i;
                this._xlu.SetNumberFormat(new XlCell(2, col), new XlCell(rows, col),
                    XlNumberFormats.Percent);
            }
        }
        protected override void MergeHeaders()
        {
            if (this._numSects < 1)
            {
                return;
            }
            int rows = this.Buffer.Size.Rows;
            int cols = this.Buffer.Size.Columns;
            this._xlu.MergeCells(new XlCell(1, 1), new XlCell(2, 2));
            for (int i = 0; i < this._numSects; i++)
            {
                int col1 = 3 + 3 * i;
                int col2 = 5 + 3 * i;
                this._xlu.MergeCells(new XlCell(1, col1), new XlCell(1, col2));
            }
            this._xlu.MergeCells(new XlCell(3, 1), new XlCell(3, 2));
            this._xlu.MergeCells(new XlCell(3, 3), new XlCell(3, cols));
            for (int i = 0; i < 4; i++)
            {
                int row1 = 4 + 3 * i;
                int row2 = 6 + 3 * i;
                this._xlu.MergeCells(new XlCell(row1, 1), new XlCell(row2, 1));
            }
            this._xlu.MergeCells(new XlCell(16, 1), new XlCell(16, cols));
            this._xlu.MergeCells(new XlCell(17, 1), new XlCell(17, 2));
            this._xlu.MergeCells(new XlCell(17, 3), new XlCell(17, cols));
            for (int i = 0; i < 9; i++)
            {
                int row1 = 18 + 9 * i;
                int row2 = 26 + 9 * i;
                this._xlu.MergeCells(new XlCell(row1, 1), new XlCell(row2, 1));
            }
            this._xlu.MergeCells(new XlCell(99, 1), new XlCell(99, cols));
            this._xlu.MergeCells(new XlCell(100, 3), new XlCell(100, cols));
            this._xlu.MergeCells(new XlCell(115, 1), new XlCell(115, cols));
            this._xlu.MergeCells(new XlCell(116, 3), new XlCell(116, cols));
            for (int i = 0; i < 15; i++)
            {
                this._xlu.MergeCells(new XlCell(100 + i, 1), new XlCell(100 + i, 2));
                this._xlu.MergeCells(new XlCell(116 + i, 1), new XlCell(116 + i, 2));
            }
        }
        protected override void SetBorders(XlCell rHeader1, XlCell rHeader2, XlCell cHeader1, XlCell cHeader2,
            XlCell dataCell1, XlCell dataCell2)
        {
            base.SetBorders(rHeader1, rHeader2, cHeader1, cHeader2, dataCell1, dataCell2);
            if (this._numSects < 1)
            {
                return;
            }
            int rows = this.Buffer.Size.Rows;
            int cols = this.Buffer.Size.Columns;
            for (int i = 0; i < this._numSects - 1; i++)
            {
                int col = 5 + 3 * i;
                this._xlu.SetBorders(new XlCell(3, col), new XlCell(rows, col), false, false, false, true, true, false,
                    XlColor.DarkBlue);
            }
            for (int i = 0; i < 4; i++)
            {
                int row = 4 + 3 * i;
                this._xlu.SetBorders(new XlCell(row, 3), new XlCell(row, cols), true, false, false, false, true, false,
                    XlColor.DarkBlue);
            }
            this._xlu.SetBorders(new XlCell(3, 1), new XlCell(15, cols), true, false, true, true, XlColor.DarkBlue);
            for (int i = 0; i < 9; i++)
            {
                int row = 18 + 9 * i;
                this._xlu.SetBorders(new XlCell(row, 3), new XlCell(row, cols), true, false, false, false, true, false,
                    XlColor.DarkBlue);
            }
            this._xlu.SetBorders(new XlCell(17, 1), new XlCell(98, cols), true, false, true, true, XlColor.DarkBlue);
            this._xlu.SetBorders(new XlCell(101, 3), new XlCell(101, cols), true, false, false, false, true, false,
                XlColor.DarkBlue);
            this._xlu.SetBorders(new XlCell(100, 1), new XlCell(114, cols), true, false, true, true, XlColor.DarkBlue);
            this._xlu.SetBorders(new XlCell(117, 3), new XlCell(117, cols), true, false, false, false, true, false,
                XlColor.DarkBlue);
            this._xlu.SetBorders(new XlCell(116, 1), new XlCell(130, cols), true, false, true, true, XlColor.DarkBlue);
        }
        protected override void SetColor(XlCell rHeader1, XlCell rHeader2, XlCell cHeader1, XlCell cHeader2,
            XlCell dataCell1, XlCell dataCell2)
        {
            base.SetColor(rHeader1, rHeader2, cHeader1, cHeader2, dataCell1, dataCell2);
            if (this._numSects < 1)
            {
                return;
            }
            int cols = this.Buffer.Size.Columns;
            this._xlu.SetFillColor(new XlCell(3, 1), new XlCell(3, cols), XlColor.LightYellow);
            for (int i = 0; i < 4; i++)
            {
                int row = 6 + 3 * i;
                this._xlu.SetFillColor(new XlCell(row, 2), new XlCell(row, cols), XlColor.LightGreen);
            }
            this._xlu.SetFillColor(new XlCell(17, 1), new XlCell(17, cols), XlColor.LightYellow);
            for (int i = 0; i < 9; i++)
            {
                int row = 26 + 9 * i;
                this._xlu.SetFillColor(new XlCell(row, 2), new XlCell(row, cols), XlColor.LightGreen);
            }
            this._xlu.SetFillColor(new XlCell(100, 1), new XlCell(100, cols), XlColor.LightYellow);
            this._xlu.SetFillColor(new XlCell(116, 1), new XlCell(116, cols), XlColor.LightYellow);
        }
        #endregion
        #endregion
        #region 
        private bool _initialized;        
        private int _minYear;
        private int _maxYear;
        private int _numSects;
        #endregion
    }
}

