using System;
using System.Drawing;
using Volpe.Cafe;
using Volpe.Cafe.Collections;
using Volpe.Cafe.Data;
using Volpe.Cafe.IO;
using Volpe.Utils;
namespace Volpe.Cafe.Reporter.ReportTypes
{
    public class ManufacturersReport : Output
    {
        #region 
        public ManufacturersReport(string path, EncryptionSettings cryptoSettings)
            : this(path, TemplateType.ManufacturersReport, cryptoSettings)
        {
        }
        internal ManufacturersReport(string path, TemplateType type, EncryptionSettings cryptoSettings)
            : base(path, type, cryptoSettings, true)
        {
            base._autoBorder = true;
            base._autoChart = false;
            base._autoMerge = true;
            base._autoSummarize = true;
            base._fillColor = XlColor.Automatic;
            this._fitToPage = new Size(1, -1);
            base._font = new XlFont("Arial");
            base._headerAlignment = new XlTextAlignment(XlHAlign.Center, XlVAlign.Center);
            base._headerFillColor = XlColor.Automatic;
            base._headerFont = new XlFont("Arial", 10, false, true, false);
            base._landscape = true;
            base._margins = new XlMargins(0.5);
            base._numberFormat = "#,##0.00";
            base._repeatHeaders = true;
            base._textAlignment = new XlTextAlignment(XlHAlign.General, XlVAlign.Center);
            this._numSects = 0;
            this._numTechs = 0;
        }
        #endregion
        #region 
        public virtual void ParseData(ModelingSettings settings, int scenarioIndex, ModelYear year,
            Industry data, Industry baselineData)
        {
            this._numSects = data.ManufacturerCount + 1;
            this._numTechs = settings.Technologies.Count;
            this._isBaseline = (scenarioIndex == 0);
            this.SetupBuffer(settings, data, "Sn " + scenarioIndex + ", MY " + year.ToString(), "Manufacturer Summary");
            this.SetupHeaders(settings);
            this.WriteData(year, data, baselineData);
            this.SetupCharts(data, baselineData);
        }
        private void SetupCharts(Industry data, Industry baselineData)
        {
            int bufIdx = this.Buffer.Index;
            string bufName = this.Buffer.Name;
            int mfrCount = data.ManufacturerCount;
            object[] chart0Values = new object[mfrCount];
            object[] chart1Values = new object[mfrCount];
            object[] chartXValues = new object[mfrCount];
            for (int i = 0, count = data.ManufacturerCount; i < count; i++)
            {
                ManufacturerModelingData mmd     = data.Manufacturers[i].ModelingData;
                ManufacturerModelingData baseMmd = baselineData.Manufacturers[i].ModelingData;
                chart0Values[i] = mmd.TechCost.TotalAll * 1e-6;
                if (!this._isBaseline)
                {    
                    chart1Values[i] = (mmd.TechCost.TotalAll - baseMmd.TechCost.TotalAll) * 1e-6;
                }
                chartXValues[i] = data.Manufacturers[i].Description.Name;
            }
            XlDataLabelsFormat labelsFormat = new XlDataLabelsFormat(null, "#,##0.00");
            XlChartFormat chart0 = new XlChartFormat(XlChartType.Pie3DExploded, bufName + "_TechCost",
                "Total Incurred Technology Costs ($m)", XlColor.White, XlChartFormat.EmptySeries, null, null);
            XlSeriesFormat chart0sf = new XlSeriesFormat(null, chart0Values, chartXValues);
            chart0sf.HasDataLabels = true;
            chart0sf.DataLabels = labelsFormat;
            chart0.AddSeries(chart0sf);
            this.Buffer.AddChartInfo(chart0);
            if (!this._isBaseline)
            {    
                XlChartFormat chart1 = new XlChartFormat(XlChartType.Pie3DExploded, bufName + "_TechCostDelta",
                    "Total Incurred Technology Costs Vs. Baseline ($m)", XlColor.White, XlChartFormat.EmptySeries,
                    null, null);
                XlSeriesFormat chart1sf = new XlSeriesFormat(null, chart1Values, chartXValues);
                chart1sf.HasDataLabels = true;
                chart1sf.DataLabels = labelsFormat;
                chart1.AddSeries(chart1sf);
                this.Buffer.AddChartInfo(chart1);
            }
        }
        protected virtual void SetupBuffer(ModelingSettings settings, Industry data, string name,
            string title)
        {
            int rows = 2 + Calculate.RcHeadersCount * (Calculate.MfrIndyVHeadersCount + 2 * this._numTechs);
            int cols = 3 + 3 * this._numSects;
            this.AddBuffer(rows, cols);
            this.Buffer.FreezePanes = new XlCell(2, 3);
            this.Buffer.Header = new XlCell(2, 3);
            this.Buffer.Name = name;
            this.Buffer.Title = title + " -- &[Tab]";
        }
        protected virtual void SetupHeaders(ModelingSettings settings)
        {
            string[] vHeaders = Calculate.GetMfrIndyVHeaders();
            string[] rcHeaders = Calculate.GetRcHeaders();
            string[] techHeaders = Calculate.GetTechHeaders(settings.Technologies, true);
            for (int i = 0; i < Calculate.MfrIndyVHeadersCount; i++)
            {
                this.Buffer[2 + i * Calculate.RcHeadersCount, 0] = vHeaders[i];
                for (int j = 0; j < Calculate.RcHeadersCount; j++)
                {
                    this.Buffer[2 + i * Calculate.RcHeadersCount + j, 1] = rcHeaders[j];
                }
            }
            int arRow = 2 + Calculate.MfrIndyVHeadersCount * Calculate.RcHeadersCount;
            int prRow = arRow + Calculate.RcHeadersCount * this._numTechs;
            this.Buffer[arRow, 0] = "Technology Application Rate";
            this.Buffer[prRow, 0] = "Technology Penetration Rate";
            for (int i = 0; i < Calculate.RcHeadersCount; i++)
            {
                this.Buffer[arRow + i * this._numTechs, 1] = rcHeaders[i];
                this.Buffer[prRow + i * this._numTechs, 1] = rcHeaders[i];
                for (int j = 0; j < this._numTechs; j++)
                {
                    this.Buffer[arRow + i * this._numTechs + j, 2] = techHeaders[j];
                    this.Buffer[prRow + i * this._numTechs + j, 2] = techHeaders[j];
                }
            }
        }
        private void WriteData(ModelYear year, Industry data, Industry baselineData)
        {
            this.Buffer[0, 0] = "Manufacturer";
            int column;
            for (int i = 0; i < this._numSects - 1; i++)
            {
                Manufacturer mfr = data.Manufacturers[i];
                column = 3 + 3 * i;
                string mfrName = mfr.Description.Name;
                mfrName = (mfrName.Length == 3) ? mfrName : Global.GetTitleCase(mfrName);
                this.Buffer[0, column] = mfrName;
                this.Buffer[1, column] = "Current\nScenario";
                this.Buffer[1, column + 1] = "Delta\n(abs.)";
                this.Buffer[1, column + 2] = "Delta\n(%)";
                this.WriteManufacturerData(year, column, mfr);
                if (baselineData != null)
                {
                    Manufacturer baseMfr = baselineData.Manufacturers[i];
                    this.WriteManufacturerDelta(year, column + 1, mfr, baseMfr);
                }
            }
            column = 3 * this._numSects;
            this.Buffer[0, column] = year.ToString() + " Total";
            this.Buffer[1, column] = "Current\nScenario";
            this.Buffer[1, column + 1] = "Delta\n(abs.)";
            this.Buffer[1, column + 2] = "Delta\n(%)";
            this.WriteIndustryData(year, column, data, baselineData);
        }
        protected void WriteManufacturerData(ModelYear year, int column, Manufacturer mfr)
        {
            ManufacturerModelingData mmd = mfr.ModelingData;
            int row = 2;    
            RCDouble cw, area;
            this.SumCwAndArea(year, mfr, out cw, out area);
            this.WriteValue(ref row, column, mmd.Sales);
            this.WriteValue(ref row, column, mmd.PreliminaryStandard, false, true, false, mmd.Sales, null);
            this.WriteValue(ref row, column, mmd.Standard, false, true, false, mmd.Sales, null);
            this.WriteValue(ref row, column, mmd.CAFE, true, false, false, mmd.Sales.TotalAll, mmd.SalesOverFE.TotalAll);
            this.WriteValue(ref row, column, this.Divide(cw, mmd.Sales), false, false, true, mmd.Sales, null);
            this.WriteValue(ref row, column, this.Divide(area, mmd.Sales), false, false, true, mmd.Sales, null);
            this.WriteValue(ref row, column, this.Divide(mmd.TechCost, mmd.Sales), false, false, true, mmd.Sales, null);
            this.WriteValue(ref row, column, this.Divide(mmd.Fines, mmd.Sales), false, false, true, mmd.Sales, null);
            this.WriteValue(ref row, column, this.Divide(mmd.RegCost, mmd.Sales), false, false, true, mmd.Sales, null);
            this.WriteValue(ref row, column, this.Divide(mmd.TechCost, 1000000));
            this.WriteValue(ref row, column, this.Divide(mmd.Fines, 1000000));
            this.WriteValue(ref row, column, this.Divide(mmd.RegCost, 1000000));
            RCDouble[] applicationRate, penetrationRate;
            this.SumTechnologyAppRates(year, mfr, out applicationRate, out penetrationRate);
            for (int i = 0; i < this._numTechs; i++)
            {
                this.WriteAppRates(row + i, column, applicationRate[i], penetrationRate[i], mmd.Sales);
            }
        }
        protected void WriteManufacturerDelta(ModelYear year, int column, Manufacturer mfr,
            Manufacturer baselineMfr)
        {
            ManufacturerModelingData mmd     = mfr.ModelingData;
            ManufacturerModelingData baseMmd = baselineMfr.ModelingData;
            int row = 2;    
            RCDouble cw, area, baseCw, baseArea;
            this.SumCwAndArea(year, mfr, out cw, out area);
            this.SumCwAndArea(year, baselineMfr, out baseCw, out baseArea);
            this.WriteValue(ref row, column, mmd.Sales, baseMmd.Sales);
            this.WriteValue(ref row, column, mmd.PreliminaryStandard, baseMmd.PreliminaryStandard, false, true, false, mmd.Sales, null, baseMmd.Sales, null);
            this.WriteValue(ref row, column, mmd.Standard, baseMmd.Standard, false, true, false, mmd.Sales, null, baseMmd.Sales, null);
            this.WriteValue(ref row, column, mmd.CAFE, baseMmd.CAFE, true, false, false, mmd.Sales.TotalAll, mmd.SalesOverFE.TotalAll, baseMmd.Sales.TotalAll, baseMmd.SalesOverFE.TotalAll);
            this.WriteValue(ref row, column, this.Divide(cw, mmd.Sales), this.Divide(baseCw, baseMmd.Sales), false, false, true, mmd.Sales, null, baseMmd.Sales, null);
            this.WriteValue(ref row, column, this.Divide(area, mmd.Sales), this.Divide(baseArea, baseMmd.Sales), false, false, true, mmd.Sales, null, baseMmd.Sales, null);
            this.WriteValue(ref row, column, this.Divide(mmd.TechCost, mmd.Sales), this.Divide(baseMmd.TechCost, baseMmd.Sales), false, false, true, mmd.Sales, null, baseMmd.Sales, null);
            this.WriteValue(ref row, column, this.Divide(mmd.Fines, mmd.Sales), this.Divide(baseMmd.Fines, baseMmd.Sales), false, false, true, mmd.Sales, null, baseMmd.Sales, null);
            this.WriteValue(ref row, column, this.Divide(mmd.RegCost, mmd.Sales), this.Divide(baseMmd.RegCost, baseMmd.Sales), false, false, true, mmd.Sales, null, baseMmd.Sales, null);
            this.WriteValue(ref row, column, this.Divide(mmd.TechCost, 1000000), this.Divide(baseMmd.TechCost, 1000000));
            this.WriteValue(ref row, column, this.Divide(mmd.Fines, 1000000), this.Divide(baseMmd.Fines, 1000000));
            this.WriteValue(ref row, column, this.Divide(mmd.RegCost, 1000000), this.Divide(baseMmd.RegCost, 1000000));
            RCDouble[] applicationRate, penetrationRate, baseApplicationRate, basePenetrationRate;
            this.SumTechnologyAppRates(year, mfr, out applicationRate, out penetrationRate);
            this.SumTechnologyAppRates(year, baselineMfr, out baseApplicationRate, out basePenetrationRate);
            for (int i = 0; i < this._numTechs; i++)
            {
                this.WriteAppRates(row + i, column, applicationRate[i], baseApplicationRate[i],
                    penetrationRate[i], basePenetrationRate[i], mmd.Sales, baseMmd.Sales);
            }
        }
        protected void WriteIndustryData(ModelYear year, int column, Industry data,
            Industry baselineData)
        {
            RCDouble sales, salesOverFe, cw, area, techCost, fines, regCost;
            RCDouble preliminaryStandard, standard, cafe;
            RCDouble[] applicationRate, penetrationRate;
            this.SumIndustryData(data, year, out sales, out salesOverFe, out cw, out area, out techCost, out fines,
                out regCost, out preliminaryStandard, out standard, out cafe, out applicationRate, out penetrationRate);
            int row = 2;    
            this.WriteValue(ref row, column, sales);
            this.WriteValue(ref row, column, preliminaryStandard, false, true, false, sales, null);
            this.WriteValue(ref row, column, standard, false, true, false, sales, null);
            this.WriteValue(ref row, column, cafe, true, false, false, sales.TotalAll, salesOverFe.TotalAll);
            this.WriteValue(ref row, column, this.Divide(cw, sales), false, false, true, sales, null);
            this.WriteValue(ref row, column, this.Divide(area, sales), false, false, true, sales, null);
            this.WriteValue(ref row, column, this.Divide(techCost, sales), false, false, true, sales, null);
            this.WriteValue(ref row, column, this.Divide(fines, sales), false, false, true, sales, null);
            this.WriteValue(ref row, column, this.Divide(regCost, sales), false, false, true, sales, null);
            this.WriteValue(ref row, column, this.Divide(techCost, 1000000));
            this.WriteValue(ref row, column, this.Divide(fines, 1000000));
            this.WriteValue(ref row, column, this.Divide(regCost, 1000000));
            for (int i = 0; i < this._numTechs; i++)
            {
                this.WriteAppRates(row + i, column, applicationRate[i], penetrationRate[i], sales);
            }
            if (baselineData != null)
            {
                RCDouble baseSales, baseSalesOverFe, baseCw, baseArea, baseTechCost, baseFines,
                    baseRegCost;
                RCDouble basePreliminaryStandard, baseStandard, baseCafe;
                RCDouble[] baseApplicationRate, basePenetrationRate;
                this.SumIndustryData(baselineData, year, out baseSales, out baseSalesOverFe, out baseCw, out baseArea,
                    out baseTechCost, out baseFines, out baseRegCost, out basePreliminaryStandard, out baseStandard,
                    out baseCafe, out baseApplicationRate, out basePenetrationRate);
                row = 2;    
                column++;    
                this.WriteValue(ref row, column, sales, baseSales);
                this.WriteValue(ref row, column, preliminaryStandard, basePreliminaryStandard, false, true, false, sales, null, baseSales, null);
                this.WriteValue(ref row, column, standard, baseStandard, false, true, false, sales, null, baseSales, null);
                this.WriteValue(ref row, column, cafe, baseCafe, true, false, false, sales.TotalAll, salesOverFe.TotalAll, baseSales.TotalAll, baseSalesOverFe.TotalAll);
                this.WriteValue(ref row, column, this.Divide(cw, sales), this.Divide(baseCw, baseSales), false, false, true, sales, null, baseSales, null);
                this.WriteValue(ref row, column, this.Divide(area, sales), this.Divide(baseArea, baseSales), false, false, true, sales, null, baseSales, null);
                this.WriteValue(ref row, column, this.Divide(techCost, sales), this.Divide(baseTechCost, baseSales), false, false, true, sales, null, baseSales, null);
                this.WriteValue(ref row, column, this.Divide(fines, sales), this.Divide(baseFines, baseSales), false, false, true, sales, null, baseSales, null);
                this.WriteValue(ref row, column, this.Divide(regCost, sales), this.Divide(baseRegCost, baseSales), false, false, true, sales, null, baseSales, null);
                this.WriteValue(ref row, column, this.Divide(techCost, 1000000), this.Divide(baseTechCost, 1000000));
                this.WriteValue(ref row, column, this.Divide(fines, 1000000), this.Divide(baseFines, 1000000));
                this.WriteValue(ref row, column, this.Divide(regCost, 1000000), this.Divide(baseRegCost, 1000000));
                for (int i = 0; i < this._numTechs; i++)
                {
                    this.WriteAppRates(row + i, column, applicationRate[i], baseApplicationRate[i],
                        penetrationRate[i], basePenetrationRate[i], sales, baseSales);
                }
            }
        }
        private void SumIndustryData(Industry data, ModelYear year, out RCDouble sales, out RCDouble salesOverFe,
            out RCDouble cw, out RCDouble area, out RCDouble techCost, out RCDouble fines, out RCDouble regCost,
            out RCDouble preliminaryStandard, out RCDouble standard, out RCDouble cafe, out RCDouble[] applicationRate,
            out RCDouble[] penetrationRate)
        {
            ManufacturerCollection mfrs = data.Manufacturers;
            int numMfrs = mfrs.Count;
            sales               = new RCDouble();
            salesOverFe         = new RCDouble();
            cw                  = new RCDouble();
            area                = new RCDouble();
            techCost            = new RCDouble();
            fines               = new RCDouble();
            regCost             = new RCDouble();
            preliminaryStandard = new RCDouble();
            standard            = new RCDouble();
            cafe                = new RCDouble();
            applicationRate     = new RCDouble[this._numTechs];
            penetrationRate     = new RCDouble[this._numTechs];
            for (int i = 0; i < numMfrs; i++)
            {
                Manufacturer mfr = mfrs[i];
                ManufacturerModelingData mmd = mfr.ModelingData;
                RCDouble mfrCw, mfrArea;
                this.SumCwAndArea(year, mfr, out mfrCw, out mfrArea);
                RCDouble[] mfrApplicationRate, mfrPenetrationRate;
                this.SumTechnologyAppRates(year, mfr, out mfrApplicationRate, out mfrPenetrationRate);
                this.AddValue(ref sales, mmd.Sales);
                this.AddValue(ref salesOverFe, mmd.SalesOverFE);
                this.AddValue(ref cw, mfrCw);
                this.AddValue(ref area, mfrArea);
                this.AddValue(ref techCost, mmd.TechCost);
                this.AddValue(ref fines, mmd.Fines);
                this.AddValue(ref regCost, mmd.RegCost);
                for (int j = 0; j < this._numTechs; j++)
                {
                    applicationRate[j] += mfrApplicationRate[j] * mmd.Sales;
                    penetrationRate[j] += mfrPenetrationRate[j] * mmd.Sales;
                }
                RCDouble prelStndSum = mmd.Sales / mmd.PreliminaryStandard;
                prelStndSum.ZeroNaNs();
                preliminaryStandard += prelStndSum;
                RCDouble stndSum = mmd.Sales / mmd.Standard;
                stndSum.ZeroNaNs();
                standard += stndSum;
            }
            for (int i = 0; i < this._numTechs; i++)
            {
                applicationRate[i] /= sales;    applicationRate[i].ZeroNaNs();
                penetrationRate[i] /= sales;    penetrationRate[i].ZeroNaNs();
            }
            cafe = sales / salesOverFe;
            preliminaryStandard = sales / preliminaryStandard;
            standard            = sales / standard;
        }
        private void AddValue(ref RCDouble sum, RCDouble value)
        {
            value.ZeroNaNs();
            sum.DomesticAuto += value.DomesticAuto;
            sum.ImportedAuto += value.ImportedAuto;
            sum.LightTruck += value.LightTruck;
            sum.Unregulated += value.Unregulated;
        }
        private void WriteAppRates(int row, int col, RCDouble applicationRate,
            RCDouble penetrationRate, RCDouble sales)
        {
            this.Buffer[row                      , col] = applicationRate.Unregulated;
            this.Buffer[row +     this._numTechs, col] = applicationRate.DomesticAuto;
            this.Buffer[row + 2 * this._numTechs, col] = applicationRate.ImportedAuto;
            this.Buffer[row + 3 * this._numTechs, col] = applicationRate.LightTruck;
            this.Buffer[row + 4 * this._numTechs, col] =
                (applicationRate.Unregulated * sales.Unregulated +
                applicationRate.DomesticAuto * sales.DomesticAuto +
                applicationRate.ImportedAuto * sales.ImportedAuto +
                applicationRate.LightTruck   * sales.LightTruck) / sales.TotalAll;
            this.Buffer[row + 5 * this._numTechs, col] = penetrationRate.Unregulated;
            this.Buffer[row + 6 * this._numTechs, col] = penetrationRate.DomesticAuto;
            this.Buffer[row + 7 * this._numTechs, col] = penetrationRate.ImportedAuto;
            this.Buffer[row + 8 * this._numTechs, col] = penetrationRate.LightTruck;
            this.Buffer[row + 9 * this._numTechs, col] =
                (penetrationRate.Unregulated * sales.Unregulated +
                penetrationRate.DomesticAuto * sales.DomesticAuto +
                penetrationRate.ImportedAuto * sales.ImportedAuto +
                penetrationRate.LightTruck   * sales.LightTruck) / sales.TotalAll;
        }
        private void WriteAppRates(int row, int col, RCDouble applicationRate,
            RCDouble baseApplicationRate, RCDouble penetrationRate, RCDouble basePenetrationRate,
            RCDouble sales, RCDouble baseSales)
        {
            double appRateTotal =
                (applicationRate.Unregulated * sales.Unregulated +
                applicationRate.DomesticAuto * sales.DomesticAuto +
                applicationRate.ImportedAuto * sales.ImportedAuto +
                applicationRate.LightTruck   * sales.LightTruck) / sales.TotalAll;
            double baseAppRateTotal =
                (baseApplicationRate.Unregulated * baseSales.Unregulated +
                baseApplicationRate.DomesticAuto * baseSales.DomesticAuto +
                baseApplicationRate.ImportedAuto * baseSales.ImportedAuto +
                baseApplicationRate.LightTruck   * baseSales.LightTruck) / baseSales.TotalAll;
            this.Buffer[row                        , col] = applicationRate.Unregulated - baseApplicationRate.Unregulated;
            this.Buffer[row +     this._numTechs, col] = applicationRate.DomesticAuto - baseApplicationRate.DomesticAuto;
            this.Buffer[row + 2 * this._numTechs, col] = applicationRate.ImportedAuto - baseApplicationRate.ImportedAuto;
            this.Buffer[row + 3 * this._numTechs, col] = applicationRate.LightTruck - baseApplicationRate.LightTruck;
            this.Buffer[row + 4 * this._numTechs, col] = appRateTotal - baseAppRateTotal;
            this.Buffer[row                        , col + 1] = (baseApplicationRate.Unregulated == 0) ? 0 :
                (applicationRate.Unregulated - baseApplicationRate.Unregulated) / baseApplicationRate.Unregulated;
            this.Buffer[row +     this._numTechs, col + 1] = (baseApplicationRate.DomesticAuto == 0) ? 0 :
                (applicationRate.DomesticAuto - baseApplicationRate.DomesticAuto) / baseApplicationRate.DomesticAuto;
            this.Buffer[row + 2 * this._numTechs, col + 1] = (baseApplicationRate.ImportedAuto == 0) ? 0 :
                (applicationRate.ImportedAuto - baseApplicationRate.ImportedAuto) / baseApplicationRate.ImportedAuto;
            this.Buffer[row + 3 * this._numTechs, col + 1] = (baseApplicationRate.LightTruck == 0) ? 0 :
                (applicationRate.LightTruck - baseApplicationRate.LightTruck) / baseApplicationRate.LightTruck;
            this.Buffer[row + 4 * this._numTechs, col + 1] = (baseAppRateTotal == 0) ? 0 :
                (appRateTotal - baseAppRateTotal) / baseAppRateTotal;
            double penRateTotal =
                (penetrationRate.Unregulated * sales.Unregulated +
                penetrationRate.DomesticAuto * sales.DomesticAuto +
                penetrationRate.ImportedAuto * sales.ImportedAuto +
                penetrationRate.LightTruck   * sales.LightTruck) / sales.TotalAll;
            double basePenRateTotal =
                (basePenetrationRate.Unregulated * baseSales.Unregulated +
                basePenetrationRate.DomesticAuto * baseSales.DomesticAuto +
                basePenetrationRate.ImportedAuto * baseSales.ImportedAuto +
                basePenetrationRate.LightTruck   * baseSales.LightTruck) / baseSales.TotalAll;
            this.Buffer[row + 5 * this._numTechs, col] = penetrationRate.Unregulated - basePenetrationRate.Unregulated;
            this.Buffer[row + 6 * this._numTechs, col] = penetrationRate.DomesticAuto - basePenetrationRate.DomesticAuto;
            this.Buffer[row + 7 * this._numTechs, col] = penetrationRate.ImportedAuto - basePenetrationRate.ImportedAuto;
            this.Buffer[row + 8 * this._numTechs, col] = penetrationRate.LightTruck - basePenetrationRate.LightTruck;
            this.Buffer[row + 9 * this._numTechs, col] = penRateTotal - basePenRateTotal;
            this.Buffer[row + 5 * this._numTechs, col + 1] = (basePenetrationRate.Unregulated == 0) ? 0 :
                (penetrationRate.Unregulated - basePenetrationRate.Unregulated) / basePenetrationRate.Unregulated;
            this.Buffer[row + 6 * this._numTechs, col + 1] = (basePenetrationRate.DomesticAuto == 0) ? 0 :
                (penetrationRate.DomesticAuto - basePenetrationRate.DomesticAuto) / basePenetrationRate.DomesticAuto;
            this.Buffer[row + 7 * this._numTechs, col + 1] = (basePenetrationRate.ImportedAuto == 0) ? 0 :
                (penetrationRate.ImportedAuto - basePenetrationRate.ImportedAuto) / basePenetrationRate.ImportedAuto;
            this.Buffer[row + 8 * this._numTechs, col + 1] = (basePenetrationRate.LightTruck == 0) ? 0 :
                (penetrationRate.LightTruck - basePenetrationRate.LightTruck) / basePenetrationRate.LightTruck;
            this.Buffer[row + 9 * this._numTechs, col + 1] = (basePenRateTotal == 0) ? 0 :
                (penRateTotal - basePenRateTotal) / basePenRateTotal;
        }
        private void WriteValue(ref int row, int col, RCDouble value)
        {
            this.WriteValue(ref row, col, value, false, false, false, null, null);
        }
        private void WriteValue(ref int row, int col, RCDouble value, bool isCafe, bool isStnd, bool isSalesWeight,
            object sales, object salesOverFe)
        {
            value.ZeroNaNs();
            this.Buffer[row++, col] = value.Unregulated;
            this.Buffer[row++, col] = value.DomesticAuto;
            this.Buffer[row++, col] = value.ImportedAuto;
            this.Buffer[row++, col] = value.LightTruck;
            if (isCafe)
            {
                this.Buffer[row++, col] = (double)sales / (double)salesOverFe;
            }
            else if (isStnd)
            {
                RCDouble rcdSales = (RCDouble)sales;
                this.Buffer[row++, col] = (double)(rcdSales.TotalAll / ((RCDouble)(rcdSales / value)).TotalAll);
            }
            else if (isSalesWeight)
            {
                RCDouble rcdSales = (RCDouble)sales;
                this.Buffer[row++, col] = ((RCDouble)(rcdSales * value)).TotalAll / rcdSales.TotalAll;
            }
            else
            {
                this.Buffer[row++, col] = value.TotalAll;
            }
        }
        private void WriteValue(ref int row, int col, RCDouble value, RCDouble baseline)
        {
            this.WriteValue(ref row, col, value, baseline, false, false, false, null, null, null, null);
        }
        private void WriteValue(ref int row, int col, RCDouble value, RCDouble baseline, bool isCafe, bool isStnd,
            bool isSalesWeight, object sales, object salesOverFe, object baseSales, object baseSalesOverFe)
        {
            value.ZeroNaNs();
            baseline.ZeroNaNs();
            this.Buffer[row, col] = value.Unregulated - baseline.Unregulated;
            this.Buffer[row++, col + 1] = (baseline.Unregulated == 0) ? 0 :
                (value.Unregulated - baseline.Unregulated) / baseline.Unregulated;
            this.Buffer[row, col] = value.DomesticAuto - baseline.DomesticAuto;
            this.Buffer[row++, col + 1] = (baseline.DomesticAuto == 0) ? 0 :
                (value.DomesticAuto - baseline.DomesticAuto) / baseline.DomesticAuto;
            this.Buffer[row, col] = value.ImportedAuto - baseline.ImportedAuto;
            this.Buffer[row++, col + 1] = (baseline.ImportedAuto == 0) ? 0 :
                (value.ImportedAuto - baseline.ImportedAuto) / baseline.ImportedAuto;
            this.Buffer[row, col] = value.LightTruck - baseline.LightTruck;
            this.Buffer[row++, col + 1] = (baseline.LightTruck == 0) ? 0 :
                (value.LightTruck - baseline.LightTruck) / baseline.LightTruck;
            double baseVal = 0, altVal = double.NaN;
            if (isCafe)
            {
                baseVal = (double)baseSales / (double)baseSalesOverFe;
                altVal    = (double)sales        / (double)salesOverFe;
            }
            else if (isStnd)
            {
                RCDouble baseRcdSales = (RCDouble)baseSales;
                RCDouble rcdSales      = (RCDouble)sales;
                baseVal = (double)(baseRcdSales.TotalAll / ((RCDouble)(baseRcdSales / baseline)).TotalAll);
                altVal    = (double)(rcdSales       .TotalAll / ((RCDouble)(rcdSales        / value      )).TotalAll);
            }
            else if (isSalesWeight)
            {
                RCDouble baseRcdSales = (RCDouble)sales;
                RCDouble rcdSales      = (RCDouble)baseSales;
                baseVal = ((RCDouble)(baseRcdSales * baseline)).TotalAll / baseRcdSales.TotalAll;
                altVal    = ((RCDouble)(rcdSales       * value     )).TotalAll / rcdSales       .TotalAll;
            }
            else
            {
                baseVal = baseline.TotalAll;
                altVal    = value.TotalAll;
            }
            if (!double.IsNaN(altVal))
            {
                double diff = altVal - baseVal;
                this.Buffer[row, col] = diff;
                this.Buffer[row++, col + 1] = (baseVal == 0) ? 0 : diff / baseVal;
            }
        }
        private RCDouble Divide(RCDouble value1, RCDouble value2)
        {
            value1.ZeroNaNs();
            value2.ZeroNaNs();
            value1.DomesticAuto /= value2.DomesticAuto;
            value1.ImportedAuto /= value2.ImportedAuto;
            value1.LightTruck    /= value2.LightTruck;
            value1.Unregulated    /= value2.Unregulated;
            value1.ZeroNaNs();
            return value1;
        }
        private RCDouble Divide(RCDouble value1, double value2)
        {
            value1.ZeroNaNs();
            if (value2 > 0)
            {
                value1.DomesticAuto /= value2;
                value1.ImportedAuto /= value2;
                value1.LightTruck    /= value2;
                value1.Unregulated    /= value2;
            }
            return value1;
        }
        private void SumCwAndArea(ModelYear year, Manufacturer mfr, out RCDouble cw, out RCDouble area)
        {
            cw = new RCDouble();
            area = new RCDouble();
            VehicleCollection vehs = mfr.GetVehicles();
            int numVehs = vehs.Count;
            int myIdx = year.Index;
            for (int i = 0; i < numVehs; i++)
            {
                Vehicle veh = vehs[i];
                if (veh.IsValid(year))
                {
                    VehicleDescription vehDescr = veh.Description;
                    RegulatoryClass regClass = veh.RegClass;
                    double vehSales = vehDescr.Sales[myIdx];
                    cw  [regClass] += vehDescr.CurbWeight * vehSales;
                    area[regClass] += vehDescr.Footprint * vehSales;
                }
            }
            cw.ZeroNaNs();
            area.ZeroNaNs();
        }
        private void SumTechnologyAppRates(ModelYear year, Manufacturer mfr,
            out RCDouble[] applicationRate, out RCDouble[] penetrationRate)
        {
            applicationRate = new RCDouble[this._numTechs];
            penetrationRate = new RCDouble[this._numTechs];
            VehicleCollection vehs = mfr.GetVehicles();
            int numVehs = vehs.Count;
            int myIdx = year.Index;
            for (int i = 0; i < numVehs; i++)
            {
                Vehicle veh = vehs[i];
                if (veh.IsValid(year))
                {
                    VehicleDescription vehDescr = veh.Description;
                    VehicleModelingData vmd = veh.ModelingData;
                    RegulatoryClass regClass = veh.RegClass;
                    double vehSales = vehDescr.Sales[myIdx];
                    for (int j = 0; j < this._numTechs; j++)
                    {
                        if (vmd.TechUsed[j] && !vmd.TechSuperseded[j] && !vmd.TechPhaseInBackfilled[j])
                        {
                            if (vmd.TechApplied[j]) { applicationRate[j][regClass] += vehSales; }
                            penetrationRate[j][regClass] += vehSales;
                        } 
                    } 
                } 
            }
            RCDouble sales = mfr.ModelingData.Sales;
            for (int i = 0; i < this._numTechs; i++)
            {
                applicationRate[i] /= sales;    applicationRate[i].ZeroNaNs();
                penetrationRate[i] /= sales;    penetrationRate[i].ZeroNaNs();
            }
        }
        #region 
        protected override void SetColor(XlCell rHeader1, XlCell rHeader2, XlCell cHeader1,
            XlCell cHeader2, XlCell dataCell1, XlCell dataCell2)
        {
            base.SetColor(rHeader1, rHeader2, cHeader1, cHeader2, dataCell1, dataCell2);
            if (this._numSects < 1) { return; }
            int cols = this.Buffer.Size.Columns;
            for (int i = 0; i < Calculate.MfrIndyVHeadersCount; i++)
            {
                int row = 7 + 5 * i;
                this._xlu.SetFillColor(new XlCell(row, 2), new XlCell(row, cols), XlColor.LightGreen);
            }
        }
        protected override void SetNumberFormat(XlCell dataCell1, XlCell dataCell2)
        {
            if (this._numSects < 1) { return; }
            int rows = this.Buffer.Size.Rows;
            int cols = this.Buffer.Size.Columns;
            this._xlu.SetNumberFormat(new XlCell(3, 4), new XlCell(32, cols), XlNumberFormats.Numeric);
            this._xlu.SetNumberFormat(new XlCell(8, 4), new XlCell(22, cols), XlNumberFormats.NumericDecimal2);
            this._xlu.SetNumberFormat(new XlCell(33, 4), new XlCell(62, cols), XlNumberFormats.AccountingDecimal2);
            this._xlu.SetNumberFormat(new XlCell(63, 4), new XlCell(rows, cols), XlNumberFormats.Percent);
            for (int i = 0; i < this._numSects; i++)
            {
                int col = 6 + 3 * i;
                this._xlu.SetNumberFormat(new XlCell(2, col), new XlCell(62, col), XlNumberFormats.Percent);
            }
        }
        protected override void SetBorders(XlCell rHeader1, XlCell rHeader2, XlCell cHeader1,
            XlCell cHeader2, XlCell dataCell1, XlCell dataCell2)
        {
            base.SetBorders(rHeader1, rHeader2, cHeader1, cHeader2, dataCell1, dataCell2);
            if (this._numSects < 1) { return; }
            int rows = this.Buffer.Size.Rows;
            int cols = this.Buffer.Size.Columns;
            for (int i = 0; i < Calculate.MfrIndyVHeadersCount; i++)
            {
                int row = 7 + 5 * i;
                this._xlu.SetBorders(new XlCell(row, 4), new XlCell(row, cols), false, true, false,
                    false, true, false, XlColor.DarkBlue);
            }
            for (int i = 0; i < 9; i++)
            {
                int row = 62 + this._numTechs * (i + 1);
                this._xlu.SetBorders(new XlCell(row, 4), new XlCell(row, cols), false, true, false,
                    false, true, false, XlColor.DarkBlue);
            }
            for (int i = 0; i < this._numSects - 1; i++)
            {
                int col = 6 + 3 * i;
                this._xlu.SetBorders(new XlCell(3, col), new XlCell(rows, col), false, false, false,
                    true, true, false, XlColor.DarkBlue);
            }
        }
        protected override void MergeHeaders()
        {
            if (this._numSects < 1) { return; }
            this._xlu.MergeCells(new XlCell(1, 1), new XlCell(2, 3));
            for (int i = 0; i < Calculate.MfrIndyVHeadersCount; i++)
            {
                int row1 = 3 + 5 * i;
                int row2 = 7 + 5 * i;
                this._xlu.MergeCells(new XlCell(row1, 1), new XlCell(row2, 1));
                for (int j = 0; j < 5; j++)
                {
                    int row = row1 + j;
                    this._xlu.MergeCells(new XlCell(row, 2), new XlCell(row, 3));
                }
            }
            this._xlu.MergeCells(new XlCell(63, 1),  new XlCell(62 + 5 * this._numTechs, 1));
            this._xlu.MergeCells(new XlCell(63 + 5 * this._numTechs, 1), new XlCell(62 + 10 * this._numTechs, 1));
            for (int i = 0; i < 10; i++)
            {
                int row1 = 63 + this._numTechs * i;
                int row2 = 62 + this._numTechs * (i + 1);
                this._xlu.MergeCells(new XlCell(row1, 2), new XlCell(row2, 2));
            }
            for (int i = 0; i < this._numSects; i++)
            {
                int col1 = 4 + 3 * i;
                int col2 = 6 + 3 * i;
                this._xlu.MergeCells(new XlCell(1, col1), new XlCell(1, col2));
            }
        }
        #endregion
        #endregion
        #region 
        protected int _numSects;
        protected int _numTechs;
        protected bool _isBaseline;
        #endregion
    }
}

