using System;
using System.Drawing;
using Volpe.Cafe.Collections;
using Volpe.Cafe.Data;
using Volpe.Cafe.IO;
using Volpe.Utils;
namespace Volpe.Cafe.Reporter.ReportTypes
{
    public class VehiclesReport : Output
    {
        #region 
        public VehiclesReport(string path, EncryptionSettings cryptoSettings)
            : base(path, TemplateType.VehiclesReport, cryptoSettings, true)
        {
            base._autoBorder = true;
            base._autoFilter = true;
            base._autoChart = true;
            base._autoMerge = true;
            base._autoSummarize = true;
            base._fillColor = XlColor.Automatic;
            this._fitToPage = new Size(2, -1);
            base._font = new XlFont("Arial");
            base._headerAlignment = new XlTextAlignment(XlHAlign.Center, XlVAlign.Center);
            base._headerFillColor = XlColor.Automatic;
            base._headerFont = new XlFont("Arial", 10, false, true, false);
            base._landscape = true;
            base._margins = new XlMargins(0.5);
            base._numberFormat = "#,##0";
            base._repeatHeaders = true;
            base._textAlignment = new XlTextAlignment(XlHAlign.General, XlVAlign.Center);
            this._numMfrs = 0;
            this._numVehs = 0;
            this._numTechs = 0;
        }
        #endregion
        #region 
        public void ParseData(ModelingSettings settings, int scenarioIndex, ModelYear year, Industry data,
            Industry initialData)
        {
            this._numMfrs = data.Manufacturers.Count;
            this._numVehs = data.VehicleCount;
            this._numTechs = settings.Technologies.Count;
            string[] headers = Calculate.GetVehHeaders();
            string[] techHeaders = Calculate.GetTechHeaders(settings.Technologies, false);
            int rows = 2 + this._numVehs;
            int cols = headers.Length + this._numTechs;
            this.AddBuffer(rows, cols);
            this.Buffer.FreezePanes = new XlCell(2, 5);
            this.Buffer.Header = new XlCell(2, 0);
            this.Buffer.Name = "Sn " + scenarioIndex + ", MY " + year.ToString();
            this.Buffer.SetFirstSortKey(2, false);
            this.Buffer.SetSecondSortKey(5, false);
            this.Buffer.SetThirdSortKey(1, false);
            this.Buffer.Title = "Vehicle Summary -- &[Tab]";
            for (int i = 0; i < headers.Length; i++)
            {
                this.Buffer[0, i] = headers[i];
            }
            this.Buffer[1, 8] = this.Buffer[1, 12] = this.Buffer[1, 14] = "Initial";
            this.Buffer[1, 9] = this.Buffer[1, 13] = this.Buffer[1, 15] = "Final";
            this.Buffer[1, 17] = "ID#";        
            this.Buffer[1, 18] = "Fuel";
            this.Buffer[1, 19] = "Disp\n(lit.)";
            this.Buffer[1, 20] = "Cyl.";
            this.Buffer[1, 21] = "ID#";        
            this.Buffer[1, 22] = "Type";
            this.Buffer[1, 23] = this.Buffer[1, 25] = "Incurred\nTech Cost";
            this.Buffer[1, 24] = "Price\nIncrease";
            this.Buffer[1, 26] = "Increase in\nSales Rev.";
            this.Buffer[0, 29] = "Technology Utilization/Applicability\n(\"-\" used in PP; \"+\" applied by model; \"-+\" or \"++\" superseded; \"%\" phase-in limit reached; \"%+\" phased-in, but backfilled; \"x\" not applicable)";
            for (int i = 0; i < this._numTechs; i++)
            {
                this.Buffer[1, 29 + i] = techHeaders[i];
            }
            int vehCount = this.WriteVehiclesData(initialData, year, data);
            if (vehCount != this._numVehs)
            {
                this.Buffer.ResizeBuffer(new XlSize(vehCount + 2, this.Buffer.Size.Columns), true);
            }
        }
        private int WriteVehiclesData(Industry initialData, ModelYear year, Industry data)
        {
            VehicleCollection initVehs = initialData.GetVehicles();
            VehicleCollection vehs = data.GetVehicles();
            if (initVehs.Count != vehs.Count)
            {    
                throw new InvalidOperationException("Number of valid vehicles in the scenario " +
                    "does not match number of valid vehicles loaded.");
            }
            int myIdx = year.Index;
            int row = 2;
            for (int i = 0; i < this._numVehs; i++)
            {
                Vehicle veh = vehs[i];
                if (veh.IsValid(year))
                {
                    ManufacturerModelingData mmd      = veh.Manufacturer.ModelingData;
                    VehicleDescription       vehDescr = veh.Description;
                    VehicleModelingData      vmd      = veh.ModelingData;
                    Engine                   eng      = veh.Engine;
                    EngineDescription        engDescr = eng.Description;
                    Transmission             trn      = veh.Transmission;
                    TransmissionDescription  trnDescr = trn.Description;
                    double vehSales = vehDescr.Sales[myIdx];
                    Vehicle initVeh = initVehs[i];
                    VehicleDescription initVehDescr = initVeh.Description;
                    string mfrName = veh.Description.Manufacturer;
                    mfrName = (mfrName.Length == 3) ? mfrName : Global.GetTitleCase(mfrName);
                    int col = 0;
                    this.Buffer[row, col++] = i;
                    this.Buffer[row, col++] = vehDescr.Code;
                    this.Buffer[row, col++] = mfrName;
                    this.Buffer[row, col++] = "'" + Global.GetTitleCase(vehDescr.Model);
                    this.Buffer[row, col++] = "'" + Global.GetTitleCase(vehDescr.Nameplate);
                    this.Buffer[row, col++] = veh.RegClass.ToString();
                    this.Buffer[row, col++] = veh.TechnologyClass.ToString();
                    this.Buffer[row, col++] = (vehDescr.PredecessorCode == 0) ? "" : vehDescr.PredecessorCode.ToString();
                    this.Buffer[row, col++] = initVehDescr.Sales[myIdx];
                    this.Buffer[row, col++] = vehSales;
                    this.Buffer[row, col++] = initVehDescr.Msrp[myIdx];
                    this.Buffer[row, col++] = initVehDescr.Price[myIdx];
                    this.Buffer[row, col++] = initVehDescr.FuelEconomy;
                    this.Buffer[row, col++] = vehDescr.FuelEconomy;
                    this.Buffer[row, col++] = initVehDescr.CurbWeight;
                    this.Buffer[row, col++] = vehDescr.CurbWeight;
                    this.Buffer[row, col++] = vehDescr.Footprint;
                    string engStr = engDescr.Code.ToString();
                    if (eng.Parent != null)
                    {   
                        engStr += " (" + eng.Parent.Description.Code.ToString() + ")";
                    }
                    this.Buffer[row, col++] = engStr;
                    this.Buffer[row, col++] = engDescr.Fuel;
                    this.Buffer[row, col++] = engDescr.Displacement;
                    this.Buffer[row, col++] = engDescr.Cylinders;
                    bool   isHev    = false;
                    string trnType  = trnDescr.Type.ToUpper();
                    string trnCntrl = trnDescr.Control.ToUpper();
                    if      (vmd.TechUsed[TechnologyIndexes.PSHEV]) { isHev = true; trnType = "PSHEV"; }
                    else if (vmd.TechUsed[TechnologyIndexes.MHEV2]) { isHev = true; trnType = "2MHEV"; }
                    else if (vmd.TechUsed[TechnologyIndexes.PHEV ]) { isHev = true; trnType = "PHEV" ; }
                    string trnStr = (isHev) ? "HEV" : trnDescr.Code.ToString();
                    if (!isHev && trn.Parent != null)
                    {   
                        trnStr += " (" + trn.Parent.Description.Code.ToString() + ")";
                    }
                    this.Buffer[row, col++] = trnStr;
                    string reportedTrnType =
                        (isHev                            ) ? trnType :
                        (trnType.StartsWith("CVT")        ) ? "CVT"   :
                        (trnType == "C" && trnCntrl == "A") ? "DCTAM" : trnCntrl + trnDescr.NumForwardGears;
                    this.Buffer[row, col++] = reportedTrnType;
                    this.Buffer[row, col++] = vmd.TechCost / vehSales;
                    this.Buffer[row, col++] = vmd.RegCost / vehSales;
                    this.Buffer[row, col++] = vmd.TechCost * 1e-3;      
                    this.Buffer[row, col++] = vmd.RegCost  * 1e-3;      
                    bool atRedesign, inShadowOfRedesign;
                    bool atRefresh , inShadowOfRefresh;
                    veh.GetRedesignState(year, out atRedesign, out inShadowOfRedesign);
                    veh.GetRefreshState (year, out atRefresh , out inShadowOfRefresh );
                    this.Buffer[row, col++] = (atRedesign) ? "At Redesign" : (inShadowOfRedesign) ? "Shadow" : null;
                    this.Buffer[row, col++] = (atRefresh ) ? "At Refresh"  : (inShadowOfRefresh ) ? "Shadow" : null;
                    for (int j = 0; j < this._numTechs; j++)
                    {
                        string techUsedStr = null;
                        if (vmd.TechUsed[j] &&  vmd.TechApplied[j] && vmd.TechPhaseInBackfilled[j] && !vmd.TechSuperseded[j])
                        {   
                            techUsedStr = "%+";
                        }
                        else
                        {   
                            techUsedStr =
                                ( vmd.TechUsed     [j] && !vmd.TechApplied[j]) ? "-" :
                                ( vmd.TechUsed     [j] &&  vmd.TechApplied[j]) ? "+" :
                                (!vmd.TechEnabled  [j]                       ) ? "x" :
                                ( mmd.TechExhausted[j][veh.RegClass]         ) ? "%" : null;
                            if (vmd.TechSuperseded[j]) { techUsedStr += "+"; }
                        }
                        this.Buffer[row, col++] = techUsedStr;
                    }
                    row++;
                }
            }
            return row - 2;
        }
        #region 
        protected override void AlignText(XlCell rHeader1, XlCell rHeader2, XlCell cHeader1,
            XlCell cHeader2, XlCell dataCell1, XlCell dataCell2)
        {
            base.AlignText(rHeader1, rHeader2, cHeader1, cHeader2, dataCell1, dataCell2);
            if (this._numMfrs < 1) { return; }
            this._xlu.AlignText(new XlCell(3, 28), new XlCell(this.Buffer.Size.Rows,
                this.Buffer.Size.Columns), new XlTextAlignment(XlHAlign.Center, XlVAlign.Center));
        }
        protected override void Autofit(XlCell cell1, XlCell cell2)
        {
            base.Autofit(cell1, cell2);
            base.XlUtils.SetRowHeight(cell1, new XlCell(1, cell2.Column), 25.5);
        }
        protected override void MergeHeaders()
        {
            if (this._numMfrs < 1) { return; }
            for (int i = 1; i < 9; i++)
            {
                this._xlu.MergeCells(new XlCell(1, i), new XlCell(2, i));
            }
            this._xlu.MergeCells(new XlCell(1, 11), new XlCell(2, 11));
            this._xlu.MergeCells(new XlCell(1, 12), new XlCell(2, 12));
            this._xlu.MergeCells(new XlCell(1, 17), new XlCell(2, 17));
            this._xlu.MergeCells(new XlCell(1, 28), new XlCell(2, 28));
            this._xlu.MergeCells(new XlCell(1, 29), new XlCell(2, 29));
            this._xlu.MergeCells(new XlCell(1,  9), new XlCell(1, 10));
            this._xlu.MergeCells(new XlCell(1, 13), new XlCell(1, 14));
            this._xlu.MergeCells(new XlCell(1, 15), new XlCell(1, 16));
            this._xlu.MergeCells(new XlCell(1, 18), new XlCell(1, 21));
            this._xlu.MergeCells(new XlCell(1, 22), new XlCell(1, 23));
            this._xlu.MergeCells(new XlCell(1, 24), new XlCell(1, 25));
            this._xlu.MergeCells(new XlCell(1, 26), new XlCell(1, 27));
            this._xlu.MergeCells(new XlCell(1, 30), new XlCell(1, 29 + this._numTechs));
        }
        protected override void SetBorders(XlCell rHeader1, XlCell rHeader2, XlCell cHeader1,
            XlCell cHeader2, XlCell dataCell1, XlCell dataCell2)
        {
            base.SetBorders(rHeader1, rHeader2, cHeader1, cHeader2, dataCell1, dataCell2);
            if (this._numMfrs < 1) { return; }
            this._xlu.SetBorders(new XlCell(3, 1), new XlCell(this.Buffer.Size.Rows, 5), false,
                true, true, false, XlColor.DarkBlue);
            this._xlu.SetBorders(new XlCell(3, 1), new XlCell(this.Buffer.Size.Rows, 5), false,
                false, false, true, true, false, XlColor.DarkBlue);
        }
        protected override void SetFont(XlCell rHeader1, XlCell rHeader2, XlCell cHeader1, XlCell cHeader2,
            XlCell dataCell1, XlCell dataCell2)
        {
            base.SetFont(rHeader1, rHeader2, cHeader1, cHeader2, dataCell1, dataCell2);
            if (this._numMfrs < 1) { return; }
            this._xlu.SetFont(new XlCell(3, 30), new XlCell(this.Buffer.Size.Rows, 29 + this._numTechs),
                this._headerFont);
        }
        protected override void SetNumberFormat(XlCell dataCell1, XlCell dataCell2)
        {
            base.SetNumberFormat(dataCell1, dataCell2);
            if (this._numMfrs < 1) { return; }
            this._xlu.SetNumberFormat(new XlCell(3, 13), new XlCell(this.Buffer.Size.Rows, 14),
                XlNumberFormats.NumericDecimal2);
            this._xlu.SetNumberFormat(new XlCell(3, 20), new XlCell(this.Buffer.Size.Rows, 20),
                XlNumberFormats.NumericDecimal1);
        }
        #endregion
        #endregion
        #region 
        private int _numMfrs;
        private int _numVehs;
        private int _numTechs;
        #endregion
    }
}

