using System;
using Volpe.Cafe.Collections;
namespace Volpe.Cafe.Data.MonteCarlo
{
    [Serializable]
    public struct TrialInfo : ICloneable
    {
        #region 
        #region 
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        public TrialInfo Clone()
        {
            TrialInfo ti = new TrialInfo();
            ti.Index                = this.Index;
            ti.FuelPriceEstimates   = this.FuelPriceEstimates;
            ti.DiscountRate         = this.DiscountRate;
            ti.ReboundEffect        = this.ReboundEffect;
            ti.CO2Costs             = this.CO2Costs;
            ti.MonopsonyCost        = this.MonopsonyCost;
            ti.PriceShockCost       = this.PriceShockCost;
            ti.MilitarySecurityCost = this.MilitarySecurityCost;
            ti.TechCostScaleFactor  = (double[])Global.CloneArray(this.TechCostScaleFactor, typeof(double));
            ti.TechFcScaleFactor    = (double[])Global.CloneArray(this.TechFcScaleFactor  , typeof(double));
            return ti;
        }
        #endregion
        public void InitializeTrial(Industry data, ModelingSettings settings)
        {
            settings.OperatingModes.FuelPriceEstimates = this.FuelPriceEstimates;
            ManufacturerCollection mfrs = data.Manufacturers;
            for (int j = 0, mfrCount = data.ManufacturerCount; j < mfrCount; j++)
            {
                mfrs[j].Description.DiscountRate = this.DiscountRate;
            }
            settings.Parameters.UpdateForMonteCarlo(this.DiscountRate, this.ReboundEffect, this.CO2Costs, this.MonopsonyCost,
                this.PriceShockCost, this.MilitarySecurityCost);
            TechnologyCollection techs = settings.Technologies;
            double[] costScale = this.TechCostScaleFactor;
            double[] fcScale   = this.TechFcScaleFactor;
            for (int i = 0, techCount = techs.Count; i < techCount; i++)
            {
                Technology tech = techs[i];
                TechnologyAttributes[] techAttr = tech.Attributes.ToArray();
                int techAttrCount = techAttr.Length;
                for (int j = 0; j < techAttrCount; j++)
                {
                    techAttr[j].CostHigh *= costScale[i];
                    techAttr[j].CostLow  *= costScale[i];
                    techAttr[j].FcHigh   *= fcScale[i];
                    techAttr[j].FcLow    *= fcScale[i];
                } 
            } 
        }
        public static string CreateCsvHeader()
        {
            string cost = "", fc = "";
            for (int i = 0; i < TechnologyIndexes.TechnologyCount; i++)
            {
                cost += "Cost(" + TechnologyIndexes.TechAbbr[i] + "),";
                fc   += "FC("   + TechnologyIndexes.TechAbbr[i] + "),";
            }
            return "Index,FuelPriceEstimates,DiscountRate,ReboundEffect,CO2Costs,MonopsonyCost,PriceShockCost," +
                "MilitarySecurityCost," + cost + fc.Remove(fc.Length - 1, 1);
        }
        public string ToCsvString()
        {
            return
                this.Index                + "," +
                this.FuelPriceEstimates   + "," +
                this.DiscountRate         + "," +
                this.ReboundEffect        + "," +
                this.CO2Costs             + "," +
                this.MonopsonyCost        + "," +
                this.PriceShockCost       + "," +
                this.MilitarySecurityCost + "," +
                this.ArrToCsv(this.TechCostScaleFactor) + "," +
                this.ArrToCsv(this.TechFcScaleFactor);
        }
        public void FromCsvString(string value)
        {
            string[] parts = value.Split(',');
            this.Index = int.Parse(parts[0]);
            this.FuelPriceEstimates =
                (Global.StringCompare(parts[1], "LOW",  false)) ? Estimates.Low  :
                (Global.StringCompare(parts[1], "HIGH", false)) ? Estimates.High : Estimates.Average;
            this.DiscountRate         = double.Parse(parts[2]);
            this.ReboundEffect        = double.Parse(parts[3]);
            this.CO2Costs             = double.Parse(parts[4]);
            this.MonopsonyCost        = double.Parse(parts[5]);
            this.PriceShockCost       = double.Parse(parts[6]);
            this.MilitarySecurityCost = double.Parse(parts[7]);
            this.TechCostScaleFactor  = new double[TechnologyIndexes.TechnologyCount];
            this.TechFcScaleFactor    = new double[TechnologyIndexes.TechnologyCount];
            for (int i = 0; i < TechnologyIndexes.TechnologyCount; i++)
            {   
                this.TechCostScaleFactor[i] = double.Parse(parts[8 + i]);
                this.TechFcScaleFactor  [i] = double.Parse(parts[8 + TechnologyIndexes.TechnologyCount + i]);
            }
        }
        string ArrToCsv(double[] arr)
        {
            string result = "";
            for (int i = 0, count = arr.Length; i < count; i++)
            {
                result += arr[i] + ",";
            }
            return result.Remove(result.Length - 1, 1);
        }
        #endregion
        #region 
        public int Index;
        public Estimates FuelPriceEstimates;
        public double DiscountRate;
        public double ReboundEffect;
        public double CO2Costs;
        public double MonopsonyCost;
        public double PriceShockCost;
        public double MilitarySecurityCost;
        public double[] TechCostScaleFactor;
        public double[] TechFcScaleFactor;
        #endregion
    }
}

