using System;
using System.Collections.Specialized;
using Volpe.Cafe;
using Volpe.Cafe.Collections;
namespace Volpe.Cafe.Data
{
    [Serializable]
    public class Vehicle : ICloneable
    {
        #region 
        private Vehicle()
        {
            this._valid = false;
            this._initialized = false;
            this._vehicleType  = VehicleType.None;
            this._engine       = null;
            this._transmission = null;
            this._namePlate    = null;
            this._manufacturer = null;
            this._predecessor  = null;
            this._successors   = new VehicleCollection(8);
            this._regClass     = RegulatoryClass.Unregulated;
            this._mobile6Class = Mobile6Class.None;
        }
        public Vehicle(VehicleDescription description)
            : this()
        {
            this._description = description;
            this._modelingData = new VehicleModelingData();
        }
        #endregion
        #region 
        #region 
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        public Vehicle Clone()
        {
            Vehicle veh = new Vehicle();
            veh._initialized = this._initialized;
            veh._valid = this._valid;
            veh._invalidReasons = (string[])Global.CloneArray(this._invalidReasons, typeof(string));
            veh._loadWarnings = (string[])Global.CloneArray(this._loadWarnings, typeof(string));
            if (this._validYears != null)
            {
                veh._validYears = new ModelYear[this._validYears.Length];
                for (int i = 0; i < this._validYears.Length; i++)
                {
                    veh._validYears[i] = new ModelYear(this._validYears[i].Year);
                }
            }
            veh._description  = this._description.Clone();
            veh._modelingData = this._modelingData.Clone();
            veh._vehicleType  = this._vehicleType;
            veh._engine       = this._engine;
            veh._transmission = this._transmission;
            veh._namePlate    = this._namePlate;
            veh._manufacturer = this._manufacturer;
            veh._predecessor  = this._predecessor;
            veh._successors.AddRange(this._successors);
            veh._regClass     = this._regClass;
            veh._mobile6Class = this._mobile6Class;
            return veh;
        }
        #endregion
        internal void Initialize(Manufacturer manufacturer, Vehicle[] vehicles)
        {
            this._manufacturer = manufacturer;
            this._engine = null;
            this._transmission = null;
            this._predecessor = null;
            Engine[] engs = manufacturer.Engines.ToArray();
            Transmission[] trns = manufacturer.Transmissions.ToArray();
            for (int i = 0; i < engs.Length; i++)
            {
                if (engs[i].Description.Code == this._description.EngineCode)
                {   
                    this._engine = engs[i];
                    this._engine.Vehicles.Add(this);
                    break;
                }
            }
            for (int i = 0; i < trns.Length; i++)
            {
                if (trns[i].Description.Code == this._description.TransmissionCode)
                {    
                    this._transmission = trns[i];
                    this._transmission.Vehicles.Add(this);
                    break;
                }
            }
            for (int i = 0; i < vehicles.Length; i++)
            {
                if (this._description.PredecessorCode == vehicles[i]._description.Code)
                {   
                    this._predecessor = vehicles[i];
                    vehicles[i]._successors.Add(this);
                    if (this._description.Style == "" && this._predecessor._description.Style != "")
                    {
                        this._description.Style = this._predecessor._description.Style;
                    }
                    if (this._description.HybridType == "" && this._predecessor._description.HybridType != "")
                    {
                        this._description.HybridType = this._predecessor._description.HybridType;
                    }
                    if (this._description.Origin == '\0' && this._predecessor._description.Origin != '\0')
                    {
                        this._description.Origin = this._predecessor._description.Origin;
                    }
                    if (this._description.GVWR <= 0)   { this._description.GVWR = this._predecessor._description.GVWR; }
                    break;
                }
            }
            this._initialized = true;
        }
        internal void ValidatePredecessor()
        {
            if (!this.Valid || this._description.PredecessorCode == 0) { return; }
            StringCollection err = new StringCollection();
            if (this._predecessor == null)
            {   
                err.Add("The Vehicle's Predecessor at Code, " + this._description.PredecessorCode + ", is not found.");
            }
            else if (!this._predecessor.Valid)
            {   
                err.Add("The Vehicle's Predecessor at Code, " + this._description.PredecessorCode + ", is not valid.");
            }
            else if (this._predecessor == this || this._predecessor._description.Code == this._description.Code)
            {   
                err.Add("The Vehicle lists itself as the predecessor.  The predecessor will be cleared.");
                this._predecessor = null;
                this._description.PredecessorCode = 0;
            }
            else if (this._validYears[0].Index < this._predecessor._validYears[0].Index)
            {   
                err.Add("The Vehicle's first valid Model Year, " + this._validYears[0].Year + ", is earlier than " +
                    "the Predecessor's, Code=" + this._description.PredecessorCode + ", first valid Model Year, " +
                    this._predecessor._validYears[0].Year + ".");
            }
            if (err.Count > 0)
            {
                this._loadWarnings = new string[err.Count];
                err.CopyTo(this._loadWarnings, 0);
            }
        }
        internal void Validate()
        {
            this._valid = true;        
            StringCollection err = new StringCollection();
            if (this._description.Origin.Equals('\0'))
            {
                double domesticContent = this._description.USContent + this._description.CanadianContent +
                    this._description.MexicanContent;
                if (domesticContent <= 1) { this._description.Origin = (domesticContent >= 0.75) ? 'D' : 'I'; }
            }
            if (!this._initialized)
            {
                this._valid = false;
                err.Add("The Vehicle has not been initialized.");
            }
            else if (
                (this._description.Code < 1) ||
                (this._description.Model.Equals(string.Empty)) ||
                (this._description.Nameplate.Equals(string.Empty)) ||
                (this._description.FuelEconomy < 1) ||
                (this._description.EngineCode < 1) ||
                (this._description.TransmissionCode < 1) ||
                (this._description.Origin.Equals('\0')) ||
                (this._description.Style.Equals(string.Empty)) ||
                (this._description.BaseWeight < 1) ||
                (this._description.TechnologyClass == TechnologyClass.None))
            {    
                this._valid = false;
                if (this._description.Code < 1) { err.Add("The Vehicle Code," + this._description.Code + ", is not valid."); }
                if (this._description.Model == "") { err.Add("The Vehicle's Model is missing."); }
                if (this._description.Nameplate == "") { err.Add("The Vehicle's Nameplate is missing."); }
                if (this._description.FuelEconomy < 1) { err.Add("The Vehicle's Fuel Economy, " +
                    this._description.FuelEconomy + ", is missing or not valid."); }
                if (this._description.EngineCode < 1) { err.Add("The Vehicle's Engine-Code, " +
                    this._description.EngineCode + ", is missing or not valid."); }
                if (this._description.TransmissionCode < 1) { err.Add("The Vehicle's Transmission-Code, " +
                    this._description.TransmissionCode + ", is missing or not valid."); }
                if (this._description.Origin == '\0') { err.Add("The Vehicle's Origin is missing."); }
                if (this._description.Style == "") { err.Add("The Vehicle's Style is missing."); }
                if (this._description.Class == "") { err.Add("The Vehicle's Class is missing."); }
                if (this._description.BaseWeight < 1) { err.Add("The Vehicle's Curb Weight and Test Weight are missing or not valid."); }
                if (this._description.TechnologyClass == TechnologyClass.None) { err.Add("The Vehicle's Technology Class assignment is missing or not valid."); }
            }
            else if (this._description.Wheelbase < 1 || this._description.FrontTrackWidth < 1 ||
                this._description.RearTrackWidth < 1)
            {   
                this._valid = false;
                if (this._description.Wheelbase       < 1) { err.Add("The Vehicle's Wheelbase, " + this._description.Wheelbase + ", is missing or not valid."); }
                if (this._description.FrontTrackWidth < 1) { err.Add("The Vehicle's Front Track Width, " + this._description.FrontTrackWidth + ", is missing or not valid."); }
                if (this._description.RearTrackWidth  < 1) { err.Add("The Vehicle's Rear Track Width, " + this._description.RearTrackWidth + ", is missing or not valid."); }
            }
            if (this._valid && this._description.Footprint < 1)
            {   
                this._description.Footprint = (this._description.Wheelbase *
                    (this._description.FrontTrackWidth + this._description.RearTrackWidth)) / 288;
            }
            if (this._valid)
            {   
                int yearCount = this._description.Sales.Length;
                int foundSMP = 0;
                this._validYears = new ModelYear[yearCount];
                for (int i = 0; i < yearCount; i++)
                {
                    if (this._description.Sales[i] > 0 && this._description.Msrp[i] > 0 && this._description.Price[i] > 0)
                    {   
                        this._validYears[foundSMP++] = new ModelYear(i + ModelYear.MinYear);
                    }
                }
                if (foundSMP > 0)
                {   
                    ModelYear[] validYears = new ModelYear[foundSMP];
                    for (int i = 0; i < foundSMP; i++)
                    {
                        validYears[i] = this._validYears[i];
                    }
                    this._validYears = validYears;
                }
                else
                {   
                    this._valid = false;
                    this._validYears = null;
                }
                if (!this._valid)
                {   
                    err.Add("The Vehicle does not have any Sales, MSRP, and Price values for the same model year.");
                }
            }
            if (this._valid)
            {    
                this._valid = (this._engine != null && this._engine.Valid && this._transmission != null &&
                    this._transmission.Valid);
                if (!this._valid)
                {   
                    if      ( this._engine == null      ) { err.Add("The Vehicle's Engine is missing.");         }
                    else if (!this._engine.Valid        ) { err.Add("The Vehicle's Engine is not valid.");       }
                    if      ( this._transmission == null) { err.Add("The Vehicle's Transmission is missing.");   }
                    else if (!this._transmission.Valid  ) { err.Add("The Vehicle's Transmission is not valid."); }
                }
            }
            if (this._valid)
            {   
                this.AssignTypeAndMobile6Class();
            }
            else
            {
                this._invalidReasons = new string[err.Count];
                err.CopyTo(this._invalidReasons, 0);
            }
        }
        private void AssignTypeAndMobile6Class()
        {
            string style  = this._description.Style;
            double gvwr   = this._description.GVWR;
            double weight = this._description.CurbWeight;
            string fuel   = this._engine.Description.Fuel;
            if (Global.StringCompareAny(style, new string[] {"WAGON", "SEDAN", "CONVERTIBLE", "COUPE", "HATCHBACK"}, false))
            {   
                this._vehicleType = (this._description.Origin == 'D') ? VehicleType.DomesticAuto :
                    VehicleType.ImportedAuto;
                this._mobile6Class = (fuel == "D") ? Mobile6Class.Lddv : Mobile6Class.Ldgv;
            }
            else if ((gvwr > 8500 || weight > 6000))    
            {   
                this._vehicleType = VehicleType.HeavyLT;
                this._mobile6Class = (fuel == "D") ? Mobile6Class.Hddv2b : Mobile6Class.Hdgv2b;
            }
            else
            {   
                this._vehicleType = VehicleType.LightTruck;
                this._mobile6Class = (fuel == "D") ?
                    ((gvwr <= 6000) ? Mobile6Class.Lddt1 : Mobile6Class.Lddt2) :
                    ((gvwr <= 6000) ? Mobile6Class.Ldgt1 : Mobile6Class.Ldgt2);
            }
        }
        internal void SetEngine(Engine engine)
        {
            this._engine = engine;
            this._description.EngineCode = engine.Description.Code;
        }
        internal void SetTransmission(Transmission transmission)
        {
            this._transmission = transmission;
            this._description.TransmissionCode = transmission.Description.Code;
        }
        internal void SetNamePlate(NamePlate namePlate)
        {
            this._namePlate = namePlate;
        }
        internal void SetModelingData(VehicleModelingData value)
        {
            this._modelingData = value;
        }
        public void SetEngineAndTransmissionFromPredecessor()
        {
            if (this._predecessor != null && this._predecessor.Valid)
            {   
                Engine       predEng = this._predecessor._engine;
                Transmission predTrn = this._predecessor._transmission;
                if (predEng.Parent != null && predEng.Parent.Description.Code == this._engine.Description.Code)
                {   
                    predEng.Vehicles.Add(this);
                    predEng.ModelYearVehicles.Add(this);
                    this._engine.Vehicles.Remove(this);
                    this._engine.ModelYearVehicles.Remove(this);
                    this.SetEngine(predEng);
                }
                if (predTrn.Parent != null && predTrn.Parent.Description.Code == this._transmission.Description.Code)
                {   
                    predTrn.Vehicles.Add(this);
                    predTrn.ModelYearVehicles.Add(this);
                    this._transmission.Vehicles.Remove(this);
                    this._transmission.ModelYearVehicles.Remove(this);
                    this.SetTransmission(predTrn);
                }
            }
        }
        public bool IsValid(ModelYear year)
        {
            return this.IsValid(year.Index);
        }
        public bool IsValid(int yrIndex)
        {
            if (!this._valid || yrIndex < 0 || yrIndex > this._description.Sales.Length)
            {   
                return false;
            }
            else
            {
                return (this._description.Sales[yrIndex] > 0) && (this._description.Msrp[yrIndex] > 0) &&
                    (this._description.Price[yrIndex] > 0);
            }
        }
        public void UpdateTypeAndMobile6Class()
        {
            if (!this._valid || this._mobile6Class == Mobile6Class.None)
            {
                throw new InvalidOperationException("The vehicle being updated is not valid or does not have a " +
                    "valid Mobile6 Class assignment.  Updating the Mobile6 Class or Vehicle Type of invalid " +
                    "vehicles is not allowed.");
            }
            this.AssignTypeAndMobile6Class();
        }
        internal void SetTypeAndMobile6Class(VehicleType type, Mobile6Class m6Class)
        {
            this._vehicleType  = type;
            this._mobile6Class = m6Class;
        }
        public int[] GetRedesignYears()
        {
            return this.GetRedesignYears(true);
        }
        private int[] GetRedesignYears(bool checkPred)
        {
            if (checkPred && this._predecessor != null)
            {   
                return this._predecessor.GetRedesignYears(true);
            }
            else
            {   
                int[] redesignYears = this._description.RedesignYears;
                for (int i = 0, successorCount = this._successors.Count; i < successorCount; i++)
                {   
                    this.MergeRedesignRefreshYears(ref redesignYears, this._successors[i].GetRedesignYears(false));
                }
                return redesignYears;
            }
        }
        public int[] GetRefreshYears()
        {
            return this.GetRefreshYears(true);
        }
        private int[] GetRefreshYears(bool checkPred)
        {
            if (checkPred && this._predecessor != null)
            {   
                return this._predecessor.GetRefreshYears(true);
            }
            else
            {   
                int[] refreshYears = this._description.RefreshYears;
                for (int i = 0, successorCount = this._successors.Count; i < successorCount; i++)
                {   
                    this.MergeRedesignRefreshYears(ref refreshYears, this._successors[i].GetRefreshYears(false));
                }
                return refreshYears;
            }
        }
        private void MergeRedesignRefreshYears(ref int[] value1, int[] value2)
        {
            int v1Len = value1.Length, v2Len = value2.Length;
            int[] value = new int[v1Len + v2Len];
            int vLen = v1Len;
            Array.Copy(value1, value, vLen);
            for (int i = 0; i < v2Len; i++)
            {
                bool duplicate = false;
                for (int j = 0; j < vLen; j++)
                {   
                    if (value[j] == value2[i]) { duplicate = true; break; }
                }
                if (!duplicate) { value[vLen++] = value2[i]; }
            }
            value1 = new int[vLen];
            Array.Copy(value, value1, vLen);
            Array.Sort(value1);
        }
        public void GetRedesignState(ModelYear modelYear, out bool atRedesign, out bool inShadowOfRedesign)
        {
            atRedesign = inShadowOfRedesign = false;    
            int year = modelYear.Year;
            int[] redesignYears = this.GetRedesignYears();
            for (int i = 0, count = redesignYears.Length; i < count; i++)
            {
                if (redesignYears[i] == year) { atRedesign = true; break; }
            }
            if (!atRedesign)
            {
                for (int i = 0, count = redesignYears.Length; i < count; i++)
                {
                    int delta = Math.Abs(redesignYears[i] - year);    
                    if (0 < delta && delta < 5) { inShadowOfRedesign = true; break; }
                }
            }
        }
        public void GetRefreshState(ModelYear modelYear, out bool atRefresh, out bool inShadowOfRefresh)
        {
            atRefresh = inShadowOfRefresh = false;
            int year = modelYear.Year;
            int[] refreshYears = this.GetRefreshYears();
            for (int i = 0, count = refreshYears.Length; i < count; i++)
            {
                if (refreshYears[i] == year) { atRefresh = true; break; }
            }
            if (!atRefresh)
            {
                for (int i = 0, count = refreshYears.Length; i < count; i++)
                {
                    int delta = Math.Abs(refreshYears[i] - year);    
                    if (0 < delta && delta < 2) { inShadowOfRefresh = true; break; }
                }
            }
        }
        #endregion
        #region 
        public bool Initialized
        {
            get { return this._initialized; }
        }
        public bool Valid
        {
            get { return this._valid; }
        }
        public string[] InvalidReasons
        {
            get { return this._invalidReasons; }
        }
        public string[] LoadWarnings
        {
            get { return this._loadWarnings; }
        }
        public ModelYear[] ValidYears
        {
            get { return this._validYears; }
        }
        public ModelYear MinYear
        {
            get { return (this._validYears == null) ? null : this._validYears[0]; }
        }
        public ModelYear MaxYear
        {
            get { return (this._validYears == null) ? null : this._validYears[this._validYears.Length - 1]; }
        }
        public VehicleDescription Description
        {
            get { return this._description; }
        }
        public VehicleModelingData ModelingData
        {
            get { return this._modelingData; }
        }
        public VehicleType VehicleType
        {
            get { return this._vehicleType; }
        }
        public Engine Engine
        {
            get { return this._engine; }
        }
        public Transmission Transmission
        {
            get { return this._transmission; }
        }
        public NamePlate NamePlate
        {
            get { return this._namePlate; }
        }
        public Manufacturer Manufacturer
        {
            get { return this._manufacturer; }
        }
        public Vehicle Predecessor
        {
            get { return this._predecessor; }
        }
        public VehicleCollection Successors
        {
            get { return this._successors; }
        }
        public bool IsRcAuto
        {
            get
            {
                return (this._regClass == RegulatoryClass.DomesticAuto ||
                    this._regClass == RegulatoryClass.ImportedAuto);
            }
        }
        public bool IsRcTruck
        {
            get { return (this._regClass == RegulatoryClass.LightTruck); }
        }
        public RegulatoryClass RegClass
        {
            get { return this._regClass; }
            set { this._regClass = value; }
        }
        public Mobile6Class Mobile6Class
        {
            get { return this._mobile6Class; }
        }
        public TechnologyClass TechnologyClass
        {
            get { return this._description.TechnologyClass; }
        }
        #endregion
        #region 
        private bool _initialized;
        private bool _valid;
        private string[] _invalidReasons;
        private string[] _loadWarnings;
        private ModelYear[] _validYears;
        private VehicleDescription _description;
        private VehicleModelingData _modelingData;
        private VehicleType _vehicleType;
        private Engine _engine;
        private Transmission _transmission;
        private NamePlate _namePlate;
        private Manufacturer _manufacturer;
        private Vehicle _predecessor;
        private VehicleCollection _successors;
        private RegulatoryClass _regClass;
        private Mobile6Class _mobile6Class;
        #endregion
    }
}

