using System;
using System.Collections.Specialized;
using Volpe.Utils;
namespace Volpe.Cafe.IO
{
    [Serializable]
    public class OutputBuffer
    {
        #region 
        public OutputBuffer(int index, XlSize size)
        {
            this._index = index;
            this._size = size;
            this.Clear();
        }
        #endregion
        #region 
        public void Clear()
        {
            this._buffer         = new object[this._size.Rows, this._size.Columns];
            this._name           = null;
            this._title          = null;
            this._header         = new XlCell(0, 0);
            this._freezePanes    = new XlCell(0, 0);
            this._numPageBreaks  = 0;
            this._pageBreaks     = new int[8];
            this._numSortKeys    = 0;
            this._sortKeys       = new int[] {-1, -1, -1};
            this._sortDescending = new bool[3];
            this._charts         = new XlChartFormat[4];
            this._chartCount     = 0;
            this._excelNamesCount       = 0;
            this._excelNames            = new StringCollection();
            this._excelNamesExpressions = new StringCollection();
        }
        public void ResizeBuffer(XlSize newSize, bool preserve)
        {
            object[,] newBuffer = new object[newSize.Rows, newSize.Columns];
            if (preserve)
            {   
                int rows = newSize.Rows;
                int cols = newSize.Columns;
                if (rows > this._size.Rows)    rows = this._size.Rows;
                if (cols > this._size.Columns) cols = this._size.Columns;
                for (int i = 0; i < rows; i++)
                {
                    for (int j = 0; j < cols; j++)
                    {
                        newBuffer[i, j] = this._buffer[i, j];
                    }
                }
            }
            this._buffer = newBuffer;
            this._size = newSize;
        }
        public void AddPageBreak(int row)
        {
            if (this._numPageBreaks == this._pageBreaks.Length)
            {   
                int[] newPageBreaks = new int[this._numPageBreaks * 2];
                for (int i = 0; i < this._numPageBreaks; i++)
                {
                    newPageBreaks[i] = this._pageBreaks[i];
                }
                this._pageBreaks = newPageBreaks;
            }
            this._pageBreaks[this._numPageBreaks++] = row;
        }
        public void SetFirstSortKey(int sortKey, bool sortDescending)
        {
            this.SetSortKey(0, sortKey, sortDescending);
        }
        public void SetSecondSortKey(int sortKey, bool sortDescending)
        {
            this.SetSortKey(1, sortKey, sortDescending);
        }
        public void SetThirdSortKey(int sortKey, bool sortDescending)
        {
            this.SetSortKey(2, sortKey, sortDescending);
        }
        private void SetSortKey(int index, int sortKey, bool sortDescending)
        {
            if (sortKey < 0 || sortKey >= this.Size.Columns)
            {
                throw new ArgumentOutOfRangeException("sortKey",
                    "The sortKey column must be between 0 and Size.Columns.");
            }
            this._sortKeys[index] = sortKey;
            this._sortDescending[index] = sortDescending;
            this._numSortKeys = 0;
            for (int i = 0; i < 3; i++)
            {
                if (this._sortKeys[i] != -1)
                {
                    this._numSortKeys++;
                }
            }
        }
        public void AddChartInfo(XlChartType type, string name, string title, XlColor plotColor,
            XlSeriesFormat[] seriesFormat, XlAxisFormat categoryAxisFormat, XlAxisFormat valuesAxisFormat)
        {
            this.AddChartInfo(new XlChartFormat(type, name, title, plotColor, seriesFormat, categoryAxisFormat,
                valuesAxisFormat));
        }
        public void AddChartInfo(XlChartFormat format)
        {
            if (this._chartCount == this._charts.Length)
            {    
                XlChartFormat[] cf = new XlChartFormat[this._chartCount * 2];
                Array.Copy(this._charts, cf, this._chartCount);
                this._charts = cf;
            }
            this._charts[this._chartCount++] = format;
        }
        public void AddExcelName(string name, string expression)
        {
            this._excelNames.Add(name);
            this._excelNamesExpressions.Add(expression);
            this._excelNamesCount++;
        }
        #endregion
        #region 
        public object this[int row, int column]
        {
            get
            {
                if (row    < 0 || row     >= this._size.Rows ||
                    column < 0 || column >= this._size.Columns)
                {
                    throw new ArgumentOutOfRangeException("row, col",
                        "The specified row or column must be between 0 and Size.Rows or Size.Columns.");
                }
                return this._buffer[row, column];
            }
            set
            {
                if (row    < 0 || row     >= this._size.Rows ||
                    column < 0 || column >= this._size.Columns)
                {
                    throw new ArgumentOutOfRangeException("row, col",
                        "The specified row or column must be between 0 and Size.Rows or Size.Columns.");
                }
                this._buffer[row, column] = value;
            }
        }
        public object[,] Buffer
        {
            get { return this._buffer; }
        }
        public XlSize Size
        {
            get { return this._size; }
        }
        public int Index
        {
            get { return this._index; }
        }
        public string Name
        {
            get { return this._name; }
            set
            {
                if (value == null || value.Trim() == "")
                {
                    throw new ArgumentNullException("value", "The buffer name cannot be null or empty.");
                }
                this._name = value.Trim();
            }
        }
        public string Title
        {
            get { return this._title; }
            set { this._title = value.Trim(); }
        }
        public XlCell Header
        {
            get { return this._header; }
            set
            {
                if (value.Row    < 0 || value.Row >= this.Size.Rows ||
                    value.Column < 0 || value.Column >= this.Size.Columns)
                {
                    throw new ArgumentOutOfRangeException("value",
                        "The header cell must be between (0, 0) and (Size.Rows, Size.Columns).");
                }
                this._header = value;
            }
        }
        public XlCell FreezePanes
        {
            get { return this._freezePanes; }
            set
            {
                if (value.Row    < 0 || value.Row >= this.Size.Rows ||
                    value.Column < 0 || value.Column >= this.Size.Columns)
                {
                    throw new ArgumentOutOfRangeException("value",
                        "The freeze-panes cell must be between (0, 0) and (Size.Rows, Size.Columns).");
                }
                this._freezePanes = value;
            }
        }
        public int[] PageBreaks
        {
            get
            {
                int[] pageBreaks = new int[this._numPageBreaks];
                for (int i = 0; i < this._numPageBreaks; i++)
                {
                    pageBreaks[i] = this._pageBreaks[i];
                }
                return pageBreaks;
            }
        }
        public int[] SortKeys
        {
            get
            {
                int[] sortKeys = new int[this._numSortKeys];
                for (int i = 0, j = 0; i < 3; i++)
                {
                    if (this._sortKeys[i] != -1)
                    {
                        sortKeys[j++] = this._sortKeys[i];
                    }
                }
                return sortKeys;
            }
        }
        public bool[] SortDescending
        {
            get
            {
                bool[] sortDescending = new bool[this._numSortKeys];
                for (int i = 0, j = 0; i < 3; i++)
                {
                    if (this._sortKeys[i] != -1)
                    {
                        sortDescending[j++] = this._sortDescending[i];
                    }
                }
                return sortDescending;
            }
        }
        public XlChartFormat[] ChartFormats
        {
            get
            {
                XlChartFormat[] cf = new XlChartFormat[this._chartCount];
                Array.Copy(this._charts, cf, this._chartCount);
                return cf;
            }
        }
        public bool HasCharts
        {
            get { return (this._chartCount > 0); }
        }
        public int ExcelNameCount { get { return this._excelNamesCount; } }
        public StringCollection ExcelNames { get { return this._excelNames; } }
        public StringCollection ExcelNamesExpressions { get { return this._excelNamesExpressions; } }
        #endregion
        #region 
        private object[,] _buffer;
        private XlSize _size;
        private int _index;
        private string _name;
        private string _title;
        private XlCell _header;
        private XlCell _freezePanes;
        private int _numPageBreaks;
        private int[] _pageBreaks;
        private int _numSortKeys;
        private int[] _sortKeys;
        private bool[] _sortDescending;
        private XlChartFormat[] _charts;
        private int _chartCount;
        private int _excelNamesCount;
        private StringCollection _excelNames;
        private StringCollection _excelNamesExpressions;
        #endregion
    }
}

