#region << Using directives >>
using System;
using System.Collections;
using System.Collections.Specialized;
using System.IO;
using System.Text;
using System.Threading;
using Volpe.Cafe;
using Volpe.Cafe.Collections;
using Volpe.Cafe.Data;
using Volpe.Cafe.Data.MonteCarlo;
using Volpe.Cafe.IO;
using Volpe.Ui;
#endregion
namespace Volpe.Cafe.Model.MonteCarlo
{
    [Serializable]
    [ModelDescription("Monte-Carlo Model", "Performs sensitivity analysis by slightly varying the initial input parameters and " +
         "re-running the standard compliance model multiple times.", 1.0F)]
    public sealed class MonteCarlo : ComplianceBase
    {
        #region 
        public MonteCarlo()
            : base()
        {
        }
        ~MonteCarlo()
        {
        }
        #endregion
        #region 
        #region 
        protected override void OnModelingStopped()
        {
            base.OnModelingStopped();
            this.AfterModelingStoppedOrCompleted();
        }
        protected override void OnModelingCompleted()
        {
            base.OnModelingCompleted();
            this.AfterModelingStoppedOrCompleted();
        }
        void AfterModelingStoppedOrCompleted()
        {
            if (this._valid && this._completedTrialCount > 0)
            {
                PromptEventArgs e = new PromptEventArgs("Generate Logs?",
                    "Would you like to generate Monte-Carlo log files for the " + this._completedTrialCount +
                    " completed trials?", PromptOption.YesNo);
                this.OnPrompt(e);
                if (e.PromptResult == PromptResult.Yes)
                {   
                    this.GenerateLogFiles();
                }
            }
            if (this._currentCompliance != null)
            {
                this._currentCompliance.Prompt -= new PromptEventHandler(this.Compliance_Prompt);
            }
        }
        #endregion
        #region 
        public override ICompliance CreateNew()
        {
            return new MonteCarlo();
        }
        protected override void StartInternal()
        {
            MonteCarloSettings mcSettings = this._settings.MonteCarloSettings;
            this._discRateCount = this._settings.Parameters.MonteCarlo.DiscountRates.Length;
            this._useTrialsFile = mcSettings.UseTrialsFile && File.Exists(mcSettings.TrialsFile);
            if (this._useTrialsFile)
            {
                this._trialsFile = mcSettings.TrialsFile;
                this._trialCount = this.GetTrialCountFromFile(this._trialsFile, out this._firstTrialIndex,
                    out this._lastTrialIndex);
                if (this._trialCount < this._discRateCount)
                {
                    this.OnPrompt(new PromptEventArgs("Monte-Carlo Error", "Error parsing the Monte-Carlo Trials file.  " +
                        "Possibly the file does not contain valid Monte-Carlo trial data.", PromptOption.Ok));
                    return;
                }
            }
            else
            {
                int trialPairs = mcSettings.TrialPairs;
                this._trialCount = trialPairs * this._discRateCount;
                this._firstTrialIndex = this._lastTrialIndex = 0;
            }
            this._initializeTrialData = true;
            this._completedTrialCount = 0;
            if (!this._valid) { this._valid = false; return; }
            this.Begin();
        }
        protected override void AbortInternal(bool abortWhenReady)
        {
            if (this._currentCompliance != null) { this._currentCompliance.Abort(abortWhenReady); }
            base.AbortInternal(abortWhenReady);
        }
        public override Industry GetData(int scenIndex, int yearIndex)
        {
            return null;
        }
        #endregion
        void Begin()
        {
            if (this._initializeTrialData) { this.InitializeTrialData(); }
            this._currentTrialIndices = new ArrayList(1);
            this._currentTrialIndices.Add(0);
            this.RunTrialLoop();
        }
        private void InitializeTrialData()
        {
            TrialInfo[] trials;
            if (this._useTrialsFile)
            {   
                StreamReader sr = new StreamReader(this._trialsFile);
                sr.ReadLine();      
                trials = new TrialInfo[this._trialCount];
                for (int i = 0; i < this._trialCount; i++)
                {
                    trials[i].FromCsvString(sr.ReadLine());
                }
                if (sr != null) { sr.Close(); }
            }
            else
            {   
                TrialGenerator generator = new TrialGenerator(this._settings);
                trials = generator.GenerateTrials(this._trialCount);
            }
            this._trialData = new TrialData[this._scenCount][];
            for (int i = 0; i < this._scenCount; i++)
            {
                this._trialData[i] = new TrialData[this._trialCount];
                for (int j = 0; j < this._trialCount; j++)
                {
                    this._trialData[i][j] = new TrialData(trials[j]);
                }
            }
            this._initializeTrialData = false;
        }
        void RunTrialLoop()
        {
            for (int i = 0; i < this._trialCount; i++)
            {   
                TrialData trialData = this._trialData[0][i];
                if (!trialData.HasData)
                {   
                    this.BeginTrial(i, trialData.TrialInfo);
                }
                if (this._abortRequested) { break; }
            } 
        }
        void BeginTrial(int trialDataIndex, TrialInfo trial)
        {
            Industry         data     = this._data.Clone();
            ModelingSettings settings = this._settings.Clone();
            trial.InitializeTrial(data, settings);
            ICompliance compliance = new Compliance();
            this._currentCompliance = compliance;
            this._currentTrialIndices[0] = trial.Index;
            this.UpdateLogs(trial.Index);   
            compliance.Prompt += new PromptEventHandler(this.Compliance_Prompt);    
            compliance.Start(data, settings, LogWriter.Empty);
            while (compliance.Running) { Thread.Sleep(1000); }                      
            compliance.Prompt -= new PromptEventHandler(this.Compliance_Prompt);    
            if (this._abortRequested) { return; }
            this.SaveTrialData(trialDataIndex, settings);
            Interlocked.Increment(ref this._completedTrialCount);
        }
        void UpdateLogs(int trialIndex)
        {
            string value = CharDash25 + " Examining MC Trial: " + (trialIndex + 1) + " / " +
                (this._trialCount + this._lastTrialIndex) + " " + CharDash25;
            LogWriter writer = this.LogWriter;
            for (int i = 0; i < this._scenCount; i++)
            {
                for (int j = 0; j < this._yearCount; j++)
                {
                    ModelYear year = new ModelYear(j + this._minYear);
                    for (int k = 0; k < this._mfrCount; k++)
                    {
                        writer.WriteComplianceLine   (this._settings.Scenarios[i], year, this._data.Manufacturers[k], value);
                        writer.WriteExtComplianceLine(this._settings.Scenarios[i], year, this._data.Manufacturers[k], value);
                    }
                } 
            } 
            writer.WriteErrorLine  (value);
            writer.WriteWarningLine(value);
            writer.WriteSummaryLine(value);
        }
        void SaveTrialData(int trialDataIndex, ModelingSettings settings)
        {
            LogWriter writer = this.LogWriter;
            for (int i = 0; i < this._scenCount; i++)
            {   
                Industry[] modelData = new Industry[this._maxYearIndex + 1];
                for (int j = this._minYearIndex; j <= this._maxYearIndex; j++)
                {
                    modelData[j] = this._currentCompliance.GetData(i, j);
                }
                this._trialData[i][trialDataIndex].SaveTrialData(settings.Scenarios[i], this._minYearIndex, this._yearCount,
                    this._trialData[0][trialDataIndex], writer, modelData);
            }
        }
        void GenerateLogFiles()
        {
            VehicleCollection vehs = this._data.GetVehicles();
            Estimates costEstimates = this._settings.OperatingModes.TechnologyCostEstimates;
            Estimates fuelEstimates = this._settings.OperatingModes.TechnologyFuelEstimates;
            ASCIIEncoding ascii = new ASCIIEncoding();
            string outPath = (this.LogWriter == null) ? this._settings.OutputSettings.OutputPath :
                this.LogWriter.LogPath + "\\..\\MC-logs";
            if (!Directory.Exists(outPath)) { Directory.CreateDirectory(outPath); }
            for (int i = 0; i < this._scenCount; i++)
            {
                FileMode mode = FileMode.Create; FileAccess access = FileAccess.Write; FileShare share = FileShare.Read;
                FileStream fsTrials = null, fsCosts  = null, fsFC = null;
                FileStream fsData   = new FileStream(outPath + "\\MC_Sn" + i + "_data.csv", mode, access, share);
                try
                {   
                    string csv;
                    byte[] buffer;
                    if (i == 0)
                    {   
                        fsTrials = new FileStream(outPath + "\\MC_trials.csv", mode, access, share);
                        csv      = TrialInfo.CreateCsvHeader() + "\r\n";
                        buffer   = ascii.GetBytes(csv);
                        fsTrials.Write(buffer, 0, buffer.Length);
                        fsCosts = new FileStream(outPath + "\\MC_tech_costs.csv", mode, access, share);
                        buffer  = ascii.GetBytes("Index," + TechnologyIndexes.TechAbbrCSV + "\r\n");
                        fsCosts.Write(buffer, 0, buffer.Length);
                        fsFC   = new FileStream(outPath + "\\MC_tech_fcs.csv", mode, access, share);
                        buffer = ascii.GetBytes("Index," + TechnologyIndexes.TechAbbrCSV + "\r\n");
                        fsFC.Write(buffer, 0, buffer.Length);
                    }
                    csv    = TrialData.CreateCsvHeader(this._data, false) + "\r\n";
                    buffer = ascii.GetBytes(csv);
                    fsData.Write(buffer, 0, buffer.Length);
                    for (int j = 0; j < this._trialCount; j++)
                    {
                        if (this._trialData[i][j].HasData)
                        {
                            if (i == 0)
                            {   
                                csv = this._trialData[i][j].TrialInfo.ToCsvString() + "\r\n";
                                buffer = ascii.GetBytes(csv);
                                fsTrials.Write(buffer, 0, buffer.Length);
                                if (j % this._discRateCount == 0)
                                {   
                                    double[] costs = this.GetTechCosts(this._trialData[i][j].TrialInfo.TechCostScaleFactor,
                                        costEstimates, vehs);
                                    csv = this._trialData[i][j].TrialInfo.Index.ToString() + ",";
                                    for (int k = 0; k < TechnologyIndexes.TechnologyCount; k++)
                                    {
                                        csv += (costs[k].ToString() + ",");
                                    }
                                    csv    = csv.Remove(csv.Length - 1, 1) + "\r\n";
                                    buffer = ascii.GetBytes(csv);
                                    fsCosts.Write(buffer, 0, buffer.Length);
                                    double[] fc = this.GetTechFC(this._trialData[i][j].TrialInfo.TechFcScaleFactor,
                                        fuelEstimates, vehs);
                                    csv = this._trialData[i][j].TrialInfo.Index.ToString() + ",";
                                    for (int k = 0; k < TechnologyIndexes.TechnologyCount; k++)
                                    {
                                        csv += (fc[k].ToString() + ",");
                                    }
                                    csv    = csv.Remove(csv.Length - 1, 1) + "\r\n";
                                    buffer = ascii.GetBytes(csv);
                                    fsFC.Write(buffer, 0, buffer.Length);
                                }
                            }
                            csv    = this._trialData[i][j].ToCsvString(false) + "\r\n";
                            buffer = ascii.GetBytes(csv);
                            fsData.Write(buffer, 0, buffer.Length);
                        }
                    }
                }
                catch (Exception ex)
                {
                    Console.WriteLine(ex);
                }
                finally
                {   
                    if (fsTrials != null) { fsTrials.Close(); }
                    if (fsCosts  != null) { fsCosts .Close(); }
                    if (fsFC     != null) { fsFC    .Close(); }
                    if (fsData   != null) { fsData  .Close(); }
                }
            }
        }
        double[] GetTechCosts(double[] scales, Estimates estimates, VehicleCollection vehs)
        {
            double[] costs = new double[TechnologyIndexes.TechnologyCount];
            for (int i = 0; i < TechnologyIndexes.TechnologyCount; i++)
            {
                double ttlSales = 0;    
                double ttlCost  = 0;    
                for (int j = 0; j < vehs.Count; j++)
                {
                    Vehicle veh = vehs[j];
                    TechnologyAttributes ta = this._settings.Technologies[i].Attributes[veh.TechnologyClass];
                    if (ta.Applicable && (ta.CostLow != 0 || ta.CostHigh != 0 || ta.FcLow != 0 || ta.FcHigh != 0))
                    {
                        double techClassCost = (estimates == Estimates.Low) ? ta.CostLow : (estimates == Estimates.High) ?
                            ta.CostHigh : (ta.CostHigh + ta.CostLow) * 0.5;
                        double[] sales = veh.Description.Sales;
                        double vehTtlSales = 0;     
                        for (int k = this._minYearIndex; k <= this._maxYearIndex; k++)
                        {
                            ttlSales    += sales[k];
                            vehTtlSales += sales[k];
                        } 
                        ttlCost += (vehTtlSales * techClassCost);
                    }
                } 
                costs[i] = ttlCost / ttlSales * scales[i];  
            } 
            return costs;
        }
        double[] GetTechFC(double[] scales, Estimates estimates, VehicleCollection vehs)
        {
            double[] fc = new double[TechnologyIndexes.TechnologyCount];
            for (int i = 0; i < TechnologyIndexes.TechnologyCount; i++)
            {
                double ttlSales = 0;    
                double ttlFC    = 0;    
                for (int j = 0; j < vehs.Count; j++)
                {
                    Vehicle veh = vehs[j];
                    TechnologyAttributes ta = this._settings.Technologies[i].Attributes[veh.TechnologyClass];
                    if (ta.Applicable && (ta.CostLow != 0 || ta.CostHigh != 0 || ta.FcLow != 0 || ta.FcHigh != 0))
                    {
                        double techClassFc = (estimates == Estimates.Low) ? ta.FcLow : (estimates == Estimates.High) ?
                            ta.FcHigh : (ta.FcHigh + ta.FcLow) * 0.5;
                        double[] sales = veh.Description.Sales;
                        double vehTtlSales = 0;     
                        for (int k = this._minYearIndex; k <= this._maxYearIndex; k++)
                        {
                            ttlSales    += sales[k];
                            vehTtlSales += sales[k];
                        } 
                        ttlFC += (vehTtlSales * techClassFc);
                    }
                } 
                fc[i] = ttlFC / ttlSales * scales[i];   
            } 
            return fc;
        }
        void Compliance_Prompt(object sender, PromptEventArgs e)
        {
            this.OnPrompt(e);
        }
        int GetTrialCountFromFile(string path, out int firstTrialIndex, out int lastTrialIndex)
        {
            FileStream fs = null;
            StreamReader sr = null;
            firstTrialIndex = lastTrialIndex = -1;
            try
            {   
                fs = new FileStream(path, FileMode.Open, FileAccess.Read, FileShare.Read);
                sr = new StreamReader(fs);
                sr.ReadLine();    
                firstTrialIndex = int.Parse(sr.ReadLine().Split(',')[0]);
                fs.Seek(-1, SeekOrigin.End);
                char c = (char)fs.ReadByte();
                while (c == '\n' || c == '\r')
                {
                    fs.Seek(-2, SeekOrigin.Current);
                    c = (char)fs.ReadByte();
                }
                int charCount = -1;
                while (c != '\n' && c != '\r')
                {
                    fs.Seek(-2, SeekOrigin.Current);
                    c = (char)fs.ReadByte();
                    if (c == ',') { charCount = -1; } 
                    else { charCount++; }
                }
                byte[] buffer = new byte[charCount];
                fs.Read(buffer, 0, charCount);
                string str = ASCIIEncoding.ASCII.GetString(buffer);
                lastTrialIndex = int.Parse(str);
                return lastTrialIndex - firstTrialIndex + 1;
            }
            catch   { return -1; }
            finally { if (fs != null) { fs.Close(); } if (sr != null) { sr.Close(); } }
        }
        #endregion
        #region 
        #region 
        public override MonteCarloData MonteCarloData
        {
            get { return new MonteCarloData(this._trialData, this._trialCount, this._completedTrialCount); }
        }
        public override IModelingProgress Progress
        {
            get
            {
                string trialsString = "";
                if (this._currentTrialIndices != null)
                {
                    lock (this._currentTrialIndices.SyncRoot)
                    {
                        for (int i = 0, trialCount = this._currentTrialIndices.Count; i < trialCount; i++)
                        {
                            trialsString += ((i == 0) ? "" : ", ") + this._currentTrialIndices[i];
                        }
                    }
                }
                trialsString = "Examing MC Trial(s): " + trialsString + ".\nNumber of trials examined: " +
                    (this._completedTrialCount) + " / " + (this._trialCount + this._firstTrialIndex) + ".";
                double runtime  = this.Runtime / 60000D;
                double avgSpeed = (runtime  == 0 || this._completedTrialCount == 0) ? -1D :
                    (Math.Round(this._completedTrialCount / (double)runtime, 1));
                int timeRemain  = (avgSpeed <= 0) ? -1 : (int)((this._trialCount - this._completedTrialCount) / avgSpeed);
                string avgSpeedString = "Avg. Speed: " + ((avgSpeed == -1) ?
                    "<calculating ...>" : avgSpeed + " t/m") + "  Approximate Time Remaining: " + ((timeRemain == -1) ?
                    "<calculating ...>" : (timeRemain < 1) ? "<1 m" : timeRemain + " m.");
                trialsString += "\r\n" + avgSpeedString;
                if (this._currentCompliance == null || !this._currentCompliance.Running) { return null; }
                IModelingProgress progress = this._currentCompliance.Progress;
                if (progress == null) { return null; }
                object additionalInfo = progress.AdditionalInfo;
                additionalInfo = trialsString + ((additionalInfo == null) ? "" : "\r\n  > " + additionalInfo.ToString());
                return new ComplianceProgress(progress.Scenarios, progress.ModelYears, progress.Manufacturers, additionalInfo);
            }
        }
        #endregion
        #endregion
        #region 
        const string CharDash25 = "-------------------------";
        int    _discRateCount;
        bool   _useTrialsFile;
        string _trialsFile;
        int    _trialCount;                                         
        int    _firstTrialIndex;                                    
        int    _lastTrialIndex;                                     
        [NonSerialized] ICompliance _currentCompliance;             
        [NonSerialized] ArrayList   _currentTrialIndices;           
        TrialData[][] _trialData;
        int           _completedTrialCount;                         
        bool          _initializeTrialData;                         
        #endregion
    }
}

