#region << Using Directives >>
using System;
using System.Collections;
using System.Text;
using Volpe.Cafe.Collections;
using Volpe.Cafe.Data;
using Volpe.Cafe.IO;
using TI = Volpe.Cafe.TechnologyIndexes;
using TA = Volpe.Cafe.Model.TechnologyApplicability;
#endregion
namespace Volpe.Cafe.Model
{
    [Serializable]
    public class ComplianceFinding
    {
        #region 
        public ComplianceFinding()
        {
            this._capacity       = 8;
            this._vehs           = new Vehicle   [this._capacity];
            this._ignore         = new int       [this._capacity];
            this._techKeys       = new ulong     [this._capacity];
            this._techLingerKeys = new ulong     [this._capacity];
            this._techImpacts    = new TechImpact[this._capacity];
            this._lingerYear     = new int       [this._capacity];
            this.Reset();
        }
        #endregion
        #region 
        #region 
        public override string ToString()
        {
            bool valid = this.IsValid;
            if (valid)
            {
                ManufacturerModelingData mmd = this._vehs[0].Manufacturer.ModelingData;
                return "ComplianceFinding # " + this._index +
                    ": {\nIsValid: true\nIsEfficient: "  + this.IsEfficient +
                    "\nEfficiency: "  + this._efficiency +
                    "\nTechCost: "    + this._techCost   +
                    "\nRPECost: "     + this._totalDeltaRPE +
                    "\nDeltaFines: "  + this._totalDeltaFines +
                    "\nFuelSavings: " + this._totalFuelSavings +
                    "\nAffectedSales: " + this._totalAffectedSales +
                    "\nCAFE (old, new): {\r\n\t  Domestic Auto: " +
                    mmd.CAFE.DomesticAuto  + ", " + this._newCafe.DomesticAuto  +
                    "\r\n\t  Imported Auto: " + mmd.CAFE.ImportedAuto  + ", " + this._newCafe.ImportedAuto  +
                    "\r\n\t  Light Truck: "   + mmd.CAFE.LightTruck    + ", " + this._newCafe.LightTruck    +
                    "\r\n\t  Unregulated: "   + mmd.CAFE.Unregulated   + ", " + this._newCafe.Unregulated   +
                    "\r\n\t}\nFines (old, new): {\r\n\t  Domestic Auto: " +
                    mmd.Fines.DomesticAuto + ", " + this._newFines.DomesticAuto +
                    "\r\n\t  Imported Auto: " + mmd.Fines.ImportedAuto + ", " + this._newFines.ImportedAuto +
                    "\r\n\t  Light Truck: "   + mmd.Fines.LightTruck   + ", " + this._newFines.LightTruck   +
                    "\r\n\t  Unregulated: "   + mmd.Fines.Unregulated  + ", " + this._newFines.Unregulated  +
                    "\r\n\t}\r\n}";
            }
            else { return "ComplianceFinding # " + this._index + ": {\nValid: false}"; }
        }
        #endregion
        #region 
        public void Initialize(Scenario scen, ModelYear year, int startYear, Industry[] modelYearData, int mfrIndex,
            ModelingSettings settings, LogWriter logWriter, TechnologyType techType, TechnologyCollection techGroup)
        {
            this._scen          = scen;
            this._year          = year;
            this._yrIndex       = year.Index;
            this._yrYear        = year.Year;
            this._startYear     = startYear;
            this._modelYearData = modelYearData;
            this._mfrIndex      = mfrIndex;
            this._settings      = settings;
            this._parameters    = settings.Parameters;
            this._costEstimates = settings.OperatingModes.TechnologyCostEstimates;
            this._fuelEstimates = settings.OperatingModes.TechnologyFuelEstimates;
            this._multiYear     = settings.OperatingModes.MultiYearModeling;
            this._ignorePhaseIn = settings.OperatingModes.IgnorePhaseIn;
            this._backfill      = settings.OperatingModes.Backfill;
            this._clipCost      = settings.OperatingModes.ClipTechCosts;
            this._clipImprv     = settings.OperatingModes.ClipTechImprovements;
            this._splitShared   = settings.OperatingModes.SplitSharedEngsOrTrns;
            this._splitMixed    = settings.OperatingModes.SplitConflictingEngsOrTrns;
            this._skipShared    = settings.OperatingModes.SkipSharedEngsOrTrns;
            this._skipMixed     = settings.OperatingModes.SkipConflictingEngsOrTrns;
            this._logWriter     = logWriter;
            this._techType      = techType;
            this._techGroup     = techGroup;
            this._techCount     = techGroup.Count;
            this.Reset();
        }
        public void Reset()
        {
            this._index             = -1;
            this._vehCount          = 0;
            this._isRcMix           = false;
            this._isEnablesMix      = false;
            this._isSplitCf         = false;
            this._isApplied         = false;
            this._efficiency        = double.NaN;
            this._techCost          = double.NaN;
            this._newCafe           = RCDouble.Zero;
            this._newFines          = RCDouble.Zero;
            this._effectiveNewFines = RCDouble.Zero;
        }
        public void Clear(bool clearArrays)
        {
            this._scen          = null;
            this._year          = null;
            this._yrIndex       = 0;
            this._yrYear        = 0;
            this._startYear     = 0;
            this._modelYearData = null;
            this._mfrIndex      = 0;
            this._settings      = null;
            this._parameters    = null;
            this._costEstimates = Estimates.Average;
            this._fuelEstimates = Estimates.Average;
            this._multiYear     = false;
            this._ignorePhaseIn = false;
            this._backfill      = false;
            this._clipCost      = false;
            this._clipImprv     = false;
            this._splitShared   = false;
            this._splitMixed    = false;
            this._skipShared    = false;
            this._skipMixed     = false;
            this._logWriter     = LogWriter.Empty;
            this._techType      = TechnologyType.Other;
            this._techGroup     = null;
            this._techCount     = 0;
            if (clearArrays)
            {
                for (int i = 0; i < this._capacity; i++)
                {
                    this._vehs          [i] = null;
                    this._ignore        [i] = 0;
                    this._techKeys      [i] = 0;
                    this._techLingerKeys[i] = 0;
                    this._techImpacts   [i].Reset();
                }
            }
            this.Reset();
        }
        public void EnsureCapacity(int capacity)
        {
            if (this._capacity < capacity)
            {
                this._capacity = ((capacity / 8) + 1) * 8;
                this.EnsureCapacity();
            }
        }
        void EnsureCapacity()
        {
            if (this._techKeys.Length < this._capacity)
            {
                this._vehs           = new Vehicle   [this._capacity];
                this._ignore         = new int       [this._capacity];
                this._techKeys       = new ulong     [this._capacity];
                this._techLingerKeys = new ulong     [this._capacity];
                this._techImpacts    = new TechImpact[this._capacity];
                this._lingerYear     = new int       [this._capacity];
            }
        }
        public void CopyFrom(ComplianceFinding cf)
        {
            this._scen          = cf._scen;
            this._year          = cf._year;
            this._yrIndex       = cf._yrIndex;
            this._yrYear        = cf._yrYear;
            this._startYear     = cf._startYear;
            this._modelYearData = cf._modelYearData;    
            this._mfrIndex      = cf._mfrIndex;
            this._settings      = cf._settings;
            this._parameters    = cf._parameters;
            this._costEstimates = cf._costEstimates;
            this._fuelEstimates = cf._fuelEstimates;
            this._multiYear     = cf._multiYear;
            this._ignorePhaseIn = cf._ignorePhaseIn;
            this._backfill      = cf._backfill;
            this._clipCost      = cf._clipCost;
            this._clipImprv     = cf._clipImprv;
            this._splitShared   = cf._splitShared;
            this._splitMixed    = cf._splitMixed;
            this._skipShared    = cf._skipShared;
            this._skipMixed     = cf._skipMixed;
            this._logWriter     = cf._logWriter;
            this._techType      = cf._techType;
            this._techGroup     = cf._techGroup;
            this._techCount     = cf._techCount;
            this._index              = cf._index;
            this._vehCount           = cf._vehCount;
            this._capacity           = cf._capacity;
            this._hasVehs            = cf._hasVehs;
            this._isRcMix            = cf._isRcMix;
            this._isEnablesMix       = cf._isEnablesMix;
            this._isSplitCf          = cf._isSplitCf;
            this._isApplied          = cf._isApplied;
            this._efficiency         = cf._efficiency;
            this._techCost           = cf._techCost;
            this._newCafe            = cf._newCafe;
            this._newFines           = cf._newFines;
            this._effectiveNewFines  = cf._effectiveNewFines;
            this._totalDeltaRPE      = cf._totalDeltaRPE;
            this._totalDeltaFines    = cf._totalDeltaFines;
            this._totalFuelSavings   = cf._totalFuelSavings;
            this._totalAffectedSales = cf._totalAffectedSales;
            this.EnsureCapacity();
            Array.Copy(cf._vehs          , 0, this._vehs          , 0, this._vehCount);
            Array.Copy(cf._ignore        , 0, this._ignore        , 0, this._vehCount);
            Array.Copy(cf._techKeys      , 0, this._techKeys      , 0, this._vehCount);
            Array.Copy(cf._techLingerKeys, 0, this._techLingerKeys, 0, this._vehCount);
            Array.Copy(cf._techImpacts   , 0, this._techImpacts   , 0, this._vehCount);
            Array.Copy(cf._lingerYear    , 0, this._lingerYear    , 0, this._vehCount);
        }
        public void ExamineFinding(int techIdx, VehicleCollection vehs, int vehIdx, ref int cfCount)
        {
            this.Reset();
            this.ParseVehTechsLists(techIdx, vehs, vehIdx);
            this._hasVehs = false;
            for (int i = 0; i < this._vehCount; i++)
            {
                if (this._vehs[i].RegClass != RegulatoryClass.Unregulated) { this._hasVehs = true; break; }
            }
            if (!this._hasVehs) { return; }     
            bool isAero           = (this._techType == TechnologyType.Aerodynamics);
            bool skipShared       = this._isRcMix && this._skipShared;
            bool skipMixedEnables = this._isEnablesMix && (isAero || this._skipMixed);
            if (skipShared || skipMixedEnables) { return; }
            bool isEng = (this._techType == TechnologyType.Engine);
            bool isTrn = (this._techType == TechnologyType.Transmission);
            bool splitShared       = this._isRcMix      && this._splitShared && (isEng || isTrn);
            bool splitMixedEnables = this._isEnablesMix && this._splitMixed  && (isEng || isTrn);
            if (splitShared)
            {   
                this.SplitShared(splitMixedEnables, ref cfCount);
            }
            else if (splitMixedEnables)
            {   
                this.SplitMixedEnables(true, ref cfCount);
            }
            else
            {   
                this.CalcEfficiency();
                this._index = cfCount++;
            }
        }
        void ParseVehTechsLists(int techIdx, VehicleCollection vehs, int vehIdx)
        {
            int allVehCount = (vehIdx == -1) ? vehs.Count : 1;
            this.EnsureCapacity((vehIdx == -1) ? vehs.Count : 1);
            Manufacturer             mfr = this._modelYearData[this._yrIndex].Manufacturers[this._mfrIndex];
            ManufacturerModelingData mmd = mfr.ModelingData;
            int vIdxOff = (vehIdx == -1) ? 0 : vehIdx;
            for (int i = 0; i < allVehCount; i++)
            {
                Vehicle veh = vehs[i + vIdxOff];    
                Manufacturer             lMfr = null;
                ManufacturerModelingData lMmd = null;
                if (veh.ModelingData.TechLingerYear != null)
                {   
                    lMfr = this._modelYearData[veh.ModelingData.TechLingerYear.Index].Manufacturers[this._mfrIndex];
                    lMmd = lMfr.ModelingData;
                }
                bool lingering;
                if (TA.IsTechValid(mmd, veh.ModelingData, veh.RegClass, this._techGroup[techIdx].Index, this._ignorePhaseIn,
                    false, lMmd, out lingering))
                {   
                    this._techKeys[this._vehCount] = TA.TechKeys[techIdx];
                    this._vehs    [this._vehCount] = veh;
                    this._ignore  [this._vehCount] = 0;
                    if (this._multiYear && lingering)
                    {   
                        this._techLingerKeys[this._vehCount] = TA.TechKeys[techIdx];
                    }
                    else
                    {   
                        this._techLingerKeys[this._vehCount] = 0;
                    }
                    if (this._backfill)
                    {
                        this.ParseVehTechsLists_Backfill(techIdx, veh, mmd);
                    }
                    this._vehCount++;
                }
            } 
            if (this._vehCount > 0)
            {   
                RegulatoryClass veh0rc = this._vehs[0].RegClass;
                for (int i = 1; i < this._vehCount; i++)
                {   
                    if (this._vehs[i].RegClass != veh0rc) { this._isRcMix = true; break; }
                }
                if (this._vehCount != allVehCount) { this._isEnablesMix = true; }
                else
                {   
                    for (int i = 1; i < this._vehCount; i++)
                    {   
                        if (this._techKeys[i] != this._techKeys[0]) { this._isEnablesMix = true; break; }
                    }
                }
            } 
        }
        void ParseVehTechsLists_Backfill(int techIdx, Vehicle veh, ManufacturerModelingData mmd)
        {
            TechnologyClass     techClass  = veh.TechnologyClass;
            RegulatoryClass     regClass   = veh.RegClass;
            VehicleModelingData vmd        = veh.ModelingData;
            for (int i = techIdx - 1; i >= 0; i--)
            {   
                int i2 = -1, iTmp = i;  
                switch (i)
                {   
                    case TI.DVVLD : i2 = this.FindTechInGroup(TI.CVVL ); break;
                    case TI.DVVLO : i2 = this.FindTechInGroup(TI.CDOHC); break;
                    case TI.CVVL: case TI.CDOHC:
                        continue;       
                }
                if (i2 != -1)
                {   
                    if (TA.GetCost(this._techGroup[i2], veh, this._modelYearData[this._yrIndex], this._costEstimates, this._year,
                                   this._startYear, this._techGroup, this._techKeys[this._vehCount], this._clipCost) <
                        TA.GetCost(this._techGroup[i ], veh, this._modelYearData[this._yrIndex], this._costEstimates, this._year,
                                   this._startYear, this._techGroup, this._techKeys[this._vehCount], this._clipCost))
                    {   
                        Global.Swap(ref i, ref i2);
                    }
                }
                for (int j = 0; j < 2; j++)
                {   
                    int bftIndex = this._techGroup[i].Index;    
                    Manufacturer             lMfr = null;
                    ManufacturerModelingData lMmd = null;
                    if (vmd.TechLingerYear != null)
                    {
                        lMfr = this._modelYearData[vmd.TechLingerYear.Index].Manufacturers[this._mfrIndex];
                        lMmd = lMfr.ModelingData;
                    }
                    bool lingering;
                    bool isTechValid = TA.IsTechValid(mmd, vmd, regClass, bftIndex, this._ignorePhaseIn, true, lMmd, out lingering);
                    bool canBackfill = TA.CanBackfill(i, this._techGroup, this._techKeys[this._vehCount], vmd.TechUsed);
                    if (isTechValid && canBackfill)
                    {
                        this._techKeys[this._vehCount] += TA.TechKeys[i];
                        if (this._multiYear && lingering)
                        {
                            this._techLingerKeys[this._vehCount] += TA.TechKeys[i];
                        }
                    }
                    if (i2 == -1) { break; }    
                    i = i2;                     
                }
                i = iTmp;
            } 
        }
        void SplitShared(bool splitMixedEnables, ref int cfCount)
        {
            double eff   = double.NaN;
            int    effRc = 0;
            for (int i = 1; i <= 3; i++)
            {   
                RegulatoryClass regClass = (RegulatoryClass)i;
                int firstRcVeh = -1;    
                int rcVehCount =  0;    
                for (int j = 0; j < this._vehCount; j++)
                {
                    if (this._vehs[j].RegClass == regClass)
                    {   
                        this._ignore[j] = 0;                        
                        if (firstRcVeh == -1) { firstRcVeh = j; }   
                        rcVehCount++;                               
                    }
                    else { this._ignore[j] = 1; }
                }
                if (rcVehCount == 0) { continue; }
                bool isEnablesMix = false;
                if (splitMixedEnables)
                {
                    for (int j = firstRcVeh + 1; j < this._vehCount; j++)
                    {   
                        if (this._ignore[j] == 0 && this._techKeys[j] != this._techKeys[firstRcVeh])
                        {   
                            isEnablesMix = true; break;
                        }
                    }
                }
                this._isEnablesMix = isEnablesMix;
                if (isEnablesMix) { this.SplitMixedEnables(i == 1, ref cfCount); }
                else              { this.CalcEfficiency(); }
                if (double.IsNaN(eff) || this._efficiency < eff)
                {
                    eff   = this._efficiency;
                    effRc = i;
                }
            } 
            if (effRc > 0)
            {
                if (effRc < 3)
                {   
                    RegulatoryClass regClass = (RegulatoryClass)effRc;
                    for (int i = 0; i < this._vehCount; i++)
                    {
                        this._ignore[i] = (this._vehs[i].RegClass == regClass) ? 0 : 1;
                    }
                    this.CalcEfficiency();
                }
                this._index     = cfCount++;
                this._isSplitCf = true;
            }
        }
        void SplitMixedEnables(bool sortLists, ref int cfCount)
        {
            if (sortLists)
            {   
                for (int i = this._vehCount - 1; i >= 0; i--)
                {
                    for (int j = 1; j <= i; j++)
                    {
                        if (this._techKeys[j - 1] > this._techKeys[j])
                        {   
                            this.Swap(this._vehs          , j - 1, j);
                            this.Swap(this._ignore        , j - 1, j);
                            this.Swap(this._techKeys      , j - 1, j);
                            this.Swap(this._techLingerKeys, j - 1, j);
                        }
                    } 
                } 
            }
            double eff    = double.NaN;     
            ulong  effKey = 0;              
            ulong  key    = 0;              
            for (int i = 0; i < this._vehCount; i++)
            {
                if ((this._ignore[i] & 1) != 1)
                {   
                    if (key == this._techKeys[i]) { this._ignore[i] -= 2; }
                    else
                    {   
                        if (key != 0)
                        {   
                            this.CalcEfficiency();
                            if (double.IsNaN(eff) || this._efficiency < eff)
                            {   
                                eff    = this._efficiency;
                                effKey = key;
                            }
                        }
                        key = this._techKeys[i];
                        for (int j = 0; j < this._vehCount; j++) { this._ignore[j] |= 2; }
                        this._ignore[i] -= 2;   
                    }
                } 
            } 
            this.CalcEfficiency();
            if (double.IsNaN(eff) || this._efficiency < eff)
            {   
                eff    = this._efficiency;
                effKey = key;
            }
            if (effKey != 0)
            {
                for (int i = 0; i < this._vehCount; i++)
                {   
                    if (this._techKeys[i] == effKey) { this._ignore[i] &= 1; }
                    else                             { this._ignore[i] |= 2; }
                }
                this.CalcEfficiency();
                this._index     = cfCount++;
                this._isSplitCf = true;
            }
        }
        int FindTechInGroup(int techIdx)
        {
            for (int i = 0; i < this._techCount; i++)
            {
                if (this._techGroup[i].Index == techIdx) { return i; }
            }
            return -1;
        }
        void Swap(Vehicle[] arr, int i, int j) { Vehicle tmp = arr[i]; arr[i] = arr[j]; arr[j] = tmp; }
        void Swap(ulong  [] arr, int i, int j) { ulong   tmp = arr[i]; arr[i] = arr[j]; arr[j] = tmp; }
        void Swap(int    [] arr, int i, int j) { int     tmp = arr[i]; arr[i] = arr[j]; arr[j] = tmp; }
        #endregion 
        #region 
        void CalcEfficiency()
        {
            this._efficiency = double.NaN;
            bool hasVehs = false;   
            for (int i = 0; i < this._vehCount; i++)
            {
                if (this._ignore[i] == 0 && this._vehs[i].RegClass != RegulatoryClass.Unregulated)
                {   
                    hasVehs = true; break;
                }
            }
            if (!hasVehs) { return; }   
            if (this._multiYear)
            {
                for (int i = 0; i < this._vehCount; i++)
                {
                    if (this._techLingerKeys[i] != 0)
                    {
                        this._lingerYear[i] = this._vehs[i].ModelingData.TechLingerYear.Year;
                    }
                }
            }
            Industry     data = this._modelYearData[this._yrIndex];
            Manufacturer mfr  = data.Manufacturers[this._mfrIndex];
            this.CalcTechImpacts(data.ModelingData);
            this._newCafe  = Standards.TestCAFE (mfr, this._year, this._techImpacts, this._vehCount, false);
            this._newFines = Standards.TestFines(mfr, this._scen, this._year, this._parameters, this._newCafe,
                this._techImpacts, this._vehCount);
            RCDouble effectiveNewCafe = Standards.TestCAFE (mfr, this._year, this._techImpacts, this._vehCount, true);
            this._effectiveNewFines   = Standards.TestFines(mfr, this._scen, this._year, this._parameters,
                effectiveNewCafe.CreateRounded(1), this._techImpacts, this._vehCount);
            RCDouble curCafe  = Standards.TestCAFE (mfr, this._year, null, 0, false);
            RCDouble curFines = Standards.TestFines(mfr, this._scen, this._year, this._parameters, curCafe, null, 0);
            RCDouble deltaFines = curFines - this._newFines;
            double totalDeltaFines = deltaFines.TotalAll;
            if (totalDeltaFines <= 0) { this.IgnoreTechnologies(); }
            else
            {   
                ManufacturerModelingData mmd = mfr.ModelingData;
                KMYType kmy = mmd.KMY[this._yrIndex];
                double techCost      = 0;
                double affectedSales = 0;
                double fcSavings     = 0;
                double deltaRpe      = 0;
                for (int i = 0; i < this._vehCount; i++)
                {
                    TechImpact impact = this._techImpacts[i];
                    if (!impact.Ignore)
                    {
                        for (int yrIndex = 0; yrIndex <= this._yrIndex; yrIndex++)
                        {
                            if (impact.HasImpact[yrIndex])
                            {
                                Vehicle         veh      = impact.Vehicle[yrIndex];
                                RegulatoryClass regClass = veh.RegClass;
                                double          fe       = veh.Description.FuelEconomy;
                                double          newFe    = fe + impact.DeltaFuelEconomy[yrIndex];
                                double          sales    = veh.Description.Sales[yrIndex];
                                string fuelType  = veh.Engine.Description.Fuel;
                                double kmyVal    = (fuelType == "D") ? kmy.Diesel : (fuelType == "E85") ? kmy.E85 :
                                    (fuelType == "H") ? kmy.Hydrogen : (fuelType == "CNG") ? kmy.CNG : kmy.Gasoline;
                                double newKmyVal = (impact.IsDieselized) ? kmy.Diesel : kmyVal;
                                techCost        += sales * impact.Cost[yrIndex];
                                affectedSales   += sales;
                                fcSavings       += sales * (kmyVal / fe - newKmyVal / newFe);
                            } 
                        } 
                    } 
                    deltaRpe = techCost * 0.77;
                } 
                double numerator   = deltaRpe - totalDeltaFines - fcSavings;
                double denominator = affectedSales;
                this._efficiency   = numerator / denominator;
                this._techCost     = techCost;
                this._totalDeltaRPE      = deltaRpe;
                this._totalDeltaFines    = totalDeltaFines;
                this._totalFuelSavings   = fcSavings;
                this._totalAffectedSales = affectedSales;
            } 
        }
        int[] CalcTechImpacts_GetVehiclePredecessorHistory(Vehicle veh, int yrIndex, int lingerYrIndex)
        {
            int[] predHistory = new int[yrIndex + 1];
            for (int i = yrIndex; i >= lingerYrIndex; i--)
            {
                if (veh.Description.Sales[i] <= 0)
                {   
                    veh = veh.Predecessor;
                }
                predHistory[i] = veh.Description.Code;
            }
            return predHistory;
        }
        Vehicle CalcTechImpacts_GetVehicle(int vehIndex, int[] predHistory, int yrIndex)
        {
            Vehicle veh = this._vehs[vehIndex];
            if (yrIndex == this._yrIndex)
            {   
                return veh;
            }
            else
            {   
                Manufacturer      lMfr  = this._modelYearData[yrIndex].Manufacturers[this._mfrIndex];
                VehicleCollection lVehs = lMfr.ModelYearVehicles;
                int     lCode = predHistory[yrIndex];
                Vehicle lVeh  = lVehs.FindByCode(lCode);
                if (lVeh == null)
                {   
                    return null;
                }
                if (lCode == veh.Description.Code)
                {   
                    return lVeh;
                }
                if ((this._techGroup.HasEngine       && lVeh.Description.EngineCode       != veh.Description.EngineCode) ||
                    (this._techGroup.HasTransmission && lVeh.Description.TransmissionCode != veh.Description.TransmissionCode))
                {   
                    return null;
                }
                return lVeh;
            } 
        }
        void CalcTechImpacts(IndustryModelingData imd)
        {
            double vehAffectedSales = 0;
            for (int i = 0; i < this._vehCount; i++)
            {   
                if (this._ignore[i] == 0) { vehAffectedSales += this._vehs[i].Description.Sales[this._yrIndex]; }
            }
            for (int i = 0; i < this._vehCount; i++)
            {   
                if (this._ignore[i] != 0) { this._techImpacts[i].Ignore = true; continue; }
                Vehicle curVeh        = this._vehs          [i];                            
                ulong   curTechKey    = this._techKeys      [i];                            
                ulong   lingerTechKey = this._techLingerKeys[i];                            
                bool    lingering     = (lingerTechKey != 0);                               
                int     lingerYrYear  = (lingering) ? this._lingerYear[i] : this._yrYear;   
                int     lingerYrIndex = lingerYrYear - ModelYear.MinYear;                   
                int[] predHistory = this.CalcTechImpacts_GetVehiclePredecessorHistory(curVeh, this._yrIndex, lingerYrIndex);
                int       yrCount      = this._yrIndex + 1;
                bool      ignoreImpact = false;
                bool   [] hasImpact    = new bool   [yrCount];
                Vehicle[] impactVeh    = new Vehicle[yrCount];
                double [] impactCost   = new double [yrCount];
                double [] impactImprv  = new double [yrCount];
                double [] impactDCW    = new double [yrCount];
                double [] impactDFE    = new double [yrCount];
                bool      isDieselized = false;
                for (int j = 0; j < yrCount; j++) { impactImprv[j] = 1; }
                for (int j = lingerYrIndex; j <= this._yrIndex; j++)
                {   
                    Vehicle veh = this.CalcTechImpacts_GetVehicle(i, predHistory, j);
                    if (veh == null) { ignoreImpact = true; break; }
                    ModelYear           year      = new ModelYear(ModelYear.FromIndex(j));  
                    Industry            data      = this._modelYearData[j];
                    VehicleDescription  vd        = veh.Description;
                    VehicleModelingData vmd       = veh.ModelingData;
                    TechnologyClass     techClass = curVeh.TechnologyClass;
                    RegulatoryClass     regClass  = curVeh.RegClass;
                    ulong               techKey   = (j == this._yrIndex) ? curTechKey : lingerTechKey;
                    if (techKey != 0)
                    {   
                        hasImpact[j] = true;
                        impactVeh[j] = veh;     
                        for (int k = 0; k < this._techCount; k++)
                        {
                            Technology tech = this._techGroup[k];
                            if ((techKey & TA.TechKeys[k]) != 0)
                            {   
                                double cost  = TA.GetCost       (tech, veh, data, this._costEstimates, year, this._startYear,
                                    this._techGroup, techKey, this._clipCost);
                                double imprv = TA.GetImprovement(tech, techClass, this._fuelEstimates, vmd.TechUsed,
                                    vmd.TechSuperseded, this._techGroup, techKey, this._clipImprv);
                                impactCost [j] += cost;
                                impactImprv[j] *= (1 - imprv);
                                if (TI.IsMaterialSubstitution(tech.Index))
                                {
                                    impactDCW[j] -= (vd.BaseWeight * tech.Attributes[techClass].Aux);
                                }
                                if (!isDieselized && TI.IsDieselization(tech.Index)) { isDieselized = true; }
                            } 
                        } 
                    } 
                } 
                this._techImpacts[i].Ignore = ignoreImpact;
                if (!ignoreImpact)
                {   
                    for (int j = lingerYrIndex; j <= this._yrIndex; j++)
                    {
                        if (hasImpact[j]) { impactDFE[j] = impactVeh[j].Description.FuelEconomy *
                                                           ((1 - impactImprv[j]) / impactImprv[j]); }
                    }
                    this._techImpacts[i].HasImpact        = hasImpact;
                    this._techImpacts[i].Vehicle          = impactVeh;
                    this._techImpacts[i].Cost             = impactCost;
                    this._techImpacts[i].Improvement      = impactImprv;
                    this._techImpacts[i].DeltaCurbWeight  = impactDCW;
                    this._techImpacts[i].DeltaFuelEconomy = impactDFE;
                    this._techImpacts[i].IsDieselized     = isDieselized;
                }
            } 
        }
        public void IgnoreTechnologies()
        {
            for (int i = 0; i < this._vehCount; i++)
            {
                if (this._ignore[i] == 0)
                {   
                    VehicleModelingData vmd = this._vehs[i].ModelingData;
                    for (int j = 0; j < this._techCount; j++)
                    {   
                        if ((this._techKeys[i] & TA.TechKeys[j]) != 0) { vmd.TechIgnored[this._techGroup[j].Index] = true; }
                    } 
                } 
            } 
        }
        public void ReserveTechnologies()
        {
            for (int i = 0; i < this._vehCount; i++)
            {
                if (this._ignore[i] == 0)
                {   
                    VehicleModelingData vmd = this._vehs[i].ModelingData;
                    for (int j = 0; j < this._techCount; j++)
                    {   
                        if ((this._techKeys[i] & TA.TechKeys[j]) != 0) { vmd.TechReserved[this._techGroup[j].Index] = true; }
                    } 
                } 
            } 
        }
        #endregion
        #region 
        public void ApplyFinding()
        {
            if (!this.IsValid  ) { throw new InvalidOperationException("The ComplianceFinding being applied is not valid."); }
            if (this._isApplied) { throw new InvalidOperationException("The ComplianceFinding has already been applied."  ); }
            this._isApplied = true;
            this.LogFinding();
            this.ApplyFinding_CheckSplitCf();
            for (int i = 0; i < this._vehCount; i++)
            {
                if (this._ignore[i] == 0)
                {   
                    this.ApplyTechImpact(this._techImpacts[i], this._techKeys[i], this._techLingerKeys[i]);
                }
            }
        }
        void ApplyTechImpact(TechImpact impact, ulong curTechKey, ulong lingerTechKey)
        {
            for (int i = 0; i <= this._yrIndex; i++)
            {
                ModelYear modelYear = new ModelYear(ModelYear.MinYear + i);
                if (impact.HasImpact[i]) 
                {   
                    Vehicle               veh = impact.Vehicle[i];
                    VehicleDescription    vd  = veh.Description;
                    VehicleModelingData   vmd = veh.ModelingData;
                    ComponentModelingData cmd =     
                        (this._techType == TechnologyType.Engine      ) ? veh.Engine      .ModelingData :
                        (this._techType == TechnologyType.Transmission) ? veh.Transmission.ModelingData :
                        (this._techType == TechnologyType.Aerodynamics) ? veh.NamePlate   .ModelingData : null;
                    ManufacturerModelingData mmd = veh.Manufacturer.ModelingData;
                    ulong techKey = (i == this._yrIndex) ? curTechKey : lingerTechKey;
                    RegulatoryClass regClass = veh.RegClass;
                    double          vehSales = vd.Sales[i];
                    ulong baseTech = 0;
                    for (int j = this._techCount - 1; j >= 0; j--)
                    {   
                        if ((techKey & TA.TechKeys[j]) != 0) { baseTech = TA.TechKeys[j]; break; }
                    }
                    for (int j = 0; j < this._techCount; j++)
                    {   
                        if ((techKey & TA.TechKeys[j]) != 0)
                        {   
                            this.ApplyTechnology(veh, vmd, cmd, mmd, regClass, vehSales, this._techGroup[j], modelYear,
                                TA.TechKeys[j] != baseTech);
                        }
                    } 
                    double techCost = vehSales * impact.Cost[i];
                    if (cmd != null)
                    {
                        cmd.TechCost        += techCost;
                        cmd.TechImprovement += impact.Improvement[i];
                    }
                    if (impact.DeltaCurbWeight [i] != 0) { vd.CurbWeight  += impact.DeltaCurbWeight [i]; }
                    if (impact.DeltaFuelEconomy[i] != 0) { vd.FuelEconomy += impact.DeltaFuelEconomy[i]; }
                    vmd.TechCost += techCost;
                    veh.UpdateTypeAndMobile6Class();
                    mmd.TechCost[regClass] += techCost;
                } 
            } 
        }
        void ApplyTechnology(Vehicle veh, VehicleModelingData vmd, ComponentModelingData cmd, ManufacturerModelingData mmd,
            RegulatoryClass regClass, double vehSales, Technology tech, ModelYear year, bool backfilling)
        {
            TA.InstallTechnologyOnVehicle(veh, tech, year, this._settings, backfilling, mmd.TechExhausted[tech.Index][regClass]);
            if (cmd != null) { cmd.TechUsed[tech.Index] = true; }
            mmd.TechUsedSales   [tech.Index][regClass] += vehSales;
            mmd.TechAppliedSales[tech.Index][regClass] += vehSales;
            TA.CheckPhaseIn(mmd, regClass, tech, year);
        }
        void ApplyFinding_CheckSplitCf()
        {
            if (!this._isSplitCf) { return; }
            if (this._techType != TechnologyType.Engine && this._techType != TechnologyType.Transmission)
            {
                throw new Exception("The ComplianceFinding was marked as split, however, the technologies selected " +
                    "for applicability are not engine or transmission type.");
            }
            VehicleCollection[] splitVehs = new VehicleCollection[this._yrIndex + 1];
            for (int i = 0; i < this._vehCount; i++)
            {
                TechImpact impact = this._techImpacts[i];
                if (!impact.Ignore)
                {
                    for (int j = 0; j <= this._yrIndex; j++)
                    {
                        if (impact.HasImpact[j])
                        {   
                            if (splitVehs[j] == null) { splitVehs[j] = new VehicleCollection(); }
                            splitVehs[j].Add(impact.Vehicle[j]);
                        }
                    } 
                } 
            } 
            int code = -1;
            for (int i = this._yrIndex; i >= 0; i--)
            {
                if (splitVehs[i] != null)
                {   
                    if (this._techType == TechnologyType.Engine)
                    {   
                        Engine eng = splitVehs[i][0].Engine.Split(splitVehs[i], code);
                        splitVehs[i][0].Manufacturer.Engines.Add(eng);
                        code = eng.Description.Code;
                    }
                    else 
                    {   
                        Transmission trn = splitVehs[i][0].Transmission.Split(splitVehs[i], code);
                        splitVehs[i][0].Manufacturer.Transmissions.Add(trn);
                        code = trn.Description.Code;
                    }
                } 
            } 
        }
        #endregion
        #region 
        public void LogFinding()
        {
            if (!this.IsValid || this._logWriter.IsEmpty) { return; }
            Manufacturer mfr = this._modelYearData[this._yrIndex].Manufacturers[this._mfrIndex];
            ComplianceWriter log    = this._logWriter.GetCompliance   (this._scen, this._year, mfr);
            ComplianceWriter extLog = this._logWriter.GetExtCompliance(this._scen, this._year, mfr);
            if (log    != null && this._isApplied) { this.LogFinding(log   , mfr.ModelingData); }
            if (extLog != null                   ) { this.LogFinding(extLog, mfr.ModelingData); }
        }
        void LogFinding(ComplianceWriter cw, ManufacturerModelingData mmd)
        {
            StringBuilder line = new StringBuilder();
            int lineNumber = cw.LineCount + 1;
            for (int i = 0; i < this._vehCount; i++)
            {
                TechImpact impact      = this._techImpacts[i];
                Vehicle    baseVeh     = impact.Vehicle[this._yrIndex];
                int        baseVehCode = baseVeh.Description.Code;
                if (!impact.Ignore)
                {   
                    for (int j = impact.HasImpact.Length - 1; j >= 0; j--)
                    {   
                        if (impact.HasImpact[j] == false) { continue; }
                        Vehicle            veh = impact.Vehicle[j];
                        VehicleDescription vd  = veh.Description;
                        ulong techKey;
                        if (j == this._yrIndex)
                        {   
                            line.Append(lineNumber++    ); line.Append(TAB);
                            line.Append(this._index     ); line.Append(TAB);
                            line.Append(this._efficiency); line.Append(TAB);
                            line.Append(this._totalDeltaRPE     .ToString("0.####")); line.Append(TAB);
                            line.Append(this._totalDeltaFines   .ToString("0.####")); line.Append(TAB);
                            line.Append(this._totalFuelSavings  .ToString("0.####")); line.Append(TAB);
                            line.Append(this._totalAffectedSales.ToString("0.####")); line.Append(TAB);
                            RegulatoryClass regClass = veh.RegClass;
                            line.Append(mmd.Standard  [regClass]); line.Append(TAB);
                            line.Append(mmd.CAFE      [regClass]); line.Append(TAB);
                            line.Append(this._newCafe [regClass]); line.Append(TAB);
                            line.Append(mmd.Fines     [regClass]); line.Append(TAB);
                            line.Append(this._newFines[regClass]); line.Append(TAB);
                            techKey = this._techKeys[i];
                        }
                        else
                        {   
                            line.Append(lineNumber++);
                            line.Append(TAB_12);
                            techKey = this._techLingerKeys[i];
                        }
                        if (baseVehCode != vd.Code)
                        {   
                            line.Append('['); line.Append(vd.Code); line.Append(']');
                        }
                        line.Append(vd.Code); line.Append(TAB);
                        bool isFirstTech = true;
                        for (int k = 0; k < this._techCount; k++)
                        {
                            if ((techKey & TA.TechKeys[k]) != 0)
                            {   
                                if (isFirstTech) { isFirstTech = false; } 
                                else { line.Append(", "); }
                                line.Append(this._techGroup[k].Abbr);
                            } 
                        } 
                        line.Append(TAB);
                        double newFE = vd.FuelEconomy + impact.DeltaFuelEconomy[j];
                        line.Append(j + ModelYear.MinYear); line.Append(TAB);
                        line.Append(vd.FuelEconomy            .ToString("0.####")); line.Append(TAB);
                        line.Append(newFE                     .ToString("0.####")); line.Append(TAB);
                        line.Append(vd.Sales               [j].ToString(        )); line.Append(TAB);
                        line.Append(impact.Cost            [j].ToString("0.####")); line.Append(TAB);
                        line.Append(impact.Improvement     [j].ToString("0.####")); line.Append(TAB);
                        line.Append(impact.DeltaCurbWeight [j].ToString("0.####")); line.Append(TAB);
                        line.Append(impact.DeltaFuelEconomy[j].ToString("0.####"));
                        line.Append(NewLine);
                    } 
                } 
            }
            cw.Write(line.ToString());
        }
        #endregion
        #endregion
        #region 
        public int Index { get { return this._index; } }
        public bool IsValid { get { return !double.IsNaN(this._efficiency); } }
        public bool IsEfficient { get { return (this._efficiency < 0); } }
        public bool InCompliance { get { return this.IsValid && (this._effectiveNewFines.TotalAll <= 0); } }
        public bool IsApplied { get { return this._isApplied; } }
        public double Efficiency { get { return this._efficiency; } }
        public double TechCost { get { return this._techCost; } }
        #endregion
        #region 
        const string TAB     = "\t";
        const string TAB_7   = "\t\t\t\t\t\t\t";
        const string TAB_12  = "\t\t\t\t\t\t\t\t\t\t\t\t";
        const string NewLine = "\r\n";
        Scenario             _scen;
        ModelYear            _year;
        int                  _yrIndex;          
        int                  _yrYear;           
        int                  _startYear;
        Industry[]           _modelYearData;
        int                  _mfrIndex;
        ModelingSettings     _settings;
        Parameters           _parameters;       
        Estimates            _costEstimates,    
                             _fuelEstimates;
        bool                 _multiYear;        
        bool                 _ignorePhaseIn;
        bool                 _backfill;
        bool                 _clipCost;
        bool                 _clipImprv;
        bool                 _splitShared;
        bool                 _splitMixed;
        bool                 _skipShared;
        bool                 _skipMixed;
        LogWriter            _logWriter;
        TechnologyType       _techType;         
        TechnologyCollection _techGroup;        
        int                  _techCount;
        internal int _index;                    
        internal int _vehCount;                 
        internal int _capacity;                 
        internal Vehicle[]    _vehs;            
        internal int    []    _ignore;          
        internal ulong     [] _techKeys;        
        internal ulong     [] _techLingerKeys;  
        internal TechImpact[] _techImpacts;     
        int                [] _lingerYear;      
        internal bool _hasVehs;
        internal bool _isRcMix;
        internal bool _isEnablesMix;
        internal bool _isSplitCf;
        internal bool _isApplied;
        internal double _efficiency;
        internal double _techCost;
        RCDouble _newCafe;
        RCDouble _newFines;
        RCDouble _effectiveNewFines;
        double _totalDeltaRPE;
        double _totalDeltaFines;
        double _totalFuelSavings;
        double _totalAffectedSales;
        #endregion
    }
}

