using System;
using System.IO;
using System.Threading;
using Volpe.Cafe;
using Volpe.Cafe.Collections;
using Volpe.Cafe.Data;
using Volpe.Cafe.Data.Optimization;
using Volpe.Cafe.IO;
using Volpe.Cafe.Model;
using Volpe.Cafe.Reporter.ReportTypes;
namespace Volpe.Cafe.Reporter
{
    [Serializable]
    public sealed class Reporter
    {
        #region 
        public event EventHandler ReportingCompleted;
        public event EventHandler ReportingStopped;
        public event EventHandler ReportingProgress;
        #endregion
        #region 
        public Reporter(ICompliance compliance, string path, string templatePath, Industry ppData, ModelingSettings settings,
            bool isOpt, int optIndSn, IterationsData[] optIndData)
        {
            this._compliance   = compliance;
            this._path         = path;
            this._templatePath = templatePath;
            this._ppData       = ppData;
            this._settings     = settings;
            this._isOpt        = isOpt;
            this._optIndSn     = optIndSn;
            this._optIndData   = optIndData;
        }
        #endregion
        #region 
        #region 
        private void OnReportingCompleted()
        {
            if (this.ReportingCompleted != null)
            {
                this.ReportingCompleted(this, EventArgs.Empty);
            }
        }
        private void OnReportingStopped()
        {
            if (this.ReportingStopped != null)
            {
                this.ReportingStopped(this, EventArgs.Empty);
            }
        }
        private void OnReportingProgress(string progress)
        {
            if (this.ReportingProgress != null)
            {
                this._progress = progress;
                this.ReportingProgress(this, EventArgs.Empty);
            }
        }
        #endregion
        public void GenerateReports(bool threaded)
        {
            if (threaded)
            {
                Thread t = new Thread(new ThreadStart(this.GenerateReports));
                t.Name = "ReporterThread";
                t.Start();
            }
            else
            {
                this.GenerateReports();
            }
        }
        public void GenerateReports()
        {
            if (!this._settings.OutputSettings.GenerateReports) { return; }
            this._path = this._path.Replace('[', '_').Replace(']', '_');
            if (!Directory.Exists(this._path)) { Directory.CreateDirectory(this._path); }
            try
            {
                this._template = (File.Exists(this._templatePath)) ? new Input(this._templatePath) : null;
            }
            catch { this._template = null; }
            try
            {   
                this._canceling = false;
                this.GenerateReportsHelper();
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.ToString());
                this._canceling = true;
            }
            finally
            {
                if (this._template != null)
                {   
                    this._template.Close();
                }
                this.FinalizeByRunReports(false);
                this.FinalizeByScenarioReports(false);
                if (this._canceling) { this.OnReportingStopped(); }
                else                 { this.OnReportingCompleted(); }
            }
        }
        private void GenerateReportsHelper()
        {
            ScenarioCollection scenarios = this._settings.Scenarios;
            int numScen = scenarios.Count;
            int minYear = this._ppData.MinYear.Year;
            int maxYear = this._ppData.MaxYear.Year;
            this.InitializeByRunReports();
            bool generated = false;         
            for (int i = 0; i < numScen; i++)
            {   
                this.InitializeByScenarioReports(i);
                bool snGenerated = false;   
                for (int j = minYear; j <= maxYear; j++)
                {
                    if (this.GenerateReportsHelper(scenarios[i], new ModelYear(j)))
                    {    
                        snGenerated = true;
                        generated   = true;
                    }
                    if (this._canceling) { return; }    
                }
                this.FinalizeByScenarioReports(snGenerated);
            }
            this.FinalizeByRunReports(generated);
        }
        private bool GenerateReportsHelper(Scenario scenario, ModelYear year)
        {
            this.OnReportingProgress("Loading data for Scenario: " + scenario.Index + ", MY: " + year.Year + " ...");
            Industry data = this._compliance.GetData(scenario.Index, year.Index);
            Industry baseData = this._compliance.GetData(0, year.Index);    
            if (data == null || baseData == null || this._ppData == null) { return false; }
            int snCount = this._settings.Scenarios.Count;
            int snIndex = scenario.Index;
            string scenName = scenario.Description;
            OutputSettings outSets = this._settings.OutputSettings;
            if (this._canceling) { return false; }
            if (outSets.GenerateIndustryReport)
            {
                this.OnReportingProgress("Initializing Report: " + Path.GetFileName(this._ind.XlUtils.Path) + " ...");
                this._ind.ParseData(this._settings, snIndex, year, data, baseData);
            }
            if (outSets.GenerateManufacturersReport)
            {
                this.OnReportingProgress("Initializing Report: " + Path.GetFileName(this._mfr.XlUtils.Path) + " ...");
                this._mfr.ParseData(this._settings, snIndex, year, data, baseData);
            }
            if (outSets.GenerateVehiclesReport)
            {
                this.OnReportingProgress("Initializing Report: " + Path.GetFileName(this._veh.XlUtils.Path) + " ...");
                this._veh.ParseData(this._settings, snIndex, year, data, this._ppData);
            }
            if (outSets.GenerateEffectsReport)
            {
                this.OnReportingProgress("Initializing Report: " + Path.GetFileName(this._eff.XlUtils.Path) + " ...");
                this._eff.ParseData(snIndex, year, data, baseData);
            }
            if (outSets.GenerateEnvironmentalAssessmentReport)
            {
                this.OnReportingProgress("Initializing Report: " + Path.GetFileName(this._eas.XlUtils.Path) + " ...");
                this._eas.ParseData(snCount, snIndex, year, data);
            }
            if (outSets.GenerateScenarioSummaryReport && snIndex > 0)
            {
                this.OnReportingProgress("Initializing Report: " + Path.GetFileName(this._ssu.XlUtils.Path) + " ...");
                this._ssu.ParseData(snCount, snIndex, scenName, year, data, baseData);
            }
            if (this._isOpt)
            {
                if (snIndex == this._optIndSn && outSets.GenerateOptimizedIndustryReportExtended)
                {
                    this.OnReportingProgress("Initializing Report: " + Path.GetFileName(this._optIndExt.XlUtils.Path) + " ...");
                    this._optIndExt.ParseData(year, this._settings, data, this._optIndData, this._optIndSn);
                }
            }
            return true;
        }
        private void InitializeByRunReports()
        {
            OutputSettings outSets = this._settings.OutputSettings;
            EncryptionSettings crypto = this._settings.EncryptionSettings;
            if (outSets.GenerateEnvironmentalAssessmentReport)
            {
                this.OnReportingProgress("Initializing Report: EA_Report.xls ...");
                this._eas = new EAReport(this._path + "\\EA_Report.xls", crypto);
            }
            if (outSets.GenerateScenarioSummaryReport)
            {
                this.OnReportingProgress("Initializing Report: ScenSummary_Report.xls ...");
                this._ssu = new ScenSummaryReport(this._path + "\\ScenSummary_Report.xls", crypto);
            }
        }
        private void InitializeByScenarioReports(int snIndex)
        {
            OutputSettings outSets = this._settings.OutputSettings;
            EncryptionSettings crypto = this._settings.EncryptionSettings;
            if (outSets.GenerateIndustryReport)
            {
                this.OnReportingProgress("Initializing Report: Industry_Report_Sn.xls ...");
                this._ind = new IndustryReport(this._path + "\\Industry_Report_Sn" + snIndex + ".xls", crypto);
            }
            if (outSets.GenerateManufacturersReport)
            {
                this.OnReportingProgress("Initializing Report: Manufacturers_Report_Sn.xls ...");
                this._mfr = new ManufacturersReport(this._path + "\\Manufacturers_Report_Sn" + snIndex + ".xls",
                    crypto);
            }
            if (outSets.GenerateVehiclesReport)
            {
                this.OnReportingProgress("Initializing Report: Vehicles_Report_Sn.xls ...");
                this._veh = new VehiclesReport(this._path + "\\Vehicles_Report_Sn" + snIndex + ".xls", crypto);
            }
            if (outSets.GenerateEffectsReport)
            {
                this.OnReportingProgress("Initializing Report: Effects_Report_Sn.xls ...");
                this._eff = new EffectsReport(this._path + "\\Effects_Report_Sn" + snIndex + ".xls", crypto);
            }
            if (this._isOpt)
            {
                if (snIndex == this._optIndSn && outSets.GenerateOptimizedIndustryReportExtended)
                {
                    this.OnReportingProgress("Initializing Report: OptIndExt_Report.xls ...");
                    this._optIndExt = new OptIndReport_Extended(this._path + "\\OptIndExt_Report.xls", crypto);
                }
            }
        }
        private void FinalizeByRunReports(bool generated)
        {
            OutputSettings outSets = this._settings.OutputSettings;
            this.FinalizeReport(this._eas, generated);
            this.FinalizeReport(this._ssu, generated);
        }
        private void FinalizeByScenarioReports(bool generated)
        {
            OutputSettings outSets = this._settings.OutputSettings;
            this.FinalizeReport(this._ind      , generated);
            this.FinalizeReport(this._mfr      , generated);
            this.FinalizeReport(this._veh      , generated);
            this.FinalizeReport(this._eff      , generated);
            this.FinalizeReport(this._optIndExt, generated);
        }
        private void FinalizeReport(Output report, bool generated)
        {
            if (report != null && !report.Disposed)
            {
                if (generated)
                {   
                    this.OnReportingProgress("Finalizing Report: " + Path.GetFileName(report.XlUtils.Path) + " ...");
                    report.WriteBuffers(this._template, true);
                    report.XlUtils.ActivateWorksheet(1);
                    report.Close(true);
                }
                else
                {   
                    string path = report.XlUtils.Path;
                    report.Close(false);
                    File.Delete(path);
                }
            }
        }
        public void CancelReports()
        {
            this._canceling = true;
            this.CancelReport(this._ind);
            this.CancelReport(this._mfr);
            this.CancelReport(this._veh);
            this.CancelReport(this._ssu);
            this.CancelReport(this._eff);
            this.CancelReport(this._eas);
            this.CancelReport(this._optIndExt);
        }
        private void CancelReport(Output report)
        {
            if (report != null)
            {
                report.CancelWrite();
            }
        }
        #endregion
        #region 
        public string Progress { get { return this._progress; } }
        #endregion
        #region 
        ICompliance        _compliance;
        string             _path;
        string             _templatePath;
        Industry           _ppData;
        ModelingSettings   _settings;
        bool               _isOpt;
        int                _optIndSn;
        IterationsData[]   _optIndData;
        Input _template;
        IndustryReport      _ind;
        ManufacturersReport _mfr;
        VehiclesReport      _veh;
        ScenSummaryReport   _ssu;
        EffectsReport       _eff;
        EAReport            _eas;
        OptIndReport_Extended _optIndExt;
        string _progress;
        bool   _canceling;
        #endregion
    }
}

