using System;
using Volpe.Cafe;
using Volpe.Cafe.Collections;
namespace Volpe.Cafe.Data
{
    [Serializable]
    public class Manufacturer : ICloneable
    {
        #region 
        private Manufacturer()
        {
        }
        public Manufacturer(ManufacturerDescription description)
        {
            this._description            = description;
            this._modelingData           = new ManufacturerModelingData();
            this._domesticPassengerAutos = new VehicleCollection(32);
            this._importedPassengerAutos = new VehicleCollection(32);
            this._lightTrucks            = new VehicleCollection(32);
            this._heavyLightTrucks       = new VehicleCollection(32);
            this._discardedVehs          = new VehicleCollection(32);
            this._modelYearVehicles      = new VehicleCollection(32);
            this._engines                = new EngineCollection(32);
            this._transmissions          = new TransmissionCollection(32);
            this._namePlates             = new NamePlateCollection(32);
            this._index                      = -1;
            this._valid                      = false;
            this._initialized                = false;
            this._lastEngCode                = -1;
            this._lastTrnCode                = -1;
            this._discardedEngineCount       = 0;
            this._discardedTransmissionCount = 0;
        }
        #endregion
        #region 
        #region 
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        public Manufacturer Clone()
        {
            Manufacturer mfr = new Manufacturer();
            mfr._index                      = this._index;
            mfr._initialized                = this._initialized;
            mfr._valid                      = this._valid;
            mfr._invalidReasons             = (string[])Global.CloneArray(this._invalidReasons, typeof(string));
            mfr._lastEngCode                = this._lastEngCode;
            mfr._lastTrnCode                = this._lastTrnCode;
            mfr._discardedEngineCount       = this._discardedEngineCount;
            mfr._discardedTransmissionCount = this._discardedTransmissionCount;
            mfr._description  = this._description.Clone();
            mfr._modelingData = this._modelingData.Clone();
            mfr._currentYear            = this._currentYear;
            mfr._domesticPassengerAutos = this._domesticPassengerAutos.Clone();
            mfr._importedPassengerAutos = this._importedPassengerAutos.Clone();
            mfr._lightTrucks            = this._lightTrucks.Clone();
            mfr._heavyLightTrucks       = this._heavyLightTrucks.Clone();
            mfr._discardedVehs          = this._discardedVehs.Clone();
            mfr._engines                = this._engines.Clone();
            mfr._transmissions          = this._transmissions.Clone();
            mfr._namePlates             = this._namePlates.Clone();
            Engine      [] engs = mfr._engines.ToArray();
            Transmission[] trns = mfr._transmissions.ToArray();
            NamePlate   [] npts = mfr._namePlates.ToArray();
            VehicleCollection vc = new VehicleCollection(this.AllVehicleCount);
            vc.AddRange(mfr._domesticPassengerAutos);
            vc.AddRange(mfr._importedPassengerAutos);
            vc.AddRange(mfr._lightTrucks);
            vc.AddRange(mfr._heavyLightTrucks);
            vc.AddRange(mfr._discardedVehs);
            Vehicle[] vehs = vc.ToArray();
            for (int i = 0; i < engs.Length; i++)
            {
                int parentCode = ((engs[i].Parent == null) ? -1 : engs[i].Parent.Description.Code);
                engs[i].Initialize(mfr);
                engs[i].SetParent(parentCode);
            }
            for (int i = 0; i < trns.Length; i++)
            {
                int parentCode = ((trns[i].Parent == null) ? -1 : trns[i].Parent.Description.Code);
                trns[i].Initialize(mfr);
                trns[i].SetParent(parentCode);
            }
            for (int i = 0; i < npts.Length; i++)
            {
                npts[i].Initialize(mfr);
            }
            for (int i = 0; i < vehs.Length; i++)
            {
                vehs[i].Successors.Clear();
            }
            for (int i = 0; i < vehs.Length; i++)
            {   
                vehs[i].Initialize(mfr, vehs);
                NamePlate np = mfr.GetNamePlate(vehs[i].Description.Model);
                np.Vehicles.Add(vehs[i]);
                vehs[i].SetNamePlate(np);
            }
            mfr._modelYearVehicles = new VehicleCollection(32);
            if (mfr._currentYear != null)
            {
                mfr.InitializeModelYearVehicles(mfr._currentYear);
            }
            return mfr;
        }
        #endregion
        public override string ToString()
        {
            return this._description.Name;
        }
        internal void Initialize(VehicleCollection vehicles, EngineCollection engines, TransmissionCollection transmissions)
        {
            Engine[] engs = engines.ToArray();
            Transmission[] trns = transmissions.ToArray();
            Vehicle[] vehs = vehicles.ToArray();
            for (int i = 0; i < engs.Length; i++)
            {
                engs[i].Initialize(this);
                engs[i].Validate();
                if (!engs[i].Valid) { this._discardedEngineCount++; }
                this._engines.Add(engs[i]);
                if (this._lastEngCode < engs[i].Description.Code)
                {
                    this._lastEngCode = engs[i].Description.Code;
                }
            }
            for (int i = 0; i < trns.Length; i++)
            {
                trns[i].Initialize(this);
                trns[i].Validate();
                if (!trns[i].Valid) { this._discardedTransmissionCount++; }
                this._transmissions.Add(trns[i]);
                if (this._lastTrnCode < trns[i].Description.Code)
                {
                    this._lastTrnCode = trns[i].Description.Code;
                }
            }
            for (int i = 0; i < vehs.Length; i++)
            {
                vehs[i].Initialize(this, vehs);
                vehs[i].Validate();
                NamePlate np = this.GetNamePlate(vehs[i].Description.Model);
                np.Vehicles.Add(vehs[i]);
                vehs[i].SetNamePlate(np);
                if (vehs[i].Valid)
                {   
                    if (vehs[i].VehicleType == VehicleType.DomesticAuto)
                    {   
                        this._domesticPassengerAutos.Add(vehs[i]);
                    }
                    else if (vehs[i].VehicleType == VehicleType.ImportedAuto)
                    {   
                        this._importedPassengerAutos.Add(vehs[i]);
                    }
                    else if (vehs[i].VehicleType == VehicleType.LightTruck)
                    {   
                        this._lightTrucks.Add(vehs[i]);
                    }
                    else if (vehs[i].VehicleType == VehicleType.HeavyLT)
                    {   
                        this._heavyLightTrucks.Add(vehs[i]);
                    }
                    else
                    {   
                        throw new System.ApplicationException("The modeling system has encountered a valid vehicle, " +
                            "however, that vehicle does not appear to have a vehicle type.  This should never happen.");
                    }
                }
                else
                {   
                    this._discardedVehs.Add(vehs[i]);
                }
            }
            for (int i = 0; i < vehs.Length; i++)
            {
                vehs[i].ValidatePredecessor();
            }
            this._initialized = true;
        }
        private NamePlate GetNamePlate(string name)
        {
            for (int i = 0; i < this._namePlates.Count; i++)
            {
                if (this._namePlates[i].Name.Equals(name.Trim().ToUpper()))
                {    
                    return this._namePlates[i];
                }
            }
            NamePlate np = new NamePlate(name);
            np.Initialize(this);
            this._namePlates.Add(np);
            return np;
        }
        internal void Validate(out int minYear, out int maxYear)
        {
            this._valid = true;        
            minYear = maxYear = -1;
            System.Collections.Specialized.StringCollection err = new System.Collections.Specialized.StringCollection();
            if (!this._initialized)
            {
                this._valid = false;
                err.Add("The Manufacturer has not been initialized.");
            }
            else
            {    
                if (this.VehicleCount == 0)
                {
                    this._valid = false;
                    err.Add("The Manufacturer does not contain any valid vehicles.");
                }
                else
                {    
                    VehicleCollection vc = this.GetVehicles();
                    minYear = int.MaxValue;
                    maxYear = -1;
                    for (int i = 0, vehCount = this.VehicleCount; i < vehCount; i++)
                    {
                        Vehicle v = vc[i];
                        if (v.ValidYears != null)
                        {
                            minYear = Math.Min(v.MinYear.Year, minYear);
                            maxYear = Math.Max(v.MaxYear.Year, maxYear);
                        }
                    }
                }
            }
            if (!this._valid)
            {
                this._invalidReasons = new string[err.Count];
                err.CopyTo(this._invalidReasons, 0);
            }
        }
        internal void SetIndex(int index)
        {
            if (!this._valid)
            {
                throw new System.InvalidOperationException(
                    "A manufacturer index cannot be assigned to invalid manufacturers");
            }
            this._index = index;
        }
        public VehicleCollection InitializeModelYearVehicles(ModelYear year)
        {
            this._currentYear = year;
            int vehCount = this.VehicleCount;
            vehCount = (vehCount < 32) ? 32 : (vehCount < 64) ? 64 : 128;
            this._modelYearVehicles = new VehicleCollection(vehCount);
            this.InitializeModelYearVehicles_BuildVehList(year, this._modelYearVehicles, this._domesticPassengerAutos);
            this.InitializeModelYearVehicles_BuildVehList(year, this._modelYearVehicles, this._importedPassengerAutos);
            this.InitializeModelYearVehicles_BuildVehList(year, this._modelYearVehicles, this._lightTrucks);
            this.InitializeModelYearVehicles_BuildVehList(year, this._modelYearVehicles, this._heavyLightTrucks);
            EngineCollection       ec  = this.Engines;
            TransmissionCollection tc  = this.Transmissions;
            NamePlateCollection    npc = this.NamePlates;
            for (int i = 0, count = ec .Count; i < count; i++) { Engine       eng = ec [i]; eng.InitializeModelYearVehicles(year); }
            for (int i = 0, count = tc .Count; i < count; i++) { Transmission trn = tc [i]; trn.InitializeModelYearVehicles(year); }
            for (int i = 0, count = npc.Count; i < count; i++) { NamePlate    np  = npc[i]; np .InitializeModelYearVehicles(year); }
            return this._modelYearVehicles;
        }
        void InitializeModelYearVehicles_BuildVehList(ModelYear year, VehicleCollection yrVehs, VehicleCollection vehs)
        {
            for (int i = 0, vehCount = vehs.Count; i < vehCount; i++)
            {
                if (vehs[i].IsValid(year)) { yrVehs.Add(vehs[i]); }
            }
        }
        public VehicleCollection GetVehicles()
        {
            VehicleCollection vc = new VehicleCollection(this.VehicleCount);
            vc.AddRange(this._domesticPassengerAutos);
            vc.AddRange(this._importedPassengerAutos);
            vc.AddRange(this._lightTrucks);
            vc.AddRange(this._heavyLightTrucks);
            return vc;
        }
        internal int GetNextEngineCode()
        {
            return ++this._lastEngCode;
        }
        internal int GetNextTransmissionCode()
        {
            return ++this._lastTrnCode;
        }
        internal void SetModelingData(ManufacturerModelingData value)
        {
            this._modelingData = value;
        }
        #endregion
        #region 
        public int Index
        {
            get { return this._index; }
        }
        public bool Initialized
        {
            get { return this._initialized; }
        }
        public bool Valid
        {
            get { return this._valid; }
        }
        public string[] InvalidReasons
        {
            get { return this._invalidReasons; }
        }
        public ManufacturerDescription Description
        {
            get { return this._description; }
        }
        public ManufacturerModelingData ModelingData
        {
            get { return this._modelingData; }
        }
        public VehicleCollection DomesticPassengerAutos
        {
            get { return this._domesticPassengerAutos; }
        }
        public VehicleCollection ImportedPassengerAutos
        {
            get { return this._importedPassengerAutos; }
        }
        public VehicleCollection LightTrucks
        {
            get { return this._lightTrucks; }
        }
        public VehicleCollection HeavyLightTrucks
        {
            get { return this._heavyLightTrucks; }
        }
        public VehicleCollection DiscardedVehicles
        {
            get { return this._discardedVehs; }
        }
        public VehicleCollection ModelYearVehicles
        {
            get { return this._modelYearVehicles; }
        }
        public EngineCollection Engines
        {
            get { return this._engines; }
        }
        public TransmissionCollection Transmissions
        {
            get { return this._transmissions; }
        }
        public NamePlateCollection NamePlates
        {
            get { return this._namePlates; }
        }
        public int ModelYearVehicleCount
        {
            get { return this._modelYearVehicles.Count; }
        }
        public int VehicleCount
        {
            get
            {
                return this._domesticPassengerAutos.Count + this._lightTrucks.Count +
                    this._importedPassengerAutos.Count + this._heavyLightTrucks.Count;
            }
        }
        public int DiscardedVehicleCount
        {
            get { return this._discardedVehs.Count; }
        }
        public int AllVehicleCount
        {
            get { return this.VehicleCount + this._discardedVehs.Count; }
        }
        public int NonDiscardedEngineCount
        {
            get { return this._engines.Count - this._discardedEngineCount; }
        }
        public int DiscardedEngineCount
        {
            get { return this._discardedEngineCount; }
        }
        public int NonDiscardedTransmissionCount
        {
            get { return this._transmissions.Count - this._discardedTransmissionCount; }
        }
        public int DiscardedTransmissionCount
        {
            get { return this._discardedTransmissionCount; }
        }
        internal int CurrentYear { get { return this._currentYear.Year; } set { this._currentYear = new ModelYear(value); } }
        internal int LastEngCode { get { return this._lastEngCode; } set { this._lastEngCode = value; } }
        internal int LastTrnCode { get { return this._lastTrnCode; } set { this._lastTrnCode = value; } }
        #endregion
        #region 
        private int _index;
        private bool _initialized;
        private bool _valid;
        private string[] _invalidReasons;
        private ManufacturerDescription _description;
        private ManufacturerModelingData _modelingData;
        private VehicleCollection _domesticPassengerAutos;
        private VehicleCollection _importedPassengerAutos;
        private VehicleCollection _lightTrucks;
        private VehicleCollection _heavyLightTrucks;
        private VehicleCollection _discardedVehs;
        private VehicleCollection _modelYearVehicles;
        private ModelYear _currentYear;
        private EngineCollection _engines;
        private TransmissionCollection _transmissions;
        private NamePlateCollection _namePlates;
        private int _lastEngCode;
        private int _lastTrnCode;
        private int _discardedEngineCount;
        private int _discardedTransmissionCount;
        #endregion
    }
}

