#region << Using Directives >>
using System;
using System.IO;
using System.Threading;
using Volpe.Cafe.Collections;
using Volpe.Cafe.Data;
using Volpe.Cafe.Data.Optimization;
using Volpe.Cafe.IO;
using Volpe.Ui;
#endregion
namespace Volpe.Cafe.Model.Optimization
{
    [Serializable]
    [ModelDescription("Optimization Model", "Optimizes the shape of the reformed CAFE standard by running multiple compliance " +
         "iterations and taking into account the resultant technology costs and benefits.", 1F)]
    public sealed class Optimization : Compliance
    {
        #region 
        public Optimization()
            : base()
        {
        }
        ~Optimization()
        {
        }
        #endregion
        #region 
        #region 
        public override ICompliance CreateNew()
        {
            return new Optimization();
        }
        public override void Start(Industry data, ModelingSettings settings, LogWriter logWriter)
        {
            if (settings.OptimizationSettings.MergedFleet)
            {   
                data = data.GetMergedFleet();
            }
            base.Start(data, settings, logWriter);
        }
        protected override void StartInternal()
        {
            if (this.PreProcess())
            {   
                this.Begin();
                this.PostProcess();
            }
        }
        protected override void Validate()
        {
            base.Validate();
            if (!this._valid) { return; }
            string message = null;
            if (this._scenCount != 2)
            {
                message = "The optimization model requires exactly 2 scenarios to be present in the scenarios file:  the " +
                    "Baseline scenario and the Optimization Definition scenario.  The number of scenarios found was " +
                    this._scenCount + ".";
            }
            if (this._settings.OptimizationSettings.Type != OptimizationType.PassengerAuto &&
                this._settings.OptimizationSettings.Type != OptimizationType.LightTruck    &&
                this._settings.OptimizationSettings.Type != OptimizationType.Default)
            {
                message = "The specified optimization type is not valid.  Valid types are PassengerAuto, LightTruck, or Default.";
            }
            if (message != null)
            {
                PromptEventArgs e = new PromptEventArgs("Validation Error", "An error has occured during validation of input " +
                    "data and/or modeling settings; the model will not run.  Please fix the error and try again.\n\nMore " +
                    "information:\n\n" + message, PromptOption.Ok);
                this.OnPrompt(e);
                this._valid = false;
            }
        }
        protected override void OnModelingChanged()
        {
            if (!this.Running) { this.ClearStatus(); }
            base.OnModelingChanged();
        }
        #endregion
        #region 
        protected override bool PreProcess()
        {
            if (this.State == ModelingState.Completed) { return false; }
            this._optSettings = this._settings.OptimizationSettings;
            if (this._optSettings.Type == OptimizationType.Default)
            {
                Scenario  typeScen = this._settings.Scenarios[1];
                ModelYear typeYear = new ModelYear(this._minYear);
                bool      typeDetermined = false;
                for (int i = 0; i < this._mfrCount; i++)
                {   
                    if (this._data.Manufacturers[i].Description.Optimize)
                    {
                        VehicleCollection vehs = this._data.Manufacturers[i].GetVehicles();
                        for (int j = 0, vehCount = vehs.Count; j < vehCount; j++)
                        {
                            RegulatoryClass regClass = Standards.GetRegClass(vehs[j].VehicleType, typeScen, typeYear);
                            if (regClass == RegulatoryClass.LightTruck)
                            {
                                this._optSettings.Type = OptimizationType.LightTruck;
                                typeDetermined = true;
                            }
                            else if (regClass != RegulatoryClass.Unregulated)
                            {
                                this._optSettings.Type = OptimizationType.PassengerAuto;
                                typeDetermined = true;
                            }
                            if (typeDetermined) { break; }
                        } 
                        if (typeDetermined) { break; }
                    } 
                } 
            }
            this._optMinYear = this._optSettings.StartYear;
            if (this._optMinYear < this._minYear) { this._optMinYear = this._minYear; }
            this._optMfrCount = 0;
            for (int i = 0; i < this._mfrCount; i++)
            {
                if (this._data.Manufacturers[i].Description.Optimize) { this._optMfrCount++; }
            }
            this._data.UpdateMinMaxFpCw();
            this._minFP = Math.Round(this._data.MinFp, 1) - 0.1;
            this._maxFP = Math.Round(this._data.MaxFp, 1) + 0.1;
            this._stage  = OptimizationStage.None;  
            this._status = string.Empty;
            this._mfrSales = new double[this._mfrCount][];
            ManufacturerCollection mfrs = this._data.Manufacturers;
            for (int i = 0; i < this._mfrCount; i++)
            {
                this._mfrSales[i] = new double[this._yearCount];
                VehicleCollection vehs = mfrs[i].GetVehicles();
                for (int j = 0; j < this._yearCount; j++)
                {
                    for (int k = 0, vehCount = vehs.Count; k < vehCount; k++)
                    {
                        if (vehs[k].IsValid(this._minYearIndex + j))
                        {
                            this._mfrSales[i][j] += vehs[k].Description.Sales[this._minYearIndex + j];
                        }
                    } 
                } 
            } 
            this._baselineData     = new IterationData [this._yearCount];
            this._optimizationData = new IterationsData[this._yearCount];
            this._iterationCount   = this._optSettings.AboveOptimum + this._optSettings.BelowOptimum + 1;
            for (int i = 0; i < this._yearCount; i++)
            {
                this._optimizationData[i].Iterations = new IterationData[this._iterationCount];
            }
            this._optimizedData = new Industry[this._yearCount];
            return base.PreProcess();
        }
        #endregion
        #region 
        void Begin()
        {
            this.ProcessBaselineScenario();
            if (this._abortRequested) { return; }
            this.ProcessOptimizationScenario();
            if (this._abortRequested) { return; }
            this._stage = OptimizationStage.None;
        }
        void ProcessBaselineScenario()
        {
            this._stage = OptimizationStage.Baseline;
            Industry data = this._data.Clone();
            base.RunScenario(this._settings.Scenarios[0], data);
            if (!this._abortRequested)
            {   
                for (int i = this._minYearIndex; i <= this._maxYearIndex; i++)
                {
                    data = this.GetData(0, i);          
                    this.SaveEntry(-1, i, data, null);  
                }
            }
        }
        void ProcessOptimizationScenario()
        {
            Scenario scen  = this._settings.Scenarios[1];
            this._stage    = OptimizationStage.Industry;
            this._scens[0] = scen;  
            for (int i = this._minYear; i <= this._maxYear; i++)
            {   
                ModelYear year = new ModelYear(i);
                this.OptimizeModelYear(scen, year);
                if (this._abortRequested) { break; }
            }
            this.ClearStatus();
        }
        void OptimizeModelYear(Scenario scen, ModelYear year)
        {
            if (year.Year < this._optMinYear)
            {   
                this.RunModelYearNonOptimized(scen, year);
            }
            else
            {   
                this.RunModelYearOptimized(scen, year);
            }
        }
        void RunModelYearNonOptimized(Scenario scen, ModelYear year)
        {
            int snIndex = scen.Index;
            int yrIndex = year.Index;
            this.InitializeModelYear(scen, yrIndex);
            base.RunModelYear(scen, year);
            this._optimizedData[yrIndex - this._minYearIndex] = this.GetData(snIndex, yrIndex).Clone();
            if (this._settings.OperatingModes.MultiYearModeling)
            {   
                for (int i = this._minYearIndex; i < yrIndex; i++)
                {
                    this._optimizedData[i - this._minYearIndex] = this.GetData(snIndex, i).Clone();
                }
            } 
        }
        void RunModelYearOptimized(Scenario scen, ModelYear year)
        {
            this._years[0] = year;
            int       yrYear  = year.Year;
            int       yrIndex = year.Index;
            bool      optLt   = (this._optSettings.Type == OptimizationType.LightTruck);
            int   []  fncType = (optLt) ? scen.LtFunctionType : scen.AutoFunctionType;
            double[,] fncCoef = (optLt) ? scen.LtCoefficients : scen.AutoCoefficients;
            bool restartYear = false;
            double aGpm = 0, bGpm = 0, lnrYGpm = 0;
            do
            {
                if (restartYear) { restartYear = false; }   
                this.PreProcessModelYearForOptimization (scen, year, fncType, fncCoef, ref aGpm, ref bGpm, ref lnrYGpm);
                this.IterationLoop                      (scen, year, fncType, fncCoef,     aGpm,     bGpm,     lnrYGpm);
                this.PostProcessModelYearForOptimization(scen, year, fncType, fncCoef, ref restartYear);
            } while (restartYear);
            if (this._abortRequested) { return; }
            this.SetStatus("Modeling for optimized MY-" + yrYear + " fleet ...");
            this.WriteLogLine("Modeling for optimized MY-" + yrYear + " fleet ...");
            this.WriteLogLine(CharAsterisk50);
            double[] optCoefs = this._optimizationData[yrYear - this._minYear].GetCoefficients();
            for (int i = 0; i < 4; i++) { fncCoef[i, yrIndex] = optCoefs[i]; }
            this.RunModelYearNonOptimized(scen, year);
        }
        void PreProcessModelYearForOptimization(Scenario scen, ModelYear year, int[] fncType, double[,] fncCoef,
            ref double aGpm, ref double bGpm, ref double lnrYGpm)
        {
            int yrIndex = year.Index;
            double coefOff = this._optSettings.Increment * this._optSettings.AboveOptimum;
            aGpm = 1 / fncCoef[0, yrIndex] + coefOff;
            bGpm = 1 / fncCoef[1, yrIndex] + coefOff;
            if (FunctionInformation.IsLinearFunction(fncType[yrIndex]))
            {   
                lnrYGpm = fncCoef[3, yrIndex] + coefOff;
            }
        }
        void PostProcessModelYearForOptimization(Scenario scen, ModelYear year, int[] fncType, double[,] fncCoef,
            ref bool restartYear)
        {
            if (this._abortRequested) { return; }
            int  yrIndex  = year.Index;
            int  yrOffset = yrIndex - this._minYearIndex;
            int  currType = fncType[yrIndex];
            bool isLinear = FunctionInformation.IsLinearFunction(currType);
            this.CalcOptimizedValues(yrIndex);
            if (this._optSettings.BackstopTargetFunction && yrIndex > this._minYearIndex)
            {   
                if (currType == fncType[yrIndex - 1])
                {   
                    double[] currCoef = this._optimizationData[yrOffset].GetCoefficients();
                    double[] prevCoef = new double[4];
                    for (int i = 0; i < 4; i++) { prevCoef[i] = fncCoef[i, yrIndex - 1]; }
                    bool boundsDecreased =
                        (Math.Round(currCoef[0], 1) < Math.Round(prevCoef[0], 1)) ||
                        (Math.Round(currCoef[1], 1) < Math.Round(prevCoef[1], 1)) ||
                        this.CheckCurvesCrossing(currCoef, prevCoef, currType, this._minFP, this._maxFP, 0.1);
                    if (boundsDecreased && (currCoef[2] == prevCoef[2] && (isLinear || currCoef[3] == prevCoef[3])))
                    {   
                        this.SetStatus("Backstopping MY-" + year.Year + " asymptotes ...");
                        this.WriteLogLine("Backstopping MY-" + year.Year + " asymptotes ...");
                        this.WriteLogLine(CharAsterisk50);
                        this._optimizationData[yrOffset].SetOptimizedIndexFromCoefs(Math.Round(prevCoef[0], 1),
                            Math.Round(prevCoef[1], 1));
                    }
                    else if (boundsDecreased)
                    {   
                        this._optimizationData[yrOffset].Clear();
                        double shapeOff = this._optSettings.Increment * (this._optimizationData[yrOffset - 1].Optimized -
                            this._optSettings.AboveOptimum);
                        fncCoef[0, yrIndex] = 1 / (1 / prevCoef[0] + shapeOff);
                        fncCoef[1, yrIndex] = 1 / (1 / prevCoef[1] + shapeOff);
                        fncCoef[2, yrIndex] = prevCoef[2];
                        fncCoef[3, yrIndex] = prevCoef[3] + ((isLinear) ? shapeOff : 0);
                        restartYear = true;
                        this.SetStatus("Backstopping MY-" + year.Year + " shape (a restart will be required) ...");
                        this.WriteLogLine("Backstopping MY-" + year.Year + " shape (a restart will be required) ...");
                        this.WriteLogLine(CharAsterisk50);
                        return;
                    }
                } 
            } 
            if (this._optSettings.StopOnFinedIterations)
            {   
                int finedIter = this.GetFinedIteration(this._optimizationData[yrOffset].Iterations);
                if (finedIter != -1)
                {   
                    IterationsData iterData = this._optimizationData[yrOffset];
                    if (finedIter < iterData.Optimized) { this._optimizationData[yrOffset].Optimized = finedIter; }
                }
            } 
        }
        void IterationLoop(Scenario scen, ModelYear year, int[] fncType, double[,] fncCoef, double aGpm, double bGpm,
            double lnrYGpm)
        {
            int cfExhaustedCount = 0;
            bool cfExhausted;
            for (int i = 0; i < this._iterationCount; i++)
            {
                this.PreProcessIteration(i, scen, year, fncType, fncCoef, aGpm, bGpm, lnrYGpm);
                base.RunModelYear(scen, year);
                this.PostProcessIteration(i, scen, year, fncCoef, out cfExhausted);
                if (cfExhausted) { cfExhaustedCount++; }
                if (cfExhaustedCount == 2) { return; }
                if (this._abortRequested) { return; }
            }
        }
        void PreProcessIteration(int iter, Scenario scen, ModelYear year, int[] fncType, double[,] fncCoef, double aGpm,
            double bGpm, double lnrYGpm)
        {
            int yrIndex = year.Index;
            this.SetStatus("Examining iteration: " + (iter + 1) + " / " + this._iterationCount);
            this.InitializeModelYear(scen, yrIndex);
            bool   isLinear  = FunctionInformation.IsLinearFunction(fncType[yrIndex]);
            double coefDelta = iter * this._optSettings.Increment;
            fncCoef[0, yrIndex] = Math.Round(1 / (aGpm - coefDelta), 1);
            fncCoef[1, yrIndex] = Math.Round(1 / (bGpm - coefDelta), 1);
            if (isLinear) { fncCoef[3, yrIndex] = lnrYGpm - coefDelta; }
        }
        void PostProcessIteration(int iter, Scenario scen, ModelYear year, double[,] fncCoef, out bool cfExhausted)
        {
            if (this._abortRequested) { cfExhausted = false; return; }
            int yrIndex = year.Index;
            Industry data = this.GetData(scen.Index, yrIndex);
            double[] coefs = new double[4];
            for (int i = 0; i < 4; i++) { coefs[i] = fncCoef[i, yrIndex]; }
            this.SaveEntry(iter, yrIndex, data, coefs);
            cfExhausted = true;
            ManufacturerCollection mfrs = data.Manufacturers;
            for (int i = 0; i < this._mfrCount; i++)
            {   
                if (mfrs[i].Description.Optimize)
                {   
                    if (!mfrs[i].ModelingData.CFExhausted) { cfExhausted = false; break; }
                }
            }
            if (iter + 1 == this._iterationCount) { this.WriteLogLine(CharAsterisk50); }
            else                                  { this.WriteLogLine(CharDash50    ); }
        }
        void InitializeModelYear(Scenario scen, int yrIndex)
        {
            int snIndex = scen.Index;
            if (yrIndex == this._minYearIndex)
            {   
                Industry data = this._data.Clone();
                base.PreProcessScenario(scen, data);
                this.SetData(snIndex, yrIndex, data);
            }
            else
            {   
                if (this._settings.OperatingModes.MultiYearModeling)
                {   
                    for (int i = this._minYearIndex; i < yrIndex; i++)
                    {
                        Industry optData = this._optimizedData[i - this._minYearIndex].Clone();
                        this.SetData(snIndex, i, optData);
                    }
                } 
            }
        }
        void SaveEntry(int iteration, int yrIndex, Industry data, double[] coefficients)
        {
            bool optAuto = (this._optSettings.Type == OptimizationType.PassengerAuto);
            bool optLt   = (this._optSettings.Type == OptimizationType.LightTruck);
            if (!optAuto && !optLt) { return; }
            int yrOffset = yrIndex - this._minYearIndex;    
            double sales = 0, stndSum = 0, cafeSum = 0;     
            if (this._stage == OptimizationStage.Baseline)
            {
                this._baselineData[yrOffset].Reset();
                this._baselineData[yrOffset].AdditionalIterationData = new IterationData[this._optMfrCount];
            }
            else if (this._stage == OptimizationStage.Industry)
            {
                this._optimizationData[yrOffset].Iterations[iteration].Reset();
                this._optimizationData[yrOffset].Iterations[iteration].AdditionalIterationData =
                    new IterationData[this._optMfrCount];
            }
            for (int i = 0, j = 0, mfrCount = data.ManufacturerCount; i < mfrCount; i++)
            {
                Manufacturer mfr = data.Manufacturers[i];
                if (!mfr.Description.Optimize) { continue; }
                IterationData id = new IterationData(); id.Valid = true;
                ManufacturerModelingData mmd = mfr.ModelingData;
                double mfrSales = 0, mfrStndSum = 0, mfrCafeSum = 0;
                if (optLt)
                {   
                    mfrSales   = mmd.Sales.LightTruck;                if (double.IsNaN(mfrSales  )) { mfrSales   = 0; }
                    mfrStndSum = mfrSales / mmd.Standard.LightTruck;  if (double.IsNaN(mfrStndSum)) { mfrStndSum = 0; }
                    mfrCafeSum = mfrSales / mmd.CAFE.LightTruck;      if (double.IsNaN(mfrCafeSum)) { mfrCafeSum = 0; }
                    id.Standard        = mfrSales / mfrStndSum;
                    id.CAFE            = mfrSales / mfrCafeSum;
                    id.TechCost        = mmd.TechCost          .LightTruck * 1e-6;
                    id.Fines           = mmd.Fines             .LightTruck * 1e-6;
                    id.Benefits        = mmd.SocialBenefits    .LightTruck * 1e-3;
                    id.DiscBenefits    = mmd.DiscSocialBenefits.LightTruck * 1e-3;
                    id.FuelConsumption = mmd.FuelConsumption   .LightTruck;
                    id.CO2Emissions    = mmd.CO2Emissions      .LightTruck;
                    id.CO2DiscCosts    = mmd.CO2DiscDamageCosts.LightTruck;
                }
                else
                {   
                    double daSales = mmd.Sales.DomesticAuto;        if (double.IsNaN(daSales)) { daSales = 0; }
                    double iaSales = mmd.Sales.ImportedAuto;        if (double.IsNaN(iaSales)) { iaSales = 0; }
                    double daStd   = mmd.Standard.DomesticAuto;     if (double.IsNaN(daStd  )) { daStd   = 0; }
                    double iaStd   = mmd.Standard.ImportedAuto;     if (double.IsNaN(iaStd  )) { iaStd   = 0; }
                    double daCafe  = mmd.CAFE.DomesticAuto;         if (double.IsNaN(daCafe )) { daCafe  = 0; }
                    double iaCafe  = mmd.CAFE.ImportedAuto;         if (double.IsNaN(iaCafe )) { iaCafe  = 0; }
                    mfrSales   = daSales + iaSales;
                    mfrStndSum = ((daStd  == 0) ? 0 : daSales / daStd ) + ((iaStd  == 0) ? 0 : iaSales / iaStd );
                    mfrCafeSum = ((daCafe == 0) ? 0 : daSales / daCafe) + ((iaCafe == 0) ? 0 : iaSales / iaCafe);
                    id.Standard        = mfrSales / mfrStndSum;
                    id.CAFE            = mfrSales / mfrCafeSum;
                    id.TechCost        = (mmd.TechCost          .DomesticAuto + mmd.TechCost          .ImportedAuto) * 1e-6;
                    id.Fines           = (mmd.Fines             .DomesticAuto + mmd.Fines             .ImportedAuto) * 1e-6;
                    id.Benefits        = (mmd.SocialBenefits    .DomesticAuto + mmd.SocialBenefits    .ImportedAuto) * 1e-3;
                    id.DiscBenefits    = (mmd.DiscSocialBenefits.DomesticAuto + mmd.DiscSocialBenefits.ImportedAuto) * 1e-3;
                    id.FuelConsumption = (mmd.FuelConsumption   .DomesticAuto + mmd.FuelConsumption   .ImportedAuto);
                    id.CO2Emissions    = (mmd.CO2Emissions      .DomesticAuto + mmd.CO2Emissions      .ImportedAuto);
                    id.CO2DiscCosts    = (mmd.CO2DiscDamageCosts.DomesticAuto + mmd.CO2DiscDamageCosts.ImportedAuto);
                }
                sales   += mfrSales;
                stndSum += mfrStndSum;
                cafeSum += mfrCafeSum;
                if (this._stage == OptimizationStage.Baseline)
                {   
                    this._baselineData[yrOffset].AdditionalIterationData[j] = id;
                }
                else if (this._stage == OptimizationStage.Industry)
                {   
                    IterationData baseId    = this._baselineData[yrOffset].AdditionalIterationData[j];
                    id.DeltaStandard        =  (id.Standard        - baseId.Standard);
                    id.DeltaCAFE            =  (id.CAFE            - baseId.CAFE);
                    id.DeltaTechCost        =  (id.TechCost        - baseId.TechCost);
                    id.DeltaFines           =  (id.Fines           - baseId.Fines);
                    id.DeltaBenefits        = -(id.Benefits        - baseId.Benefits);
                    id.DeltaDiscBenefits    = -(id.DiscBenefits    - baseId.DiscBenefits);
                    id.DeltaFuelConsumption =  (id.FuelConsumption - baseId.FuelConsumption);
                    id.DeltaCO2Emissions    =  (id.CO2Emissions    - baseId.CO2Emissions);
                    id.DeltaCO2DiscCosts    =  (id.CO2DiscCosts    - baseId.CO2DiscCosts);
                    id.FuelSavings = baseId.FuelConsumption - id.FuelConsumption;
                    id.CO2EmissionSavings = baseId.CO2Emissions - id.CO2Emissions;
                    id.CO2MonetizedSavings = baseId.CO2DiscCosts - id.CO2DiscCosts;
                    id.NetBenefits =  (id.DeltaDiscBenefits - id.DeltaTechCost);
                    id.BCRatio     =  (id.DeltaDiscBenefits / id.DeltaTechCost);
                    this._optimizationData[yrOffset].Iterations[iteration].AdditionalIterationData[j] = id;
                }
                j++;
            } 
            if (this._stage == OptimizationStage.Baseline)
            {   
                this._baselineData[yrOffset].Valid    = true;
                this._baselineData[yrOffset].Standard = sales / stndSum;
                this._baselineData[yrOffset].CAFE     = sales / cafeSum;
                for (int i = 0; i < this._mfrCount; i++)
                {
                    this._baselineData[yrOffset].TechCost        += this._baselineData[yrOffset].AdditionalIterationData[i].TechCost;
                    this._baselineData[yrOffset].Fines           += this._baselineData[yrOffset].AdditionalIterationData[i].Fines;
                    this._baselineData[yrOffset].Benefits        += this._baselineData[yrOffset].AdditionalIterationData[i].Benefits;
                    this._baselineData[yrOffset].DiscBenefits    += this._baselineData[yrOffset].AdditionalIterationData[i].DiscBenefits;
                    this._baselineData[yrOffset].FuelConsumption += this._baselineData[yrOffset].AdditionalIterationData[i].FuelConsumption;
                    this._baselineData[yrOffset].CO2Emissions    += this._baselineData[yrOffset].AdditionalIterationData[i].CO2Emissions;
                    this._baselineData[yrOffset].CO2DiscCosts    += this._baselineData[yrOffset].AdditionalIterationData[i].CO2DiscCosts;
                }
            }
            else if (this._stage == OptimizationStage.Industry)
            {   
                IterationData id = this._optimizationData[yrOffset].Iterations[iteration];
                id.Valid    = true;
                id.Standard = sales / stndSum;
                id.CAFE     = sales / cafeSum;
                for (int i = 0; i < this._mfrCount; i++)
                {
                    id.TechCost        += id.AdditionalIterationData[i].TechCost;
                    id.Fines           += id.AdditionalIterationData[i].Fines;
                    id.Benefits        += id.AdditionalIterationData[i].Benefits;
                    id.DiscBenefits    += id.AdditionalIterationData[i].DiscBenefits;
                    id.FuelConsumption += id.AdditionalIterationData[i].FuelConsumption;
                    id.CO2Emissions    += id.AdditionalIterationData[i].CO2Emissions;
                    id.CO2DiscCosts    += id.AdditionalIterationData[i].CO2DiscCosts;
                    id.FuelSavings         += id.AdditionalIterationData[i].FuelSavings;
                    id.CO2EmissionSavings  += id.AdditionalIterationData[i].CO2EmissionSavings;
                    id.CO2MonetizedSavings += id.AdditionalIterationData[i].CO2MonetizedSavings;
                }
                id.DeltaStandard        =  (id.Standard        - this._baselineData[yrOffset].Standard);
                id.DeltaCAFE            =  (id.CAFE            - this._baselineData[yrOffset].CAFE);
                id.DeltaTechCost        =  (id.TechCost        - this._baselineData[yrOffset].TechCost);
                id.DeltaFines           =  (id.Fines           - this._baselineData[yrOffset].Fines);
                id.DeltaBenefits        = -(id.Benefits        - this._baselineData[yrOffset].Benefits);
                id.DeltaDiscBenefits    = -(id.DiscBenefits    - this._baselineData[yrOffset].DiscBenefits);
                id.DeltaFuelConsumption =  (id.FuelConsumption - this._baselineData[yrOffset].FuelConsumption);
                id.DeltaCO2Emissions    =  (id.CO2Emissions    - this._baselineData[yrOffset].CO2Emissions);
                id.DeltaCO2DiscCosts    =  (id.CO2DiscCosts    - this._baselineData[yrOffset].CO2DiscCosts);
                if (this._optSettings.DiscardMfrsWithFines)
                {   
                    for (int i = 0; i < this._mfrCount; i++)
                    {
                        if (id.AdditionalIterationData[i].Fines == 0)
                        {
                            id.NetBenefits += id.AdditionalIterationData[iteration].NetBenefits;
                        }
                    }
                }
                else
                {   
                    id.NetBenefits = (id.DeltaDiscBenefits - id.DeltaTechCost);
                    if (this._optSettings.CountFinesTowardNB)
                    {   
                        id.NetBenefits -= id.DeltaFines;
                    }
                }
                id.BCRatio = (id.DeltaDiscBenefits / id.DeltaTechCost);
                id.Coefficients = new double[4];
                Array.Copy(coefficients, 0, id.Coefficients, 0, 4);
                this._optimizationData[yrOffset].Iterations[iteration] = id;
            }
        }
        void CalcOptimizedValues(int yrIndex)
        {
            int yrOffset = yrIndex - this._minYearIndex;
            IterationData[] iterations = this._optimizationData[yrOffset].Iterations;
            this.CalcMBCRatios(iterations);
            int optIdx = 0;
            if      (this._optSettings.Mode == OptimizationMode.MaximumNetBenefits) { optIdx = this.GetMaxNetBenefits (iterations); }
            else if (this._optSettings.Mode == OptimizationMode.ZeroNetBeneftis   ) { optIdx = this.GetZeroNetBenefits(iterations); }
            else if (this._optSettings.Mode == OptimizationMode.AverageOfPeaks    ) { optIdx = this.GetAverageOfPeaks (iterations, yrOffset); }
            this.SaveOptimizedData(yrOffset, optIdx, this._optimizationData);
        }
        void CalcMBCRatios(IterationData[] iterations)
        {
            int smoothOff = 1;
            int iterationCount = this._iterationCount - smoothOff;      
            for (int i = 1; i < iterationCount; i++)
            {   
                IterationData next = iterations[i + smoothOff];     
                IterationData prev = iterations[i - 1];             
                if (!next.Valid || !prev.Valid) { continue; }
                iterations[i].MBCRatio = this.CalcMBCRatiosHelper(ref next, ref prev);
                for (int j = 0; j < this._mfrCount; j++)
                {
                    iterations[i].AdditionalIterationData[j].MBCRatio = this.CalcMBCRatiosHelper(
                        ref next.AdditionalIterationData[j],        
                        ref prev.AdditionalIterationData[j]);       
                }
            } 
        }
        double CalcMBCRatiosHelper(ref IterationData next, ref IterationData prev)
        {
            double benefits = next.DeltaDiscBenefits - prev.DeltaDiscBenefits;
            double techCost = next.DeltaTechCost - prev.DeltaTechCost;
            return (Math.Abs(techCost) < 1 || (benefits < 0 && techCost > 0)) ? double.NaN : (benefits / techCost);
        }
        int GetMaxNetBenefits(IterationData[] iterations)
        {
            int max = 0;    
            for (int i = 0, count = iterations.Length; i < count; i++)
            {
                if (iterations[i].Valid)
                {   
                    if (iterations[max].NetBenefits < iterations[i].NetBenefits)
                    {   
                        max = i;
                    }
                }
            } 
            return (iterations[max].Valid) ? max : -1;    
        }
        int GetZeroNetBenefits(IterationData[] iterations)
        {
            int max = this.GetMaxNetBenefits(iterations);
            if (max == - 1) { return -1; }
            int min = max, lastValid = max;
            for (int i = max, count = iterations.Length; i < count; i++)
            {
                if (iterations[i].Valid)
                {   
                    if (iterations[i].NetBenefits < 0) { break; }
                    if (iterations[i].NetBenefits >= 0 && iterations[i].NetBenefits < iterations[min].NetBenefits)
                    {   
                        min = i;
                    }
                }
            } 
            return (iterations[min].Valid) ? min : -1;    
        }
        int GetAverageOfPeaks(IterationData[] iterations, int yrOffset)
        {
            int[] max = new int[this._mfrCount];
            for (int i = 0, count = iterations.Length; i < count; i++)
            {
                if (iterations[i].Valid)
                {   
                    for (int j = 0; j < this._mfrCount; j++)
                    {
                        if (iterations[max[j]].AdditionalIterationData[j].NetBenefits <
                            iterations[i].AdditionalIterationData[j].NetBenefits)
                        {   
                            max[j] = i;
                        }
                    } 
                }
            } 
            for (int i = 0; i < this._mfrCount; i++)
            {
                if(max[i] == 0)
                {   
                    for (int j = 0, count = iterations.Length; j < count; j++)
                    {
                        if (iterations[j].AdditionalIterationData[i].Fines > 0) { break; }
                        max[i] = j;
                    }
                }
            } 
            double average = 0;
            double sales   = 0;
            for (int i = 0; i < this._mfrCount; i++)
            {
                double mfrSales = this._mfrSales[i][yrOffset];
                sales   += mfrSales;
                average += mfrSales * max[i];
            }
            return (int)(average / sales);
        }
        void SaveOptimizedData(int yrOffset, int optIdx, IterationsData[] data)
        {
            data[yrOffset].Intersections = new int[0];
            data[yrOffset].PosAve        = double.NaN;
            data[yrOffset].NegAve        = double.NaN;
            if      (optIdx == -1 && yrOffset > 0) { data[yrOffset].Optimized = data[yrOffset - 1].Optimized; }
            else if (optIdx == -1                ) { data[yrOffset].Optimized = -1; }
            else if (this._optSettings.BackstopStandard && yrOffset > 0 && data[yrOffset - 1].Optimized >= 0 &&
                data[yrOffset].Iterations[optIdx].Standard < data[yrOffset - 1].GetStandard())
            {
                data[yrOffset].SetOptimizedIndexFromStnd(data[yrOffset - 1].GetStandard());
            }
            else { data[yrOffset].Optimized = optIdx; }
        }
        int GetFinedIteration(IterationData[] iterations)
        {
            for (int i = 0, count = iterations.Length; i < count; i++)
            {
                if (iterations[i].Fines > 0.0) { return i - 1; }    
            }
            return -1;
        }
        bool CheckCurvesCrossing(double[] currCoef, double[] prevCoef, int fncType, double fpStart, double fpEnd,
            double fpIncrement)
        {
            for (double fp = fpStart; fp < fpEnd; fp += fpIncrement)
            {   
                double currTarget = Standards.GetTarget(fp, 0, fncType, currCoef);
                double prevTarget = Standards.GetTarget(fp, 0, fncType, prevCoef);
                if (currTarget > prevTarget) { return true; }
            }
            return false;
        }
        void SetStatus(string status)
        {
            this._status = status;
        }
        void ClearStatus()
        {
            this._status = "";
        }
        void WriteLogLine(string value)
        {
            Scenario scen = this._scens[0];
            for (int i = 0; i < this._mfrCount; i++)
            {
                this._logWriter.WriteComplianceLine   (scen, this._years[0], this._data.Manufacturers[i], value);
                this._logWriter.WriteExtComplianceLine(scen, this._years[0], this._data.Manufacturers[i], value);
            }
        }
        #endregion
        #endregion
        #region 
        #region 
        public override OptimizationData OptimizationData
        {
            get { return new OptimizationData(null, this._optimizationData); }
        }
        public override IModelingProgress Progress
        {
            get
            {
                if (this.Running)
                {
                    if (this._stage == OptimizationStage.Baseline) { return base.Progress; }
                    else if (this._stage == OptimizationStage.Industry)
                    {
                        return new ComplianceProgress(this._scens, this._years, this._mfrs, this._status);
                    }
                }
                return null;
            }
        }
        #endregion
        #endregion
        #region 
        const string CharAsterisk50 = "**************************************************";
        const string CharDash50     = "--------------------------------------------------";
        OptimizationSettings _optSettings;
        int _optMinYear;
        int _optMfrCount;
        double _minFP, _maxFP;
        OptimizationStage _stage;
        string _status;
        double[][] _mfrSales;                   
        IterationData [] _baselineData;         
        IterationsData[] _optimizationData;     
        int              _iterationCount;
        Industry[] _optimizedData;
        #endregion
    }
}

