using System;
using Volpe.Cafe;
using Volpe.Cafe.Collections;
using Volpe.Cafe.Data;
using AdjUE = Volpe.Cafe.Parameters.AdjustedUpstreamEmissions;
namespace Volpe.Cafe.Model
{
    [Serializable]
    public class Effects
    {
        #region 
        private Effects(Scenario scenario, ModelYear year, Industry data, ModelingSettings settings)
        {
            this._data     = data;
            this._settings = settings;
            this._isBaseline = scenario.IsBaseline;
            this._myIdx      = year.Index;
            this._mYear      = year.Year;
            this._mfrCount   = data.ManufacturerCount;
            Parameters parameters = settings.Parameters;
            Parameters.EmissionDamageCosts emissionCosts = parameters.EmissionCosts;
            this._dmgCostCO  = emissionCosts.CO ;
            this._dmgCostVOC = emissionCosts.VOC;
            this._dmgCostNOX = emissionCosts.NOX;
            this._dmgCostPM  = emissionCosts.PM ;
            this._dmgCostSOX = emissionCosts.SOX;
            this._dmgCostCO2 = emissionCosts.CO2;
            this._dmgCostCO2Pct     = emissionCosts.CO2Pct;
            this._dmgCostCO2RefYear = emissionCosts.CO2RefYear;
            Parameters.EconomicDataValues  economicValues = parameters.EconomicValues;
            Parameters.ExternalCostsValues externalCosts  = parameters.ExternalCosts;
            Parameters.EconomicCostsValues economicCosts  = parameters.EconomicCosts;
            this._testVsOnRoadGap        = economicValues.TestVsOnRoadGap;
            this._reboundEffect          = economicValues.ReboundEffect;
            this._discountRate           = economicValues.DiscountRate;
            this._vehicleTravelTimeValue = economicValues.VehicleTravelTimeValue;
            this._congestion             = externalCosts .Congestion;
            this._accident               = externalCosts .Accident;
            this._noise                  = externalCosts .Noise;
            this._monopsonyCost          = economicCosts .Monopsony;
            this._priceShockCost         = economicCosts .PriceShock;
            this._militarySecurityCost   = economicCosts .MilitarySecurity;
            this._convGasMix             = parameters.GasolineMix.GasolineConventional;
        }
        #endregion
        #region 
        public static void Process(Scenario scenario, ModelYear year, Industry data, ModelingSettings settings)
        {
            Effects eff = new Effects(scenario, year, data, settings);
            eff.CalculateAggregatedEffects();
        }
        private void CalculateVehicleEffects(int mfrIdx, AggregateEffectsData ied, ManufacturerModelingData mmd,
            VehicleCollection vc)
        {
            Parameters     parameters = this._settings.Parameters;
            EmissionsRates emissions  = this._settings.EmissionsRates;
            Estimates      fuelPriceEstimates = this._settings.OperatingModes.FuelPriceEstimates;
            for (int i = 0; i < vc.Count; i++)
            {    
                Vehicle veh = vc[i];
                string fuel = veh.Engine.Description.Fuel;
                double fuelEconomy = veh.Description.FuelEconomy;
                double fuelCapacity = veh.Description.FuelCapacity;
                double sales = veh.Description.Sales[this._myIdx];
                Mobile6Class m6class = veh.Mobile6Class;
                RegulatoryClass regClass = veh.RegClass;
                double[] survRate = parameters.GetVehicleSurvivability(veh);
                double[] milesDrv = parameters.GetVehicleMilesDriven(veh, fuelPriceEstimates);
                double[] fuelPrice, fuelTaxArr, preTaxFuelArr;
                double fuelLowImp, fuelReducedRef, domReducedRef, impReducedRef;
                double massDensity, carbonContent, soxRate;
                AdjUE adjCO, adjVOC, adjNOX, adjPM, adjSOX, adjCO2;
                this.InitEffectsByFuel(fuel, out fuelPrice, out fuelTaxArr, out preTaxFuelArr, out fuelLowImp,
                    out fuelReducedRef, out domReducedRef, out impReducedRef, out massDensity, out carbonContent,
                    out soxRate, out adjCO, out adjVOC, out adjNOX, out adjPM, out adjSOX, out adjCO2);
                double[] coRate  = this.GetEmissionsRates(emissions.CO , m6class, this._convGasMix, this._myIdx);
                double[] vocRate = this.GetEmissionsRates(emissions.VOC, m6class, this._convGasMix, this._myIdx);
                double[] noxRate = this.GetEmissionsRates(emissions.NOX, m6class, this._convGasMix, this._myIdx);
                double[] pmRate  = this.GetEmissionsRates(emissions.PM , m6class, this._convGasMix, this._myIdx);
                double baselineFe = (this._isBaseline) ? fuelEconomy :
                    this._data.ModelingData.BaselineInfo[this._myIdx].FinalFuelEconomy[mfrIdx][i];
                ied.Sales      [m6class] += sales;
                ied.FuelEconomy[m6class] += (sales / veh.Description.FuelEconomy);      
                for (int j = 0; j < Global.CY; j++)
                {
                    double fuelPrice_j  = fuelPrice [Math.Min(this._myIdx + j, fuelPrice .Length - 1)];
                    double fuelTax_j    = fuelTaxArr[Math.Min(this._myIdx + j, fuelTaxArr.Length - 1)];
                    double preTaxFuel_j = preTaxFuelArr[Math.Min(this._myIdx + j, preTaxFuelArr.Length - 1)];
                    double discRatePow  = Math.Pow(1 + this._discountRate, -j);
                    double fleet = sales * survRate[j];
                    double cpm         = fuelPrice_j / fuelEconomy * (1 - this._testVsOnRoadGap);
                    double baselineCpm = fuelPrice_j / baselineFe  * (1 - this._testVsOnRoadGap);
                    double baselineVmt = fleet * milesDrv[j] * 1e-3;
                    double vmt;     
                    if (this._isBaseline) { vmt = baselineVmt; }
                    else
                    {   
                        vmt = baselineVmt + (baselineVmt * ((cpm / baselineCpm) - 1) * this._reboundEffect);
                    }
                    double gallons             = vmt / (fuelEconomy * (1 - this._testVsOnRoadGap));
                    double fuelTax             = gallons * fuelTax_j;
                    double discFuelTax         = fuelTax * discRatePow;
                    double preTaxFuel          = gallons * preTaxFuel_j;
                    double discPreTaxFuel      = preTaxFuel * discRatePow;
                    double driveSurplus        = -0.5 * (baselineVmt * ((cpm / baselineCpm) - 1) * this._reboundEffect) * (baselineCpm - cpm) + ((baselineVmt - vmt) * cpm);
                    double discDriveSurplus    = driveSurplus * discRatePow;
                    const double refuelPct     = 0.55;
                    const double refuelTime    = 5D / 60D;
                    double refuelSurplus       = -((milesDrv[j] / (refuelPct * fuelCapacity * (1 - this._testVsOnRoadGap) * baselineFe)) - (milesDrv[j] * (1 + ((cpm / baselineCpm) - 1) * this._reboundEffect) / (refuelPct * fuelCapacity * (1 - this._testVsOnRoadGap) * fuelEconomy))) * fleet * (refuelTime * this._vehicleTravelTimeValue * 1e-3);
                    double discRefuelSurplus   = refuelSurplus * discRatePow;
                    double congestionCosts     = vmt * this._congestion;
                    double discCongestionCosts = congestionCosts * discRatePow;
                    double accidentCosts       = vmt * this._accident;
                    double discAccidentCosts   = accidentCosts * discRatePow;
                    double noiseCosts          = vmt * this._noise;
                    double discNoiseCosts      = noiseCosts * discRatePow;
                    double economicCosts       = gallons * (fuelLowImp + fuelReducedRef * impReducedRef) * (this._monopsonyCost + this._priceShockCost + this._militarySecurityCost);
                    double discEconomicCosts   = economicCosts * discRatePow;
                    double coEmissions  = this.CalculateEmission(domReducedRef, fuelReducedRef, adjCO , gallons, vmt, coRate , this._myIdx, j);
                    double vocEmissions = this.CalculateEmission(domReducedRef, fuelReducedRef, adjVOC, gallons, vmt, vocRate, this._myIdx, j);
                    double noxEmissions = this.CalculateEmission(domReducedRef, fuelReducedRef, adjNOX, gallons, vmt, noxRate, this._myIdx, j);
                    double pmEmissions  = this.CalculateEmission(domReducedRef, fuelReducedRef, adjPM , gallons, vmt, pmRate , this._myIdx, j);
                    double soxEmissions = ((fuelReducedRef * (domReducedRef * (adjSOX.Extraction + adjSOX.Transportation) + adjSOX.Refining) + adjSOX.TSD) + soxRate) * gallons * Effects.EmFract;
                    double co2Emissions = ((adjCO2.Extraction + adjCO2.Transportation + adjCO2.Refining + adjCO2.TSD) + (massDensity * carbonContent * Effects.FourtyFourTwelfths)) * gallons * 1e-9;
                    double coDamage, coDiscDamage, vocDamage, vocDiscDamage, noxDamage, noxDiscDamage, pmDamage, pmDiscDamage, soxDamage, soxDiscDamage, co2Damage, co2DiscDamage;
                    this.CalculateEmissionCosts(coEmissions , this._dmgCostCO , discRatePow, out coDamage , out coDiscDamage , 1e-3);
                    this.CalculateEmissionCosts(vocEmissions, this._dmgCostVOC, discRatePow, out vocDamage, out vocDiscDamage, 1e-3);
                    this.CalculateEmissionCosts(noxEmissions, this._dmgCostNOX, discRatePow, out noxDamage, out noxDiscDamage, 1e-3);
                    this.CalculateEmissionCosts(pmEmissions , this._dmgCostPM , discRatePow, out pmDamage , out pmDiscDamage , 1e-3);
                    this.CalculateEmissionCosts(soxEmissions, this._dmgCostSOX, discRatePow, out soxDamage, out soxDiscDamage, 1e-3);
                    this.CalculateEmissionCosts(co2Emissions, this._dmgCostCO2 * Math.Pow(1 + this._dmgCostCO2Pct, this._mYear - this._dmgCostCO2RefYear + j), discRatePow, out co2Damage, out co2DiscDamage, 1e3);
                    ied.Vmt                  [j][m6class] += vmt;
                    ied.Gallons              [j][m6class] += gallons;
                    ied.FuelTax              [j][m6class] += fuelTax;
                    ied.DiscFuelTax          [j][m6class] += discFuelTax;
                    ied.PreTaxFuel           [j][m6class] += preTaxFuel;
                    ied.DiscPreTaxFuel       [j][m6class] += discPreTaxFuel;
                    ied.DriveSurplus         [j][m6class] += driveSurplus;
                    ied.DiscDriveSurplus     [j][m6class] += discDriveSurplus;
                    ied.RefuelSurplus        [j][m6class] += refuelSurplus;
                    ied.DiscRefuelSurplus    [j][m6class] += discRefuelSurplus;
                    ied.CongestionCosts      [j][m6class] += congestionCosts;
                    ied.DiscCongestionCosts  [j][m6class] += discCongestionCosts;
                    ied.AccidentCosts        [j][m6class] += accidentCosts;
                    ied.DiscAccidentCosts    [j][m6class] += discAccidentCosts;
                    ied.NoiseCosts           [j][m6class] += noiseCosts;
                    ied.DiscNoiseCosts       [j][m6class] += discNoiseCosts;
                    ied.EconomicCosts        [j][m6class] += economicCosts;
                    ied.DiscEconomicCosts    [j][m6class] += discEconomicCosts;
                    ied.COEmissions          [j][m6class] += coEmissions;
                    ied.CODamage             [j][m6class] += coDamage;
                    ied.CODiscDamage         [j][m6class] += coDiscDamage;
                    ied.VOCEmissions         [j][m6class] += vocEmissions;
                    ied.VOCDamage            [j][m6class] += vocDamage;
                    ied.VOCDiscDamage        [j][m6class] += vocDiscDamage;
                    ied.NOXEmissions         [j][m6class] += noxEmissions;
                    ied.NOXDamage            [j][m6class] += noxDamage;
                    ied.NOXDiscDamage        [j][m6class] += noxDiscDamage;
                    ied.PMEmissions          [j][m6class] += pmEmissions;
                    ied.PMDamage             [j][m6class] += pmDamage;
                    ied.PMDiscDamage         [j][m6class] += pmDiscDamage;
                    ied.SOXEmissions         [j][m6class] += soxEmissions;
                    ied.SOXDamage            [j][m6class] += soxDamage;
                    ied.SOXDiscDamage        [j][m6class] += soxDiscDamage;
                    ied.CO2Emissions         [j][m6class] += co2Emissions;
                    ied.CO2Damage            [j][m6class] += co2Damage;
                    ied.CO2DiscDamage        [j][m6class] += co2DiscDamage;
                    ied.RcGallons            [regClass] += gallons;
                    ied.RcPreTaxFuel         [regClass] += preTaxFuel;
                    ied.RcDiscPreTaxFuel     [regClass] += discPreTaxFuel;
                    ied.RcDriveSurplus       [regClass] += driveSurplus;
                    ied.RcDiscDriveSurplus   [regClass] += discDriveSurplus;
                    ied.RcRefuelSurplus      [regClass] += refuelSurplus;
                    ied.RcDiscRefuelSurplus  [regClass] += discRefuelSurplus;
                    ied.RcCongestionCosts    [regClass] += congestionCosts;
                    ied.RcDiscCongestionCosts[regClass] += discCongestionCosts;
                    ied.RcAccidentCosts      [regClass] += accidentCosts;
                    ied.RcDiscAccidentCosts  [regClass] += discAccidentCosts;
                    ied.RcNoiseCosts         [regClass] += noiseCosts;
                    ied.RcDiscNoiseCosts     [regClass] += discNoiseCosts;
                    ied.RcEconomicCosts      [regClass] += economicCosts;
                    ied.RcDiscEconomicCosts  [regClass] += discEconomicCosts;
                    ied.RcCODamage           [regClass] += coDamage;
                    ied.RcCODiscDamage       [regClass] += coDiscDamage;
                    ied.RcVOCDamage          [regClass] += vocDamage;
                    ied.RcVOCDiscDamage      [regClass] += vocDiscDamage;
                    ied.RcNOXDamage          [regClass] += noxDamage;
                    ied.RcNOXDiscDamage      [regClass] += noxDiscDamage;
                    ied.RcPMDamage           [regClass] += pmDamage;
                    ied.RcPMDiscDamage       [regClass] += pmDiscDamage;
                    ied.RcSOXDamage          [regClass] += soxDamage;
                    ied.RcSOXDiscDamage      [regClass] += soxDiscDamage;
                    ied.RcCO2Damage          [regClass] += co2Damage;
                    ied.RcCO2DiscDamage      [regClass] += co2DiscDamage;
                    mmd.FuelConsumption   [regClass] += gallons;
                    mmd.SocialBenefits    [regClass] += (preTaxFuel + driveSurplus + refuelSurplus +
                        economicCosts + congestionCosts + accidentCosts + noiseCosts +
                        co2Damage + coDamage  + vocDamage + noxDamage + pmDamage + soxDamage);
                    mmd.DiscSocialBenefits[regClass] += (discPreTaxFuel + discDriveSurplus + discRefuelSurplus +
                        discEconomicCosts + discCongestionCosts + discAccidentCosts + discNoiseCosts +
                        co2DiscDamage + coDiscDamage + vocDiscDamage + noxDiscDamage + pmDiscDamage + soxDiscDamage);
                    mmd.CO2Emissions      [regClass] += co2Emissions;
                    mmd.CO2DamageCosts    [regClass] += co2Damage;
                    mmd.CO2DiscDamageCosts[regClass] += co2DiscDamage;
                } 
            } 
            ied.RcSocialBenefits = ied.RcPreTaxFuel + ied.RcDriveSurplus + ied.RcRefuelSurplus +
                ied.RcEconomicCosts + ied.RcCongestionCosts + ied.RcAccidentCosts + ied.RcNoiseCosts +
                ied.RcCO2Damage + ied.RcCODamage + ied.RcVOCDamage + ied.RcNOXDamage + ied.RcPMDamage + ied.RcSOXDamage;
            ied.RcDiscSocialBenefits = ied.RcDiscPreTaxFuel + ied.RcDiscDriveSurplus + ied.RcDiscRefuelSurplus +
                ied.RcDiscEconomicCosts + ied.RcDiscCongestionCosts + ied.RcDiscAccidentCosts + ied.RcDiscNoiseCosts +
                ied.RcCO2DiscDamage + ied.RcCODiscDamage + ied.RcVOCDiscDamage + ied.RcNOXDiscDamage + ied.RcPMDiscDamage +
                ied.RcSOXDiscDamage;
        }
        private double CalculateEmission(double domReducedRef, double fuelReducedRef, AdjUE adjEm,
            double gallons, double vmt, double[] rate, int myIdx, int cyIdx)
        {
            return ((fuelReducedRef * (domReducedRef * (adjEm.Extraction + adjEm.Transportation) + adjEm.Refining) + adjEm.TSD) *
                gallons + vmt * rate[Math.Min(cyIdx + myIdx, rate.Length - 1)]) * Effects.EmFract;
        }
        private void CalculateEmissionCosts(double emission, double dmgCost, double discRatePowMy, out double damage,
            out double discDamage, double factor)
        {
            damage = emission * dmgCost * factor;
            discDamage = damage * discRatePowMy;
        }
        private void InitEffectsByFuel(string fuel, out double[] fuelPrice, out double[] fuelTax, out double[] preTaxFuel,
            out double fuelLowImp, out double fuelReducedRef, out double domReducedRef, out double impReducedRef,
            out double massDensity, out double carbonContent, out double soxRate, out AdjUE adjCO, out AdjUE adjVOC,
            out AdjUE adjNOX, out AdjUE adjPM, out AdjUE adjSOX, out AdjUE adjCO2)
        {
            Estimates  fuelPriceEstimates = this._settings.OperatingModes.FuelPriceEstimates;
            Parameters parameters         = this._settings.Parameters;
            switch (fuel)
            {
                case "D":
                    fuelPrice      = parameters.ForecastData.GetDieselFuelPrice(fuelPriceEstimates);
                    fuelTax        = parameters.ForecastData.DieselFuelTax;
                    preTaxFuel     = parameters.ForecastData.GetDieselPreTaxFuelPrice(fuelPriceEstimates);
                    fuelLowImp     = parameters.DieselImportAssumptions.FuelSavingsLeadingToLowerImports;
                    fuelReducedRef = parameters.DieselImportAssumptions.FuelSavingsLeadingToReducedRefining;
                    domReducedRef  = parameters.DieselImportAssumptions.ReducedRefiningFromDomesticCrude;
                    impReducedRef  = parameters.DieselImportAssumptions.ReducedRefiningFromImportedCrude;
                    adjCO          = parameters.CODieselEmissions;
                    adjVOC         = parameters.VOCDieselEmissions;
                    adjNOX         = parameters.NOXDieselEmissions;
                    adjPM          = parameters.PMDieselEmissions;
                    adjSOX         = parameters.SOXDieselEmissions;
                    adjCO2         = parameters.CO2DieselEmissions;
                    massDensity    = parameters.DieselMassDensity;
                    carbonContent  = parameters.DieselCarbonContent;
                    soxRate        = parameters.DieselSOXRate;
                    break;
                case "E85":
                    fuelPrice      = parameters.ForecastData.GetE85FuelPrice(fuelPriceEstimates);
                    fuelTax        = parameters.ForecastData.E85FuelTax;
                    preTaxFuel     = parameters.ForecastData.GetE85PreTaxFuelPrice(fuelPriceEstimates);
                    fuelLowImp     = parameters.E85ImportAssumptions.FuelSavingsLeadingToLowerImports;
                    fuelReducedRef = parameters.E85ImportAssumptions.FuelSavingsLeadingToReducedRefining;
                    domReducedRef  = parameters.E85ImportAssumptions.ReducedRefiningFromDomesticCrude;
                    impReducedRef  = parameters.E85ImportAssumptions.ReducedRefiningFromImportedCrude;
                    adjCO          = parameters.COE85Emissions;
                    adjVOC         = parameters.VOCE85Emissions;
                    adjNOX         = parameters.NOXE85Emissions;
                    adjPM          = parameters.PME85Emissions;
                    adjSOX         = parameters.SOXE85Emissions;
                    adjCO2         = parameters.CO2E85Emissions;
                    massDensity    = parameters.E85MassDensity;
                    carbonContent  = parameters.E85CarbonContent;
                    soxRate        = parameters.E85SOXRate;
                    break;
                case "CNG":
                    fuelPrice      = parameters.ForecastData.GetCNGFuelPrice(fuelPriceEstimates);
                    fuelTax        = parameters.ForecastData.CNGFuelTax;
                    preTaxFuel     = parameters.ForecastData.GetCNGPreTaxFuelPrice(fuelPriceEstimates);
                    fuelLowImp     = parameters.CNGImportAssumptions.FuelSavingsLeadingToLowerImports;
                    fuelReducedRef = parameters.CNGImportAssumptions.FuelSavingsLeadingToReducedRefining;
                    domReducedRef  = parameters.CNGImportAssumptions.ReducedRefiningFromDomesticCrude;
                    impReducedRef  = parameters.CNGImportAssumptions.ReducedRefiningFromImportedCrude;
                    adjCO          = parameters.COCNGEmissions;
                    adjVOC         = parameters.VOCCNGEmissions;
                    adjNOX         = parameters.NOXCNGEmissions;
                    adjPM          = parameters.PMCNGEmissions;
                    adjSOX         = parameters.SOXCNGEmissions;
                    adjCO2         = parameters.CO2CNGEmissions;
                    massDensity    = parameters.CNGMassDensity;
                    carbonContent  = parameters.CNGCarbonContent;
                    soxRate        = parameters.CNGSOXRate;
                    break;
                case "H":
                    fuelPrice      = parameters.ForecastData.GetHydrogenFuelPrice(fuelPriceEstimates);
                    fuelTax        = parameters.ForecastData.HydrogenFuelTax;
                    preTaxFuel     = parameters.ForecastData.GetHydrogenPreTaxFuelPrice(fuelPriceEstimates);
                    fuelLowImp     = parameters.HydrogenImportAssumptions.FuelSavingsLeadingToLowerImports;
                    fuelReducedRef = parameters.HydrogenImportAssumptions.FuelSavingsLeadingToReducedRefining;
                    domReducedRef  = parameters.HydrogenImportAssumptions.ReducedRefiningFromDomesticCrude;
                    impReducedRef  = parameters.HydrogenImportAssumptions.ReducedRefiningFromImportedCrude;
                    adjCO          = parameters.COHydrogenEmissions;
                    adjVOC         = parameters.VOCHydrogenEmissions;
                    adjNOX         = parameters.NOXHydrogenEmissions;
                    adjPM          = parameters.PMHydrogenEmissions;
                    adjSOX         = parameters.SOXHydrogenEmissions;
                    adjCO2         = parameters.CO2HydrogenEmissions;
                    massDensity    = parameters.HydrogenMassDensity;
                    carbonContent  = parameters.HydrogenCarbonContent;
                    soxRate        = parameters.HydrogenSOXRate;
                    break;
                case "SP1":
                    fuelPrice      = parameters.ForecastData.GetSpare1FuelPrice(fuelPriceEstimates);
                    fuelTax        = parameters.ForecastData.Spare1FuelTax;
                    preTaxFuel     = parameters.ForecastData.GetSpare1PreTaxFuelPrice(fuelPriceEstimates);
                    fuelLowImp     = parameters.Spare1ImportAssumptions.FuelSavingsLeadingToLowerImports;
                    fuelReducedRef = parameters.Spare1ImportAssumptions.FuelSavingsLeadingToReducedRefining;
                    domReducedRef  = parameters.Spare1ImportAssumptions.ReducedRefiningFromDomesticCrude;
                    impReducedRef  = parameters.Spare1ImportAssumptions.ReducedRefiningFromImportedCrude;
                    adjCO          = parameters.COSpare1Emissions;
                    adjVOC         = parameters.VOCSpare1Emissions;
                    adjNOX         = parameters.NOXSpare1Emissions;
                    adjPM          = parameters.PMSpare1Emissions;
                    adjSOX         = parameters.SOXSpare1Emissions;
                    adjCO2         = parameters.CO2Spare1Emissions;
                    massDensity    = parameters.Spare1MassDensity;
                    carbonContent  = parameters.Spare1CarbonContent;
                    soxRate        = parameters.Spare1SOXRate;
                    break;
                case "SP2":
                    fuelPrice      = parameters.ForecastData.GetSpare2FuelPrice(fuelPriceEstimates);
                    fuelTax        = parameters.ForecastData.Spare2FuelTax;
                    preTaxFuel     = parameters.ForecastData.GetSpare2PreTaxFuelPrice(fuelPriceEstimates);
                    fuelLowImp     = parameters.Spare2ImportAssumptions.FuelSavingsLeadingToLowerImports;
                    fuelReducedRef = parameters.Spare2ImportAssumptions.FuelSavingsLeadingToReducedRefining;
                    domReducedRef  = parameters.Spare2ImportAssumptions.ReducedRefiningFromDomesticCrude;
                    impReducedRef  = parameters.Spare2ImportAssumptions.ReducedRefiningFromImportedCrude;
                    adjCO          = parameters.COSpare2Emissions;
                    adjVOC         = parameters.VOCSpare2Emissions;
                    adjNOX         = parameters.NOXSpare2Emissions;
                    adjPM          = parameters.PMSpare2Emissions;
                    adjSOX         = parameters.SOXSpare2Emissions;
                    adjCO2         = parameters.CO2Spare2Emissions;
                    massDensity    = parameters.Spare2MassDensity;
                    carbonContent  = parameters.Spare2CarbonContent;
                    soxRate        = parameters.Spare2SOXRate;
                    break;
                default:
                    fuelPrice      = parameters.ForecastData.GetGasolineFuelPrice(fuelPriceEstimates);
                    fuelTax        = parameters.ForecastData.GasolineFuelTax;
                    preTaxFuel     = parameters.ForecastData.GetGasolinePreTaxFuelPrice(fuelPriceEstimates);
                    fuelLowImp     = parameters.GasolineImportAssumptions.FuelSavingsLeadingToLowerImports;
                    fuelReducedRef = parameters.GasolineImportAssumptions.FuelSavingsLeadingToReducedRefining;
                    domReducedRef  = parameters.GasolineImportAssumptions.ReducedRefiningFromDomesticCrude;
                    impReducedRef  = parameters.GasolineImportAssumptions.ReducedRefiningFromImportedCrude;
                    adjCO          = parameters.COGasolineEmissions;
                    adjVOC         = parameters.VOCGasolineEmissions;
                    adjNOX         = parameters.NOXGasolineEmissions;
                    adjPM          = parameters.PMGasolineEmissions;
                    adjSOX         = parameters.SOXGasolineEmissions;
                    adjCO2         = parameters.CO2GasolineEmissions;
                    massDensity    = parameters.GasolineMassDensity;
                    carbonContent  = parameters.GasolineCarbonContent;
                    soxRate        = parameters.GasolineSOXRate;
                    break;
            }
        }
        private double[] GetEmissionsRates(EmissionsRates.Emissions em, Mobile6Class m6class, double convGasMix, int myIdx)
        {
            int ubound = em.Diesel.Ldv.GetUpperBound(0) + 1;
            double[,] emRate = null;
            double[,] gasRfg = null;
            double[] rate = new double[ubound];
            switch (m6class)
            {
                case Mobile6Class.Ldgv:   emRate = em.Gasoline.Ldv;   gasRfg = em.GasolineRfg.Ldv;   break;
                case Mobile6Class.Ldgt1:  emRate = em.Gasoline.Ldt1;  gasRfg = em.GasolineRfg.Ldt1;  break;
                case Mobile6Class.Ldgt2:  emRate = em.Gasoline.Ldt2;  gasRfg = em.GasolineRfg.Ldt2;  break;
                case Mobile6Class.Hdgv2b: emRate = em.Gasoline.Hdv2b; gasRfg = em.GasolineRfg.Hdv2b; break;
                case Mobile6Class.Lddv:   emRate = em.Diesel.Ldv;   break;
                case Mobile6Class.Lddt1:  emRate = em.Diesel.Ldt1;  break;
                case Mobile6Class.Lddt2:  emRate = em.Diesel.Ldt2;  break;
                case Mobile6Class.Hddv2b: emRate = em.Diesel.Hdv2b; break;
            }
            if (gasRfg != null)
            {   
                for (int i = 0; i < ubound; i++)
                {
                    rate[i] = emRate[myIdx, i] * convGasMix + gasRfg[myIdx, i] * (1 - convGasMix);
                }
            }
            else if (emRate != null)
            {   
                for (int i = 0; i < ubound; i++)
                {
                    rate[i] = emRate[myIdx, i];
                }
            }
            return rate;
        }
        private void CalculateAggregatedEffects()
        {
            AggregateEffectsData ied = this._data.ModelingData.EffectsData; 
            ied.Reset();
            ManufacturerCollection mfrs = this._data.Manufacturers;
            for (int i = 0; i < this._mfrCount; i++)
            {
                Manufacturer mfr = mfrs[i];
                ManufacturerModelingData mmd = mfr.ModelingData;            
                VehicleCollection vc = mfr.ModelYearVehicles;
                mmd.FuelConsumption   .Clear();
                mmd.SocialBenefits    .Clear();
                mmd.DiscSocialBenefits.Clear();
                mmd.CO2Emissions      .Clear();
                mmd.CO2DamageCosts    .Clear();
                mmd.CO2DiscDamageCosts.Clear();
                this.CalculateVehicleEffects(mfr.Index, ied, mmd, vc);
            } 
            for (int i = 1; i <= 8; i++)
            {
                Mobile6Class m6 = (Mobile6Class)i;
                double fe = ied.FuelEconomy[m6];
                ied.FuelEconomy[m6] = (fe == 0) ? 0 : ied.Sales[m6] / fe;
            }
        }
        #endregion
        #region 
        private const double EmFract = 1e6 / 908e6;
        private const double OneTwelfth = 1D / 12D;
        private const double FourtyFourTwelfths = 44D / 12D;
        Industry         _data;
        ModelingSettings _settings;
        bool   _isBaseline;
        int    _myIdx, _mYear;
        int    _mfrCount;
        double _dmgCostCO, _dmgCostVOC, _dmgCostNOX, _dmgCostPM, _dmgCostSOX, _dmgCostCO2;
        double _dmgCostCO2Pct;
        int    _dmgCostCO2RefYear;
        double _testVsOnRoadGap, _reboundEffect, _discountRate, _vehicleTravelTimeValue;
        double _congestion, _accident, _noise;
        double _monopsonyCost, _priceShockCost, _militarySecurityCost, _convGasMix;
        #endregion
    }
}

