using System;
using System.Drawing;
using System.Windows.Forms;
using System.Runtime.InteropServices;
namespace Volpe.Cafe.Ui
{
    public sealed class LabelButton : Label
    {
        #region 
        public new event EventHandler Click;
        #endregion
        #region 
        public LabelButton()
            : base()
        {
            this.BorderStyle = BorderStyle.FixedSingle;
            this.TextAlign = ContentAlignment.MiddleCenter;
            this.ImageAlign = ContentAlignment.MiddleCenter;
            this.Size = new Size(80, 20);
            this._themed = false;
            this._imageTag = "";
            this._useLargeSet = false;
        }
        #endregion
        #region 
        public void ThemeInitialize(string imageTag, bool useLargeSet)
        {
            this.BorderStyle = BorderStyle.None;
            this._themed = true;        
            this._imageTag = imageTag.Trim();
            this._useLargeSet = useLargeSet;
            this.ThemeRefresh();
            Cafe2d.Themes.ThemeIndexChanged += new EventHandler(this.ThemeIndexChanged);
            Cafe2d.Themes.ThemeAlternativeChanged += new EventHandler(this.ThemeAlternativeChanged);
        }
        void ThemeRefresh()
        {
            Theme theme = Cafe2d.Themes.SelectedTheme;
            int alter = Cafe2d.Themes.ThemeAlternative;
            if (alter >= theme.Alternatives) { alter = 0; }     
            if (theme.Scaling)
            {
                this.Image = (this._useLargeSet) ?
                    theme.X32[alter].GetImage(this._imageTag) :
                    theme.X22[alter].GetImage(this._imageTag);
            }
            else if (theme.Rollover)
            {
                this.Image = (this._useLargeSet) ?
                    theme.BaseLarge[alter].GetImage(this._imageTag) :
                    theme.BaseSmall[alter].GetImage(this._imageTag);
            }
        }
        void ThemeIndexChanged(object sender, EventArgs e)
        {
            this.ThemeRefresh();
        }
        void ThemeAlternativeChanged(object sender, EventArgs e)
        {
            this.ThemeRefresh();
        }
        public override void Refresh()
        {
            this.ProcessMouseLeave();
            base.Refresh();
        }
        protected override void WndProc(ref Message m)
        {
            IntPtr lParam = m.LParam;
            int x = Win32.LoWord(lParam.ToInt32());
            int y = Win32.HiWord(lParam.ToInt32());
            switch ((Win32.Messages)m.Msg)
            {
                case Win32.Messages.WM_MOUSEMOVE:
                    this.ProcessMouseMove(x, y);
                    break;
                case Win32.Messages.WM_MOUSELEAVE:
                    this.ProcessMouseLeave();
                    break;
                case Win32.Messages.WM_LBUTTONDOWN:
                    this.ProcessLMouseDown(x, y);
                    break;
                case Win32.Messages.WM_LBUTTONUP:
                    this.ProcessLMouseUp(x, y);
                    break;
            }
            try
            {
                base.WndProc(ref m);
            } catch (Exception ex) { Console.WriteLine(ex.ToString()); }
        }
        void ProcessMouseMove(int x, int y)
        {
            if (this.IsOutOfBounds(x, y))
            {   
                this.ProcessMouseLeave();
            }
            else if (!this._mouseHover)
            {
                this.SuspendLayout();
                this._mouseHover = true;
                this.BorderStyle = BorderStyle.FixedSingle;
                this.BackColor =  Color.Lavender;
                if (this._themed) { this.ThemeMouseEnter(); }
                this.ResumeLayout(true);
            }
        }
        void ProcessMouseLeave()
        {
            if (this._mouseHover)
            {
                this.SuspendLayout();
                this._mouseHover = false;
                this._mouseDown  = false;
                this.BorderStyle = (this._themed) ? BorderStyle.None : BorderStyle.FixedSingle;
                this.BackColor   = Control.DefaultBackColor;
                if (this._themed) { this.ThemeMouseLeave(); }
                this.ResumeLayout(true);
            }
        }
        void ProcessLMouseDown(int x, int y)
        {
            if (!this._mouseDown)
            {
                this._mouseDown = true;
                this.BorderStyle = BorderStyle.Fixed3D;
            }
        }
        void ProcessLMouseUp(int x, int y)
        {
            if (this._mouseDown)
            {
                this._mouseDown = false;
                this.BorderStyle = BorderStyle.FixedSingle;
                this.ProcessMouseLeave();
                if (this.Click != null)
                {   
                    this.Click(this, new MouseEventArgs(MouseButtons.Left, 1, x, y, 0));
                }
            }
        }
        void ThemeMouseEnter()
        {
            Theme theme = Cafe2d.Themes.SelectedTheme;
            int   alter = Cafe2d.Themes.ThemeAlternative;
            if (theme.Scaling)
            {
                if (!this._useLargeSet)
                {
                    this.SetBounds(this.Left - 4, this.Top - 4, this.Width + 8, this.Height + 8);
                    this.Image = theme.X32[alter].GetImage(this._imageTag);
                    this.BringToFront();
                }
            }
            else if (theme.Rollover)
            {
                this.Image = (this._useLargeSet) ?
                    theme.AltLarge[alter].GetImage(this._imageTag) :
                    theme.AltSmall[alter].GetImage(this._imageTag);
            }
        }
        void ThemeMouseLeave()
        {
            Theme theme = Cafe2d.Themes.SelectedTheme;
            int   alter = Cafe2d.Themes.ThemeAlternative;
            if (theme.Scaling)
            {
                if (!this._useLargeSet)
                {
                    this.SetBounds(this.Left + 4, this.Top + 4, this.Width - 8, this.Height - 8, BoundsSpecified.All);
                    this.Image = theme.X22[alter].GetImage(this._imageTag);
                }
            }
            else if (theme.Rollover)
            {
                this.Image = (this._useLargeSet) ?
                    theme.BaseLarge[alter].GetImage(this._imageTag) :
                    theme.BaseSmall[alter].GetImage(this._imageTag);
            }
        }
        bool IsOutOfBounds(Point pt)
        {
            return this.IsOutOfBounds(pt.X, pt.Y);
        }
        bool IsOutOfBounds(int x, int y)
        {
            int xl = 0, yl = 0, xr = this.Width, yr = this.Height;
            if      (this.BorderStyle == BorderStyle.FixedSingle) { xl--; yl--; xr+=2; yr+=2; }
            else if (this.BorderStyle == BorderStyle.Fixed3D    ) { xl--; yl--; }
            return (x < xl || y < yl || x > xr || y > yr);
        }
        #endregion
        #region 
        public new bool Enabled
        {
            get { return base.Enabled; }
            set
            {
                if (value != base.Enabled)
                {
                    if (value == false) { this.ProcessMouseLeave(); }
                    base.Enabled = value;
                }
            }
        }
        #endregion
        #region 
        bool   _themed;
        string _imageTag;
        bool   _useLargeSet;
        bool _mouseDown  = false;
        bool _mouseHover = false;
        #endregion
    }
}

