using System;
using System.ComponentModel;
using System.Drawing;
using System.IO;
using System.Threading;
using System.Windows.Forms;
using Volpe.Cafe.Data;
using Volpe.Cafe.IO;
using Volpe.Cafe.Ui;
namespace Volpe.Cafe.Ui.Panels
{
    public class InputPanel : Panel
    {
        #region 
        private class VerifierEventArgs : EventArgs
        {
            #region 
            public VerifierEventArgs(string filename, DataFileType filetype, bool complete)
                : base()
            {
                this._filename = filename;
                this._filetype = filetype;
                this._complete = complete;
            }
            #endregion
            #region 
            public string Filename
            {
                get
                {
                    return this._filename;
                }
            }
            public DataFileType Filetype
            {
                get
                {
                    return this._filetype;
                }
            }
            public bool Complete
            {
                get
                {
                    return this._complete;
                }
            }
            #endregion
            #region 
            private string _filename;
            private DataFileType _filetype;
            private bool _complete;
            #endregion
        }
        private abstract class AbstractVerifier
        {
            #region 
            protected AbstractVerifier()
            {
                this._errors = "";
            }
            #endregion
            #region 
            protected void Start(string threadName, ThreadStart threadStart)
            {
                this._thread = new Thread(threadStart);
                this._thread.Name = threadName;
                this._thread.Start();
            }
            #endregion
            #region 
            public object Tag
            {
                get
                {
                    return this._tag;
                }
                set
                {
                    this._tag = value;
                }
            }
            public string Errors
            {
                get
                {
                    return this._errors;
                }
            }
            #endregion
            #region 
            protected object _tag;
            protected string _errors;
            protected static Mutex _mutex = new Mutex(false);
            protected Thread _thread;
            #endregion
        }
        private class FileVerifier : AbstractVerifier
        {
            #region 
            public event EventHandler VerificationComplete;
            #endregion
            #region 
            public FileVerifier(string filename, string password, params string[] desiredTitles)
                : base()
            {
                if (desiredTitles.Length < 1)
                {
                    throw new ArgumentOutOfRangeException("desiredTitles");
                }
                this._filename = filename;
                this._password = password;
                this._titles = desiredTitles;
                this._verified = false;
            }
            #endregion
            #region 
            public void Start()
            {
                this.Start("FileVerifier::" + this._titles[0],
                    new ThreadStart(this.StartThread));
            }
            private void StartThread()
            {
                FileVerifier._mutex.WaitOne();
                this._verified = FileVerifier.Verify(this._filename, this._password,
                    out this._errors, this._titles);
                if (this.VerificationComplete != null)
                {
                    VerificationComplete(this, EventArgs.Empty);
                }
                FileVerifier._mutex.ReleaseMutex();
            }
            public static bool Verify(string filename, string password, out string errors,
                params string[] desiredTitles)
            {
                filename = filename.Trim();
                if (filename == null || filename == "")
                {
                    errors = "The path provided for " + desiredTitles[0] + " file is blank.";
                    return false;
                }
                else if (!File.Exists(filename))
                {
                    errors = "The specified " + desiredTitles[0] + " file cannot be found on disk.";
                    return false;
                }
                else
                {
                    try
                    {
                        Input input = new Input(filename, password);
                        input.Close();
                        return FileVerifier.Verify(input, out errors, desiredTitles);
                    }
                    catch (Exception ex)
                    {
                        errors = "The specified " + desiredTitles[0] + " file cannot be loaded: "
                            + ex.Message;
                        return false;
                    }
                }
            }
            public static bool Verify(string filename, string password,
                params string[] desiredTitles)
            {
                string errors;    
                return FileVerifier.Verify(filename, password, out errors, desiredTitles);
            }
            public static bool Verify(Input inputFile, out string errors,
                params string[] desiredTitles)
            {
                errors = "";
                bool found = false;
                for (int i = 0; i < desiredTitles.Length; i++)
                {
                    if (inputFile.Compare(inputFile.Title, desiredTitles[i], true))
                    {
                        found = true;
                        break;
                    }
                }
                if (!found)
                {
                    errors = "The specified " + desiredTitles[0] + " file has incorrect " +
                        "workbook title information.";
                }
                return found;
            }
            public static bool Verify(Input inputFile, params string[] desiredTitles)
            {
                string errors;    
                return FileVerifier.Verify(inputFile, out errors, desiredTitles);
            }
            #endregion
            #region 
            public string Filename
            {
                get
                {
                    return this._filename;
                }
            }
            public bool Verified
            {
                get
                {
                    return this._verified;
                }
            }
            #endregion
            #region 
            private string _filename;
            private string _password;
            private string[] _titles;
            private bool _verified;
            #endregion
        }
        private class FileParser : AbstractVerifier
        {
            #region 
            public event VerifierEventHandler ParsingStart;
            public event VerifierEventHandler ParsingComplete;
            public event VerifierEventHandler Parsed;
            #endregion
            #region 
            public FileParser(string path, string password, bool recurseDirs, bool keepExisting,
                string[] existingPaths, bool[] ignorePaths)
                : this(new string[] {path}, password, recurseDirs, keepExisting, existingPaths, ignorePaths)
            {
            }
            public FileParser(string[] paths, string password, bool recurseDirs, bool keepExisting,
                string[] existingPaths, bool[] ignorePaths)
                : base()
            {
                this._paths = paths;
                this._password = password;
                this._recurseDirs = recurseDirs;
                this._keepExisting = keepExisting;
                this._existingPaths = existingPaths;
                this._ignorePaths = ignorePaths;
                this._errors = "";
            }
            #endregion
            #region 
            public void Start()
            {
                this.Start("FileParser::" + DateTime.Now.ToString(), new ThreadStart(this.StartThread));
            }
            private void StartThread()
            {
                FileParser._mutex.WaitOne();
                this.ParseFiles();
                if (this.Parsed != null)
                {
                    this.Parsed(this, new VerifierEventArgs("", DataFileType.None, true));
                }
                FileParser._mutex.ReleaseMutex();
            }
            private void ParseFiles()
            {
                bool[] filesFound = new bool[5];
                if (this._keepExisting)
                {    
                    if (File.Exists(this._existingPaths[0])) { filesFound[0] = true; }
                    if (File.Exists(this._existingPaths[1])) { filesFound[1] = true; }
                    if (File.Exists(this._existingPaths[2])) { filesFound[2] = true; }
                    if (File.Exists(this._existingPaths[3])) { filesFound[3] = true; }
                    if (File.Exists(this._existingPaths[4])) { filesFound[4] = true; }
                }
                this.ParseFiles_Helper(this._paths, ref filesFound);
            }
            private void ParseFiles_Helper(string[] paths, ref bool[] filesFound)
            {
                for (int i = 0; i < paths.Length; i++)
                {    
                    if (filesFound[0] && filesFound[1] && filesFound[2] && filesFound[3] && filesFound[4]) { return; }
                    string path = paths[i];
                    if (Directory.Exists(path))
                    {    
                        string[] files = Directory.GetFiles(path, "*.xls");
                        this.ParseFiles_Helper(files, ref filesFound);
                        if (this._recurseDirs)
                        {
                            string[] dirs = Directory.GetDirectories(path);
                            this.ParseFiles_Helper(dirs, ref filesFound);
                        }
                    }
                    else if (File.Exists(path))
                    {    
                        this.ParseFile(path, ref filesFound);
                    }
                    else
                    {
                        this._errors += "The specified location \"" + path + "\" cannot be found on disk.\r\n\r\n";
                    }
                }
            }
            private void ParseFile(string filename, ref bool[] filesFound)
            {
                if (this.ParsingStart != null)
                {
                    this.ParsingStart(this, new VerifierEventArgs(filename, DataFileType.None, false));
                }
                DataFileType filetype = DataFileType.None;
                try
                {
                    Input input = new Input(filename, this._password);
                    input.Close();
                    if (!this._ignorePaths[0] && FileVerifier.Verify(input, "CAFE Market Data"))
                    {
                        if (!filesFound[(int)DataFileType.MarketData])
                        {
                            this._existingPaths[(int)DataFileType.MarketData] = filename;
                            filesFound[(int)DataFileType.MarketData] = true;
                            filetype = DataFileType.MarketData;
                        }
                    }
                    else if (!this._ignorePaths[1] && FileVerifier.Verify(input, "CAFE Technologies"))
                    {
                        if (!filesFound[(int)DataFileType.Technologies])
                        {
                            this._existingPaths[(int)DataFileType.Technologies] = filename;
                            filesFound[(int)DataFileType.Technologies] = true;
                            filetype = DataFileType.Technologies;
                        }
                    }
                    else if (!this._ignorePaths[2] && FileVerifier.Verify(input, "CAFE Parameters"))
                    {
                        if (!filesFound[(int)DataFileType.Parameters])
                        {
                            this._existingPaths[(int)DataFileType.Parameters] = filename;
                            filesFound[(int)DataFileType.Parameters] = true;
                            filetype = DataFileType.Parameters;
                        }
                    }
                    else if (!this._ignorePaths[3] && FileVerifier.Verify(input, "CAFE Emissions Rates", "CAFE Emissions"))
                    {
                        if (!filesFound[(int)DataFileType.EmissionsRates])
                        {
                            this._existingPaths[(int)DataFileType.EmissionsRates] = filename;
                            filesFound[(int)DataFileType.EmissionsRates] = true;
                            filetype = DataFileType.EmissionsRates;
                        }
                    }
                    else if (!this._ignorePaths[4] && FileVerifier.Verify(input, "CAFE Scenarios"))
                    {
                        if (!filesFound[(int)DataFileType.Scenarios])
                        {
                            this._existingPaths[(int)DataFileType.Scenarios] = filename;
                            filesFound[(int)DataFileType.Scenarios] = true;
                            filetype = DataFileType.Scenarios;
                        }
                    }
                    else
                    {
                        this._errors += "The \"" + filename + "\" file cannot be parsed.\r\n\r\n";
                    }
                }
                catch (Exception ex)
                {
                    this._errors += "The \"" + filename + "\" file cannot be parsed: " +
                        ex.Message + "\r\n\r\n";
                }
                if (this.ParsingComplete != null)
                {
                    this.ParsingComplete(this, new VerifierEventArgs(filename, filetype, true));
                }
            }
            #endregion
            #region 
            public string[] ExistingPaths
            {
                get
                {
                    return this._existingPaths;
                }
            }
            #endregion
            #region 
            private string[] _paths;
            private string _password;
            private bool _recurseDirs;
            private bool _keepExisting;
            private string[] _existingPaths;
            private bool[] _ignorePaths;
            #endregion
        }
        private class FileLoader : AbstractVerifier
        {
            #region 
            public event VerifierEventHandler LoadingStart;
            public event VerifierEventHandler LoadingComplete;
            public event VerifierEventHandler Loaded;
            #endregion
            #region 
            public FileLoader(string[] fileNames, string password, bool[] filesLoaded, bool verifyFiles,
                bool[] ignorePaths)
                : base()
            {
                this._fileNames = fileNames;
                this._password = password;
                this._filesLoaded = filesLoaded;
                this._verify = verifyFiles;
                this._data = new object[fileNames.Length];
                this._ignorePaths = ignorePaths;
                this._errors = "";
            }
            #endregion
            #region 
            public void Start()
            {
                this.Start("FileLoader::" + DateTime.Now.ToString(), new ThreadStart(this.StartThread));
            }
            private void StartThread()
            {
                FileParser._mutex.WaitOne();
                this.LoadFiles();
                if (this.Loaded != null)
                {
                    this.Loaded(this, new VerifierEventArgs("", DataFileType.None, true));
                }
                FileParser._mutex.ReleaseMutex();
            }
            private void LoadFiles()
            {
                for (int i = 0; i < this._fileNames.Length; i++)
                {    
                    if (this._ignorePaths[i]) { continue; }
                    if (this.LoadingStart != null)
                    {
                        this.LoadingStart(this, new VerifierEventArgs(this._fileNames[i], (DataFileType)i, false));
                    }
                    if (!this._filesLoaded[i])
                    {
                        bool verified = true;
                        if (this._verify)
                        {
                            verified = FileVerifier.Verify(this._fileNames[i], this._password,
                                out this._errors, InputPanel.DesiredTitles[i]);
                        }
                        if (verified)
                        {
                            this._filesLoaded[i] = true;
                            try
                            {
                                if (i == (int)DataFileType.MarketData)
                                {
                                    this._data[i] = new Industry(this._fileNames[i], this._password);
                                }
                                else if (i == (int)DataFileType.Technologies)
                                {
                                    this._data[i] = new Technologies(this._fileNames[i], this._password);
                                }
                                else if (i == (int)DataFileType.Parameters)
                                {
                                    this._data[i] = new Parameters(this._fileNames[i], this._password);
                                }
                                else if (i == (int)DataFileType.EmissionsRates)
                                {
                                    this._data[i] = new EmissionsRates(this._fileNames[i], this._password);
                                }
                                else if (i == (int)DataFileType.Scenarios)
                                {
                                    this._data[i] = new Scenarios(this._fileNames[i], this._password);
                                }
                                else
                                {
                                    this._errors += "The file \"" + this._fileNames[i] + "\" cannot be loaded.";
                                    this._filesLoaded[i] = false;
                                }
                            }
                            catch (InputException ex)
                            {
                                this._errors += ex.Message;
                                this._filesLoaded[i] = false;
                            }
                            catch (Exception ex)
                            {
                                this._errors += "The file \"" + this._fileNames[i] + "\" cannot be loaded: " +
                                    ex.Message;
                                this._filesLoaded[i] = false;
                            }
                        }
                    }
                    if (this.LoadingComplete != null)
                    {
                        this.LoadingComplete(this, new VerifierEventArgs(this._fileNames[i], (DataFileType)i, true));
                    }
                }
            }
            #endregion
            #region 
            public bool[] FilesLoaded
            {
                get
                {
                    return this._filesLoaded;
                }
            }
            public object[] Data
            {
                get
                {
                    return this._data;
                }
            }
            #endregion
            #region 
            private string[] _fileNames;
            private string _password;
            private bool[] _filesLoaded;
            private bool _verify;
            private object[] _data;
            private bool[] _ignorePaths;
            #endregion
        }
        #endregion
        #region 
        private delegate void VerifierEventHandler(object sender, VerifierEventArgs e);
        #endregion
        #region 
        public event InputStateChangedEventHandler InputStateChanged;
        public event EventHandler FilesLoaded;
        #endregion
        #region 
        public InputPanel()
            : base()
        {
            this.InitializeComponent();
            this._data = new object[5];
            this._fileNames = new string[5];
            this._filesLoaded = new bool[5];
            this._verifying = false;
            this._messages = new Messages();
        }
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                if (components != null)
                {
                    components.Dispose();
                }
            }
            base.Dispose(disposing);
        }
        #endregion
        #region 
        #region 
        private void InitializeComponent()
        {
            this.ux_browseLabel = new System.Windows.Forms.Label();
            this.ux_marketDataLabel = new System.Windows.Forms.Label();
            this.ux_marketDataTextBox = new System.Windows.Forms.TextBox();
            this.ux_marketDataLabelButton = new Volpe.Cafe.Ui.LabelButton();
            this.ux_technologiesLabel = new System.Windows.Forms.Label();
            this.ux_technologiesTextBox = new System.Windows.Forms.TextBox();
            this.ux_technologiesLabelButton = new Volpe.Cafe.Ui.LabelButton();
            this.ux_parametersLabel = new System.Windows.Forms.Label();
            this.ux_parametersTextBox = new System.Windows.Forms.TextBox();
            this.ux_parametersLabelButton = new Volpe.Cafe.Ui.LabelButton();
            this.ux_emissionsLabel = new System.Windows.Forms.Label();
            this.ux_emissionsTextBox = new System.Windows.Forms.TextBox();
            this.ux_emissionsLabelButton = new Volpe.Cafe.Ui.LabelButton();
            this.ux_scenariosLabel = new System.Windows.Forms.Label();
            this.ux_scenariosLabelButton = new Volpe.Cafe.Ui.LabelButton();
            this.ux_scenariosTextBox = new System.Windows.Forms.TextBox();
            this.ux_reloadAllFromDiskCheckBox = new CheckBox();
            this.SuspendLayout();
            this.ux_browseLabel.Anchor = AnchorStyles.Top | AnchorStyles.Right;
            this.ux_browseLabel.AutoSize = true;
            this.ux_browseLabel.Font = new System.Drawing.Font(this.Font.Name, 8.25F,
                System.Drawing.FontStyle.Underline);
            this.ux_browseLabel.Location = new System.Drawing.Point(312, 5);
            this.ux_browseLabel.Name = "ux_browseLabel";
            this.ux_browseLabel.Size = new System.Drawing.Size(41, 17);
            this.ux_browseLabel.TabIndex = 0;
            this.ux_browseLabel.Text = "Browse";
            this.ux_marketDataLabel.AutoSize = true;
            this.ux_marketDataLabel.Location = new System.Drawing.Point(8, 8);
            this.ux_marketDataLabel.Name = "ux_marketDataLabel";
            this.ux_marketDataLabel.Size = new System.Drawing.Size(121, 17);
            this.ux_marketDataLabel.TabIndex = 1;
            this.ux_marketDataLabel.Text = "Select Market-Data file:";
            this.ux_marketDataTextBox.Anchor = AnchorStyles.Top | AnchorStyles.Left | AnchorStyles.Right;
            this.ux_marketDataTextBox.BorderStyle = BorderStyle.FixedSingle;
            this.ux_marketDataTextBox.Location = new System.Drawing.Point(8, 24);
            this.ux_marketDataTextBox.Name = "ux_marketDataTextBox";
            this.ux_marketDataTextBox.Size = new System.Drawing.Size(312, 21);
            this.ux_marketDataTextBox.TabIndex = 2;
            this.ux_marketDataTextBox.Text = "";
            this.ux_marketDataTextBox.TextChanged += new System.EventHandler(this.InputFileTextBox_TextChanged);
            this.ux_marketDataLabelButton.Anchor = AnchorStyles.Top | AnchorStyles.Right;
            this.ux_marketDataLabelButton.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
            this.ux_marketDataLabelButton.Location = new System.Drawing.Point(320, 22);
            this.ux_marketDataLabelButton.Name = "ux_marketDataLabelButton";
            this.ux_marketDataLabelButton.Size = new System.Drawing.Size(24, 24);
            this.ux_marketDataLabelButton.TabIndex = 3;
            this.ux_marketDataLabelButton.TextAlign = System.Drawing.ContentAlignment.MiddleCenter;
            this.ux_marketDataLabelButton.Click += new System.EventHandler(this.InputFileLabelButton_Click);
            this.ux_technologiesLabel.AutoSize = true;
            this.ux_technologiesLabel.Location = new System.Drawing.Point(8, 48);
            this.ux_technologiesLabel.Name = "ux_technologiesLabel";
            this.ux_technologiesLabel.Size = new System.Drawing.Size(125, 17);
            this.ux_technologiesLabel.TabIndex = 4;
            this.ux_technologiesLabel.Text = "Select Technologies file:";
            this.ux_technologiesTextBox.Anchor = AnchorStyles.Top | AnchorStyles.Left | AnchorStyles.Right;
            this.ux_technologiesTextBox.BorderStyle = BorderStyle.FixedSingle;
            this.ux_technologiesTextBox.Location = new System.Drawing.Point(8, 64);
            this.ux_technologiesTextBox.Name = "ux_technologiesTextBox";
            this.ux_technologiesTextBox.Size = new System.Drawing.Size(312, 21);
            this.ux_technologiesTextBox.TabIndex = 5;
            this.ux_technologiesTextBox.Text = "";
            this.ux_technologiesTextBox.TextChanged += new System.EventHandler(this.InputFileTextBox_TextChanged);
            this.ux_technologiesLabelButton.Anchor = AnchorStyles.Top | AnchorStyles.Right;
            this.ux_technologiesLabelButton.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
            this.ux_technologiesLabelButton.Location = new System.Drawing.Point(320, 62);
            this.ux_technologiesLabelButton.Name = "ux_technologiesLabelButton";
            this.ux_technologiesLabelButton.Size = new System.Drawing.Size(24, 24);
            this.ux_technologiesLabelButton.TabIndex = 6;
            this.ux_technologiesLabelButton.TextAlign = System.Drawing.ContentAlignment.MiddleCenter;
            this.ux_technologiesLabelButton.Click += new System.EventHandler(this.InputFileLabelButton_Click);
            this.ux_parametersLabel.AutoSize = true;
            this.ux_parametersLabel.Location = new System.Drawing.Point(8, 88);
            this.ux_parametersLabel.Name = "ux_parametersLabel";
            this.ux_parametersLabel.Size = new System.Drawing.Size(116, 17);
            this.ux_parametersLabel.TabIndex = 7;
            this.ux_parametersLabel.Text = "Select Parameters file:";
            this.ux_parametersTextBox.Anchor = AnchorStyles.Top | AnchorStyles.Left | AnchorStyles.Right;
            this.ux_parametersTextBox.BorderStyle = BorderStyle.FixedSingle;
            this.ux_parametersTextBox.Location = new System.Drawing.Point(8, 104);
            this.ux_parametersTextBox.Name = "ux_parametersTextBox";
            this.ux_parametersTextBox.Size = new System.Drawing.Size(312, 21);
            this.ux_parametersTextBox.TabIndex = 8;
            this.ux_parametersTextBox.Text = "";
            this.ux_parametersTextBox.TextChanged += new System.EventHandler(this.InputFileTextBox_TextChanged);
            this.ux_parametersLabelButton.Anchor = AnchorStyles.Top | AnchorStyles.Right;
            this.ux_parametersLabelButton.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
            this.ux_parametersLabelButton.Location = new System.Drawing.Point(320, 102);
            this.ux_parametersLabelButton.Name = "ux_parametersLabelButton";
            this.ux_parametersLabelButton.Size = new System.Drawing.Size(24, 24);
            this.ux_parametersLabelButton.TabIndex = 9;
            this.ux_parametersLabelButton.TextAlign = System.Drawing.ContentAlignment.MiddleCenter;
            this.ux_parametersLabelButton.Click += new System.EventHandler(this.InputFileLabelButton_Click);
            this.ux_emissionsLabel.AutoSize = true;
            this.ux_emissionsLabel.Location = new System.Drawing.Point(8, 128);
            this.ux_emissionsLabel.Name = "ux_emissionsLabel";
            this.ux_emissionsLabel.Size = new System.Drawing.Size(140, 17);
            this.ux_emissionsLabel.TabIndex = 10;
            this.ux_emissionsLabel.Text = "Select Emissions-Rates file:";
            this.ux_emissionsTextBox.Anchor = AnchorStyles.Top | AnchorStyles.Left | AnchorStyles.Right;
            this.ux_emissionsTextBox.BorderStyle = BorderStyle.FixedSingle;
            this.ux_emissionsTextBox.Location = new System.Drawing.Point(8, 144);
            this.ux_emissionsTextBox.Name = "ux_emissionsTextBox";
            this.ux_emissionsTextBox.Size = new System.Drawing.Size(312, 21);
            this.ux_emissionsTextBox.TabIndex = 11;
            this.ux_emissionsTextBox.Text = "";
            this.ux_emissionsTextBox.TextChanged += new System.EventHandler(this.InputFileTextBox_TextChanged);
            this.ux_emissionsLabelButton.Anchor = AnchorStyles.Top | AnchorStyles.Right;
            this.ux_emissionsLabelButton.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
            this.ux_emissionsLabelButton.Location = new System.Drawing.Point(320, 142);
            this.ux_emissionsLabelButton.Name = "ux_emissionsLabelButton";
            this.ux_emissionsLabelButton.Size = new System.Drawing.Size(24, 24);
            this.ux_emissionsLabelButton.TabIndex = 12;
            this.ux_emissionsLabelButton.TextAlign = System.Drawing.ContentAlignment.MiddleCenter;
            this.ux_emissionsLabelButton.Click += new System.EventHandler(this.InputFileLabelButton_Click);
            this.ux_scenariosLabel.AutoSize = true;
            this.ux_scenariosLabel.Location = new System.Drawing.Point(8, 168);
            this.ux_scenariosLabel.Name = "ux_scenariosLabel";
            this.ux_scenariosLabel.Size = new System.Drawing.Size(107, 17);
            this.ux_scenariosLabel.TabIndex = 13;
            this.ux_scenariosLabel.Text = "Select Scenarios file:";
            this.ux_scenariosTextBox.Anchor = AnchorStyles.Top | AnchorStyles.Left | AnchorStyles.Right;
            this.ux_scenariosTextBox.BorderStyle = BorderStyle.FixedSingle;
            this.ux_scenariosTextBox.Location = new System.Drawing.Point(8, 184);
            this.ux_scenariosTextBox.Name = "ux_scenariosTextBox";
            this.ux_scenariosTextBox.Size = new System.Drawing.Size(312, 21);
            this.ux_scenariosTextBox.TabIndex = 14;
            this.ux_scenariosTextBox.Text = "";
            this.ux_scenariosTextBox.TextChanged += new System.EventHandler(this.InputFileTextBox_TextChanged);
            this.ux_scenariosLabelButton.Anchor = AnchorStyles.Top | AnchorStyles.Right;
            this.ux_scenariosLabelButton.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
            this.ux_scenariosLabelButton.Location = new System.Drawing.Point(320, 182);
            this.ux_scenariosLabelButton.Name = "ux_scenariosLabelButton";
            this.ux_scenariosLabelButton.Size = new System.Drawing.Size(24, 24);
            this.ux_scenariosLabelButton.TabIndex = 15;
            this.ux_scenariosLabelButton.TextAlign = System.Drawing.ContentAlignment.MiddleCenter;
            this.ux_scenariosLabelButton.Click += new System.EventHandler(this.InputFileLabelButton_Click);
            this.ux_reloadAllFromDiskCheckBox.FlatStyle = FlatStyle.Flat;
            this.ux_reloadAllFromDiskCheckBox.Location = new Point(8, 208);
            this.ux_reloadAllFromDiskCheckBox.Name = "ux_reloadAllFromDiskCheckBox";
            this.ux_reloadAllFromDiskCheckBox.Size = new Size(192, 16);
            this.ux_reloadAllFromDiskCheckBox.TabIndex = 16;
            this.ux_reloadAllFromDiskCheckBox.Text = "Reload all selected files from disk.";
            this.ux_reloadAllFromDiskCheckBox.TextAlign = ContentAlignment.MiddleLeft;
            this.ux_reloadAllFromDiskCheckBox.CheckedChanged += new EventHandler(this.ReloadAllFromDiskCheckBox_CheckedChanged);
            this.Controls.AddRange(
                new Control[]{
                                 ux_browseLabel,
                                 ux_marketDataLabel, ux_marketDataTextBox, ux_marketDataLabelButton,
                                 ux_technologiesLabel, ux_technologiesTextBox, ux_technologiesLabelButton,
                                 ux_parametersLabel, ux_parametersTextBox, ux_parametersLabelButton,
                                 ux_emissionsLabel, ux_emissionsTextBox, ux_emissionsLabelButton,
                                 ux_scenariosLabel, ux_scenariosTextBox, ux_scenariosLabelButton,
                                 ux_reloadAllFromDiskCheckBox
                             });
            this.Location = new Point(0, 0);
            this.Name = "InputPanel";
            this.Size = new Size(352, 232);
            this.ux_marketDataTextBox.BringToFront();
            this.ux_technologiesTextBox.BringToFront();
            this.ux_parametersTextBox.BringToFront();
            this.ux_emissionsTextBox.BringToFront();
            this.ux_scenariosTextBox.BringToFront();
            this.VisibleChanged += new EventHandler(this.InputPanel_VisibleChanged);
            this.ResumeLayout(false);
        }
        #endregion
        #region 
        private void InputPanel_VisibleChanged(object sender, EventArgs e)
        {
            if (this._filesToParse != null && this.Visible)
            {
                this.ParseFiles(this._filesToParse, false);
                this._filesToParse = null;
            }
        }
        private void InputPanel_DragEnter(object sender, DragEventArgs e)
        {
            if (e.Data.GetDataPresent(DataFormats.FileDrop, false))
            {
                e.Effect = DragDropEffects.All;
            }
        }
        private void InputPanel_DragDrop(object sender, DragEventArgs e)
        {
            string[] files = (string[])e.Data.GetData(DataFormats.FileDrop, false);
            this.ParseFiles(files, false);
        }
        private void FileParser_Parsing(object sender, VerifierEventArgs e)
        {
            if (e.Complete)
            {
                this._messages.RemoveMessage(e.Filename);
                TextBox tb = null;
                switch (e.Filetype)
                {
                    case DataFileType.MarketData:
                        tb = this.ux_marketDataTextBox;
                        break;
                    case DataFileType.Technologies:
                        tb = this.ux_technologiesTextBox;
                        break;
                    case DataFileType.Parameters:
                        tb = this.ux_parametersTextBox;
                        break;
                    case DataFileType.EmissionsRates:
                        tb = this.ux_emissionsTextBox;
                        break;
                    case DataFileType.Scenarios:
                        tb = this.ux_scenariosTextBox;
                        break;
                }
                if (tb != null)
                {
                    tb.Text = e.Filename;
                    this._messages.RemoveMessage(tb);
                    tb.BackColor = Color.White;
                    tb.ForeColor = Color.Black;
                }
            }
            else
            {
                this._messages.AddMessage("Checking \"" + e.Filename + "\" file...",
                    Color.Blue, e.Filename);
            }
        }
        private void FileParser_Parsed(object sender, VerifierEventArgs e)
        {
            if (Cafe2d.Settings.InputSettings.BackgroundLoadFiles == TriState.Enabled)
            {
                this.LoadInputFiles(false);
            }
            else
            {
                this.ChangeInputState(true);
            }
        }
        private void InputFileTextBox_TextChanged(object sender, EventArgs e)
        {
            TextBox tb = (TextBox)sender;
            int index = (int)tb.Tag;
            this._filesLoaded[index] = false;
            if (!this._verifying)
            {
                if (Cafe2d.Settings.InputSettings.BackgroundLoadFiles == TriState.Enabled)
                {
                    this.LoadInputFiles(true);
                }
                else if (Cafe2d.Settings.InputSettings.VerifySelectedFiles == TriState.Enabled)
                {
                    FileVerifier fv = new FileVerifier(tb.Text, this._inputPassword,
                        InputPanel.DesiredTitles[index]);
                    fv.Tag = tb;
                    fv.VerificationComplete += new EventHandler(
                        this.TextBoxFileVerifier_VerificationComplete);
                    fv.Start();
                }
            }
        }
        private void TextBoxFileVerifier_VerificationComplete(object sender, EventArgs e)
        {
            FileVerifier fv = (FileVerifier)sender;
            this.UpdateTextboxes((TextBox)fv.Tag, fv.Verified, fv.Errors);
        }
        private void FileLoader_Loading(object sender, VerifierEventArgs e)
        {
            TextBox tb = this.GetTextBox(e.Filetype);
            if (e.Complete)
            {
                this._messages.RemoveMessage(tb);
                FileLoader fl = (FileLoader)sender;
                int index = (int)e.Filetype;
                if (fl.FilesLoaded[index] && fl.Data[index] != null)
                {
                    this._data[index] = fl.Data[index];
                    this._fileNames[index] = e.Filename;
                }
                this.UpdateTextboxes(tb, fl.FilesLoaded[index], fl.Errors);
            }
            else
            {
                this._messages.AddMessage("Loading \"" + e.Filename + "\" file...",
                    Color.Blue, tb);
            }
        }
        private void FileLoader_Loaded(object sender, VerifierEventArgs e)
        {
            this.ChangeInputState(true);
            if (this.FilesLoaded != null)
            {
                this.FilesLoaded(this, EventArgs.Empty);
            }
        }
        private void InputFileLabelButton_Click(object sender, System.EventArgs e)
        {
            TextBox txt = (TextBox)((LabelButton)sender).Tag;
            OpenFileDialog ofd = new OpenFileDialog();
            ofd.Filter = "All Files (*.*)|*.*|All Microsoft Excel Files (*.xl*)|*.xl*|" +
                "Microsoft Excel Workbook Files (*.xls)|*.xls";
            ofd.FilterIndex = 3;
            ofd.Multiselect = false;
            ofd.FileName = File.Exists(txt.Text) ? txt.Text : "";
            if (ofd.ShowDialog() == DialogResult.OK)
            {
                txt.Text = ofd.FileName;
            }
        }
        private void ReloadAllFromDiskCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            this._reloadFromDisk = this.ux_reloadAllFromDiskCheckBox.Checked;
        }
        #endregion
        #region 
        private void ChangeInputState(bool enabled)
        {
            if (this.Visible && this.IsHandleCreated)
            {
                object[] args = new object[] {new object[] {enabled}};
                this.Invoke(new GenericEventHandler(this.ChangeInputState_Invoke), args);
            }
        }
        private void ChangeInputState_Invoke(object[] args)
        {
            bool enabled = (bool)args[0];
            this._verifying = !enabled;
            if (enabled && this.DragAndDrop)
            {
                this.DragEnter += new DragEventHandler(this.InputPanel_DragEnter);
                this.DragDrop += new DragEventHandler(this.InputPanel_DragDrop);
            }
            else
            {
                this.DragEnter -= new DragEventHandler(this.InputPanel_DragEnter);
                this.DragDrop -= new DragEventHandler(this.InputPanel_DragDrop);
            }
            this.ux_marketDataLabelButton.Enabled = enabled;
            this.ux_technologiesLabelButton.Enabled = enabled;
            this.ux_parametersLabelButton.Enabled = enabled;
            this.ux_emissionsLabelButton.Enabled = enabled;
            this.ux_scenariosLabelButton.Enabled = enabled;
            if (this.InputStateChanged != null)
            {
                this.InputStateChanged(this, enabled);
            }
        }
        private void UpdateTextboxes(TextBox textbox, bool verified, string errors)
        {
            try
            {
                if (this.IsHandleCreated)
                {
                    object[] args = new object[] {new object[] {textbox, verified, errors}};
                    this.Invoke(new GenericEventHandler(this.UpdateTextboxes_Invoke), args);
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.ToString());
            }
        }
        private void UpdateTextboxes_Invoke(object[] args)
        {
            TextBox textbox = (TextBox)args[0];
            bool verified = (bool)args[1];
            string errors = (string)args[2];
            if (verified)
            {
                textbox.BackColor = Color.White;
                textbox.ForeColor = Color.Black;
                this._messages.RemoveMessage(textbox);
            }
            else
            {
                textbox.BackColor = Color.LightYellow;
                textbox.ForeColor = Color.Red;
                this._messages.AddMessage(errors, Color.Red, textbox);
            }
        }
        private TextBox GetTextBox(DataFileType filetype)
        {
            if (filetype == DataFileType.MarketData)
            {
                return this.ux_marketDataTextBox;
            }
            else if (filetype == DataFileType.Technologies)
            {
                return this.ux_technologiesTextBox;
            }
            else if (filetype == DataFileType.Parameters)
            {
                return this.ux_parametersTextBox;
            }
            else if (filetype == DataFileType.EmissionsRates)
            {
                return this.ux_emissionsTextBox;
            }
            else if (filetype == DataFileType.Scenarios)
            {
                return this.ux_scenariosTextBox;
            }
            else
            {
                return null;
            }
        }
        private void ParseFiles(string path, bool keepExisting)
        {
            this.ParseFiles(new string[] {path}, keepExisting);
        }
        private void ParseFiles(string[] paths, bool keepExisting)
        {
            if (paths == null || paths.Length == 0) { return; }
            string[] existingPaths =
                new string[] {
                                 this.ux_marketDataTextBox.Text,
                                 this.ux_technologiesTextBox.Text,
                                 this.ux_parametersTextBox.Text,
                                 this.ux_emissionsTextBox.Text,
                                 this.ux_scenariosTextBox.Text
                             };
            bool[] ignorePaths =
                new bool[] {
                               !this._showMarketData,
                               !this._showTechnologies,
                               !this._showParameters,
                               !this._showEmissions,
                               !this._showScenarios
                           };
            FileParser fp = new FileParser(paths, this._inputPassword,
                Cafe2d.Settings.InputSettings.RecurseDirectories == TriState.Enabled,
                keepExisting, existingPaths, ignorePaths);
            fp.ParsingStart += new VerifierEventHandler(this.FileParser_Parsing);
            fp.ParsingComplete += new VerifierEventHandler(this.FileParser_Parsing);
            fp.Parsed += new VerifierEventHandler(this.FileParser_Parsed);
            this.ChangeInputState(false);
            fp.Start();
        }
        private void LoadInputFiles(bool verifyFiles)
        {
            string[] fileNames =
                new string[] {
                                 this.ux_marketDataTextBox.Text,
                                 this.ux_technologiesTextBox.Text,
                                 this.ux_parametersTextBox.Text,
                                 this.ux_emissionsTextBox.Text,
                                 this.ux_scenariosTextBox.Text
                             };
            bool[] ignorePaths =
                new bool[] {
                               !this._showMarketData,
                               !this._showTechnologies,
                               !this._showParameters,
                               !this._showEmissions,
                               !this._showScenarios
                           };
            if (this._reloadFromDisk)
            {   
                for (int i = 0; i < this._filesLoaded.Length; i++)
                {
                    this._filesLoaded[i] = false;
                }
                this.ReloadFromDisk = false;
            }
            FileLoader fl = new FileLoader(fileNames, this._inputPassword, this._filesLoaded, verifyFiles, ignorePaths);
            fl.LoadingStart += new VerifierEventHandler(this.FileLoader_Loading);
            fl.LoadingComplete += new VerifierEventHandler(this.FileLoader_Loading);
            fl.Loaded += new VerifierEventHandler(this.FileLoader_Loaded);
            this.ChangeInputState(false);
            fl.Start();
        }
        #endregion
        public void SetInitialState(InputSettings inputSettings, string inputPassword)
        {
            this._inputSettings = inputSettings;
            this._inputPassword = inputPassword;
            this.ux_marketDataLabelButton.ThemeInitialize("config-files", false);
            this.ux_technologiesLabelButton.ThemeInitialize("config-files", false);
            this.ux_parametersLabelButton.ThemeInitialize("config-files", false);
            this.ux_emissionsLabelButton.ThemeInitialize("config-files", false);
            this.ux_scenariosLabelButton.ThemeInitialize("config-files", false);
            this.ux_marketDataLabelButton.Tag = this.ux_marketDataTextBox;
            this.ux_technologiesLabelButton.Tag = this.ux_technologiesTextBox;
            this.ux_parametersLabelButton.Tag = this.ux_parametersTextBox;
            this.ux_emissionsLabelButton.Tag = this.ux_emissionsTextBox;
            this.ux_scenariosLabelButton.Tag = this.ux_scenariosTextBox;
            this.ux_marketDataTextBox.Tag = DataFileType.MarketData;
            this.ux_technologiesTextBox.Tag = DataFileType.Technologies;
            this.ux_parametersTextBox.Tag = DataFileType.Parameters;
            this.ux_emissionsTextBox.Tag = DataFileType.EmissionsRates;
            this.ux_scenariosTextBox.Tag = DataFileType.Scenarios;
            this.ux_marketDataTextBox.Text = this._inputSettings.MarketDataFile;
            this.ux_technologiesTextBox.Text = this._inputSettings.TechnologiesFile;
            this.ux_parametersTextBox.Text = this._inputSettings.ParametersFile;
            this.ux_emissionsTextBox.Text = this._inputSettings.EmissionsRatesFile;
            this.ux_scenariosTextBox.Text = this._inputSettings.ScenariosFile;
            this._filesLoaded[(int)DataFileType.MarketData] = this._inputSettings.MarketDataFileLoaded;
            this._filesLoaded[(int)DataFileType.Technologies] = this._inputSettings.TechnologiesFileLoaded;
            this._filesLoaded[(int)DataFileType.Parameters] = this._inputSettings.ParametersFileLoaded;
            this._filesLoaded[(int)DataFileType.EmissionsRates] = this._inputSettings.EmissionsRatesFileLoaded;
            this._filesLoaded[(int)DataFileType.Scenarios] = this._inputSettings.ScenariosFileLoaded;
            if (this.DragAndDrop)
            {
                this.AllowDrop = true;
                this.DragEnter += new DragEventHandler(this.InputPanel_DragEnter);
                this.DragDrop += new DragEventHandler(this.InputPanel_DragDrop);
            }
            if (Cafe2d.Settings.InputSettings.SearchPrimaryDataPath == TriState.Enabled)
            {
                this.ParseFiles(this._inputSettings.DataPath, true);
            }
        }
        public void UpdatePassword(string inputPassword)
        {
            this._inputPassword = inputPassword;
        }
        public void ParseFiles(string[] files)
        {
            if (this.Visible)
            {
                this.ParseFiles(files, false);
            }
            else
            {
                this._filesToParse = files;
            }
        }
        public void SaveChanges()
        {
            if (this.SettingsChanged)
            {
                this.LoadInputFiles(true);
            }
        }
        private void ShowHideFilePaths()
        {
            int m = 8, t = 48, p = 88, e = 128, s = 168;
            t = (this._showMarketData    ) ? m + 40 : m;
            p = (this._showTechnologies    ) ? t + 40 : t;
            e = (this._showParameters    ) ? p + 40 : p;
            s = (this._showEmissions    ) ? e + 40 : e;
            this.ux_marketDataLabel.Visible = this._showMarketData;
            this.ux_marketDataTextBox.Visible = this._showMarketData;
            this.ux_marketDataLabelButton.Visible = this._showMarketData;
            this.ux_technologiesLabel.Visible = this._showTechnologies;
            this.ux_technologiesTextBox.Visible = this._showTechnologies;
            this.ux_technologiesLabelButton.Visible = this._showTechnologies;
            this.ux_technologiesLabel.Top = t;
            this.ux_technologiesTextBox.Top = t + 16;
            this.ux_technologiesLabelButton.Top = t + 14;
            this.ux_parametersLabel.Visible = this._showParameters;
            this.ux_parametersTextBox.Visible = this._showParameters;
            this.ux_parametersLabelButton.Visible = this._showParameters;
            this.ux_parametersLabel.Top = p;
            this.ux_parametersTextBox.Top = p + 16;
            this.ux_parametersLabelButton.Top = p + 14;
            this.ux_emissionsLabel.Visible = this._showEmissions;
            this.ux_emissionsTextBox.Visible = this._showEmissions;
            this.ux_emissionsLabelButton.Visible = this._showEmissions;
            this.ux_emissionsLabel.Top = e;
            this.ux_emissionsTextBox.Top = e + 16;
            this.ux_emissionsLabelButton.Top = e + 14;
            this.ux_scenariosLabel.Visible = this._showScenarios;
            this.ux_scenariosTextBox.Visible = this._showScenarios;
            this.ux_scenariosLabelButton.Visible = this._showScenarios;
            this.ux_scenariosLabel.Top = s;
            this.ux_scenariosTextBox.Top = s + 16;
            this.ux_scenariosLabelButton.Top = s + 14;
        }
        #endregion
        #region 
        [Browsable(true)]
        [DefaultValue(true)]
        [Category("Appearance")]
        [Description("Specifies whether to display the Market-Data file prompt.")]
        public bool ShowMarketData
        {
            get { return this._showMarketData; }
            set { this._showMarketData = value; this.ShowHideFilePaths(); }
        }
        [Browsable(true)]
        [DefaultValue(true)]
        [Category("Appearance")]
        [Description("Specifies whether to display the Technologies file prompt.")]
        public bool ShowTechnologies
        {
            get { return this._showTechnologies; }
            set { this._showTechnologies = value; this.ShowHideFilePaths(); }
        }
        [Browsable(true)]
        [DefaultValue(true)]
        [Category("Appearance")]
        [Description("Specifies whether to display the Parameters file prompt.")]
        public bool ShowParameters
        {
            get { return this._showParameters; }
            set { this._showParameters = value; this.ShowHideFilePaths(); }
        }
        [Browsable(true)]
        [DefaultValue(true)]
        [Category("Appearance")]
        [Description("Specifies whether to display the Emissions-Rates file prompt.")]
        public bool ShowEmissions
        {
            get { return this._showEmissions; }
            set { this._showEmissions = value; this.ShowHideFilePaths(); }
        }
        [Browsable(true)]
        [DefaultValue(true)]
        [Category("Appearance")]
        [Description("Specifies whether to display the Scenarios file prompt.")]
        public bool ShowScenarios
        {
            get { return this._showScenarios; }
            set { this._showScenarios = value; this.ShowHideFilePaths(); }
        }
        [Browsable(true)]
        [DefaultValue(false)]
        [Category("Behavior")]
        [Description("Specifies whether to reload the data files from disk, even if there were no text changes.")]
        public bool ReloadFromDisk
        {
            get { return this._reloadFromDisk; }
            set { this._reloadFromDisk = value; this.ux_reloadAllFromDiskCheckBox.Checked = value; }
        }
        [Browsable(false)]
        public bool SettingsChanged
        {
            get
            {
                if (this._reloadFromDisk && !this._verifying) { return true; }
                bool[] showFiles =
                    new bool[] {
                                   this._showMarketData,
                                   this._showTechnologies,
                                   this._showParameters,
                                   this._showEmissions,
                                   this._showScenarios
                               };
                if (!this._verifying)
                {
                    for (int i = 0; i < this._filesLoaded.Length; i++)
                    {
                        if (!this._filesLoaded[i] && showFiles[i]) { return true; }
                    }
                }
                return false;
            }
        }
        [Browsable(false)]
        private bool DragAndDrop
        {
            get
            {
                return Cafe2d.Settings.InputSettings.DragAndDrop == TriState.Enabled ||
                    (Cafe2d.Settings.InputSettings.DragAndDrop == TriState.NotConfigured &&
                    Cafe2d.Settings.DragAndDrop == TriState.Enabled);
            }
        }
        [Browsable(false)]
        public Messages Messages
        {
            get
            {
                return this._messages;
            }
        }
        [Browsable(false)]
        public Industry MarketData
        {
            get
            {
                if (this._filesLoaded[(int)DataFileType.MarketData])
                {
                    return (Industry)this._data[(int)DataFileType.MarketData];
                }
                else
                {
                    return null;
                }
            }
        }
        [Browsable(false)]
        public Technologies Technologies
        {
            get
            {
                if (this._filesLoaded[(int)DataFileType.Technologies])
                {
                    return (Technologies)this._data[(int)DataFileType.Technologies];
                }
                else
                {
                    return null;
                }
            }
        }
        [Browsable(false)]
        public Parameters Parameters
        {
            get
            {
                if (this._filesLoaded[(int)DataFileType.Parameters])
                {
                    return (Parameters)this._data[(int)DataFileType.Parameters];
                }
                else
                {
                    return null;
                }
            }
        }
        [Browsable(false)]
        public EmissionsRates EmissionsRates
        {
            get
            {
                if (this._filesLoaded[(int)DataFileType.EmissionsRates])
                {
                    return (EmissionsRates)this._data[(int)DataFileType.EmissionsRates];
                }
                else
                {
                    return null;
                }
            }
        }
        [Browsable(false)]
        public Scenarios Scenarios
        {
            get
            {
                if (this._filesLoaded[(int)DataFileType.Scenarios])
                {
                    return (Scenarios)this._data[(int)DataFileType.Scenarios];
                }
                else
                {
                    return null;
                }
            }
        }
        [Browsable(false)]
        public string MarketDataFile
        {
            get
            {
                if (this._filesLoaded[(int)DataFileType.MarketData])
                {
                    return this._fileNames[(int)DataFileType.MarketData];
                }
                else
                {
                    return null;
                }
            }
        }
        [Browsable(false)]
        public string TechnologiesFile
        {
            get
            {
                if (this._filesLoaded[(int)DataFileType.Technologies])
                {
                    return this._fileNames[(int)DataFileType.Technologies];
                }
                else
                {
                    return null;
                }
            }
        }
        [Browsable(false)]
        public string ParametersFile
        {
            get
            {
                if (this._filesLoaded[(int)DataFileType.Parameters])
                {
                    return this._fileNames[(int)DataFileType.Parameters];
                }
                else
                {
                    return null;
                }
            }
        }
        [Browsable(false)]
        public string EmissionsRatesFile
        {
            get
            {
                if (this._filesLoaded[(int)DataFileType.EmissionsRates])
                {
                    return this._fileNames[(int)DataFileType.EmissionsRates];
                }
                else
                {
                    return null;
                }
            }
        }
        [Browsable(false)]
        public string ScenariosFile
        {
            get
            {
                if (this._filesLoaded[(int)DataFileType.Scenarios])
                {
                    return this._fileNames[(int)DataFileType.Scenarios];
                }
                else
                {
                    return null;
                }
            }
        }
        #endregion
        #region 
        #region 
        private System.Windows.Forms.Label ux_browseLabel;
        private System.Windows.Forms.Label ux_marketDataLabel;
        private System.Windows.Forms.TextBox ux_marketDataTextBox;
        private Volpe.Cafe.Ui.LabelButton ux_marketDataLabelButton;
        private System.Windows.Forms.Label ux_technologiesLabel;
        private System.Windows.Forms.TextBox ux_technologiesTextBox;
        private Volpe.Cafe.Ui.LabelButton ux_technologiesLabelButton;
        private System.Windows.Forms.Label ux_parametersLabel;
        private System.Windows.Forms.TextBox ux_parametersTextBox;
        private Volpe.Cafe.Ui.LabelButton ux_parametersLabelButton;
        private System.Windows.Forms.Label ux_emissionsLabel;
        private System.Windows.Forms.TextBox ux_emissionsTextBox;
        private Volpe.Cafe.Ui.LabelButton ux_emissionsLabelButton;
        private System.Windows.Forms.Label ux_scenariosLabel;
        private System.Windows.Forms.TextBox ux_scenariosTextBox;
        private Volpe.Cafe.Ui.LabelButton ux_scenariosLabelButton;
        private CheckBox ux_reloadAllFromDiskCheckBox;
        private System.ComponentModel.IContainer components = null;
        #endregion
        private static string[][] DesiredTitles =
            new string[][] {
                               new string[] {"CAFE Market Data"},
                               new string[] {"CAFE Technologies"},
                               new string[] {"CAFE Parameters"},
                               new string[] {"CAFE Emissions Rates", "CAFE Emissions"},
                               new string[] {"CAFE Scenarios"}
                           };
        private object[] _data;
        private string[] _fileNames;
        private bool[] _filesLoaded;
        private bool _reloadFromDisk;
        private bool _verifying;
        private Messages _messages;
        private InputSettings _inputSettings;
        private string _inputPassword;
        private string[] _filesToParse;
        private bool _showMarketData;
        private bool _showTechnologies;
        private bool _showParameters;
        private bool _showEmissions;
        private bool _showScenarios;
        #endregion
    }
}

