using System;
using System.Drawing;
using System.Collections;
using System.ComponentModel;
using System.IO;
using System.Reflection;
using System.Runtime.Serialization.Formatters.Binary;
using System.Threading;
using System.Windows.Forms;
using Volpe.Cafe;
using Volpe.Cafe.Collections;
using Volpe.Cafe.Data;
using Volpe.Cafe.Model;
using Volpe.Ui;
namespace Volpe.Cafe.Ui
{
    public class Session2d : System.Windows.Forms.Form
    {
        #region 
        public event EventHandler SessionOpened;
        public event EventHandler SessionSaved;
        #endregion
        #region 
        public Session2d(Cafe2d parent)
        {
            this.InitializeComponent();
            this._parent = parent;
            this._session = new Session();
            this._session.Prompt += new PromptEventHandler(this.Session_Prompt);
            this._session.Updating += new EventHandler(this.Session_Updating);
            this.Text = this._session.Text;
        }
        protected override void Dispose(bool disposing)
        {
            this._stopRequested = false;
            this._session.Dispose();
            this._session = null;
            this.SessionOpened = null;
            this.SessionSaved  = null;
            if (disposing)
            {
                if (this.components != null)
                {
                    this.components.Dispose();
                }
            }
            base.Dispose(disposing);
        }
        #endregion
        #region 
        #region 
        private void InitializeComponent()
        {
            System.Resources.ResourceManager resources = new System.Resources.ResourceManager(typeof(Session2d));
            this.ux_progressRichTextBox = new System.Windows.Forms.RichTextBox();
            this.ux_statusPanel = new System.Windows.Forms.Panel();
            this.ux_additionalInfoRichTextBox = new System.Windows.Forms.RichTextBox();
            this.ux_statusLabel = new System.Windows.Forms.Label();
            this.ux_splitter = new System.Windows.Forms.Splitter();
            this.ux_statusPanel.SuspendLayout();
            this.SuspendLayout();
            this.ux_progressRichTextBox.Dock = System.Windows.Forms.DockStyle.Fill;
            this.ux_progressRichTextBox.Location = new System.Drawing.Point(0, 0);
            this.ux_progressRichTextBox.Name = "ux_progressRichTextBox";
            this.ux_progressRichTextBox.ReadOnly = true;
            this.ux_progressRichTextBox.Size = new System.Drawing.Size(472, 209);
            this.ux_progressRichTextBox.TabIndex = 0;
            this.ux_progressRichTextBox.Text = "Ready";
            this.ux_progressRichTextBox.WordWrap = false;
            this.ux_statusPanel.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
            this.ux_statusPanel.Controls.Add(this.ux_additionalInfoRichTextBox);
            this.ux_statusPanel.Controls.Add(this.ux_statusLabel);
            this.ux_statusPanel.Dock = System.Windows.Forms.DockStyle.Bottom;
            this.ux_statusPanel.Location = new System.Drawing.Point(0, 213);
            this.ux_statusPanel.Name = "ux_statusPanel";
            this.ux_statusPanel.Size = new System.Drawing.Size(472, 80);
            this.ux_statusPanel.TabIndex = 1;
            this.ux_additionalInfoRichTextBox.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
                | System.Windows.Forms.AnchorStyles.Left) 
                | System.Windows.Forms.AnchorStyles.Right)));
            this.ux_additionalInfoRichTextBox.BackColor = System.Drawing.SystemColors.Control;
            this.ux_additionalInfoRichTextBox.BorderStyle = System.Windows.Forms.BorderStyle.None;
            this.ux_additionalInfoRichTextBox.Location = new System.Drawing.Point(8, 32);
            this.ux_additionalInfoRichTextBox.Name = "ux_additionalInfoRichTextBox";
            this.ux_additionalInfoRichTextBox.ReadOnly = true;
            this.ux_additionalInfoRichTextBox.Size = new System.Drawing.Size(456, 40);
            this.ux_additionalInfoRichTextBox.TabIndex = 1;
            this.ux_additionalInfoRichTextBox.TabStop = false;
            this.ux_additionalInfoRichTextBox.Text = "";
            this.ux_additionalInfoRichTextBox.WordWrap = false;
            this.ux_statusLabel.AutoSize = true;
            this.ux_statusLabel.Location = new System.Drawing.Point(8, 8);
            this.ux_statusLabel.Name = "ux_statusLabel";
            this.ux_statusLabel.Size = new System.Drawing.Size(59, 17);
            this.ux_statusLabel.TabIndex = 0;
            this.ux_statusLabel.Text = "Runtime: 0";
            this.ux_splitter.BackColor = System.Drawing.SystemColors.HotTrack;
            this.ux_splitter.Dock = System.Windows.Forms.DockStyle.Bottom;
            this.ux_splitter.Location = new System.Drawing.Point(0, 209);
            this.ux_splitter.Name = "ux_splitter";
            this.ux_splitter.Size = new System.Drawing.Size(472, 4);
            this.ux_splitter.TabIndex = 2;
            this.ux_splitter.TabStop = false;
            this.AutoScaleBaseSize = new System.Drawing.Size(5, 14);
            this.ClientSize = new System.Drawing.Size(472, 293);
            this.Controls.Add(this.ux_progressRichTextBox);
            this.Controls.Add(this.ux_splitter);
            this.Controls.Add(this.ux_statusPanel);
            this.Font = new System.Drawing.Font("Tahoma", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
            this.Icon = ((System.Drawing.Icon)(resources.GetObject("$this.Icon")));
            this.Name = "Session2d";
            this.Text = "Session2d";
            this.Load += new System.EventHandler(this.Session2d_Load);
            this.ux_statusPanel.ResumeLayout(false);
            this.ResumeLayout(false);
        }
        #endregion
        #region 
        protected void OnSessionOpened()
        {
            this._parent.AddRecentSession(this.SessionName, this.SessionPath);
            if (this.SessionOpened != null)
            {
                this.SessionOpened(this, EventArgs.Empty);
            }
        }
        protected void OnSessionSaved()
        {
            this._parent.AddRecentSession(this.SessionName, this.SessionPath);
            if (this.SessionSaved != null)
            {
                this.SessionSaved(this, EventArgs.Empty);
            }
        }
        #endregion
        public override void Refresh()
        {
            if (this._session != null)
            {
                this._session.Refresh();
            }
            base.Refresh();
        }
        private void Session2d_Load(object sender, System.EventArgs e)
        {
            this.Font = Cafe2d.DefaultFont;
            this.Closing += new CancelEventHandler(Session2d_Closing);
        }
        private void Session2d_Closing(object sender, CancelEventArgs e)
        {
            DialogResult dr = DialogResult.None;
            if (this._session.Running || this._session.Reporting)
            {   
                dr = MessageBox.Show(
                    "The session is currently " + ((this._session.Running) ? "running" : "reporting") + ".  Would " +
                    "you like to stop the " + ((this._session.Running) ? "modeling" : "reporting") + " process?  " +
                    "Selecting \"No\" will continue " + ((this._session.Running) ? "modeling" : "reporting") + ".",
                    "Stop " + ((this._session.Running) ? "Modeling?" : "Reporting?"),
                    MessageBoxButtons.YesNo, MessageBoxIcon.Question, MessageBoxDefaultButton.Button2);
                if (dr == DialogResult.No)
                {
                    e.Cancel = true;
                    return;
                }
                if (this.Running)
                {
                    this._session.Stop(false);
                    while (this._session.Running) { Thread.Sleep(125); Application.DoEvents(); }
                }
                else
                {
                    this._session.CancelReports();
                    while (this._session.Reporting) { Thread.Sleep(125); Application.DoEvents(); }
                }
            }
            dr = MessageBox.Show("Would you like to save the session \"" + this.Text + "\" before closing?",
                "Save Session?", MessageBoxButtons.YesNoCancel, MessageBoxIcon.Question, MessageBoxDefaultButton.Button3);
            if (dr == DialogResult.Cancel)
            {   
                e.Cancel = true;
            }
            else
            {
                if (dr == DialogResult.Yes)
                {    
                    this.SaveSession();
                }
                this._session.Dispose();
                e.Cancel = false;
                GC.Collect();
                GC.WaitForPendingFinalizers();
                GC.Collect();
            }
        }
        private void Session_Prompt(object sender, PromptEventArgs e)
        {
            MessageBoxButtons buttons =
                (e.PromptOption == PromptOption.OkCancel)    ? MessageBoxButtons.OKCancel :
                (e.PromptOption == PromptOption.YesNoCancel) ? MessageBoxButtons.YesNoCancel :
                (e.PromptOption == PromptOption.YesNo)       ? MessageBoxButtons.YesNo :
                MessageBoxButtons.OK;
            MessageBoxDefaultButton defaultButton =
                (e.PromptOption == PromptOption.YesNoCancel) ? MessageBoxDefaultButton.Button3 :
                (e.PromptOption == PromptOption.YesNo ||
                    e.PromptOption == PromptOption.OkCancel) ? MessageBoxDefaultButton.Button2 :
                MessageBoxDefaultButton.Button1;
            DialogResult dr =  MessageBox.Show(e.Message, e.Title, buttons, MessageBoxIcon.None, defaultButton);
            e.PromptResult =
                (dr == DialogResult.OK)        ? PromptResult.Ok :
                (dr == DialogResult.Cancel) ? PromptResult.Cancel :
                (dr == DialogResult.Yes)    ? PromptResult.Yes :
                (dr == DialogResult.No)        ? PromptResult.No : PromptResult.None;
        }
        private void Session_Updating(object sender, EventArgs e)
        {
            this.UpdateProgress();
            this._parent.RefreshSession();
        }
        public bool OpenSession()
        {
            return this.OpenSession(null, false);
        }
        public bool OpenSession(string filename, bool silent)
        {
            if (filename == null || filename == "")
            {
                OpenFileDialog ofd = new OpenFileDialog();
                ofd.Filter = "All Files (*.*)|*.*|CAFE Model Session Data (*.cmsd)|*.cmsd";
                ofd.FilterIndex = 2;
                if (ofd.ShowDialog() != DialogResult.OK) { return false; }    
                filename = ofd.FileName;
            }
            this._parent.SetStatus(this, "Loading session: " + filename + "...");
            this._session.Load(filename, silent);
            this.Text = this._session.Text;
            this._parent.SetStatus(this, "Session " + ((this._session.Loaded) ? "loaded." : "loading failed."));
            if (this._session.Loaded) { this.OnSessionOpened(); }
            else                      { this._parent.RemoveRecentSession(filename); }
            return this._session.Loaded;
        }
        public void CloseSession()
        {
            this.Close();
        }
        public bool SaveSession()
        {
            string path = this._session.Path;
            if (path != null && path.Trim() != "" && (!File.Exists(path) ||
                (File.GetAttributes(path) & FileAttributes.ReadOnly) != FileAttributes.ReadOnly))
            {
                this._parent.SetStatus(this, "Saving session: " + path + " ...");
                this._session.Save(path);
                this.OnSessionSaved();
                this._parent.SetStatus(this, "Session saved at: " + path + ".");
                return true;
            }
            else
            {   
                return this.SaveSessionAs();
            }
        }
        public bool SaveSessionAs()
        {
            SaveFileDialog sfd = new SaveFileDialog();
            sfd.FileName = this._session.Path;
            if (sfd.FileName == "")
            {
                sfd.FileName = this._session.Name + ".cmsd";
            }
            sfd.Filter = "CAFE Model Session Data (*.cmsd)|*.cmsd";
            sfd.Title = "Please specify the location where you would like to save the '" + this._session.Name +
                "' session.";
            if (sfd.ShowDialog() == DialogResult.OK)
            {   
                this._session.Path = sfd.FileName;
                return this.SaveSession();
            }
            else { return false; }
        }
        public void StartModeling()
        {
            this._stopRequested = false;
            this._session.Start();
        }
        public void StopModeling()
        {
            if (this._stopRequested)
            {
                this._session.Stop(false);
                this._stopRequested = false;
            }
            else
            {
                this._session.Stop();
                this._stopRequested = true;
                Thread waitToStop = new Thread(new ThreadStart(this.WaitToStopThread));
                waitToStop.Name = "Session-" + this._session.Name + "::WaitToStopThread";
                waitToStop.Start();
            }
        }
        public void GenerateReports()
        {
            try
            {
                this._session.GenerateReports();
            }
            catch
            {
                MessageBox.Show("An error has occurred while generating one or more of the modeling reports.  Please" +
                    " save your work, restart the application, and try generating the reports again.  If the problem" +
                    " persists, please contact " + Program.ContactName + " at " + Program.ContactEmail + ".",
                    "Error Generating Reports", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }
        public void CancelReports()
        {
            this._session.CancelReports();
        }
        private void WaitToStopThread()
        {
            for (int i = 0; i < 10; i++)
            {
                Thread.Sleep(500);
                if (!this._stopRequested || !this._session.Running) { return; }
            }
            this._stopRequested = false;
        }
        private void UpdateProgress()
        {
            try
            {
                this.Invoke(new GenericEventHandler(this.UpdateProgress_Invoke), new object[] {null});
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.ToString());
            }
        }
        private void UpdateProgress_Invoke(object[] args)
        {
            Session.ProgressInfo progress = this._session.ProgressInformation;
            string text = "";
            if (this._stopRequested && this._session.Running)
            {    
                text += "Waiting to stop...  Press stop again to terminate immediately.\r\n\r\n";
            }
            text += progress.GetCurrent();
            this.ux_progressRichTextBox.Text = text;
            Win32.SendMessage(this.ux_progressRichTextBox.Handle, (int)Win32.Messages.WM_VSCROLL, 7, new IntPtr(0));
            this.ux_statusLabel.Text = "Runtime: " + Global.GetTimeString(this._session.Runtime);
            object additionalInfo = progress.GetAdditionalInfo();
            this.ux_additionalInfoRichTextBox.Text = (additionalInfo == null) ? "" :
                "Additional Info\r\n" + additionalInfo.ToString();
            this._parent.SetStatus(this, this._session.Status);
        }
        #endregion
        #region 
        public int RefreshInterval
        {
            get { return this._session.RefreshInterval; }
            set { this._session.RefreshInterval = value; }
        }
        public Industry Data
        {
            get { return this._session.Data; }
            set { this._session.Data = value; }
        }
        public ModelingSettings Settings
        {
            get { return this._session.Settings; }
            set { this._session.Settings = value; }
        }
        public ICompliance Compliance
        {
            get { return this._session.Compliance; }
            set { this._session.Compliance = value; }
        }
        public bool Running
        {
            get { return this._session.Running; }
        }
        public string Status
        {
            get { return this._session.Status; }
        }
        public bool Reporting
        {
            get { return this._session.Reporting; }
        }
        public string SessionName
        {
            get { return this._session.Name; }
            set
            {
                this._session.Name = value;
                this.Text = this._session.Text;
            }
        }
        public string SessionPath
        {
            get { return this._session.Path; }
        }
        #endregion
        #region 
        #region 
        private System.ComponentModel.Container components = null;
        private System.Windows.Forms.Panel ux_statusPanel;
        private System.Windows.Forms.RichTextBox ux_additionalInfoRichTextBox;
        private System.Windows.Forms.Label ux_statusLabel;
        private System.Windows.Forms.Splitter ux_splitter;
        private System.Windows.Forms.RichTextBox ux_progressRichTextBox;
        #endregion
        private Cafe2d _parent;
        private Session _session;
        private bool _stopRequested;
        #endregion
    }
}

