using System;
using Volpe.Cafe;
using Volpe.Cafe.Collections;
namespace Volpe.Cafe.IO
{
    [Serializable]
    public class Technologies : Input
    {
        #region 
        public Technologies(string path)
            : this(path, null)
        {
        }
        public Technologies(string path, string password)
            : base(path, password)
        {
            try
            {   
                this.ParseFile();
            }
            catch (Exception ex)
            {   
                throw new InputException("Errors were encountered while parsing file: " + ex.Message, path, ex);
            }
            finally
            {   
                this.Close();
            }
        }
        #endregion
        #region 
        private void ParseFile()
        {
            this._data = new TechnologyCollection(TechnologyIndexes.TechnologyCount + 1);
            string[] sheets = this.SheetNames;
            string[] names =
                new string[] {                                                                              
                                 "Subcompact PC", "Subcompact Perf. PC", "Compact PC", "Compact Perf. PC",  
                                 "Midsize PC"   , "Midsize Perf. PC"   , "Large PC"  , "Large Perf. PC"  ,  
                                 "Minivan LT"   , "Small LT"           , "Midsize LT", "Large LT"        ,  
                                 "FC Synergies" , "Cost Synergies"     , "Monte-Carlo"                      
                             };
            int[] indexes = new int[names.Length];
            for (int i = 0; i < indexes.Length; i++) { indexes[i] = -1; }
            TechnologyClass[] classes = 
                new TechnologyClass[] {
                                          TechnologyClass.SubcompactPC, TechnologyClass.SubcompactPerfPC,
                                          TechnologyClass.CompactPC   , TechnologyClass.CompactPerfPC   ,
                                          TechnologyClass.MidsizePC   , TechnologyClass.MidsizePerfPC   ,
                                          TechnologyClass.LargePC     , TechnologyClass.LargePerfPC     ,
                                          TechnologyClass.MinivanLT   , TechnologyClass.SmallLT         ,
                                          TechnologyClass.MidsizeLT   , TechnologyClass.LargeLT
                                      };
            for (int i = 0; i < sheets.Length; i++)
            {
                string sheet = sheets[i].Trim().ToUpper();
                for (int j = 0; j < names.Length; j++)
                {
                    if (this.Compare(sheet, names[j], true)) { indexes[j] = i; break; }
                }
            }
            this.VerifyIndexes(indexes, names, 0, 12);
            this._data.AddRange(new Technology[TechnologyIndexes.TechnologyCount]);
            for (int i = 0, techClassCount = (int)TechnologyClass.Max; i < techClassCount; i++)
            {   
                this.ParseFile_LoadTechnologies(indexes[i], classes[i]);
            }
            TechnologySynergiesTable[] imprvSynergies = null, costSynergies = null;
            if (indexes[12] != -1) { imprvSynergies = this.ParseFile_LoadSynergies(indexes[12], classes); }
            if (indexes[13] != -1) { costSynergies  = this.ParseFile_LoadSynergies(indexes[13], classes); }
            for (int i = 0; i < TechnologyIndexes.TechnologyCount; i++)
            {
                if (indexes[12] != -1) { this._data[i].ImprvSynergies = imprvSynergies[i]; }
                if (indexes[13] != -1) { this._data[i].CostSynergies  = costSynergies [i]; }
            }
            if (indexes[14] != -1)
            {   
                this.ActivateWorksheet(indexes[14]);
                for (int i = 0; i < TechnologyIndexes.TechnologyCount; i++)
                {
                    char complexity = this.GetChar(2 + i, 4);
                    this._data[i].Complexity    = (complexity == 'L') ? TechnologyComplexity.Low    :
                                                  (complexity == 'M') ? TechnologyComplexity.Medium :
                                                  (complexity == 'H') ? TechnologyComplexity.High   : TechnologyComplexity.None;
                    this._data[i].FcVariation   = this.GetDouble(2 + i, 5);
                    this._data[i].CostVariation = this.GetDouble(2 + i, 6);
                }
            }
        }
        void ParseFile_LoadTechnologies(int index, TechnologyClass techClass)
        {
            this.ActivateWorksheet(index);
            int rows = this.Rows;
            int cols = this.Columns;
            string[] names = {                                                                                  
                                 "Technology"    , "Abbr"          , "TechType"     , "Year Avail"        ,     
                                 "FC-Lower"      , "FC-Upper"      , "Cost-Lower"   , "Cost-Upper"        ,     
                                 "Learning Type" , "Learning Start", "Learning End" , "Learning Threshold",     
                                 "Learning Rate" , "Aux."          , "Applicability", "Phase-In Values"         
                             };
            int[] indexes = new int[names.Length];
            for (int i = 0; i < indexes.Length; i++) { indexes[i] = -1; }
            int phaseInCount = 0;   
            for (int i = 0; i < cols; i++)
            {
                string column = this.GetString(1, i);
                if (indexes[15] == -1 && Global.StringStartsWith(column, "PHASE-IN", true))
                {   
                    indexes[15] = i;    
                    while (Global.StringStartsWith(column, "PHASE-IN", true))
                    {   
                        phaseInCount++; i++;
                        if (i == cols) { break; }
                        column = this.GetString(1, i);
                    }
                }
                else
                {   
                    for (int j = 0; j < names.Length; j++)
                    {
                        if (this.Compare(column, names[j], true)) { indexes[j] = i; break; }
                    }
                }
            }
            this.VerifyIndexes(indexes, names);
            int phaseInMinYear = Global.GetInt32(
                Global.StringRemoveWhitespacePunctuationAndSpecialChars(this.GetString(1, indexes[15])).Replace("PHASEIN", ""));
            if (phaseInMinYear < ModelYear.MinYear)
            {
                string path = this._xlu.Path;
                this.Close();
                throw new InputException("The phase-in section is not valid.", path);
            }
            for (int i = 2; i < rows; i++)
            {
                Technology tech = this.ParseFile_LoadTechnology(i, indexes, techClass, phaseInMinYear, phaseInCount);
                if (tech.Index == -1) { continue; }     
                if (this._data[tech.Index] == null) { this._data[tech.Index] = tech; }
                else
                {   
                    this._data[tech.Index].Attributes[techClass] = tech.Attributes[techClass];
                }
            }
        }
        Technology ParseFile_LoadTechnology(int row, int[] indexes, TechnologyClass techClass, int phaseInMinYear,
            int phaseInCount)
        {
            Technology           tech = new Technology();
            TechnologyAttributes ta   = new TechnologyAttributes();
            tech.Name  = this.GetString(row, indexes[0]);
            tech.Abbr  = this.GetString(row, indexes[1]);
            tech.Index = TechnologyIndexes.GetIndex(tech.Abbr);     
            string techType = this.GetString(row, indexes[2]);
            tech.Type =
                (techType == "ENGMOD") ? TechnologyType.Engine :
                (techType == "TRMOD" ) ? TechnologyType.Transmission :
                (techType == "ELEC"  ) ? TechnologyType.Elec :
                (techType == "HEV"   ) ? TechnologyType.Hybrid :
                (techType == "MSM" ||
                 techType == "MSR"   ) ? TechnologyType.MSR :
                (techType == "DLR"   ) ? TechnologyType.DLR :
                (techType == "AERO"  ) ? TechnologyType.Aerodynamics : TechnologyType.Other;
            ta.TechnologyClass   = techClass;
            ta.Year              = this.GetInt32 (row, indexes[ 3]);
            ta.FcLow             = this.GetDouble(row, indexes[ 4]);
            ta.FcHigh            = this.GetDouble(row, indexes[ 5]);
            ta.CostLow           = this.GetDouble(row, indexes[ 6]);
            ta.CostHigh          = this.GetDouble(row, indexes[ 7]);
            string learningType  = this.GetString(row, indexes[ 8]);
            ta.LearningType      =
                Global.StringCompare(learningType, "Volume", true) ? LearningType.Volume :
                Global.StringCompare(learningType, "Time"  , true) ? LearningType.Time   : LearningType.None;
            ta.LearningStartYear = this.GetInt32 (row, indexes[ 9]);
            ta.LearningEndYear   = this.GetInt32 (row, indexes[10]);
            ta.LearningThreshold = this.GetDouble(row, indexes[11]);
            ta.LearningRate      = this.GetDouble(row, indexes[12]);
            ta.Aux               = this.GetDouble(row, indexes[13]);
            ta.Applicable        = this.GetBool  (row, indexes[14]);
            tech.Attributes[techClass] = ta;
            int phaseInStart = phaseInMinYear - ModelYear.MinYear;
            tech.PhaseIn = new double[phaseInStart + phaseInCount];
            for (int i = 0; i < phaseInCount; i++)
            {
                tech.PhaseIn[phaseInStart + i] = Math.Min(1.0, this.GetDouble(row, indexes[15] + i));
            }
            return tech;
        }
        TechnologySynergiesTable[] ParseFile_LoadSynergies(int index, TechnologyClass[] classes)
        {
            this.ActivateWorksheet(index);
            int rows = this.Rows;
            int cols = this.Columns;
            string[] names =
                new string[] {                                                                              
                                 "Type", "Technology A", "Technology B",                                    
                                 "Subcompact PC", "Subcompact Perf. PC", "Compact PC", "Compact Perf. PC",  
                                 "Midsize PC"   , "Midsize Perf. PC"   , "Large PC"  , "Large Perf. PC"  ,  
                                 "Minivan LT"   , "Small LT"           , "Midsize LT", "Large LT"           
                             };
            int[] indexes = new int[names.Length];
            for (int i = 0; i < indexes.Length; i++) { indexes[i] = -1; }
            for (int i = 0; i < cols; i++)
            {   
                string colName = this.GetString(1, i);
                for (int j = 0; j < names.Length; j++)
                {
                    if (this.Compare(colName, names[j], true)) { indexes[j] = i; break; }
                }
            }
            int row = 2;    
            TechnologySynergiesTable[] synergies = new TechnologySynergiesTable[TechnologyIndexes.TechnologyCount];
            for (int i = 0; i < TechnologyIndexes.TechnologyCount; i++)
            {
                synergies[i] = new TechnologySynergiesTable();
            }
            do
            {   
                string techA = this.GetString(row, indexes[1]);
                if (techA == "") { break; }     
                string techB = this.GetString(row, indexes[2]);
                SynergyType type = SynergyType.None;
                if (indexes[0] != -1)
                {
                    string typeStr = this.GetString(row, indexes[0]).ToUpper();
                    type = (typeStr == "ACCOUNTING") ? SynergyType.Accounting :
                        (typeStr == "PHYSICAL") ? SynergyType.Physical : SynergyType.None;
                }
                int techAIndex = TechnologyIndexes.GetIndex(techA);
                int techBIndex = TechnologyIndexes.GetIndex(techB);
                if ((techAIndex >= 0) && (techBIndex >= 0))
                {   
                    for (int i = 3; i < indexes.Length; i++)
                    {
                        if (indexes[i] != -1)
                        {   
                            double synergy = this.GetDouble(row, indexes[i]);
                            synergies[techAIndex][classes[i - 3]].SetSynergy(techBIndex, synergy, type);
                            synergies[techBIndex][classes[i - 3]].SetSynergy(techAIndex, synergy, type);
                        }
                    }
                }
                row++;
            } while (row < rows);
            return synergies;
        }
        #endregion
        #region 
        public TechnologyCollection Data
        {
            get { return this._data; }
        }
        #endregion
        #region 
        private TechnologyCollection _data;
        #endregion
    }
}

