using System;
using Volpe.Cafe.Collections;
using Volpe.Cafe.IO;
namespace Volpe.Cafe
{
    [Serializable]
    public class ModelingSettings : ICloneable
    {
        #region 
        [Serializable]
        public struct SettingsChanges
        {
            #region 
            public void Reset()
            {
                this.InputChanged      = false;
                this.OutputChanged     = false;
                this.EncryptionChanged = false;
                this.OperatingModesChanged = false;
                this.OptimizationChanged   = false;
                this.MonteCarloChanged     = false;
                this.TechnologiesDataChanged = false;
                this.ParametersDataChanged   = false;
                this.EmissionsDataChanged    = false;
                this.ScenariosDataChanged    = false;
                this.ModelingDataChanged     = false;
            }
            #endregion
            #region 
            public bool SettingsChanged
            {
                get
                {
                    return
                        this.InputChanged          || this.OutputChanged       || this.EncryptionChanged ||
                        this.OperatingModesChanged || this.OptimizationChanged ||
                        this.MonteCarloChanged     || this.InputDataChanged;
                }
            }
            public bool InputDataChanged
            {
                get
                {
                    return this.TechnologiesDataChanged || this.ParametersDataChanged ||
                        this.EmissionsDataChanged       || this.ScenariosDataChanged  || this.ModelingDataChanged;
                }
            }
            #endregion
            #region 
            public bool InputChanged;
            public bool OutputChanged;
            public bool EncryptionChanged;
            public bool OperatingModesChanged;
            public bool OptimizationChanged;
            public bool MonteCarloChanged;
            public bool TechnologiesDataChanged;
            public bool ParametersDataChanged;
            public bool EmissionsDataChanged;
            public bool ScenariosDataChanged;
            public bool ModelingDataChanged;
            #endregion
        }
        #endregion
        #region 
        private ModelingSettings()
        {
        }
        public ModelingSettings(string dataPath, string outputPath)
        {
            this.Changes = new SettingsChanges();
            this.InputSettings = new InputSettings(dataPath);
            this.OutputSettings = new OutputSettings(outputPath);
            this.EncryptionSettings = new EncryptionSettings();
            this.EncryptionSettings.EncryptionType = Volpe.Cafe.EncryptionSettings.EncryptionTypes[6];
            this.EncryptionSettings.EncryptionKeyLength = 128;
            this.OperatingModes = new OperatingModes();
            this.OperatingModes.TechnologyFuelEstimates = Estimates.Average;
            this.OperatingModes.TechnologyCostEstimates = Estimates.Average;
            this.OperatingModes.FuelPriceEstimates = Estimates.Average;
            this.OptimizationSettings = new OptimizationSettings();
            this.MonteCarloSettings = new MonteCarloSettings();
        }
        #endregion
        #region 
        #region 
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        public ModelingSettings Clone()
        {
            ModelingSettings settings = new ModelingSettings();
            settings.Changes = this.Changes;
            settings.InputSettings      = this.InputSettings;
            settings.OutputSettings     = this.OutputSettings;
            settings.EncryptionSettings = this.EncryptionSettings;
            settings.OperatingModes          = this.OperatingModes;
            settings.OptimizationSettings    = this.OptimizationSettings;
            settings.MonteCarloSettings      = this.MonteCarloSettings.Clone();
            settings._technologies = (this._technologies == null) ? null : this._technologies.Clone();
            settings._parameters   = (this._parameters   == null) ? null : this._parameters  .Clone();
            settings._emissions    = (this._emissions    == null) ? null : this._emissions   .Clone();
            settings._scenarios    = (this._scenarios    == null) ? null : this._scenarios   .Clone();
            return settings;
        }
        #endregion
        public void WriteSummaryLog(LogWriter writer)
        {
            if (writer.Summary == null) { return; }
            writer.Summary.WriteLine("---------- Modeling Settings ----------");
            this.InputSettings          .WriteSummaryLog(writer);
            this.OutputSettings         .WriteSummaryLog(writer);
            this.EncryptionSettings     .WriteSummaryLog(writer);
            this.OperatingModes         .WriteSummaryLog(writer);
            this.OptimizationSettings   .WriteSummaryLog(writer);
            this.MonteCarloSettings     .WriteSummaryLog(writer);
            if (this._technologies != null && this._technologies.Count > 0)
            {
                string techStr = "--- Technologies Analyzed ---";
                for (int i = 0, techCount = this._technologies.Count; i < techCount; i++)
                {
                    Technology tech = this._technologies[i];
                    techStr += "\r\n  " + tech.Index + ". " + tech.Abbr + " - " + tech.Name;
                }
                writer.Summary.WriteLine(techStr + "\r\n");
            }
            if (this._scenarios != null && this._scenarios.Count > 0)
            {
                string scenStr = "--- Scenarios Analyzed ---";
                for (int i = 0, scenCount = this._scenarios.Count; i < scenCount; i++)
                {
                    Scenario scen = this._scenarios[i];
                    scenStr += "\r\n  " + scen.Index + ". " + scen.Description +
                        ((scen.IsBaseline) ? "  (Baseline)" : "");
                }
                writer.Summary.WriteLine(scenStr + "\r\n");
            }
        }
        public void UseRecommendedSettings()
        {
            this.OutputSettings.GenerateReports                         = true;
            this.OutputSettings.GenerateIndustryReport                  = true;
            this.OutputSettings.GenerateManufacturersReport             = false;
            this.OutputSettings.GenerateVehiclesReport                  = false;
            this.OutputSettings.GenerateScenarioSummaryReport           = true;
            this.OutputSettings.GenerateEffectsReport                   = false;
            this.OutputSettings.GenerateEnvironmentalAssessmentReport   = false;
            this.OutputSettings.GenerateOptimizedIndustryReportExtended = false;
            this.EncryptionSettings.EncryptInput  = false;
            this.EncryptionSettings.EncryptOutput = false;
            this.OperatingModes.ModelingApproach                  = ModelingApproach.LowCostFirst;
            this.OperatingModes.AutoToggleApproach                = false;
            this.OperatingModes.NoFines                           = false;
            this.OperatingModes.MultiYearModeling                 = true;
            this.OperatingModes.MultiYearStartYear                = 0;
            this.OperatingModes.SplitSharedEngsOrTrns             = true;
            this.OperatingModes.SplitConflictingEngsOrTrns        = true;
            this.OperatingModes.SkipSharedEngsOrTrns              = false;
            this.OperatingModes.SkipConflictingEngsOrTrns         = false;
            this.OperatingModes.TechnologyFuelEstimates           = Estimates.Average;
            this.OperatingModes.TechnologyCostEstimates           = Estimates.Average;
            this.OperatingModes.Backfill                          = true;
            this.OperatingModes.ClipTechImprovements              = true;
            this.OperatingModes.ClipTechCosts                     = false;
            this.OperatingModes.ReserveTechnologiesIfInCompliance = false;
            this.OperatingModes.IgnoreYearAvailable               = false;
            this.OperatingModes.IgnorePhaseIn                     = false;
            this.OperatingModes.IgnoreRefreshRedesign             = false;
            this.OperatingModes.IgnoreWeightInMsApplication       = true;
            this.OperatingModes.ShutoffEngineTechsAfterHEV        = false;
            this.OperatingModes.ShutoffTransmissionTechsAfterHEV  = true;
            this.OperatingModes.ShutoffElecTechsAfterHEV          = true;
            this.OperatingModes.FuelPriceEstimates                = Estimates.Average;
            this.OptimizationSettings.Type                   = OptimizationType.Default;
            this.OptimizationSettings.Mode                   = OptimizationMode.MaximumNetBenefits;
            this.OptimizationSettings.AboveOptimum           = 75;
            this.OptimizationSettings.BelowOptimum           = 75;
            this.OptimizationSettings.Increment              = 0.0001;
            this.OptimizationSettings.MergedFleet            = false;
            this.OptimizationSettings.BackstopStandard       = true;
            this.OptimizationSettings.BackstopTargetFunction = false;
            this.OptimizationSettings.StartYear              = 0;
            this.OptimizationSettings.StopOnFinedIterations  = false;
            this.OptimizationSettings.CountFinesTowardNB     = false;
            this.OptimizationSettings.DiscardMfrsWithFines   = false;
            this.MonteCarloSettings.TrialPairs    = 10000;
            this.MonteCarloSettings.UseTrialsFile = false;
        }
        #endregion
        #region 
        public TechnologyCollection Technologies
        {
            get { return this._technologies; }
            set
            {
                if (value == null)
                {
                    throw new ArgumentNullException("Technologies", "The specified value cannot be null.");
                }
                this._technologies = value;
            }
        }
        public Parameters Parameters
        {
            get { return this._parameters; }
            set
            {
                if (value == null)
                {
                    throw new ArgumentNullException("Parameters", "The specified value cannot be null.");
                }
                this._parameters = value;
            }
        }
        public EmissionsRates EmissionsRates
        {
            get { return this._emissions; }
            set
            {
                if (value == null)
                {
                    throw new ArgumentNullException("EmissionsRates", "The specified value cannot be null.");
                }
                this._emissions = value;
            }
        }
        public ScenarioCollection Scenarios
        {
            get { return this._scenarios; }
            set
            {
                if (value == null)
                {
                    throw new ArgumentNullException("Scenarios", "The specified value cannot be null.");
                }
                this._scenarios = value;
            }
        }
        #endregion
        #region 
        public ModelingSettings.SettingsChanges Changes;
        public InputSettings      InputSettings;
        public OutputSettings     OutputSettings;
        public EncryptionSettings EncryptionSettings;
        public OperatingModes          OperatingModes;
        public OptimizationSettings    OptimizationSettings;
        public MonteCarloSettings      MonteCarloSettings;
        private TechnologyCollection _technologies;
        private Parameters           _parameters;
        private EmissionsRates       _emissions;
        private ScenarioCollection   _scenarios;
        #endregion
    }
}

