using System;
using Volpe.Cafe.Collections;
using Volpe.Cafe.Data;
using Volpe.Cafe.Data.MonteCarlo;
namespace Volpe.Cafe.Model.MonteCarlo
{
    [Serializable]
    public class TrialGenerator
    {
        #region 
        [Serializable]
        private struct TechInfo
        {
            public double
                FcLow,          
                FcHigh,         
                FcMean,         
                FcStdDev;       
            public double
                CostLow,        
                CostHigh,       
                CostMean,       
                CostStdDev;     
        }
        #endregion
        #region 
        public TrialGenerator(ModelingSettings settings)
        {
            this._trialIndex = 0;
            this._betaRng = new Random(DateTime.Now.Second * DateTime.Now.Hour);
            this._normRng = new Random(DateTime.Now.Second * DateTime.Now.Minute);
            this._settings = settings;
            this._discRates = settings.Parameters.MonteCarlo.DiscountRates;
            this._discRateCount = this._discRates.Length;
            this._forecastData  = settings.Parameters.ForecastData.Clone();
            this._economicCosts = settings.Parameters.EconomicCosts;
            TechnologyCollection techs = this._settings.Technologies;
            this._techCount = techs.Count;
            this._techInfo = new TrialGenerator.TechInfo[this._techCount];
            for (int i = 0; i < this._techCount; i ++)
            {
                Technology tech = techs[i];
                TechnologyAttributesTable techAttr = tech.Attributes;
                double costLow = 0, costHigh = 0, fcLow = 0, fcHigh = 0;
                TechnologyAttributes[] attrs = techAttr.ToArray();
                int attrCount = attrs.Length;
                int sumAttrCount = 0;       
                for (int j = 0; j < attrCount; j++)
                {
                    if (attrs[j].Applicable &&
                        (attrs[j].CostLow != 0 || attrs[j].CostHigh != 0 || attrs[j].FcLow != 0 || attrs[j].FcHigh != 0))
                    {
                        costLow  += attrs[j].CostLow;
                        costHigh += attrs[j].CostHigh;
                        fcLow    += attrs[j].FcLow;
                        fcHigh   += attrs[j].FcHigh;
                        sumAttrCount++;
                    }
                }
                if (sumAttrCount != 0)
                {
                    costLow  /= sumAttrCount;
                    costHigh /= sumAttrCount;
                    fcLow    /= sumAttrCount;
                    fcHigh   /= sumAttrCount;
                }
                this.InitTechInfo(costLow, costHigh, this._settings.Technologies[i].CostVariation,
                    out this._techInfo[i].CostLow, out this._techInfo[i].CostHigh, out this._techInfo[i].CostMean,
                    out this._techInfo[i].CostStdDev);
                this.InitTechInfo(fcLow, fcHigh, this._settings.Technologies[i].FcVariation,
                    out this._techInfo[i].FcLow, out this._techInfo[i].FcHigh, out this._techInfo[i].FcMean,
                    out this._techInfo[i].FcStdDev);
            }
        }
        #endregion
        #region 
        void InitTechInfo(double inLow, double inHigh, double variation,
            out double low, out double high, out double mean,  out double stdDev)
        {
            mean   = (inLow + inHigh) * 0.5;
            low    = mean * (1 - variation);
            high   = mean * (1 + variation);
            stdDev = Math.Abs(high - low) / 6;
        }
        public TrialInfo[] GenerateTrials(int count)
        {
            TrialInfo[] trials = new TrialInfo[count];
            for (int i = 0; i < count; i++)
            {
                trials[i] = this.GenerateTrial();
            }
            return trials;
        }
        public TrialInfo GenerateTrial()
        {
            TrialInfo t;
            if (this._trialIndex == 0)
            {   
                t.Index                = this._trialIndex++;
                t.FuelPriceEstimates   = Estimates.Average;
                t.DiscountRate         = this._discRates[0];
                t.ReboundEffect        = this._settings.Parameters.EconomicValues.ReboundEffect;
                t.CO2Costs             = this._settings.Parameters.EmissionCosts.CO2;
                t.MonopsonyCost        = this._settings.Parameters.EconomicCosts.Monopsony;
                t.PriceShockCost       = this._settings.Parameters.EconomicCosts.PriceShock;
                t.MilitarySecurityCost = this._settings.Parameters.EconomicCosts.MilitarySecurity;
                t.TechCostScaleFactor  = new double[this._techCount];
                t.TechFcScaleFactor    = new double[this._techCount];
                for (int i = 0; i < this._techCount; i++)
                {
                    t.TechCostScaleFactor[i] = 1;
                    t.TechFcScaleFactor  [i] = 1;
                }
            }
            else if (this._trialIndex % this._discRateCount == 0)
            {   
                Parameters.MonteCarloValues mcv = this._settings.Parameters.MonteCarlo;
                t.Index                = this._trialIndex++;
                double fuelPath        = this._normRng.RanUnif();
                t.FuelPriceEstimates   = (fuelPath < mcv.FuelPathLow                        ) ? Estimates.Low     :
                                         (fuelPath < mcv.FuelPathLow + mcv.FuelPathReference) ? Estimates.Average : Estimates.High;
                t.DiscountRate         = this._discRates[0];
                t.ReboundEffect        = Math.Min(mcv.ReboundBase, mcv.ReboundBase + mcv.ReboundScale * this._betaRng.RanBeta(mcv.ReboundAlphaShape, mcv.ReboundBetaShape));
                t.CO2Costs             = this._normRng.RanNormalConstrained(mcv.CO2Mean             , mcv.CO2StandardDeviation             , 8, 0, double.NaN, true);
                t.MonopsonyCost        = this._normRng.RanNormalConstrained(mcv.MonopsonyMean       , mcv.MonopsonyStandardDeviation       , 4, 0, double.NaN, true);
                t.PriceShockCost       = this._normRng.RanNormalConstrained(mcv.PriceShockMean      , mcv.PriceShockStandardDeviation      , 4, 0, double.NaN, true);
                double milSecurityCost = this._normRng.RanUnif();
                t.MilitarySecurityCost = (milSecurityCost <= mcv.MilitarySecurityAlternativeProbability) ? mcv.MilitarySecurityAlternativeCost : this._economicCosts.MilitarySecurity;
                t.TechCostScaleFactor  = new double[this._techCount];
                t.TechFcScaleFactor    = new double[this._techCount];
                for (int i = 0; i < this._techCount; i++)
                {
                    double cost = this._normRng.RanNormal(this._techInfo[i].CostMean, this._techInfo[i].CostStdDev);
                    double fc   = this._normRng.RanNormal(this._techInfo[i].FcMean  , this._techInfo[i].FcStdDev  );
                    t.TechCostScaleFactor[i] = cost / this._techInfo[i].CostMean;
                    t.TechFcScaleFactor  [i] = fc   / this._techInfo[i].FcMean;
                    if (t.TechCostScaleFactor[i] < 0 || this._techInfo[i].CostMean == 0) { t.TechCostScaleFactor[i] = 1.0D; }
                    if (t.TechFcScaleFactor  [i] < 0 || this._techInfo[i].FcMean   == 0) { t.TechFcScaleFactor  [i] = 1.0D; }
                }
            }
            else
            {   
                t = this._lastTrial.Clone();
                t.Index = this._trialIndex++;
                t.DiscountRate = this._discRates[(t.Index % this._discRateCount)];
            }
            this._lastTrial = t;
            return t;
        }
        #endregion
        #region 
        public int TrialIndex { get { return this._trialIndex; } }
        #endregion
        #region 
        private int       _trialIndex;      
        private TrialInfo _lastTrial;       
        private Random _betaRng;
        private Random _normRng;
        private ModelingSettings _settings;
        private int      _techCount;
        private double[] _discRates;        
        private int      _discRateCount;    
        private Parameters.ForecastDataValues  _forecastData;       
        private Parameters.EconomicCostsValues _economicCosts;      
        private TrialGenerator.TechInfo[] _techInfo;    
        #endregion
    }
}

