using System;
using System.Collections.Specialized;
using System.Drawing;
using System.IO;
using Volpe.Utils;
namespace Volpe.Cafe.IO
{
    [Serializable]
    public class Output
    {
        #region 
        public Output(string path)
            : this(path, TemplateType.None, new EncryptionSettings(), false)
        {
        }
        public Output(string path, TemplateType type, EncryptionSettings cryptoSettings, bool saveOnError)
        {
            this.Open(path, cryptoSettings);
            this._type = type;
            this._buffers      = new OutputBuffer[8];
            this._bufferCount  = 0;
            this._activeBuffer = -1;
            this._readOnly     = false;
            this._writing      = false;
            this._cancelWrite  = false;
            this._font          = null;
            this._fillColor     = XlColor.Automatic;
            this._textAlignment = null;
            this._headerFont      = null;
            this._headerFillColor = XlColor.Automatic;
            this._headerAlignment = null;
            this._numberFormat    = null;
            this._repeatHeaders = false;
            this._autoMerge     = false;
            this._autoBorder    = false;
            this._autoFilter    = false;
            this._autoChart     = false;
            this._autoSummarize = false;
            this._landscape = false;
            this._margins   = null;
            this._fitToPage = Size.Empty;
            this._saveOnError = saveOnError;
        }
        ~Output()
        {
            if (!this.Disposed)
            {
                this.Close(false);
            }
        }
        #endregion
        #region 
        protected void Open(string path, EncryptionSettings cryptoSettings)
        {
            XlCryptoSettings crypto;
            string openPw   = null;
            string modifyPw = null;
            if (cryptoSettings.EncryptOutput)
            {
                openPw = cryptoSettings.OutputPasswordToOpen;
                modifyPw = cryptoSettings.OutputPasswordToModify;
                crypto = new XlCryptoSettings();
                crypto.EncryptionProvider  = cryptoSettings.EncryptionType.Provider;
                crypto.EncryptionAlgorithm = cryptoSettings.EncryptionType.Algorithm;
                crypto.EncryptionKeyLength = cryptoSettings.EncryptionKeyLength;
                crypto.EncryptDocumentProperties = cryptoSettings.EncryptDocumentProperties;
            }
            else
            {
                crypto = XlCryptoSettings.Empty;
            }
            if (File.Exists(path)) { File.Delete(path); }
            this._xlu = new XlUtilities(path, false, openPw, modifyPw, crypto);
        }
        public void Save()
        {
            this._xlu.Save();
        }
        public void Close(bool saveChanges)
        {
            this._xlu.Close(saveChanges);
        }
        public void AddBuffer(int rows, int cols)
        {
            this.AddBuffer(new XlSize(rows, cols));
        }
        public void AddBuffer(XlSize size)
        {
            if (this._readOnly) { throw new InvalidOperationException("Cannot add a new buffer; the Output is read-only."); }
            if (this._bufferCount == this._buffers.Length)
            {
                OutputBuffer[] newBuffers = new OutputBuffer[this._bufferCount * 2];
                for (int i = 0; i < this._bufferCount; i++)
                {
                    newBuffers[i] = this._buffers[i];
                }
                this._buffers = newBuffers;
            }
            OutputBuffer buffer = new OutputBuffer(this._bufferCount, size);
            this._buffers[this._bufferCount] = buffer;
            this._bufferCount++;
            this._activeBuffer++;    
        }
        protected void ActivateBuffer(int index)
        {
            if (index >= 0 && index < this._bufferCount) { this._activeBuffer = index; }
        }
        public void RemoveBuffer()
        {
            if (this._readOnly)
            {
                throw new InvalidOperationException("Cannot remove the buffer; " +
                    "the Output is read-only.");
            }
            this._buffers[this._activeBuffer] = null;
            this._bufferCount--;
            this._activeBuffer--;    
        }
        public void RemoveAllBuffers()
        {
            if (this._readOnly)
            {
                throw new InvalidOperationException("Cannot remove the buffers; " +
                    "the Output is read-only.");
            }
            this._buffers = new OutputBuffer[8];
            this._bufferCount = 0;
            this._activeBuffer = -1;
        }
        public void ClearBuffer()
        {
            if (this._readOnly)
            {
                throw new InvalidOperationException("Cannot clear the active buffer; " +
                    "the Output is read-only.");
            }
            this._buffers[this._activeBuffer].Clear();
        }
        public virtual void WriteBuffers(Input template, bool templateFirstBuffer)
        {
            if (this._writing || this._cancelWrite) { return; }
            this._readOnly = true;
            this._writing = true;    
            this._useTemplate = (template != null && this._type != TemplateType.None);
            try
            {   
                if (!this._cancelWrite)
                {
                    this._xlu.DeleteAllWorksheets();
                }
                for (int i = 0; i < this._bufferCount; i++)
                {
                    this._templateFirstBuffer = templateFirstBuffer && (i > 0) &&
                        (this._buffers[0].Size.Rows == this._buffers[i].Size.Rows) &&
                        (this._buffers[0].Size.Columns == this._buffers[i].Size.Columns);
                    if (this._cancelWrite) break;
                    if (this._templateFirstBuffer)
                    {   
                        this._xlu.CopyWorksheet(this._xlu, 1, i + 1);
                    }
                    else if (this._useTemplate)
                    {   
                        this._xlu.CopyWorksheet(template.XlUtils, (int)this._type, i + 1);
                        if (i == 0) { this._xlu.DeleteWorksheet(2); }
                    }
                    else if (i != 0)
                    {
                        this._xlu.AddWorksheet("TempSheet" + i);
                    }
                    this._activeBuffer = i;
                    this.WriteBuffer();
                    this.InsertExcelNames();
                    this.DrawCharts();
                }
            }
            catch (Exception ex)
            {
                if (!this.Disposed) { this.Close(this._saveOnError); }
                throw ex;   
            }
            finally
            {
                this._writing = false;    
            }
        }
        public void CancelWrite()
        {
            this._cancelWrite = true;
            this._readOnly = true;
        }
        protected virtual void WriteBuffer()
        {
            if (this._cancelWrite) return;
            this._xlu.SetData(this.Buffer.Buffer);
            this._xlu.SetWorksheetName(this.Buffer.Name);
            XlSize bufSize = this.Buffer.Size;
            int hRow = this.Buffer.Header.Row;
            int hCol = this.Buffer.Header.Column;
            XlCell topCell = new XlCell(1, 1);        
            XlCell rHeader2 = (hRow > 0) ? new XlCell(hRow, bufSize.Columns) : new XlCell(0, 0);
            XlCell cHeader2 = (hCol > 0) ? new XlCell(bufSize.Rows, hCol) : new XlCell(0, 0);
            XlCell dataCell1 = new XlCell((hRow > 0) ? hRow + 1 : 1, (hCol > 0) ? hCol + 1 : 1);
            XlCell dataCell2 = new XlCell(bufSize.Rows, bufSize.Columns);
            if (!this._cancelWrite) this.SortData(topCell, dataCell2, hRow, hCol);
            if (!this._templateFirstBuffer)
            {
                if (!this._cancelWrite) this.MergeHeaders();
                if (!this._cancelWrite) this.SetFont (topCell, rHeader2, topCell, cHeader2, dataCell1, dataCell2);
                if (!this._cancelWrite) this.SetColor(topCell, rHeader2, topCell, cHeader2, dataCell1, dataCell2);
                if (!this._cancelWrite) this.SetNumberFormat(dataCell1, dataCell2);
                if (!this._cancelWrite) this.SetBorders(topCell, rHeader2, topCell, cHeader2, dataCell1, dataCell2);
                if (!this._cancelWrite) this.SetAutoFilter(topCell, rHeader2, topCell, cHeader2, dataCell1, dataCell2);
                if (!this._cancelWrite) this.AlignText(topCell, rHeader2, topCell, cHeader2, dataCell1, dataCell2);
            }
            if (!this._cancelWrite) this.Autofit(topCell, dataCell2);
            if (!this._templateFirstBuffer)
            {
                if (!this._cancelWrite) this.PageSetup(hRow, hCol);
            }
        }
        protected virtual void InsertExcelNames()
        {
            StringCollection excelNames            = this.Buffer.ExcelNames;
            StringCollection excelNamesExpressions = this.Buffer.ExcelNamesExpressions;
            for (int i = 0, count = this.Buffer.ExcelNameCount; i < count; i++)
            {
                this._xlu.AddExcelName(excelNames[i], excelNamesExpressions[i]);
            }
        }
        protected virtual void DrawCharts()
        {
            if (this._autoChart && this.Buffer.HasCharts)
            {
                XlChartFormat[] formats = this.Buffer.ChartFormats;
                for (int i = 0, count = formats.Length; i < count; i++)
                {
                    try
                    {
                        this._xlu.DrawChart(this.Buffer.Name, formats[i]);
                    }
                    catch
                    {
                        XlChartFormat errFormat = new XlChartFormat(formats[i].Type, formats[i].Name,
                            "This chart could not be drawn -- the input data may have contained errors.",
                            XlColor.White, new XlSeriesFormat("Error!", "1", "1"), null, null);
                        this._xlu.DrawChart(this.Buffer.Name, errFormat);
                    }
                }
            }
        }
        protected virtual void SortData(XlCell cell1, XlCell cell2, int rowHeaders, int columnHeaders)
        {
            OutputBuffer buffer = this.Buffer;
            int[] keys = buffer.SortKeys;
            if (keys.Length > 0)
            {
                bool[] desc = buffer.SortDescending;
                int key1 = keys[0] + 1;
                int key2 = (keys.Length > 1) ? keys[1] + 1: -1;
                int key3 = (keys.Length > 2) ? keys[2] + 1: -1;
                bool des1 = desc[0];
                bool des2 = (desc.Length > 1) ? desc[1] : false;
                bool des3 = (desc.Length > 2) ? desc[2] : false;
                if (rowHeaders > 0)
                {
                    cell1.Row = rowHeaders + 1;
                }
                this._xlu.SortData(cell1, cell2, key1, des1, key2, des2, key3, des3);
            }
        }
        protected virtual void MergeHeaders()
        {
        }
        protected virtual void SetFont(XlCell rHeader1, XlCell rHeader2, XlCell cHeader1, XlCell cHeader2,
            XlCell dataCell1, XlCell dataCell2)
        {
            if (this._headerFont != null)
            {
                if (rHeader1.IsValid && rHeader2.IsValid)
                {
                    this._xlu.SetFont(rHeader1, rHeader2, this._headerFont);
                }
                if (cHeader1.IsValid && cHeader2.IsValid)
                {
                    this._xlu.SetFont(cHeader1, cHeader2, this._headerFont);
                }
            }
            if (this._font != null)
            {
                this._xlu.SetFont(dataCell1, dataCell2, this._font);
            }
        }
        protected virtual void SetColor(XlCell rHeader1, XlCell rHeader2, XlCell cHeader1,
            XlCell cHeader2, XlCell dataCell1, XlCell dataCell2)
        {
            if (this._headerFillColor != XlColor.Automatic)
            {
                if (rHeader1.IsValid && rHeader2.IsValid)
                {
                    this._xlu.SetFillColor(rHeader1, rHeader2, this._headerFillColor);
                }
                if (cHeader1.IsValid && cHeader2.IsValid)
                {
                    this._xlu.SetFillColor(cHeader1, cHeader2, this._headerFillColor);
                }
            }
            if (this._fillColor != XlColor.Automatic)
            {
                this._xlu.SetFillColor(dataCell1, dataCell2, this._fillColor);
            }
        }
        protected virtual void SetNumberFormat(XlCell dataCell1, XlCell dataCell2)
        {
            if (this._numberFormat != null && this._numberFormat.Trim() != "")
            {
                this._xlu.SetNumberFormat(dataCell1, dataCell2, this._numberFormat);
            }
        }
        protected virtual void SetBorders(XlCell rHeader1, XlCell rHeader2, XlCell cHeader1, XlCell cHeader2,
            XlCell dataCell1, XlCell dataCell2)
        {
            if (this._autoBorder)
            {
                this._xlu.SetBorders(dataCell1, dataCell2, false, true, false, false, XlColor.Blue);
                this._xlu.SetBorders(dataCell1, dataCell2, true, false, true, true, XlColor.DarkBlue);
                if (rHeader1.IsValid && rHeader2.IsValid)
                {
                    this._xlu.SetBorders(rHeader1, rHeader2, false, true, true, false, XlColor.DarkBlue);
                }
                if (cHeader1.IsValid && cHeader2.IsValid)
                {
                    this._xlu.SetBorders(cHeader1, cHeader2, false, true, true, false, XlColor.DarkBlue);
                    this._xlu.SetBorders(cHeader1, cHeader2, true, false, true, true, XlColor.DarkBlue);
                }
                if (rHeader1.IsValid && rHeader2.IsValid)
                {
                    this._xlu.SetBorders(rHeader1, rHeader2, true, false, true, true, XlColor.DarkBlue);
                }
            }
        }
        protected virtual void SetAutoFilter(XlCell rHeader1, XlCell rHeader2, XlCell cHeader1, XlCell cHeader2,
            XlCell dataCell1, XlCell dataCell2)
        {
            XlCell cell1 = new XlCell(1, 1);
            XlCell cell2 = new XlCell(1, dataCell2.Column);
            if (rHeader1.IsValid && rHeader2.IsValid)
            {    
                int row = ((rHeader1.Row > rHeader2.Row) ? rHeader1.Row : rHeader2.Row);
                cell1.Row = row;
                cell2.Row = row;
            }
            this._xlu.AutoFilter(cell1, cell2);
        }
        protected virtual void AlignText(XlCell rHeader1, XlCell rHeader2, XlCell cHeader1, XlCell cHeader2,
            XlCell dataCell1, XlCell dataCell2)
        {
            if (this._headerAlignment != null)
            {
                if (rHeader1.IsValid && rHeader2.IsValid)
                {   
                    this._xlu.AlignText(rHeader1, rHeader2, this._headerAlignment);
                }
                if (cHeader1.IsValid && cHeader2.IsValid)
                {   
                    this._xlu.AlignText(cHeader1, cHeader2, this._headerAlignment);
                }
            }
            if (this._textAlignment != null)
            {
                this._xlu.AlignText(dataCell1, dataCell2, this._textAlignment);
            }
        }
        protected virtual void Autofit(XlCell cell1, XlCell cell2)
        {
            this._xlu.AutoFitText(cell1, cell2);
        }
        protected virtual void PageSetup(int rowHeaders, int columnHeaders)
        {
            if (this._useTemplate)
            {    
            }
            else
            {
                this._xlu.PageSetup(this._landscape, this._fitToPage, this._margins,
                    (this._repeatHeaders) ? rowHeaders : -1, (this._repeatHeaders) ? columnHeaders : -1);
                XlHeaderFooterText header = new XlHeaderFooterText();
                header.SetDefaultHeader();
                header.Center = this.Buffer.Title;
                XlHeaderFooterText footer = new XlHeaderFooterText();
                footer.SetDefaultFooter();
                this._xlu.SetHeaderAndFooter(header, footer);
                if (!this.Buffer.FreezePanes.IsEmpty)
                {
                    XlCell freezePanes = new XlCell(this.Buffer.FreezePanes.Row + 1,
                        this.Buffer.FreezePanes.Column + 1);        
                    this._xlu.FreezePanes(freezePanes);
                }
            }
            int[] pageBreaks = this.Buffer.PageBreaks;
            if (pageBreaks.Length > 0)
            {
                this._xlu.InsertPageBreaks(pageBreaks);
            }
        }
        #endregion
        #region 
        public XlUtilities XlUtils
        {
            get { return this._xlu; }
        }
        public TemplateType Type
        {
            get { return this._type; }
        }
        public bool Disposed
        {
            get { return (this._xlu == null || this._xlu.Disposed); }
        }
        protected OutputBuffer[] Buffers
        {
            get { return this._buffers; }
        }
        public OutputBuffer Buffer
        {
            get
            {
                if (this._activeBuffer < 0) { return null; }
                else                        { return this._buffers[this._activeBuffer]; }
            }
        }
        public int BufferCount
        {
            get { return this._bufferCount; }
        }
        public bool ReadOnly
        {
            get { return this._readOnly; }
        }
        public bool WritingBuffers
        {
            get { return this._writing; }
        }
        public XlFont Font
        {
            get { return this._font; }
            set { this._font = value; }
        }
        public XlColor FillColor
        {
            get { return this._fillColor; }
            set { this._fillColor = value; }
        }
        public XlTextAlignment TextAlignment
        {
            get { return this._textAlignment; }
            set { this._textAlignment = value; }
        }
        public XlFont HeaderFont
        {
            get { return this._headerFont; }
            set { this._headerFont = value; }
        }
        public XlColor HeaderFillColor
        {
            get { return this._headerFillColor; }
            set { this._headerFillColor = value; }
        }
        public XlTextAlignment HeaderTextAlignment
        {
            get { return this._headerAlignment; }
            set { this._headerAlignment = value; }
        }
        public string NumberFormat
        {
            get { return this._numberFormat; }
            set { this._numberFormat = value; }
        }
        public bool RepeatHeaders
        {
            get { return this._repeatHeaders; }
            set { this._repeatHeaders = value; }
        }
        public bool AutoMerge
        {
            get { return this._autoMerge; }
            set { this._autoMerge = value; }
        }
        public bool AutoBorder
        {
            get { return this._autoBorder; }
            set { this._autoBorder = value; }
        }
        public bool AutoFilter
        {
            get { return this._autoFilter; }
            set { this._autoFilter = true; }
        }
        public bool AutoChart
        {
            get { return this._autoChart; }
            set { this._autoChart = value; }
        }
        public bool AutoSummarize
        {
            get { return this._autoSummarize; }
            set { this._autoSummarize = value; }
        }
        public bool Landscape
        {
            get { return this._landscape; }
            set { this._landscape = value; }
        }
        public XlMargins Margins
        {
            get { return this._margins; }
            set { this._margins = value; }
        }
        public Size FitToPage
        {
            get { return this._fitToPage; }
            set { this._fitToPage = value; }
        }
        public bool SaveOnError
        {
            get { return this._saveOnError; }
            set { this._saveOnError = true; }
        }
        #endregion
        #region 
        protected XlUtilities _xlu;
        protected TemplateType _type;
        private OutputBuffer[] _buffers;
        private int _bufferCount;
        private int _activeBuffer;
        private bool _readOnly;
        private bool _templateFirstBuffer;
        protected bool _writing;
        protected bool _cancelWrite;
        protected bool _useTemplate;
        protected XlFont _font;
        protected XlColor _fillColor;
        protected XlTextAlignment _textAlignment;
        protected XlFont _headerFont;
        protected XlColor _headerFillColor;
        protected XlTextAlignment _headerAlignment;
        protected string _numberFormat;
        protected bool _repeatHeaders;
        protected bool _autoMerge;
        protected bool _autoBorder;
        protected bool _autoFilter;
        protected bool _autoChart;
        protected bool _autoSummarize;
        protected bool _landscape;
        protected XlMargins _margins;
        protected Size _fitToPage;
        protected bool _saveOnError;
        #endregion
    }
}

