using System;
using System.Runtime.Serialization;
using Volpe.Cafe.IO;
namespace Volpe.Cafe
{
    [Serializable]
    public class Scenario : ISerializable, ICloneable
    {
        #region 
        private Scenario()
        {
        }
        public Scenario(Input input, int index)
        {
            this._index = index;
            this._isBaseline = (index == 0);
            this.Parse(input);
        }
        public Scenario(Scenario prototype, int newIndex, string newDescription)
        {
            prototype.CloneTo(this);
            this._index = newIndex;
            if (newDescription != null && newDescription != "")
            {
                this._description = newDescription;
            }
        }
        private Scenario(SerializationInfo info, StreamingContext context)
            : this()
        {
            this._index            = info.GetInt32("_index");
            this._isBaseline       = info.GetBoolean("_isBaseline");
            this._description      = info.GetString("_description");
            this._regMerging       = (string[] )info.GetValue("_regMerging"      , typeof(string[] ));
            this._autoFunctionType = (int   [] )info.GetValue("_autoFunctionType", typeof(int   [] ));
            this._autoCoefficients = (double[,])info.GetValue("_autoCoefficients", typeof(double[,]));
            this._ltFunctionType   = (int   [] )info.GetValue("_ltFunctionType"  , typeof(int   [] ));
            this._ltCoefficients   = (double[,])info.GetValue("_ltCoefficients"  , typeof(double[,]));
            try
            {
                this._autoMinStndMpg = (double[])info.GetValue("_autoMinStndMpg", typeof(double[]));
                this._autoMinStndPct = (double[])info.GetValue("_autoMinStndPct", typeof(double[]));
                this._ltMinStndMpg   = (double[])info.GetValue("_ltMinStndMpg"  , typeof(double[]));
                this._ltMinStndPct   = (double[])info.GetValue("_ltMinStndPct"  , typeof(double[]));
            }
            catch
            {
                int len = this._regMerging.Length;
                this._autoMinStndMpg = new double[len];
                this._autoMinStndPct = new double[len];
                this._ltMinStndMpg   = new double[len];
                this._ltMinStndPct   = new double[len];
            }
        }
        #endregion
        #region 
        #region 
        public void GetObjectData(SerializationInfo info, StreamingContext context)
        {
            info.AddValue("_index"           , this._index);
            info.AddValue("_isBaseline"      , this._isBaseline);
            info.AddValue("_description"     , this._description);
            info.AddValue("_regMerging"      , this._regMerging);
            info.AddValue("_autoFunctionType", this._autoFunctionType);
            info.AddValue("_autoCoefficients", this._autoCoefficients);
            info.AddValue("_autoMinStndMpg"  , this._autoMinStndMpg);
            info.AddValue("_autoMinStndPct"  , this._autoMinStndPct);
            info.AddValue("_ltFunctionType"  , this._ltFunctionType);
            info.AddValue("_ltCoefficients"  , this._ltCoefficients);
            info.AddValue("_ltMinStndMpg"    , this._ltMinStndMpg);
            info.AddValue("_ltMinStndPct"    , this._ltMinStndPct);
        }
        #endregion
        #region 
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        public Scenario Clone()
        {
            Scenario sn = new Scenario();
            this.CloneTo(sn);
            return sn;
        }
        private void CloneTo(Scenario sn)
        {
            sn._index = this._index;
            sn._isBaseline  = this._isBaseline;
            sn._description = this._description;
            int xBound = this._ltCoefficients.GetUpperBound(0) + 1;
            int yBound = this._ltCoefficients.GetUpperBound(1) + 1;
            sn._regMerging       = new string[        yBound];
            sn._autoFunctionType = new int   [        yBound];
            sn._autoCoefficients = new double[xBound, yBound];
            sn._autoMinStndMpg   = new double[        yBound];
            sn._autoMinStndPct   = new double[        yBound];
            sn._ltFunctionType   = new int   [        yBound];
            sn._ltCoefficients   = new double[xBound, yBound];
            sn._ltMinStndMpg     = new double[        yBound];
            sn._ltMinStndPct     = new double[        yBound];
            for (int i = 0; i < yBound; i++)
            {
                sn._regMerging      [i] = this._regMerging      [i];
                sn._autoFunctionType[i] = this._autoFunctionType[i];
                sn._autoMinStndMpg  [i] = this._autoMinStndMpg  [i];
                sn._autoMinStndPct  [i] = this._autoMinStndPct  [i];
                sn._ltFunctionType  [i] = this._ltFunctionType  [i];
                sn._ltMinStndMpg    [i] = this._ltMinStndMpg    [i];
                sn._ltMinStndPct    [i] = this._ltMinStndPct    [i];
                for (int j = 0; j < xBound; j++)
                {
                    sn._autoCoefficients[j, i] = this._autoCoefficients[j, i];
                    sn._ltCoefficients  [j, i] = this._ltCoefficients  [j, i];
                }
            }
        }
        #endregion
        public override string ToString()
        {
            return this._index + ". " + Global.GetTitleCase(this._description);
        }
        public string ToString(int descrLength)
        {
            int len = this._description.Length;
            bool outOfBounds = (descrLength < len);
            descrLength = (outOfBounds) ? descrLength - 3 : len;
            string descr = Global.GetTitleCase(this._description.Substring(0, descrLength));
            if (outOfBounds) { descr += " ..."; }
            return this._index + ". " + descr;
        }
        private void Parse(Input input)
        {
            int minYear, maxYear;
            input.GetMinMaxYears(4, 2, out minYear, out maxYear);
            int years  = maxYear - minYear + 1;
            int offset = minYear - ModelYear.MinYear;
            int numCoef = 0;
            while (input.GetString(11 + numCoef, 1) != string.Empty)
            {
                numCoef++;
            }
            bool hasAltMin = Global.StringCompare(input.GetString(11 + numCoef + 1, 1), "Alt. Minimum", true);
            int arrLen = maxYear - ModelYear.MinYear + 1;
            this._description      = input.GetString(2, 2);
            this._regMerging       = new string[         arrLen];
            this._autoFunctionType = new int   [         arrLen];
            this._autoCoefficients = new double[numCoef, arrLen];
            this._autoMinStndMpg   = new double[         arrLen];
            this._autoMinStndPct   = new double[         arrLen];
            this._ltFunctionType   = new int   [         arrLen];
            this._ltCoefficients   = new double[numCoef, arrLen];
            this._ltMinStndMpg     = new double[         arrLen];
            this._ltMinStndPct     = new double[         arrLen];
            for (int i = 0; i < years; i++)
            {
                this._regMerging      [i + offset] = input.GetString(5, i + 2);
                this._autoFunctionType[i + offset] = input.GetInt32 (8, i + 2);
                this._autoMinStndMpg  [i + offset] = (hasAltMin) ? input.GetDouble(13 + numCoef, i + 2) : 0;
                this._autoMinStndPct  [i + offset] = (hasAltMin) ? input.GetDouble(14 + numCoef, i + 2) : 0;
                this._ltFunctionType  [i + offset] = input.GetInt32(13 + numCoef + ((hasAltMin) ? 4 : 0), i + 2);
                this._ltMinStndMpg    [i + offset] = (hasAltMin) ? input.GetDouble(26 + numCoef, i + 2) : 0;
                this._ltMinStndPct    [i + offset] = (hasAltMin) ? input.GetDouble(27 + numCoef, i + 2) : 0;
                for (int j = 0; j < numCoef; j++)
                {
                    this._autoCoefficients[j, i + offset] = input.GetDouble(11 + j, i + 2);
                    this._ltCoefficients  [j, i + offset] = input.GetDouble(16 + numCoef + j + ((hasAltMin) ? 4 : 0), i + 2);
                }
            }
        }
        #endregion
        #region 
        public int    Index { get { return this._index; } }
        public bool   IsBaseline { get { return this._isBaseline; } }
        public string Description { get { return this._description; } }
        public string[] RegulatoryMerging { get { return this._regMerging; } }
        public int[]     AutoFunctionType { get { return this._autoFunctionType; } }
        public double[,] AutoCoefficients { get { return this._autoCoefficients; } }
        public double[]  AutoMinStndMpg { get { return this._autoMinStndMpg; } }
        public double[]  AutoMinStndPct { get { return this._autoMinStndPct; } }
        public int[]     LtFunctionType { get { return this._ltFunctionType; } }
        public double[,] LtCoefficients { get { return this._ltCoefficients; } }
        public double[]  LtMinStndMpg { get { return this._ltMinStndMpg; } }
        public double[]  LtMinStndPct { get { return this._ltMinStndPct; } }
        #endregion
        #region 
        private int    _index;
        private bool   _isBaseline;
        private string _description;
        private string[] _regMerging;
        private int[]     _autoFunctionType;
        private double[,] _autoCoefficients;
        private double[]  _autoMinStndMpg;
        private double[]  _autoMinStndPct;
        private int[]     _ltFunctionType;
        private double[,] _ltCoefficients;
        private double[]  _ltMinStndMpg;
        private double[]  _ltMinStndPct;
        #endregion
    }
}

