using System;
using System.Collections.Generic;
using System.Linq;
using Volpe.Cafe;
using Volpe.Cafe.Utils;
using Volpe.Cafe.HelperClasses;

namespace Volpe.Cafe.Data
{
    /// <summary>
    /// Provides an abstract base class for <see cref="Engine"/>s, <see cref="Transmission"/>s, <see cref="Platform"/>s, and
    /// other vehicle components.
    /// </summary>
    /// <seealso cref="Engine"/>
    /// <seealso cref="Transmission"/>
    /// <seealso cref="Platform"/>
    [Serializable]
    public abstract class Component
    {

        #region /*** Nested Types ***/

        /// <summary>
        /// Provides compliance model data for a <see cref="Component"/>, which is updated during runtime.
        /// </summary>
        [Serializable]
        public class CModelData
        {

            #region /*** Ctors ***/

            /// <summary>
            /// Initializes a new instance of the <see cref="Component.CModelData"/> class.
            /// </summary>
            internal CModelData() { }

            #endregion

            #region /*** Methods ***/

            /// <summary>
            /// Creates a new object that is a copy of the current <see cref="Component.CModelData"/> instance.
            /// </summary>
            /// <returns>A new object that is a copy of this <see cref="Component.CModelData"/>.</returns>
            internal Component.CModelData Clone()
            {
                Component.CModelData value = new Component.CModelData();
                //
                value.TechUsed           = (bool[])Interaction.CloneArray(this.TechUsed          , typeof(bool));
                value.TechApplied        = (bool[])Interaction.CloneArray(this.TechApplied       , typeof(bool));
                value.TechAppliedYear    = (int [])Interaction.CloneArray(this.TechAppliedYear   , typeof(int ));
                value.TechSuperseded     = (bool[])Interaction.CloneArray(this.TechSuperseded    , typeof(bool));
                value.TechSupersededYear = (int [])Interaction.CloneArray(this.TechSupersededYear, typeof(int ));
                //
                return value;
            }

            #endregion

            #region /*** Variables ***/

            /// <summary>Specifies that the technology is currently being used by the <see cref="Component"/>.</summary>
            public bool[] TechUsed;
            /// <summary>Specifies that the technology has been applied during the modeling process and is currently being used
            ///   by the <see cref="Component"/>.</summary>
            public bool[] TechApplied;
            /// <summary>Specifies the model year when the technology was applied to the <see cref="Component"/> during the
            ///   modeling process.</summary>
            public int[] TechAppliedYear;
            /// <summary>Specifies that the technology has been used by the <see cref="Component"/> at some point, but was later
            ///   superseded by another technology. Technologies can be marked as superseded if they were present in the
            ///   <see cref="Manufacturer"/>'s product plan, or applied during the modeling process.</summary>
            public bool[] TechSuperseded;
            /// <summary>Specifies the model year when the technology was superseded on the <see cref="Component"/> by another
            ///   technology during the modeling process.</summary>
            public int[] TechSupersededYear;

            #endregion

        }

        /// <summary>
        /// Provides version information for a <see cref="Component"/>.
        /// </summary>
        [Serializable]
        class ComponentVersion
        {

            #region /*** Ctors ***/

            // Private constructor (for cloning).
            ComponentVersion() { }
            /// <summary>
            /// Initializes a new instance of the <see cref="Component.CModelData"/> class.
            /// </summary>
            public ComponentVersion(ulong techKey, int year, int revision)
            {
                this.TechKey   = techKey;
                this.Year      = year;
                this.Revision  = revision;
                this.Revisions = new List<ComponentVersion>();
            }

            #endregion

            #region /*** Methods ***/

            /// <summary>
            /// Creates a new object that is a copy of the current <see cref="Component.ComponentVersion"/> instance.
            /// </summary>
            /// <returns>A new object that is a copy of this <see cref="Component.ComponentVersion"/>.</returns>
            internal ComponentVersion Clone()
            {
                ComponentVersion value = new ComponentVersion();
                //
                value.TechKey   = this.TechKey;
                value.Year      = this.Year;
                value.Revision  = this.Revision;
                value.Revisions = new List<ComponentVersion>();
                for (int i = 0; i < this.Revisions.Count; i++)
                {
                    value.Revisions.Add(this.Revisions[i].Clone());
                }
                //
                return value;
            }

            /// <summary>
            /// Returns the string representation of this <see cref="ComponentVersion"/> instance.
            /// </summary>
            /// <returns>The string representation of the <see cref="ComponentVersion"/> instance.</returns>
            public override string ToString()
            {
                return "{TechKey=" + this.TechKey + ", Year=" + this.Year + ", Revision=" + this.Revision + "}";
            }

            #endregion

            #region /*** Properties ***/

            /// <summary>Gets the version string representing this <see cref="ComponentVersion"/> instance.</summary>
            public string Version
            {
                get
                {
                    return ((this.Year     > 0) ? "v." + this.Year     : (this.Year == -1) ? "pre-baseline" : "baseline") +
                           ((this.Revision > 0) ? "-R" + this.Revision : string.Empty);
                }
            }

            public ulong TechKey { get; private set; }
            public int Year { get; private set; }
            public int Revision { get; private set; }

            public bool HasRevisions { get { return this.Revisions.Count > 0; } }
            public List<ComponentVersion> Revisions { get; private set; }

            #endregion

        }

        #endregion

        #region /*** Constructors ***/

        /// <summary>
        /// Initializes a new <see cref="Component"/>.
        /// </summary>
        protected Component()
        {
            this._modelData          = new Component.CModelData();
            this._versions           = new List<ComponentVersion>();
            this._manufacturer       = null;
            this._vehicles           = new List<Vehicle>();
            this._platformLeader     = null;
            this._platformLeaderCode = -1;
        }

        #endregion

        #region /*** Methods ***/

        /// <summary>
        /// Initializes the <see cref="Component"/> by setting the manufacturer reference and clearing the vehicles list.
        /// </summary>
        internal virtual void Initialize(Manufacturer manufacturer)
        {
            // set manufacturer reference
            this._manufacturer = manufacturer;
            // clear out vehicle lists
            if (this._vehicles == null) { this._vehicles = new List<Vehicle>(); } else { this._vehicles.Clear(); }
        }

        /// <summary>
        /// Returns the version of the component currently in use on the specified vehicle.
        /// </summary>
        /// <param name="veh">The vehicle to search for.</param>
        public string GetComponentVersion(Vehicle veh)
        {
            ulong vehTechKey = this.ComputeTechnologyKey(this.TechnologyList, veh.ModelData.TechUsed, veh.ModelData.TechSuperseded);
            ComponentVersion version = this.FindComponentVersion(vehTechKey, false);
            if (version == null)
            {
                return "UNDEFINED";
                //version = new ComponentVersion(0, 9999, 0);
            }
            return version.Version;
        }
        /// <summary>
        /// Updates the component with version information for the specified model year.
        /// </summary>
        /// <param name="year">The model year for which to update the component version information.</param>
        /// <param name="prev">The component from the previous model year to use for obtaining previous version information. For
        ///   the first model year, this value should be null.</param>
        public void UpdateComponentVersion(ModelYear year, Component prev)
        {
            if (prev != null)
            {   // copy over version infromation from the previous component
                this._versions.Clear();
                for (int i = 0; i < prev._versions.Count; i++)
                {
                    this._versions.Add(prev._versions[i].Clone());
                }
            }
            // update version infromation for this component for the current year
            this.UpdateComponentVersion(year);
        }
        /// <summary>
        /// Updates the component with version information for the specified model year.
        /// </summary>
        /// <param name="year">The model year for which to update the component version information.</param>
        void UpdateComponentVersion(ModelYear year)
        {
            ulong techKey = this.ComputeTechnologyKey(this.TechnologyList, this._modelData.TechUsed, this._modelData.TechSuperseded);
            // check if a component with this key already exists
            if (!this.HasComponentVersion(techKey))
            {
                this.AddComponentVersion(this._versions, techKey, (year == null) ? 0 : year.Year, 0);
            }
            // scan each vehicle in this component to obtain "revisions"
            foreach (Vehicle veh in this._vehicles)
            {   // get tech-key for the vehicle
                ulong vehTechKey = this.ComputeTechnologyKey(this.TechnologyList, veh.ModelData.TechUsed, veh.ModelData.TechSuperseded);
                // search the component for the vehicle tech-key
                if (!this.HasComponentVersion(vehTechKey))
                {   // this vehicle does not use a technology set from a previously defined version of a component --
                    // find a "closest" base version and create a revision of a component to represent this vehicle
                    ComponentVersion baseVersion = this.FindComponentVersion(vehTechKey, true);
                    if (baseVersion != null)
                    {
                        this.AddComponentVersion(baseVersion.Revisions,
                                                 vehTechKey, baseVersion.Year, baseVersion.Revisions.Count + 1);
                    }
                    else if (year == null)
                    {   // if closest match still not found AND evaluating baseline component version, assume vehicle
                        // is at a "pre-baseline" state (this could happen for vehicle platforms with mixed tech usage)
                        this.AddComponentVersion(this._versions, vehTechKey, -1, 0);
                    }
                }
            }
        }
        void AddComponentVersion(List<ComponentVersion> versions, ulong techKey, int year, int revision)
        {
            ComponentVersion version = new ComponentVersion(techKey, year, revision);
            versions.Add(version);
            versions.Sort((x, y) => { return x.TechKey.CompareTo(y.TechKey); });
        }
        ulong ComputeTechnologyKey(int[] techList, bool[] techUsed, bool[] techSuperseded)
        {
            ulong techKey = 0;
            for (int i = 0; i < techList.Length; i++)
            {
                int techIndex = techList[i];
                if (techUsed[techIndex] && !techSuperseded[techIndex])
                {
                    techKey |= KeyHelper.Keys[i];
                }
            }
            return techKey;
        }
        /// <summary>
        /// Searches for a version of a <see cref="Component"/> with the specified technology key.
        /// </summary>
        /// <param name="techKey">The technology key to search for.</param>
        /// <param name="closestBaseVersion">true, to search only the base component versions (excluding their revisions) for a
        ///   closest tech-key match; false, to search all component versions and its revisions for an exact match.</param>
        /// <returns>A version information object, if a version of a <see cref="Component"/> with the specified technology key
        ///   has been created; null, if version information does not exist.</returns>
        ComponentVersion FindComponentVersion(ulong techKey, bool closestBaseVersion)
        {
            return this.FindComponentVersion_Internal(techKey, this._versions, closestBaseVersion);
        }
        ComponentVersion FindComponentVersion_Internal(ulong techKey, List<ComponentVersion> versions, bool closestBaseVersion)
        {
            for (int i = versions.Count - 1; i >= 0; i--)
            {
                ComponentVersion version = versions[i];
                if ((techKey == version.TechKey) || (closestBaseVersion && (techKey & version.TechKey) == version.TechKey))
                {
                    return version;
                }
                else if (!closestBaseVersion && version.HasRevisions)
                {
                    version = this.FindComponentVersion_Internal(techKey, version.Revisions, closestBaseVersion);
                    if (version != null) { return version; }
                }
            }
            // couldn't detect component version -- if using closestBaseVersion option, try again using
            // alternate methods (closest version wasn't detected probably because some tech was superseded)
            if (closestBaseVersion)
            {
                for (int i = 0; i < versions.Count; i++)
                {
                    if (techKey > versions[i].TechKey) { return versions[i]; }
                }
            }
            return null;
        }
        /// <summary>
        /// Determines whether version information with the specified technology key has been created.
        /// </summary>
        /// <param name="techKey">The technology key to search for.</param>
        /// <returns>true, if version information with the specified technology key has been found; false, otherwise.</returns>
        bool HasComponentVersion(ulong techKey)
        {
            return (this.FindComponentVersion(techKey, false) != null);
        }

        /// <summary>
        /// Copies the <see cref="Component"/> instance to the specified value.
        /// </summary>
        /// <param name="value">The <see cref="Component"/> value to copy to.</param>
        protected void CopyTo(Component value)
        {
            value._modelData          = this._modelData.Clone();
            value._manufacturer       = this._manufacturer;
            value._vehicles           = new List<Vehicle>(this._vehicles);
            value._platformLeader     = null; // platform leader needs to be re-assigned from veh code
            value._platformLeaderCode = this._platformLeaderCode;
            //
            for (int i = 0; i < this._versions.Count; i++)
            {
                value._versions.Add(this._versions[i].Clone());
            }
        }

        /// <summary>
        /// Removes the specified vehicle from this <see cref="Component"/>.
        /// </summary>
        /// <param name="veh">The vehicle to remove.</param>
        internal virtual void RemoveVehicle(Vehicle veh)
        {
            this._vehicles.Remove(veh);
            if (this.IsPlatformLeader(veh))
            {
                this._platformLeader     = null;
                this._platformLeaderCode = -1;
            }
        }

        /// <summary>
        /// Determines whether the specified vehicle is the platform leader for this component.
        /// </summary>
        /// <param name="veh">The vehicle to test.</param>
        /// <returns>true, if the specified vehicle is the platform leader for this component; false, otherwise.</returns>
        public bool IsPlatformLeader(Vehicle veh)
        {
            return (veh == this.GetPlatformLeader());
        }
        /// <summary>
        /// Returns the platform leader of the component.
        /// </summary>
        /// <returns>The platform leader of the component.</returns>
        public Vehicle GetPlatformLeader()
        {
            if (this._vehicles == null || this._vehicles.Count == 0) { return null; }
            if (this._platformLeader == null)
            {   // if a veh code for a platform leader exists (probably component was cloned),
                // attempt to scan list of user vehicles
                if (this._platformLeaderCode != -1)
                {
                    this._platformLeader = this.SelectPlatformLeaderByCode(this._vehicles);
                }
                if (this._platformLeader == null)
                {   // if veh code for a platform leader does not exist OR not found in existing
                    // list of user vehicles (probably removed), select new leader
                    this._platformLeader     = this.SelectPlatformLeader(this._vehicles);
                    this._platformLeaderCode = (this._platformLeader == null) ? -1 : this._platformLeader.Description.Code;
                }
            }
            return this._platformLeader;
        }
        Vehicle SelectPlatformLeaderByCode(List<Vehicle> vehicles)
        {
            for (int i = 0; i < vehicles.Count; i++)
            {
                if (vehicles[i].Description.Code == this._platformLeaderCode)
                {
                    return vehicles[i];
                }
            }
            return null;
        }
        /// <summary>
        /// Selects the platform leader for the component from among the specified vehicles (note: all input vehicles must belong
        /// to this component). The base method assigns a platform leader by picking a vehicle with the lowest sales. In the event
        /// of a tie (if multiple vehicles have the same sales), the first vehicle, in chronological order, with the highest MSRP
        /// is selected as a leader. Derived classes may override this method to provide additional functionality.
        /// </summary>
        /// <param name="vehicles">The vehicles from which to compute the platform leader.</param>
        /// <returns>The platform leader of the component.</returns>
        protected virtual Vehicle SelectPlatformLeader(List<Vehicle> vehicles)
        {
            //
            // Pick vehicle leader as follows:
            //  1) filter out vehicles with low sales volumes:
            //      a) scan all vehicles to find maximum number of model years with sales
            //      b) filter for vehicles with at least 2/3 model years with sales
            //      c) if no vehicles found, filter for vehicles with at least 1/3 model years with sales
            //      d) if still no vehicles found, consider all
            //  2) pick leader from the filtered list of vehicles:
            //      a) pick vehicle with lower sales as leader
            //      b) if sales match, pick vehicle with higher MSRP as leader
            //
            List<Vehicle> filteredVehicles = vehicles;
            double maxProdYears = vehicles.Max(v => v.ProductionYears);
            double queryProdYears = Math.Ceiling(maxProdYears * 2 / 3);
            var query = vehicles.Where(item => (item.ProductionYears >= queryProdYears &&
                                                item.Description.ZEVCandidate == HEVType.None));
            if (query.Count() == 0)
            {   // no vehicles found with ~2/3 or greater production years -- try again
                // looking for vehicles with at least ~1/3 or greater production years
                queryProdYears = Math.Ceiling(maxProdYears / 3);
                query = vehicles.Where(item => (item.ProductionYears >= queryProdYears &&
                                                item.Description.ZEVCandidate == HEVType.None));
            }
            if (query.Count() > 0)
            {   // if "2/3" or "1/3" query produced a list of vehicles, use those to
                // pick a leader -- otherwise, default to using the entire list
                filteredVehicles = query.ToList();
            }
            //
            Vehicle leader = null;
            double  lSales = int.MaxValue;
            double  lMSRP  = int.MaxValue;
            //
            for (int i = 0; i < filteredVehicles.Count; i++)
            {
                Vehicle veh = filteredVehicles[i];
                //
                if (veh.Description.ZEVCandidate == HEVType.None)
                {
                    double vSales = veh.Description.CalcAverageSales();
                    double vMSRP  = veh.Description.CalcAverageMSRP();
                    if (vSales < lSales || (vSales == lSales && vMSRP > lMSRP))
                    {
                        leader = veh;
                        lSales = vSales;
                        lMSRP  = vMSRP;
                    }
                }
            }
            //
            return leader;
        }

        /// <summary>
        /// Returns an array of redesign years of the component.
        /// </summary>
        /// <returns>An array of redesign years of the component.</returns>
        public int[] GetRedesignYears()
        {
            Vehicle veh = this.GetPlatformLeader();
            return (veh == null) ? new int[] { } : veh.Description.RedesignYears;
        }
        /// <summary>
        /// Determines whether the component is at redesign for the specified model year.
        /// </summary>
        /// <param name="year">The model year for which to check the <see cref="Component"/>'s redesign state.</param>
        /// <returns>true, if the component is at redesign for the specified model year; false, otherwise.</returns>
        public bool IsAtRedesign(ModelYear year)
        {
            int[] redesignYears = this.GetRedesignYears();
            for (int i = 0; i < redesignYears.Length; i++)
            {
                if (redesignYears[i] == year.Year) { return true; }
            }
            return false;
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>
        /// Gets an array of technology indexes that are applicable to the <see cref="Component"/>.
        /// </summary>
        /// <returns>An array of technology indexes that are applicable to the <see cref="Component"/>.</returns>
        public abstract int[] TechnologyList { get; }

        /// <summary>Gets compliance model data for the <see cref="Component"/>, which is updated during runtime.</summary>
        public Component.CModelData ModelData { get { return this._modelData; } }

        /// <summary>Gets a reference to the <see cref="Manufacturer"/> of the <see cref="Component"/>.</summary>
        public Manufacturer Manufacturer { get { return this._manufacturer; } }
        /// <summary>Gets a list of all <see cref="Vehicle"/>s that use the <see cref="Component"/>.</summary>
        public List<Vehicle> Vehicles { get { return this._vehicles; } }

        #endregion

        #region /*** Variables ***/

        /// <summary>Specifies compliance model data for the <see cref="Component"/>, which is updated during runtime.</summary>
        protected Component.CModelData _modelData;

        /// <summary>Specifies a reference to the <see cref="Manufacturer"/> of the <see cref="Component"/>.</summary>
        protected Manufacturer _manufacturer;
        /// <summary>Represents a list of <see cref="Vehicle"/>s that use the <see cref="Component"/>.</summary>
        protected List<Vehicle> _vehicles;

        Vehicle _platformLeader;
        int _platformLeaderCode;

        /// <summary>Specifies version information for a <see cref="Component"/>.</summary>
        List<ComponentVersion> _versions;

        #endregion

    }
}
