using System;
using System.Collections.Generic;
using Volpe.Cafe;
using Volpe.Cafe.Settings;
using Volpe.Cafe.Utils;
using Volpe.Cafe.Generic;
using Volpe.Cafe.Model;

namespace Volpe.Cafe.Data
{
    /// <summary>
    /// Represents a single vehicle manufacturer.
    /// </summary>
    [Serializable]
    public sealed class Manufacturer
    {

        #region /*** Nested Types ***/

        /// <summary>
        /// Provides a description of engineering characteristics for a <see cref="Manufacturer"/>.
        /// </summary>
        [Serializable]
        public sealed class CDescription
        {

            #region /*** Constructors ***/

            /// <summary>
            /// Initializes a new instance of the <see cref="Manufacturer.CDescription"/> class.
            /// </summary>
            internal CDescription() { }

            #endregion

            #region /*** Methods ***/

            /// <summary>
            /// Creates a new object that is a copy of the current <see cref="Manufacturer.CDescription"/> instance.
            /// </summary>
            /// <returns>A new object that is a copy of this <see cref="Manufacturer.CDescription"/>.</returns>
            internal CDescription Clone()
            {
                Manufacturer.CDescription md = new Manufacturer.CDescription();
                //
                md.Code                 = this.Code;
                md.Name                 = this.Name;
                md.ProperName           = this.ProperName;
                md.PreferFines          = this.PreferFines;
                md.DiscountRate         = this.DiscountRate.Clone();
                md.PaybackPeriod        = this.PaybackPeriod.Clone();
                md.PaybackPeriod_OC     = this.PaybackPeriod_OC.Clone();
                //
                md.BankedCredits        = this.BankedCredits.Clone();
                md.BankedCreditsMinYear = this.BankedCreditsMinYear;
                md.BankedCreditsMaxYear = this.BankedCreditsMaxYear;
                md.FFVCredits           = this.FFVCredits.Clone();
                md.FFVCreditsMinYear    = this.FFVCreditsMinYear;
                md.FFVCreditsMaxYear    = this.FFVCreditsMaxYear;
                //
                md.ZEVSalesShare        = this.ZEVSalesShare;
                md.ZEVCreditShare       = this.ZEVCreditShare;
                //
                return md;
            }

            /// <summary>
            /// Returns the string representation of this <see cref="Manufacturer.CDescription"/> instance.
            /// </summary>
            /// <returns>The string representation of the <see cref="Manufacturer.CDescription"/> instance.</returns>
            public override string ToString()
            {
                return this.ProperName;
            }

            /// <summary>
            /// Returns the manufacturer's available credits, banked from model years preceding the start of analysis, for the
            /// specified regulatory class and model year.
            /// </summary>
            /// <returns>The manufacturer's available banked credits, or zero, if credit information is not available for the
            ///   specified year.</returns>
            public double GetBankedCredits(RegulatoryClass regClass, int year)
            {
                return this.GetCreditsHelper(regClass, year, this.BankedCredits, this.BankedCreditsMinYear, this.BankedCreditsMaxYear);
            }
            /// <summary>
            /// Returns the manufacturer's available FFV credits for the specified regulatory class and model year.
            /// </summary>
            /// <returns>The manufacturer's available FFV credits, or zero, if credit information is not available for the
            ///   specified year.</returns>
            public double GetFFVCredits(RegulatoryClass regClass, int year)
            {
                return this.GetCreditsHelper(regClass, year, this.FFVCredits, this.FFVCreditsMinYear, this.FFVCreditsMaxYear);
            }
            double GetCreditsHelper(RegulatoryClass regClass, int year, RCObject<double[]> data, int minYear, int maxYear)
            {
                if (year < minYear || year > maxYear) { return 0; }
                return data[regClass][year - minYear];
            }

            #endregion

            #region /*** Variables ***/

            /// <summary>Represents the internal manufacturer specific code.</summary>
            public int Code;
            /// <summary>Represents the name of the manufacturer.</summary>
            public string Name;
            /// <summary>Represents the name of the manufacturer, specified as "Title Case".</summary>
            public string ProperName;
            /// <summary>Specifies whether the manufacturer pefers to pay CAFE fines instead of applying technology.</summary>
            public bool PreferFines;
            /// <summary>Specifies the manufacturer specific discount rate to apply to the effective cost calculation for the
            ///   fuel price for the remaining lifetime of the vehicle.</summary>
            public VCXValue<double> DiscountRate;
            /// <summary>Specifies the number of years required for an initial investment to be repaid in the form of future
            ///   benefits or cost savings.</summary>
            public VCXValue<double> PaybackPeriod;
            /// <summary>Specifies the number of years required for an initial investment to be repaid in the form of future
            ///   benefits or cost savings (applicable after reaching compliance).</summary>
            public VCXValue<double> PaybackPeriod_OC;

            /// <summary>Represents the manufacturer's available credits, banked from model years preceding the start of analysis,
            ///   specified by regulatory class.</summary>
            internal RCObject<double[]> BankedCredits;
            internal int BankedCreditsMinYear;
            internal int BankedCreditsMaxYear;

            /// <summary>Represents the manufacturer's available FFV credits, specified by regulatory class.</summary>
            internal RCObject<double[]> FFVCredits;
            internal int FFVCreditsMinYear;
            internal int FFVCreditsMaxYear;

            /// <summary>Gets the percentage of manufacturer's total fleet assumed to be sold in California and S177 states.</summary>
            public double ZEVSalesShare;
            /// <summary>Gets the percentage of manufacturer's ZEV credits assumed to be generated in California and S177 states.</summary>
            public double ZEVCreditShare;

            #endregion

        }

        /// <summary>
        /// Provides compliance model data for a <see cref="Manufacturer"/>, which is updated during runtime.
        /// </summary>
        [Serializable]
        public sealed class CModelData
        {

            #region /*** Constructors ***/

            /// <summary>
            /// Initializes a new instance of the <see cref="Manufacturer.CModelData"/> class.
            /// </summary>
            internal CModelData() { }

            #endregion

            #region /*** Methods ***/

            /// <summary>
            /// Creates a new object that is a copy of the current <see cref="Manufacturer.CModelData"/> instance.
            /// </summary>
            /// <returns>A new object that is a copy of this <see cref="Manufacturer.CModelData"/>.</returns>
            internal Manufacturer.CModelData Clone()
            {
                Manufacturer.CModelData mmd = new Manufacturer.CModelData();
                //
                // clone compliance data
                mmd.KMY    = new KMYType[this.KMY   .Length];
                mmd.KMY_OC = new KMYType[this.KMY_OC.Length];
                //
                for (int i = 0; i < this.KMY   .Length; i++) { mmd.KMY   [i] = this.KMY   [i].Clone(); }
                for (int i = 0; i < this.KMY_OC.Length; i++) { mmd.KMY_OC[i] = this.KMY_OC[i].Clone(); }
                //
                if (this.DefaultStandard != null)
                {
                    mmd.DefaultStandard = new RCDouble[this.DefaultStandard.Length];
                    for (int i = 0; i < this.DefaultStandard.Length; i++)
                    {
                        mmd.DefaultStandard[i] = this.DefaultStandard[i].Clone();
                    }
                }
                mmd.AverageStandard     = this.AverageStandard      .Clone();
                mmd.PreliminaryStandard = this.PreliminaryStandard  .Clone();
                mmd.Standard            = this.Standard             .Clone();
                mmd.Sales               = this.Sales                .Clone();
                mmd.SalesOverFE         = this.SalesOverFE          .Clone();
                mmd.CAFE                = this.CAFE                 .Clone();
                mmd.CAFE_2Bag           = this.CAFE_2Bag            .Clone();
                mmd.Credits             = this.Credits              .Clone();
                mmd.Fines               = this.Fines                .Clone();
                mmd.TCreditsInCapped    = this.TCreditsInCapped     .Clone();
                mmd.TCreditsIn          = this.TCreditsIn           .Clone();
                mmd.TCreditsOut         = this.TCreditsOut          .Clone();
                mmd.TechCost            = this.TechCost             .Clone();
                mmd.RegCost             = this.RegCost              .Clone();
                mmd.DiscCost            = this.DiscCost             .Clone();
                mmd.ConsumerValuation   = this.ConsumerValuation    .Clone();
                mmd.RelativeLossOfValue = this.RelativeLossOfValue  .Clone();
                mmd.MaintenanceCost     = this.MaintenanceCost      .Clone();
                mmd.RepairCost          = this.RepairCost           .Clone();
                mmd.TotalConsumerCosts  = this.TotalConsumerCosts   .Clone();
                mmd.TotalSocialCosts    = this.TotalSocialCosts     .Clone();
                mmd.TaxesAndFees        = this.TaxesAndFees         .Clone();
                mmd.FinancingCost       = this.FinancingCost        .Clone();
                mmd.InsuranceCost       = this.InsuranceCost        .Clone();
                if (this.TechUsedSales != null)
                {
                    int len = TechUsedSales.Length;
                    mmd.TechUsedSales     = new RCDouble[len];
                    mmd.TechAppliedSales  = new RCDouble[len];
                    mmd.TechExhausted     = new bool    [len];
                    for (int i = 0; i < len; i++)
                    {
                        mmd.TechUsedSales    [i] = this.TechUsedSales    [i].Clone();
                        mmd.TechAppliedSales [i] = this.TechAppliedSales [i].Clone();
                        mmd.TechExhausted    [i] = this.TechExhausted    [i];
                    }
                }
                mmd.CFExhausted = this.CFExhausted;
                //
                return mmd;
            }

            #endregion

            #region /*** Variables ***/

            /// <summary>Represents the fuel price for the remaining lifetime of the vehicle.</summary>
            public KMYType[] KMY;
            /// <summary>Represents the fuel price for the remaining lifetime of the vehicle (applicable after reaching compliance).</summary>
            public KMYType[] KMY_OC;

            /// <summary>Represents the default, manufacturer specific CAFE standard, in miles per gallon (mpg), for all model years
            ///   for the manufacturer, per regulatory class.  This value is typically applicable to the optimization model, where it
            ///   is normally assigned, and is not usually used by regular compliance runs.</summary>
            public RCDouble[] DefaultStandard;
            /// <summary>Represents the calculated average CAFE fuel economy standard, in miles per gallon (mpg), for the entire
            ///   industry, per regulatory class.</summary>
            /// <remarks>The average standard value should be reinitialized each time before starting a new model year.  This value
            ///   should be used with scenarios that utilize the "minimum alternative standard" feature.</remarks>
            public RCDouble AverageStandard = new RCDouble();
            /// <summary>Represents the preliminary calculated CAFE fuel economy standard, in miles per gallon (mpg), for the
            ///   manufacturer, per regulatory class.</summary>
            /// <remarks>The preliminary standard value should be cleared each time before starting a new model year.  This value
            ///   should be updated whenever a new standard value is calculated.  The final value of CAFE standard is then based on
            ///   this value and additional constraints.</remarks>
            public RCDouble PreliminaryStandard = new RCDouble();
            /// <summary>Represents the final calculated CAFE fuel economy standard, in miles per gallon (mpg), for the manufacturer,
            ///   per regulatory class.</summary>
            /// <remarks>The value of the CAFE standard should be cleared each time before starting a new model year.</remarks>
            public RCDouble Standard = new RCDouble();

            /// <summary>Represents the overall sales for the manufacturer, per regulatory class, for the last or current model year
            ///   analyzed.</summary>
            /// <remarks>The manufacturer sales should be recalculated each time before starting a new model year.</remarks>
            public RCDouble Sales = new RCDouble();
            /// <summary>Represents the overall sales to fuel economy ratio for the manufacturer, per regulatory class.</summary>
            public RCDouble SalesOverFE = new RCDouble();

            /// <summary>Represents the sales weighted corporate average fuel economy (CAFE) value for the manufacturer, per
            ///   regulatory class.</summary>
            public RCDouble CAFE = new RCDouble();
            /// <summary>Represents the sales weighted corporate average fuel economy (CAFE) value for the manufacturer, per
            ///   regulatory class, aggregated from unadjusted 2-cycle vehicle fuel economy values.</summary>
            public RCDouble CAFE_2Bag = new RCDouble();
            /// <summary>Represents the total credits accumulated by the manufacturer, per regulatory class.</summary>
            /// <remarks>The manufacturer credits, for a given regulatory class (regClass), are defined as:
            ///   <see cref="Sales"/>[regClass] * ( <see cref="CAFE"/>[regClass] - <see cref="Standard"/>[regClass] ).</remarks>
            public RCDouble Credits = new RCDouble();
            /// <summary>Represents the total fines owed by the manufacturer, per regulatory class.</summary>
            /// <remarks>The manufacturer fines, for a given regulatory class (regClass), are defined as:
            ///   -<i>FineRate</i> * Min( <see cref="Credits"/>[regClass], 0 ).</remarks>
            public RCDouble Fines = new RCDouble();

            /// <summary>Represents the total transferable credits transferred into the manufacturer, per regulatory class, which
            ///   are subject to transfer caps (for example, fleet transfers are subject to transfer caps; however, carry forward
            ///   and carry backward are not).</summary>
            /// <remarks>This value should be reset each time before starting a new model year.</remarks>
            public RCDouble TCreditsInCapped = new RCDouble();
            /// <summary>Represents the total transferable credits transferred into the manufacturer, per regulatory class.</summary>
            /// <remarks>This value should be reset each time before starting a new model year.</remarks>
            public RCDouble TCreditsIn = new RCDouble();
            /// <summary>Represents the total transferable credits transferred out of the manufacturer, per regulatory class.</summary>
            /// <remarks>This value should be reset each time before starting a new model year.</remarks>
            public RCDouble TCreditsOut = new RCDouble();

            /// <summary>Represents the total technology costs accumulated by the manufacturer.</summary>
            /// <remarks>This variable should be initialized/reinitialized during model year pre-processing and updated with
            ///   individual vehicle technology costs.</remarks>
            public RCDouble TechCost = new RCDouble();
            /// <summary>Represents the total regulatory costs accumulated by the manufacturer.</summary>
            /// <remarks>This variable should be initialized/reinitialized during model year post-processing typically by a
            ///   cost-allocator.</remarks>
            public RCDouble RegCost = new RCDouble();
            /// <summary>Represents the total discounted technology costs accumulated by the manufacturer.</summary>
            /// <remarks>This variable should be initialized/reinitialized during model year post-processing.</remarks>
            public RCDouble DiscCost = new RCDouble();
            /// <summary>Represents the changes in consumer valuation due to application of technologies accumulated by all
            ///   vehicles of the manufacturer.</summary>
            /// <remarks>This variable should be initialized/reinitialized during model year pre-processing and updated with
            ///   individual vehicle consumer-valuation.</remarks>
            public RCDouble ConsumerValuation = new RCDouble();
            /// <summary>Represents the total relative loss in value to the consumer due to application of technologies accumulated
            ///   by the manufacturer.</summary>
            /// <remarks>This variable should be initialized/reinitialized during model year post-processing.</remarks>
            public RCDouble RelativeLossOfValue = new RCDouble();
            /// <summary>Represents the total technology maintenance costs accumulated by the manufacturer.</summary>
            /// <remarks>This variable should be initialized/reinitialized during model year pre-processing and updated with
            ///   individual vehicle technology costs.</remarks>
            public RCDouble MaintenanceCost = new RCDouble();
            /// <summary>Represents the total technology repair costs accumulated by the manufacturer.</summary>
            /// <remarks>This variable should be initialized/reinitialized during model year pre-processing and updated with
            ///   individual vehicle technology costs.</remarks>
            public RCDouble RepairCost = new RCDouble();

            /// <summary>Represents the total consumer costs accumulated by the manufacturer.</summary>
            public RCDouble TotalConsumerCosts = new RCDouble();
            /// <summary>Represents the total social costs accumulated by the manufacturer.</summary>
            public RCDouble TotalSocialCosts = new RCDouble();

            /// <summary>Represents the total taxes and fees accumulated accross all vehicles by the manufacturer.</summary>
            /// <remarks>This variable should be initialized/reinitialized during model year post-processing.</remarks>
            public RCDouble TaxesAndFees = new RCDouble();
            /// <summary>Represents the total financing costs accumulated accross all vehicles by the manufacturer.</summary>
            /// <remarks>This variable should be initialized/reinitialized during model year post-processing.</remarks>
            public RCDouble FinancingCost = new RCDouble();
            /// <summary>Represents the total insurance costs accumulated accross all vehicles by the manufacturer.</summary>
            /// <remarks>This variable should be initialized/reinitialized during model year post-processing.</remarks>
            public RCDouble InsuranceCost = new RCDouble();

            /// <summary>Specifies the overall sales volumes of technologies that are currently in use by all vehicles of a specific
            ///   manufacturer's fleet.  The volumes are represented for each technology, per regulatory class.</summary>
            /// <remarks>The technology used sales volumes should be reinitialized each time before starting a new model year.  In
            ///   the beginning of the model year, the manufacturer volumes should be calculated based on the current "tech-used"
            ///   states of the manufacturer's fleet.  The volumes <b>should</b> also be updated during the compliance modeling
            ///   process, whenever a new technology is applied to a <see cref="Vehicle"/>.</remarks>
            public RCDouble[] TechUsedSales;
            /// <summary>Specifies the overall sales volumes of technologies that were applied during the compliance modeling process
            ///   to all vehicles of a specific manufacturer's fleet.  The volumes are represented for each technology, per
            ///   regulatory class.</summary>
            /// <remarks>The technology applied sales volumes should be reinitialized each time before starting a new model year.  In
            ///   the beginning of the model year, the manufacturer volumes should be calculated based on the current "tech-applied"
            ///   states of the manufacturer's fleet.  The volumes <b>should</b> also be updated during the compliance modeling
            ///   process, whenever a new technology is applied to a <see cref="Vehicle"/>.</remarks>
            public RCDouble[] TechAppliedSales;
            /// <summary>Specifies that the technologies have exceeded its phase-in threshold for all vehicles of a specific
            ///   manufacturer's fleet.  The exhausted states are represented for each technology, per regulatory class.</summary>
            /// <remarks>The technology exhausted states should be reinitialized each time before starting a new model year.  In the
            ///   beginning of the model year, the exhausted states should be set to true if the manufacturer's
            ///   <see cref="TechAppliedSales"/> exceeds a predefined phase-in percentage of a technology.  The exhausted sates
            ///   should also be reevaluated during the compliance modeling process, whenever a new technology is applied to a
            ///   <see cref="Vehicle"/>.</remarks>
            public bool[] TechExhausted;

            /// <summary>Specifies that the model has exhausted all compliance finding solutions for the current model year.</summary>
            /// <remarks>This variable should be updated during the manufacturer loop.</remarks>
            public bool CFExhausted;

            #endregion

        }

        #endregion


        #region /*** Constructors ***/

        // Private constructor used for internal usage (such as cloning).
        Manufacturer() { }

        /// <summary>
        /// Initializes a new instance of the <see cref="Manufacturer"/> class.
        /// </summary>
        /// <param name="description">A description of engineering characteristics for this <see cref="Manufacturer"/>.</param>
        internal Manufacturer(Manufacturer.CDescription description)
        {
            this._index = -1;
            //
            this._description   = description;
            this._modelData     = null;
            this._vehicles      = new List<Vehicle>     (8);
            this._engines       = new List<Engine>      (8);
            this._transmissions = new List<Transmission>(8);
            this._platforms     = new List<Platform>    (8);
        }

        #endregion


        #region /*** Methods ***/

        #region /* Manufacturer cloning methods */

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="Manufacturer"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="Manufacturer"/>.</returns>
        internal Manufacturer Clone()
        {
            Manufacturer mfr = new Manufacturer();
            //
            // clone the manufacturer
            mfr._index         = this._index;
            //
            mfr._description   = this._description.Clone();
            mfr._modelData     = (this._modelData   == null) ? null : this._modelData  .Clone();
            //
            mfr._vehicles      = this.CloneVehicles     (this._vehicles);
            mfr._engines       = this.CloneEngines      (this._engines);
            mfr._transmissions = this.CloneTransmissions(this._transmissions);
            mfr._platforms     = this.ClonePlatforms    (this._platforms);
            //
            // next, fix engine, transmission, nameplate, and vehicle references
            Engine      [] engs = mfr._engines      .ToArray();
            Transmission[] trns = mfr._transmissions.ToArray();
            Platform    [] plts = mfr._platforms    .ToArray();
            Vehicle     [] vehs = mfr._vehicles     .ToArray();
            //
            // re-initialzie each engine, transmission, and nameplate to set the correct mfr reference and clear their vehicle lists
            //
            // engines
            for (int i = 0; i < engs.Length; i++)
            {
                engs[i].Initialize(mfr);
            }
            // transmissions
            for (int i = 0; i < trns.Length; i++)
            {
                trns[i].Initialize(mfr);
            }
            // platforms
            for (int i = 0; i < plts.Length; i++)
            {
                plts[i].Initialize(mfr);
            }
            //
            // re-initialize each vehicle
            for (int i = 0; i < vehs.Length; i++)
            {   // set mfr, engine, transmission, and predecessor references, and set up engine and transmission vehicle lists
                vehs[i].Initialize(mfr);

                // parse and initialize platforms for each vehicle
                Platform plt = mfr.GetPlatform(vehs[i]);
                plt.Vehicles.Add(vehs[i]);
                // set the nameplate reference for the vehicle
                vehs[i].SetPlatform(plt);
            }
            //
            return mfr;
        }
        List<Vehicle> CloneVehicles(List<Vehicle> list)
        {
            // initialize a new list instance
            List<Vehicle> newList = new List<Vehicle>(list.Count);
            // traverse the current list and clone each element
            for (int i = 0, count = list.Count; i < count; i++)
            {
                newList.Add(list[i].Clone());
            }
            // return the new list
            return newList;
        }
        List<Engine> CloneEngines(List<Engine> list)
        {
            // initialize a new list instance
            List<Engine> newList = new List<Engine>(list.Count);
            // traverse the current list and clone each element
            for (int i = 0, count = list.Count; i < count; i++)
            {
                newList.Add(list[i].Clone());
            }
            // return the new list
            return newList;
        }
        List<Transmission> CloneTransmissions(List<Transmission> list)
        {
            // initialize a new list instance
            List<Transmission> newList = new List<Transmission>(list.Count);
            // traverse the current list and clone each element
            for (int i = 0, count = list.Count; i < count; i++)
            {
                newList.Add(list[i].Clone());
            }
            // return the new list
            return newList;
        }
        List<Platform> ClonePlatforms(List<Platform> list)
        {
            // initialize a new list instance
            List<Platform> newList = new List<Platform>(list.Count);
            // traverse the current list and clone each element
            for (int i = 0, count = list.Count; i < count; i++)
            {
                newList.Add(list[i].Clone());
            }
            // return the new list
            return newList;
        }

        #endregion

        /// <summary>
        /// Returns the string representation of this <see cref="Manufacturer"/> instance.
        /// </summary>
        /// <returns>The string representation of the <see cref="Manufacturer"/> instance.</returns>
        public override string ToString()
        {
            return this._description.Name;
        }

        /// <summary>
        /// Initializes the <see cref="Manufacturer"/>.
        /// </summary>
        /// <param name="vehicles">A list of <see cref="Vehicle"/>s produced by the <see cref="Manufacturer"/>.</param>
        /// <param name="engines">A list of <see cref="Engine"/>s produced by the <see cref="Manufacturer"/>.</param>
        /// <param name="transmissions">A list of <see cref="Transmission"/>s produced by the <see cref="Manufacturer"/>.</param>
        /// <param name="minYear"></param>
        /// <param name="maxYear"></param>
        /// <remarks>
        /// Initialization should occur directly after loading and parsing of the market data file.
        /// </remarks>
        internal void Initialize(List<Vehicle> vehicles, List<Engine> engines, List<Transmission> transmissions, out int minYear, out int maxYear)
        {
            // parse and initialize each engine, transmission, and vehicle
            Engine      [] engs = engines      .ToArray();
            Transmission[] trns = transmissions.ToArray();
            Vehicle     [] vehs = vehicles     .ToArray();

            // parse and initialize each engine
            for (int i = 0; i < engs.Length; i++)
            {
                engs[i].Initialize(this);
                this._engines.Add(engs[i]);
            }

            // parse and initialize each transmission
            for (int i = 0; i < trns.Length; i++)
            {
                trns[i].Initialize(this);
                this._transmissions.Add(trns[i]);
            }

            // initialize and validate each vehicle
            for (int i = 0; i < vehs.Length; i++)
            {
                vehs[i].Initialize(this);

                // parse and initialize nameplates for each vehicle
                Platform plt = this.GetPlatform(vehs[i]);
                plt.Vehicles.Add(vehs[i]);
                // set the nameplate reference for the vehicle
                vehs[i].SetPlatform(plt);
                // add vehicle to the list
                this._vehicles.Add(vehs[i]);
            }

            // obtain min and max model years available for modeling from all vehicles
            List<Vehicle> vc = this.Vehicles;
            minYear = int.MaxValue;
            maxYear = -1;
            for (int i = 0, vehCount = this.VehicleCount; i < vehCount; i++)
            {
                Vehicle v = vc[i];
                if (v.MinYear != null)
                {
                    minYear = Math.Min(v.MinYear.Year, minYear);
                    maxYear = Math.Max(v.MaxYear.Year, maxYear);
                }
            }
        }
        Platform GetPlatform(Vehicle veh)
        {
            string vehPlatformName = veh.Description.Platform;
            // check for existing platform
            for (int i = 0; i < this._platforms.Count; i++)
            {
                if (this._platforms[i].Name.Equals(vehPlatformName, StringComparison.InvariantCultureIgnoreCase))
                {
                    return this._platforms[i];
                }
            }
            // platform does not exist -- create a new one and it to the list
            Platform value = new Platform(vehPlatformName);
            value.Initialize(this);
            this._platforms.Add(value);
            //
            return value;
        }

        /// <summary>
        /// Calculates and returns total employment hours for all vehicles produced by the <see cref="Manufacturer"/>, specified
        /// as thousand labor-hours, for the specified model year.
        /// </summary>
        /// <param name="year">The model year for which to compute employment hours.</param>
        /// <param name="regClass">The regulatory class based on which to determine the compliance modeling data to return.
        ///   Alternatively, specify -1 to return the sum of the compliance modeling data of all regulatory classes.</param>
        /// <returns>The total employment hours for all vehicles produced by the <see cref="Manufacturer"/>, specified as thousand
        ///   labor-hours.</returns>
        public double CalcLaborHours(ModelYear year, RegulatoryClass regClass)
        {
            double hours = 0;
            for (int i = 0, vehCount = this._vehicles.Count; i < vehCount; i++)
            {
                if (regClass == RegulatoryClass.All || regClass == this._vehicles[i].RegClass)
                {
                    hours += this._vehicles[i].CalcLaborHours(year);
                }
            }
            return hours;
        }
        /// <summary>
        /// Calculates and returns total employment hours for all vehicles produced by the <see cref="Manufacturer"/>, specified
        /// as thousand labor-hours, for the specified range of model years.
        /// </summary>
        /// <param name="minYear">The lower bound model year for which to compute employment hours.</param>
        /// <param name="maxYear">The upper bound model year for which to compute employment hours.</param>
        /// <param name="regClass">The regulatory class based on which to determine the compliance modeling data to return.
        ///   Alternatively, specify -1 to return the sum of the compliance modeling data of all regulatory classes, not including
        ///   the unregulated class, or -2 to return the sum of the compliance modeling data of all regulatory classes, including
        ///   the unregulated class.</param>
        /// <returns>The total employment hours for all vehicles produced by the <see cref="Manufacturer"/>, specified as thousand
        ///   labor-hours.</returns>
        public double CalcLaborHours(ModelYear minYear, ModelYear maxYear, RegulatoryClass regClass)
        {
            double hours = 0;
            for (int i = 0, vehCount = this._vehicles.Count; i < vehCount; i++)
            {
                if (regClass == RegulatoryClass.All || regClass == this._vehicles[i].RegClass)
                {
                    hours += this._vehicles[i].CalcLaborHours(minYear, maxYear);
                }
            }
            return hours;
        }

        /// <summary>
        /// Calculates and returns the average vehicle footprint for the manufacturer.
        /// </summary>
        /// <returns>The average vehicle footprint for the manufacturer.</returns>
        public RCDouble CalcAverageFootprint(Scenario scen, ModelYear year)
        {
            RCDouble sales = new RCDouble();
            RCDouble value = new RCDouble();
            //
            foreach (Vehicle veh in this.Vehicles)
            {
                var    vd       = veh.Description;
                var    regClass = veh.RegClass;
                double vehSales = vd .Sales[year.Index];
                //
                sales[regClass] += vehSales;
                value[regClass] += vehSales * vd.Footprint;
            }
            //
            foreach (RegulatoryClass regClass in RCDouble.Classes)
            {
                if (sales[regClass] != 0) { value[regClass] /= sales[regClass]; }
            }
            //
            return value;
        }
        /// <summary>
        /// Calculates and returns the average vehicle curb weight for the manufacturer.
        /// </summary>
        /// <returns>The average vehicle curb weight for the manufacturer.</returns>
        public RCDouble CalcAverageCurbWeight(Scenario scen, ModelYear year)
        {
            RCDouble sales = new RCDouble();
            RCDouble value = new RCDouble();
            //
            foreach (Vehicle veh in this.Vehicles)
            {
                var    vd       = veh.Description;
                var    regClass = veh.RegClass;
                double vehSales = vd .Sales[year.Index];
                //
                sales[regClass] += vehSales;
                value[regClass] += vehSales * vd.CurbWeight;
            }
            //
            foreach (RegulatoryClass regClass in RCDouble.Classes)
            {
                if (sales[regClass] != 0) { value[regClass] /= sales[regClass]; }
            }
            //
            return value;
        }
        /// <summary>
        /// Calculates and returns the average vehicle work factor for the manufacturer.
        /// </summary>
        /// <returns>The average vehicle work factor for the manufacturer.</returns>
        public RCDouble CalcAverageWorkFactor(Scenario scen, ModelYear year)
        {
            RCDouble sales = new RCDouble();
            RCDouble value = new RCDouble();
            //
            foreach (Vehicle veh in this.Vehicles)
            {
                var    vd       = veh.Description;
                var    regClass = veh.RegClass;
                double vehSales = vd .Sales[year.Index];
                //
                sales[regClass] += vehSales;
                value[regClass] += vehSales * Standards.GetWorkFactor(veh, scen, year);
            }
            //
            foreach (RegulatoryClass regClass in RCDouble.Classes)
            {
                if (sales[regClass] != 0) { value[regClass] /= sales[regClass]; }
            }
            //
            return value;
        }

        #endregion


        #region /*** Properties ***/

        /// <summary>Gets the internal index of the <see cref="Manufacturer"/> assigned during load time.</summary>
        public int Index { get { return this._index; } internal set { this._index = value; } }

        // ----- manufacturer characteristics -----
        /// <summary>Gets a description of engineering characteristics for the <see cref="Manufacturer"/>.</summary>
        public Manufacturer.CDescription Description { get { return this._description; } }
        /// <summary>Gets compliance model data for the <see cref="Manufacturer"/>, which is updated during runtime.</summary>
        public Manufacturer.CModelData ModelData { get { return this._modelData; } set { this._modelData = value; } }

        /// <summary>Gets a list of <see cref="Vehicle"/>s that the <see cref="Manufacturer"/> produces.</summary>
        public List<Vehicle> Vehicles { get { return this._vehicles; } }
        /// <summary>Gets the total count of <see cref="Vehicle"/>s that this <see cref="Manufacturer"/> produces.</summary>
        public int VehicleCount { get { return this._vehicles.Count; } }
        /// <summary>Gets a list of <see cref="Engine"/>s that the <see cref="Manufacturer"/> produces.</summary>
        public List<Engine> Engines { get { return this._engines; } }
        /// <summary>Gets a list of <see cref="Transmission"/>s that the <see cref="Manufacturer"/> produces.</summary>
        public List<Transmission> Transmissions { get { return this._transmissions; } }
        /// <summary>Gets a list of <see cref="Vehicle"/>s that the <see cref="Manufacturer"/> produces, split up by <see cref="Platform"/>s.</summary>
        public List<Platform> Platforms { get { return this._platforms; } }

        #endregion


        #region /*** Variables ***/

        int _index; // internal index for each mfr (primarily used for log writing)

        // ----- manufacturer characteristics -----
        Manufacturer.CDescription _description;
        Manufacturer.CModelData   _modelData;

        List<Vehicle>      _vehicles;
        List<Engine>       _engines;
        List<Transmission> _transmissions;
        List<Platform>     _platforms;

        #endregion

    }
}
