using System;

namespace Volpe.Cafe
{
    /// <summary>
    /// Specifies the technology overrides for manufacturers, vehicles, engines, and transmissions; whether a given
    /// technology is used, should be skipped or forced applicability.
    /// </summary>
    [Serializable]
    public enum TechnologyOverride : int
    {
        /// <summary>Specifies that the technology override setting is not set.</summary>
        None = 0,
        /// <summary>Specifies that a specific technology is used on the engine, transmission, or vehicle, or on all
        ///   vehicles of this manufacturer.</summary>
        Used = 1,
        /// <summary>Specifies that the technology override setting is set to skip applicability of a specific
        ///   technology on the engine, tranmission, vehicle, or manufacturer.</summary>
        Skip = 2,
        /// <summary>Specifies that the technology override setting is set to force applicability of a specific
        ///   technology on the engine, tranmission, vehicle, or manufacturer.</summary>
        Force = 3,
        /// <summary>Specifies that a specific technology is not used on the engine, transmission, or vehicle, or on all
        ///   vehicles of this manufacturer.</summary>
        Unused = 4
    }

    /// <summary>
    /// Specifies whether to use low, average, high, or very high estimates for modeling.
    /// </summary>
    /// <remarks>
    /// Applicable to technology cost, fuel consumption, fuel cost, and CO-2 cost.
    /// </remarks>
    [Serializable]
    public enum Estimates : int
    {
        /// <summary>Specifies that the low estimates should be used for modeling.</summary>
        Low = 0,
        /// <summary>Specifies that the average estimates should be used for modeling. This should be the default option for most
        ///   settings.</summary>
        Average = 1,
        /// <summary>Specifies that the high estimates should be used for modeling.</summary>
        High = 2,
        /// <summary>Specifies that the very high estimates should be used for modeling.</summary>
        VeryHigh = 3
    }

    /// <summary>
    /// Specifies the complexity level of a technology.
    /// </summary>
    [Serializable]
    public enum TechnologyComplexity : int
    {
        /// <summary>Specifies that the complexity of a technology is unknown.</summary>
        None = 0,
        /// <summary>Specifies that the technology complexity is low.</summary>
        Low = 1,
        /// <summary>Specifies that the technology complexity is medium.</summary>
        Medium = 2,
        /// <summary>Specifies that the technology complexity is high.</summary>
        High = 3
    }

    /// <summary>
    /// Specifies the learning curve that a technology uses.
    /// </summary>
    [Serializable]
    public enum LearningType : int
    {
        /// <summary>Specifies that the technology does not use a learning curve.</summary>
        None   = 0,
        /// <summary>Specifies that the technology uses a volume-based learning curve.</summary>
        Volume = 1,
        /// <summary>Specifies that the technology uses a time-based learning curve.</summary>
        Time   = 2,
        /// <summary>Specifies that the technology uses a sloping-based learning decline.</summary>
        Slope  = 3,
        /// <summary>Specifies that the technology uses a predefined learning cost table.</summary>
        Table  = 4
    }

    /// <summary>
    /// Specifies the synergy type relation between two technologies.
    /// </summary>
    [Serializable]
    public enum SynergyType : int
    {
        /// <summary>Specifies that no synergy relation exists.</summary>
        None = 0,
        /// <summary>Specifies that the synergy relation between two technologies is to provide accounting adjustments for the
        ///   decision trees.</summary>
        Accounting = 1,
        /// <summary>Specifies that the synergy relation between two technologies is to address physical energy losses.</summary>
        Physical = 2
    }

    /// <summary>
    /// Specifies the vehicle's classification.
    /// </summary>
    [Serializable]
    public enum VehicleClass : int
    {
        /// <summary>Specifies a combination of all available vehicle classes. This value is usefull for reporting purposes.</summary>
        All = -1,
        /// <summary>Specifies that the vehicle's class is unknown or undefined.</summary>
        None = 0,
        /// <summary>Specifies that the vehicle is classfieid as a light duty passenger vehicle.</summary>
        LDV = 1,
        /// <summary>Specifies that the vehicle is classfieid as a class-1 light duty truck, with a gross vehicle weight rating
        ///   (GVWR) ranging from 0 to 6,000 pounds.</summary>
        LDT1 = 2,
        /// <summary>Specifies that the vehicle is classfieid as a class-2a light duty truck, with a gross vehicle weight rating
        ///   (GVWR) ranging from 6,001 to 8,500 pounds.</summary>
        LDT2a = 3,
        /// <summary>Specifies that the vehicle is classfieid as a class-2b light duty truck, with a gross vehicle weight rating
        ///   (GVWR) ranging from 8,501 to 10,000 pounds.</summary>
        LDT2b = 4,
        /// <summary>Specifies that the vehicle is classfieid as a class-3 light duty truck, with a gross vehicle weight rating
        ///   (GVWR) ranging from 10,001 to 14,000 pounds.</summary>
        LDT3 = 5,

        //--- joint classification ---
        /// <summary>Specifies a joint vehicle classification for vehicles that are classfieid as either a class-1 light duty
        ///   truck, with a gross vehicle weight rating (GVWR) ranging from 0 to 6,000 pounds, or a class-2a light duty truck,
        ///   with a gross vehicle weight rating (GVWR) ranging from 6,001 to 8,500 pounds.</summary>
        LDT12a = 101,
        /// <summary>Specifies a joint vehicle classification for vehicles that are classfieid as either a class-2b light duty
        ///   truck, with a gross vehicle weight rating (GVWR) ranging from 8,501 to 10,000 pounds, or a class-3 light duty truck,
        ///   with a gross vehicle weight rating (GVWR) ranging from 10,001 to 14,000 pounds.</summary>
        LDT2b3 = 102,
    }

    /// <summary>
    /// Specifies the vehicle type, whether the vehicle is domestic or imported passenger automobile, light truck, or
    /// heavy light truck.
    /// </summary>
    [Serializable]
    public enum VehicleType : int
    {
        //------------------------------------------------------------------------------------------//
        // ** WARNING **                                                                            //
        //                                                                                          //
        // Do not change the coding of the variable names defined in this enum.                     //
        //                                                                                          //
        // Changing the values of the enumeration constants may render parts of the model unusable. //
        //                                                                                          //
        //------------------------------------------------------------------------------------------//

        /// <summary>Specifies that the vehicle type is unknown or has not yet been determined.</summary>
        None = 0,

        /// <summary>Specifies that the vehicle is a domesticly manufactured automobile.</summary>
        Domestic = 1,
        /// <summary>Specifies that the vehicle is a non-domesticly manufactured automobile.</summary>
        Imported = 2,

        /// <summary>Specifies that the vehicle is a passenger automobile.</summary>
        PassengerAuto = 4,
        /// <summary>Specifies that the vehicle is a light truck.</summary>
        LightTruck = 8,
        /// <summary>Specifies that the vehicle is a heavy light truck.</summary>
        HeavyLT = 16
    }

    /// <summary>
    /// Specifies the vehicle style.
    /// </summary>
    [Serializable]
    public enum VehicleStyle : int
    {
        /// <summary>Specifies that the vehicle style is unknown or has not yet been determined.</summary>
        None = 0,

        /// <summary>Specifies that the vehicle is a Convertible.</summary>
        Convertible = 1,
        /// <summary>Specifies that the vehicle is a Coupe.</summary>
        Coupe = 2,
        /// <summary>Specifies that the vehicle is a Hatchback.</summary>
        Hatchback = 3,
        /// <summary>Specifies that the vehicle is a Sedan.</summary>
        Sedan = 4,
        /// <summary>Specifies that the vehicle is a Wagon.</summary>
        Wagon = 5,

        /// <summary>Specifies that the vehicle is a Sport Utility Vehicle.</summary>
        SportUtility = 6,
        /// <summary>Specifies that the vehicle is a Minivan.</summary>
        Minivan = 7,
        /// <summary>Specifies that the vehicle is a Van.</summary>
        Van = 8,
        /// <summary>Specifies that the vehicle is a Passenger Van.</summary>
        PassengerVan = 9,
        /// <summary>Specifies that the vehicle is a Cargo Van.</summary>
        CargoVan = 10,

        /// <summary>Specifies that the vehicle is a Pickup Truck.</summary>
        Pickup = 11,
        /// <summary>Specifies that the vehicle is a Large Pickup Truck.</summary>
        LargePickup = 12,
        /// <summary>Specifies that the vehicle is a Chassis Cab Truck.</summary>
        ChassisCab = 13,
        /// <summary>Specifies that the vehicle is a Cutaway Van.</summary>
        Cutaway = 14
    }

    /// <summary>
    /// Specifies the vehicle's hybrid/electric type, whether the vehicle is a mild hybrid, strong hybrid, plug-in hybrid, pure
    /// electric vehicle, or fuel-cell vehicle.
    /// </summary>
    [Serializable]
    public enum HEVType : int
    {
        /// <summary>Specifies that the vehicle is not a hybrid/electric.</summary>
        None = 0,

        /// <summary>Specifies that the vehicle is a mild hybrid.</summary>
        MildHybrid = 1,
        /// <summary>Specifies that the vehicle is a strong hybrid.</summary>
        StrongHybrid = 2,
        /// <summary>Specifies that the vehicle is a plug-in hybrid.</summary>
        PlugInHybrid = 3,
        /// <summary>Specifies that the vehicle is a pure-electric vehicle.</summary>
        PureElectric = 4,
        /// <summary>Specifies that the vehicle is a fuel cell vehicle.</summary>
        FuelCell = 5
    }

    /// <summary>
    /// Specifies the fuel type used by a vehicle.  For flex-fuel or multi-fuel vehicles, multiple fuel types may be combined.
    /// </summary>
    [Serializable]
    [Flags]
    public enum FuelType : int
    {
        //------------------------------------------------------------------------------------------//
        // ** WARNING **                                                                            //
        //                                                                                          //
        // Do not change the coding of the variable names defined in this enum.                     //
        //                                                                                          //
        // Changing the values of the enumeration constants may render parts of the model unusable. //
        //                                                                                          //
        //------------------------------------------------------------------------------------------//

        /// <summary>Specifies that the fuel type of a vehicle is unknown or has not yet been determined.</summary>
        None = 0,

        /// <summary>Specifies that the vehicle runs on Gasoline fuel.</summary>
        Gasoline = 1,
        /// <summary>Specifies that the vehicle runs on E85 fuel (blend of 85% Ethanol, 15% Gasoline).</summary>
        Ethanol85 = 2,

        /// <summary>Specifies that the vehicle runs on Diesel fuel.</summary>
        Diesel = 16,

        /// <summary>Specifies that the vehicle runs on Electricity.</summary>
        Electricity = 256,
        /// <summary>Specifies that the vehicle runs on Hydrogen fuel.</summary>
        Hydrogen = 512,
        /// <summary>Specifies that the vehicle runs on Compressed Natural Gas fuel.</summary>
        CNG = 1024,

        /// <summary>Specifies that the vehicle runs on Gasoline and Ethanol-85 fuels.</summary>
        GasolineE85 = Gasoline | Ethanol85,
        /// <summary>Specifies that the vehicle runs on Gasoline fuel and on Electricity.</summary>
        GasolineElectric = Gasoline | Electricity,
        ///// <summary>Specifies that the vehicle runs on Gasoline and Ethanol-85 fuels, and on Electricity.</summary>
        //GasolineE85Electric = Gasoline | Ethanol85 | Electricity,

        /// <summary>Specifies that the vehicle runs on Diesel fuel and on Electricity.</summary>
        DieselElectric = Diesel | Electricity,

        /// <summary>Specifies a combination of all available fuel types.</summary>
        All =
            Gasoline | Ethanol85 |
            Diesel |
            Electricity | Hydrogen | CNG
    }

    /// <summary>
    /// Specifies a pollutant for which vehicle upstream and tailpipe emissions are calculated.
    /// </summary>
    [Serializable]
    public enum Pollutant : int
    {
        //----- emissions for effects modeling -----
        /// <summary>Specifies the Carbon Monoxide pollutant.</summary>
        CO = 0,
        /// <summary>Specifies the Volatile Organic Compound pollutant.</summary>
        VOC = 1,
        /// <summary>Specifies the Nitrogen Oxides pollutant.</summary>
        NOx = 2,
        /// <summary>Specifies the Sulfur Dioxide pollutant.</summary>
        SO2 = 3,
        /// <summary>Specifies the Particulate Matter (diameter of ~2.5 micrometers) pollutant.</summary>
        PM = 4,
        /// <summary>Specifies the Carbon Dioxide pollutant.</summary>
        CO2 = 5,
        /// <summary>Specifies the Methane pollutant.</summary>
        CH4 = 6,
        /// <summary>Specifies the Nitrous Oxide pollutant.</summary>
        N2O = 7,

        //----- additional emissions for EIS modeling -----
        /// <summary>Specifies the Acetaldehyde pollutant.</summary>
        Acetaldehyde = 8,
        /// <summary>Specifies the Acrolein pollutant.</summary>
        Acrolein = 9,
        /// <summary>Specifies the Benzene pollutant.</summary>
        Benzene = 10,
        /// <summary>Specifies the 1,3-Butadiene pollutant.</summary>
        Butadiene = 11,
        /// <summary>Specifies the Formaldehyde pollutant.</summary>
        Formaldehyde = 12,
        /// <summary>Specifies the Diesel Particulate Matter (diameter of ~10 micrometers) pollutant.</summary>
        DPM10 = 13,
        /// <summary>Specifies the Methyl Tertiary Butyl Ether pollutant.</summary>
        MTBE = 14
    }

    /// <summary>
    /// Specifies the CAFE regulatory class assignment of a vehicle.
    /// </summary>
    [Serializable]
    public enum RegulatoryClass : int
    {
        /// <summary>Specifies a combination of all available regulatory classes. This value is usefull for reporting purposes.</summary>
        All = -1,
        /// <summary>Specifies that the vehicle's regulatory class has not been assigned.</summary>
        None = 0,
        /// <summary>Specifies that the vehicle is regulated using passenger car standards.</summary>
        PassengerCar = 1,
        /// <summary>Specifies that the vehicle is regulated using the light truck standards.</summary>
        LightTruck = 2,
        /// <summary>Specifies that the vehicle is regulated using the class-2b/3 light truck standards.</summary>
        LightTruck2b3 = 3
    }

    /// <summary>
    /// Specifies whether the fuel consumption improvement of a technology applies to the vehicle's fuel economy on primary fuel
    /// only, to the vehicle's fuel economy on secondary fuel only, or to the vehicle's fuel economy on all fuel types.
    /// </summary>
    [Serializable]
    public enum FCModLevel : int
    {
        /// <summary>Specifies that the fuel consumption improvement of a technology applies to the vehicle's fuel economy on all
        ///   fuel types.</summary>
        Default = 0,
        /// <summary>Specifies that the fuel consumption improvement of a technology applies to the vehicle's fuel economy on
        ///   primary fuel only.</summary>
        Primary = 1,
        /// <summary>Specifies that the fuel consumption improvement of a technology applies to the vehicle's fuel economy on
        ///   secondary fuel only.</summary>
        Secondary = 2
    }

    /// <summary>
    /// Specifies whether the optimization model will optimize domestic and imported passenger automobiles or light trucks.
    /// </summary>
    [Serializable]
    public enum OptimizationType : int
    {
        /// <summary>Specifies that the model should auto-detect the appropriate optimization type based on the input data.</summary>
        Default = 0,
        /// <summary>Specifies that the domestic and imported passenger automobiles will be optimized.</summary>
        PassengerAuto = 1,
        /// <summary>Specifies that the nonpassenger automobiles will be optimized.</summary>
        LightTruck = 2
    }

    /// <summary>
    /// Specifies the optimization mode the model should use when determining the optimum value of the CAFE standard.
    /// </summary>
    [Serializable]
    public enum OptimizationMode : int
    {
        /// <summary>Specifies that the optimization model should optimize the value of CAFE standard based on the
        ///   difference between the discounted social benefits and technology costs (Disc Benefits - Costs), by
        ///   maximizing that difference.</summary>
        MaximumNetBenefits = 1,
        /// <summary>Specifies that the optimization model should optimize the value of CAFE standard based on the difference
        ///   between the discounted social benefits and technology costs (Disc Benefits - Costs), by finding the lowest positive
        ///   difference after the maximum difference has occured.  This mode only applies to industry optimization; during
        ///   mfr-level optimization, Maximum NB mode will be used.</summary>
        ZeroNetBeneftis = 2,
        /// <summary>Specifies that the optimization model should optimize the value of CAFE standard based on the
        ///   difference between the discounted private benefits and private costs (Disc Benefits - Costs), by
        ///   maximizing that difference.</summary>
        MaximumPrivateNetBenefits = 3
    }

    /// <summary>
    /// Specifies the approach to use when fitting a functional form standard such as:  linear, logistic, or exponential
    /// functions.
    /// </summary>
    [Serializable]
    public enum FunctionalFitMode : int
    {
        /// <summary>The functional fit mode is not specified.  This option would normally be used when using a predefined
        ///   functional shape.</summary>
        None = 0,

        // ----- weighted/unweighted -----
        //
        /// <summary>Specifies that sales-weighted regression will be used to fit the functional form.</summary>
        Weighted   = 1,
        /// <summary>Specifies that un-weighted regression will be used to fit the functional form.</summary>
        Unweighted = 2,

        // ----- various functional fit approaches -----
        //
        /// <summary>Specifies that a Least Squares (LSQ) approach will be used fit the functional form.</summary>
        Lsq = 4,
        /// <summary>Specifies that a Minimum Absolute Deviation (MAD) approach will be used fit the functional form.</summary>
        Mad = 8,
        /// <summary>Specifies that a Two-Step Least Squares (TS-LSQ) approach will be used fit the functional form.</summary>
        /// <remarks>The TS-LSQ approach works by, first approximating the intermediate piecewise linear curve, then fitting the
        ///   user defined function to that curve, using the LSQ approach.</remarks>
        TsLsq = 16,
        /// <summary>Specifies that a Two-Step Minimum Absolute Deviation (TS-MAD) approach will be used fit the functional form.</summary>
        /// <remarks>The TS-MAD approach works by, first approximating the intermediate piecewise linear curve, then fitting the
        ///   user defined function to that curve, using the MAD approach.</remarks>
        TsMad = 32,
        /// <summary>Specifies that a Linear Regression (LNR) approach will be used fit the functional form.</summary>
        /// <remarks>The LNR approach will always produce a linear fit, regardless of the functional form being examined by the
        ///   user.  This approach does not consider the input lower and upper bound asymptotes.</remarks>
        Lnr = 1024,

        // ----- actual functional fit modes -----
        //
        /// <summary>Specifies that the sales-weighted Least Squares (LSQ) approach should be used to fit a functional form.</summary>
        WeightedLsq = Weighted | Lsq,
        /// <summary>Specifies that an un-weighted Least Squares (LSQ) approach should be used to fit a functional form.</summary>
        UnweightedLsq = Unweighted | Lsq,
        /// <summary>Specifies that the sales-weighted Two-Step Least Squares (TS-LSQ) approach should be used to fit a
        ///   functional form.</summary>
        WeightedTsLsq = Weighted | TsLsq,
        /// <summary>Specifies that an un-weighted Two-Step Least Squares (TS-LSQ) approach should be used to fit a functional
        ///   form.</summary>
        UnweightedTsLsq = Unweighted | TsLsq,
        /// <summary>Specifies that the sales-weighted Minimum Absolute Deviation (MAD) approach should be used to fit a
        ///   functional form.</summary>
        WeightedMad = Weighted | Mad,
        /// <summary>Specifies that an un-weighted Minimum Absolute Deviation (MAD) approach should be used to fit a functional
        ///   form.</summary>
        UnweightedMad = Unweighted | Mad,
        /// <summary>Specifies that the sales-weighted Two-Step Minimum Absolute Deviation (TS-MAD) approach should be used to
        ///   fit a functional form.</summary>
        WeightedTsMad = Weighted | TsMad,
        /// <summary>Specifies that an un-weighted Two-Step Minimum Absolute Deviation (TS-MAD) approach should be used to fit a
        ///   functional form.</summary>
        UnweightedTsMad = Unweighted | TsMad,
        /// <summary>Specifies that the sales-weighted Linear Regression (LNR) approach should be used to fit a functional form.</summary>
        /// <remarks>The LNR approach will always produce a linear fit, regardless of the functional form being examined by the
        ///   user.  This approach does not consider the input lower and upper bound asymptotes.</remarks>
        WeightedLnr = Weighted | Lnr,
        /// <summary>Specifies that an un-weighted Linear Regression (LNR) approach should be used to fit a functional form.</summary>
        /// <remarks>The LNR approach will always produce a linear fit, regardless of the functional form being examined by the
        ///   user.  This approach does not consider the input lower and upper bound asymptotes.</remarks>
        UnweightedLnr = Unweighted | Lnr
    }

    /// <summary>
    /// Specifies the calibration mode to use when determining the lower and upper bound asymptotes when using a functional form
    /// standard such as:  linear, logistic, or exponential functions.
    /// </summary>
    [Serializable]
    public enum CalibrationMode : int
    {
        /// <summary>Specifies that the model should auto-choose the appropriate calibration mode based on the
        ///   <see cref="FunctionalFitMode"/> selected by the user.</summary>
        /// <remarks>For weighted functional fit modes, the model will default to <see cref="CalibrationMode.WeightedByPct"/>
        ///   calibration (weighted by percent).  For un-weighted functional fit modes, the model will default to
        ///   <see cref="CalibrationMode.UnweightedByPct"/> calibration (un-weighted by percent).
        /// </remarks>
        Default = Weighted | Unweighted | ByPercent,

        // ----- weighted/unweighted -----
        //
        /// <summary>Specifies that the lower and upper bound asymptotes will be determined using sales weighted averaging.</summary>
        Weighted = 1,
        /// <summary>Specifies that the lower and upper bound asymptotes will be determined using un-weighted averaging.</summary>
        Unweighted = 2,

        // ----- various calibration approaches -----
        //
        /// <summary>Specifies that the lower and upper bound asymptotes will be determined based on a percentage of total fleet
        ///   sales or number of models.</summary>
        ByPercent = 4,
        /// <summary>Specifies that the lower and upper bound asymptotes will be determined based on a vehicle footprint or curb
        ///   weight.</summary>
        ByFpOrCw = 8,
        /// <summary>Specifies that the lower bound asymptote should be determined based on all passenger automobiles, and the
        ///   upper bound asymptote should be determined based on all non-passenger automobiles.</summary>
        ByVehicleType = 16,

        // ----- actual calibration modes -----
        //
        /// <summary>Specifies that the lower and upper bound asymptotes should be determined based on a percentage of total
        ///   fleet sales, using sales weighted averaging.  The lower bound asymptote is computed as a user-specified percent of
        ///   vehicles with a smaller footprint or weight, while the upper bound asymptote is computed as a user-specified
        ///   percent of vehicles with a larger footprint or weight.</summary>
        WeightedByPct = Weighted | ByPercent,
        /// <summary>Specifies that the lower and upper bound asymptotes should be determined based on a percentage of total
        ///   models in the fleet, using un-weighted averaging.  The lower bound asymptote is computed as a user-specified
        ///   percent of vehicles with a smaller footprint or weight, while the upper bound asymptote is computed as a
        ///   user-specified percent of vehicles with a larger footprint or weight.</summary>
        UnweightedByPct = Unweighted | ByPercent,
        /// <summary>Specifies that the lower and upper bound asymptotes should be determined based on a vehicle footprint or
        ///   curb weight, using sales-weighted averaging.  The lower bound asymptote is computed based on vehicles with a
        ///   footprint or curb weight below a user-specified value, while the upper bound asymptote is computed based on
        ///   vehicles with a footprint or curb weight above a user-specified value.</summary>
        WeightedByFpCw = Weighted | ByFpOrCw,
        /// <summary>Specifies that the lower and upper bound asymptotes should be determined based on a vehicle footprint or
        ///   curb weight, using un-weighted averaging.  The lower bound asymptote is computed based on vehicles with a footprint
        ///   or curb weight below a user-specified value, while the upper bound asymptote is computed based on vehicles with a
        ///   footprint or curb weight above a user-specified value.</summary>
        UnweightedByFpCw = Unweighted | ByFpOrCw,
        /// <summary>Specifies that the lower bound asymptote should be determined based on all passenger automobiles, and the
        ///   upper bound asymptote should be determined based on all non-passenger automobiles, using sales weighted averaging.</summary>
        WeightedByVehicleType = Unweighted | ByVehicleType
    }

    /// <summary>
    /// Specifies the equalization mode to use when equalizing certain model years for industry optimization.
    /// </summary>
    [Serializable]
    public enum EqualizationMode : int
    {
        /// <summary>Specifies that the model years should not be equalized.</summary>
        None = 0,
        /// <summary>Specifies that the model years should be equalized based on the incremental costs to the entire
        ///   industry.</summary>
        Cost = 1,
        /// <summary>Specifies that the model years should be equalized based on the average required fuel economy of
        ///   the entire industry.</summary>
        Standard = 2,
        /// <summary>Specifies that the model years should be equalized based on the average achieved fuel economy of
        ///   the entire industry.  This feature is not implemented.</summary>
        FuelEconomy = 3,
        /// <summary>Specifies that the model years should be equalized based on the incremental fuel savings for the
        ///   entire industry.  This feature is not implemented.</summary>
        FuelSavings = 4,
        /// <summary>Specifies that the model years should be equalized based on the incremental Carbon Dioxide (C02)
        ///   for the entire industry.  This feature is not implemented.</summary>
        CO2 = 5
    }

    /// <summary>
    /// Specifies the Mobile6 class assignment of a vehicle.
    /// </summary>
    [Serializable]
    [Obsolete("The use of Mobile6Class is obsolete.", true)]
    public enum Mobile6Class : int
    {
        /// <summary>Specifies that the Mobile6 class is unknown or has not yet been determined.</summary>
        None = 0,
        /// <summary>Specifies that the vehicle is a Light Duty Vehicle.</summary>
        LDV = 1,
        /// <summary>Specifies that the vehicle is a Light Duty Truck with a Gross Vehicle Weight Rating less than or equal to 6000 lbs.</summary>
        LDT1 = 2,
        /// <summary>Specifies that the vehicle is a Light Duty Truck with a Gross Vehicle Weight Rating between 6001 and 8500 lbs.</summary>
        LDT2 = 3,
        /// <summary>Specifies that the vehicle is a Heavy Duty Vehicle with a Gross Vehicle Weight Rating greater than 8500 lbs.</summary>
        HDV = 4
    }

    /// <summary>
    /// Specifies the Safety class assignment of a vehicle.
    /// </summary>
    [Serializable]
    public enum SafetyClass : int
    {
        /// <summary>Specifies that the Safety class is unknown or has not yet been determined.</summary>
        None = 0,

        /// <summary>Specifies that the vehicle belongs to a "PC" Safety class.</summary>
        PC = 1,
        /// <summary>Specifies that the vehicle belongs to a "LT/SUV" Safety class.</summary>
        LT = 2,
        /// <summary>Specifies that the vehicle belongs to a "CUV/Minivan" Safety class.</summary>
        CM = 3

//        // ----- cars -----
//        /// <summary>Specifies that the vehicle belongs to a "Very Small Car" Safety class.</summary>
//        VerySmallCar = 1,
//        /// <summary>Specifies that the vehicle belongs to a "Small Car" Safety class.</summary>
//        SmallCar = 2,
//        /// <summary>Specifies that the vehicle belongs to a "Midsize Car" Safety class.</summary>
//        MidsizeCar = 3,
//        /// <summary>Specifies that the vehicle belongs to a "Large Car" Safety class.</summary>
//        LargeCar = 4,
//        /// <summary>Specifies that the vehicle belongs to a "Sporty 2-Door Car" Safety class.</summary>
//        SportyCar = 5,
//        /// <summary>Specifies that the vehicle belongs to a "Muscle 2-Door Car" Safety class.</summary>
//        MuscleCar = 6,
//        /// <summary>Specifies that the vehicle belongs to an "Economy 2-Door Car" Safety class.</summary>
//        EconomyCar = 7,
//        /// <summary>Specifies that the vehicle belongs to an "Other 2-Door Car" Safety class.</summary>
//        OtherCar = 8,
//        // ----- SUVs -----
//        /// <summary>Specifies that the vehicle belongs to a "Small Utility" Safety class.</summary>
//        SmallUtility = 9,
//        /// <summary>Specifies that the vehicle belongs to a "Midsize Utility" Safety class.</summary>
//        MidsizeUtility = 10,
//        /// <summary>Specifies that the vehicle belongs to a "Large Utility" Safety class.</summary>
//        LargeUtility = 11,
//        /// <summary>Specifies that the vehicle belongs to a "2-Door Utility" Safety class.</summary>
//        TwoDoorUtility = 12,
//        // ----- trucks -----
//        /// <summary>Specifies that the vehicle belongs to a "Compact Pickup" Safety class.</summary>
//        CompactPickup = 13,
//        /// <summary>Specifies that the vehicle belongs to a "Largy Pickup" Safety class.</summary>
//        LargePickup = 14,
//        /// <summary>Specifies that the vehicle belongs to a "Heavy Pickup" Safety class.</summary>
//        HeavyPickup = 15,
//        // ----- vans -----
//        /// <summary>Specifies that the vehicle belongs to a "Minivan" Safety class.</summary>
//        Minivan = 16,
//        /// <summary>Specifies that the vehicle belongs to a "Largy Van" Safety class.</summary>
//        LargeVan = 17,
//
//        //
//        // "double" classes -- some vehicles may belong to two safety classes; these categories are used to represent those vehicles
//        //
//        // ----- cars -----
//        /// <summary>Some vehicles may belong to two Safety classes.  Using this class specifies that the vehicle belongs to the
//        ///   "Very Small Car" and "Economy 2-Door Car" Safety classes.</summary>
//        VerySmallEconomyCar = 107,
//        /// <summary>Some vehicles may belong to two Safety classes.  Using this class specifies that the vehicle belongs to the
//        ///   "Small Car" and "Economy 2-Door Car" Safety classes.</summary>
//        SmallEconomyCar = 207,
//        /// <summary>Some vehicles may belong to two Safety classes.  Using this class specifies that the vehicle belongs to the
//        ///   "Midsize Car" and "Economy 2-Door Car" Safety classes.</summary>
//        MidsizeEconomyCar = 307,
//        /// <summary>Some vehicles may belong to two Safety classes.  Using this class specifies that the vehicle belongs to the
//        ///   "Midsize Car" and "Other 2-Door Car" Safety classes.</summary>
//        MidsizeOtherCar = 308,
//        /// <summary>Some vehicles may belong to two Safety classes.  Using this class specifies that the vehicle belongs to the
//        ///   "Large Car" and "Other 2-Door Car" Safety classes.</summary>
//        LargeOtherCar = 408,
//        // ----- SUVs -----
//        /// <summary>Some vehicles may belong to two Safety classes.  Using this class specifies that the vehicle belongs to the
//        ///   "Small Utility" and "2-Door Utility" Safety classes.</summary>
//        SmallTwoDoorUtility = 912,
//        /// <summary>Some vehicles may belong to two Safety classes.  Using this class specifies that the vehicle belongs to the
//        ///   "Midsize Utility" and "2-Door Utility" Safety classes.</summary>
//        MidsizeTwoDoorUtility = 1012
    }

    /// <summary>
    /// Specfies the CAFE technology class assignment for technology applicability of different vehicle types.
    /// </summary>
    [Serializable]
    [Obsolete("The use of TechnologyClass is obsolete.", true)]
    public enum TechnologyClass : int
    {
        /// <summary>Specifies that the technology applicability is unknown or has not yet been determined.</summary>
        None = 0,
        /// <summary>Specifies that the technology applicability is for subcompact passenger automobiles.</summary>
        SubcompactPC = 1,
        /// <summary>Specifies that the technology applicability is for subcompact performance passenger automobiles.</summary>
        SubcompactPerfPC = 2,
        /// <summary>Specifies that the technology applicability is for compact passenger automobiles.</summary>
        CompactPC = 3,
        /// <summary>Specifies that the technology applicability is for compact performance passenger automobiles.</summary>
        CompactPerfPC = 4,
        /// <summary>Specifies that the technology applicability is for midsize passenger automobiles.</summary>
        MidsizePC = 5,
        /// <summary>Specifies that the technology applicability is for midsize performance passenger automobiles.</summary>
        MidsizePerfPC = 6,
        /// <summary>Specifies that the technology applicability is for large passenger automobiles.</summary>
        LargePC = 7,
        /// <summary>Specifies that the technology applicability is for large performance passenger automobiles.</summary>
        LargePerfPC = 8,
        /// <summary>Specifies that the technology applicability is for minivans.</summary>
        MinivanLT = 9,
        /// <summary>Specifies that the technology applicability is for small light trucks and SUV's.</summary>
        SmallLT = 10,
        /// <summary>Specifies that the technology applicability is for midsize light trucks and SUV's.</summary>
        MidsizeLT = 11,
        /// <summary>Specifies that the technology applicability is for large light trucks and SUV's.</summary>
        LargeLT = 12,
        /// <summary>Specifies that the technology applicability is for class-2b/3 light trucks.</summary>
        Truck2b3 = 13,
        /// <summary>Specifies that the technology applicability is for class-2b/3 vans.</summary>
        Van2b3 = 14,
        /// <summary>Specifies the maximum value of this enumeration, that is, the total count of available technology classes.</summary>
        Max = 14
    }

    /// <summary>
    /// Specfies the relative size of the engine, with respect to technology application.
    /// </summary>
    [Serializable]
    public enum EngineSize : int
    {
        /// <summary>Specifies that the engine size is unknown or has not yet been determined.</summary>
        None = 0,

        /// <summary>Specifies a small sized engine.</summary>
        Small = 1,
        /// <summary>Specifies a medium sized engine.</summary>
        Medium = 2,
        /// <summary>Specifies a large sized engine.</summary>
        Large = 3
    }

    /// <summary>
    /// Specifies whether the credits should be adjusted for total fuel use when transferred between compliance categories or
    /// transferred "as is".
    /// </summary>
    [Serializable]
    public enum CreditAdjustmentMode : int
    {
        /// <summary>Specifies that the credits should be adjusted for total fuel use when transferred between compliance categories,
        ///   according to the following formula: (uVMT * eCAFE * eSTND) / (eVMT * uCAFE * uSTND).</summary>
        Variable = 0,
        /// <summary>Specifies that the credits should be transferred "as is" between compliance categories without adjusting for
        ///   total fuel use. This mode is useful for debugging.</summary>
        Fixed = 1
    }

}
