using System;
using System.IO;
using System.Text;

namespace Volpe.Cafe.IO
{
    /// <summary>
    /// Provides support for logging compliance findings to a text file.
    /// </summary>
    [Serializable]
    public sealed class ComplianceWriter
    {

        #region /*** Constructors ***/

        /// <summary>
        /// Initializes a new instance of the <see cref="ComplianceWriter"/> class for the specified file on the specified path,
        /// using the default encoding and buffer size.  If the file exists, it can be either overwritten or appended to.  If the
        /// file does not exist, this constructor creates a new file.
        /// </summary>
        /// <param name="path">The complete file path to write to.</param>
        /// <param name="append">Specifies whether data is to be appended to the file.  If the file exists and append is false,
        ///   the file is overwritten.  If the file exists and append is true , the data is appended to the file. Otherwise, a
        ///   new file is created.</param>
        public ComplianceWriter(string path, bool append)
        {
            this._lineCount = 0;
            this._sw        = new StreamWriter(path, append);
            //this._sb        = new StringBuilder(1024);
        }

        #endregion


        #region /*** Methods ***/

        /// <summary>
        /// Writes a string to the text stream.
        /// </summary>
        /// <param name="value">The string to write to the stream.</param>
        public void Write(string value)
        {
            this._sw.Write(value);
            int index = -1;
            while (true)
            {
                index = value.IndexOf(NewLine, ++index);
                if (index == -1) { break; }
                this._lineCount++;
            }
        }
        /// <summary>
        /// Writes a string followed by a line terminator to the text stream.
        /// </summary>
        /// <param name="value">The string to write to the stream.  If value is null, only the line termination characters are written.</param>
        public void WriteLine(string value)
        {
            this.Write(value);
            this._sw.Write(NewLine);
            this._lineCount++;
        }
        /// <summary>
        /// Closes the current writer and releases any system resources associated with it.
        /// </summary>
        public void Close()
        {
            this._sw.Close();
        }

        #endregion


        #region /*** Properties ***/

        /// <summary>Gets the total number of lines written to the text stream thus far.</summary>
        public int LineCount { get { return this._lineCount; } }

        #endregion


        #region /*** Variables ***/

        const string NewLine = "\n";

        int           _lineCount;
        StreamWriter  _sw;
        //StringBuilder _sb;

        #endregion

    }
}
