﻿#region << Using Directives >>
using System;
using System.IO;
using Volpe.Cafe.Data;
using Volpe.Cafe.Model;
using Volpe.Cafe.Settings;
using Volpe.Cafe.Utils;
using Volpe.XlLib;
#endregion

namespace Volpe.Cafe.IO.Reporting.XL
{
    /// <summary>
    /// Provides the default Excel report generator for the compliance model.
    /// </summary>
    [Serializable]
    public class XlReportGenerator
    {

        #region /*** Ctors ***/

        XlReportGenerator()
        {
            this._isEmpty = true;
        }
        /// <summary>
        /// Initializes a new instance of the <see cref="XlReportGenerator"/> class.
        /// </summary>
        public XlReportGenerator(ICompliance compliance)
        {
            this._compliance = compliance;
            this._compliance.ScenarioCompleted += new ModelingProgressEventHandler(this.Compliance_ScenarioCompleted);
            this._compliance.ModelingCompleted += new ModelingEventHandler(this.Compliance_ModelingStoppedOrCompleted);
            this._compliance.ModelingStopped   += new ModelingEventHandler(this.Compliance_ModelingStoppedOrCompleted);
            this._isEmpty = false;
            //
            this._minYear = compliance.Data.MinYear.Year;
            this._maxYear = compliance.Data.MaxYear.Year;
        }

        #endregion


        #region /*** Methods ***/

        void Compliance_ModelingStoppedOrCompleted(object sender, ModelingEventArgs e)
        {
            // de-register events
            this._compliance.ScenarioCompleted -= new ModelingProgressEventHandler(this.Compliance_ScenarioCompleted);
            this._compliance.ModelingCompleted -= new ModelingEventHandler(this.Compliance_ModelingStoppedOrCompleted);
            this._compliance.ModelingStopped   -= new ModelingEventHandler(this.Compliance_ModelingStoppedOrCompleted);
        }
        void Compliance_ScenarioCompleted(object sender, ModelingProgressEventArgs e)
        {
            this.GenerateReportsForScenario((e.Progress == null) ? null : e.Progress.Scenario);
        }
        /// <summary>
        /// Begins generating reports for the specified <see cref="Scenario"/>.
        /// </summary>
        /// <param name="scen">The compliance scenario for which to generate modeling reports.</param>
        protected void GenerateReportsForScenario(Scenario scen)
        {
            if (scen == null || this.IsModelStopped()) { return; }
            //
            try
            {
                this._reporting = true;
                this._reportingProgress = "Initializing reports ...";

                this.OpenReports(scen.Index);   // open reports

                if (!this.IsModelStopped())
                {   // get data
                    Industry[] scenData = this._compliance.GetData(scen);
                    Industry[] baseData = (scen.Index == 0) ? scenData : this._compliance.GetData(this._compliance.Settings.Scenarios[0]);
                    Industry   ppData   = this._compliance.Data;
                    // parse data
                    for (int i = 0; i < this._compliance.ModelYears.Length; i++)
                    {
                        if (this.IsModelStopped()) { break; }
                        //
                        ModelYear year = this._compliance.ModelYears[i];
                        this.WriteReportData(scen, year, scenData[year.Index], baseData[year.Index], ppData);
                    } // next i (model year)
                }
            }
            catch (OutputException ex) { throw ex; }
            catch (Exception       ex) { throw new OutputException("<unknown file>", null, ex); }
            finally
            {
                this.CloseReports(scen.Index);  // save & close the reports
                //
                this._reportingProgress = "Reporting for scenario " + scen.Index + " completed.";
                this._reporting = false;
            }
        }
        /// <summary>
        /// Opens modeling reports.
        /// </summary>
        protected virtual void OpenReports(int snIndex) { }
        /// <summary>
        /// Writes report data.
        /// </summary>
        /// <param name="scen"></param>
        /// <param name="year"></param>
        /// <param name="scenData">Data from the current sceanrio.</param>
        /// <param name="baseData">Data from the baseline sceanrio.</param>
        /// <param name="ppData">Data from the product plan.</param>
        protected virtual void WriteReportData(Scenario scen, ModelYear year, Industry scenData, Industry baseData, Industry ppData) { }
        /// <summary>
        /// Saves and closes any open modeling reports.
        /// </summary>
        protected virtual void CloseReports(int snIndex) { }
        /// <summary>
        /// Saves and closes the specified report.  If modeling was stopped, the report will be closed without being saved.
        /// </summary>
        /// <param name="report">The Excel report to close.</param>
        protected void CloseReport(XlReportingBase report)
        {
            if (report != null && !report.Disposed)
            {
                this.UpdateProgress("Saving " + report.FileName + " report.");
                bool saveReport = !this.IsModelStopped();
                report.Close(saveReport);
                if (!saveReport) { File.Delete(report.Path); }
            }
        }

        /// <summary>
        /// Updates the reporting progress with the specified progress string.
        /// </summary>
        protected void UpdateProgress(string progress)
        {
            this._reportingProgress = progress;
        }
        /// <summary>
        /// Returns true, if model state is stopped or stopping; false, otherwise.
        /// </summary>
        protected bool IsModelStopped()
        {
            return (this._compliance.Stopped || this._compliance.State == ModelingState.StopRequested);
        }

        #endregion


        #region /*** Properties ***/

        /// <summary>Gets whether the generator is currently parsing or writing reports.</summary>
        public bool Reporting { get { return !this._isEmpty && this._reporting; } }
        /// <summary>Gets the report generator's progress.</summary>
        public string ReportingProgress { get { return this._reportingProgress; } }

        /// <summary>Gets the compliance model with which this <see cref="XlReportGenerator"/> is associated.</summary>
        protected ICompliance Compliance { get { return this._compliance; } }

        /// <summary>Gets the default directory where output files are saved to.</summary>
        protected string OutputPath { get { return this._compliance.Settings.OutputSettings.OutputPath + "\\reports-xls"; } }

        #endregion


        #region /*** Variables ***/

        // ----- constant/readonly variables -----
        /// <summary>Represents an empty <see cref="XlReportGenerator"/> instance.  This value is read-only.</summary>
        public static readonly XlReportGenerator Empty = new XlReportGenerator();

        // ----- compliance & reporting variables -----
        ICompliance _compliance;

        // ----- runtime variables -----
        bool   _isEmpty;
        bool   _reporting;
        string _reportingProgress;

        int _minYear;
        int _maxYear;

        #endregion

    }
}
