#region << Using Directives >>
using System;
using System.Collections.Generic;
using System.Text;
using System.Linq;
using Volpe.Cafe.Data;
using Volpe.Cafe.Generic;
using Volpe.Cafe.IO;
using Volpe.Cafe.Settings;
using TA = Volpe.Cafe.Model.TechnologyApplicability;
using TI = Volpe.Cafe.TechnologyIndexes;
#endregion

namespace Volpe.Cafe.Model
{
    /// <summary>
    /// Provides the standard compliance modeling implementation of the compliance modeling process.
    /// </summary>
    /// <remarks>
    /// Runs a regular compliance model that estimates technology costs and benefits under scenarios with predefined flat or
    /// reformed CAFE standards.
    /// </remarks>
    [Serializable]
    [ModelDescription("Standard Compliance Model", "Runs a regular compliance model that estimates technology costs and " +
         "benefits under scenarios with predefined flat or reformed CAFE standards.", 1.0F)]
    public class Compliance : ComplianceBase
    {

        #region /*** Constructors ***/

        /// <summary>
        /// Initializes a new instance of the <see cref="Compliance"/> class.
        /// </summary>
        public Compliance() : base() { }

        #endregion


        #region /*** Methods ***/

        #region /* ComplianceBase Members */

        /// <summary>
        /// Creates and returns a new <see cref="Compliance"/> instance.
        /// </summary>
        /// <returns>A new <see cref="Compliance"/> instance.</returns>
        public override ICompliance CreateNew()
        {
            return new Compliance();
        }

        /// <summary>
        /// Starts the compliance modeling process.
        /// </summary>
        protected override void StartInternal()
        {
            if (this.PreProcess())
            {   // pre-processing returns true if the the model has not yet completed
                //
                // begin ...
                this.ScenarioLoop();
                this.PostProcess();
            }
        }

        #endregion

        #region /* Main pre-/post-processing methods */

        /// <summary>
        /// Pre-process data and settings for modeling.
        /// </summary>
        /// <returns>true, to begin the modeling process; false, if modeling has previously completed.</returns>
        protected virtual bool PreProcess()
        {
            // the model has previously completed -- no need to run again
            if (this.State == ModelingState.Completed) { return false; }

            // verify manufacturer-specific discount rate and set up kmy-type
            for (int i = 0; i < this.MfrCount; i++)
            {
                Manufacturer mfr = this.Data.Manufacturers[i];
                // initialize model data
                mfr.ModelData = new Manufacturer.CModelData();
                // initialize KMY values
                Estimates priceEstimates = this.Settings.OperatingModes.FuelPriceEstimates;
                mfr.ModelData.KMY    = KMYType.GetKmyValues(this.MinYear, this.MaxYear, this.Settings.Parameters, priceEstimates, mfr, false);
                mfr.ModelData.KMY_OC = KMYType.GetKmyValues(this.MinYear, this.MaxYear, this.Settings.Parameters, priceEstimates, mfr, true );
            }

            // reset progress variables
            this._scen = null;
            this._year = null;
            this._mfr  = null;
            this._effectsYear = -1;

            // initialize the effects model
            this._effectsModel = new EffectsModel(this.Settings, this.MinYear, this.MaxYear, this.MaxEffectsYear);

            // return true to specify that the pre-processing succeeded
            return true;
        }

        /// <summary>
        /// Post-process data and settings for modeling.
        /// </summary>
        protected virtual void PostProcess()
        {
        }

        #endregion

        #region /* Compliance processing loops */

        #region /* Scenario processing */

        /// <summary>
        /// Begins the scenario loop for the compliance modeling process.
        /// </summary>
        protected virtual void ScenarioLoop()
        {
            // start the loop
            for (int i = 0; i < this.ScenCount; i++)
            {   // obtain the scenario to process
                Scenario scen = this.Settings.Scenarios[i];

                // create a copy of the modeling data for the new scenario
                Industry data = this.Data.Clone();
                // start the scenario processing
                this.RunScenario(scen, data);

                // if user requested to terminate modeling, skip remaining scenarios
                if (this.AbortRequested) { break; }
            } // next i (scenario)
        }
        /// <summary>
        /// Begins examining the specified scenario.
        /// </summary>
        /// <param name="scen">The scenario to analyze.</param>
        /// <param name="data">The modeling data to analyze with the current scenario.</param>
        protected virtual void RunScenario(Scenario scen, Industry data)
        {
            this.PreProcessScenario (scen, data);
            this.ModelYearLoop      (scen, data);
            this.PostProcessScenario(scen);
        }
        /// <summary>
        /// Pre-processes the scenario for analysis.
        /// </summary>
        /// <param name="scen">The current scenario being analyzed.</param>
        /// <param name="data">The modeling data being analyzed with the current scenario.</param>
        protected virtual void PreProcessScenario(Scenario scen, Industry data)
        {
            // update progress ...
            this._scen = scen;
            this._year = null;
            this._mfr  = null;
            this._effectsYear = -1;
            this.ResetActiveScenarioData();
            this.OnScenarioStarted();

            // initialize compliance finder
            this._lcf = new LinearComplianceFinder(scen, this.MinYear, this.MaxYear, this.Settings, this.LogWriter);

            // init complaince industry values
            data.ModelData.TechUsedSales = new RCDouble[TI.TechnologyCount];
            for (int j = 0; j < TI.TechnologyCount; j++)
            {
                data.ModelData.TechUsedSales[j] = new RCDouble();
            }

            // iterate through each mfr
            for (int i = 0; i < this.MfrCount; i++)
            {
                Manufacturer            mfr = data.Manufacturers[i];
                Manufacturer.CModelData mmd = mfr.ModelData;

                // init compliance mfr values
                mmd.TechUsedSales     = new RCDouble[TI.TechnologyCount];
                mmd.TechAppliedSales  = new RCDouble[TI.TechnologyCount];
                mmd.TechExhausted     = new bool    [TI.TechnologyCount];
                //
                for (int j = 0; j < TI.TechnologyCount; j++)
                {
                    mmd.TechUsedSales    [j] = new RCDouble();
                    mmd.TechAppliedSales [j] = new RCDouble();
                    //mmd.TechExhausted    [j] = new bool    ();
                }

                // initialize mfr vehicles, engines, transmissions, and platforms
                foreach (Vehicle veh in mfr.Vehicles)
                {
                    veh.RegClass = Standards.GetRegClass(veh);              // assign vehicle reg-class
                    TechnologyApplicability.Initialize(veh, this.Settings); // initialize
                }
                foreach (Engine       eng in mfr.Engines      ) { TechnologyApplicability.Initialize(eng, this.Settings); }
                foreach (Transmission trn in mfr.Transmissions) { TechnologyApplicability.Initialize(trn, this.Settings); }
                foreach (Platform     plt in mfr.Platforms    ) { TechnologyApplicability.Initialize(plt, this.Settings); }
            } // next mfr
        }
        /// <summary>
        /// Post-processes the scenario after the analysis for that scenario has completed.  If the scenario did not complete,
        /// post-processing will not occur.
        /// </summary>
        /// <param name="scen">The current scenario being analyzed.</param>
        protected virtual void PostProcessScenario(Scenario scen)
        {
            // if user requested to terminate modeling, return without post-processing
            if (this.AbortRequested) { return; }

            // start the loop
            for (int i = 0; i < this.YearCount; i++)
            {   // get current model year and start the model year processing
                ModelYear year     = this.ModelYears[i];
                Industry  data     = this.GetData(scen, year);
                Industry  prevData = (i == 0) ? null : this.GetData(scen, this.ModelYears[i - 1]);
                //
                for (int j = 0; j < this.MfrCount; j++)
                {   // update engine, transmission, and platform versions/revisions
                    Manufacturer mfr     =                   data    .Manufacturers[j];
                    Manufacturer prevMfr = (i == 0) ? null : prevData.Manufacturers[j];
                    //
                    for (int k = 0; k < mfr.Engines.Count; k++)
                    {
                        mfr.Engines[k].UpdateComponentVersion(year, (i == 0) ? null : prevMfr.Engines[k]);
                    }
                    for (int k = 0; k < mfr.Transmissions.Count; k++)
                    {
                        mfr.Transmissions[k].UpdateComponentVersion(year, (i == 0) ? null : prevMfr.Transmissions[k]);
                    }
                    for (int k = 0; k < mfr.Platforms.Count; k++)
                    {
                        mfr.Platforms[k].UpdateComponentVersion(year, (i == 0) ? null : prevMfr.Platforms[k]);
                    }
                }

                // if user requested to terminate modeling, skip remaining years
                if (this.AbortRequested) { break; }
            } // next i (model year)

            if (this.Settings.OperatingModes.FleetAnalysis)
            {   // run the Effects Sub-Model for additional years
                ModelYear year  = new ModelYear(this.MaxYear);
                Scenario  bScen = this.Settings.Scenarios[0];
                Industry  bData = this.GetData(bScen, year);
                Industry  data  = this.GetData(scen , year);

                // historic years
                for (int i = this.MinEffectsYear; i < this.MinYear; i++)
                {
                    EffectsData effectsData;
                    this._effectsYear = i;
                    this._effectsModel.CalculateHistoricEffects(scen, i, data.VehClasses, out effectsData);
                    this.SetEffectsData(i, effectsData);

                    // if user requested to terminate modeling, skip remaining effects years
                    if (this.AbortRequested) { return; }
                }

                // future years
                for (int i = this.MaxYear + 1; i <= this.MaxEffectsYear; i++)
                {
                    EffectsData effectsData;
                    this._effectsYear = i;
                    this._effectsModel.CalculateEffects(scen, i, bData.Vehicles, data.Vehicles, out effectsData);
                    this.SetEffectsData(i, effectsData);

                    // if user requested to terminate modeling, skip remaining effects years
                    if (this.AbortRequested) { return; }
                }
                this._effectsYear = -1;
            }

            this._lcf = null;
            this.OnScenarioCompleted();
        }

        #endregion

        #region /* Model year processing */

        /// <summary>
        /// Begins the model year loop for the compliance modeling process.
        /// </summary>
        /// <param name="scen">The current scenario being analyzed.</param>
        /// <param name="data">The modeling data being analyzed with the current scenario.</param>
        protected virtual void ModelYearLoop(Scenario scen, Industry data)
        {
            // save the product plan data into the first year
            this.SetActiveScenarioData(this.ModelYears[0], data);

            // start the loop
            for (int i = 0; i < this.YearCount; i++)
            {   // get current model year and start the model year processing
                ModelYear year = this.ModelYears[i];
                this.RunModelYear(scen, year, true);

                // if user requested to terminate modeling, skip remaining years
                if (this.AbortRequested) { break; }
            } // next i (model year)
        }
        /// <summary>
        /// Begins examining the specified model year.
        /// </summary>
        /// <param name="scen">The current scenario being analyzed.</param>
        /// <param name="year">The model year to analyze.</param>
        /// <param name="delayPostProcessing">true, to delay model year post processing until evaluation of the last model year;
        ///   false, otherwise. This setting takes effect only when multiyear modeling is enabled.</param>
        protected virtual void RunModelYear(Scenario scen, ModelYear year, bool delayPostProcessing)
        {
            this.PreProcessModelYear(scen, year);
            this.ManufacturerLoop   (scen, year);
            if (!delayPostProcessing || !this.Settings.OperatingModes.MultiYearModeling || year.Year == this.MaxYear)
            {
                this.PostProcessModelYear(scen, year);
            }
        }
        /// <summary>
        /// Pre-processes the model year for analysis.
        /// </summary>
        /// <param name="scen">The current scenario being analyzed.</param>
        /// <param name="year">The current model year being analyzed.</param>
        protected virtual void PreProcessModelYear(Scenario scen, ModelYear year)
        {
            // update progress ...
            this._year = year;
            this.OnModelYearStarted();

            // initialize variables ...
            int yrIndex = year.Index;

            // initialize the model year data for the current year
            // (if examining the first year, this was done before entering the loop)
            Industry prevData = null, data = null;
            if (yrIndex != this.MinYearIndex)
            {   // clone the data from the previously completed model year
                prevData = this.GetData(scen, this.ModelYears[yrIndex - this.MinYearIndex - 1]);
                this.SetActiveScenarioData(year, prevData.Clone());
            }
            // get reference to the current year's data
            data = this.GetData(scen, year);

            // adjust PC/LT fleet share, if settings allow
            if (this.Settings.OperatingModes.DynamicFleetShare)
            {
                DynamicFleetShare.AdjustFleetShare(data, prevData, this.Settings, scen, year);
            }

            // clear the tech volumes for the industry
            Industry.CModelData imd = data.ModelData;
            for (int i = 0; i < TI.TechnologyCount; i++) { imd.TechUsedSales[i].Clear(); }

            // get the list of technologies available
            TechnologyInfo[] technologies = this.Settings.Technologies.TechnologyList;

            // initialize manufacturers
            for (int i = 0; i < this.MfrCount; i++)
            {   // iterate through each manufacturer
                Manufacturer            mfr = data.Manufacturers[i];
                Manufacturer.CModelData mmd = mfr.ModelData;

                // clear certain mfr-specific values
                mmd.PreliminaryStandard.Clear();
                mmd.Standard           .Clear();
                mmd.TCreditsInCapped   .Clear();
                mmd.TCreditsIn         .Clear();
                mmd.TCreditsOut        .Clear();
                mmd.TechCost           .Clear();
                mmd.RegCost            .Clear();
                mmd.ConsumerValuation  .Clear();
                mmd.RelativeLossOfValue.Clear();
                mmd.MaintenanceCost    .Clear();
                mmd.RepairCost         .Clear();
                mmd.TaxesAndFees       .Clear();
                mmd.FinancingCost      .Clear();
                mmd.InsuranceCost      .Clear();
                //
                for (int j = 0; j < TI.TechnologyCount; j++)
                {   // clear the tech volumes and exhausted-state for the mfr
                    mmd.TechUsedSales   [j].Clear();
                    mmd.TechAppliedSales[j].Clear();
                    mmd.TechExhausted   [j] = false;
                    //if (yrIndex == this.MinYearIndex || this.Settings.OperatingModes.MfrAdjPhaseInRecalcByYear)
                    //{   // clear out phase-in offsets if analyzing first year or if "recalc offset by year" option is set
                    //    mmd.TechPhaseInOffset[j].Clear();
                    //}
                }

                // update technology enabled states for all technologies on all vehicles
                List<Vehicle> mfrVehs = mfr.Vehicles;
                for (int j = 0, mfrVehCount = mfrVehs.Count; j < mfrVehCount; j++)
                {
                    TechnologyApplicability.Enable(mfrVehs[j], year, -1, this.Settings);
                }

                // update veh/mfr sales and phase-in
                for (int j = 0, mfrVehCount = mfrVehs.Count; j < mfrVehCount; j++)
                {   // iterate through each mfr's vehicle
                    Vehicle              veh = mfrVehs[j];
                    Vehicle.CDescription vd  = veh.Description;
                    Vehicle.CModelData   vmd = veh.ModelData;

                    // get vehicle reg-class
                    RegulatoryClass regClass = veh.RegClass;

                    // scan each technology and update industry- and mfr-level tech volumes
                    for (int k = 0; k < TI.TechnologyCount; k++)
                    {
                        if (vmd.TechUsed[k])
                        {   // this technology is used by the vehicle -- adjust industry- and mfr-level sales volumes
                            imd.TechUsedSales[k][regClass] += vd.Sales[yrIndex];
                            mmd.TechUsedSales[k][regClass] += vd.Sales[yrIndex];
                            if (vmd.TechApplied[k])
                            {   // this technology was applied by the model -- adjust mfr-level "tech-applied" sales volume
                                mmd.TechAppliedSales[k][regClass] += vd.Sales[yrIndex];
                            }
                        }
                    } // next k (technology)
                } // next j (vehicle)
            } // next i (mfr)

            // calculate industry average CAFE standard (not required if not using "min stnd pct" option)
            RCDouble indStandard = Standards.GetIndustryStandard(data, scen, year);

            // re-calculate mfr vehicle costs and update phase-in
            for (int i = 0; i < this.MfrCount; i++)
            {   // iterate through each manufacturer
                Manufacturer            mfr     = data.Manufacturers[i];
                Manufacturer.CModelData mmd     = mfr .ModelData;
                List<Vehicle>           mfrVehs = mfr .Vehicles;

                for (int j = 0, mfrVehCount = mfrVehs.Count; j < mfrVehCount; j++)
                {   // iterate through each mfr's vehicle
                    Vehicle              veh              = mfrVehs[j];
                    Vehicle.CDescription vd               = veh.Description;
                    Vehicle.CModelData   vmd              = veh.ModelData;
                    double               vehCost          = 0;
                    double               vehMaintCost     = 0;
                    double               vehRepairCost    = 0;
                    double               vehConsumerValue = 0;
                    RegulatoryClass      regClass         = veh.RegClass;
                    VehicleClass         vehClass         = veh.VehicleClass;

                    // scan each technology and update industry- and mfr-level tech volumes and vehicle costs
                    for (int k = 0; k < TI.TechnologyCount; k++)
                    {
                        TechnologyInfo tech = technologies[k];
                        //
                        if (vmd.TechUsed[k])
                        {
                            if (TI.IsFCTimeBased(k) && !vmd.TechSuperseded[k])
                            {   // add in current cost for any time based technology that hasn't been superseded
                                vehCost += vmd.TechCostTimeBased[k];
                            }
                            else if (vmd.TechApplied[k])
                            {   // this technology was applied by the model -- update vehicle costs
                                vehCost       += TechnologyApplicability.GetCost           (tech, veh, year, this.Settings.Technologies, this.MinYear, this.MaxYear);
                                vehMaintCost  += TechnologyApplicability.GetMaintenanceCost(tech, veh, year);
                                vehRepairCost += TechnologyApplicability.GetRepairCost     (tech, veh, year);
                                // accumulate loss of value as well
                                vehConsumerValue += tech.GetAttributes(veh).ConsumerValuation;
                            }
                        }
                    } // next k (technology)

                    // update vehicle cost to include adjustment for improvements in air conditioning
                    if (scen.ScenInfo[regClass].IncludeAC[year.Index])
                    {
                        vehCost += scen.ScenInfo[regClass].ACCost[year.Index];
                    }

                    // update vehicle (and mfr) costs and loss of value based on the current year sales and present state of
                    // technologies applied to a vehicle by the model
                    double vehSales = vd.Sales[yrIndex];
                    //
                    vmd.TechCost                     =             vehCost          ;
                    vmd.ConsumerValuation            =             vehConsumerValue ;
                    vmd.MaintenanceCost              =             vehMaintCost     ;
                    vmd.RepairCost                   =             vehRepairCost    ;
                    mmd.TechCost         [regClass] += (vehSales * vehCost         );
                    mmd.ConsumerValuation[regClass] += (vehSales * vehConsumerValue);
                    mmd.MaintenanceCost  [regClass] += (vehSales * vehMaintCost    );
                    mmd.RepairCost       [regClass] += (vehSales * vehRepairCost   );
                } // next j (vehicle)

                // update vehicle-specific fc-adj-key
                for (int j = 0, mfrVehCount = mfrVehs.Count; j < mfrVehCount; j++)
                {
                    TechnologyApplicability.InitializeFCAdjKey(mfrVehs[j]);
                } // next j (vehicle)

                // update some mfr-specific values
                mmd.AverageStandard = indStandard;                      // update to use ind-overall standard
                mmd.Sales           = Standards.GetSales(mfr, year);    // update sales

                if (!this.Settings.OperatingModes.IgnorePhaseIn)
                {   // update mfr's tech-exhausted states based on the updated "tech-applied" sales volumes
                    for (int j = 0; j < TI.TechnologyCount; j++)
                    {
                        TechnologyApplicability.CheckPhaseIn(this.Settings, mmd, technologies[j], year, this.MinYearIndex);
                    } // next j (technology)
                }
            } // next i (mfr)
        }
        /// <summary>
        /// Post-processes the model year after the analysis for that year has completed.  If the model year did not complete,
        /// post-processing will not occur.
        /// </summary>
        /// <param name="scen">The current scenario being analyzed.</param>
        /// <param name="year">The current model year being analyzed.</param>
        protected virtual void PostProcessModelYear(Scenario scen, ModelYear year)
        {
            // if user requested to terminate modeling, return without post-processing
            if (this.AbortRequested) { return; }

            // initialize variables ...
            int      techCount = TI.TechnologyCount;
            Industry data      = this.GetData(scen, year);
            Industry bData     = (scen.Index == 0) ? data : this.GetData(this.Settings.Scenarios[0], year);

            // calculate tax credit for all vehicle
            Standards.CalcVehicleTaxCredit(data, scen, year);

            // allocate costs
            MarketSimulation.AllocateCosts(data, this.Settings, scen, year);

            // calculate taxes and fees, financing, and insurance costs
            EconomicValues economicValues = this.Settings.Parameters.EconomicValues;
            for (int i = 0, mfrCount = this.MfrCount; i < mfrCount; i++)
            {
                Manufacturer            mfr  = data.Manufacturers[i];
                Manufacturer.CModelData mmd  = mfr .ModelData;
                List<Vehicle>           vehs = mfr .Vehicles;
                //
                // clear out previous mfr variables
                mmd.DiscCost           .Clear();
                mmd.ConsumerValuation  .Clear();
                mmd.RelativeLossOfValue.Clear();
                mmd.TaxesAndFees       .Clear();
                mmd.FinancingCost      .Clear();
                mmd.InsuranceCost      .Clear();
                //
                for (int j = 0, vehCount = vehs.Count; j < vehCount; j++)
                {
                    Vehicle              veh      = vehs[j];
                    Vehicle.CModelData   vmd      = veh.ModelData;
                    Vehicle.CDescription vd       = veh.Description;
                    double               vehMsrp  = vd .Msrp + vmd.RegCost;
                    double               vehSales = vd .Sales[year.Index];
                    RegulatoryClass      vehRC    = veh.RegClass;
                    VehicleClass         vehClass = veh.VehicleClass;
                    OperatingCosts       opCosts  = economicValues.OperatingCosts[vehClass];
                    double               drExp    = Math.Max(0, year.Year - economicValues.DRBaseYear[vehClass]);
                    double               discRate = (economicValues.DRBaseYear[vehClass] <= 0) ? 1 : Math.Pow(1 + economicValues.DiscountRate[vehClass], -drExp);
                    //
                    vmd.DiscCost          = vmd.TechCost          * discRate;
                    vmd.ConsumerValuation = vmd.ConsumerValuation * discRate;
                    if (vehs[j].HEVType == HEVType.PureElectric)
                    {
                        vmd.RelativeLossOfValue = vehMsrp * opCosts.RelativeValueLoss;
                    }
                    vmd.TaxesAndFees  = vehMsrp * opCosts.TaxesAndFees;
                    vmd.FinancingCost = vehMsrp * opCosts.Financing;
                    vmd.InsuranceCost = vehMsrp * opCosts.Insurance;
                    //
                    mmd.DiscCost           [vehRC] += (vehSales * vmd.DiscCost           );
                    mmd.ConsumerValuation  [vehRC] += (vehSales * vmd.ConsumerValuation  );
                    mmd.RelativeLossOfValue[vehRC] += (vehSales * vmd.RelativeLossOfValue);
                    mmd.TaxesAndFees       [vehRC] += (vehSales * vmd.TaxesAndFees       );
                    mmd.FinancingCost      [vehRC] += (vehSales * vmd.FinancingCost      );
                    mmd.InsuranceCost      [vehRC] += (vehSales * vmd.InsuranceCost      );
                }

                // calculate subtotals
                mmd.TotalConsumerCosts = mmd.DiscCost + mmd.Fines + mmd.TaxesAndFees + mmd.FinancingCost + mmd.InsuranceCost +
                                         mmd.MaintenanceCost + mmd.RepairCost + mmd.ConsumerValuation + mmd.RelativeLossOfValue;
                mmd.TotalSocialCosts   = mmd.DiscCost +
                                         mmd.MaintenanceCost + mmd.RepairCost + mmd.ConsumerValuation + mmd.RelativeLossOfValue;
            }

            // run the Effects Sub-Model
            EffectsData effectsData;
            this._effectsModel.CalculateEffects(scen, year.Year, bData.Vehicles, data.Vehicles, out effectsData);
            this.SetEffectsData(year.Year, effectsData);

            if (this.Settings.OperatingModes.MultiYearModeling)
            {   // if in multi-year mode, need to re-run the model year post-processing for all previous years
                // by recursively passing the previous year to the "PostProcessModelYear"
                if (year.Year > this.MinYear)
                {   // there are still "redesign" years available -- perform post processing
                    this.PostProcessModelYear(scen, this.ModelYears[year.Year - this.MinYear - 1]);
                }
            }

            // raise model-year-completed event only for the current year (instead of all "multi-years")
            if (year.Year == this._year.Year)
            {
                this.OnModelYearCompleted();
            }
        }

        #endregion

        #region /* Manufacturer processing */

        /// <summary>
        /// Begins the model year loop for the compliance modeling process.
        /// </summary>
        /// <param name="scen">The current scenario being analyzed.</param>
        /// <param name="year">The current model year being analyzed.</param>
        protected virtual void ManufacturerLoop(Scenario scen, ModelYear year)
        {
            // get model year data
            Industry data = this.GetData(scen, year);

            // begin the mfr loop
            for (int i = 0; i < this.MfrCount; i++)
            {   // get next mfr and start processing
                Manufacturer mfr = data.Manufacturers[i];
                this.RunManufacturer(scen, year, mfr);
                // if user requested to terminate modeling, skip remaining mfrs
                if (this.AbortRequested) { break; }
            } // next i (manufacturer)
        }
        /// <summary>
        /// Begins examining the specified manufacturer.
        /// </summary>
        /// <param name="scen">The current scenario being analyzed.</param>
        /// <param name="year">The current model year being analyzed.</param>
        /// <param name="mfr">The manufacturer to analyze.</param>
        protected virtual void RunManufacturer(Scenario scen, ModelYear year, Manufacturer mfr)
        {
            // pre-process the manufacturer
            this.PreProcessManufacturer(scen, year, mfr);

            // initialize variables ...
            var            mmd          = mfr.ModelData;
            double         mfrFines     = mmd.Fines.Total;
            OperatingModes opModes      = this.Settings.OperatingModes;
            bool           noFines      = opModes.NoFines || !mfr.Description.PreferFines;
            bool           allowCT      = opModes.AllowCreditTrading;
            bool           allowOC      = opModes.Overcomply; // && (mfr.Description.PaybackPeriod > 0);
            bool           doOvercomply = allowOC && (mfrFines <= 0);
            bool           expCrUsed    = false; // whether expiring credits have been used
            int            maxExpCrYr   = this.Settings.Parameters.CreditTradingValues.MaxExpiringYears;
            bool           ignoredReset = false;

            ComplianceFinding finding = null;

            // clear cf-exhausted state
            mmd.CFExhausted = false;

            // start the inner mfr loop
            while (mfrFines > 0 || doOvercomply)
            {   // execute the compliance loop as long as there are fines
                finding = this.ComplianceFindingLoop(scen, year, mfr, noFines, doOvercomply, allowCT);

                if (finding == null)
                {   // if finding is null, the model cannot generate any more compliance solutions
                    if (ignoredReset)
                    {   // the model tried reseting tech "ignored" states, but still cannot find any more valid
                        // CFs -- set cf-exhausted state and exit the loop
                        mmd.CFExhausted = true;
                        break;
                    }
                    else
                    {   // reset "ignored" states for all vehicles in case synergies of already applied
                        // technologies improves the efficiency of ignored technologies
                        List<Vehicle> mfrVehs = mfr.Vehicles;
                        for (int i = 0, vehCount = mfrVehs.Count; i < vehCount; i++)
                        {
                            Vehicle.CModelData vmd = mfrVehs[i].ModelData;
                            for (int j = 0; j < TI.TechnologyCount; j++) { vmd.TechIgnored[j] = false; }
                        }
                        ignoredReset = true; // indicates that the model tried reseting the ignored states already
                    }
                }
                else if (!finding.IsValid)
                {   // the finding is not valid -- efficiency has not been calculated for some reason (SHOULD NEVER HAPPEN)
                    throw new Exception("The compliance finding is not valid!\r\n\r\n" +
                        "This should never happen and typically indicates an internal error in the modeling system.");
                }
                else if (mfrFines > 0 || (doOvercomply && finding.IsEfficient))
                {   //----------------------------------------------------------------------------------------------------//
                    // apply finding if:
                    //  * mfr has not yet reached compliance
                    //  * mfr has reached compliance, but the model is overcomplying AND the finding is efficient
                    //
                    // if finding is not cost-effective and the model hasn't yet tried using credits that will expire
                    // the following year, attempt to carry-forward those credits first
                    //----------------------------------------------------------------------------------------------------//
                    if (!expCrUsed && !finding.IsEfficient)
                    {
                        CreditTrading.CT_CarryForward  (this.GetData(scen), scen, year, mfr.Index, this.Settings, true, year.Index + maxExpCrYr);
                        CreditTrading.CT_FleetTransfers(this.GetData(scen), scen, year, mfr.Index, this.Settings, true, year.Index + maxExpCrYr);
                        this.Recomply(scen, year, mfr);
                        expCrUsed = true;
                    }
                    else
                    {
                        finding.ApplyFinding();
                        this.Recomply(scen, year, mfr);
                        //
                        //if (finding.IsMultiYear)
                        //{   // if current finding applied technology in a preceding year (due to multiyear effect),
                        //    // clear out "expiring credits used" flag (this flag will be set once inefficient findings
                        //    // are encountered)
                        //    expCrUsed = false;
                        //}
                    }
                }
                else { break; }

                // update fines
                mfrFines = mmd.Fines.Total;

                if (mfrFines <= 0)
                {   // if mfrFines were reduced to 0, check if the model should start overcomplying
                    doOvercomply = allowOC;
                }

                if (this.AbortRequested) { return; }
            } // end while

            // if the manufacturer still paying fines, attemp to transfer credits
            if (mfrFines > 0)
            {
                CreditTrading.ExamineCredits(this.GetData(scen), scen, year, mfr.Index, this.Settings);
                this.Recomply(scen, year, mfr);
            }

            // post-process the manufacturer
            this.PostProcessManufacturer(scen, year, mfr);
        }
        /// <summary>
        /// Pre-processes the manufacturer for analysis.
        /// </summary>
        /// <param name="scen">The current scenario being analyzed.</param>
        /// <param name="year">The current model year being analyzed.</param>
        /// <param name="mfr">The current manufacturer being analyzed.</param>
        protected virtual void PreProcessManufacturer(Scenario scen, ModelYear year, Manufacturer mfr)
        {
            // initialize variables ...
            // update progress ...
            this._mfr = mfr;
            this.OnManufacturerStarted();

            // recomply manufacturer -- recalculate standard, sales, CAFE value, credits, and fines
            this.Recomply(scen, year, mfr);

            // before evaluating mfr for CAFE, force compliance with CA+S177 ZEV targets
            this.ComplyWithZEV(scen, year, mfr);
            this.Recomply(scen, year, mfr);

            // evaluate technologies with ForcedApplication flag set 
            //this.EvaluateForcedApplicationTechnologies(scen, year, mfr);
            //this.Recomply(scen, year, mfr);

            // check for inherited tecchnologies
            this.InheritTechnologies(scen, year, mfr);
            this.Recomply(scen, year, mfr);

            // evaluate technologies with FCTimeBased flag set
            this.EvaluateFCTimeBasedTechnologies(scen, year, mfr);
            this.Recomply(scen, year, mfr);

            // carry-forward all credits that will expire next year
            CreditTrading.CT_CarryForward  (this.GetData(scen), scen, year, mfr.Index, this.Settings, true, year.Index);
            CreditTrading.CT_FleetTransfers(this.GetData(scen), scen, year, mfr.Index, this.Settings, true, year.Index);
            this.Recomply(scen, year, mfr);
        }
        /// <summary>
        /// Post-processes the manufacturer after the analysis for that manufacturer has completed.  If user terminate modeling
        /// before manufacturer processing completed, post-processing will not occur.
        /// </summary>
        /// <param name="scen">The current scenario being analyzed.</param>
        /// <param name="year">The current model year being analyzed.</param>
        /// <param name="mfr">The current manufacturer being analyzed.</param>
        protected virtual void PostProcessManufacturer(Scenario scen, ModelYear year, Manufacturer mfr)
        {
            // if user requested to terminate modeling, return without post-processing
            if (this.AbortRequested) { return; }

            // initialize variables ...
            int yrIndex = year.Index;

            // recomply manufacturer -- recalculate standard, sales, CAFE value, credits, and fines
            this.Recomply(scen, year, mfr);

            // check for inherited tecchnologies
            this.InheritTechnologies(scen, year, mfr);
            this.Recomply(scen, year, mfr);

            if (this.Settings.OperatingModes.MultiYearModeling && this.Settings.OperatingModes.MultiYearStartYear <= year.Year)
            {   // check for redesign year technologies which are enabled but not used
                List<Vehicle> mfrVehs = mfr.Vehicles;
                //
                for (int i = 0, mfrVehCount = mfrVehs.Count; i < mfrVehCount; i++)
                {   // iterate through each mfr's vehicle
                    Vehicle              veh = mfrVehs[i];
                    Vehicle.CDescription vd  = veh.Description;
                    Vehicle.CModelData   vmd = veh.ModelData;

                    if (veh.IsAtRedesign(year))
                    {
                        vmd.TechLingerYear.Add(year);
                    } // end if (atRedesign)
                }

                // recomply for previous years also ...
                for (int i = this.MinYearIndex; i < yrIndex; i++)
                {
                    ModelYear prevYear = this.ModelYears[i - this.MinYearIndex];
                    Industry prevData = this.GetData(scen, prevYear);
                    Manufacturer lMfr = prevData.Manufacturers[mfr.Index];
                    this.Recomply(scen, prevYear, lMfr);
                }
            } // end if (multi-year)

            this.OnManufacturerCompleted();
        }
        /// <summary>
        /// Recomplies the manufacturer currently being analyzed by recalculating the Standard, CAFE, Credits, and Fines for the
        /// manufacturer, scenario, and model year being examined.
        /// </summary>
        /// <param name="scen">The current scenario being analyzed.</param>
        /// <param name="year">The current model year being analyzed.</param>
        /// <param name="mfr">The current manufacturer being analyzed.</param>
        protected virtual void Recomply(Scenario scen, ModelYear year, Manufacturer mfr)
        {
            // get the compliance manufacturer
            Manufacturer.CModelData mmd = mfr.ModelData;
            // recalculate standard, sales and CAFE value, credits, and fines
            Standards.CalcStandard(mfr, scen, year               );
            Standards.CalcCAFE    (mfr, scen, year, this.Settings);     // calc-cafe safes mfr-sales and CAFE
            Standards.CalcCredits (mfr, scen, year, this.Settings);
            Standards.CalcFines   (mfr, scen, year, this.Settings);
        }

        void ComplyWithZEV(Scenario scen, ModelYear year, Manufacturer mfr)
        {
            // upgrade any ZEV candidate vehicle which is at redesign and hasn't been upgraded yet
            for (int i = 0; i < mfr.VehicleCount; i++)
            {
                Vehicle veh = mfr.Vehicles[i];
                HEVType zevCandidate = veh.Description.ZEVCandidate;
                //
                if (zevCandidate != HEVType.None         && // * check if veh is a ZEV candidate
                    veh.HEVType  != zevCandidate         && // * check if veh was already upgraded to desired ZEV state
                    veh.HEVType  != HEVType.PureElectric && // * if desired ZEV state is PHEV, check if veh was upgraded
                    veh.HEVType  != HEVType.FuelCell     && //   beyond that to EV or FCV
                    veh.IsAtRedesign(year))                 // * check if veh is at redesign
                {   //
                    // when upgrading ZEV candidate vehicle, apply technologies in the following order:
                    //  - apply SHEVPS (if not already SHEVPS or SHEVP2)
                    //  - apply PHEV50 (if desired ZEV state is PHEV)
                    //  - apply BEV200 (if desired ZEV state is EV)
                    // before applying SHEVPS, also need to apply all missing prerequisite techs
                    //  - basic engine path is required for proper accounting (VVT, VVL, SGID, and DEAC)
                    //  - transmission path is NOT required since the model negates back to AT5
                    //  - electrification path will be backfilled when SHEVPS is applied
                    //
                    this.ConvertToZEV(scen, year, mfr, veh);
                }
            }
        }
        void ConvertToZEV(Scenario scen, ModelYear year, Manufacturer mfr, Vehicle veh)
        {
            ComplianceFinding      cf = new ComplianceFinding();
            TechnologyInfo[] techList = this.Settings.Technologies.TechnologyList;
            List<Vehicle>        vehs = new List<Vehicle> { veh }; // create list around desried veh for CF processing
            bool[]           techUsed = veh.ModelData.TechUsed;
            bool[]        techEnabled = veh.ModelData.TechEnabled;
            bool[]      techAvailable = veh.ModelData.TechAvailable;

            // determine desired tech -- PHEV or EV
            int zevTechIdx = (veh.Description.ZEVCandidate == HEVType.PlugInHybrid) ? TI.PHEV50 : TI.BEV200;
            // if ZEV tech is not compatible with veh, return without modifying the vehicle
            // (for example: veh has manual transmission or non-gas fuel -- either veh wasn't properly classified
            //  as ZEV candidate in the input file -or- model applied an incompatible tech before it had a chance
            //  to convert the vehicle to ZEV)
            if (!techAvailable[zevTechIdx]) { return; }

            // apply any missing basic engine techs
            // generate tech group
            List<TechnologyInfo> techGroup = new List<TechnologyInfo>();
            techGroup.Add(techList[TI.VVT ]);
            techGroup.Add(techList[TI.VVL ]);
            techGroup.Add(techList[TI.SGDI]);
            techGroup.Add(techList[TI.DEAC]);
            // initialize CF
            cf.Initialize(scen, year, this.MinYear, this.MaxYear, this.GetData(scen), mfr.Index, this.Settings, this.LogWriter,
                techGroup);
            // scan vehicle used and enabled states to apply missing basic engine techs
            for (int i = 0; i < techGroup.Count; i++)
            {
                int techIndex = techGroup[i].Index;
                if (!techUsed[techIndex] && techEnabled[techIndex])
                {
                    cf.ExamineForcedApplication(i, vehs, 0, true);
                    cf.ApplyFinding();
                }
            }

            // once the basic engine path is complete, apply PHEV50 or BEV200 to convert vehicle to ZEV
            // (any missing tech on the electrification path will be backfilled)
            if (!techEnabled[zevTechIdx]) { return; }
            //
            // initialize tech group
            techGroup.Clear();
            for (int i = 0; i < techList.Length; i++)
            {
                if (TI.IsElectrificationPath   (techList[i].Index) ||
                    TI.IsHybridElectricPath    (techList[i].Index) ||
                    TI.IsAdvancedHybridElecPath(techList[i].Index)) { techGroup.Add(techList[i]); }
            }
            // find desired tech idx within tech-group
            int desiredIdx = 0;
            for (int i = techGroup.Count - 1; i >= 0; i--)
            {
                if (techGroup[i].Index == zevTechIdx) { desiredIdx = i; break; }
            }
            // initialize CF and examing finding (used force application)
            cf.Clear();
            cf.Initialize(scen, year, this.MinYear, this.MaxYear, this.GetData(scen), mfr.Index, this.Settings, this.LogWriter, techGroup);
            cf.ExamineForcedApplication(desiredIdx, vehs, 0, true);
            cf.ApplyFinding();
        }

        void InheritTechnologies(Scenario scen, ModelYear year, Manufacturer mfr)
        {
            // for each engine, transmission, and platform, check each vehicle at redesign and inherit missing technologies --
            // multiple passes are required, since inheriting techs from one path (e.g., 8-speed transmission) may unlock
            // techs in another path (e.g., turbo2 engine and beyond)
            for (int i = 0; i < 3; i++)
            {
                foreach (TechnologyInfo tech in this.Settings.Technologies.TechnologyList)
                {
                    if (TI.IsEngineLevel(tech.Index))
                    {
                        foreach (Engine value in mfr.Engines)
                        {
                            this.InheritTechnology(scen, year, mfr, value, tech);
                        }
                    }
                    else if (TI.IsTransmissionLevel(tech.Index))
                    {
                        foreach (Transmission value in mfr.Transmissions)
                        {
                            this.InheritTechnology(scen, year, mfr, value, tech);
                        }
                    }
                    else if (TI.IsPlatformLevel(tech.Index))
                    {
                        foreach (Platform value in mfr.Platforms)
                        {
                            this.InheritTechnology(scen, year, mfr, value, tech);
                        }
                    }
                } // next tech
            }
        }
        void InheritTechnology(Scenario scen, ModelYear year, Manufacturer mfr, Component component, TechnologyInfo tech)
        {
            //----------------------------------------------------------------------------------//
            // This function propogates the specified technology from the specified component   //
            // down to all vehicles.                                                            //
            //----------------------------------------------------------------------------------//
            if (!component.ModelData.TechUsed[tech.Index]) { return; }

            List<TechnologyInfo> techGroup = new List<TechnologyInfo> { tech };
            ComplianceFinding cf = new ComplianceFinding();
            cf.Initialize(scen, year, this.MinYear, this.MaxYear, this.GetData(scen), mfr.Index, this.Settings, this.LogWriter,
                techGroup);

            for (int i = 0; i < component.Vehicles.Count; i++)
            {
                Vehicle veh = component.Vehicles[i];
                if (!veh.ModelData.TechEnabled[tech.Index]) { continue; } // || veh.Description.Sales[year.Index] == 0) { continue; }

                cf.ExamineTechInheriting(0, component.Vehicles, i);
                cf.ApplyFinding();
                this.Recomply(scen, year, mfr);
            } // next veh in component
        }

        void EvaluateFCTimeBasedTechnologies(Scenario scen, ModelYear year, Manufacturer mfr)
        {
            foreach (Vehicle veh in mfr.Vehicles)
            {
                bool[] techUsed       = veh.ModelData.TechUsed;
                bool[] techSuperseded = veh.ModelData.TechSuperseded;
                bool   atRedesign     = veh.IsAtRedesign(year);
                bool   atRefresh      = veh.IsAtRefresh (year);
                //
                foreach (TechnologyInfo tech in this.Settings.Technologies.TechnologyList)
                {
                    int tIndex = tech.Index;
                    if (  TI.IsFCTimeBased      (tIndex) &&
                          techUsed              [tIndex] &&
                         !techSuperseded        [tIndex] &&
                        (!TI.IsEngineLevel      (tIndex) || veh.Engine       != null) &&
                        (!TI.IsTransmissionLevel(tIndex) || veh.Transmission != null) &&
                        (!TI.IsPlatformLevel    (tIndex) || veh.Platform     != null))
                    {
                        bool isTiedToRedesign = TI.IsTiedToRedesign(tIndex);
                        bool isTiedToRefresh  = TI.IsTiedToRefresh (tIndex);
                        //
                        if (atRedesign || (!isTiedToRedesign && (atRefresh || !isTiedToRefresh)))
                        {   // technology is eligible for time-based FC adjustment on the vehicle
                            this.EvaluateFCTimeBasedTechnology(scen, year, veh, tech);
                        }
                    }
                } // next tech
            } // next veh
        }
        void EvaluateFCTimeBasedTechnology(Scenario scen, ModelYear year, Vehicle veh, TechnologyInfo tech)
        {
            double lastFC, currFC, lastCost, currCost;
            TechnologyApplicability.GetEstimatedFCTimeBased  (year, veh, tech, out lastFC  , out currFC  );
            TechnologyApplicability.GetEstimatedCostTimeBased(year, veh, tech, out lastCost, out currCost);
            //
            if (lastFC != currFC)
            {
                FuelValue oldFE = veh.Description.FuelEconomy;
                // adjust vehicle's fuel economy to include this technology's time-based FC gain
                foreach (FuelType fuel in FTValue<object>.Classes)
                {
                    if (veh.Description.FuelEconomy[fuel] != 0)
                    {
                        veh.Description.FuelEconomy[fuel] *= (1 - lastFC) / (1 - currFC);
                    }
                }
                // adjust tech cost
                veh.ModelData.TechCost += (currCost - lastCost);
                veh.ModelData.TechCostTimeBased[tech.Index] = currCost;
                // log time-based effect
                this.LogFCTimeBasedTechnology(scen, year, veh, tech, oldFE, lastFC, currFC, lastCost, currCost);
            }
        }
        void LogFCTimeBasedTechnology(
            Scenario scen, ModelYear year, Vehicle veh, TechnologyInfo tech,
            FuelValue oldFE, double lastFC, double currFC, double lastCost, double currCost)
        {
            if (!this.LogWriter.IsEmpty)
            {
                ComplianceWriter log    = (this.LogWriter.CFs    == null) ? null : this.LogWriter.CFs   [this._scen.Index];
                ComplianceWriter extLog = (this.LogWriter.ExtCFs == null) ? null : this.LogWriter.ExtCFs[this._scen.Index];
                if (log != null || extLog != null)
                {
                    this.LogFCTimeBasedTechnology(log, extLog, scen, year, veh, tech, oldFE, lastFC, currFC, lastCost, currCost);
                }
            }
        }
        void LogFCTimeBasedTechnology(
            ComplianceWriter log, ComplianceWriter extLog, Scenario scen, ModelYear year, Vehicle veh, TechnologyInfo tech,
            FuelValue oldFE, double lastFC, double currFC, double lastCost, double currCost)
        {
            var vd = veh.Description;
            int ln1 = (log    == null) ? 0 : log   .LineCount + 1;
            int ln2 = (extLog == null) ? 0 : extLog.LineCount + 1;
            StringBuilder line = new StringBuilder();
            line.Append("\tT\t");
            line.Append(scen.Index);
            line.Append("\t");
            line.Append(vd.Manufacturer);
            line.Append("\tT\t\t\t\t\t\t");
            line.Append(vd.Sales[year.Index]);
            line.Append("\t");
            line.Append(veh.RegClass);
            line.Append("\t\t\t\t\t\t");
            line.Append(vd.Code); line.Append("\t");
            line.Append(vd.EngineCode); line.Append("\t");
            line.Append(vd.TransmissionCode); line.Append("\t");
            line.Append(veh.TechClassString); line.Append("\t");
            line.Append(tech.Name); line.Append("\t");
            line.Append(year.Year); line.Append("\t");
            line.Append(   oldFE      .ToString("0.####", true)); line.Append("\t");
            line.Append(vd.FuelShare  .ToString("0.####", true)); line.Append("\t");
            line.Append(vd.FuelEconomy.ToString("0.####", true)); line.Append("\t");
            line.Append(vd.FuelShare  .ToString("0.####", true)); line.Append("\t");
            line.Append(vd.Sales[year.Index]); line.Append("\t");
            line.Append(currCost - lastCost);
            line.Append("\t\t");
            line.Append((1 - 1 / (1 - lastFC) * (1 - currFC)).ToString("0.########"));
            line.Append("\t\t\t\t");
            line.Append((vd.FuelEconomy - oldFE).ToString("0.####", true));
            line.Append("\t\t\t\t\t");
            //
            if (log    != null) { log   .Write(ln1.ToString()); log   .WriteLine(line.ToString()); }
            if (extLog != null) { extLog.Write(ln2.ToString()); extLog.WriteLine(line.ToString()); }
        }

        void EvaluateForcedApplicationTechnologies(Scenario scen, ModelYear year, Manufacturer mfr)
        {
            Industry[] modelYearData = this.GetData(scen);
            //
            foreach (TechnologyInfo tech in this.Settings.Technologies.TechnologyList)
            {
                if (!TI.IsForcedApplication(tech.Index)) { continue; }

                List<TechnologyInfo> techGroup = new List<TechnologyInfo> { tech };
                ComplianceFinding cf = new ComplianceFinding();
                cf.Initialize(scen, year, this.MinYear, this.MaxYear, modelYearData, mfr.Index, this.Settings, this.LogWriter,
                    techGroup);

                for (int i = 0; i < mfr.Vehicles.Count; i++)
                {
                    Vehicle veh = mfr.Vehicles[i];
                    ModelYear lingerYear;
                    bool lingering;
                    bool isTechValid = TA.IsTechValid(modelYearData, year, veh, tech.Index, true, false, false, out lingering, out lingerYear);
                    if (isTechValid && !lingering)
                    {   // if tech is valid in the current year (and NOT lingering), apply it
                        cf.ExamineForcedApplication(0, mfr.Vehicles, i, false);
                        if (cf.IsValid)
                        {
                            cf.ApplyFinding();
                            this.Recomply(scen, year, mfr);
                        }
                    }
                } // next veh
            } // next tech
        }

        #endregion

        #region /* Compliance finding processing */

        /// <summary>
        /// Begins the compliance finding loop for the compliance modeling process.
        /// </summary>
        /// <param name="scen">The current scenario being analyzed.</param>
        /// <param name="year">The current model year being analyzed.</param>
        /// <param name="mfr">The current manufacturer being analyzed.</param>
        /// <param name="noFines">Specifies whether the manufacturer is willing to pay fines.</param>
        /// <param name="overcomply">Specifies whether the model has entered over-compliance mode.</param>
        /// <param name="creditTrade">Specifies whether credit trading (transfers and carry fwd/back) is allowed.</param>
        /// <returns>The most efficient compliance finding for a manufacturer for the scenario and model year being analyzed, or
        ///   null, if no such finding exists.  If a finding is not efficient and the manufacutrer is not willing to pay fines,
        ///   the return value will be null.</returns>
        protected virtual ComplianceFinding ComplianceFindingLoop(Scenario scen, ModelYear year, Manufacturer mfr, bool noFines,
            bool overcomply, bool creditTrade)
        {
            // initialize variables ...
            int snIndex = scen.Index;
            int yrIndex = year.Index;
            var mmd     = mfr.ModelData;

            ComplianceFinding finding = null;
            bool              loop    = true;

            // examine compliance findings
            while (loop)
            {   // get the compliance finding
                finding = this._lcf.GetNextFinding(year, this.GetData(scen), mfr.Index, overcomply, creditTrade);

                // check the compliance finding -- this point will be reached only if a solution was found, or if no valid
                // solution was found after exhausting all technology/vehicle combinations
                //
                // * if no valid solution was found, terminate the loop
                // * if a solution is valid and the finding is efficient, or the model is in no fines mode, terminate the loop
                // * if a solution is valid and the finding is not efficient, while the mfr is not in no fines mode, reset the
                //   finding to null and terminate the loop
                if (finding == null || !finding.IsValid || noFines || finding.IsEfficient)
                {   // terminate the loop per above criteria
                    loop = false;
                }
                else
                {   // finding is not efficient -- clear the finding and terminate the loop
                    finding = null;
                    loop    = false;
                }
            } // end while(loop)

            return finding;
        }

        #endregion

        #endregion

        #endregion


        #region /*** Properties ***/

        #region /* ComplianceBase Members */

        /// <summary>Gets the current modeling progress, or null if the compliance model is not running.</summary>
        public override IModelingProgress Progress
        {
            get
            {
                bool   xlReporting          = (this.XlReportGenerator  != null && this.XlReportGenerator .Reporting);
                bool   csvReporting         = (this.CsvReportGenerator != null && this.CsvReportGenerator.Reporting);
                bool   reporting            =  xlReporting || csvReporting;
                string xlReportingProgress  = (xlReporting) ? this.XlReportGenerator.ReportingProgress : null;
                string csvReportingProgress = (csvReporting) ? this.CsvReportGenerator.ReportingProgress : null;
                string reportingProgress    =  xlReportingProgress + ((xlReporting) ? "\n" : "") + csvReportingProgress;
                bool   effectsModelRunning  = (this._effectsYear != -1 && (this._effectsYear < this.MinYear || this._effectsYear > this.MaxYear));
                string effectsModelProgress = (effectsModelRunning) ? "Running effects analysis for additional year: " + this._effectsYear : "";
                string additionalInfo       = effectsModelProgress + ((effectsModelRunning) ? "\n" : "") + reportingProgress;
                //
                return (this.Running || reporting) ? new ModelingProgress(this._scen, this._year, this._mfr, additionalInfo) : null;
            }
        }

        #endregion

        #endregion


        #region /*** Variables ***/

        /// <summary>Provides compliance finders to use for generating compliance findings for scenarios (indexed by scenario).</summary>
        [NonSerialized] internal LinearComplianceFinder _lcf;

        /// <summary>Provides the effects sub-model for the emissions calculations.</summary>
        [NonSerialized] EffectsModel _effectsModel;

        #endregion

    }
}
