#region << Using Directives >>
using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using Volpe.Cafe.Data;
using Volpe.Cafe.IO;
using Volpe.Cafe.Settings;
using Volpe.Cafe.Utils;
using TI = Volpe.Cafe.TechnologyIndexes;
using TA = Volpe.Cafe.Model.TechnologyApplicability;
using Volpe.Cafe.Generic;
#endregion

namespace Volpe.Cafe.Model
{
    /// <summary>
    /// Provides a compliance solution for applying one or more technologies to a vehicle or a group of vehicles.
    /// <br></br><br></br>
    /// This class contains an <see cref="ExamineFinding"/> method, which is used to examine a finding based on a single
    /// vehicles/technologies combination.
    /// </summary>
    [Serializable]
    public class ComplianceFinding
    {

        #region /*** Constructors ***/

        /// <summary>
        /// Initializes a new instance of the <see cref="ComplianceFinding"/> class.
        /// </summary>
        public ComplianceFinding()
        {
            // init internal runtime arrays
            this._capacity                 = 8;
            this._vehs                     = new Vehicle   [this._capacity];
            this._techKeys                 = new ulong     [this._capacity];
            this._techLingerKeys           = new ulong     [this._capacity];
            this._lingerYear               = new int       [this._capacity];
            this._lingering                = new bool      [this._capacity];
            this._techImpacts              = new TechImpact[this._capacity];
            this._yearOfInitialApplication = new int       [this._capacity][];
            //
            this._committedERPCosts        = new bool      [TI.TechnologyCount];
            //
            // clear other vars
            this.Reset();
        }

        #endregion

        #region /*** Methods ***/

        #region /* Overriden from Object */

        /// <summary>
        /// Returns the string representation of this <see cref="ComplianceFinding"/> instance.
        /// </summary>
        /// <returns>The string representation of the <see cref="ComplianceFinding"/> instance.</returns>
        public override string ToString()
        {
            bool valid = this.IsValid;
            if (valid)
            {
                Manufacturer.CModelData mmd = this._vehs[0].Manufacturer.ModelData;
                string s = 
                    "ComplianceFinding # " + this._index +
                    ": {\nIsValid: true\nIsEfficient: "  + this.IsEfficient +
                    "\nEfficiency: "    + this._efficiency         +
                    "\nTechCost: "      + this._techCost           +
                    "\nRPECost: "       + this._totalDeltaRPE      +
                    "\nDeltaFines: "    + this._totalDeltaFines    +
                    "\nDeltaCredits: "  + this._totalDeltaCredits  +
                    "\nFuelSavings: "   + this._totalFuelSavings   +
                    "\nAffectedSales: " + this._totalAffectedSales +
                    "\nCAFE (old, new): {";
                for (int i = 0; i < RCValue<object>.Classes.Length; i++)
                {
                    s += "\n\t  " + RCValue<object>.Names[i] + ": " + this._curCafe.Items[i] + ", " + this._newCafe.Items[i];
                }
                s += "\n\t}\nFines (old, new): {";
                for (int i = 0; i < RCValue<object>.Classes.Length; i++)
                {
                    s += "\n\t  " + RCValue<object>.Names[i] + ": " + this._curFines.Items[i] + ", " + this._newFines.Items[i];
                }
                s += "\n\t}\n}";
                return s;
            }
            else { return "ComplianceFinding # " + this._index + ": {\nValid: false}"; }
        }

        #endregion

        #region /* CF Processing (init, reset/clear, building) */

        /// <summary>
        /// Initializes this <see cref="ComplianceFinding"/> instance for processing the specified scenario, model year, and
        /// manufacturer, using the provided modeling settings, log writer, and technologies of the given type.
        /// </summary>
        /// <param name="scen">The current scenario being analyzed.</param>
        /// <param name="year">The current model year being analyzed.</param>
        /// <param name="startYear">The value of the model year when modeling began.</param>
        /// <param name="endYear">The value of the model year when modeling ends.</param>
        /// <param name="modelYearData">An array of modeling data values, indexed by year, being analyzed with the current
        ///   scenario.  This array is populated with each subsequent model year.</param>
        /// <param name="mfrIndex">The index of the current manufacturer being analyzed.</param>
        /// <param name="settings">The modeling settings used for modeling.</param>
        /// <param name="logWriter">The <see cref="LogWriter"/> to use for saving the modeling logs.</param>
        /// <param name="techGroup">The group of technologies, all of the same type and corresponding to techType, to analyze.</param>
        public void Initialize(Scenario scen, ModelYear year, int startYear, int endYear, Industry[] modelYearData, int mfrIndex,
            ModelingSettings settings, LogWriter logWriter, List<TechnologyInfo> techGroup)
        {
            this._scen           = scen;
            this._year           = year;
            this._yrIndex        = year.Index;
            this._yrYear         = year.Year;
            this._startYear      = startYear;
            this._endYear        = endYear;
            this._modelYearData  = modelYearData;
            this._mfrIndex       = mfrIndex;
            this._settings       = settings;
            this._parameters     = settings.Parameters;
            this._overcomply     = false;
            this._creditTrade    = false;
            this._multiYear      = settings.OperatingModes.MultiYearModeling;
            this._ignorePhaseIn  = settings.OperatingModes.IgnorePhaseIn;
            this._backfill       = settings.OperatingModes.Backfill;
            this._logWriter      = logWriter;
            this._techGroup      = techGroup;
            this._techCount      = techGroup.Count;
            this._doInherit      = false;
            this._forcedTech     = false;
            // reset vars
            this.Reset();
        }

        /// <summary>
        /// Resets the runtime variables, counters, and effectivenes claculations associated with this
        /// <see cref="ComplianceFinding"/> instance.  The variables assigned by calling the <see cref="Initialize"/> method will
        /// not be reset.
        /// </summary>
        public void Reset()
        {
            this._index              = -1;
            this._vehCount           = 0;
            this._techIndex          = 0;
            this._minLingerYear      = 0;
            this._affectedRC         = RCBoolean.False;
            this._hasVehs            = false;
            this._isApplied          = false;
            this._efficiency         = double.NaN;
            this._techCost           = double.NaN;
            this._newCafe            = RCDouble.Zero;
            this._newFines           = RCDouble.Zero;
            this._curCafe            = RCDouble.Zero;
            this._curFines           = RCDouble.Zero;
            this._totalDeltaRPE      = 0;
            this._totalDeltaFines    = 0;
            this._totalDeltaCredits  = 0;
            this._totalFuelSavings   = 0;
            this._totalAffectedSales = 0;
        }

        /// <summary>
        /// Clears all elements in this <see cref="ComplianceFinding"/> instance.
        /// </summary>
        public void Clear()
        {
            this._scen           = null;
            this._year           = null;
            this._yrIndex        = 0;
            this._yrYear         = 0;
            this._startYear      = 0;
            this._endYear        = 0;
            this._modelYearData  = null;
            this._mfrIndex       = 0;
            this._settings       = null;
            this._parameters     = null;
            this._overcomply     = false;
            this._creditTrade    = false;
            this._multiYear      = false;
            this._ignorePhaseIn  = false;
            this._backfill       = false;
            this._logWriter      = LogWriter.Empty;
            this._techGroup      = null;
            this._techCount      = 0;
            this._doInherit      = false;
            this._forcedTech     = false;
            // clear other vars
            this.Reset();
        }

        /// <summary>
        /// Ensures that all arrays used internally by this <see cref="ComplianceFinding"/> instance at a minimum accomodate the
        /// specified capacity.
        /// </summary>
        /// <param name="capacity">The minimum capacity of the internal arrays.</param>
        public void EnsureCapacity(int capacity)
        {
            if (this._capacity < capacity)
            {
                this._capacity = ((capacity / 8) + 1) * 8;
                this.EnsureCapacity();
            }
        }
        void EnsureCapacity()
        {
            if (this._techKeys.Length < this._capacity)
            {
                this._vehs                     = new Vehicle   [this._capacity];
                this._techKeys                 = new ulong     [this._capacity];
                this._techLingerKeys           = new ulong     [this._capacity];
                this._lingerYear               = new int       [this._capacity];
                this._lingering                = new bool      [this._capacity];
                this._techImpacts              = new TechImpact[this._capacity];
                this._yearOfInitialApplication = new int       [this._capacity][];
            }
            //
            int impactLen = this._endYear - ModelYear.MinYear + 1;
            for (int i = 0; i < this._capacity; i++)
            {
                if (this._techImpacts[i] == null)
                {
                    this._techImpacts[i] = new TechImpact(impactLen);
                }
                if (this._yearOfInitialApplication[i] == null)
                {
                    this._yearOfInitialApplication[i] = new int[TI.TechnologyCount];
                }
            }
        }

        /// <summary>
        /// Copies all elements from the specified <see cref="ComplianceFinding"/> to this instance.
        /// </summary>
        /// <param name="cf">The <see cref="ComplianceFinding"/> whose elements to copy.</param>
        public void CopyFrom(ComplianceFinding cf)
        {
            // input variables
            this._scen               = cf._scen;
            this._year               = cf._year;
            this._yrIndex            = cf._yrIndex;
            this._yrYear             = cf._yrYear;
            this._startYear          = cf._startYear;
            this._endYear            = cf._endYear;
            this._modelYearData      = cf._modelYearData;   // by-ref copy of input data is OK
            this._mfrIndex           = cf._mfrIndex;
            this._settings           = cf._settings;
            this._parameters         = cf._parameters;
            this._overcomply         = cf._overcomply;
            this._creditTrade        = cf._creditTrade;
            this._multiYear          = cf._multiYear;
            this._ignorePhaseIn      = cf._ignorePhaseIn;
            this._backfill           = cf._backfill;
            this._logWriter          = cf._logWriter;
            this._techGroup          = cf._techGroup;
            this._techCount          = cf._techCount;
            this._doInherit          = cf._doInherit;
            this._forcedTech         = cf._forcedTech;
            // runtime variables
            this._index              = cf._index;
            this._vehCount           = cf._vehCount;
            this._capacity           = cf._capacity;
            this._techIndex          = cf._techIndex;
            this._minLingerYear      = cf._minLingerYear;
            this._affectedRC         = cf._affectedRC.Clone();
            this._hasVehs            = cf._hasVehs;
            this._isApplied          = cf._isApplied;
            this._overcomply         = cf._overcomply;
            this._creditTrade        = cf._creditTrade;
            this._efficiency         = cf._efficiency;
            this._techCost           = cf._techCost;
            this._newCafe            = cf._newCafe.Clone();
            this._newFines           = cf._newFines.Clone();
            this._curCafe            = cf._curCafe.Clone();
            this._curFines           = cf._curFines.Clone();
            this._totalDeltaRPE      = cf._totalDeltaRPE;
            this._totalDeltaFines    = cf._totalDeltaFines;
            this._totalDeltaCredits  = cf._totalDeltaCredits;
            this._totalFuelSavings   = cf._totalFuelSavings;
            this._totalAffectedSales = cf._totalAffectedSales;
            // copy arrays
            this.EnsureCapacity();
            Array.Copy(cf._vehs          , 0, this._vehs          , 0, this._vehCount);
            Array.Copy(cf._techKeys      , 0, this._techKeys      , 0, this._vehCount);
            Array.Copy(cf._techLingerKeys, 0, this._techLingerKeys, 0, this._vehCount);
            Array.Copy(cf._lingerYear    , 0, this._lingerYear    , 0, this._vehCount);
            Array.Copy(cf._lingering     , 0, this._lingering     , 0, this._vehCount);
            for (int i = 0; i < this._vehCount; i++)
            {
                cf._techImpacts[i].CopyTo(this._techImpacts[i]);
                Array.Copy(cf._yearOfInitialApplication[i], 0, this._yearOfInitialApplication[i], 0, TI.TechnologyCount);
            }
        }
        internal ComplianceFinding Clone()
        {
            ComplianceFinding cf = new ComplianceFinding();
            cf.CopyFrom(this);
            return cf;
        }

        /// <summary>
        /// Examines this <see cref="ComplianceFinding"/> using the parameters specified in the <see cref="Initialize"/> method
        /// and applying the given technology and vehicle(s) considerations.
        /// </summary>
        /// <param name="techIdx">The index of a technology in techGroup to examine.  The techGroup variable must be specified
        ///   by using the <see cref="Initialize"/> method before invoking this method.</param>
        /// <param name="vehs">A list of vehicles to examine for the next compliance finding.</param>
        /// <param name="vehIdx">The index of the vehicle in vehs to consider, or -1, to consider all vehicles.</param>
        /// <param name="cfCount">A reference to the total count of findings detected thus far.  When this method returns, if a
        ///   new finding was found, this value will be incremented.</param>
        /// <param name="overcomply">Specifies whether the model has entered over-compliance mode.</param>
        /// <param name="creditTrade">Specifies whether credit trading (transfers and carry fwd/back) is allowed.</param>
        public void ExamineFinding(int techIdx, List<Vehicle> vehs, int vehIdx, ref int cfCount, bool overcomply, bool creditTrade)
        {
            // generate the updated _cfTmp with vehs and tech-keys
            this.Reset();
            this.ParseVehTechsLists(techIdx, vehs, vehIdx, false);

            // check if any compliance vehicles were generated (this does not include vehs in unregulated class)
            if (!this._hasVehs) { return; }

            // continue with the cf ...
            this._overcomply  = overcomply;
            this._creditTrade = creditTrade;
            this._doInherit   = false;
            this._forcedTech  = false;

            this.CalcEfficiency();
            this._index = cfCount++;
        }
        /// <summary>
        /// Examines the effect of inheriting the specified technology on a given vehicle.
        /// </summary>
        /// <param name="techIdx">The index of a technology in techGroup to examine.  The techGroup variable must be specified
        ///   by using the <see cref="Initialize"/> method before invoking this method.</param>
        /// <param name="vehs">A list of vehicles to examine for the next compliance finding.</param>
        /// <param name="vehIdx">The index of the vehicle in vehs to consider, or -1, to consider all vehicles.</param>
        public void ExamineTechInheriting(int techIdx, List<Vehicle> vehs, int vehIdx)
        {
            this.Reset();
            this._doInherit  = true;
            this._forcedTech = false;
            this.ParseVehTechsLists(techIdx, vehs, vehIdx, false);
            this.CalcEfficiency();
        }
        /// <summary>
        /// Examines the effect of forcing the specified technology on a given vehicle.
        /// </summary>
        /// <param name="techIdx">The index of a technology in techGroup to examine.  The techGroup variable must be specified
        ///   by using the <see cref="Initialize"/> method before invoking this method.</param>
        /// <param name="vehs">A list of vehicles to examine for the next compliance finding.</param>
        /// <param name="vehIdx">The index of the vehicle in vehs to consider, or -1, to consider all vehicles.</param>
        /// <param name="ignorePlatform">true to ignore platform dependency on engine, transmission, and platform level techs.</param>
        public void ExamineForcedApplication(int techIdx, List<Vehicle> vehs, int vehIdx, bool ignorePlatfrom)
        {
            this.Reset();
            this._doInherit  = false;
            this._forcedTech = true;
            this.ParseVehTechsLists(techIdx, vehs, vehIdx, ignorePlatfrom);
            if (this._hasVehs)
            {   // continue only if any compliance vehicles were generated
                this.CalcEfficiency();
            }
        }

        void ParseVehTechsLists(int techIdx, List<Vehicle> vehs, int vehIdx, bool ignorePlatfrom)
        {
            // resize internal arrays in cf-tmp to ensure room for all vehs
            int allVehCount = (vehIdx == -1) ? vehs.Count : 1;
            this.EnsureCapacity(allVehCount);

            // save base tech index
            this._techIndex = techIdx;

            // inititialize variables ...
            Manufacturer            mfr = this._modelYearData[this._yrIndex].Manufacturers[this._mfrIndex];
            Manufacturer.CModelData mmd = mfr.ModelData;
            int yrLen = this._endYear - ModelYear.MinYear + 1;

            // examine each vehicle
            int vIdxOff = (vehIdx == -1) ? 0 : vehIdx;
            for (int i = 0; i < allVehCount; i++)
            {
                Vehicle veh = vehs[i + vIdxOff];
                //
                ModelYear lingerYear;
                bool      lingering;
                bool      isTechValid = TA.IsTechValid(this._modelYearData, this._year, veh, this._techGroup[techIdx].Index,
                                                       this._ignorePhaseIn, ignorePlatfrom, false, out lingering, out lingerYear);
                if (this._doInherit || isTechValid)
                {   // this technology can be installed on the vehicle (it's valid) -- examine & backfill
                    this._vehs          [this._vehCount] = veh;
                    this._techKeys      [this._vehCount] = TA.TechKeys[techIdx];
                    this._techLingerKeys[this._vehCount] = (lingering) ? TA.TechKeys[techIdx] : 0;
                    this._lingerYear    [this._vehCount] = (lingering) ? lingerYear.Year      : 0;
                    this._lingering     [this._vehCount] = (lingering) ? true                 : false;

                    if (veh.RegClass != RegulatoryClass.None)
                    {
                        this._hasVehs = true;
                        this._affectedRC[veh.RegClass] = true;
                    }

                    // initialize some additional variables
                    if (this._techImpacts[this._vehCount] == null)
                    {
                        this._techImpacts[this._vehCount] = new TechImpact(yrLen);
                    }
                    if (this._yearOfInitialApplication[this._vehCount] == null)
                    {
                        this._yearOfInitialApplication[this._vehCount] = new int[TI.TechnologyCount];
                    }

                    // if backfill is on, check and add the previous techs
                    if (this._backfill)
                    {
                        this.ParseVehTechsLists_Backfill(techIdx, veh, mmd, ignorePlatfrom);
                    }

                    // if any of the vehicles are lingering, compute min-linger-year
                    if (lingering)
                    {
                        if (this._minLingerYear == 0 || this._minLingerYear > lingerYear.Year)
                        {
                            this._minLingerYear = lingerYear.Year;
                        }
                    }

                    // increment vehicle counter
                    this._vehCount++;
                }
            } // next i (vehicle)
        }
        void ParseVehTechsLists_Backfill(int techIdx, Vehicle veh, Manufacturer.CModelData mmd, bool ignorePlatfrom)
        {
            Vehicle.CModelData vmd = veh.ModelData;

            for (int i = techIdx - 1; i >= 0; i--)
            {   // start from one before tech-index and examine all techs going backwards
                //
                // check if the tech can be backfilled and whether it violates backfill constraints
                int bftIndex = this._techGroup[i].Index;    // bftIndex = back-fill tech index
                //
                ModelYear lingerYear;
                bool      lingering;
                bool      isTechValid = TA.IsTechValid(this._modelYearData, this._year, veh, bftIndex, this._ignorePhaseIn,
                                                       ignorePlatfrom, true, out lingering, out lingerYear);
                bool      canBackfill = TA.CanBackfill(i, this._techGroup, this._techKeys[this._vehCount], vmd.TechUsed);
                if (isTechValid && canBackfill)
                {
                    this._techKeys[this._vehCount] += TA.TechKeys[i];
                    if (lingering || this._lingering[this._vehCount])
                    {
                        this._techLingerKeys[this._vehCount] += TA.TechKeys[i];
                        if (lingering)
                        {
                            this._lingerYear[this._vehCount] = lingerYear.Year;
                            this._lingering [this._vehCount] = true;
                        }
                    }
                }
            } // next i (backfill tech)
        }

        #endregion

        #region /* Calculating Efficiency */

        /// <summary>
        /// Calculates the efficiency rating of this <see cref="ComplianceFinding"/> instance.
        /// </summary>
        void CalcEfficiency()
        {
            //----------------------------------------------------------------------------------//
            // Calculates the efficiency rating for this CF (for LCF, this is called on _cfTmp) //
            //----------------------------------------------------------------------------------//
            this._efficiency = double.NaN;

            // get references to current year's data and mfr
            Industry     data = this._modelYearData[this._yrIndex];
            Manufacturer mfr  = data.Manufacturers[this._mfrIndex];

            //----------------------------------------------------------------------------------//
            // Check to see if the mfr is already in compliance for all reg-classes represented //
            // by the vehicles of this compliance finding.                                      //
            //----------------------------------------------------------------------------------//
            bool mfrInCompliance = true;
            foreach (RegulatoryClass rc in RCValue<object>.Classes)
            {   // if a mfr has fines for at least one affected reg-class in this CF, continue with efficiency calculation
                if (this._affectedRC[rc] && mfr.ModelData.Fines[rc] > 0) { mfrInCompliance = false; break; }
            }

            // if mfr is already in compliance, skip eff-calc, unless overcomplying or technologies are being inherited or forced
            if (mfrInCompliance && !this._overcomply && !this._doInherit && !this._forcedTech) { return; }

            // calculate technology impacts for each affected vehicle
            this.CalcTechImpacts(data.ModelData);

            // calc new cafe and new fines
            this._newCafe  = Standards.TestCAFE (mfr, this._scen, this._year, this._settings,                this._techImpacts, this._vehCount, false);
            this._newFines = Standards.TestFines(mfr, this._scen, this._year, this._settings, this._newCafe, this._techImpacts, this._vehCount, false);
            // get the current unrounded mfr fines
            this._curCafe  = Standards.TestCAFE (mfr, this._scen, this._year, this._settings,                null, 0, false);
            this._curFines = Standards.TestFines(mfr, this._scen, this._year, this._settings, this._curCafe, null, 0, false);

            // scan the impact of techs on each valid vehicle to obtain costs, sales, and fc-savings
            Manufacturer.CModelData mmd = mfr.ModelData;
            double techCost      = 0;
            double affectedSales = 0;
            double fcSavings     = 0;
            double consumerValue = 0;
            //
            for (int i = 0; i < this._vehCount; i++)
            {
                TechImpact impact = this._techImpacts[i];
                for (int yrIndex = 0; yrIndex <= this._yrIndex; yrIndex++)
                {
                    if (impact.HasImpact[yrIndex])
                    {
                        // this vehicle is selected for applicability
                        Vehicle         veh      = impact.Vehicle[yrIndex];
                        RegulatoryClass regClass = veh.RegClass;
                        VehicleClass    vehClass = veh.VehicleClass;
                        VehicleStyle    vehStyle = veh.VehicleStyle;
                        HEVType         hevType  = veh.HEVType;
                        FuelValue       vehFE    = veh.Description.FuelEconomy;
                        FuelValue       vehShare = veh.Description.FuelShare;
                        FuelValue       newFE    = impact.NewFuelEconomy[yrIndex];
                        FuelValue       newShare = impact.NewFuelShare  [yrIndex];
                        double          sales    = veh.Description.Sales[yrIndex];
                        ModelYear       lingerMY = ModelYear.NewModelYearFromIndex(yrIndex);
                        // compute fuel cost
                        KMYType   kmy = (mfrInCompliance) ? mmd.KMY_OC[yrIndex] : mmd.KMY[yrIndex];
                        FuelValue gap = this._parameters.EconomicValues.OnRoadGap[veh.VehicleClass];
                        double vehFuelCost = Standards.GetVehicleFuelCost(vehFE, vehShare, regClass, this._scen, lingerMY, gap, kmy[vehClass, vehStyle, hevType]);
                        double newFuelCost = Standards.GetVehicleFuelCost(newFE, newShare, regClass, this._scen, lingerMY, gap, kmy[vehClass, vehStyle, hevType]);
                        //
                        techCost      += sales * impact.Cost[yrIndex];
                        affectedSales += sales;
                        fcSavings     += sales * (vehFuelCost - newFuelCost);
                        consumerValue += sales * impact.ConsumerValuation[yrIndex];
                    } // end if (hasImpacts[yrIndex])
                } // next yrIndex
            } // next i (vehicle)

            //----------------------------------------------------------------------------------//
            // Check to see if the technologies generated a change in FC savings or reduction   //
            // in fines:                                                                        //
            //  - if FC savings exist or fines were reduced, finalize efficiency calcs          //
            //  - if there are no FC savings or reduction in fines, most likely the vehicle's   //
            //    fuel economy did not change -- ignore technologies and return                 //
            //  - if lingering for multiyear, check CAFE in preceding years and do not allow it //
            //    to decrease, since generated credits from those years may have already been   //
            //    used; if delta CAFE decreases -- ignore technologies and return               //
            //  - if technologies are being inherited or forced, finalize efficiency calcs even //
            //    if there was no change in FC savings or fines                                 //
            //----------------------------------------------------------------------------------//
            RCDouble deltaFines = this._curFines - this._newFines;
            if (!this._doInherit && !this._forcedTech)
            {
                if (deltaFines.Total < 0 || (deltaFines.Total == 0 && fcSavings <= 0))
                {
                    this.IgnoreTechnologies();
                    return;
                }
                // if lingering, calc new CAFE levels in preceding model years and ignore techs if it drops
                if (this.IsMultiYear)
                {
                    for (int yrIndex = this._minLingerYear - ModelYear.MinYear; yrIndex <= this._yrIndex; yrIndex++)
                    {
                        Manufacturer lMfr  = this._modelYearData[yrIndex].Manufacturers[this._mfrIndex];
                        ModelYear    lYear = ModelYear.NewModelYearFromIndex(yrIndex);
                        //
                        // calc new cafe and new fines
                        RCDouble lCurCafe = Standards.TestCAFE(lMfr, this._scen, lYear, this._settings, null, 0, true);
                        RCDouble lNewCafe = Standards.TestCAFE(lMfr, this._scen, lYear, this._settings, this._techImpacts, this._vehCount, true);
                        for (int rc = 0; rc < lCurCafe.Items.Length; rc++)
                        {
                            if (Math.Round(lNewCafe.Items[rc], 1) < Math.Round(lCurCafe.Items[rc], 1))
                            {
                                this.IgnoreTechnologies();
                                return;
                            }
                        }
                    } // nexy linger year
                }
            }

            // compute total delta fines for affected reg-classes --
            // only consider change in fines if scenario allows fine payment and mfr prefers fines
            double totalDeltaFines = 0;
            foreach (RegulatoryClass rc in RCValue<object>.Classes)
            {
                if (this._affectedRC   [rc] &&
                    this._scen.ScenInfo[rc].AllowFines[this._yrIndex] &&
                    mfr .Description.PreferFines)
                {
                    totalDeltaFines += deltaFines[rc];
                }
            }

            // final steps -- compute efficiency
            if ((this._doInherit || this._forcedTech) && affectedSales == 0)
            {   // do not compute eff if inheriting or forcing technology on a vehicle with 0 sales
                this._efficiency = 0;
            }
            else
            {
                this._efficiency = (techCost - totalDeltaFines - fcSavings - consumerValue) / affectedSales;
            }

            // save additional info
            this._techCost           = techCost;
            this._totalDeltaRPE      = techCost;
            this._totalDeltaFines    = deltaFines.Total;
            this._totalDeltaCredits  = 0;
            this._totalFuelSavings   = fcSavings;
            this._totalAffectedSales = affectedSales;
        }
        Vehicle CalcTechImpacts_GetVehicle(int vehIndex, int yrIndex)
        {
            // get the current year's vehicle
            Vehicle veh = this._vehs[vehIndex];
            //
            if (yrIndex == this._yrIndex)
            {   // if working with the current model year, return the current year's vehicle
                return veh;
            }
            else
            {   // obtain the vehicle from the lingering data
                Manufacturer  lMfr  = this._modelYearData[yrIndex].Manufacturers[this._mfrIndex];
                List<Vehicle> lVehs = lMfr.Vehicles;
                //
                int     lCode = veh.Description.Code;
                Vehicle lVeh  = null;
                for (int i = 0; i < lVehs.Count; i++)
                {
                    if (lVehs[i].Description.Code == lCode) { lVeh = lVehs[i]; break;  }
                }
                //
                return lVeh;
            } // end if
        }
        void CalcTechImpacts(Industry.CModelData imd)
        {
            Technologies techData = this._settings.Technologies;
            //
            for (int i = 0; i < this._vehCount; i++)
            {   // reset the previous impact
                this._techImpacts[i].Reset();

                // initialize variables ...
                Vehicle curVeh        = this._vehs          [i];                            // the current year's vehicle
                ulong   curTechKey    = this._techKeys      [i];                            // the current year's tech-key
                ulong   lingerTechKey = this._techLingerKeys[i];                            // tech-key for lingering years
                bool    lingering     = this._lingering     [i];                            // the veh is using lingering techs if the linger key != 0
                int     lingerYrYear  = (lingering) ? this._lingerYear[i] : this._yrYear;   // if lingering, get linger yr, otherwise, use current yr
                int     lingerYrIndex = lingerYrYear - ModelYear.MinYear;                   // get the linger year index also

                // get reference to the tech-impact for the current veh
                TechImpact impact = this._techImpacts[i];

                // yearOfInitialApplication - indicates the initial year when a technology is applied;
                // this is useful for multiyear modeling so that veh.TechAppliedYear is properly assigned
                Array.Clear(this._yearOfInitialApplication[i], 0, TI.TechnologyCount);

                //------------------------------------//
                // Examine techs during each year ... //
                //------------------------------------//
                for (int j = lingerYrIndex; j <= this._yrIndex; j++)
                {   // obtain the appropriate vehicle for the year being examined
                    //  * if the year being examined is the current year, the current vehicle will be returned
                    //  * if the year being examined is not the current year (e.g., in multi-year mode), the vehicle
                    //    from the previous year's data will be returned
                    Vehicle veh = this.CalcTechImpacts_GetVehicle(i, j);

                    // if veh is null, the impact should be ignored
                    if (veh == null) { ThrowHelper.InternalModelError(); }

                    // prepare additional data ...
                    ModelYear            year = ModelYear.NewModelYearFromIndex(j); // the model year being examined
                    Industry             data = this._modelYearData[j];
                    Vehicle.CDescription vd   = veh.Description;
                    Vehicle.CModelData   vmd  = veh.ModelData;
                    // for the last year, use tech-key, for all others, use tech-linger-key
                    ulong                techKey   = (j == this._yrIndex) ? curTechKey : lingerTechKey;

                    // get current fuel and HEV status on a vehicle --
                    // as techs are evaluated for application below, these will change to reflect selected technologies
                    HEVType  vehHevType  = veh.HEVType;
                    FuelType vehFuelType = vd .FuelShare.FuelType;

                    if (techKey != 0)
                    {   // tech-key has valid data -- indicate that the year being examined has impact data
                        impact.HasImpact[j] = true;
                        impact.Vehicle  [j] = veh;      // save veh associated with the impact

                        // get the current fuel share and fuel economy for the vehicle
                        impact.NewFuelShare  [j] = vd.FuelShare;
                        impact.NewFuelEconomy[j] = vd.FuelEconomy;

                        // set initial improvement values
                        impact.FC               [j] = 0;
                        impact.FCSecondary      [j] = 0;
                        impact.FCAdjustment     [j] = 1;
                        impact.FCPowerAdjustment[j] = 1;

                        // reset committed EPR-costs --
                        // when techs are backfilled, penalty costs could be double counted if two or more technologies being
                        // applied supersede another technology that has penalty costs; once a backfill tech accrues a penalty,
                        // the EPR-costs of a superseded tech are "committed" and won't be counted for the next backfill tech
                        Array.Clear(this._committedERPCosts, 0, TI.TechnologyCount);

                        // special handling needs to occur when evaluating the incremental cost and FC of PHEV30 technology,
                        // while backfilling SHEVP2 technology in the same step --
                        // this flag tracks whether the model has selected SHEVP2 for application and adjusts accordingly
                        bool isShevP2Selected = false;

                        // iterate all available technologies
                        for (int k = 0; k < this._techCount; k++)
                        {
                            if ((techKey & TA.TechKeys[k]) != 0)
                            {   // this tech is selected to be applied on the current vehicle
                                TechnologyInfo           tech = this._techGroup[k];
                                TechnologyAttributes techAttr = tech.GetAttributes(veh);
                                //
                                // aggregate the consumer valuation of the technologies
                                impact.ConsumerValuation[j] += techAttr.ConsumerValuation;

                                // update the year of initial application as necessary
                                if (this._yearOfInitialApplication[i][tech.Index] == 0) { this._yearOfInitialApplication[i][tech.Index] = j; }

                                // set SHEVP2 flag if the tech is selected for application
                                if (tech.Index == TI.SHEVP2) { isShevP2Selected = true; }

                                // compute the cost
                                double erpCost;
                                double cost = TA.GetCost(tech, veh, year, techData, isShevP2Selected,
                                    this._yearOfInitialApplication[i][tech.Index], this._startYear, this._endYear,
                                    out erpCost, ref this._committedERPCosts);
                                impact.Cost           [j] += cost;
                                impact.ERPCost        [j] += erpCost;
                                impact.MaintenanceCost[j] += TA.GetMaintenanceCost(tech, veh, year);
                                impact.RepairCost     [j] += TA.GetRepairCost     (tech, veh, year);

                                // compute the FC and fuel share
                                double fc1, fc2, fs2, fcAdj, fcPwrAdj, deltaPwr;
                                // get fc values
                                TA.GetImprovement(tech, veh, this._scen, year, this._settings, impact, isShevP2Selected,
                                    out fc1, out fc2, out fs2, out fcAdj, out fcPwrAdj, out deltaPwr);

                                // compute off-cycle credit
                                impact.OffCycleCredit[j] += tech.OffCycleCredit[veh.RegClass];

                                // save ZEV credit for the current technology
                                // when backfilling multiple techs, the ZEV credit from latest technology will override
                                // all preceding ones
                                if (TI.IsConversionToZEV(tech.Index))
                                {
                                    impact.ZEVCredit[j] = tech.ZEVCredits;
                                }

                                // get info about current primary fuel (this value may change based on the last tech examined)
                                FuelType primaryFuel = impact.NewFuelShare[j].PrimaryFuel;

                                //--------------------------------------------------------------//
                                // Examine certain technologies with special considerations     //
                                //  1)  detect if the technology triggers a fuel type change    //
                                //  2a) if there is no fuel type change, proceed as usual       //
                                //  2b) if there is a fuel type change, migrate fuel shares and //
                                //      fuel economy to new fuel type                           //
                                //--------------------------------------------------------------//
                                if (TI.IsConversionToDiesel(tech.Index) && (vehFuelType & FuelType.Diesel) != FuelType.Diesel)
                                {
                                    this.CalcTechImpacts_ConvertFuel(impact, j, primaryFuel, FuelType.Diesel, fc1);
                                    vehFuelType = FuelType.Diesel;  // update veh fuel type to reflect conversion to new tech
                                }
                                else if (TI.IsConversionToCNG(tech.Index) && (vehFuelType & FuelType.CNG) != FuelType.CNG)
                                {
                                    this.CalcTechImpacts_ConvertFuel(impact, j, primaryFuel, FuelType.CNG, fc1);
                                    vehFuelType = FuelType.CNG;     // update veh fuel type to reflect conversion to new tech
                                }
                                //else if (TI.IsConversionToLNG(tech.Index) && (vehFuelType & FuelType.LNG) != FuelType.LNG)
                                //{
                                //    this.CalcTechImpacts_ConvertFuel(impact, j, primaryFuel, FuelType.LNG, fc1);
                                //    vehFuelType = FuelType.LNG;     // update veh fuel type to reflect conversion to new tech
                                //}
                                //else if (TI.IsConversionToLPG(tech.Index) && (vehFuelType & FuelType.LPG) != FuelType.LPG)
                                //{
                                //    this.CalcTechImpacts_ConvertFuel(impact, j, primaryFuel, FuelType.LPG, fc1);
                                //    vehFuelType = FuelType.LPG;     // update veh fuel type to reflect conversion to new tech
                                //}
                                else if (TI.IsConversionToSHEV(tech.Index) && vehHevType != HEVType.StrongHybrid)
                                {
                                    this.CalcTechImpacts_ConvertFuel(impact, j, primaryFuel, primaryFuel, fc1);
                                    vehHevType = HEVType.StrongHybrid;  // update veh HEV type to reflect conversion to new tech
                                }
                                else if (TI.IsConversionToPHEV(tech.Index))
                                {
                                    if (vehHevType != HEVType.PlugInHybrid)
                                    {   // save FE on primary fuel (which is the basis for new FE values)
                                        double primaryFE = impact.NewFuelEconomy[j][primaryFuel];
                                        // compute new FE and FS on primary fuel and on electricity
                                        impact.NewFuelEconomy[j][primaryFuel         ] = primaryFE / (1 - fc1);
                                        impact.NewFuelEconomy[j][FuelType.Electricity] = primaryFE / (1 - fc2);
                                        impact.NewFuelShare  [j][primaryFuel         ] = (1 - fs2);
                                        impact.NewFuelShare  [j][FuelType.Electricity] =      fs2;
                                        // calculate and update improvement values
                                        impact.FCSecondary   [j]  = impact.FC[j];
                                        impact.FC            [j]  = 1 - (1 - impact.FC         [j]) * (1 - fc1);
                                        impact.FCSecondary   [j]  = 1 - (1 - impact.FCSecondary[j]) * (1 - fc2);
                                    }
                                    else
                                    {   // vehicle is already a PHEV/EREV -- no conversion necessary
                                        impact.NewFuelEconomy[j][primaryFuel         ] /= (1 - fc1);
                                        impact.NewFuelEconomy[j][FuelType.Electricity] /= (1 - fc2);
                                        impact.NewFuelShare  [j][primaryFuel         ]  = (1 - fs2);
                                        impact.NewFuelShare  [j][FuelType.Electricity]  =      fs2;
                                        // calculate and update improvement values
                                        impact.FC            [j] = 1 - (1 - impact.FC         [j]) * (1 - fc1);
                                        impact.FCSecondary   [j] = 1 - (1 - impact.FCSecondary[j]) * (1 - fc2);
                                    }
                                    // update veh HEV type to reflect conversion to new tech
                                    vehHevType = HEVType.PlugInHybrid;
                                }
                                else if (TI.IsConversionToEV(tech.Index) && vehHevType != HEVType.PureElectric)
                                {
                                    impact.FC[j] = impact.FCSecondary[j];
                                    impact.FCSecondary[j] = 0;
                                    this.CalcTechImpacts_ConvertFuel(impact, j, FuelType.Electricity, FuelType.Electricity, fc1);
                                    vehHevType = HEVType.PureElectric; // update veh HEV type to reflect conversion to new tech
                                }
                                else if (TI.IsConversionToFCV(tech.Index) && vehHevType != HEVType.FuelCell)
                                {
                                    impact.FC[j] = impact.FCSecondary[j];
                                    impact.FCSecondary[j] = 0;
                                    this.CalcTechImpacts_ConvertFuel(impact, j, FuelType.Electricity, FuelType.Hydrogen, fc1);
                                    vehHevType = HEVType.FuelCell; // update veh HEV type to reflect conversion to new tech
                                }
                                else
                                {   // the technology being examined does not have any special considerations
                                    if (impact.NewFuelShare[j].IsMultiFuel)
                                    {
                                        foreach (FuelType f in FTValue<object>.Classes)
                                        {
                                            if (impact.NewFuelShare[j][f] > 0)
                                            {
                                                impact.NewFuelEconomy[j][f] /= (1 - fc1);
                                            }
                                        }
                                    }
                                    else
                                    {
                                        impact.NewFuelEconomy[j][primaryFuel] /= (1 - fc1);
                                    }
                                    // update improvement values
                                    impact.FC[j]  = 1 - (1 - impact.FC[j]) * (1 - fc1);
                                }

                                if (TI.IsFCAdjKey(tech.Index))
                                {   // save FC adj factor for the current technology
                                    // when backfilling multiple techs, the adj factor from latest technology will override
                                    // all preceding ones
                                    impact.FCAdjustment[j] = fcAdj;
                                }

                                // save FC power adj factor and delta power (if any)
                                impact.FCPowerAdjustment[j] *= fcPwrAdj;
                                impact.DeltaPower       [j] -= deltaPwr;

                                // compute changes in weight for the vehcile
                                if (TI.IsCostPerPound    (tech.Index) ||
                                    TI.IsConversionToSHEV(tech.Index) ||
                                    TI.IsConversionToPHEV(tech.Index) ||
                                    TI.IsConversionToEV  (tech.Index))
                                {
                                    double deltaCW, deltaGVWR, deltaGCWR;
                                    //
                                    TechnologyApplicability.GetWeightChange(tech, veh, this._scen, year, impact, out deltaCW, out deltaGVWR, out deltaGCWR);
                                    impact.DeltaCurbWeight[j] -= deltaCW;
                                    impact.DeltaGVWR      [j] -= deltaGVWR;
                                    impact.DeltaGCWR      [j] -= deltaGCWR;
                                }
                            } // end if (tech selected for applicability)
                        } // next k (technology)

                        // after all technologies within the backfill stream have been evaluated:
                        //  - apply adj factors to final FE
                        //  - compute delta FE
                        FuelValue vehFE = impact.Vehicle[j].Description.FuelEconomy;
                        foreach (FuelType f in FTValue<object>.Classes)
                        {
                            if (impact.NewFuelShare[j][f] > 0)
                            {
                                impact.NewFuelEconomy[j][f] = impact.NewFuelEconomy   [j][f] /
                                                              impact.FCAdjustment     [j]    /
                                                              impact.FCPowerAdjustment[j];
                            }
                            impact.DeltaFuelEconomy[j][f] = impact.NewFuelEconomy[j][f] - vehFE[f];
                        }
                    } // end if (techKey != 0)
                } // next j ([linger] year index)
            } // next i (veh)
        }
        void CalcTechImpacts_ConvertFuel(TechImpact impact, int yr, FuelType fromFuel, FuelType toFuel, double fc1)
        {
            // compute new fuel economy value using old fuel type
            double newFE = impact.NewFuelEconomy[yr][fromFuel] / (1 - fc1);
            // clear existing fuel economy and fuel share values
            impact.NewFuelEconomy[yr].Clear();
            impact.NewFuelShare  [yr].Clear();
            // save new fuel economy and fuel share to new fuel type
            impact.NewFuelEconomy[yr][toFuel] = newFE;
            impact.NewFuelShare  [yr][toFuel] = 1;
            // save improvement values
            impact.FC            [yr] = 1 - (1 - impact.FC[yr]) * (1 - fc1);
        }

        /// <summary>
        /// Sets the TechIgnored flag to true for all vehicle/technology combinations associated with this
        /// <see cref="ComplianceFinding"/> instance.
        /// </summary>
        /// <remarks>
        /// Once the TechIgnored flag is set on a vehicle for a particular technology, that technology will not be available for
        /// applicability for the associated manufacturer until it is cleared. Typically the Compliance model clears the
        /// TechIgnored flag for all vehicles at the beginning of the model year loop.
        /// </remarks>
        public void IgnoreTechnologies()
        {
            for (int i = 0; i < this._vehCount; i++)
            {   // the veh is being considered for applicability
                Vehicle.CModelData vmd = this._vehs[i].ModelData;
                for (int j = 0; j < this._techCount; j++)
                {   // if the tech-key from the i-th vehicle is built upon the global key from the j-th tech,
                    // then the j-th tech is selected to be applied on the i-th vehicle
                    if ((this._techKeys[i] & TA.TechKeys[j]) != 0)
                    {
                        vmd.TechIgnored[this._techGroup[j].Index] = true;
                    }
                } // next j (technology)
            } // next i (vehicle)
        }

        #endregion

        #region /* Applying CF */

        /// <summary>
        /// Applies all technology/vehicle combinations available in this <see cref="ComplianceFinding"/> instance.
        /// </summary>
        public void ApplyFinding()
        {
            if (!this.IsValid  ) { throw new InvalidOperationException("The ComplianceFinding being applied is not valid."); }
            if (this._isApplied) { throw new InvalidOperationException("The ComplianceFinding has already been applied."  ); }

            // mark finding as applied and log it
            this._isApplied = true;
            this.LogFinding();

            // begin applying the finding for each valid vehicle
            for (int i = 0; i < this._vehCount; i++)
            {
                this.ApplyTechImpact(this._techImpacts[i], this._techKeys[i], this._techLingerKeys[i], this._lingerYear[i],
                    this._yearOfInitialApplication[i]);
            }

            // increment "applied" counter
            //LCF._cfApplied++;
        }
        void ApplyTechImpact(TechImpact impact, ulong curTechKey, ulong lingerTechKey, int lingerYear, int[] yearOfInitApplication)
        {
            for (int i = 0; i <= this._yrIndex; i++)
            {
                ModelYear year = new ModelYear(ModelYear.MinYear + i);
                if (impact.HasImpact[i])
                {   // the year has an impact (either for the current or linger year)
                    Vehicle                   veh = impact.Vehicle[i];
                    Vehicle     .CDescription vd  = veh.Description;
                    Vehicle     .CModelData   vmd = veh.ModelData;
                    Manufacturer.CModelData   mmd = veh.Manufacturer.ModelData;
                    // for the last year, use tech-key, for all others, use tech-linger-key
                    ulong techKey = (i == this._yrIndex) ? curTechKey : lingerTechKey;

                    RegulatoryClass regClass = veh.RegClass;
                    VehicleClass    vehClass = veh.VehicleClass;
                    double          vehSales = vd.Sales[i];

                    // find base technology being applied (all others are being backfilled)
                    ulong baseTech = 0;
                    for (int j = this._techCount - 1; j >= 0; j--)
                    {   // iterate all available techs
                        if ((techKey & TA.TechKeys[j]) != 0) { baseTech = TA.TechKeys[j]; break; }
                    }
                    // apply technologies
                    for (int j = 0; j < this._techCount; j++)
                    {   // iterate all available techs
                        if ((techKey & TA.TechKeys[j]) != 0)
                        {   // this tech is selected to be applied on the current vehicle, for the current year
                            int techIndex = this._techGroup[j].Index;
                            this.ApplyTechnology(veh, vmd, mmd, regClass, vehSales, this._techGroup[j], year, lingerYear,
                                yearOfInitApplication[techIndex], TA.TechKeys[j] != baseTech);
                        }
                    } // next j (technology)

                    // update vehicle power, CW, GVWR, GCWR
                    if (impact.HasPowerChanged (i)) { vd.VehiclePower += impact.DeltaPower     [i]; }
                    if (impact.HasWeightChanged(i)) { vd.CurbWeight   += impact.DeltaCurbWeight[i]; }
                    if (impact.HasGVWRChanged  (i)) { vd.GVWR         += impact.DeltaGVWR      [i]; }
                    if (impact.HasGCWRChanged  (i)) { vd.GCWR         += impact.DeltaGCWR      [i]; }
                    // update vehicle FE and fuel share
                    if (!impact.NewFuelShare[i].IsValid || !impact.NewFuelEconomy[i].IsValid)
                    {
                        throw new InvalidOperationException(
                            "Error in ComplianceFinding.  Vehicle's new Fuel Economy and/or Fuel Share values are not valid.");
                    }
                    else
                    {
                        vd.FuelShare   = impact.NewFuelShare  [i];
                        vd.FuelEconomy = impact.NewFuelEconomy[i];
                    }
                    // update vehicle off-cycle credit
                    vmd.OffCycleCredits             +=             impact.OffCycleCredit   [i] ;
                    // calculate and update vehicle (and mfr) costs and loss of value
                    vmd.TechCost                    +=             impact.Cost             [i] ;
                    vmd.ConsumerValuation           +=             impact.ConsumerValuation[i] ;
                    vmd.MaintenanceCost             +=             impact.MaintenanceCost  [i] ;
                    vmd.RepairCost                  +=             impact.RepairCost       [i] ;
                    mmd.TechCost         [regClass] += (vehSales * impact.Cost             [i]);
                    mmd.ConsumerValuation[regClass] += (vehSales * impact.ConsumerValuation[i]);
                    mmd.MaintenanceCost  [regClass] += (vehSales * impact.MaintenanceCost  [i]);
                    mmd.RepairCost       [regClass] += (vehSales * impact.RepairCost       [i]);
                } // end if (hasImpact[yrIndex])
            } // next yrIndex
        }
        void ApplyTechnology(Vehicle veh, Vehicle.CModelData vmd, Manufacturer.CModelData mmd, RegulatoryClass regClass,
            double vehSales, TechnologyInfo tech, ModelYear year, int lingerYear, int yearOfInitApplication, bool backfilling)
        {
            //----------------------------------------------------------------------------------//
            // Clarification of certain params:                                                 //
            //  year:  current model year for which to apply the technology to a vehicle        //
            //  yearOfInitialApplication:                                                       //
            //         if tech is applied due to multiyear, this is the initial year when the   //
            //         tech was applied; otherwise, this will be same as "year" parameter       //
            //----------------------------------------------------------------------------------//

            // update vehicle configuration based on the current technology
            bool isInherited = (this._doInherit || lingerYear != this._minLingerYear);
            TA.InstallTechnologyOnVehicle(veh, tech, year, yearOfInitApplication, isInherited, this._settings);

            // update sales volumes on the mfr
            mmd.TechUsedSales   [tech.Index][regClass] += vehSales;
            mmd.TechAppliedSales[tech.Index][regClass] += vehSales;

            if (!this._settings.OperatingModes.IgnorePhaseIn)
            {   // check phase-in caps (and update as necessary)
                TA.CheckPhaseIn(this._settings, mmd, tech, year, this._startYear - ModelYear.MinYear);
            }
        }

        #endregion

        #region /* Logging CF */

        /// <summary>
        /// Writes the details of this <see cref="ComplianceFinding"/> instance to a log file.
        /// </summary>
        public void LogFinding()
        {
            if (!this.IsValid || this._logWriter.IsEmpty) { return; }
            // get the current year's mfr reference
            Manufacturer mfr = this._modelYearData[this._yrIndex].Manufacturers[this._mfrIndex];
            // log the finding
            ComplianceWriter log    = (this._logWriter.CFs    == null) ? null : this._logWriter.CFs   [this._scen.Index];
            ComplianceWriter extLog = (this._logWriter.ExtCFs == null) ? null : this._logWriter.ExtCFs[this._scen.Index];
            if (log    != null && this._isApplied) { this.LogFinding(log   , mfr.ModelData); }
            if (extLog != null                   ) { this.LogFinding(extLog, mfr.ModelData); }
        }
        void LogFinding(ComplianceWriter cw, Manufacturer.CModelData mmd)
        {
            StringBuilder line = new StringBuilder(1024);
            int lineNumber = cw.LineCount + 1;
            // examine each vehicle impact
            for (int i = 0; i < this._vehCount; i++)
            {
                TechImpact impact = this._techImpacts[i];
                //
                Vehicle baseVeh     = impact.Vehicle[this._yrIndex];
                int     baseVehCode = baseVeh.Description.Code;
                //if (isFirstVeh) { isFirstVeh = false; } else { line.Append(lineNumber++); line.Append(TAB_7); }
                // scan each "linger-year" vehicle
                for (int j = impact.HasImpact.Length - 1; j >= 0; j--)
                {   // skip yeas without an impact
                    if (!impact.HasImpact[j]) { continue; }

                    // this year has an impact -- log vehicle and impact data
                    Vehicle              veh = impact.Vehicle[j];
                    Vehicle.CDescription vd  = veh.Description;
                    ulong techKey    = (j == this._yrIndex) ? this._techKeys[i] : this._techLingerKeys[i];
                    bool isInherited = (this._doInherit || this._lingerYear[i] != this._minLingerYear);
                    //
                    // write base info
                    line.Append(lineNumber++);                                      line.Append(TAB);
                    line.Append((this._forcedTech) ? "F" :
                                (this._doInherit ) ? "I" : this._index.ToString()); line.Append(TAB);
                    line.Append(this._scen.Index);                                  line.Append(TAB);
                    line.Append(vd.Manufacturer);                                   line.Append(TAB);
                    line.Append((!this._isApplied  ) ? '\0' :
                                ( this._forcedTech ) ?  'F' :
                                ( isInherited      ) ?  'I' :
                                ( this._overcomply ) ?  'O' : 'A');                 line.Append(TAB);
                    line.Append(this._efficiency);                                  line.Append(TAB);
                    line.Append(this._totalDeltaRPE     .ToString("0.####"));       line.Append(TAB);
                    line.Append(this._totalDeltaFines   .ToString("0.####"));       line.Append(TAB);
                    line.Append(this._totalDeltaCredits .ToString("0.####"));       line.Append(TAB);
                    line.Append(this._totalFuelSavings  .ToString("0.####"));       line.Append(TAB);
                    line.Append(this._totalAffectedSales.ToString("0.####"));       line.Append(TAB);
                    //
                    // write reg-class, stnd, cafe, and fines
                    RegulatoryClass regClass = veh.RegClass;
                    line.Append(               regClass );  line.Append(TAB);
                    line.Append(mmd.Standard  [regClass]);  line.Append(TAB);
                    line.Append(this._curCafe [regClass]);  line.Append(TAB);
                    line.Append(this._newCafe [regClass]);  line.Append(TAB);
                    line.Append(this._curFines[regClass]);  line.Append(TAB);
                    line.Append(this._newFines[regClass]);  line.Append(TAB);
                    //
                    if (baseVehCode != vd.Code)
                    {   // a predecessor is being used in the linger year -- note that in the log
                        line.Append('['); line.Append(vd.Code); line.Append(']');
                    }
                    line.Append(vd .Code            );      line.Append(TAB);
                    line.Append(vd .EngineCode      );      line.Append(TAB);
                    line.Append(vd .TransmissionCode);      line.Append(TAB);
                    line.Append(veh.TechClassString );      line.Append(TAB);
                    //
                    // examine techs selected for applicability
                    bool isFirstTech = true;
                    for (int k = 0; k < this._techCount; k++)
                    {
                        if ((techKey & TA.TechKeys[k]) != 0)
                        {   // this tech is selected for applicability
                            if (isFirstTech) { isFirstTech = false; } 
                            else { line.Append(", "); }
                            line.Append(this._techGroup[k].Name);
                        } // end if (tech selected for applicability)
                    } // next k (technology)
                    line.Append(TAB);
                    //
                    line.Append(j + ModelYear.MinYear);                                line.Append(TAB);
                    line.Append(vd    .FuelEconomy         .ToString("0.####", true)); line.Append(TAB);
                    line.Append(vd    .FuelShare           .ToString("0.####", true)); line.Append(TAB);
                    line.Append(impact.NewFuelEconomy   [j].ToString("0.####", true)); line.Append(TAB);
                    line.Append(impact.NewFuelShare     [j].ToString("0.####", true)); line.Append(TAB);
                    line.Append(vd    .Sales            [j].ToString(              )); line.Append(TAB);
                    line.Append(impact.Cost             [j].ToString("0.####"      )); line.Append(TAB);
                    line.Append(impact.ERPCost          [j].ToString("0.####"      )); line.Append(TAB);
                    line.Append(impact.FC               [j].ToString("0.########"  )); line.Append(TAB);
                    line.Append(impact.FCSecondary      [j].ToString("0.########"  )); line.Append(TAB);
                    line.Append(impact.FCAdjustment     [j].ToString("0.########"  )); line.Append(TAB);
                    line.Append(impact.FCPowerAdjustment[j].ToString("0.########"  )); line.Append(TAB);
                    line.Append(impact.DeltaFuelEconomy [j].ToString("0.####", true)); line.Append(TAB);
                    line.Append(impact.OffCycleCredit   [j].ToString("0.####"      )); line.Append(TAB);
                    line.Append(impact.DeltaPower       [j].ToString("0.####"      )); line.Append(TAB);
                    line.Append(impact.DeltaCurbWeight  [j].ToString("0.####"      )); line.Append(TAB);
                    line.Append(impact.DeltaGVWR        [j].ToString("0.####"      )); line.Append(TAB);
                    line.Append(impact.DeltaGCWR        [j].ToString("0.####"      ));
                    // add line breaker to the end
                    line.Append(NewLine);
                } // next j (impact from linger year)
            }
            // write the log ...
            cw.Write(line.ToString());
        }

        #endregion

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets the unique index number for the ComplianceFinding.</summary>
        public int Index { get { return this._index; } }

        /// <summary>Gets whether this ComplianceFinding instance is valid.  A ComplianceFinding is valid if
        ///   CalcEfficiency has been called on the instance and the resultant efficiency calculation represents a
        ///   valid value.</summary>
        /// <remarks>If, during ComplianceFinding.CalcEfficiency(), the fines to the manufacturer do not change as a result of
        ///   the available technology's impact, or if it turns out that the ComplianceFinding does not have any vehicles, the
        ///   efficiency calculation, and the associated technology costs would be assigned a value of Double.NaN.</remarks>
        public bool IsValid { get { return !double.IsNaN(this._efficiency); } }

        /// <summary>Gets whether this ComplianceFinding instance is efficient.</summary>
        /// <remarks>A ComplianceFinding is considered efficient if its efficiency rating is less than 0.  Efficiency values of
        ///   greater than or equal to 0 are considered inefficient.  Invalid ComplianceFinding instances are automatically
        ///   considered inefficient.</remarks>
        public bool IsEfficient { get { return (this._efficiency < 0); } }

        /// <summary>Gets whether this ComplianceFinding instance represents a multiyear solution.</summary>
        public bool IsMultiYear { get { return this._minLingerYear != 0; } }

        /// <summary>Gets whether this ComplianceFinding instance has been applied.</summary>
        public bool IsApplied { get { return this._isApplied; } }

        /// <summary>Gets the efficiency rating of this ComplianceFinding instance.</summary>
        /// <remarks>If the value of the efficiency rating is Double.NaN, the ComplianceFinding is considered invalid and should
        ///   be discarded.  Invalid findings will throw an exception when a call to ApplyFinding is made.</remarks>
        public double Efficiency { get { return this._efficiency; } }

        /// <summary>Gets the total technology costs that a manufacturer will incur as a result of applying the technologies
        ///   represented by this ComplianceFinding instance to all vehicles in the instance.</summary>
        /// <remarks>Not all technologies will be applied to all vehicles.  If a technology is marked as not applicable for a
        ///   certain vehicle, that technology will not be applied and its costs will not be accumulated for that vehicle.</remarks>
        public double TechCost { get { return this._techCost; } }

        #endregion

        #region /*** Variables ***/

        const string TAB     = "\t";
        const string NewLine = "\n";

        // ----- input variables ----- //
        Scenario             _scen;
        ModelYear            _year;
        int                  _yrIndex;          // the index of "_year"
        int                  _yrYear;           // the actual model year of "_year"
        int                  _startYear;
        int                  _endYear;
        Industry[]           _modelYearData;
        int                  _mfrIndex;
        ModelingSettings     _settings;
        Parameters           _parameters;       // derived from "_settings"
        bool                 _multiYear;
        bool                 _ignorePhaseIn;
        bool                 _backfill;
        LogWriter            _logWriter;
        List<TechnologyInfo> _techGroup;        // techs selected for applicability, filtered using _techKeys for each veh
        int                  _techCount;
        bool                 _doInherit;        // indicates if model is inheriting previous technology (instead of doing regular compliance finding)
        bool                 _forcedTech;       // indicates if model is forcing application of a technology (instead of doing regular compliance finding)

        // ----- runtime variables ----- //
        internal int _index;                    // the index assigned to this compliance finding
        internal int _vehCount;                 // number of vehs (including those to ignore) in this CF
        internal int _capacity;                 // used internally

        internal Vehicle   [] _vehs;            // vehs selected for applicability
        int                   _techIndex;       // base tech index being examined (techKeys include this index and others that were backfilled)
        internal ulong     [] _techKeys;        // one tech-key per vehicle
        internal ulong     [] _techLingerKeys;  // indicates if the technology is lingering from previous redesign
        int                [] _lingerYear;      // the lingering redesing year
        int                   _minLingerYear;   // lowest lingering redesing year accross all vehicles in this CF
        bool               [] _lingering;       // if a vehicle is lingering
        internal TechImpact[] _techImpacts;     // the impact of techs on vehs selected for applicability
        // yearOfInitialApplication - indicates the initial year when a technology is applied;
        // this is useful for multiyear modeling so that veh.TechAppliedYear is properly assigned
        // the outer index of this variable correspond to the "_vehs" and "_techImpacts" arrays, while the inner index is the tech count
        int              [][] _yearOfInitialApplication;

        // when techs are backfilled, penalty costs could be double counted if two or more technologies being
        // applied supersede another technology that has penalty costs; once a backfill tech accrues a penalty,
        // the EPR-costs of a superseded tech are "committed" and won't be counted for the next backfill tech
        // NOTE** this is a temporary pre-allocated variable used for performance only, and should not be used
        //        anywhere other than CalcTechImpacts
        bool[] _committedERPCosts;

        internal RCBoolean _affectedRC;         // specifies which reg-classes are represented by this CF
        internal bool      _hasVehs;            // specifies if there is at least one regulated veh in this CF
        internal bool      _isApplied;
        internal bool      _overcomply;
        internal bool      _creditTrade;

        internal double    _efficiency;
        internal double    _techCost;

        RCDouble _newCafe;                      // new CAFE rating after tech-application (not rounded)
        RCDouble _newFines;                     // new fines after tech-application, using unrounded CAFE
        RCDouble _curCafe;                      // CAFE rating before tech-application (not rounded)
        RCDouble _curFines;                     // fines before tech-application, using unrounded CAFE

        double _totalDeltaRPE;
        double _totalDeltaFines;
        double _totalDeltaCredits;
        double _totalFuelSavings;
        double _totalAffectedSales;

        #endregion

    }
}
