using System;
using Volpe.Cafe.IO;
using Volpe.Cafe.Utils;
using Volpe.XlLib;

namespace Volpe.Cafe.Settings
{
    /// <summary>
    /// Provides output configuration settings for the modeling system.
    /// </summary>
    [Serializable]
    public class OutputSettings : ISettings, ICloneable
    {

        #region /*** Methods ***/

        #region /* ICloneable/ISettings Members */

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="OutputSettings"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="OutputSettings"/>.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="OutputSettings"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="OutputSettings"/>.</returns>
        ISettings ISettings.Clone()
        {
            return this.Clone();
        }
        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="OutputSettings"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="OutputSettings"/>.</returns>
        OutputSettings Clone()
        {
            OutputSettings value = new OutputSettings();
            Clone(this, value);
            return value;
        }
        static void Clone(OutputSettings src, OutputSettings dst)
        {
            dst.OutputPath            = src.OutputPath;
            //
            dst.DisableLogWriter      = src.DisableLogWriter;
            dst.WriteLogFiles         = src.WriteLogFiles;
            dst.WriteExtendedLogFiles = src.WriteExtendedLogFiles;
            //
            dst.GenerateCSVReports    = src.GenerateCSVReports;
            dst.GenerateXLReports     = src.GenerateXLReports;
        }

        #endregion

        #region /* ISettings Members */

        /// <summary>
        /// Sets the initial state for the <see cref="ISettings"/> object to the default values.
        /// </summary>
        void ISettings.SetDefaultSettings()
        {
            this.OutputPath            = ApplicationInfo.DefaultOutputPath;
            //
            this.DisableLogWriter      = false;
            this.WriteLogFiles         = true;
            this.WriteExtendedLogFiles = false;
            //
            this.GenerateCSVReports    = true;
            this.GenerateXLReports     = true;
        }

        /// <summary>
        /// Updates this <see cref="OutputSettings"/> object with values from the specified new settings.
        /// </summary>
        /// <param name="newSettings">An <see cref="OutputSettings"/> object containing new runtime settings for this instance.</param>
        void ISettings.UpdateSettings(ISettings newSettings)
        {
            this.UpdateSettings((OutputSettings)newSettings);
        }
        /// <summary>
        /// Updates this <see cref="OutputSettings"/> object with values from the specified new settings.
        /// </summary>
        /// <param name="newSettings">An <see cref="OutputSettings"/> object containing new runtime settings for this instance.</param>
        public void UpdateSettings(OutputSettings newSettings)
        {
            Clone(newSettings, this);
        }

        /// <summary>
        /// Writes settings data to the summary.txt log file.
        /// </summary>
        void ISettings.WriteSummaryLog(LogWriter writer)
        {
            if (writer.Summary == null) { return; }
            string outStr = "--- Output Settings ---" +
                "\n  OutputPath="              + this.OutputPath            +
                "\n  DisableLogWriter="        + this.DisableLogWriter      +
                "\n  WriteLogFiles="           + this.WriteLogFiles         +
                "\n    WriteExtendedLogFiles=" + this.WriteExtendedLogFiles +
                "\n  GenerateCSVReports="      + this.GenerateCSVReports    +
                "\n  GenerateXLReports="       + this.GenerateXLReports;
            writer.Summary.WriteLine(outStr + "\n");
        }

        #endregion

        #endregion


        #region /*** Variables ***/

        /// <summary>Specifies the default <see cref="XlCryptoSettings"/> used for weak output encryption.  This field is
        ///   read-only.</summary>
        public static readonly XlCryptoSettings DefaultWeakEncryption = XlCryptoSettings.OfficeXor;
        /// <summary>Specifies the default <see cref="XlCryptoSettings"/> used for strong output encryption.  This field is
        ///   read-only.</summary>
        public static readonly XlCryptoSettings DefaultStrongEncryption = XlCryptoSettings.RC4_MsftRSASChannel;

        /// <summary>Specifies the default output path, where the diagnostic files and reports will be written to.</summary>
        public string OutputPath;

        /// <summary>Specifies whether the log writer should be disabled (this will supress all logs, including summary, errors,
        ///   and warnings files).</summary>
        public bool DisableLogWriter;
        /// <summary>Specifies whether compliance log files should be generated.</summary>
        public bool WriteLogFiles;
        /// <summary>Specifies whether extended log files (a log entry for each compliance finding evaluated) should be generated.</summary>
        public bool WriteExtendedLogFiles;

        /// <summary>Specifies whether the model should generate CSV reports.</summary>
        public bool GenerateCSVReports;
        /// <summary>Specifies whether the model should generate Excel reports.</summary>
        public bool GenerateXLReports;

        #endregion

    }
}
