﻿#region << Using Directives >>
using System;
using Volpe.Cafe.Utils;
#endregion

namespace Volpe.Cafe.Settings
{
    /// <summary>
    /// Defines compliance scenario attributes for a single regulatory class.
    /// </summary>
    [Serializable]
    public class ScenarioInfo : ICloneable
    {

        #region /*** Ctors ***/

        // Private constructor used for internal usage (such as cloning).
        ScenarioInfo() { }
        /// <summary>
        /// Initializes a new instance of the <see cref="ScenarioInfo"/> class.
        /// </summary>
        internal ScenarioInfo(
            RegulatoryClass regClass,
            int[] function, double[][] coefficients, double[] minStndMpg, double[] minStndPct,
            bool[] allowFines, double[] fineRate,
            int[] multiFuel, double[] multiFuelFFVShare, double[] multiFuelPHEVShare,
            bool[] includeAC, double[] acAdjustment, double[] acCost,
            double[] offCycleCap, double[] ffvCap,
            double[] shevTaxCredit, double[] phevTaxCredit, double[] evTaxCredit,
            int[] twFunction, double[] payloadReturn, double[] towingReturn,
            int[] creditCarryFwd)
        {
            this.RegClass           = regClass;
            // CAFE standard
            this.Function           = function;
            this.Coefficients       = coefficients;
            this.MinStndMpg         = minStndMpg;
            this.MinStndPct         = minStndPct;
            // CAFE fines
            this.AllowFines         = allowFines;
            this.FineRate           = fineRate;
            // mutli-fuel applicability
            this.MultiFuel          = multiFuel;
            this.MultiFuelFFVShare  = multiFuelFFVShare;
            this.MultiFuelPHEVShare = multiFuelPHEVShare;
            // air-conditioning
            this.IncludeAC          = includeAC;
            this.ACAdjustment       = acAdjustment;
            this.ACCost             = acCost;
            // CAFE credit caps
            this.OffCycleCap        = offCycleCap;
            this.FFVCap             = ffvCap;
            // tax credits ($)
            this.SHEVTaxCredit      = shevTaxCredit;
            this.PHEVTaxCredit      = phevTaxCredit;
            this.EVTaxCredit        = evTaxCredit;
            // test weight calculations
            this.TWFunction         = twFunction;
            this.PayloadReturn      = payloadReturn;
            this.TowingReturn       = towingReturn;
            // credit trading settings
            this.CreditCarryFwd     = creditCarryFwd;
        }

        #endregion

        #region /*** Methods ***/

        #region /* ICloneable Members */

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="ScenarioInfo"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="ScenarioInfo"/>.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="ScenarioInfo"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="ScenarioInfo"/>.</returns>
        public ScenarioInfo Clone()
        {
            ScenarioInfo scen = new ScenarioInfo();
            //
            scen.RegClass           = this.RegClass;
            // CAFE standard
            scen.Function           = this.Function          .CloneArray();
            scen.Coefficients       = this.Coefficients      .CloneArray();
            scen.MinStndMpg         = this.MinStndMpg        .CloneArray();
            scen.MinStndPct         = this.MinStndPct        .CloneArray();
            // CAFE fines
            scen.AllowFines         = this.AllowFines        .CloneArray();
            scen.FineRate           = this.FineRate          .CloneArray();
            // mutli-fuel applicability
            scen.MultiFuel          = this.MultiFuel         .CloneArray();
            scen.MultiFuelFFVShare  = this.MultiFuelFFVShare .CloneArray();
            scen.MultiFuelPHEVShare = this.MultiFuelPHEVShare.CloneArray();
            // air-conditioning
            scen.IncludeAC          = this.IncludeAC         .CloneArray();
            scen.ACAdjustment       = this.ACAdjustment      .CloneArray();
            scen.ACCost             = this.ACCost            .CloneArray();
            // CAFE credit caps
            scen.OffCycleCap        = this.OffCycleCap       .CloneArray();
            scen.FFVCap             = this.FFVCap            .CloneArray();
            // tax credits ($)
            scen.SHEVTaxCredit      = this.SHEVTaxCredit     .CloneArray();
            scen.PHEVTaxCredit      = this.PHEVTaxCredit     .CloneArray();
            scen.EVTaxCredit        = this.EVTaxCredit       .CloneArray();
            // test weight calculations
            scen.TWFunction         = this.TWFunction        .CloneArray();
            scen.PayloadReturn      = this.PayloadReturn     .CloneArray();
            scen.TowingReturn       = this.TowingReturn      .CloneArray();
            // credit trading settings
            scen.CreditCarryFwd     = this.CreditCarryFwd    .CloneArray();
            //
            return scen;
        }

        #endregion

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets the <see cref="RegulatoryClass"/> for which the compliance scenario attributes are defined.</summary>
        public RegulatoryClass RegClass { get; private set; }

        //----- CAFE standard -----
        /// <summary>Gets the functional form to use for computing the vehicle target.</summary>
        public int[] Function { get; private set; }
        /// <summary>Gets the coefficients associated with the functional form to use for computing the vehicle target.</summary>
        public double[][] Coefficients { get; private set; }
        /// <summary>Gets the minimum CAFE standard that each manufacturer must attain, specified as a flat-standard in miles/gallon,
        ///   or 0 if not applicable.</summary>
        public double[] MinStndMpg { get; private set; }
        /// <summary>Gets the minimum CAFE standard that each manufacturer must attain, specified as a percentage of the average
        ///   requirement under the function-based standard, or 0 if not applicable.</summary>
        public double[] MinStndPct { get; private set; }

        //----- CAFE fines -----
        /// <summary>Gets whether manufacturers are allowed to pay CAFE fines for non-compliance instead of applying additional
        ///   technologies.</summary>
        public bool[] AllowFines { get; private set; }
        /// <summary>Gets the CAFE fine rate in $/credit, applicable whenever a manufacturer's fleet is under compliance.</summary>
        public double[] FineRate { get; private set; }

        //----- mutli-fuel applicability -----
        /// <summary>Gets the applicability of multi-fuel vehicles for compliance calculations (does not apply to single-fuel
        ///   vehicles). The following modes are available:
        ///   0: only gasoline fuel economy value is considered (gasoline fuel share is assumed to be 100%);
        ///   1: for Gasoline/Ethanol-85 vehicles, only the gasoline fuel economy value is considered (gasoline fuel share is assumed
        ///      to be 100%); for Gasoline/Electricity vehicles, both fuel economy values are considered;
        ///   2: for Gasoline/Ethanol-85 and Gasoline/Electricity vehicles, both fuel economy values are considered.</summary>
        public int[] MultiFuel { get; private set; }
        /// <summary>Gets the statutory fuel share to use for compliance for flex-fuel vehicles (FFVs), whenever the
        ///   <see cref="MultiFuel"/> mode is 2. This fuel share applies only to vehicles operating on gasoline and ethanol-85 fuel
        ///   types. If this value is 0, the vehicle's assumed on-road fuel share will be used for compliance.</summary>
        public double[] MultiFuelFFVShare { get; private set; }
        /// <summary>Gets the statutory fuel share to use for compliance for plug-in hybrid/electric vehicles (SHEVs), whenever the
        ///   <see cref="MultiFuel"/> mode is either 1 or 2. This fuel share applies only to vehicles operating on gasoline and
        ///   electricity fuel types. If this value is 0, the vehicle's assumed on-road fuel share will be used for compliance.</summary>
        public double[] MultiFuelPHEVShare { get; private set; }

        //----- air-conditioning -----
        /// <summary>Gets whether to include adjustments for improvements in air conditioning (AC adjustments) for compliance and
        ///   effects calculations.</summary>
        public bool[] IncludeAC { get; private set; }
        /// <summary>Gets the AC adjustment factor in grams/mile of CO2.</summary>
        public double[] ACAdjustment { get; private set; }
        /// <summary>Gets the cost of AC adjustment in dollars per unit.</summary>
        public double[] ACCost { get; private set; }

        //----- CAFE credit caps -----
        /// <summary>Gets the maximum amount of credit a manufacturer may accrue from Off-Cycle technology improvments, specified in
        ///   grams/mile of CO2.</summary>
        public double[] OffCycleCap { get; private set; }
        /// <summary>Gets the maximum amount of credit a manufacturer may accrue from flex-fuel vehicles (FFVs), specified in
        ///   vehcle-miles/gallon.</summary>
        public double[] FFVCap { get; private set; }

        //----- tax credits ($) -----
        /// <summary>Gets the amount of Federal Incentives (tax credits) a buyer receives for purchasing a strong hybrid/electric
        /// vehicle (SHEV).</summary>
        public double[] SHEVTaxCredit { get; private set; }
        /// <summary>Gets the amount of Federal Incentives (tax credits) a buyer receives for purchasing a plug-in hybrid/electric
        ///   vehicle (PHEV).</summary>
        public double[] PHEVTaxCredit { get; private set; }
        /// <summary>Gets the amount of Federal Incentives (tax credits) a buyer receives for purchasing a pure electric vehicle (EV).</summary>
        public double[] EVTaxCredit { get; private set; }

        //----- test weight calculations -----
        /// <summary>Gets the functional form to use for computing the vehicle's test weight.</summary>
        public int[] TWFunction { get; private set; }
        /// <summary>Gets the percentage of curb weight reduction returned to payload capacity.</summary>
        /// <remarks>The payload return value applies whenever mass reduction technology is installed to a vehicle. For example,
        ///   if payload return is 0%, the vehicle's payload capacity remains the same; if payload return is 100%, the vehicle's
        ///   reduction in curb weight goes entirely to payload.</remarks>
        public double[] PayloadReturn { get; private set; }
        /// <summary>Gets the percentage of GVWR reduction returned to towing capacity.</summary>
        /// <remarks>The towing return value applies whenever mass reduction technology is installed to a vehicle. For example,
        ///   if towing return is 0%, the vehicle’s towing capacity remains the same; if towing return is 100%, the vehicle’s
        ///   reduction in GVWR goes entirely to towing.</remarks>
        public double[] TowingReturn { get; private set; }

        //----- credit trading settings -----
        /// <summary>Gets the maximum number of years to carry forward. If a value is specified, this setting overrides the value
        ///   present in the parameters file. If this value is 0, the default value from the parameters file will be used.</summary>
        public int[] CreditCarryFwd { get; private set; }

        #endregion

    }
}
