using System;
using System.Collections.Generic;
using Volpe.Cafe.Data;
using Volpe.Cafe.Utils;
//
using TC = Volpe.Cafe.TechnologyClasses;
using TI = Volpe.Cafe.TechnologyIndexes;

namespace Volpe.Cafe.Settings
{
    /// <summary>
    /// Provides information relating to a single vehicle technology.
    /// </summary>
    [Serializable]
    public sealed class TechnologyInfo
    {

        #region /*** Constructors ***/

        /// <summary>
        /// Initializes a new instance of the Technology class.
        /// </summary>
        /// <param name="index">The name of the technology.</param>
        internal TechnologyInfo(string name)
        {
            this.Name          = name;
            this.Index         = TI.GetIndex(name);
            this.IsEngineLevel = TI.IsEngineLevel(this.Index);
            //
            this._attributes   = new TechnologyAttributes[                                             TC.ClassCount];
            this._costs        = new TechnologyCosts     [(this.IsEngineLevel) ? TC.EngineClassCount : TC.ClassCount];
        }

        #endregion

        #region /*** Methods ***/

        #region /* ICloneable Members */

        /// <summary>
        /// Creates a new object that is a copy of the current Technology instance.
        /// </summary>
        /// <returns>A new object that is a copy of this Technology.</returns>
        internal TechnologyInfo Clone()
        {
            TechnologyInfo tech = new TechnologyInfo(this.Name);
            //
            tech.PhaseInCap     = this.PhaseInCap;
            tech.OffCycleCredit = this.OffCycleCredit.Clone();
            tech.ZEVCredits     = this.ZEVCredits;
            tech._attributes    = this._attributes.CloneArray(x => x.Clone());
            tech._costs         = this._costs     .CloneArray(x => x.Clone());
            //
            return tech;
        }

        #endregion

        #region /* GetPhaseIn methods */

        /// <summary>
        /// Returns the total phase in cap value for the specified model year.
        /// </summary>
        /// <param name="yrIndex">The index of the model year currently being examined.</param>
        /// <param name="startYrIndex">The index of the model year when modeling began.</param>
        /// <returns>The total phase in cap value for the specified model year.</returns>
        public double GetPhaseIn(int yrIndex, int startYrIndex)
        {
            return Math.Min(this.PhaseInCap * Math.Max(yrIndex - startYrIndex + 1, 0), 1);
        }

        #endregion

        /// <summary>
        /// Returns the string representation of this <see cref="TechnologyInfo"/> instance.
        /// </summary>
        /// <returns>The string representation of the <see cref="TechnologyInfo"/> instance.</returns>
        public override string ToString()
        {
            return "{" + this.Index.ToString("00") + "-" + this.Name + "}";
        }

        /// <summary>
        /// Returns attributes associated with the current technology, applicable to the specified vehicle.
        /// </summary>
        /// <param name="veh">The vehicle for which to retrieve technology attributes.</param>
        /// <returns>Attributes associated with the current technology.</returns>
        public TechnologyAttributes GetAttributes(Vehicle veh)
        {
            return this._attributes[veh.TechnologyClass];
        }
        internal void SetAttributes(int techClass, TechnologyAttributes attr)
        {
            this._attributes[techClass] = attr;
        }
        /// <summary>
        /// Returns costs associated with the current technology, applicable to the specified vehicle.
        /// </summary>
        /// <param name="veh">The vehicle for which to retrieve technology costs.</param>
        /// <returns>Costs associated with the current technology.</returns>
        public TechnologyCosts GetCosts(Vehicle veh)
        {
            if (this.IsEngineLevel)
            {
                return this._costs[veh.EngineTechnologyClass];
            }
            else
            {
                return this._costs[veh.TechnologyClass];
            }
        }
        internal void SetCosts(int techClass, TechnologyCosts costs)
        {
            this._costs[techClass] = costs;
        }

        /// <summary>
        /// Scales the FC improvements and the costs of this technology for all tech classes by the specified values.
        /// </summary>
        /// <param name="fcScale">The technology FC improvement scale.</param>
        /// <param name="costScale">The technology cost scale.</param>
        public void ScaleForMonteCarlo(double fcScale, double costScale)
        {
            // scale FC
            for (int i = 0; i < this._attributes.Length; i++)
            {
                this._attributes[i].FC *= fcScale;
            }
            // scale costs
            for (int i = 0; i < this._costs.Length; i++)
            {
                for (int j = 0; j < this._costs[i].CostTable.Length; j++)
                {
                    this._costs[i].CostTable[j] *= costScale;
                }
            }
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets the name of the technology.</summary>
        public string Name { get; private set; }
        /// <summary>Gets the index of the technology.</summary>
        public int Index { get; private set; }
        /// <summary>Gets whether the technology is engine-level.</summary>
        public bool IsEngineLevel { get; private set; }

        #endregion

        #region /*** Variables ***/

        /// <summary>Specifies the percentage of the entire fleet to which the current technology can be applied, per model year.</summary>
        public double PhaseInCap;
        /// <summary>Specifies the off-cycle credit that a vehicle inccurs as a result of applying the current technology.</summary>
        public RCDouble OffCycleCredit;
        /// <summary>Specifies the amount of ZEV credits that a vehicle generates as a result of applying the current technology.</summary>
        public double ZEVCredits;

        /// <summary>Provides attributes associated with the current technology for each technology class.</summary>
        TechnologyAttributes[] _attributes;
        /// <summary>Provides costs associated with the current technology for each technology class.</summary>
        TechnologyCosts[] _costs;

        #endregion

    }
}
