#region << Using Directives >>
using System;
using System.Collections;
using System.IO;
using System.Threading;
using Volpe.Cafe;
using Volpe.Cafe.Collections;
using Volpe.Cafe.Data;
using Volpe.Cafe.Data.MonteCarlo;
using Volpe.Cafe.Data.Optimization;
using Volpe.Cafe.IO;
using Volpe.Cafe.Model;
using Volpe.Cafe.Settings;
using Volpe.Ui;
#endregion
namespace Volpe.Cafe.Model
{
    [Serializable]
    [ModelDescription("ComplianceBase", "Provides an abstract base class for the compliance model implementation.", 1.0F)]
    public abstract class ComplianceBase : ICompliance
    {
        #region 
        #region 
        public event PromptEventHandler Prompt;
        public event ModelingEventHandler ModelingStarted;
        public event ModelingEventHandler ModelingStopped;
        public event ModelingEventHandler ModelingCompleted;
        public event ModelingEventHandler ModelingChanged;
        #endregion
        #endregion
        #region 
        protected ComplianceBase()
        {
            this._state = ModelingState.Unstarted;
            this._abortRequested = false;
        }
        ~ComplianceBase()
        {
            this.Abort(false);
            this.ModelingStarted   = null;
            this.ModelingStopped   = null;
            this.ModelingCompleted = null;
            this.ModelingChanged   = null;
        }
        #endregion
        #region 
        #region 
        protected virtual void OnPrompt(PromptEventArgs e)
        {
            if (this.Prompt != null)
            {
                this.Prompt(this, e);
            }
        }
        protected virtual void OnModelingStarted()
        {
            if (this.ModelingStarted != null)
            {
                this.ModelingStarted(this, new ModelingEventArgs(this._state));
            }
        }
        protected virtual void OnModelingStopped()
        {
            if (this.ModelingStopped != null)
            {
                this.ModelingStopped(this, new ModelingEventArgs(this._state));
            }
        }
        protected virtual void OnModelingCompleted()
        {
            if (this.ModelingCompleted != null)
            {
                this.ModelingCompleted(this, new ModelingEventArgs(this._state));
            }
        }
        protected virtual void OnModelingChanged()
        {
            if (this.ModelingChanged != null)
            {
                this.ModelingChanged(this, new ModelingEventArgs(this._state));
            }
        }
        #endregion
        #region 
        public abstract ICompliance CreateNew();
        public virtual void Start(Industry data, ModelingSettings settings, LogWriter logWriter)
        {
            if (this.Running || this._state == ModelingState.StartRequested)
            {   
                string errString = "The compliance modeling process has already started at: " +
                    this._startTime.ToString("M/dd/yyyy h:mm:ss tt") + " -- exiting ..."; 
                this._logWriter.WriteErrorLine(errString);
                this.OnPrompt(new PromptEventArgs("CAFE Model Error", errString, PromptOption.Ok));
                return;
            }
            this._startTime = new DateTime(0);
            this.UpdateState(ModelingState.StartRequested);     
            this._data      = data;
            this._settings  = settings;
            this._logWriter = (logWriter == null) ? LogWriter.Empty : logWriter;
            if (this._settings != null)
            {
                ScenarioCollection scenarios = this._settings.Scenarios;
                if (scenarios != null)
                {
                    this._scenCount = scenarios.Count;
                }
                if (this._settings.ParametersOverrides.OverrideFuelPriceEstimates)
                {
                    this._settings.OperatingModes.FuelPriceEstimates = this._settings.ParametersOverrides.FuelPriceEstimates;
                }
                if (this._settings.ParametersOverrides.OverrideCO2Estimates)
                {
                    this._settings.OperatingModes.CO2Estimates = this._settings.ParametersOverrides.CO2Estimates;
                }
                if (this._settings.Parameters != null)
                {
                    this._settings.Parameters.UpdateParametersOverrides(this._settings.ParametersOverrides);
                }
            }
            if (this._data != null)
            {
                this._minYear      = this._data.MinYear.Year;
                this._maxYear      = this._data.MaxYear.Year;
                this._minYearIndex = this._data.MinYear.Index;
                this._maxYearIndex = this._data.MaxYear.Index;
                this._yearCount    = this._maxYear - this._minYear + 1;
                this._mfrCount     = this._data.ManufacturerCount;
                this._modelYears   = new ModelYear[this._yearCount];
                for (int i = this._minYear; i <= this._maxYear; i++)
                {
                    this._modelYears[i - this._minYear] = new ModelYear(i);
                }
            }
            this.Validate();
            if (!this._valid)
            {   
                string errString = "The compliance model is not valid -- exiting!\n\nTry modifying the modeling settings " +
                    "or restarting the model.";
                this._logWriter.WriteErrorLine(errString);
                this.OnPrompt(new PromptEventArgs("CAFE Model Error", errString, PromptOption.Ok));
                this.UpdateState(ModelingState.Stopped);        
                return;
            }
            this.Start();
        }
        public virtual void Abort()
        {
            this.Abort(false);
        }
        public virtual void Abort(bool abortWhenReady)
        {
            if (this._state == ModelingState.Running || (!abortWhenReady && this._state == ModelingState.StopRequested))
            {   
                this.UpdateState(ModelingState.StopRequested);
                this._abortRequested = true;
                this.AbortInternal(abortWhenReady);
            }
        }
        public abstract Industry GetData(int scenIndex, int yearIndex);
        #endregion
        protected virtual void Validate()
        {
            this._valid = true;
            string message = "";
            if (this.RequiresMarketData)
            {
                if (this._data == null)
                {
                    message += "\n    The modeling data was not specified.";
                }
                else if (this._data.ManufacturerCount == 0 || this._data.VehicleCount == 0)
                {
                    message += "\n    The specified modeling data does not contain any valid manufacturers or vehicles.";
                }
            }
            if (this._settings == null)
            {
                message += "\n    The modeling settings were not specified.";
            }
            else
            {
                string[] missingSets = new string[4];
                int missingCntr = 0;
                if (this.RequiresTechnologies && (this._settings.Technologies  == null ||
                    this._settings.Technologies.Count == 0))                          { missingSets[missingCntr++] = "Technologies";    }
                if (this.RequiresParameters && this._settings.Parameters     == null) { missingSets[missingCntr++] = "Parameters";      }
                if (this.RequiresEmissions  && this._settings.EmissionsRates == null) { missingSets[missingCntr++] = "Emissions Rates"; }
                if (this.RequiresScenarios  && (this._settings.Scenarios     == null ||
                    this._settings.Scenarios.Count == 0))                             { missingSets[missingCntr++] = "Scenarios";       }
                if (missingCntr > 0)
                {
                    string messageSets = "";
                    for (int i = 0; i < missingCntr - 1; i++)
                    {
                        messageSets += missingSets[i] + ", ";
                    }
                    if (missingCntr == 1) { messageSets = missingSets[0]; }
                    else
                    {
                        if (missingCntr == 2) { messageSets = messageSets.Remove(messageSets.Length - 2, 2) + " "; }
                        messageSets += "and " + missingSets[missingCntr - 1];
                    }
                    message += "\n    The " + messageSets + " are missing from the modeling settings.";
                }
            }
            if (message != "")
            {
                this._valid = false;
                message = "The following errors have occured during initialization:\n" + message +
                    "\n\nThe model will not run.";
                this._logWriter.WriteErrorLine(new string('-', 25));
                this._logWriter.WriteErrorLine(message);
                this._logWriter.WriteErrorLine(new string('-', 25));
                this.OnPrompt(new PromptEventArgs("CAFE Model Error", message, PromptOption.Ok));
            }
        }
        void Start()
        {
            if (!this._logWriter.IsOpen)
            {
                this._logWriter.Open(false, this._scenCount, this._mfrCount, this._minYearIndex, this._maxYearIndex,
                    (this._mfrCount == 0) ? null : this._data.Manufacturers);
            }
            this._complianceThread = new Thread(new ThreadStart(this.StartMainComplianceThread));
            this._complianceThread.Name = "MainComplianceThread";
            this._startTime = DateTime.Now;
            this._abortRequested = false;
            this.UpdateState(ModelingState.Running);                            
            this._complianceThread.Start();
        }
        void StartMainComplianceThread()
        {
            ModelingState endState = ModelingState.Unstarted;   
            string title = "<UNNAMED-MODEL>";
            try
            {   
                title = ((ModelDescriptionAttribute)
                    Attribute.GetCustomAttribute(this.GetType(), typeof(ModelDescriptionAttribute))).Title;
                this._logWriter.WriteSummaryLine(title + " started at " + DateTime.Now + ".");
                this.StartInternal();
                endState = (this._abortRequested) ? ModelingState.Stopped : ModelingState.Completed;
            }
            catch (Exception ex)
            {   
                this._logWriter.WriteErrorLine(ex.ToString());
                endState = ModelingState.Stopped;
            }
            finally
            {
                this._logWriter.WriteSummaryLine(new string('=', 50));
                this._logWriter.WriteSummaryLine("---------- User Settings ----------");
                this._logWriter.WriteSummaryLine("  Machine Name: " + System.Environment.MachineName);
                this._logWriter.WriteSummaryLine("  OS Version:   " + System.Environment.OSVersion);
                this._logWriter.WriteSummaryLine("  CLR Version:  " + System.Environment.Version.ToString());
                this._logWriter.WriteSummaryLine("  CAFE Version: " + Global.ApplicationTimestamp);
                this._logWriter.WriteSummaryLine("  User Name:    " + System.Environment.UserName);
                this._logWriter.WriteSummaryLine(new string('=', 50));
                if (this._settings != null)
                {
                    this._settings.WriteSummaryLog(this._logWriter);
                }
                if (this._data != null)
                {
                    this._data.WriteSummaryLog(this._logWriter);
                }
                this._logWriter.WriteSummaryLine(new string('=', 50));
                this._logWriter.WriteSummaryLine(title + " " + endState + " at " + DateTime.Now + " in " +
                    (this.Runtime / 1000) + " sec.");
                this.UpdateState(endState);
            }
        }
        protected abstract void StartInternal();
        protected virtual void AbortInternal(bool abortWhenReady)
        {
            if (!abortWhenReady)
            {   
                try
                {
                    if (this._complianceThread != null && this._complianceThread.IsAlive)
                    {
                        this._complianceThread.Abort();
                        this._complianceThread = null;
                    }
                }
                catch (Exception ex)
                {
                    this._logWriter.WriteErrorLine("Failed to terminate thread: " + ((this._complianceThread == null) ? "null" :
                        this._complianceThread.Name) + "\nException: " + ex.Message);
                }
                this.UpdateState(ModelingState.Stopped);
            }
        }
        protected void UpdateState(ModelingState newState)
        {
            this._state = newState;
            if (this.StoppedOrCompleted)
            {   
                if (this._startTime.Ticks != 0) { this._stopTime = DateTime.Now; }
                this._logWriter.Close();
            }
            if      (this._state == ModelingState.Running  ) { this.OnModelingStarted();   }
            else if (this._state == ModelingState.Stopped  ) { this.OnModelingStopped();   }
            else if (this._state == ModelingState.Completed) { this.OnModelingCompleted(); }
            this.OnModelingChanged();
        }
        #endregion
        #region 
        #region 
        public bool IsValid { get { return this._valid; } }
        public virtual IEISModel EISModel { get { return null; } }
        public virtual bool IsOptimization { get { return false; } }
        public virtual bool IsMonteCarlo { get { return false; } }
        public virtual OptimizationData OptimizationData { get { return null; } }
        public virtual MonteCarloData   MonteCarloData   { get { return null; } }
        public LogWriter LogWriter { get { return this._logWriter; } }
        public Industry Data { get { return this._data; } }
        public ModelingSettings Settings { get { return this._settings; } }
        public ModelingState State { get { return this._state; } }
        public bool Running   { get { return (this._state == ModelingState.Running || this._state == ModelingState.StopRequested); } }
        public bool Stopped   { get { return (this._state == ModelingState.Stopped); } }
        public bool Completed { get { return (this._state == ModelingState.Completed); } }
        public bool StoppedOrCompleted { get { return (this._state == ModelingState.Stopped || this._state == ModelingState.Completed); } }
        public abstract IModelingProgress Progress { get; }
        public DateTime StartTime { get { return this._startTime; } }
        public DateTime StopTime  { get { return (this.StoppedOrCompleted) ? DateTime.MinValue : this._stopTime; } }
        public long Runtime
        {
            get
            {
                return (this._startTime.Ticks == 0) ? 0 :
                    (((this.StoppedOrCompleted) ? this._stopTime.Ticks : DateTime.Now.Ticks) - this._startTime.Ticks) / 10000;
            }
        }
        public virtual bool RequiresMarketData   { get { return true; } }
        public virtual bool RequiresTechnologies { get { return true; } }
        public virtual bool RequiresParameters   { get { return true; } }
        public virtual bool RequiresEmissions    { get { return true; } }
        public virtual bool RequiresScenarios    { get { return true; } }
        #endregion
        #endregion
        #region 
        protected bool _valid;
        protected LogWriter _logWriter;
        protected Industry _data;
        protected ModelingSettings _settings;
        ModelingState _state;
        DateTime _startTime;
        DateTime _stopTime;
        [NonSerialized] protected Thread _complianceThread;
        [NonSerialized] protected bool   _abortRequested;
        protected int _scenCount;
        protected int _yearCount;
        protected int _mfrCount;
        protected int _minYear;
        protected int _maxYear;
        protected int _minYearIndex;
        protected int _maxYearIndex;
        protected ModelYear[] _modelYears;
        #endregion
    }
}

