#region << Using Directives >>
using System;
using System.ComponentModel;
using System.Drawing;
using System.IO;
using System.Runtime.Serialization.Formatters.Binary;
using System.Text;
using System.Threading;
using System.Windows.Forms;
using Volpe.Cafe;
using Volpe.Cafe.Data;
using Volpe.Cafe.Data.Optimization;
using Volpe.Cafe.IO;
using Volpe.Cafe.Model;
using Volpe.Cafe.Model.MonteCarlo;
using Volpe.Cafe.Model.Optimization;
using Volpe.Cafe.Reports;
using Volpe.Cafe.Settings;
using Volpe.Ui;
using Volpe.Utils;
#endregion
namespace Volpe.Cafe.Ui
{
	public class Session : System.Windows.Forms.Form
	{
        #region 
        sealed class SessionMessages
        {
            public const string CAFEModelError = "CAFE Model Error";
            public const string CAFEModel      = "CAFE Model";
            public const string OPEN_FileNotFound = "Error opening session file.\n\nThe specified file cannot be found on disk.";
            public const string OPEN_SessionNotValid = "Error opening session file.\n\nThe specified file may not represent a valid session, or the session format is no longer supported.";
            public const string OPEN_STATUS_LoadingSession = "Loading session ...";
            public const string OPEN_STATUS_SessionLoadingFailed = "Session loading failed.";
            public const string OPEN_STATUS_SessionLoaded = "Session loaded.";
            public const string SAVE_SessionRunning = "The session is currently running.\n\nPlease wait for modeling to complete, or stop the session before saving again.";
            public const string SAVE_STATUS_SavingSession = "Saving session ...";
            public const string SAVE_STATUS_SessionSaved = "Session saved.";
            public const string CLOSE_SessionRunning = "The session is currently running.\n\nPlease wait for modeling to complete, or stop the session before closing.";
            public const string CLOSE_SessionNotSaved = "The session has not been saved yet.\n\nWould you like to save the changes before closing?";
            public const string START_SessionRunning = "The session is already running.";
            public const string START_NoComplianceModel = "The session cannot be run since no valid compliance model has been selected.\n\nPlease open the Modeling Settings window to select a compliance model you wish to run.";
            public const string STOP_SessionNotRunning = "The session is not currently running.";
            public const string REPORTS_SessionReporting = "The reports for the session are already being generated.";
            public const string REPORTS_ComplianceModelNotRun = "The reports for the session cannot be generated because the compliance model was not run or did not complete successfully.";
            public const string REPORTS_Error = "Error generating reports.\n\nOne or more of the modeling reports could not be generated.  Please restart the application and try again.";
            public const string REPORTS_STATUS_Starting = "Preparing to generate reports ...";
            public const string REPORTS_STATUS_GeneratingReports = "Generating Reports ...";
            public const string REPORTS_STATUS_Stopped = "Reporting Stopped.";
            public const string REPORTS_STATUS_Completed = "Reporting Completed!";
            public const string REPORTS_SessionNotReporting = "The reports for the session are not currently being generated.";
            public const string REPORTS_STATUS_CancelReports = "Canceling Reports (this may take some time) ...";
            public const string COMPLIANCE_STATUS_StartRequested = "Starting the Modeling Process ...";
            public const string COMPLIANCE_STATUS_Running        = "Model Running ...";
            public const string COMPLIANCE_STATUS_StopRequested  = "Stopping the Modeling Process ...";
            public const string COMPLIANCE_STATUS_Stopped        = "Modeling Stopped.";
            public const string COMPLIANCE_STATUS_Completed      = "Modeling Completed!";
            public const string COMPLIANCE_STATUS_Unstarted      = "The Modeling Process has not Started.";
        }
        #endregion
        #region 
        public Session(Cafe2d parent)
        {
            this.InitializeComponent();
            this._parent       = parent;
            this._index        = SessionCount++;
            this._compressor   = new ZipUtilities(ZipFormat.Zip);
            this._saveRequired = true;
            this.Text          = this.SessionName;
            this._settings     = new ModelingSettings(Cafe2d.DefaultDataPath, Cafe2d.DefaultOutputPath);
        }
        public Session(Cafe2d parent, string path)
            : this(parent)
        {
            this._path         = path;
            this._saveRequired = false;
            this.OpenSession();
        }
        protected override void Dispose(bool disposing)
        {
            base.Dispose(disposing);
        }
        #endregion
        #region 
        #region 
        void InitializeComponent()
        {
            System.Resources.ResourceManager resources = new System.Resources.ResourceManager(typeof(Session));
            this.ux_splitter = new System.Windows.Forms.Splitter();
            this.ux_progress = new System.Windows.Forms.RichTextBox();
            this.ux_statusPanel = new System.Windows.Forms.Panel();
            this.ux_additionalInfo = new System.Windows.Forms.RichTextBox();
            this.ux_runtime = new System.Windows.Forms.Label();
            this.ux_statusPanel.SuspendLayout();
            this.SuspendLayout();
            this.ux_splitter.BackColor = System.Drawing.SystemColors.HotTrack;
            this.ux_splitter.Dock = System.Windows.Forms.DockStyle.Bottom;
            this.ux_splitter.Location = new System.Drawing.Point(0, 209);
            this.ux_splitter.Name = "ux_splitter";
            this.ux_splitter.Size = new System.Drawing.Size(472, 4);
            this.ux_splitter.TabIndex = 5;
            this.ux_splitter.TabStop = false;
            this.ux_progress.BackColor = System.Drawing.SystemColors.Window;
            this.ux_progress.DetectUrls = false;
            this.ux_progress.Dock = System.Windows.Forms.DockStyle.Fill;
            this.ux_progress.Font = new System.Drawing.Font("Lucida Console", 9F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
            this.ux_progress.Location = new System.Drawing.Point(0, 0);
            this.ux_progress.Name = "ux_progress";
            this.ux_progress.ReadOnly = true;
            this.ux_progress.Size = new System.Drawing.Size(472, 213);
            this.ux_progress.TabIndex = 1;
            this.ux_progress.Text = "Ready";
            this.ux_progress.WordWrap = false;
            this.ux_statusPanel.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
            this.ux_statusPanel.Controls.Add(this.ux_additionalInfo);
            this.ux_statusPanel.Controls.Add(this.ux_runtime);
            this.ux_statusPanel.Dock = System.Windows.Forms.DockStyle.Bottom;
            this.ux_statusPanel.Location = new System.Drawing.Point(0, 213);
            this.ux_statusPanel.Name = "ux_statusPanel";
            this.ux_statusPanel.Size = new System.Drawing.Size(472, 80);
            this.ux_statusPanel.TabIndex = 2;
            this.ux_additionalInfo.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
                | System.Windows.Forms.AnchorStyles.Left) 
                | System.Windows.Forms.AnchorStyles.Right)));
            this.ux_additionalInfo.BackColor = System.Drawing.SystemColors.Control;
            this.ux_additionalInfo.BorderStyle = System.Windows.Forms.BorderStyle.None;
            this.ux_additionalInfo.Location = new System.Drawing.Point(8, 32);
            this.ux_additionalInfo.Name = "ux_additionalInfo";
            this.ux_additionalInfo.ReadOnly = true;
            this.ux_additionalInfo.Size = new System.Drawing.Size(456, 40);
            this.ux_additionalInfo.TabIndex = 1;
            this.ux_additionalInfo.TabStop = false;
            this.ux_additionalInfo.Text = "";
            this.ux_additionalInfo.WordWrap = false;
            this.ux_runtime.AutoSize = true;
            this.ux_runtime.Location = new System.Drawing.Point(8, 8);
            this.ux_runtime.Name = "ux_runtime";
            this.ux_runtime.Size = new System.Drawing.Size(59, 17);
            this.ux_runtime.TabIndex = 0;
            this.ux_runtime.Text = "Runtime: 0";
            this.AutoScaleBaseSize = new System.Drawing.Size(5, 14);
            this.ClientSize = new System.Drawing.Size(472, 293);
            this.Controls.Add(this.ux_splitter);
            this.Controls.Add(this.ux_progress);
            this.Controls.Add(this.ux_statusPanel);
            this.Font = new System.Drawing.Font("Tahoma", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
            this.Icon = ((System.Drawing.Icon)(resources.GetObject("$this.Icon")));
            this.Name = "Session";
            this.Text = "Session";
            this.ux_statusPanel.ResumeLayout(false);
            this.ResumeLayout(false);
        }
        #endregion
        #region 
        protected override void OnSizeChanged(EventArgs e)
        {
            float zoom = this.Width / 400.0F;
            if (zoom < 0.75F) { zoom = 0.75F; } else if (zoom > 3.0F ) { zoom = 3.0F ; }
            if (this.ux_progress.ZoomFactor != zoom) { this.ux_progress.ZoomFactor = zoom; }
            base.OnSizeChanged(e);
        }
        protected override void OnClosing(CancelEventArgs e)
        {
            if (this.RunningOrReporting)
            {   
                MessageBox.Show(SessionMessages.CLOSE_SessionRunning, SessionMessages.CAFEModel, MessageBoxButtons.OK,
                    MessageBoxIcon.Information);
                e.Cancel = true;
                return;
            }
            if (this._saveRequired)
            {   
                DialogResult dr = MessageBox.Show(SessionMessages.CLOSE_SessionNotSaved, SessionMessages.CAFEModel,
                    MessageBoxButtons.YesNoCancel, MessageBoxIcon.Question, MessageBoxDefaultButton.Button3);
                if (dr == DialogResult.Cancel)
                {   
                    e.Cancel = true;
                }
                else
                {   
                    e.Cancel = false;
                    if (dr == DialogResult.Yes)
                    {   
                        this.SaveSession();
                    }
                    this._compliance = null;
                    this._data       = null;
                    this._settings   = null;
                    GC.Collect();
                    GC.WaitForPendingFinalizers();
                    GC.Collect();
                }
            }
            base.OnClosing(e);
        }
        #endregion
        void OpenSession()
        {
            if (!File.Exists(this._path)) { throw new IOException(SessionMessages.OPEN_FileNotFound); }
            this.UpdateStatus(SessionMessages.OPEN_STATUS_LoadingSession);
            BinaryFormatter bf = new BinaryFormatter();
            Stream[] streams = null;
            try
            {   
                this._compressor.Decompress(this._path, out streams, true, null);
                streams[0].Seek(0, SeekOrigin.Begin);
                object[] sessionData = (object[])bf.Deserialize(streams[0]);
                this._compliance      = (ICompliance     )sessionData[0];
                this._data            = (Industry        )sessionData[1];
                this._settings        = (ModelingSettings)sessionData[2];
                this._refreshInterval = (int             )sessionData[3];
            }
            catch (Exception ex)
            {   
                this.UpdateStatus(SessionMessages.OPEN_STATUS_SessionLoadingFailed);
                Console.WriteLine(ex.ToString());
                throw new Exception(SessionMessages.OPEN_SessionNotValid);
            }
            finally
            {
                if (streams != null)
                {   
                    for (int i = 0; i < streams.Length; i++)
                    {
                        if (streams[i] != null) { streams[i].Close(); }
                    }
                }
            }
            this.Text = this.SessionName;
            this.UpdateStatus(SessionMessages.OPEN_STATUS_SessionLoaded);
            this.RefreshSession();
        }
        public void CloseSession()
        {
            if (!this.RunningOrReporting) { base.Close(); }
        }
        public bool SaveSession()
        {
            return this.SaveSession_Internal(false, null);
        }
        public bool SaveSession(string name)
        {
            if (name != null && name.Trim() != string.Empty && name != this.SessionName)
            {
                this._saveRequired = true;
            }
            return this.SaveSession_Internal(false, name);
        }
        public bool SaveSessionAs()
        {
            return this.SaveSession_Internal(true, null);
        }
        bool SaveSession_Internal(bool displaySaveAs, string name)
        {
            if (!displaySaveAs && !this._saveRequired) { return false; }
            if (this.RunningOrReporting)
            {
                MessageBox.Show(SessionMessages.SAVE_SessionRunning, SessionMessages.CAFEModel, MessageBoxButtons.OK,
                    MessageBoxIcon.Information);
                return false;
            }
            bool silent = false;
            if (name != null)
            {
                name = name.Trim();
                if (File.Exists(this._path) && name != string.Empty)
                {
                    this._path = Path.GetDirectoryName(this._path) + "\\" + name + ".cmsd";
                    silent = true;
                }
            }
            if (displaySaveAs || this._path == null || (!silent && (!File.Exists(this._path) ||
                (File.GetAttributes(this._path) & FileAttributes.ReadOnly) == FileAttributes.ReadOnly)))
            {
                SaveFileDialog dlg = new SaveFileDialog();
                dlg.FileName = File.Exists(this._path) ? this._path : (name != null) ? name + ".cmsd" : this.SessionName + ".cmsd";
                dlg.Filter = "CAFE Model Session Data (*.cmsd)|*.cmsd";
                dlg.Title = "Please specify the location where you would like to save the '" +
                    ((name != null) ? name : this.SessionName) + "' session:";
                if (dlg.ShowDialog() == DialogResult.OK)
                {   
                    this._path = dlg.FileName;
                }
                else { return false; }  
            }
            this.UpdateStatus(SessionMessages.SAVE_STATUS_SavingSession);
            object[] sessionData = new object[4];
            sessionData[0] = (this._compliance == null) ? null : this._compliance.CreateNew();
            sessionData[1] = this._data;
            sessionData[2] = this._settings;
            sessionData[3] = this._refreshInterval;
            MemoryStream stream = new MemoryStream();
            BinaryFormatter bf = new BinaryFormatter();
            bf.Serialize(stream, sessionData);
            stream.Seek(0, SeekOrigin.Begin);
            this._compressor.Compress(new Stream[] {stream}, new string[] {"cmsd"}, this._path);
            stream.Close();
            this.Text = this.SessionName;
            this.SaveRequired = false;
            this.UpdateStatus(SessionMessages.SAVE_STATUS_SessionSaved);
            this.RefreshSession();
            return true;
        }
        public void StartModeling()
        {
            if (this.RunningOrReporting)
            {
                MessageBox.Show(SessionMessages.START_SessionRunning, SessionMessages.CAFEModel, MessageBoxButtons.OK,
                    MessageBoxIcon.Information);
                return;
            }
            if (this._compliance == null)
            {
                MessageBox.Show(SessionMessages.START_NoComplianceModel, SessionMessages.CAFEModelError, MessageBoxButtons.OK,
                    MessageBoxIcon.Exclamation);
                return;
            }
            this._stopRequested = false;
            string    logPath   = this._settings.OutputSettings.OutputPath + "\\" + this.SessionName + "\\logs";
            Directory.CreateDirectory(logPath);
            LogWriter logWriter = new LogWriter(logPath, this._settings.OperatingModes.GenerateExtendedLogFiles);
            this._compliance.ModelingChanged += new ModelingEventHandler(this.Compliance_ModelingChanged);
            this._compliance.Prompt          += new PromptEventHandler  (this.Compliance_Prompt);
            Industry         complianceData     = (this._data     == null) ? null : this._data.Clone();
            ModelingSettings complianceSettings = (this._settings == null) ? null : this._settings.Clone();
            this._compliance.Start(complianceData, complianceSettings, logWriter);
            this.RefreshSession();
            this.InitRefreshSessionThread();
        }
        public void StopModeling()
        {
            if (!this.Running)
            {
                MessageBox.Show(SessionMessages.STOP_SessionNotRunning, SessionMessages.CAFEModel, MessageBoxButtons.OK,
                    MessageBoxIcon.Information);
                return;
            }
            if (this._stopRequested)
            {
                this._compliance.Abort(false);
                this._stopRequested = false;
            }
            else
            {
                this._compliance.Abort(true);
                this._stopRequested = true;
                this.InitWaitToStopThread();
            }
        }
        public void GenerateReports()
        {
            if (this.Reporting)
            {
                MessageBox.Show(SessionMessages.REPORTS_SessionReporting, SessionMessages.CAFEModel, MessageBoxButtons.OK,
                    MessageBoxIcon.Information);
                return;
            }
            if (this._compliance == null || !this._compliance.Completed)
            {
                MessageBox.Show(SessionMessages.REPORTS_ComplianceModelNotRun, SessionMessages.CAFEModelError,
                    MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                return;
            }
            this.UpdateStatus(SessionMessages.REPORTS_STATUS_Starting);
            try
            {
                ModelingSettings settings     = this._settings.Clone();
                string           path         = settings.OutputSettings.OutputPath + "\\" + this.SessionName + "\\reports";
                string           templatePath = Cafe2d.DefaultDataPath + "\\template.xls";
                bool             isOpt      = (this._compliance is Optimization);
                OptimizationData optData    =  this._compliance.OptimizationData;
                IterationsData[] optIndData = (optData != null) ? optData.IndustryData : null;
                this._reporter = new Reporter(this._compliance, path, templatePath, this._compliance.Data, settings, isOpt,
                    (isOpt ? 1 : -1), optIndData);
                this._reporter.ReportingCompleted += new EventHandler(this.Reporter_ReportingCompleted);
                this._reporter.ReportingStopped   += new EventHandler(this.Reporter_ReportingStopped);
                this._reporter.ReportingProgress  += new EventHandler(this.Reporter_ReportingProgress);
                this._reporter.GenerateReports(true);
                this.UpdateStatus(SessionMessages.REPORTS_STATUS_GeneratingReports);
                this.RefreshSession();
            }
            catch
            {
                if (this._reporter != null)
                {   
                    this._reporter.ReportingCompleted -= new EventHandler(this.Reporter_ReportingCompleted);
                    this._reporter.ReportingStopped   -= new EventHandler(this.Reporter_ReportingStopped);
                    this._reporter.ReportingProgress  -= new EventHandler(this.Reporter_ReportingProgress);
                    this._reporter = null;
                }
                MessageBox.Show(SessionMessages.CAFEModelError, SessionMessages.REPORTS_Error, MessageBoxButtons.OK,
                    MessageBoxIcon.Error);
            }
        }
        public void CancelReports()
        {
            if (!this.Reporting)
            {
                MessageBox.Show(SessionMessages.REPORTS_SessionNotReporting, SessionMessages.CAFEModel, MessageBoxButtons.OK,
                    MessageBoxIcon.Information);
                return;
            }
            this._reporter.CancelReports();
            this.UpdateStatus(SessionMessages.REPORTS_STATUS_CancelReports);
        }
        void Compliance_ModelingChanged(object sender, ModelingEventArgs e)
        {
            ModelingState state = e.State;
            string status = string.Empty;
            if      (state == ModelingState.StartRequested) { status = SessionMessages.COMPLIANCE_STATUS_StartRequested; }
            else if (state == ModelingState.Running       ) { status = SessionMessages.COMPLIANCE_STATUS_Running;        }
            else if (state == ModelingState.StopRequested ) { status = SessionMessages.COMPLIANCE_STATUS_StopRequested;  }
            else if (state == ModelingState.Stopped       ) { status = SessionMessages.COMPLIANCE_STATUS_Stopped;        }
            else if (state == ModelingState.Completed     ) { status = SessionMessages.COMPLIANCE_STATUS_Completed;      }
            else if (state == ModelingState.Unstarted     ) { status = SessionMessages.COMPLIANCE_STATUS_Unstarted;      }
            if (state == ModelingState.Stopped || state == ModelingState.Completed)
            {   
                this._compliance.ModelingChanged -= new ModelingEventHandler(this.Compliance_ModelingChanged);
                this._compliance.Prompt          -= new PromptEventHandler  (this.Compliance_Prompt);
            }
            this.UpdateStatus(status);
            this.RefreshSession();
            if (state == ModelingState.Completed && this._compliance.Settings.OutputSettings.GenerateReports &&
                !this._compliance.IsMonteCarlo)
            {   
                this.GenerateReports();
            }
        }
        void Compliance_Prompt(object sender, PromptEventArgs e)
        {
            MessageBoxButtons buttons =
                (e.PromptOption == PromptOption.OkCancel   ) ? MessageBoxButtons.OKCancel    :
                (e.PromptOption == PromptOption.YesNoCancel) ? MessageBoxButtons.YesNoCancel :
                (e.PromptOption == PromptOption.YesNo      ) ? MessageBoxButtons.YesNo       : MessageBoxButtons.OK;
            MessageBoxDefaultButton defaultButton =
                (e.PromptOption == PromptOption.YesNoCancel) ? MessageBoxDefaultButton.Button3 :
                (e.PromptOption == PromptOption.YesNo ||
                 e.PromptOption == PromptOption.OkCancel   ) ? MessageBoxDefaultButton.Button2 : MessageBoxDefaultButton.Button1;
            DialogResult dr =  MessageBox.Show(e.Message, e.Title, buttons, MessageBoxIcon.Information, defaultButton);
            e.PromptResult =
                (dr == DialogResult.OK    ) ? PromptResult.Ok     :
                (dr == DialogResult.Cancel) ? PromptResult.Cancel :
                (dr == DialogResult.Yes   ) ? PromptResult.Yes    :
                (dr == DialogResult.No    ) ? PromptResult.No     : PromptResult.None;
        }
        void Reporter_ReportingStoppedOrCompleted(string status)
        {
            this._reporter.ReportingCompleted -= new EventHandler(this.Reporter_ReportingCompleted);
            this._reporter.ReportingStopped   -= new EventHandler(this.Reporter_ReportingStopped  );
            this._reporter.ReportingProgress  -= new EventHandler(this.Reporter_ReportingProgress );
            this._reporter = null;
            this.UpdateStatus(status);
            this.RefreshSession();
        }
        void Reporter_ReportingCompleted(object sender, EventArgs e)
        {
            this.Reporter_ReportingStoppedOrCompleted("Reporting Completed!");
        }
        void Reporter_ReportingStopped(object sender, EventArgs e)
        {
            this.Reporter_ReportingStoppedOrCompleted("Reporting Stopped.");
        }
        void Reporter_ReportingProgress(object sender, EventArgs e)
        {
            this.UpdateSessionProgress();
        }
        void InitRefreshSessionThread()
        {
            Thread refreshThread = new Thread(new ThreadStart(this.RefreshSessionThread));
            refreshThread.Name = "Session-" + this.SessionName + "::SessionRefreshThread";
            refreshThread.Start();
        }
        void RefreshSessionThread()
        {
            while (this.Running)
            {
                for (int i = 0; i < this._refreshInterval / Session.MinRefreshInterval; i++)
                {
                    Thread.Sleep(Session.MinRefreshInterval);
                    if (!this.Running) { break; }
                }
                if (this._refreshInterval == -1) { Thread.Sleep(500); } else { this.RefreshSession(); }
            }
        }
        void InitWaitToStopThread()
        {
            Thread waitToStopThread = new Thread(new ThreadStart(this.WaitToStopThread));
            waitToStopThread.Name = "Session-" + this.SessionName + "::WaitToStopThread";
            waitToStopThread.Start();
        }
        void WaitToStopThread()
        {
            for (int i = 0; i < 10; i++)
            {
                Thread.Sleep(500);
                if (!this._stopRequested || !this.Running) { return; }
            }
            this._stopRequested = false;
        }
        public void RefreshSession()
        {
            if (this.InvokeRequired)
            {
                this.Invoke(new EmptyEventHandler(this.RefreshSession));
            }
            else
            {
                this.UpdateSessionProgress();
                this._parent.RefreshSession();
            }
        }
        void UpdateSessionProgress()
        {
            if (this.Running)
            {   
                IModelingProgress progress = this._compliance.Progress;
                StringBuilder sb = new StringBuilder();
                if (this._refreshInterval == -1)
                {
                    sb.Append("Automatic refreshing is currently paused.\n(Press F5 to refresh manually.)\n\n");
                }
                if (progress == null || progress.Scenario == null)
                {
                    if (this._refreshInterval != -1) { sb.Append("Refreshing compliance model progress ..."); }
                }
                else
                {
                    if (this._stopRequested) { sb.Append("Waiting to stop ...\n(Press stop again to terminate immediately.)\n\n"); }
                    sb.Append("Scenario: "); sb.Append(Global.GetTitleCase(progress.Scenario.ToString())); sb.Append("\n");
                    if (progress.ModelYear == null) { sb.Append("Model Year: N/A\n"); }
                    else { sb.Append("Model Year: "); sb.Append(progress.ModelYear); sb.Append("\n"); }
                    Manufacturer mfr = progress.Manufacturer;
                    if (mfr == null)
                    {
                        sb.Append("Manufacturer: N/A\n");
                    }
                    else
                    {
                        ManufacturerModelingData mmd = progress.Manufacturer.ModelingData;
                        sb.Append("Manufacturer: "); sb.Append(Global.GetTitleCase(progress.Manufacturer.Description.Name, 4));
                        sb.Append("\n");
                        sb.Append("         pSTND  STND  CAFE  Credits  Fines  Costs\n");
                        sb.Append("    RC0:  "); this.UpdateSessionProgress_Helper_UpdateRC(sb, mmd, RegulatoryClass.Unregulated );
                        sb.Append("    RC1:  "); this.UpdateSessionProgress_Helper_UpdateRC(sb, mmd, RegulatoryClass.DomesticAuto);
                        sb.Append("    RC2:  "); this.UpdateSessionProgress_Helper_UpdateRC(sb, mmd, RegulatoryClass.ImportedAuto);
                        sb.Append("    RC3:  "); this.UpdateSessionProgress_Helper_UpdateRC(sb, mmd, RegulatoryClass.LightTruck  );
                    }
                }
                this.ux_progress.Text = sb.ToString();
                this.ux_additionalInfo.Text = (progress == null || progress.AdditionalInfo == null) ? string.Empty :
                    progress.AdditionalInfo.ToString();
                this.ux_runtime.Text = Global.GetTimeString(this._compliance.Runtime);
            }
            else if (this.Reporting)
            {   
                this.ux_additionalInfo.Text = this._reporter.Progress;
            }
            else
            {
                this.ux_additionalInfo.Clear();
            }
        }
        void UpdateSessionProgress_Helper_UpdateRC(StringBuilder sb, ManufacturerModelingData mmd, RegulatoryClass regClass)
        {
            if (double.IsNaN(mmd.CAFE[regClass]))
            {
                sb.Append(" -     -     -        -      -      -\n");
            }
            else
            {
                double d;
                d = mmd.PreliminaryStandard[regClass];
                if (d < 10 || (d >= 100 && d < 1000)) { sb.Append(" "); }
                sb.Append((d < 100) ? d.ToString("0.0") : d.ToString("0")); sb.Append("  ");
                d = mmd.Standard[regClass];
                if (d < 10 || (d >= 100 && d < 1000)) { sb.Append(" "); }
                sb.Append((d < 100) ? d.ToString("0.0") : d.ToString("0")); sb.Append("  ");
                d = mmd.CAFE[regClass];
                if (d < 10 || (d >= 100 && d < 1000)) { sb.Append(" "); }
                sb.Append((d < 100) ? d.ToString("0.0") : d.ToString("0")); sb.Append("   ");
                this.UpdateSessionProgress_Helper_FormatValue(sb, mmd.Credits [regClass]);
                this.UpdateSessionProgress_Helper_FormatValue(sb, mmd.Fines   [regClass]);
                this.UpdateSessionProgress_Helper_FormatValue(sb, mmd.TechCost[regClass]);
                sb.Append("\n");
            }
        }
        void UpdateSessionProgress_Helper_FormatValue(StringBuilder sb, double d)
        {
            double abs_d = Math.Abs(d);
            string unit = "  ";
            if      (abs_d > 1000000000) { d /= 1000000000; unit = " b"; }
            else if (abs_d >    1000000) { d /=    1000000; unit = " m"; }
            else if (abs_d >       1000) { d /=       1000; unit = " k"; }
            if      (d < -100) {                   }
            else if (d <  -10) { sb.Append(  " "); }
            else if (d <    0) { sb.Append( "  "); }
            else if (d <   10) { sb.Append("   "); }
            else if (d <  100) { sb.Append( "  "); }
            else               { sb.Append(  " "); }
            sb.Append(d.ToString("0"));
            sb.Append(unit);
            sb.Append(' ');
        }
        void UpdateStatus(string value)
        {
            this._status = value;
            this._parent.SetStatus(this, value);
        }
        string GetGenericSessionName()
        {
            return "Session " + this._index.ToString();
        }
        #endregion
        #region 
        public string SessionName
        {
            get { return (this._path == null) ? this.GetGenericSessionName() : Path.GetFileNameWithoutExtension(this._path); }
        }
        public string SessionPath { get { return this._path; } }
        public ICompliance Compliance { get { return this._compliance; } set { this._compliance = value; this.SaveRequired = true; } }
        public Industry Data { get { return this._data; } set { this._data = value; this.SaveRequired = true; } }
        public ModelingSettings Settings { get { return this._settings; } set { this._settings = value; this.SaveRequired = true; } }
        public bool Running { get { return (this._compliance != null && this._compliance.Running); } }
        public bool Reporting { get { return this._reporter != null; } }
        public bool RunningOrReporting { get { return this.Running || this.Reporting; } }
        public string Status { get { return this._status; } }
        public int RefreshInterval
        {
            get { return this._refreshInterval; }
            set
            {
                if (value == -1) { this._refreshInterval = -1; }    
                else
                {
                    this._refreshInterval = (value < MinRefreshInterval) ? MinRefreshInterval :
                        value - (value % MinRefreshInterval);       
                }
                this.SaveRequired = true;
            }
        }
        public bool SaveRequired { get { return this._saveRequired; } set { this._saveRequired = value; this.RefreshSession(); } }
        #endregion
        #region 
        #region 
        System.Windows.Forms.RichTextBox ux_progress;
        System.Windows.Forms.Splitter ux_splitter;
        System.Windows.Forms.Panel ux_statusPanel;
        System.Windows.Forms.Label ux_runtime;
        System.Windows.Forms.RichTextBox ux_additionalInfo;
        #endregion
        public const int MinRefreshInterval = 125;
        static int SessionCount = 1;
        Cafe2d       _parent;
        int          _index;
        string       _path;
        ZipUtilities _compressor;
        ICompliance      _compliance;
        Industry         _data;
        ModelingSettings _settings;
        Reporter         _reporter;
        string _status;
        int    _refreshInterval;    
        bool   _saveRequired;
        bool   _stopRequested;
        #endregion
	}
}

