using System;
using System.Drawing;
using Volpe.Cafe.IO;
namespace Volpe.Cafe
{
    [Serializable]
    public class EmissionsRates : ICloneable
    {
        #region 
        [Serializable]
        public struct RateValues : ICloneable
        {
            #region 
            object ICloneable.Clone()
            {
                return this.Clone();
            }
            public RateValues Clone()
            {
                RateValues rt = new RateValues();
                int ubound0 = this.Ldv.GetUpperBound(0) + 1;
                int ubound1 = this.Ldv.GetUpperBound(1) + 1;
                rt.Ldv   = new double[ubound0, ubound1];
                rt.Ldt1  = new double[ubound0, ubound1];
                rt.Ldt2  = new double[ubound0, ubound1];
                rt.Hdv2b = new double[ubound0, ubound1];
                for (int i = 0; i < ubound0; i++)
                {
                    for (int j = 0; j < ubound1; j++)
                    {
                        rt.Ldv  [i, j] = this.Ldv[i, j];
                        rt.Ldt1 [i, j] = this.Ldt1[i, j];
                        rt.Ldt2 [i, j] = this.Ldt2[i, j];
                        rt.Hdv2b[i, j] = this.Hdv2b[i, j];
                    }
                }
                return rt;
            }
            #endregion
            public void Initialize(int length)
            {
                this.Ldv   = new double[length, length];
                this.Ldt1  = new double[length, length];
                this.Ldt2  = new double[length, length];
                this.Hdv2b = new double[length, length];
            }
            public double[,] Ldv;
            public double[,] Ldt1;
            public double[,] Ldt2;
            public double[,] Hdv2b;
        }
        [Serializable]
        public struct Emissions : ICloneable
        {
            #region 
            object ICloneable.Clone()
            {
                return this.Clone();
            }
            public Emissions Clone()
            {
                Emissions em   = new Emissions();
                em.Gasoline    = this.Gasoline.Clone();
                em.GasolineRfg = this.GasolineRfg.Clone();
                em.Diesel      = this.Diesel.Clone();
                em.E85         = this.E85.Clone();
                em.CNG         = this.CNG.Clone();
                em.Hydrogen    = this.Hydrogen.Clone();
                em.Spare1      = this.Spare1.Clone();
                em.Spare2      = this.Spare2.Clone();
                return em;
            }
            #endregion
            public RateValues Gasoline;
            public RateValues GasolineRfg;
            public RateValues Diesel;
            public RateValues E85;
            public RateValues CNG;
            public RateValues Hydrogen;
            public RateValues Spare1;
            public RateValues Spare2;
        }
        #endregion
        #region 
        private EmissionsRates()
        {
        }
        public EmissionsRates(string path)
            : this(path, "")
        {
        }
        public EmissionsRates(string path, string password)
            : this()
        {
            Input input = this.LoadFile(path, password);
            try
            {   
                this.ParseFile(input);
            }
            catch (Exception ex)
            {   
                throw new InputException("Errors were encountered while parsing file: " +
                    ex.Message, path, ex);
            }
            finally
            {   
                input.Close();
            }
        }
        #endregion
        #region 
        #region 
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        public EmissionsRates Clone()
        {
            EmissionsRates er = new EmissionsRates();
            er._co  = this._co .Clone();
            er._voc = this._voc.Clone();
            er._nox = this._nox.Clone();
            er._pm  = this._pm .Clone();
            er._sox = this._sox.Clone();
            return er;
        }
        #endregion
        private Input LoadFile(string path, string password)
        {
            return new Input(path);
        }
        private void ParseFile(Input input)
        {
            string[] sheets = input.SheetNames;
            string[] names = {
                                 "CO Rates - Gas", "CO Rates - Gas Rfg", "CO Rates - Diesel", "CO Rates - E85",
                                 "CO Rates - CNG", "CO Rates - Hydrogen", "CO Rates - Spare1", "CO Rates - Spare2",
                                 "VOC Rates - Gas", "VOC Rates - Gas Rfg", "VOC Rates - Diesel", "VOC Rates - E85",
                                 "VOC Rates - CNG", "VOC Rates - Hydrogen", "VOC Rates - Spare1", "VOC Rates - Spare2",
                                 "NOX Rates - Gas", "NOX Rates - Gas Rfg", "NOX Rates - Diesel", "NOX Rates - E85", 
                                 "NOX Rates - CNG", "NOX Rates - Hydrogen", "NOX Rates - Spare1", "NOX Rates - Spare2",
                                 "PM Rates - Gas", "PM Rates - Gas Rfg", "PM Rates - Diesel", "PM Rates - E85", 
                                 "PM Rates - CNG", "PM Rates - Hydrogen", "PM Rates - Spare1", "PM Rates - Spare2",
                                 "SO2 Rates - Gas", "SO2 Rates - Gas Rfg", "SO2 Rates - Diesel", "SO2 Rates - E85", 
                                 "So2 Rates - CNG", "SO2 Rates - Hydrogen", "SO2 Rates - Spare1", "SO2 Rates - Spare2"
                             };
            int[] indexes = {
                                -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 
                                -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
                                -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
                                -1, -1, -1, -1, -1, -1, -1, -1, -1, -1
                            };
            for (int i = 0; i < sheets.Length; i++)
            {
                for (int j = 0; j < names.Length; j++)
                {
                    if (input.Compare(sheets[i], names[j], true))
                    {
                        indexes[j] = i;
                        break;
                    }
                }
            }
            input.VerifyIndexes(indexes, names);
            this.ParseFile_LoadSheet(input, indexes[ 0], ref this._co.Gasoline);
            this.ParseFile_LoadSheet(input, indexes[ 1], ref this._co.GasolineRfg);
            this.ParseFile_LoadSheet(input, indexes[ 2], ref this._co.Diesel);
            this.ParseFile_LoadSheet(input, indexes[ 3], ref this._co.E85);
            this.ParseFile_LoadSheet(input, indexes[ 4], ref this._co.CNG);
            this.ParseFile_LoadSheet(input, indexes[ 5], ref this._co.Hydrogen);
            this.ParseFile_LoadSheet(input, indexes[ 6], ref this._co.Spare1);
            this.ParseFile_LoadSheet(input, indexes[ 7], ref this._co.Spare2);
            this.ParseFile_LoadSheet(input, indexes[ 8], ref this._voc.Gasoline);
            this.ParseFile_LoadSheet(input, indexes[ 9], ref this._voc.GasolineRfg);
            this.ParseFile_LoadSheet(input, indexes[10], ref this._voc.Diesel);
            this.ParseFile_LoadSheet(input, indexes[11], ref this._voc.E85);
            this.ParseFile_LoadSheet(input, indexes[12], ref this._voc.CNG);
            this.ParseFile_LoadSheet(input, indexes[13], ref this._voc.Hydrogen);
            this.ParseFile_LoadSheet(input, indexes[14], ref this._voc.Spare1);
            this.ParseFile_LoadSheet(input, indexes[15], ref this._voc.Spare2);
            this.ParseFile_LoadSheet(input, indexes[16], ref this._nox.Gasoline);
            this.ParseFile_LoadSheet(input, indexes[17], ref this._nox.GasolineRfg);
            this.ParseFile_LoadSheet(input, indexes[18], ref this._nox.Diesel);
            this.ParseFile_LoadSheet(input, indexes[19], ref this._nox.E85);
            this.ParseFile_LoadSheet(input, indexes[20], ref this._nox.CNG);
            this.ParseFile_LoadSheet(input, indexes[21], ref this._nox.Hydrogen);
            this.ParseFile_LoadSheet(input, indexes[22], ref this._nox.Spare1);
            this.ParseFile_LoadSheet(input, indexes[23], ref this._nox.Spare2);
            this.ParseFile_LoadSheet(input, indexes[24], ref this._pm.Gasoline);
            this.ParseFile_LoadSheet(input, indexes[25], ref this._pm.GasolineRfg);
            this.ParseFile_LoadSheet(input, indexes[26], ref this._pm.Diesel);
            this.ParseFile_LoadSheet(input, indexes[27], ref this._pm.E85);
            this.ParseFile_LoadSheet(input, indexes[28], ref this._pm.CNG);
            this.ParseFile_LoadSheet(input, indexes[29], ref this._pm.Hydrogen);
            this.ParseFile_LoadSheet(input, indexes[30], ref this._pm.Spare1);
            this.ParseFile_LoadSheet(input, indexes[31], ref this._pm.Spare2);
            this.ParseFile_LoadSheet(input, indexes[32], ref this._sox.Gasoline);
            this.ParseFile_LoadSheet(input, indexes[33], ref this._sox.GasolineRfg);
            this.ParseFile_LoadSheet(input, indexes[34], ref this._sox.Diesel);
            this.ParseFile_LoadSheet(input, indexes[35], ref this._sox.E85);
            this.ParseFile_LoadSheet(input, indexes[36], ref this._sox.CNG);
            this.ParseFile_LoadSheet(input, indexes[37], ref this._sox.Hydrogen);
            this.ParseFile_LoadSheet(input, indexes[38], ref this._sox.Spare1);
            this.ParseFile_LoadSheet(input, indexes[39], ref this._sox.Spare2);
        }
        private void ParseFile_LoadSheet(Input input, int index, ref RateValues rate)
        {
            input.ActivateWorksheet(index);
            int minYear, maxYear;
            input.GetMinMaxYears(2, 1, out minYear, out maxYear);
            int years  = maxYear - minYear + 1;
            int offset = minYear - ModelYear.MinYear;
            rate.Initialize(maxYear - (ModelYear.MinYear - 1));
            int ldv   = 3;
            int ldt1  = ldv  + years + 4;
            int ldt2  = ldt1 + years + 4;
            int hdv2b = ldt2 + years + 4;
            for (int i = 0; i < years; i++)
            {
                for (int j = 0; j < years; j++)
                {
                    rate.Ldv  [i + offset, j + offset] = input.GetDouble(i + ldv  , j + 1);
                    rate.Ldt1 [i + offset, j + offset] = input.GetDouble(i + ldt1 , j + 1);
                    rate.Ldt2 [i + offset, j + offset] = input.GetDouble(i + ldt2 , j + 1);
                    rate.Hdv2b[i + offset, j + offset] = input.GetDouble(i + hdv2b, j + 1);
                }
            }
        }
        #endregion
        #region 
        public EmissionsRates.Emissions CO  { get { return this._co;  } }
        public EmissionsRates.Emissions VOC { get { return this._voc; } }
        public EmissionsRates.Emissions NOX { get { return this._nox; } }
        public EmissionsRates.Emissions PM  { get { return this._pm;  } }
        public EmissionsRates.Emissions SOX { get { return this._sox; } }
        #endregion
        #region 
        private EmissionsRates.Emissions _co;
        private EmissionsRates.Emissions _voc;
        private EmissionsRates.Emissions _nox;
        private EmissionsRates.Emissions _pm;
        private EmissionsRates.Emissions _sox;
        #endregion
    }
}

