using System;
using System.Runtime.Serialization;
using Volpe.Cafe.Collections;
using Volpe.Cafe.IO;
namespace Volpe.Cafe.Settings
{
    [Serializable]
    public sealed class ModelingSettings : ICloneable, ISerializable
    {
        #region 
        ModelingSettings(SerializationInfo info, StreamingContext context)
        {
            this.InputSettings          = (InputSettings         )info.GetValue("InputSettings"         , typeof(InputSettings));
            this.OutputSettings         = (OutputSettings        )info.GetValue("OutputSettings"        , typeof(OutputSettings));
            this.OperatingModes         = (OperatingModes        )info.GetValue("OperatingModes"        , typeof(OperatingModes));
            try
            {
                this.ParametersOverrides = (ParametersOverrides  )info.GetValue("ParametersOverrides"   , typeof(ParametersOverrides));
            }
            catch
            {
                this.ParametersOverrides = new ParametersOverrides();
                this.ParametersOverrides.FuelPriceEstimates     = Estimates.Average;
                this.ParametersOverrides.ReboundEffect          = -0.10D;
                this.ParametersOverrides.DiscountRate           =  0.03D;
                this.ParametersOverrides.VehicleTravelTimeValue = 24.64D;
                this.ParametersOverrides.MilitarySecurity       =  0.00D;
                this.ParametersOverrides.CO2Estimates           = CO2Estimates.Average;
                this.ParametersOverrides.ConsumerBenefits       =  0.25D;
            }
            this.EISSettings            = (EISSettings           )info.GetValue("EISSettings"           , typeof(EISSettings));
            this.OptimizationSettings   = (OptimizationSettings  )info.GetValue("OptimizationSettings"  , typeof(OptimizationSettings));
            this.MonteCarloSettings     = (MonteCarloSettings    )info.GetValue("MonteCarloSettings"    , typeof(MonteCarloSettings));
            this._technologies          = (TechnologyCollection  )info.GetValue("_technologies"         , typeof(TechnologyCollection));
            this._parameters            = (Parameters            )info.GetValue("_parameters"           , typeof(Parameters));
            this._emissions             = (EmissionsRates        )info.GetValue("_emissions"            , typeof(EmissionsRates));
            this._scenarios             = (ScenarioCollection    )info.GetValue("_scenarios"            , typeof(ScenarioCollection));
        }
        ModelingSettings()
        {
        }
        public ModelingSettings(string dataPath, string outputPath)
        {
            this.InputSettings = new InputSettings(dataPath);
            this.OutputSettings = new OutputSettings(outputPath);
            this.OperatingModes = new OperatingModes();
            this.OperatingModes.TechnologyFuelEstimates = Estimates.Average;
            this.OperatingModes.TechnologyCostEstimates = Estimates.Average;
            this.OperatingModes.FuelPriceEstimates = Estimates.Average;
            this.OperatingModes.CO2Estimates = CO2Estimates.Average;
            this.OperatingModes.Multithread = true;
            this.ParametersOverrides = new ParametersOverrides();
            this.OptimizationSettings = new OptimizationSettings();
            this.MonteCarloSettings = new MonteCarloSettings();
            this.EISSettings = new EISSettings();
        }
        #endregion
        #region 
        #region 
        public void GetObjectData(SerializationInfo info, StreamingContext context)
        {
            info.AddValue("InputSettings"         , this.InputSettings);
            info.AddValue("OutputSettings"        , this.OutputSettings);
            info.AddValue("OperatingModes"        , this.OperatingModes);
            info.AddValue("ParametersOverrides"   , this.ParametersOverrides);
            info.AddValue("EISSettings"           , this.EISSettings);
            info.AddValue("OptimizationSettings"  , this.OptimizationSettings);
            info.AddValue("MonteCarloSettings"    , this.MonteCarloSettings);
            info.AddValue("_technologies"         , this._technologies);
            info.AddValue("_parameters"           , this._parameters);
            info.AddValue("_emissions"            , this._emissions);
            info.AddValue("_scenarios"            , this._scenarios);
        }
        #endregion
        #region 
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        public ModelingSettings Clone()
        {
            ModelingSettings settings = new ModelingSettings();
            settings.InputSettings  = this.InputSettings;
            settings.OutputSettings = this.OutputSettings;
            settings.OperatingModes       = this.OperatingModes;
            settings.ParametersOverrides  = this.ParametersOverrides;
            settings.EISSettings          = this.EISSettings;
            settings.OptimizationSettings = this.OptimizationSettings;
            settings.MonteCarloSettings   = this.MonteCarloSettings.Clone();
            settings._technologies = (this._technologies == null) ? null : this._technologies.Clone();
            settings._parameters   = (this._parameters   == null) ? null : this._parameters  .Clone();
            settings._emissions    = (this._emissions    == null) ? null : this._emissions   .Clone();
            settings._scenarios    = (this._scenarios    == null) ? null : this._scenarios   .Clone();
            return settings;
        }
        #endregion
        public void WriteSummaryLog(LogWriter writer)
        {
            if (writer.Summary == null) { return; }
            writer.Summary.WriteLine("---------- Modeling Settings ----------");
            this.InputSettings          .WriteSummaryLog(writer);
            this.OutputSettings         .WriteSummaryLog(writer);
            this.OperatingModes         .WriteSummaryLog(writer);
            this.ParametersOverrides    .WriteSummaryLog(writer);
            this.EISSettings            .WriteSummaryLog(writer);
            this.OptimizationSettings   .WriteSummaryLog(writer);
            this.MonteCarloSettings     .WriteSummaryLog(writer);
            if (this._technologies != null && this._technologies.Count > 0)
            {
                string techStr = "--- Technologies Analyzed ---";
                for (int i = 0, techCount = this._technologies.Count; i < techCount; i++)
                {
                    Technology tech = this._technologies[i];
                    techStr += "\n  " + tech.Index + ". " + tech.Abbr + " - " + tech.Name;
                }
                writer.Summary.WriteLine(techStr + "\n");
            }
            if (this._scenarios != null && this._scenarios.Count > 0)
            {
                string scenStr = "--- Scenarios Analyzed ---";
                for (int i = 0, scenCount = this._scenarios.Count; i < scenCount; i++)
                {
                    Scenario scen = this._scenarios[i];
                    scenStr += "\n  " + scen.Index + ". " + scen.Description +
                        ((scen.IsBaseline) ? "  (Baseline)" : "");
                }
                writer.Summary.WriteLine(scenStr + "\n");
            }
        }
        public void UseRecommendedSettings()
        {
            this.InputSettings.EncryptInput  = false;
            this.InputSettings.InputPassword = string.Empty;
            this.OutputSettings.EncryptOutput                 = false;
            this.OutputSettings.OutputPassword                = string.Empty;
            this.OutputSettings.UseStrongEncryption           = false;
            this.OutputSettings.GenerateReports               = true ;
            this.OutputSettings.ReportIndustry                = true ;
            this.OutputSettings.ReportManufacturers           = false;
            this.OutputSettings.ReportVehicles                = true ;
            this.OutputSettings.ReportScenarioSummary         = true ;
            this.OutputSettings.ReportEffects                 = false;
            this.OutputSettings.ReportEnvironmentalAssessment = false;
            this.OutputSettings.ReportOptimizedIndustry       = false;
            this.OperatingModes.ModelingApproach                  = ModelingApproach.LowCostFirst;
            this.OperatingModes.AutoToggleApproach                = false;
            this.OperatingModes.NoFines                           = false;
            this.OperatingModes.MultiYearModeling                 = true;
            this.OperatingModes.MultiYearStartYear                = 2012;
            this.OperatingModes.SplitSharedEngsOrTrns             = true;
            this.OperatingModes.SplitConflictingEngsOrTrns        = true;
            this.OperatingModes.SkipSharedEngsOrTrns              = false;
            this.OperatingModes.SkipConflictingEngsOrTrns         = false;
            this.OperatingModes.TechnologyFuelEstimates           = Estimates.Average;
            this.OperatingModes.TechnologyCostEstimates           = Estimates.Average;
            this.OperatingModes.Backfill                          = true;
            this.OperatingModes.ClipTechImprovements              = true;
            this.OperatingModes.ClipTechCosts                     = false;
            this.OperatingModes.ReserveTechnologiesIfInCompliance = false;
            this.OperatingModes.IgnoreYearAvailable               = false;
            this.OperatingModes.IgnorePhaseIn                     = false;
            this.OperatingModes.IgnoreRefreshRedesign             = false;
            this.OperatingModes.IgnoreWeightInMsApplication       = true;
            this.OperatingModes.ShutoffEngineTechsAfterHEV        = false;
            this.OperatingModes.ShutoffTransmissionTechsAfterHEV  = true;
            this.OperatingModes.ShutoffElecTechsAfterHEV          = true;
            this.OperatingModes.GenerateExtendedLogFiles          = false;
            this.OperatingModes.FuelPriceEstimates                = Estimates.Average;
            this.OperatingModes.CO2Estimates                      = CO2Estimates.Average;
            this.OperatingModes.Multithread                       = true;
            this.ParametersOverrides.OverrideFuelPriceEstimates     = false;
            this.ParametersOverrides.OverrideReboundEffect          = false;
            this.ParametersOverrides.OverrideDiscountRate           = false;
            this.ParametersOverrides.OverrideVehicleTravelTimeValue = false;
            this.ParametersOverrides.OverrideMilitarySecurity       = false;
            this.ParametersOverrides.OverrideCO2Estimates           = false;
            this.ParametersOverrides.ScaleConsumerBenefits          = false;
            this.ParametersOverrides.FuelPriceEstimates             = Estimates.Average;
            this.ParametersOverrides.ReboundEffect                  = -0.10D;
            this.ParametersOverrides.DiscountRate                   =  0.03D;
            this.ParametersOverrides.VehicleTravelTimeValue         = 24.64D;
            this.ParametersOverrides.MilitarySecurity               =  0.00D;
            this.ParametersOverrides.CO2Estimates                   = CO2Estimates.Average;
            this.ParametersOverrides.ConsumerBenefits               =  0.25D;
            this.EISSettings.RunEISModelFlatCAFE      = false;
            this.EISSettings.RunEISModelGrowCAFE      = false;
            this.EISSettings.EISParametersFile        = string.Empty;
            this.EISSettings.EISTailpipeEmissionsFile = string.Empty;
            this.OptimizationSettings.Type                   = OptimizationType.Default;
            this.OptimizationSettings.Mode                   = OptimizationMode.MaximumNetBenefits;
            this.OptimizationSettings.AboveOptimum           = 75;
            this.OptimizationSettings.BelowOptimum           = 75;
            this.OptimizationSettings.Increment              = 0.0001;
            this.OptimizationSettings.MergedFleet            = false;
            this.OptimizationSettings.BackstopStandard       = true;
            this.OptimizationSettings.BackstopTargetFunction = false;
            this.OptimizationSettings.StartYear              = 2012;
            this.OptimizationSettings.StopOnFinedIterations  = false;
            this.OptimizationSettings.CountFinesTowardNB     = false;
            this.OptimizationSettings.DiscardMfrsWithFines   = false;
            this.MonteCarloSettings.TrialPairs    = 10000;
            this.MonteCarloSettings.UseTrialsFile = false;
            this.MonteCarloSettings.Distributed   = false;
        }
        #endregion
        #region 
        public TechnologyCollection Technologies
        {
            get { return this._technologies; }
            set
            {
                if (value == null)
                {
                    throw new ArgumentNullException("Technologies", "The specified value cannot be null.");
                }
                this._technologies = value;
            }
        }
        public Parameters Parameters
        {
            get { return this._parameters; }
            set
            {
                if (value == null)
                {
                    throw new ArgumentNullException("Parameters", "The specified value cannot be null.");
                }
                this._parameters = value;
            }
        }
        public EmissionsRates EmissionsRates
        {
            get { return this._emissions; }
            set
            {
                if (value == null)
                {
                    throw new ArgumentNullException("EmissionsRates", "The specified value cannot be null.");
                }
                this._emissions = value;
            }
        }
        public ScenarioCollection Scenarios
        {
            get { return this._scenarios; }
            set
            {
                if (value == null)
                {
                    throw new ArgumentNullException("Scenarios", "The specified value cannot be null.");
                }
                this._scenarios = value;
            }
        }
        #endregion
        #region 
        public InputSettings InputSettings;
        public OutputSettings OutputSettings;
        public OperatingModes OperatingModes;
        public ParametersOverrides ParametersOverrides;
        public EISSettings EISSettings;
        public OptimizationSettings OptimizationSettings;
        public MonteCarloSettings MonteCarloSettings;
        private TechnologyCollection _technologies;
        private Parameters _parameters;
        private EmissionsRates _emissions;
        private ScenarioCollection _scenarios;
        #endregion
    }
}

