function [varargout] = LoadDxChannels_big(File, varargin)
%---------------------------------------------------------------------------------------
% LoadDxChannels: Load DX3 channels.  It is used when the file is bigger
% than 200 M.  It runs slower than LoadDxChannels.  LoadDxChannels may
% cause 'out of memory' problem if the file is too big
%
% Syntax:
%
% [Dt] = LoadDxChannels('DX3_File');
%
%  Load all channels into struct Dt including:
%  XIncr - sample time interval (sec) ,
%  EventPts - Event starting index
%  time0 - time channel contains pre-trigger time
%
%
% load some channels
% [Chn1, Chn2, ...] = LoadDxChannels('DX3_File','channel_1','channel_2',...)
%
% %Load channels and sample time XIncr
% [Chn1, Chn2, ..., XIncr] = LoadDxChannels('DX3_File','channel_1','channel_2',..., 1)
%
% %Load channels,sample time XIncr, and time channel
% [Chn1, Chn2, ..., TimeChn, XIncr] = LoadDxChannels('DX3_File','channel_1','channel_2',..., 2)
%
%
%
%RExamples:
%DX3_File = 'D:\FAT\EIVD\00095.dx3';
%addpath E:\Application\GxMatLabLib; %add Lib path
%%load some channels
%[Brake, F1, PDT] = LoadDxChannels(DX3_File, 'Brake_Light', 'FTC', 'PDT_Button');
%[Brake, F1, PDT, Xinc] = LoadDxChannels(DX3_File, 'Brake_Light', 'FTC', 'PDT_Button', 1);
%[Brake, F1, PDT, time, Xinc] = LoadDxChannels(DX3_File, 'Brake_Light', 'FTC', 'PDT_Button', 2);
%
%%Loda all channels into Dt struct
%[Dt] = LoadDxChannels(DX3_File);% Dt has all data channels
%
% * Note: It is better to load all channels you need once as to save times
%
% See also: getDxChnNames,  getDxInfo
% 5/23/06 - GX added
% Return Data structure Dt also contains EventPts (pre-trigger points) and XIncr (sample time interval)

%
% problem - It has problem 'out of memory' when loading big file (253M)
% second time. This program load Dx channels one by one since it does not
% need a big block of continous memory to load all data once. --- GX 6/8/06
%
%---------------------------------------------------------------------------------------

nIns = nargin;
nOuts = nargout;
% OutChnNameAry = varargout;

fid=fopen(File,'r'); % existing ?
if fid == -1
    mstr = ['Loading DX Error: Failed to open file ', File, '!'];
    msgbox( mstr, 'Error','error');
    return;
else
    fclose(fid);
end

%Flag -----
Fplus = 0;
nInPars = length(varargin);
if nInPars > 0
    Vlast = varargin(nInPars);
    tem = cell2mat(Vlast);
    if length(tem) == 1  % load XIncr
        Fplus = tem; % = 2;load time channel that include pre-triger event time
    end
end
FLoadAllChannels = 0;
if Fplus == 0 % no XIncr flag
    if nIns == 1 % load all channels
        FLoadAllChannels = 1;
    end
elseif Fplus == 2 | Fplus == 1%load XIncr
    if nIns == 2 % load all channels
        FLoadAllChannels = 1;
    end
else
    mstr = ['Loading DX Error: Last parameter is ', num2str(Fplus), 'not correct. It should be 1 or 2!'];
    msgbox( mstr, 'Error','error');
    return;
end

% check if num of output is correct
if FLoadAllChannels
    if nOuts > 1
        mstr = ['Loading DX Error: Num of return variables =,' num2str(nOuts), ' > 1 !  It should be one structure variable. '];
        msgbox( mstr, 'Error','error');
        return;
    end
    
    if nIns > 1
        mstr = 'Loading DX Error: No flag inputs are needed for loading all DX file!';
        msgbox( mstr, 'Error','error');
        return;
    end    
else
    if  Fplus == 0
        nout_es = nIns - 1; %num of put variables suppose to be
    else
        nout_es = nIns - 2 + Fplus; %num of put variables suppose to be
    end
    if nOuts ~= nout_es
        mstr = ['Loading DX Error: Num of return variables = ', num2str(nOuts), ', should be ', num2str(nout_es), ' !'];
        msgbox( mstr, 'Error','error');
        return;
    end
end

% load DLL ---
fl = which('LoadDxChannels'); % find file's path
[pathstr, name, ext, versn] = fileparts(fl);
hd = [pathstr, '\LoadDxOut.h'];

% hd = 'E:\My_Utility\LoadDx\LoadDxOut.h'
% pathstr ='E:\My_Utility\Call'  %fortest ??????????????????

DllLib =[pathstr, '\LoadDx'];

loadlibrary (DllLib, hd);
%loadlibrary (DllLib, @HDproto);

if ~libisloaded('LoadDx')
    mstr = ['Loading DX Error: Failed to load DLL = ', DllLib];
    msgbox( mstr, 'Error','error');
    return;
end
%libfunctions LoadDx -full

%load Dx channel info ---
maxChns = 1000; %number of channels
sm.nChns = 0;
sm.pChnPoints = zeros(maxChns,1);
sc = libpointer('DX_INFO1', sm);
calllib('LoadDx', 'get_dx_info', sc,File ); % ---

DxHdr = get(sc, 'Value');
clear sc

chnNameAry = [];
if FLoadAllChannels == 1 % load all channels
    nInChns = DxHdr.nChns;
else
    if Fplus == 0 % no XIncr flag
        chnNameAry = varargin;
        nInChns = nIns -1;
    else %load XIncr
        nInChns = nIns - 2;
        chnNameAry = varargin(1:nInChns);
    end
end

%Load channels one by one to avoid using big block of memory ---
memblokLimit = 50e6;
Chn.XIncr = 0.000;
Chn.pChnPts = zeros(nInChns + 3,1);
if FLoadAllChannels == 1 % load all channels into DT structure
    % try to load all channels in several parts, each part should not be too
    % big < 100M ?
    totalMemory = DxHdr.nChns * DxHdr.nMaxPoints * 8;
    nblocks = ceil(totalMemory / memblokLimit);
    nBlockChns =  ceil(DxHdr.nChns / nblocks );
    istartChn = 1;
    iendChn = 0;
    for iblock = 1:nblocks
        iendChn = iendChn + nBlockChns;
        if iendChn > DxHdr.nChns
            iendChn = DxHdr.nChns;
        end
        [Ch] = loadSomeChns(DxHdr, istartChn, iendChn, File);
        %put into DT struct
        ncs = iendChn - istartChn + 1;
        for i1 = 1:ncs
            chnName = correctVarName(Ch(i1).nm);
            cmdstr= ['DT.', chnName, '=', 'Ch(i1).dt ;'];
            eval(cmdstr);
        end

        istartChn = iendChn + 1;
    end
    DT.XIncr = Ch(1).XIncr;
else % load only some channels
    LenNameStr = DxHdr.TotalFieldNamesLength + DxHdr.nChns; % get all channel names includes cammas
    nstr = blanks(LenNameStr);%make empty str to hold return name string
    nstr = calllib('LoadDx', 'getAllChnNames', nstr, File );
    [AllChnNmAry] = parseCommaStr(nstr);
    ntotalChns = length(AllChnNmAry);

    iSel = [];
    for i = 1:nInChns %find index of each channel
        %         iSel(i) = [];
        for j = 1:ntotalChns
            if strcmpi(chnNameAry(i), AllChnNmAry(j))
                iSel(i) = j;
                break;
            end
        end
        if isempty(iSel(i))
            mstr = ['Loading DX Error: Fail to find channel: ', chnNameAry{i}];
            msgbox( mstr, 'Error','error');
            return;
        end
    end

    NmStr = blanks(200);%make empty str to hold return name string
    for i = 1:nInChns
        idx = iSel(i);
        n1Dpoints = DxHdr.pChnPoints(idx);
        Chn.pD = zeros(n1Dpoints, 1); % make space for 1D array
        pChn = libpointer('CHN_OB', Chn);
        [pChn, ChnName] = calllib('LoadDx', 'LoadIthChns',pChn, idx - 1,NmStr,File );
        dChn = pChn;

        cmdstr= [ChnName, '=', 'dChn.pD(1:', num2str(n1Dpoints), ') ;'];
        eval(cmdstr);
        cms2 = [ 'varargout(i) = {', ChnName '} ;'];
        eval(cms2);
        clear Chn.pD pChn n1Dpoints;
    end
    DT.XIncr = dChn.XIncr;
end
clear pChn.pD
clear pChn;
unloadlibrary LoadDx;

% time0, XIncr ---
DT.EventPts = DxHdr.EventPts;
tm = ((1:DxHdr.nMaxPoints)- DxHdr.EventPts) * DT.XIncr;;

if FLoadAllChannels == 1 % load all channels into DT structure
    DT.time0 = tm' ;
    cms2 = 'varargout(1) = {DT} ;';
    eval(cms2);
else

    i = nInChns + 1;
    if Fplus == 1
        Xinc = dChn.XIncr;
        varargout(i) = {Xinc} ;
    elseif Fplus == 2
        Xinc = dChn.XIncr;
        time0 = tm';
        varargout(i) = {time0} ;
        varargout(i+1) = {Xinc} ;;
    end
end
% problem - It has problem when loading big file (253M) second time. not
% problem during first loading ???
%=====================================
function [Ch] = loadSomeChns(DxHdr, ist, iend, File)
% load some channels
ch = [];
nstr = blanks(2000);
n1Dpoints = 0;
for i = ist:iend
    n1Dpoints = n1Dpoints + DxHdr.pChnPoints(i);
end

Chn.pD = zeros(n1Dpoints, 1); % make space for 1D array

pChn = libpointer('CHN_OB', Chn);

is0 = ist -1;
ie0 = iend  -1;
[pChn,nstr] = calllib('LoadDx', 'LoadSomeChns',pChn, is0, ie0, nstr, File );

chnNameAry = parseCommaStr(nstr);

is = 1;
ie = 0;
ict = 1;
for i = ist:iend
    ie = ie + DxHdr.pChnPoints(i);
    Ch(ict).dt = pChn.pD(is:ie);
    Ch(ict).nm = chnNameAry{ict};
    Ch(ict).XIncr = pChn.XIncr;
    is = ie + 1   ;
    ict = ict + 1;
end
return;













